//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 37412 $
// $Date: 2019-01-31 15:31:22 +0100 (Thu, 31 Jan 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Time Domain Frequency Offset Compensation Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/TDFOCmp.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module TDFOCmp #(parameter NPIPE            =  7,  //Number of Pipeline stages
                 parameter NCOMBSTG         =  2,  //Number of Combinatorial stages
                 parameter SCALING          =  1,  //Scaling to [-4,4] range
                 parameter CROTDATAWIDTH    = 13,  //Width of Input Data
                 parameter CROTANGLEWIDTH   = 25,  //Width of Input Angle
                 parameter CVECTDATAWIDTH   = 16,  //Input Data Width
                 parameter CVECTANGLEWIDTH  = 21 //Width of Output Angle
                )(

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input   wire                                   nPhyRst, // Active LOW Reset
            input   wire                                   PhyClk,  // PHY Clock

            ///////////////////////////////////////////////
            // CordicVectTop Signals
            ///////////////////////////////////////////////
            input   wire                                   CVectDataEnIn,  // Enable for CordicVec
            input   wire signed      [CVECTDATAWIDTH-1:0]  CVReDataIn,     // Real Comp of Data
            input   wire signed      [CVECTDATAWIDTH-1:0]  CVImDataIn,     // Imag Comp of Data
            //
            output  wire signed      [CVECTANGLEWIDTH-1:0] CVAngleOut,     // Angle from Cordic Vect
            output  wire                                   CVectDone,      // Cordic Vect Done

            ///////////////////////////////////////////////
            // CordicRotTop Signals
            ///////////////////////////////////////////////
            // Common control
            input   wire                                   TDFOCmpMode, // 1 = Fine Compensation
            input   wire                                   CRotEnable,  // Enable for CordicRot

            // Secondary data 20MHz
`ifndef RW_NX_DERIV_CHBW20ONLY
            // Secondary 20MHz Cordic 0
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot0ReDataIn20S,  // Real Comp of Data Secondary
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot0ImDataIn20S,  // Imag Comp of Data Secondary
            output  wire signed      [CROTDATAWIDTH-1:0]   CRot0ReDataOut20S, // Real Comp of Data Secondary
            output  wire signed      [CROTDATAWIDTH-1:0]   CRot0ImDataOut20S, // Imag Comp of Data Secondary
            
            // Secondary 20MHz Cordic 1
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot1ReDataIn20S,   // Real Comp of Data Secondary
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot1ImDataIn20S,   // Imag Comp of Data Secondary
            output  reg  signed      [CROTDATAWIDTH-1:0]   CRot01ReDataOut20S, // Real Comp of Data Secondary
            output  reg  signed      [CROTDATAWIDTH-1:0]   CRot01ImDataOut20S, // Imag Comp of Data Secondary
`endif
            
            // Cordic 0
            input   wire signed      [CROTANGLEWIDTH-1:0]  CRot0AngleIn,     // Angle for CordicRot0
            input   wire                                   CRot0DataInValid, // Qualifies Input

            input   wire signed      [CROTDATAWIDTH-1:0]   CRot0ReDataIn0,   // Real Comp of Data 0
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot0ImDataIn0,   // Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot0ReDataIn1,   // Real Comp of Data 1
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot0ImDataIn1,   // Imag Comp of Data 1
`endif
            // Coarse LTF Compensation and Coarse+Fine Data Compensation (Cordic 0)
            output  wire signed      [CROTDATAWIDTH-1:0]   CRot0ReDataOut0,  // Real Comp of Data 0
            output  wire signed      [CROTDATAWIDTH-1:0]   CRot0ImDataOut0,  // Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            output  wire signed      [CROTDATAWIDTH-1:0]   CRot0ReDataOut1,  // Real Comp of Data 1
            output  wire signed      [CROTDATAWIDTH-1:0]   CRot0ImDataOut1,  // Imag Comp of Data 1
`endif
            output  wire                                   CRot0DataOutValid, // Qualifies output

            // Cordic 1
            input   wire signed      [CROTANGLEWIDTH-1:0]  CRot1AngleIn,     // Angle for CordicRot1
            input   wire                                   CRot1DataInValid, // Qualifies Input
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot1ReDataIn0,   // Real Comp of Data 0
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot1ImDataIn0,   // Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot1ReDataIn1,   // Real Comp of Data 1
            input   wire signed      [CROTDATAWIDTH-1:0]   CRot1ImDataIn1,   // Imag Comp of Data 1
`endif
            // Fine LTF Compensation (Averaged output of Cordics 0 & 1)
            output  reg  signed      [CROTDATAWIDTH-1:0]   CRot01ReDataOut0, // Real Comp of Data 0
            output  reg  signed      [CROTDATAWIDTH-1:0]   CRot01ImDataOut0, // Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            output  reg  signed      [CROTDATAWIDTH-1:0]   CRot01ReDataOut1, // Real Comp of Data 1
            output  reg  signed      [CROTDATAWIDTH-1:0]   CRot01ImDataOut1, // Imag Comp of Data 1
`endif
            output  reg                                    CRot01DataOutValid // Qualifies output
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam                            CONST_FINE = 1'b1;
localparam signed [CROTDATAWIDTH-1:0] CONST_ZERO = {{CROTDATAWIDTH}{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                                   CRot1DataOutValid;
wire    signed     [CROTDATAWIDTH-1:0] CRot1ReDataOut0;
wire    signed     [CROTDATAWIDTH-1:0] CRot1ImDataOut0;
wire                                   CRot1Enable;
wire    signed     [CROTDATAWIDTH:0]   CRot01ReDataOut0Int;
wire    signed     [CROTDATAWIDTH:0]   CRot01ImDataOut0Int;
wire    signed     [CROTDATAWIDTH-1:0] CRot01ReDataOut0Rnd;
wire    signed     [CROTDATAWIDTH-1:0] CRot01ImDataOut0Rnd;
`ifdef RW_NX_DERIV_PATH1
wire    signed     [CROTDATAWIDTH-1:0] CRot1ReDataOut1;
wire    signed     [CROTDATAWIDTH-1:0] CRot1ImDataOut1;
wire    signed     [CROTDATAWIDTH:0]   CRot01ReDataOut1Int;
wire    signed     [CROTDATAWIDTH:0]   CRot01ImDataOut1Int;
wire    signed     [CROTDATAWIDTH-1:0] CRot01ReDataOut1Rnd;
wire    signed     [CROTDATAWIDTH-1:0] CRot01ImDataOut1Rnd;
`endif
`ifndef RW_NX_DERIV_CHBW20ONLY
wire    signed     [CROTDATAWIDTH-1:0] CRot1ReDataOut20S;
wire    signed     [CROTDATAWIDTH-1:0] CRot1ImDataOut20S;
wire    signed     [CROTDATAWIDTH:0]   CRot01ReDataOut20SInt;
wire    signed     [CROTDATAWIDTH:0]   CRot01ImDataOut20SInt;
wire    signed     [CROTDATAWIDTH-1:0] CRot01ReDataOut20SRnd;
wire    signed     [CROTDATAWIDTH-1:0] CRot01ImDataOut20SRnd;
`endif

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module instantiates the following Cordics
//1 CordicVectTop module - this calculates the Coarse and Fine TDFO Angle
//2 TDFOCordicRotTop modules - this performs the rotation (TDFO compensation)

//Cordic for Phase computation
//Instanatiate CordicVectTop
CordicVectTop # (
                 //Parameters
                 .DATAWIDTH(CVECTDATAWIDTH),
                 .ANGLEWIDTH(CVECTANGLEWIDTH),
                 .SCALING(SCALING)
                 ) U_CVTop (
                            //Inputs
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .DataInEn(CVectDataEnIn),
                            .ReDataIn(CVReDataIn),
                            .ImDataIn(CVImDataIn),

                            //Outputs
                            .AngleOut(CVAngleOut),
                            .CordicDone(CVectDone)
                            );

//Cordics for rotation
//Although only 1 Cordic rotator is required for Data Compensation, we need to
//instantiate 2 Cordics since we need to Fine+Coarse compensation L1 and L2 in
//parallel and then average them before sending them out to FFT Memory

TDFOCordicRotTop # (
                //Parameters
                .NPIPE(NPIPE),
                .NCOMBSTG(NCOMBSTG),
                .SCALING(SCALING),
                .DATAWIDTH(CROTDATAWIDTH),
                .ANGLEWIDTH(CROTANGLEWIDTH)
               ) U_CRTop0 (
                           //Inputs
                           .PhyClk(PhyClk),
                           .nPhyRst(nPhyRst),
                           .Enable(CRotEnable),
                           .AngleIn(CRot0AngleIn),
                           .DataInValid(CRot0DataInValid),
                           .ReDataIn0(CRot0ReDataIn0),
                           .ImDataIn0(CRot0ImDataIn0),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataIn1(CRot0ReDataIn1),
                           .ImDataIn1(CRot0ImDataIn1),
`endif

                           //Outputs
                           .ReDataOut0(CRot0ReDataOut0),
                           .ImDataOut0(CRot0ImDataOut0),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataOut1(CRot0ReDataOut1),
                           .ImDataOut1(CRot0ImDataOut1),
`endif
                           .DataOutValid(CRot0DataOutValid)
                           );

TDFOCordicRotTop # (
                //Parameters
                .NPIPE(NPIPE),
                .NCOMBSTG(NCOMBSTG),
                .SCALING(SCALING),
                .DATAWIDTH(CROTDATAWIDTH),
                .ANGLEWIDTH(CROTANGLEWIDTH)
               ) U_CRTop1 (
                           //Inputs
                           .PhyClk(PhyClk),
                           .nPhyRst(nPhyRst),
                           .Enable(CRot1Enable),
                           .AngleIn(CRot1AngleIn),
                           .DataInValid(CRot1DataInValid),
                           .ReDataIn0(CRot1ReDataIn0),
                           .ImDataIn0(CRot1ImDataIn0),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataIn1(CRot1ReDataIn1),
                           .ImDataIn1(CRot1ImDataIn1),
`endif
                           //Outputs
                           .ReDataOut0(CRot1ReDataOut0),
                           .ImDataOut0(CRot1ImDataOut0),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataOut1(CRot1ReDataOut1),
                           .ImDataOut1(CRot1ImDataOut1),
`endif
                           .DataOutValid(CRot1DataOutValid)
                           );

//Enable for Cordic 1
assign CRot1Enable = CRotEnable & TDFOCmpMode;

//During LTF Compensation the outputs of CordicRot have to be averaged
//Otherwise the output of the first Cordic needs to be sent out
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRot01Out_Blk
      if (nPhyRst == 1'b0) begin
         CRot01ReDataOut0 <= CONST_ZERO;
         CRot01ImDataOut0 <= CONST_ZERO;
`ifdef RW_NX_DERIV_PATH1
         CRot01ReDataOut1 <= CONST_ZERO;
         CRot01ImDataOut1 <= CONST_ZERO;
`endif
      end
      else if (TDFOCmpMode == CONST_FINE) begin
         CRot01ReDataOut0 <= CRot01ReDataOut0Rnd;
         CRot01ImDataOut0 <= CRot01ImDataOut0Rnd;
`ifdef RW_NX_DERIV_PATH1
         CRot01ReDataOut1 <= CRot01ReDataOut1Rnd;
         CRot01ImDataOut1 <= CRot01ImDataOut1Rnd; 
`endif
      end
      else begin
         CRot01ReDataOut0 <= CONST_ZERO;
         CRot01ImDataOut0 <= CONST_ZERO;
`ifdef RW_NX_DERIV_PATH1
         CRot01ReDataOut1 <= CONST_ZERO;
         CRot01ImDataOut1 <= CONST_ZERO;
`endif
      end

   end //CRot01Out_Blk

//Adding L1 & L2 and rounding by 1 bit to generate (L1+L2/2)
assign CRot01ReDataOut0Int = CRot1ReDataOut0 + CRot0ReDataOut0;
assign CRot01ImDataOut0Int = CRot1ImDataOut0 + CRot0ImDataOut0;

Round #(
        .INPUT_WIDTH(CROTDATAWIDTH+1),
        .OUTPUT_WIDTH(CROTDATAWIDTH)
       )
        U_ROUNDCMP0(
                    .InputData(CRot01ReDataOut0Int),
                    .RoundData(CRot01ReDataOut0Rnd)
                   );

Round #(
        .INPUT_WIDTH(CROTDATAWIDTH+1),
        .OUTPUT_WIDTH(CROTDATAWIDTH)
       )
        U_ROUNDCMP1(
                    .InputData(CRot01ImDataOut0Int),
                    .RoundData(CRot01ImDataOut0Rnd)
                   );

`ifdef RW_NX_DERIV_PATH1
      assign CRot01ReDataOut1Int = CRot1ReDataOut1 + CRot0ReDataOut1;
      assign CRot01ImDataOut1Int = CRot1ImDataOut1 + CRot0ImDataOut1;

      Round #(
              .INPUT_WIDTH(CROTDATAWIDTH+1),
              .OUTPUT_WIDTH(CROTDATAWIDTH)
             )
              U_ROUNDCMP2(
                          .InputData(CRot01ReDataOut1Int),
                          .RoundData(CRot01ReDataOut1Rnd)
                         );

      Round #(
              .INPUT_WIDTH(CROTDATAWIDTH+1),
              .OUTPUT_WIDTH(CROTDATAWIDTH)
             )
              U_ROUNDCMP3(
                          .InputData(CRot01ImDataOut1Int),
                          .RoundData(CRot01ImDataOut1Rnd)
                         );
`endif

// Generate CRot01DataOutValid
always @ (posedge PhyClk or negedge nPhyRst)
   begin: DataValid_Blk
      if (nPhyRst == 1'b0)
         CRot01DataOutValid <= 1'b0;
      else
         CRot01DataOutValid <= CRot1DataOutValid;
   end //DataValid_Blk


// Secondary data 20MHz
`ifndef RW_NX_DERIV_CHBW20ONLY

TDFOCordicRotTop # (
                //Parameters
                .NPIPE(NPIPE),
                .NCOMBSTG(NCOMBSTG),
                .SCALING(SCALING),
                .DATAWIDTH(CROTDATAWIDTH),
                .ANGLEWIDTH(CROTANGLEWIDTH)
               ) U_CRTop20S0 (
                           //Inputs
                           .PhyClk(PhyClk),
                           .nPhyRst(nPhyRst),
                           .Enable(CRotEnable),
                           .AngleIn(CRot0AngleIn),
                           .DataInValid(CRot0DataInValid),
                           .ReDataIn0(CRot0ReDataIn20S),
                           .ImDataIn0(CRot0ImDataIn20S),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataIn1(CONST_ZERO),
                           .ImDataIn1(CONST_ZERO),
`endif

                           //Outputs
                           .ReDataOut0(CRot0ReDataOut20S),
                           .ImDataOut0(CRot0ImDataOut20S),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataOut1(),
                           .ImDataOut1(),
`endif
                           .DataOutValid()
                           );

TDFOCordicRotTop # (
                //Parameters
                .NPIPE(NPIPE),
                .NCOMBSTG(NCOMBSTG),
                .SCALING(SCALING),
                .DATAWIDTH(CROTDATAWIDTH),
                .ANGLEWIDTH(CROTANGLEWIDTH)
               ) U_CRTop20S1 (
                           //Inputs
                           .PhyClk(PhyClk),
                           .nPhyRst(nPhyRst),
                           .Enable(CRot1Enable),
                           .AngleIn(CRot1AngleIn),
                           .DataInValid(CRot1DataInValid),
                           .ReDataIn0(CRot1ReDataIn20S),
                           .ImDataIn0(CRot1ImDataIn20S),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataIn1(CONST_ZERO),
                           .ImDataIn1(CONST_ZERO),
`endif
                           //Outputs
                           .ReDataOut0(CRot1ReDataOut20S),
                           .ImDataOut0(CRot1ImDataOut20S),
`ifdef RW_NX_DERIV_PATH1
                           .ReDataOut1(),
                           .ImDataOut1(),
`endif
                           .DataOutValid()
                           );

//During LTF Compensation the outputs of CordicRot have to be averaged
//Otherwise the output of the first Cordic needs to be sent out
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRot0120SOut_Blk
      if (nPhyRst == 1'b0) begin
         CRot01ReDataOut20S <= CONST_ZERO;
         CRot01ImDataOut20S <= CONST_ZERO;
      end
      else if (TDFOCmpMode == CONST_FINE) begin
         CRot01ReDataOut20S <= CRot01ReDataOut20SRnd;
         CRot01ImDataOut20S <= CRot01ImDataOut20SRnd;
      end
      else begin
         CRot01ReDataOut20S <= CONST_ZERO;
         CRot01ImDataOut20S <= CONST_ZERO;
      end
   end //CRot0120SOut_Blk

//Adding L1 & L2 and rounding by 1 bit to generate (L1+L2/2)
assign CRot01ReDataOut20SInt = CRot1ReDataOut20S + CRot0ReDataOut20S;
assign CRot01ImDataOut20SInt = CRot1ImDataOut20S + CRot0ImDataOut20S;

Round #(
        .INPUT_WIDTH(CROTDATAWIDTH+1),
        .OUTPUT_WIDTH(CROTDATAWIDTH)
       )
        U_ROUNDCMP20S0(
                    .InputData(CRot01ReDataOut20SInt),
                    .RoundData(CRot01ReDataOut20SRnd)
                   );

Round #(
        .INPUT_WIDTH(CROTDATAWIDTH+1),
        .OUTPUT_WIDTH(CROTDATAWIDTH)
       )
        U_ROUNDCMP20S1(
                    .InputData(CRot01ImDataOut20SInt),
                    .RoundData(CRot01ImDataOut20SRnd)
                   );

`endif


endmodule //TDFOCmp

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
