//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 40019 $
// $Date: 2019-10-07 16:19:21 +0200 (Mon, 07 Oct 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : TDFO Estimation and Compensation Controller
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/TDFOCntrl.v $
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module TDFOCntrl #(parameter ANGLEWIDTH = 21   // Cordic Vect Angle Width
                  )(

            ///////////////////////////////////////////////
            // Clock and Reset Manager interface
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                            nPhyRst,      // Active LOW Reset
            input   wire                            PhyClk,       // PHY Clock

            ///////////////////////////////////////////////
            // TD FSM interface
            ///////////////////////////////////////////////
            input   wire        [1:0]               Bandwidth,    // Bandwidth of current pkt
            input   wire                            TDFOStrtP,    // TDFO Estimation Start
            input   wire                            DataCmpStrtP, // Data Compensation Start
            input   wire                            TDFOEn,       // TDFO block enable
            //
            output  reg                             CmpLTFDoneP,  // LTF Compensation Done
            output  reg                             FineFODone,   // Fine FO estimation Done
            
            ///////////////////////////////////////////////
            // Rx FD interface
            ///////////////////////////////////////////////
            input  wire         [21:0]              FDCpeSlope,   // CPE Slope from FDO

            ///////////////////////////////////////////////
            // AGC interface
            ///////////////////////////////////////////////
            output  reg         [7:0]               FOAngle,      // FO Angle

            ///////////////////////////////////////////////
            // Timing Boundary Estimation interface
            ///////////////////////////////////////////////
            input   wire        [7:0]               TBECount,
            input   wire                            TBECountValid,
            //
            output  wire                            RxLTFValid,    // Qualifies LTF Outputs

            ///////////////////////////////////////////////
            // Registers interface
            ///////////////////////////////////////////////
            input   wire                            Force20,       // 20 MHz Mode only
            input   wire                            Force40,       // 20/40 MHz Mode only
            input   wire        [8:0]               AutoCorrPlatSumStart,
            input   wire        [8:0]               AutoCorrPlatSumEnd,
            input   wire        [8:0]               AutoCorrPlatFallSearchStart,
            input   wire        [3:0]               AutoCorrPlatFallIndexOffset,
            input   wire        [8:0]               AutoCorrPhSumStart,
            input   wire        [8:0]               AutoCorrPhSumEnd,
            input   wire                            TDFOCompEn,
            input   wire                            TDFOFineEstEn,
            input   wire                            TDFOCpeSlopeEn,
            //
            output wire signed  [ANGLEWIDTH-1:0]    CoarseFOAngle, // Coarse FO Angle
            output wire signed  [ANGLEWIDTH-1:0]    FineFOAngle,   // Fine FO Angle

            ///////////////////////////////////////////////
            // Internal Control interface
            ///////////////////////////////////////////////
            input  wire                             DataValid,          // Qualifies TDFO Data Ip
            //
            input  wire                             PlatFallLowP,       // Indicates Plateau Fall low
            input  wire                             PlatFallHighP,      // Indicates Plateau Fall high
            input  wire                             PlatThreshDoneP,    // Threshold Calc done
            input  wire                             CorrValAccValidP,   // Qualifies CorrValAcc
            //
            output reg                              CorrValAccEn,       // Enables CorrVal Acc
            output reg                              AbsCValAccEn,       // Enables AbsCorrVal Acc
            output reg                              PlatFallLowDetEn,   // Enables Plateau Fall low Detection
            output reg                              PlatFallHighDetEn,  // Enables Plateau Fall high Detection
            output wire                             ModApproxCompEn,    // En ModApprox computation
            output reg                              CorrEn,             // Enables Correlation
            output reg                              MovAvgEn,           // Enables Moving Avg Block
            output wire                             MovAvgClr,          // Clear Mov Avg Value
            output wire                             TDFOEstMode,        // 1 = Fine TDFO Estimation
            output wire                             TDFOCmpMode,        // 1 = LTF Fine Compensation
            output wire                             TDFOLTFCmpMode,     // 1 = LTF Coarse Compensation
            output wire                             TDFOCmpDataMode,    // 1 = Data Coarse+Fine Compensation
            output reg                              CorValFineDataEn,   // Data Enable for Fine Auto Correlation
            // Cordic Vect control
            input  wire signed  [ANGLEWIDTH-1:0]    CVAngleOut,         // Angle from Cordic Vect
            input  wire                             CVectDone,          // Angle Computation Done
            output reg                              CVectDataEnIn,      // Enable for Cordic Vect
            output wire                             CVectOngoing,       // Cordic Vect on going
            // Cordic Rot control
            input  wire                             CRot0DataOutValid,  // Qualifies C Rot Op
            input  wire                             CRot01DataOutValid, // Qualifies C Rot Op
            output reg                              CRotEnable,         // Enable for Cordic Rot
            output wire signed [ANGLEWIDTH+3:0]     CRot0Angle,         // Angle for Cordic Rot 0
            output reg                              CRot0DataInValid,   // Qualifies Cordic Rot 0 Inputs
            output wire signed [ANGLEWIDTH+3:0]     CRot1Angle,         // Angle for Cordic Rot 1
            output reg                              CRot1DataInValid,   // Qualifies Cordic Rot 1 Inputs

            ///////////////////////////////////////////////
            // FFT Memory interface
            ///////////////////////////////////////////////
            output wire                             FFTMemWrEn,         // Write Enable
            output reg                              FFTMemSel,          // Select memory control

            ///////////////////////////////////////////////
            // TDFO Memory interface
            ///////////////////////////////////////////////
            output reg                              MemWrEn,            // Write Enable
            output reg                              MemWrSel,           // Buffer selection
            output wire                             MemRdEn,            // Read Enable
            output wire         [7:0]               MemRdAddr,          // Read Address
            output reg          [7:0]               MemWrAddr,          // Write Address
            output wire                             MemRdColOut         // Corresponds to L1 column
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
//Bandwidth
localparam BW80 = 2'd2,
           BW40 = 2'd1,                      
           BW20 = 2'd0;

//Plateau fall counter max number
localparam TIMECOUNTPLATFALLMAX = 4'd9;

//TDFO Controller FSM
//Size
localparam ESTFSM_SIZE = 2;
localparam CMPFSM_SIZE = 3;

//FSM States
//Estimation FSM
localparam [ESTFSM_SIZE-1:0] ESTIDLE   = 2'd0,
                             ESTCOARSE = 2'd1,
                             ESTFINE   = 2'd2,
                             ESTDONE   = 2'd3;
//Compensation FSM
localparam [CMPFSM_SIZE-1:0] COMPIDLE           = 3'd0,
                             CALCOARSE          = 3'd1,
                             COMPWAITPF         = 3'd2,
                             COMPCOARSELTF      = 3'd3,
                             CALFINE            = 3'd4,
                             COMPCOARSEFINELTF  = 3'd5,
                             COMPWAITDATA       = 3'd6,
                             COMPCOARSEFINEDATA = 3'd7;

// Memory read latency
localparam MEM_RD_LATENCY = 2;

// Memory addresses adjustment
localparam [9:0] MEMADDROFFSET_2080 = 10'd29; 
localparam [9:0] MEMADDROFFSET_2040 = 10'd29;
localparam [9:0] MEMADDROFFSET_2020 = 10'd29;

localparam [9:0] MEMADDROFFSET_MAX_20 = 10'd127;

//General Param
localparam    signed   [ANGLEWIDTH - 1:0] CONST_ZERO_ANGLEWIDTH    = {{ANGLEWIDTH}{1'b0}};
localparam    signed   [ANGLEWIDTH    :0] CONST_ZERO_ANGLEWIDTH_P1 = {{ANGLEWIDTH + 1}{1'b0}};
localparam    signed   [ANGLEWIDTH + 3:0] CONST_ZERO_ANGLEWIDTH_P4 = {{ANGLEWIDTH + 4}{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                               CorrEnIntD;
wire                               CorrEnInt8D;
wire                               CorrEnInt9D;
wire                               MemRdEnOut;
wire signed [ANGLEWIDTH-2:0]       CoarseFOAngleRnd40;
wire signed [ANGLEWIDTH-3:0]       CoarseFOAngleRnd80;
wire signed [ANGLEWIDTH-3:0]       FineFOAngleRnd20;
wire signed [ANGLEWIDTH-4:0]       FineFOAngleRnd40;
wire signed [ANGLEWIDTH-5:0]       FineFOAngleRnd80;
wire        [CMPFSM_SIZE-1:0]      CmpCntrlCSD;
wire                               ByPass;
wire        [9:0]                  MemRdAddrOffset;
wire        [9:0]                  MemWrAddrSync;
wire        [8:0]                  TimeCountCoarse;
wire        [8:0]                  TimeCountAdjust;
wire        [9:0]                  hmemaddroffset_20_mux; 
wire signed [ANGLEWIDTH+3:0]       CRotAngleLTFShift;
wire signed [ANGLEWIDTH+3:0]       CRotAngleDataShift;
wire signed [ANGLEWIDTH:0]         CRotAngleInc;
wire signed [ANGLEWIDTH-10:0]      FOAngleInter;
wire signed [8:0]                  FOAngleInterSat;
wire        [3:0]                  TimeCountPlatFallLimit;
wire                               TimeCountPlatFallDone;
wire        [11:0]                 FDCpeSlopeCoef;
wire signed [23:0]                 FDCpeSlopeMultRnd;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                MemRdEnInt;
reg                                MemRdEnFinal;
reg                                MemRdCol;
reg        [8:0]                   TimeCount;
reg        [3:0]                   TimeCountPlatFall;
reg        [ESTFSM_SIZE-1:0]       EstCntrlCS;
reg        [ESTFSM_SIZE-1:0]       EstCntrlNS;
reg        [CMPFSM_SIZE-1:0]       CmpCntrlCS;
reg        [CMPFSM_SIZE-1:0]       CmpCntrlNS;
reg        [9:0]                   MemWrAddrInt;
reg        [9:0]                   MemWrAddrStop;
reg        [9:0]                   MemWrAddrIntPlatFall;
reg        [9:0]                   MemRdAddrInt;
reg        [7:0]                   MemRdAddrFinal;
reg        [9:0]                   MemRdStopEst;
reg        [9:0]                   MemRdStartCmp;
reg                                PlatFallLowFlag;
reg                                PlatFallHighFlag;
reg                                PlatThreshDoneFlag;
reg                                CorrEnInt;
reg        [8:0]                   CorrEnIntDel;
reg                                CorrFineEnd;
reg                                MemWrStopVal;
reg                                MemRdStartVal;
reg                                DataValidD1;
reg                                DataValidD2;
reg                                DataValidD3;
reg signed [ANGLEWIDTH:0]          CRotAngle;
reg                                CRot0DataOutValidD;
reg        [7:0]                   FFTMemAddrInt;
reg signed [ANGLEWIDTH+3:0]        CRot0AngleTemp;
reg signed [ANGLEWIDTH+3:0]        CRot1AngleTemp;
reg                                CRotAngleInitDone;
reg                                MovAvgClrInt;
reg                                FFTMemWrEnInt;
reg signed [ANGLEWIDTH-1:0]        CoarseFOAngleInter;
reg signed [ANGLEWIDTH-1:0]        FineFOAngleInter;
reg signed [ANGLEWIDTH-3:0]        FineFOAngleRnd20D1;
reg signed [23:0]                  FDCpeSlopeInc;
reg signed [33:0]                  FDCpeSlopeMult;

//String definition to display FSM states in simulation environment
`ifdef RW_SIMU_ON
reg        [20*8:0]                EstCntrlCSStr;
reg        [20*8:0]                EstCntrlNSStr;
reg        [20*8:0]                CmpCntrlCSStr;
reg        [20*8:0]                CmpCntrlNSStr;
`endif //RW_SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module is the controller for the Time Domain Frequency Offset
//Estimation & Compensation procedure
//The following phases are executed
//1. Coarse Offset Estimation
//2. Fine Offset Estimation
//3. Coarse Freq Compensation
//4. Fine+Coarse Freq Compensation
//This module also generates the necessary control signals for the
//Plateau Fall Detection module. The Plateau Fall point is used as the
//reference for triggering TBE.

//TDFO Coarse & Fine Estimation and Plateau Fall Detection
//--------------------------------------------------------

//Timing Counter
//Align start of Timing Counter
always @ (posedge PhyClk or negedge nPhyRst)
   begin: TimeCount_Blk
      if (nPhyRst == 1'b0)
         TimeCount <= 9'd0;
      else if (EstCntrlCS == ESTIDLE)
         TimeCount <= 9'd0;
      else if ((EstCntrlCS == ESTCOARSE) && DataValid)
            TimeCount <= TimeCount + 9'd1;
   end //TimeCount_Blk

//TDFO Estimation FSM
//Sequential Block
always @ (posedge PhyClk or negedge nPhyRst)
   begin: EstFSM_Seq
      if (nPhyRst == 1'b0)
         EstCntrlCS <= ESTIDLE;
      else if (!TDFOEn)
         EstCntrlCS <= ESTIDLE;
      else
         EstCntrlCS <= EstCntrlNS;
   end //EstFSM_Seq

//Combinatorial Block
always @ (*)
   begin: EstFSM_Combo
      case(EstCntrlCS)

         ESTIDLE: begin
            if (TDFOStrtP && !TDFOCmpDataMode)
               //From RxFSM 
               EstCntrlNS = ESTCOARSE;
            else
               EstCntrlNS = ESTIDLE;
         end //IDLE

         ESTCOARSE: begin
            if (CorrEnIntD && !CorrEnInt)
               EstCntrlNS = ESTFINE;
            else
               EstCntrlNS = ESTCOARSE;
         end

         ESTFINE: begin
            if (CorrEnInt9D && !CorrEnInt8D && MemRdStartVal)
               EstCntrlNS = ESTDONE;
            else
               EstCntrlNS = ESTFINE;
         end
         
         // ESTDONE is default
         default: EstCntrlNS = ESTIDLE;
         
      endcase
   end //EstFSM_Combo

//Plateau Fall Flags
//This flag indicates that Plateau Fall has happened.
//It is used by the FSM to transition to FINE state
//This transition has happened only after Plateau Fall has happened
//and 2.4 us worth of samples have been accumulated (whichever is later)
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallFlag_Blk
      if (nPhyRst == 1'b0) begin
         PlatFallLowFlag  <= 1'b0;
         PlatFallHighFlag <= 1'b0;
      end
      else if (EstCntrlCS == ESTCOARSE && MovAvgClrInt) begin
         //Reset Flag at the beginning of Coarse
         PlatFallLowFlag  <= 1'b0;
         PlatFallHighFlag <= 1'b0;
      end
      else begin
         if (PlatFallLowP)
            PlatFallLowFlag <= 1'b1;
         if (PlatFallHighP)
            PlatFallHighFlag <= 1'b1;
      end
   end //PlatFallFlag_Blk

assign ModApproxCompEn = ~PlatFallLowFlag;

//TDFOEstMode
assign TDFOEstMode = (EstCntrlCS == ESTFINE) ? 1'b1 : 1'b0;

//TDFOCmpMode
assign TDFOCmpMode = (CmpCntrlCS == COMPCOARSEFINELTF) ? 1'b1 : 1'b0;

//TDFOLTFCmpMode
assign TDFOLTFCmpMode = (CmpCntrlCS == COMPCOARSELTF || TimeCountPlatFallDone) ? 1'b1 : 1'b0;

//TDFOCmpDataMode
assign TDFOCmpDataMode = (CmpCntrlCS == COMPCOARSEFINEDATA) ? 1'b1 : 1'b0;

//Generate Control Signals
//CorrEn & CorrFineEnd
//This signal enables Correlation in TDFOCorrVal module
//It needs to be generated during Coarse as well as Fine mode
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrEnIntBlk
      if (nPhyRst == 1'b0) begin
         CorrEnInt   <= 1'b0;
         CorrFineEnd <= 1'b0;
      end
      else if (!TDFOEn) begin
         CorrEnInt   <= 1'b0;
         CorrFineEnd <= 1'b0;
      end
      else if (EstCntrlCS == ESTCOARSE) begin
         if ((TimeCount >= AutoCorrPhSumEnd) && PlatFallLowFlag)
            CorrEnInt <= 1'b0;
         else
            CorrEnInt <= 1'b1;
      end
      else if (EstCntrlCS == ESTFINE) begin
          if (MemRdAddrInt == 10'd64 && !CorrEnInt && !CorrFineEnd) // $todo : add counter offset from register
            CorrEnInt <= 1'b1;
          else if ((MemRdAddrInt == MemRdStopEst) && MemWrStopVal && CorrEnInt) begin
            CorrEnInt   <= 1'b0;
            CorrFineEnd <= 1'b1;
          end
      end
      else begin
         CorrEnInt   <= 1'b0;
         CorrFineEnd <= 1'b0;
      end
   end //CorrEnIntBlk

//CorrEn needs to take care of the hardware delays
always @ (posedge PhyClk or negedge nPhyRst)
   begin:CorrEnBlk
      if (nPhyRst == 1'b0)
         CorrEn <= 1'b0;
      else if (!TDFOEn)
         CorrEn <= 1'b0;
      else
         CorrEn <= CorrEnInt | CorrEnInt8D;
   end //CorrEnBlk

//MovAvgEn
//This signal enables the Moving Average Block in TDFOCorrVal module.
//In case of COARSE, this needs to be enabled 16 samples after CorrEn
//for 20 MHz. 
//Further, hardware delay in obtaining CorrValSum in TDFOCorrVal module
//needs to be taken care. (9 clocks delay)
//For FINE TDFO, MovAvgEn is a 6 clock delayed version of CorrEn
//Also it needs to take care of the 3 clock delay in the Moving Average Block
//itself.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MovAvgEn_Blk
      if (nPhyRst == 1'b0) begin
         MovAvgEn <= 1'b0;
      end
      else if (EstCntrlCS == ESTCOARSE) begin
         if (TimeCount >= TimeCountCoarse)
              MovAvgEn <= 1'b1;
         else
            MovAvgEn <= 1'b0;
      end
      else begin //EstCntrlCS = ESTFINE
         MovAvgEn <= CorrEnInt8D;
      end
    end //MovAvgEn_Blk

//16 + 3 = 19 in 20/40/80 MHz
assign TimeCountCoarse = 9'd19;

//MovAvgClr
//The Moving Average accumulator has to be cleared before Coarse and Fine
//to ensure that previously accumulated values do not impact the new
//estimation.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MovAvgClr_Blk
      if (nPhyRst == 1'b0)
         MovAvgClrInt <= 1'b0;
      else if (CorrEnInt && !CorrEnIntD)
         MovAvgClrInt <= 1'b1;
      else
         MovAvgClrInt <= 1'b0;
   end //MovAvgClr_Blk

assign MovAvgClr = MovAvgClrInt;

//BYPASS DISABLED
assign ByPass = 1'b0;

//Generate delayed versions of CorrEnInt
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrEnDel_Blk
      if (nPhyRst == 1'b0)
         CorrEnIntDel <= 9'b0;
      else if (!TDFOEn)
         CorrEnIntDel <= 9'b0;
      else
         CorrEnIntDel[8:0] <= {CorrEnIntDel[7:0],CorrEnInt};
   end //CorrEnDel_Blk

assign CorrEnIntD   = CorrEnIntDel[0];
assign CorrEnInt8D  = CorrEnIntDel[7];
assign CorrEnInt9D  = CorrEnIntDel[8];

//CorrValAccEn
//Enable 2.4 us of accumulation of CorrVal
//This will be used as the input to the CordicVect, for calculating the angle
//of rotation in Coarse TDFO
//Hardware delays in the CorrVal module need to be taken into consideration
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrValAccEn_Blk
      if (nPhyRst == 1'b0)
         CorrValAccEn <= 1'b0;
      else if (EstCntrlCS == ESTCOARSE && 
               TimeCount >  (AutoCorrPhSumStart + TimeCountAdjust) &&
               TimeCount <= (AutoCorrPhSumEnd   + TimeCountAdjust) &&
               !PlatFallLowFlag) begin
               CorrValAccEn <= 1'b1;
      end
      else
         CorrValAccEn <= 1'b0;
   end //CorrValAccEn_Blk

// 1 in 20 MHz (DC Offset latency)
// 2 in 40 MHz (no DC Offset latency)
// 3 in 80 MHz (no DC Offset latency)
assign TimeCountAdjust = (Force20)  ? 9'd1 : 
                         (!Force40) ? 9'd3 : 9'd2;

//AbsCValAccEn
//This enables accumulation of the absolute value for a period of 0.8 us
always @ (posedge PhyClk or negedge nPhyRst)
   begin: AbsCValAccEn_Blk
      if (nPhyRst == 1'b0)
         AbsCValAccEn <= 1'b0;
      else if ((EstCntrlCS == ESTCOARSE) && 
               (TimeCount >  AutoCorrPlatSumStart) &&
               (TimeCount <= AutoCorrPlatSumEnd)) begin
               AbsCValAccEn <= 1'b1;
      end
      else
         AbsCValAccEn <= 1'b0;
   end //AbsCValAccEn_Blk

//PlatThreshDoneFlag
//Indicates that Plateau Fall Threshold has been calculated by PlatDet module
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatThreshDoneFlag_Blk
      if (nPhyRst == 1'b0)
         PlatThreshDoneFlag <= 1'b0;
      else if (EstCntrlCS == ESTCOARSE && MovAvgClrInt)
         //Clear the flag at the beginning of every packet
         PlatThreshDoneFlag <= 1'b0;
      else if (PlatThreshDoneP)
         PlatThreshDoneFlag <= 1'b1;
   end //PlatThreshDoneFlag_Blk

//PlatFallLowDetEn
//This enables plateau detection low window
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallLowDetEn_Blk
      if (nPhyRst == 1'b0)
         PlatFallLowDetEn <= 1'b0;
      else if ((EstCntrlCS == ESTCOARSE) && (!PlatFallLowFlag) &&
               (TimeCount >= (AutoCorrPlatFallSearchStart + 9'd2)) && PlatThreshDoneFlag)
         PlatFallLowDetEn <= 1'b1;
      else
         PlatFallLowDetEn <= 1'b0;
   end //PlatFallLowDetEn_Blk

//PlatFallHighDetEn
//This enables plateau detection high window
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallHighDetEn_Blk
      if (nPhyRst == 1'b0)
         PlatFallHighDetEn <= 1'b0;
      else if ((EstCntrlCS == ESTCOARSE) && (!PlatFallHighFlag) &&
               (TimeCount >= (AutoCorrPlatFallSearchStart + 9'd2)) && PlatThreshDoneFlag)
         PlatFallHighDetEn <= 1'b1;
      else
         PlatFallHighDetEn <= 1'b0;
   end //PlatFallHighDetEn_Blk

//TDFO Coarse and Fine Compensation
//--------------------------------------------------------
//This logic controls the compensation part. This includes:
//1. Calculating Angle using CordicVect for both Coarse and Fine (from
//CorrValSum).
//2. Compensation with Coarse and then (Coarse + Fine) as and when they become
//available.
//TDFO Compensation FSM
//Sequential Block
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CmpFSM_Seq
      if (nPhyRst == 1'b0)
         CmpCntrlCS <= COMPIDLE;
      else if (!TDFOEn) 
         CmpCntrlCS <= COMPIDLE;
      else
         CmpCntrlCS <= CmpCntrlNS;
   end //CmpFSM_Seq

//Combinatorial Block
always @ (*)
   begin: CmpFSM_Combo
      case(CmpCntrlCS)
         COMPIDLE: begin
            if (CorrValAccValidP || PlatFallHighP) //From TDFOPlatDet 
               CmpCntrlNS = CALCOARSE;
            else
               CmpCntrlNS = COMPIDLE;
         end

         CALCOARSE: begin
            if (CVectDone) begin //From CordicVectTop
               if (TimeCountPlatFallDone)
                  CmpCntrlNS = COMPCOARSELTF;
               else
                  CmpCntrlNS = COMPWAITPF;
            end
            else
               CmpCntrlNS = CALCOARSE;
         end

         COMPWAITPF: begin
            if (TimeCountPlatFallDone)
               CmpCntrlNS = COMPCOARSELTF;
            else
               CmpCntrlNS = COMPWAITPF;
         end

         COMPCOARSELTF: begin
            if (EstCntrlCS == ESTDONE)
               CmpCntrlNS = CALFINE;
            else
               CmpCntrlNS = COMPCOARSELTF;
         end

         CALFINE: begin
            if (CVectDone)
               CmpCntrlNS = COMPCOARSEFINELTF;
            else
               CmpCntrlNS = CALFINE;
         end

         COMPCOARSEFINELTF: begin
            if ((FFTMemAddrInt == 8'd63) && FFTMemWrEnInt)
               CmpCntrlNS = COMPWAITDATA;
            else
               CmpCntrlNS = COMPCOARSEFINELTF;
         end

         COMPWAITDATA: begin
            if (DataCmpStrtP) 
               CmpCntrlNS = COMPCOARSEFINEDATA;
            else
               CmpCntrlNS = COMPWAITDATA;
         end

         COMPCOARSEFINEDATA: begin
           CmpCntrlNS = COMPCOARSEFINEDATA;
         end

         //pragma coverage off
         default: CmpCntrlNS = COMPIDLE;
         //pragma coverage on

      endcase
   end //CmpFSM_Combo

assign CVectOngoing = (CmpCntrlCS == CALCOARSE);

//Control Signals for Compensation
//CVectDataEnIn
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CVectDataEnIn_Blk
      if (nPhyRst == 1'b0)
         CVectDataEnIn <= 1'b0;
      else if ((CmpCntrlCS == CALCOARSE && CmpCntrlCSD == COMPIDLE) ||
               (CmpCntrlCS == CALFINE && CmpCntrlCSD == COMPCOARSELTF))
         CVectDataEnIn <= 1'b1;
      else
         CVectDataEnIn <= 1'b0;
   end //CVectDataEnIn_Blk

//CRotEnable
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRotEnable_Blk
      if (nPhyRst == 1'b0)
         CRotEnable <= 1'b0;
      else if (!TDFOEn)
         CRotEnable <= 1'b0;
      else if (TDFOStrtP)
         CRotEnable <= 1'b1;
   end //CRotEnable_Blk

//CoarseFOAngle
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CoarseFOAngle_Blk
      if (nPhyRst == 1'b0)
         CoarseFOAngleInter <= CONST_ZERO_ANGLEWIDTH;
      else if (CmpCntrlCS == CALCOARSE && CVectDone)
         CoarseFOAngleInter <= -CVAngleOut;
   end //CoarseFOAngle_Blk

//FineFOAngleInter
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FineFOAngle_Blk
      if (nPhyRst == 1'b0)
         FineFOAngleInter <= CONST_ZERO_ANGLEWIDTH;
      else if (TDFOFineEstEn == 1'b0)
         FineFOAngleInter <= CONST_ZERO_ANGLEWIDTH;
      else if (CmpCntrlCS == CALFINE && CVectDone)
         FineFOAngleInter <= -CVAngleOut;
   end //FineFOAngle_Blk

//FineFOAngleRnd20 delayed by 1cc for timing cut
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FineFOAngleRnd20_Blk
      if (nPhyRst == 1'b0)
         FineFOAngleRnd20D1 <= {ANGLEWIDTH-2{1'b0}};
      else
         FineFOAngleRnd20D1 <= FineFOAngleRnd20;
   end //FineFOAngleRnd20_Blk


// TBE Control
//--------------------------------------------------------

//Timing Counter for plateau fall
//Align start of TBE with plateau fall generation
always @ (posedge PhyClk or negedge nPhyRst)
   begin: TimeCountPlatFall_Blk
      if (nPhyRst == 1'b0)
         TimeCountPlatFall <= 4'd0;
      else if (EstCntrlCS == ESTIDLE)
         TimeCountPlatFall <= 4'd0;
      else if (PlatFallLowP == 1'b1)
         TimeCountPlatFall <= 4'd1;
      else if ((TimeCountPlatFall < TIMECOUNTPLATFALLMAX) && (TimeCountPlatFall > 4'd0) && (DataValid == 1'b1))
            TimeCountPlatFall <= TimeCountPlatFall + 4'd1;
   end //TimeCountPlatFall_Blk

assign RxLTFValid = (TDFOLTFCmpMode == 1'b1) ? CRot0DataOutValidD :
                                               DataValidD3 & ~PlatFallLowP & ~PlatFallLowFlag;

// Adjust TimeCountPlatFall from register
assign TimeCountPlatFallLimit = TIMECOUNTPLATFALLMAX - AutoCorrPlatFallIndexOffset;

assign TimeCountPlatFallDone = (TimeCountPlatFall >= TimeCountPlatFallLimit);

// Memory Control
//--------------------------------------------------------

//The Memory is used for storing long preamble samples.
//The incoming samples are continously stored, and once Fine Estimation
//starts, L1 samples are getting out one on every clock from TBE delay line.
//Note that during estimation, we get out 3.2 us worth
//of samples starting with (TBECount-2.4us).
//For compensation we need to read out L1 and L2 from memory 
//(starting from TBECount - 1.6 us).
//Port 0 is used for reading and Port 1 for writing

//Write Address 
//A 10 bit counter is used so that looping back and arithmetic operations
//on the address are simplified. The external address will still be
//6/7/8 bits depending on 20/40/80 MHz mode respectively.
//Note that the DataValid signal is used to increment the 
//address.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MemWrAddrInt_Blk
      if (nPhyRst == 1'b0)
         MemWrAddrInt <= 10'd0;
      else if ((CmpCntrlCS == COMPIDLE) && (EstCntrlCS == ESTIDLE))
         MemWrAddrInt <= 10'd0;
      else if (DataValid == 1'b1) begin
         if (MemWrAddrInt == (MemWrAddrStop + 10'd1) && MemWrStopVal == 1'b1)
            MemWrAddrInt <= MemWrAddrInt;
         else
            MemWrAddrInt <= MemWrAddrInt + 10'd1;
      end
   end //MemWrAddrInt_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: MemAddrBlk
      if (nPhyRst == 1'b0)
         MemWrAddr <= 8'd0;  
      else
         //20 MHz Packet
         MemWrAddr <= {1'b0, 1'b1, MemWrAddrInt[5:0]};
   end //MemAddrBlk

//Write Enable
//This is for one Rx Chain. It will be expanded to the required number of Rx
//Chains at the top level.
//Samples are only coming in once in 3 or 6 clocks, so the delayed version of
//CRot0DataOutValid can be used as Write Enable
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MemWrEn_Blk
      if (nPhyRst == 1'b0) begin
         MemWrEn  <= 1'b0;
         MemWrSel <= 1'b0;
      end
      else if (((CmpCntrlCS == COMPIDLE) && (EstCntrlCS == ESTIDLE)) ||
               ((MemWrAddrInt > MemWrAddrStop) && (MemWrStopVal == 1'b1))) begin
         MemWrEn  <= 1'b0;
         MemWrSel <= 1'b0;
      end
      else begin
         MemWrEn <= DataValidD1;
         //20 MHz Packet
         if (MemWrAddrInt[6] == 1'b1)
            //Use Second Column
            MemWrSel <= 1'b1;
         else
            //Use First Column
            MemWrSel <= 1'b0;
      end
   end //MemWrEn_Blk

//MemWrAddrStop & MemWrStopVal
//This indicates the address at which the MemWrAddrInt should stop
//incrementing. This ensures that we have L1 and L2 in Memory which
//can be used for compensation.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: WrAddrStop_Blk
      if (nPhyRst == 1'b0) begin
         MemWrAddrStop <= 10'd0;
         MemWrStopVal  <=  1'b0;
      end
      else if (EstCntrlCS == ESTCOARSE && MovAvgClrInt == 1'b1) begin
         //Reset at the beginning of Coarse
         MemWrAddrStop <= 10'd0;
         MemWrStopVal  <=  1'b0;
      end
      else if (TBECountValid) begin
         MemWrStopVal <= 1'b1;
         //20 MHz Packet
         MemWrAddrStop <= MemWrAddrSync + MEMADDROFFSET_MAX_20 - hmemaddroffset_20_mux;
      end
   end //WrAddrStop_Blk

//Adjust memory index with TBE counter
assign MemWrAddrSync = MemWrAddrIntPlatFall + {2'b0,TBECount};

//Adjust memory index
assign hmemaddroffset_20_mux = (Force20 == 1'b0 && Force40 == 1'b0) ? MEMADDROFFSET_2080 :
                               (Force20 == 1'b0)                    ? MEMADDROFFSET_2040 : MEMADDROFFSET_2020;

//MemWrAddrIntPlatFall
//This indicates the address at which plateau fall has occured.
//When PlatFallP Flag is set from PlatDet, the corresponding MemWrAddrInt
//represents the MemWrAddrIntPlatFall address.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: HMA1IntPF_Blk
      if (nPhyRst == 1'b0)
         MemWrAddrIntPlatFall <= 10'd0;
      else if (EstCntrlCS == ESTCOARSE && MovAvgClrInt == 1'b1)
         //Reset at the beginning of Coarse
         MemWrAddrIntPlatFall <= 10'd0;
      else if (TimeCountPlatFall == 4'd2 && DataValid) begin
         // Store minus 2 as final address is delayed by 1 cc (MemWrAddr)
         // and 1 cc for Bandwidth generation after PlatFallP
         MemWrAddrIntPlatFall <= MemWrAddrInt - 10'd3 - {6'b0,AutoCorrPlatFallIndexOffset};
      end
   end // HMA1IntPF_Blk

//Read Address and Read Enable
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MemRdAddrInt_Blk
      if (nPhyRst == 1'b0) begin
         MemRdAddrInt     <= 10'd0;
         MemRdEnInt       <= 1'b0;
         CorValFineDataEn <= 1'b0;
      end
      else if (TDFOLTFCmpMode) begin
         MemRdEnInt <= 1'b0;
         //TDFO Fine Estimation
         //MemRdAddrInt is used as counter only for fine estimation
         if (CorrEnInt == 1'b0 && MemRdAddrInt == 10'd64 && CorrFineEnd == 1'b0) begin
            //Re-initialize when delay line 64 is reached
            MemRdAddrInt    <= 10'd0;
         end
         else if (MemRdStartVal && (MemRdAddrInt == MemRdStopEst) && CorrFineEnd) begin
            //Last Address Completed
            MemRdAddrInt     <= MemRdAddrInt;
            CorValFineDataEn <= 1'b0;
         end
         else if (CRot0DataOutValid == 1'b1) begin
            CorValFineDataEn <= CorrEnInt & MemRdStartVal;
            MemRdAddrInt     <= MemRdAddrInt + 10'd1;
         end
         else
           CorValFineDataEn <= 1'b0;
      end //ESTFINE
      else if (CmpCntrlCS == CALFINE) begin
         //Initialize to first address
         MemRdAddrInt <= MemRdStartCmp;
         MemRdEnInt   <= 1'b0;
      end //CALFINE
      else if (CmpCntrlCS == COMPCOARSEFINELTF) begin
         //TDFO Coarse + Fine Compensation of L1 & L2
         if ((MemRdAddrOffset < MemWrAddrInt) && 
               ((MemWrAddrInt > MemWrAddrStop) && MemWrStopVal)) begin
            //Increment but dont overshoot write address
            //Note that L1 and L2 are read together and hence the offset
            if (MemRdEnInt)
              MemRdAddrInt <= MemRdAddrInt + 10'd1;
            MemRdEnInt   <= ~MemRdEnInt | Force20;
         end
         else begin
            MemRdEnInt <= 1'b0;
         end
      end //COMPCOARSEFINELTF
      else begin
         MemRdAddrInt <= 10'd0;
         MemRdEnInt   <= 1'b0;
      end
   end //MemRdAddrInt_Blk

//Read Address Offset limit
assign MemRdAddrOffset = 10'd64 + MemRdAddrInt + {9'b0,Force20};

always @ (posedge PhyClk or negedge nPhyRst)
   begin: Mem0Final_Blk
      if (nPhyRst == 1'b0) begin
         MemRdAddrFinal <= 8'd0;
         MemRdEnFinal   <= 1'b0;
      end
      else if (CmpCntrlCS == COMPCOARSEFINELTF) begin
         MemRdEnFinal   <= MemRdEnInt;
         MemRdAddrFinal <= {1'b0,1'b1, MemRdAddrInt[5:0]}; // Add offset avoiding memory overlap
      end
      else
         MemRdEnFinal <= 1'b0;
   end //Mem0Final_Blk

assign MemRdAddr = MemRdAddrFinal;
assign MemRdEn   = MemRdEnFinal;

//MemRdStopEst, MemRdStartCmp, MemRdStartVal
//These signals indicate the starting address during Estimation and
//Compensation phases of TDFO Fine mode.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MemRdStart_Blk
      if (nPhyRst == 1'b0) begin
         MemRdStopEst   <= 10'd0;
         MemRdStartCmp  <= 10'd0;
         MemRdStartVal  <=  1'd0;
      end
      else if (EstCntrlCS == ESTCOARSE && MovAvgClrInt == 1'b1) begin
         //Reset at the beginning of Coarse
         MemRdStopEst   <= 10'd0;
         MemRdStartCmp  <= 10'd0;
         MemRdStartVal  <=  1'd0;
      end
      else if (TBECountValid) begin
         MemRdStartVal <=  1'd1;
         // Start compensation memory index
         MemRdStartCmp <= MemWrAddrSync - hmemaddroffset_20_mux;
         
         if (TBECount < 8'd45)
            // Stop estimation index : 64-(44-TBECount)=20+TBECount (+ 1 for data completion)
            MemRdStopEst <= {2'b0,TBECount} + 10'd20;
         else
            // Stop estimation index : 0 to 63 (+ 1 for data completion)
            MemRdStopEst <= 10'd64;
      end
   end //MemRdStart_Blk

//MemRdCol
//This signal indicates which column contains L1
//Although both columns are read together (L1 and L2), it is necessary to know
//which column contains L1 for the current address. This is because we need to
//conjugate L1 alone before sending it to TDFOCorrVal
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MemRdCol_Blk
      if (nPhyRst == 1'b0)
         MemRdCol <= 1'd0;  
      else if (CmpCntrlCS == COMPCOARSEFINELTF)
         MemRdCol <= MemRdAddrInt[6];
      else
         MemRdCol <= 1'd0;  
   end //MemRdCol_Blk

//Generate delayed version of DataValid and CRot0OutDataValid
always @ (posedge PhyClk or negedge nPhyRst)
   begin: DataValidD_Blk
      if (nPhyRst == 1'b0) begin
         DataValidD1        <= 1'b0;
         DataValidD2        <= 1'b0;
         DataValidD3        <= 1'b0;
         CRot0DataOutValidD <= 1'b0;
      end
      else begin
         DataValidD1        <= DataValid;
         DataValidD2        <= DataValidD1;
         DataValidD3        <= DataValidD2;
         CRot0DataOutValidD <= CRot0DataOutValid;
      end
   end //DataValidD_Blk

//FFT Memory Control
//After Fine compensation of L1 and L2, they will be averaged and written in
//to the FFT Memory. To increment the address, the CRot01DataOutValid signal
//will be used. The Data has to be delayed by a clock in the top level to
//align it with the Address and WrEn.
//FFT Memory Address and Write Enable 
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FFTMemAddr_Blk
      if (nPhyRst == 1'b0) begin
         FFTMemAddrInt <= 8'b0;
         FFTMemWrEnInt <= 1'b0;
      end
      else if (CmpCntrlCS == CALFINE && CVectDone == 1'b1) begin
         //Initiliaze FFT Addr when entering COMPCOARSEFINELTF
         FFTMemAddrInt <= {8{1'b1}};
         FFTMemWrEnInt <= 1'b0;
      end
      else if (CmpCntrlCS == COMPCOARSEFINELTF) begin
         FFTMemWrEnInt <= CRot01DataOutValid;
         if (CRot01DataOutValid == 1'b1)
            FFTMemAddrInt <= FFTMemAddrInt + 8'b1;
      end
      else if (CmpCntrlCS == COMPCOARSEFINEDATA) begin
         FFTMemAddrInt <= 8'd0; //This will be generated by RxFSM for Data
         FFTMemWrEnInt <= CRot0DataOutValid; //Output for Data is from CRot0
      end
      else begin
         FFTMemAddrInt <= 8'b0;
         FFTMemWrEnInt <= 1'b0;
      end
   end //FFTMemAddr_Blk

assign FFTMemWrEn = FFTMemWrEnInt;

// Select memory control either from TDFO or Rx TD FSM
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FFTMemSel_Blk
      if (nPhyRst == 1'b0)
         FFTMemSel <= 1'b0;
      else if (CmpCntrlCS == COMPIDLE) 
         FFTMemSel <= 1'b0;
      else if ((CmpCntrlCS == COMPCOARSEFINELTF) && MemRdEnInt)
         FFTMemSel <= 1'b1;
   end //FFTMemSel_Blk

//Cordic Angle for Rotation
//During compensation, the angle (coarse+fine) needs to be multiplied with
//a linearly increasing sample index.
//Generate the Cordic Angle depending on the Compensation Mode
//There are 2 Cordics being used
//CordicRot0 will do Coarse compensation till end of LTF, 
//                do Coarse + Fine compensation on LTF and then 
//                do Coarse + Fine compensation on Data
//Cordic1 will do Coarse + Fine compensation during LTF
//CRot0Angle
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CordicRot0Angle_Blk
      if (nPhyRst == 1'b0)begin
         CRot0AngleTemp    <= CONST_ZERO_ANGLEWIDTH_P4;
         CRotAngleInitDone <= 1'b0;
      end
      else if (CmpCntrlCS == COMPIDLE || CmpCntrlCS == CALCOARSE) begin
         CRot0AngleTemp    <= CONST_ZERO_ANGLEWIDTH_P4;
         CRotAngleInitDone <= 1'b0;
      end
      else if ((CmpCntrlCS == COMPCOARSELTF || CmpCntrlCS == CALFINE) && DataValidD1)
         CRot0AngleTemp <= CRot0AngleTemp + $signed({{3{CRotAngle[ANGLEWIDTH]}},CRotAngle});
      else if (CmpCntrlCS == COMPCOARSEFINELTF) begin
         if (MemRdEnFinal && !MemRdEnOut && !CRotAngleInitDone) begin
            CRot0AngleTemp    <= CRotAngleLTFShift;
            CRotAngleInitDone <= 1'b1;
         end
         else if (MemRdEnOut)
            CRot0AngleTemp <= CRot0AngleTemp + $signed({{3{CRotAngle[ANGLEWIDTH]}},CRotAngle});
      end
      else if (CmpCntrlCS == COMPCOARSEFINEDATA) begin
         if (CmpCntrlCSD == COMPWAITDATA)
            CRot0AngleTemp <= CRot0AngleTemp + CRotAngleDataShift + $signed({{3{CRotAngleInc[ANGLEWIDTH]}},CRotAngleInc});
         else if (DataCmpStrtP) begin // Re-init at the end of HT_STF
            CRot0AngleTemp <= CONST_ZERO_ANGLEWIDTH_P4;
         end
         else if (DataValidD1)
            CRot0AngleTemp <= CRot0AngleTemp + $signed({{3{CRotAngle[ANGLEWIDTH]}},CRotAngle}) 
                                             - $signed({FDCpeSlopeInc[23],FDCpeSlopeInc});
      end
   end //CordicRot0Angle_Blk

assign CRotAngleLTFShift  = CRotAngle <<< 3'd6;
assign CRotAngleDataShift = CRotAngle <<< 3'd4; // Shift by 16 samples
assign CRotAngleInc       = (DataValidD1) ? CRotAngle : {ANGLEWIDTH+1{1'b0}};

assign CRot0Angle = (TDFOCompEn) ? CRot0AngleTemp : CONST_ZERO_ANGLEWIDTH_P4;

//CRot1AngleFine
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CordicRot1Angle_Blk
      if (nPhyRst == 1'b0)
         CRot1AngleTemp <= CONST_ZERO_ANGLEWIDTH_P4;
      else if (CmpCntrlCS == COMPCOARSEFINELTF) begin
         if (MemRdEnFinal && !MemRdEnOut && !CRotAngleInitDone)
            CRot1AngleTemp <= CONST_ZERO_ANGLEWIDTH_P4;
         else if (MemRdEnOut)
            CRot1AngleTemp <= CRot1AngleTemp + $signed({{3{CRotAngle[ANGLEWIDTH]}},CRotAngle});
      end
   end //CordicRot1Angle_Blk

assign CRot1Angle = (TDFOCompEn) ? CRot1AngleTemp : CONST_ZERO_ANGLEWIDTH_P4;

//FDO Cpe slope shifting
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FDCpeSlopeMult_Blk
      if (nPhyRst == 1'b0)
         FDCpeSlopeMult <= 34'b0;
      else
         FDCpeSlopeMult <= $signed(FDCpeSlope) * $signed({1'b0,FDCpeSlopeCoef});
   end //FDCpeSlopeMult_Blk

assign FDCpeSlopeCoef = (Force20 == 1'b0 && Force40 == 1'b0 && (Bandwidth == BW80)) ? 12'd819  :
                        (Force20 == 1'b0 && (Bandwidth == BW40))                    ? 12'd1639 :
                                                                                      12'd3277;

Round #(
        .INPUT_WIDTH(34),
        .OUTPUT_WIDTH(24)
       )
        U_ROUNDCPESLOPE(
                    .InputData(FDCpeSlopeMult),
                    .RoundData(FDCpeSlopeMultRnd)
                   );

always @ (posedge PhyClk or negedge nPhyRst)
   begin: FDCpeSlopeInc_Blk
      if (nPhyRst == 1'b0)
         FDCpeSlopeInc <= 24'b0;
      else if (TDFOStrtP)
         FDCpeSlopeInc <= 24'b0;
      else if ((CmpCntrlCS == COMPCOARSEFINEDATA) && DataCmpStrtP && TDFOCpeSlopeEn)
         // Capture CPE slope from FD
         FDCpeSlopeInc <= FDCpeSlopeMultRnd;
   end //FDCpeSlopeInc_Blk


//Generate the sum of coarse and fine angles.
//The Fine FO Angle needs to be rounded by 2 in case of 20 MHz,
//3 in case of 40 MHz and 4 in case of 80 MHz.
//The Coarse FO Angle needs to be rounded by 1 in case of 40 MHz
//mode, 2 in case of 80 MHz mode and there is no rouding in 20 MHz mode.
//So 5 rounding blocks are required
Round #(
        .INPUT_WIDTH(ANGLEWIDTH),
        .OUTPUT_WIDTH(ANGLEWIDTH-1)
       )
        U_ROUNDANGCOARSE40(
                    .InputData(CoarseFOAngleInter),
                    .RoundData(CoarseFOAngleRnd40)
                   );

Round #(
        .INPUT_WIDTH(ANGLEWIDTH),
        .OUTPUT_WIDTH(ANGLEWIDTH-2)
       )
        U_ROUNDANGCOARSE80(
                    .InputData(CoarseFOAngleInter),
                    .RoundData(CoarseFOAngleRnd80)
                   );

Round #(
        .INPUT_WIDTH(ANGLEWIDTH),
        .OUTPUT_WIDTH(ANGLEWIDTH-2)
       )
        U_ROUNDANGFINE20(
                    .InputData(FineFOAngleInter),
                    .RoundData(FineFOAngleRnd20)
                   );

Round #(
        .INPUT_WIDTH(ANGLEWIDTH-2),
        .OUTPUT_WIDTH(ANGLEWIDTH-3)
       )
        U_ROUNDANGFINE40(
                    .InputData(FineFOAngleRnd20D1),
                    .RoundData(FineFOAngleRnd40)
                   );

Round #(
        .INPUT_WIDTH(ANGLEWIDTH-2),
        .OUTPUT_WIDTH(ANGLEWIDTH-4)
       )
        U_ROUNDANGFINE80(
                    .InputData(FineFOAngleRnd20D1),
                    .RoundData(FineFOAngleRnd80)
                   );

//Calculate the total angle by muxing the rounded outputs depending on the
//Bandwidth
//Cordic 0 is used for Coarse Comp of LTF, Coarse+Fine Comp of LTF and Data
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRotAngle_Blk
      if (nPhyRst == 1'b0)
         CRotAngle <= CONST_ZERO_ANGLEWIDTH_P1;
      else if ((Force20 == 1'b0) && (Force40 == 1'b0) && (Bandwidth == BW80) && 
               (CmpCntrlCS == COMPCOARSEFINEDATA)) begin
         //80 MHz Packet
         CRotAngle <= $signed({{1{CoarseFOAngleRnd80[ANGLEWIDTH-3]}},CoarseFOAngleRnd80}) + 
                      $signed({{3{FineFOAngleRnd80[ANGLEWIDTH-5]}},FineFOAngleRnd80});
      end
      else if (Force20 == 1'b0 && (Bandwidth == BW40)) begin
         //40 MHz Packet
         if ((CmpCntrlCS == COMPCOARSEFINELTF && CmpCntrlCSD == COMPCOARSEFINELTF) ||
             CmpCntrlCS == COMPCOARSEFINEDATA || CmpCntrlCS == COMPWAITDATA)
            CRotAngle <= CoarseFOAngleRnd40 + $signed({{3{FineFOAngleRnd40[ANGLEWIDTH-4]}},FineFOAngleRnd40});
         else
            CRotAngle <= -CVAngleOut;
      end
      else begin

         if ((CmpCntrlCS == COMPCOARSEFINELTF && CmpCntrlCSD == COMPCOARSEFINELTF) ||
              CmpCntrlCS == COMPCOARSEFINEDATA || CmpCntrlCS == COMPWAITDATA)
            CRotAngle <= CoarseFOAngleInter + $signed({{3{FineFOAngleRnd20D1[ANGLEWIDTH-3]}},FineFOAngleRnd20D1});
         else
            CRotAngle <= -CVAngleOut;
      end
   end //CRotAngle_Blk

//Delayed version of CmpCntrlCS
DelayLine # (
             .DATAWIDTH(CMPFSM_SIZE),  //Width of CmpCntrlCS
             .LATENCY(1)               //1 clock latency
             )
              U_DelayLineR (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(CmpCntrlCS),
                            .DataOut(CmpCntrlCSD)
                            );

//Delayed Version of MemRdEn
DelayLine # (
             .DATAWIDTH(1),            // Width of MemRdEnFinal
             .LATENCY(MEM_RD_LATENCY)  // Clock latency
             )
              U_DelayLine_MemRdEn (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(MemRdEnFinal),
                            .DataOut(MemRdEnOut)
                            );

//Delayed Version of MemRdEn
DelayLine # (
             .DATAWIDTH(1),            // Width of MemRdCol
             .LATENCY(MEM_RD_LATENCY)  // Clock latency
             )
              U_DelayLine_MemRdColOut (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(MemRdCol),
                            .DataOut(MemRdColOut)
                            );

//DataValid for CordicRot 0
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRot0DataInValid_Blk
      if (nPhyRst == 1'b0)
         CRot0DataInValid <= 1'b0;
      else if (CmpCntrlCS == COMPCOARSEFINELTF)
         CRot0DataInValid <= MemRdEnOut;
      else if ((CmpCntrlCS == COMPCOARSELTF) || (CmpCntrlCS == COMPCOARSEFINEDATA) ||
              ((CmpCntrlCS == CALCOARSE) && (TimeCountPlatFall > TimeCountPlatFallLimit)))
         CRot0DataInValid <= DataValidD1;
      else
         CRot0DataInValid <= 1'b0;
   end //CRot0DataInValid_Blk

//DataValid for CordicRot 1 and 2
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRot1DataInValid_Blk
      if (nPhyRst == 1'b0) 
         CRot1DataInValid <= 1'b0;
      else if (CmpCntrlCS == COMPCOARSEFINELTF)
         CRot1DataInValid <= MemRdEnOut;
      else
         CRot1DataInValid <= 1'b0;
   end //CRot1DataInValid_Blk

//CmpLTFDoneP - Indicates that LTF Fine Compensation is over
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CmpLTFDoneP_Blk
      if (nPhyRst == 1'b0)
         CmpLTFDoneP <= 1'b0;
      else if (CmpCntrlCS == COMPCOARSEFINELTF && CmpCntrlNS == COMPWAITDATA)
         CmpLTFDoneP <= 1'b1;
      else
         CmpLTFDoneP <= 1'b0;
   end //CmpLTFDoneP_Blk

//FineFODone - Indicates that Fine FO estimation is over
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FineFODone_Blk
      if (nPhyRst == 1'b0)
         FineFODone <= 1'b0;
      else if (!TDFOEn)
         FineFODone <= 1'b0;
      else if (CmpCntrlCS == CALFINE && CmpCntrlNS == COMPCOARSEFINELTF)
         FineFODone <= 1'b1;
   end //FineFODone_Blk

//Status registers outputs
assign CoarseFOAngle = CoarseFOAngleInter;
assign FineFOAngle   = $signed({{2{FineFOAngleRnd20D1[ANGLEWIDTH-3]}},FineFOAngleRnd20D1});

//FO output to AGC after 10-bit floor + 9-bit saturation
assign FOAngleInter = $signed(CoarseFOAngle[ANGLEWIDTH-1:10]) + $signed(FineFOAngle[ANGLEWIDTH-1:10]);

SatSigned #(
   .INPUT_WIDTH(ANGLEWIDTH-9),
   .OUTPUT_WIDTH(9)
   )
   U_SATFO(
      .InputData(FOAngleInter),
      .SatData(FOAngleInterSat)
      );

always @ (posedge PhyClk or negedge nPhyRst)
   begin: FOAngle_Blk
      if (nPhyRst == 1'b0)
         FOAngle <= 8'b0;
      // Get absolute value
      else if (FOAngleInterSat[8:7]==2'b10)
         FOAngle <= {8{1'b1}};
      else if (FOAngleInterSat[8]==1'b1)
         FOAngle <= -FOAngleInterSat[7:0];
      else
         FOAngle <= FOAngleInter[7:0];
   end //FOAngle_Blk

//Additional Code to ease verification
//Display FSM State as a string in RTL simulation waveform
//pragma coverage off
`ifdef RW_SIMU_ON
always @ (*)
   begin: String_Blk
      case (EstCntrlCS)
         ESTIDLE   : EstCntrlCSStr  = {"ESTIDLE"};
         ESTCOARSE : EstCntrlCSStr  = {"ESTCOARSE"};
         ESTFINE   : EstCntrlCSStr  = {"ESTFINE"};
         ESTDONE   : EstCntrlCSStr  = {"ESTDONE"};
         default   : EstCntrlCSStr  = {"UNKNOWN"};
      endcase

      case (EstCntrlNS)
         ESTIDLE   : EstCntrlNSStr  = {"ESTIDLE"};
         ESTCOARSE : EstCntrlNSStr  = {"ESTCOARSE"};
         ESTFINE   : EstCntrlNSStr  = {"ESTFINE"};
         ESTDONE   : EstCntrlNSStr  = {"ESTDONE"};
         default   : EstCntrlNSStr  = {"UNKNOWN"};
      endcase

      case (CmpCntrlCS)
         COMPIDLE           : CmpCntrlCSStr  = {"COMPIDLE"};
         CALCOARSE          : CmpCntrlCSStr  = {"CALCOARSE"};
         COMPWAITPF         : CmpCntrlCSStr  = {"COMPWAITPF"};
         COMPCOARSELTF      : CmpCntrlCSStr  = {"COMPCOARSELTF"};
         CALFINE            : CmpCntrlCSStr  = {"CALFINE"};
         COMPCOARSEFINELTF  : CmpCntrlCSStr  = {"COMPCOARSEFINELTF"};
         COMPWAITDATA       : CmpCntrlCSStr  = {"COMPWAITDATA"};
         COMPCOARSEFINEDATA : CmpCntrlCSStr  = {"COMPCOARSEFINEDATA"};
         default            : CmpCntrlCSStr  = {"UNKNOWN"};
      endcase

      case (CmpCntrlNS)
         COMPIDLE           : CmpCntrlNSStr  = {"COMPIDLE"};
         CALCOARSE          : CmpCntrlNSStr  = {"CALCOARSE"};
         COMPWAITPF         : CmpCntrlNSStr  = {"COMPWAITPF"};
         COMPCOARSELTF      : CmpCntrlNSStr  = {"COMPCOARSELTF"};
         CALFINE            : CmpCntrlNSStr  = {"CALFINE"};
         COMPCOARSEFINELTF  : CmpCntrlNSStr  = {"COMPCOARSEFINELTF"};
         COMPWAITDATA       : CmpCntrlNSStr  = {"COMPWAITDATA"};
         COMPCOARSEFINEDATA : CmpCntrlNSStr  = {"COMPCOARSEFINEDATA"};
         default            : CmpCntrlNSStr  = {"UNKNOWN"};
      endcase
   end //String_Blk
`endif //RW_SIMU_ON
//pragma coverage on


endmodule //TDFOCntrl

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
