//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 34361 $
// $Date: 2018-06-22 11:57:21 +0200 (Fri, 22 Jun 2018) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Top Level File for Cordic Rotator
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       : 
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/TDFOCordicRotTop.v $
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module TDFOCordicRotTop #(parameter NPIPE       =  7, //Number of Pipeline stages
                          parameter NCOMBSTG    =  2, //Number of Combinatorial stages
                          parameter SCALING     =  1, //Scaling to [-4,4] range 
                          parameter DATAWIDTH   = 13, //Width of Input Data
                          parameter ANGLEWIDTH  = 25 //Width of Input Angle
                          )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            // Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock
            // Control Signals
            input    wire                                  Enable,  //Block Enable 
            input    wire                                  DataInValid, //Qualifies Input
            input    wire      [ANGLEWIDTH-1:0]            AngleIn, //Angle for Rotation
            // Data
            input    wire      [DATAWIDTH-1:0]             ReDataIn0, //Real Comp of Data 0
            input    wire      [DATAWIDTH-1:0]             ImDataIn0, //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            input    wire      [DATAWIDTH-1:0]             ReDataIn1, //Real Comp of Data 1
            input    wire      [DATAWIDTH-1:0]             ImDataIn1, //Imag Comp of Data 1
`endif

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            // Data
            output reg  signed [DATAWIDTH-1:0]             ReDataOut0, //Real Comp of Data 0
            output reg  signed [DATAWIDTH-1:0]             ImDataOut0, //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            output reg  signed [DATAWIDTH-1:0]             ReDataOut1, //Real Comp of Data 1
            output reg  signed [DATAWIDTH-1:0]             ImDataOut1, //Imag Comp of Data 1
`endif
            // Control
            output wire                                    DataOutValid //Qualifies Output
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed [DATAWIDTH-1:0]  CONST_ZERO_DATAWIDTH  = {{DATAWIDTH}{1'b0}};
localparam signed [DATAWIDTH+1:0]  CONST_1_DATAWIDTH_P2  = {{{(DATAWIDTH+1)}{1'b0}},1'b1};
localparam signed [DATAWIDTH+1:0]  CONST_2_DATAWIDTH_P2  = 'd2;
localparam signed [DATAWIDTH+1:0]  CONST_4_DATAWIDTH_P2  = 'd4;
localparam signed [DATAWIDTH+1:0]  CONST_5_DATAWIDTH_P2  = 'd5;
localparam signed [DATAWIDTH+1:0]  CONST_7_DATAWIDTH_P2  = 'd7;
localparam signed [DATAWIDTH+1:0]  CONST_8_DATAWIDTH_P2  = 'd8;
localparam signed [DATAWIDTH+1:0]  CONST_11_DATAWIDTH_P2 = 'd11;
localparam signed [DATAWIDTH+1:0]  CONST_12_DATAWIDTH_P2 = 'd12;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [DATAWIDTH+1:0]     ReDataOut0Int;
wire    signed     [DATAWIDTH+1:0]     ImDataOut0Int;
wire    signed     [DATAWIDTH+1:0]     ReDataOut0Norm;
wire    signed     [DATAWIDTH+1:0]     ImDataOut0Norm;
wire    signed     [DATAWIDTH-1:0]     ReDataOut0Sat;
wire    signed     [DATAWIDTH-1:0]     ImDataOut0Sat;
wire                                   DataOutValidInt;
`ifdef RW_NX_DERIV_PATH1
wire    signed     [DATAWIDTH+1:0]     ReDataOut1Int;
wire    signed     [DATAWIDTH+1:0]     ImDataOut1Int;
wire    signed     [DATAWIDTH+1:0]     ReDataOut1Norm;
wire    signed     [DATAWIDTH+1:0]     ImDataOut1Norm;
wire    signed     [DATAWIDTH-1:0]     ReDataOut1Sat;
wire    signed     [DATAWIDTH-1:0]     ImDataOut1Sat;
`endif

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This block instantiates the CordicRot module. It also performs
//normalization on the outputs. It then saturates the output to DATAWIDTH. 
//This operation adds one clock latency to the Cordic operation. 
//The DataOutValid is hence delayed by a clock.

//Instantiate the TDFOCordicRot module
TDFOCordicRot #(
            //Parameters
           .DATAWIDTH(DATAWIDTH),
           .ANGLEWIDTH(ANGLEWIDTH),
           .NCOMBSTG(NCOMBSTG),
           .NPIPE(NPIPE),
           .SCALING(SCALING)
           )U_CR (
                  //Inputs
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Enable(Enable),
                  .DataInValid(DataInValid),
                  .AngleIn(AngleIn),
                  .ReDataIn0(ReDataIn0),
                  .ImDataIn0(ImDataIn0),
`ifdef RW_NX_DERIV_PATH1
                  .ReDataIn1(ReDataIn1),
                  .ImDataIn1(ImDataIn1),
`endif

                  //Outputs
                  .ReDataOut0(ReDataOut0Int),
                  .ImDataOut0(ImDataOut0Int),
`ifdef RW_NX_DERIV_PATH1
                  .ReDataOut1(ReDataOut1Int),
                  .ImDataOut1(ImDataOut1Int),
`endif
                  .DataOutValid(DataOutValidInt)
                 );

// Normalize the outputs
assign ReDataOut0Norm = $signed(NormalizeFct(ReDataOut0Int));
assign ImDataOut0Norm = $signed(NormalizeFct(ImDataOut0Int));
`ifdef RW_NX_DERIV_PATH1
assign ReDataOut1Norm = $signed(NormalizeFct(ReDataOut1Int));
assign ImDataOut1Norm = $signed(NormalizeFct(ImDataOut1Int));
`endif

// Saturate the Normalized Outputs
   SatSymSigned #(
                  .INPUT_WIDTH(DATAWIDTH+2),
                  .OUTPUT_WIDTH(DATAWIDTH)
                 )
                  U_SATCRT0(
                           .InputData(ReDataOut0Norm),
                           .SatSymData(ReDataOut0Sat)
                           );

   SatSymSigned #(
                  .INPUT_WIDTH(DATAWIDTH+2),
                  .OUTPUT_WIDTH(DATAWIDTH)
                 )
                  U_SATCRT1(
                           .InputData(ImDataOut0Norm),
                           .SatSymData(ImDataOut0Sat)
                           );

`ifdef RW_NX_DERIV_PATH1
      SatSymSigned #(
                     .INPUT_WIDTH(DATAWIDTH+2),
                     .OUTPUT_WIDTH(DATAWIDTH)
                    )
                     U_SATCRT2(
                              .InputData(ReDataOut1Norm),
                              .SatSymData(ReDataOut1Sat)
                              );

      SatSymSigned #(
                     .INPUT_WIDTH(DATAWIDTH+2),
                     .OUTPUT_WIDTH(DATAWIDTH)
                    )
                     U_SATCRT3(
                              .InputData(ImDataOut1Norm),
                              .SatSymData(ImDataOut1Sat)
                              );
`endif

//Register the final outputs
always @ (posedge PhyClk or negedge nPhyRst)
   begin: OpReg_Blk
      if (nPhyRst == 1'b0) begin
         ReDataOut0 <= CONST_ZERO_DATAWIDTH;
         ImDataOut0 <= CONST_ZERO_DATAWIDTH;
`ifdef RW_NX_DERIV_PATH1
         ReDataOut1 <= CONST_ZERO_DATAWIDTH;
         ImDataOut1 <= CONST_ZERO_DATAWIDTH;
`endif
      end
      else if (Enable == 1'b1) begin
         ReDataOut0 <= ReDataOut0Sat;
         ImDataOut0 <= ImDataOut0Sat;
`ifdef RW_NX_DERIV_PATH1
         ReDataOut1 <= ReDataOut1Sat;
         ImDataOut1 <= ImDataOut1Sat;
`endif
      end //Enable == 1'b1

   end //OpReg_Blk

//Function to generate the Normalized Value of the output
//Multiply by normalization factor 2487/4096

function           [DATAWIDTH+1:0]                NormalizeFct;

//Inputs
input    signed    [DATAWIDTH+1:0]                DataIn;

//Internal Registers
reg      signed    [DATAWIDTH+13:0]               Normal;

   begin
      //Multiply by 2487
      //2487 = 2048 + 256 + 128 + 32 + 16 + 4 + 2 + 1
      Normal =   (DataIn <<< CONST_11_DATAWIDTH_P2) + (DataIn <<< CONST_8_DATAWIDTH_P2) + (DataIn <<< CONST_7_DATAWIDTH_P2)
               + (DataIn <<< CONST_5_DATAWIDTH_P2)  + (DataIn <<< CONST_4_DATAWIDTH_P2) + (DataIn <<< CONST_2_DATAWIDTH_P2)
               + (DataIn <<< CONST_1_DATAWIDTH_P2)  + (DataIn);

      // Divide by 4096
      NormalizeFct = $signed(Normal[DATAWIDTH+13:12]);

   end

endfunction  //NormalizeFct;

//Delay DataValidOut by a clock
DelayLine # (
             .DATAWIDTH(1),     //Width of Enable
             .LATENCY(1)        //Latency
             )
              U_DelayLineCRT (
                             .PhyClk(PhyClk),
                             .nPhyRst(nPhyRst),
                             .Bypass(~Enable),
                             .DataIn(DataOutValidInt),
                             .DataOut(DataOutValid)
                             );

endmodule //CordicRotTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
