//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 37860 $
// $Date: 2019-03-04 16:40:03 +0100 (Mon, 04 Mar 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Time Domain Frequency Offset - Plateau Fall Detect
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/TDFOPlatDet.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module TDFOPlatDet #(parameter TDFOCVWIDTH = 16 //Datawidth of Auto Correlation Value
                    )(

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input  wire                                  nPhyRst, // Active LOW Reset
            input  wire                                  PhyClk,  // PHY Clock

            ///////////////////////////////////////////////
            // Registers interface
            ///////////////////////////////////////////////
            input  wire      [9:0]                       AutoCorrCompareRatioLow,  // Low Scaling Factor
            input  wire      [9:0]                       AutoCorrCompareRatioHigh, // High Scaling Factor
            input  wire      [3:0]                       AutoCorrFallCount,        // Count for Plateau Fall Det

            ///////////////////////////////////////////////
            // Control Signals
            ///////////////////////////////////////////////
            input  wire                                  CorrValAccEn,      // Enable CorrVal Acc
            input  wire                                  CorrValAccClr,     // Clear Acc
            input  wire                                  AbsCValAccEn,      // Enable AbsCorrVal Acc
            input  wire                                  PlatFallLowDetEn,  // Enable Plateau Fall low Det
            input  wire                                  PlatFallHighDetEn, // Enable Plateau Fall high Det
            input  wire                                  ModApproxCompEn,   // Enable ModApprox
            //
            output reg                                   PlatThreshDoneP,   // Threshold Calc done
            output reg                                   PlatFallLowP,      // Indicates Plateau Fall low
            output reg                                   PlatFallHighP,     // Indicates Plateau Fall high
            
            ///////////////////////////////////////////////
            // Data
            ///////////////////////////////////////////////
            input  wire signed     [TDFOCVWIDTH-1:0]     CorrValRe,    // Real Component
            input  wire signed     [TDFOCVWIDTH-1:0]     CorrValIm,    // Imaginary Component
            input  wire                                  CorrValValid, // Qualifies CorrVal Data

            // Accumulated Data for Coarse TDFO
            output reg signed     [TDFOCVWIDTH-1:0]      CorrValAccRe,    // Real Component
            output reg signed     [TDFOCVWIDTH-1:0]      CorrValAccIm,    // Imaginary Component
            output reg                                   CorrValAccValidP // Qualifies CorrValAcc
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
// General param
localparam signed [TDFOCVWIDTH-1:0]    CONST_ZERO_TDFOCVWIDTH    = {{TDFOCVWIDTH}{1'b0}};
localparam signed [TDFOCVWIDTH+6:0]    CONST_ZERO_TDFOCVWIDTH_P7 = {{(TDFOCVWIDTH + 7)}{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [TDFOCVWIDTH-1:0]   CorrValAccRndRe;
wire    signed     [TDFOCVWIDTH-1:0]   CorrValAccRndIm;
wire               [TDFOCVWIDTH-1:0]   CorrValAbs;
wire               [TDFOCVWIDTH-1:0]   CorrValAbsAccRnd;
wire               [TDFOCVWIDTH-1:0]   CorrPlatAccScaleLowRnd;
wire               [TDFOCVWIDTH-1:0]   CorrPlatAccScaleHighRnd;
wire                                   CorrValValidMux;
wire                                   CorrValValid2D;
wire                                   CorrValValidMux2D;
wire                                   CorrValAccEnD;
wire                                   AbsCValAccEn2D;
wire                                   AbsCValAccEn3D;
wire                                   ByPass;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [TDFOCVWIDTH+6:0]   CorrValAccIntRe;
reg     signed     [TDFOCVWIDTH+6:0]   CorrValAccIntIm;
reg                [TDFOCVWIDTH+4:0]   CorrValAbsAccInt;
reg                [TDFOCVWIDTH-1:0]   CorrPlatAcc;
reg                [TDFOCVWIDTH+9:0]   CorrPlatAccScaleLow;
reg                [TDFOCVWIDTH+9:0]   CorrPlatAccScaleHigh;
reg                [TDFOCVWIDTH-1:0]   PlatThreshLow;
reg                [TDFOCVWIDTH-1:0]   PlatThreshHigh;
reg                [3:0]               PlatFallCntLow;
reg                [3:0]               PlatFallCntHigh;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Bypass disabled
assign ByPass = 1'b0;

//This block has 3 functions
//1)It accumulates the CorrSum value coming from TDFOCorrVal block when
//CorrValAccEn is high. This is usually for 2.4 us
//2)It calculates the Plateau Fall Threshold Value when AbsCValAccEn is high
//3)It compares the latest abs value of the CorrSum with the threshold
//to detect the plateau fall when PlatFallDetEn is high

//Accumulate CorrSum for 2.4 us
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrAccIntBlk
      if (nPhyRst == 1'b0) begin
         CorrValAccIntRe <= CONST_ZERO_TDFOCVWIDTH_P7;
         CorrValAccIntIm <= CONST_ZERO_TDFOCVWIDTH_P7;
      end
      else if (CorrValAccClr == 1'b1) begin
         CorrValAccIntRe <= CONST_ZERO_TDFOCVWIDTH_P7;
         CorrValAccIntIm <= CONST_ZERO_TDFOCVWIDTH_P7;
      end
      else if (CorrValAccEn == 1'b1 && CorrValValid == 1'b1) begin
         CorrValAccIntRe <= CorrValAccIntRe + $signed({{7{CorrValRe[TDFOCVWIDTH-1]}},CorrValRe});
         CorrValAccIntIm <= CorrValAccIntIm + $signed({{7{CorrValIm[TDFOCVWIDTH-1]}},CorrValIm});
      end

   end //CorrAccIntBlk

//Round CorrValAccInt to TDFOCVWIDTH bits
//Rounding is from TDFOCVWIDTH+6 to TDFOCVWIDTH (6 LSBs)
Round #(
        .INPUT_WIDTH(TDFOCVWIDTH+6),
        .OUTPUT_WIDTH(TDFOCVWIDTH)
       )
        U_ROUNDCORA(
                    .InputData(CorrValAccIntRe[TDFOCVWIDTH+5:0]),
                    .RoundData(CorrValAccRndRe)
                   );

Round #(
        .INPUT_WIDTH(TDFOCVWIDTH+6),
        .OUTPUT_WIDTH(TDFOCVWIDTH)
       )
        U_ROUNDCORB(
                    .InputData(CorrValAccIntIm[TDFOCVWIDTH+5:0]),
                    .RoundData(CorrValAccRndIm)
                   );

//Register the output
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrValAccBlk
      if (nPhyRst == 1'b0) begin
         CorrValAccRe <= CONST_ZERO_TDFOCVWIDTH;
         CorrValAccIm <= CONST_ZERO_TDFOCVWIDTH;
      end
      else begin
         CorrValAccRe <= CorrValAccRndRe;
         CorrValAccIm <= CorrValAccRndIm;
      end
   end //CorrValAccBlk

//Generate CorrValAccValidP signal
//This will be asserted 2 clocks after CorrValAccEn is deasserted
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrValAccValid_Blk
      if (nPhyRst == 1'b0)
         CorrValAccValidP <= 1'b0;
      else if (CorrValAccEnD == 1'b1 && CorrValAccEn == 1'b0)
         CorrValAccValidP <= 1'b1;
      else
         CorrValAccValidP <= 1'b0;
   end //CorrValAccValid_Blk


//Generate CorrValAccEnD
DelayLine # (
             .DATAWIDTH(1), //Width of CorrValAccEn
             .LATENCY(1)    //1 clock latency
             )
              U_DelayLineY (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(CorrValAccEn),
                            .DataOut(CorrValAccEnD)
                           );

//Plateau Fall Threshold Calculation
//Find the absolute value of the incoming CorrVal
ModApprox #(
            .SUM_WIDTH(TDFOCVWIDTH),    //Input Width
            .CORRVAL_WIDTH(TDFOCVWIDTH) //Output Width
           )
            U_MdApprox0(
                        //Inputs
                        .PhyClk(PhyClk),
                        .nPhyRst(nPhyRst),
                        .ComputeOn(ModApproxCompEn),
                        .ISum(CorrValRe),
                        .QSum(CorrValIm),

                        //Output
                        .CorrVal(CorrValAbs)
                       );


//ModApprox block has 2 clocks latency, so we need to 
//delay CorrValValid by 2 cocks
DelayLine # (
             .DATAWIDTH(1), //Width of CorrValValid
             .LATENCY(2)    //2 clocks latency
             )
              U_DelayLineZ1 (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(CorrValValid),
                            .DataOut(CorrValValid2D)
                           );

// Enable signal only after block is enable
DelayLine # (
             .DATAWIDTH(1), //Width of CorrValValid
             .LATENCY(2)    //2 clocks latency
             )
              U_DelayLineZ2 (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(CorrValValidMux),
                            .DataOut(CorrValValidMux2D)
                           );

assign CorrValValidMux = CorrValValid & PlatFallLowDetEn;

//Accumulate CorrValAbs when AbsCValAccEn is asserted
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrValAbsAccInt_Blk
      if (nPhyRst == 1'b0)
         CorrValAbsAccInt <= {{(TDFOCVWIDTH+5)}{1'b0}};
      else if (CorrValAccClr == 1'b1)
         CorrValAbsAccInt <= {{(TDFOCVWIDTH+5)}{1'b0}};
      else if (AbsCValAccEn == 1'b1 && CorrValValid2D == 1'b1)
         CorrValAbsAccInt <= CorrValAbsAccInt + {5'b0,CorrValAbs};

   end //CorrValAbsAccInt_Blk

//Unsigned Rounding
//Rounding is from TDFOCVWIDTH+4 to TDFOCVWIDTH (4 LSBs)
USgnRound #(
            .INPUT_WIDTH(TDFOCVWIDTH+4),
            .OUTPUT_WIDTH(TDFOCVWIDTH)
           )
            U_ROUNDCORY(
                        .InputData(CorrValAbsAccInt[TDFOCVWIDTH+3:0]),
                        .RoundData(CorrValAbsAccRnd)
                       );

//Register the required value
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrPlatAccBlk
      if (nPhyRst == 1'b0)
         CorrPlatAcc <= {{TDFOCVWIDTH}{1'b0}};
      else
         CorrPlatAcc <= CorrValAbsAccRnd;
   end //CorrPlatAccBlk

//Scaling by AutoCorrCompareRatio low and high
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrPlatAccScBlk
      if (nPhyRst == 1'b0) begin
         CorrPlatAccScaleLow  <= {{(TDFOCVWIDTH+10)}{1'b0}};
         CorrPlatAccScaleHigh <= {{(TDFOCVWIDTH+10)}{1'b0}};
      end
      else begin
         CorrPlatAccScaleLow  <= CorrPlatAcc * {6'b0,AutoCorrCompareRatioLow};
         CorrPlatAccScaleHigh <= CorrPlatAcc * {6'b0,AutoCorrCompareRatioHigh};
      end
   end //CorrPlatAccScBlk

//Unsigned Rounding from TDFOCVWIDTH+10 to TDFOCVWIDTH (10 LSBs)
USgnRound #(
            .INPUT_WIDTH(TDFOCVWIDTH+10),
            .OUTPUT_WIDTH(TDFOCVWIDTH)
           )
            U_ROUNDCORR_LOW(
                        .InputData(CorrPlatAccScaleLow),
                        .RoundData(CorrPlatAccScaleLowRnd)
                       );


USgnRound #(
            .INPUT_WIDTH(TDFOCVWIDTH+10),
            .OUTPUT_WIDTH(TDFOCVWIDTH)
           )
            U_ROUNDCORR_HIGH(
                        .InputData(CorrPlatAccScaleHigh),
                        .RoundData(CorrPlatAccScaleHighRnd)
                       );

//Register the Threshold Value
always @ (posedge PhyClk or negedge nPhyRst)
   begin: ThreshReg_Blk
      if (nPhyRst == 1'b0) begin
         PlatThreshLow  <= {{TDFOCVWIDTH}{1'b0}};
         PlatThreshHigh <= {{TDFOCVWIDTH}{1'b0}};
      end
      else begin
         PlatThreshLow  <= CorrPlatAccScaleLowRnd;
         PlatThreshHigh <= CorrPlatAccScaleHighRnd;
      end
   end //ThreshReg_Blk

//Generate PlatThreshDoneP
//The value of Plateau Fall Threshold will be valid 4 clocks after
//AbsCValAccEn has been deasserted
DelayLine # (
             .DATAWIDTH(1), //Width of AbsCValAccEn
             .LATENCY(2)    //2 clocks latency
             )
              U_DelayLineM (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(AbsCValAccEn),
                            .DataOut(AbsCValAccEn2D)
                           );

DelayLine # (
             .DATAWIDTH(1), //Width of AbsCValAccEn2D
             .LATENCY(1)    //1 clock latency
             )
              U_DelayLineN (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(ByPass),
                            .DataIn(AbsCValAccEn2D),
                            .DataOut(AbsCValAccEn3D)
                           );

always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatThreshDoneP_Blk
      if (nPhyRst == 1'b0)
         PlatThreshDoneP <= 1'b0;
      else if (AbsCValAccEn3D == 1'b1 && AbsCValAccEn2D == 1'b0)
         PlatThreshDoneP <= 1'b1;
      else
         PlatThreshDoneP <= 1'b0;
   end //PlatThreshDoneP_Blk

//Plateau Fall Detection
//The plateau fall search window is enabled when PlatFallDetEn is 
//asserted. During this period, the value of CorrValAbs is continuously
//compared with the value of PlatThresh. When the instantaneous value is
//equal to or less than the threshold for "n" consecutive samples,
//PlatFallP is asserted. Here "n" is given by a register programmable value,
//namely, AutoCorrFallCount.

//Count the number of consecutive cycles for which plateau fall has occured
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallLowCnt_Blk
      if (nPhyRst == 1'b0) begin
         PlatFallCntLow  <= 4'b0;
      end
      else if (PlatFallLowDetEn == 1'b0) begin
         PlatFallCntLow  <= 4'b0;
      end
      else if (CorrValValidMux2D == 1'b1) begin
         if (CorrValAbs < PlatThreshLow)
           PlatFallCntLow  <= PlatFallCntLow + 4'b1;
      end
   end //PlatFallLowCnt_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallHighCnt_Blk
      if (nPhyRst == 1'b0) begin
         PlatFallCntHigh <= 4'b0;
      end
      else if (PlatFallHighDetEn == 1'b0) begin
         PlatFallCntHigh <= 4'b0;
      end
      else if (CorrValValidMux2D == 1'b1) begin
         if (CorrValAbs < PlatThreshHigh)
           PlatFallCntHigh <= PlatFallCntHigh + 4'b1;
      end
   end //PlatFallHighCnt_Blk

//Assert PlatFall pulse when PlatFall counters equal AutoCorrFallCount
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallLowP_Blk
      if (nPhyRst == 1'b0)
         PlatFallLowP  <= 1'b0;
      else if (PlatFallLowDetEn == 1'b0)
         PlatFallLowP  <= 1'b0;
      else if (PlatFallLowP == 1'b1)
         PlatFallLowP  <= 1'b0;
      else if ((CorrValValid == 1'b1) && (PlatFallCntLow == AutoCorrFallCount))
         PlatFallLowP  <= 1'b1;
      else
         PlatFallLowP  <= 1'b0;
   end //PlatFallLowP_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: PlatFallHighP_Blk
      if (nPhyRst == 1'b0)
         PlatFallHighP <= 1'b0;
      else if (PlatFallHighDetEn == 1'b0)
         PlatFallHighP <= 1'b0;
      else if (PlatFallHighP == 1'b1)
         PlatFallHighP <= 1'b0;
      else if ((CorrValValid == 1'b1) && (PlatFallCntHigh == AutoCorrFallCount))
         PlatFallHighP <= 1'b1;
      else
         PlatFallHighP <= 1'b0;
   end //PlatFallHighP_Blk

endmodule //TDFOPlatDet

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
