//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39422 $
// $Date: 2019-07-12 12:05:19 +0200 (Fri, 12 Jul 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : TDFO Estimation and Compensation Top Level Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/TDFOTop.v $
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module TDFOTop (
            ///////////////////////////////////////////////
            // Clock and Reset Manager interface
            ///////////////////////////////////////////////
            input  wire                             nPhyRst,     // Active LOW Reset
            input  wire                             PhyClk,      // PHY Clock
            input  wire                             TDFOEstGClk, // Gated Clk for Est Blocks

            ///////////////////////////////////////////////
            // Registers interface
            ///////////////////////////////////////////////
            input  wire                             Force20,      // 20 MHz Mode only
            input  wire                             Force40,      // 20/40 MHz Mode only
            //
            input  wire         [9:0]               AutoCorrCompareRatioLow,
            input  wire         [9:0]               AutoCorrCompareRatioHigh,
            input  wire         [3:0]               AutoCorrFallCount,
            //
            input  wire         [8:0]               AutoCorrPlatSumStart,
            input  wire         [8:0]               AutoCorrPlatSumEnd,
            input  wire         [8:0]               AutoCorrPlatFallSearchStart,
            input  wire         [3:0]               AutoCorrPlatFallIndexOffset,
            input  wire         [8:0]               AutoCorrPhSumStart,
            input  wire         [8:0]               AutoCorrPhSumEnd,
            //
            input  wire                             TDFOCompEn,
            input  wire                             TDFOFineEstEn,
            input  wire                             TDFOCpeSlopeEn,
            //
            output wire signed  [20:0]              CoarseFOAngle, // Coarse FO Angle       
            output wire signed  [20:0]              FineFOAngle,   // Fine FO Angle           

            ///////////////////////////////////////////////
            // Rx TD FSM interface
            ///////////////////////////////////////////////
            input  wire         [1:0]               Bandwidth,    // Bandwidth of current pkt
            input  wire                             TDFOStrtP,    // TDFO Estimation Start
            input  wire                             DataCmpStrtP, // Start Data Compensation
            input  wire                             TDFOEn,       // TDFO block enable
            //
            output wire                             PlatFallP,    // Indicates Plateau Fall   
            output wire                             CmpLTFDoneP,  // LTF Compensation Done Pulse
            output wire                             FineFODone,   // Fine FO estimation Done

            ///////////////////////////////////////////////
            // Rx FD interface
            ///////////////////////////////////////////////
            input  wire         [21:0]              FDCpeSlope,   // CPE Slope from FDO
`ifdef RW_NX_TDFO_NSR_EN
            output reg          [11:0]              RatioNoiseSignal, // NSR to SB compression
            output reg          [5:0]               RatioShift,
`endif

            ///////////////////////////////////////////////
            // AGC interface
            ///////////////////////////////////////////////
            output wire         [7:0]               FOAngle,      // FO Angle

            ///////////////////////////////////////////////
            // Data path
            ///////////////////////////////////////////////
            // Primary data 20/40/80MHz from Rx Chains
            input  wire signed  [12:0]              RxDataInRe0,    // Real Comp of Data 0
            input  wire signed  [12:0]              RxDataInIm0,    // Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            input  wire signed  [12:0]              RxDataInRe1,    // Real Comp of Data 1
            input  wire signed  [12:0]              RxDataInIm1,    // Imag Comp of Data 1
`endif
            input  wire                             RxDataValid,    // Qualifies Rx Data Input

            // Secondary data 20MHz
            input  wire signed  [12:0]              RxDataIn20SRe,  // Real Comp of Secondary Data
            input  wire signed  [12:0]              RxDataIn20SIm,  // Imag Comp of Secondary Data

            ///////////////////////////////////////////////
            // Timing Boundary Estimation interface
            ///////////////////////////////////////////////
            input  wire         [7:0]               TBECount,       // From TBE Block
            input  wire                             TBECountValid,  // Qualifies TBE counters
            //
            input  wire signed  [9:0]               TBE16DRxI0,     // Real Comp of Delay Line 0
            input  wire signed  [9:0]               TBE16DRxQ0,     // Imag Comp of Delay Line 0
`ifdef RW_NX_DERIV_PATH1
            input  wire signed  [9:0]               TBE16DRxI1,     // Real Comp of Delay Line 1
            input  wire signed  [9:0]               TBE16DRxQ1,     // Imag Comp of Delay Line 1
`endif
            input  wire signed  [9:0]               TBE64DRxI0,     // Real Comp of Delay Line 0
            input  wire signed  [9:0]               TBE64DRxQ0,     // Imag Comp of Delay Line 0
`ifdef RW_NX_DERIV_PATH1
            input  wire signed  [9:0]               TBE64DRxI1,     // Real Comp of Delay Line 1
            input  wire signed  [9:0]               TBE64DRxQ1,     // Imag Comp of Delay Line 1
`endif

            output wire signed  [9:0]               RxLTFRe0,       // Real Comp of Data 0
            output wire signed  [9:0]               RxLTFIm0,       // Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            output wire signed  [9:0]               RxLTFRe1,       // Real Comp of Data 1
            output wire signed  [9:0]               RxLTFIm1,       // Imag Comp of Data 1
`endif
            output wire                             RxLTFValid,     // Qualifies Rx LTF Output  

  
            ///////////////////////////////////////////////
            // TDFO Memory interface
            ///////////////////////////////////////////////
            input  wire         [12:0]              TDFOMem0RdDataRe0,    // Real Read Data 0 Rx Path 0
            input  wire         [12:0]              TDFOMem0RdDataIm0,    // Imag Read Data 0 Rx Path 0
            input  wire         [12:0]              TDFOMem1RdDataRe0,    // Real Read Data 1 Rx Path 0
            input  wire         [12:0]              TDFOMem1RdDataIm0,    // Imag Read Data 1 Rx Path 0
`ifdef RW_NX_DERIV_PATH1
            input  wire         [12:0]              TDFOMem0RdDataRe1,    // Real Read Data 0 Rx Path 1
            input  wire         [12:0]              TDFOMem0RdDataIm1,    // Imag Read Data 0 Rx Path 1
            input  wire         [12:0]              TDFOMem1RdDataRe1,    // Real Read Data 1 Rx Path 1
            input  wire         [12:0]              TDFOMem1RdDataIm1,    // Imag Read Data 1 Rx Path 1
`endif
            input  wire         [12:0]              TDFOMem0RdData20SRe,  // Real Read Data 0 Secondary
            input  wire         [12:0]              TDFOMem0RdData20SIm,  // Imag Read Data 0 Secondary
            input  wire         [12:0]              TDFOMem1RdData20SRe,  // Real Read Data 1 Secondary
            input  wire         [12:0]              TDFOMem1RdData20SIm,  // Imag Read Data 1 Secondary
            //
            output wire                             TDFOMemRdEn,          // Read Enable
            output wire         [7:0]               TDFOMemRdAddr,        // Read Address
            output wire         [12:0]              TDFOMemWrDataRe0,     // Real Write Data Rx Path 0
            output wire         [12:0]              TDFOMemWrDataIm0,     // Imag Write Data Rx Path 0
`ifdef RW_NX_DERIV_PATH1
            output wire         [12:0]              TDFOMemWrDataRe1,     // Real Write Data Rx Path 1
            output wire         [12:0]              TDFOMemWrDataIm1,     // Imag Write Data Rx Path 1
`endif
            output wire         [12:0]              TDFOMemWrData20SRe,   // Real Write Data Secondary Rx Path 0
            output wire         [12:0]              TDFOMemWrData20SIm,   // Imag Write Data Secondary Rx Path 0
            output wire         [7:0]               TDFOMemWrAddr,        // Write Address
            output wire                             TDFOMemWrEn,          // Write Enable
            output wire                             TDFOMemWrSel,         // Buffer selection

            ///////////////////////////////////////////////
            // FFT Memory interface
            ///////////////////////////////////////////////
            output reg          [12:0]              FFTMemWrDataRe0,      // Real Write Data Rx Path 0
            output reg          [12:0]              FFTMemWrDataIm0,      // Imag Write Data Rx Path 0
`ifdef RW_NX_DERIV_PATH1
            output reg          [12:0]              FFTMemWrDataRe1,      // Real Write Data Rx Path 1
            output reg          [12:0]              FFTMemWrDataIm1,      // Imag Write Data Rx Path 1
`endif
            output reg          [12:0]              FFTMemWrData20SRe,    // Real Write Data Secondary Rx Path 0
            output reg          [12:0]              FFTMemWrData20SIm,    // Imag Write Data Secondary Rx Path 0
            output wire                             FFTMemWrEn,           // Write Enable         
            output wire                             FFTMemSel             // Select memory control
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////

// Fixed Parameters
localparam ANGLEWIDTH   = 21; // Cordic Vect Angle Width
localparam TDFOIPWIDTH  = 10; // Data width of input to TDFO estimation
localparam TDFOCVWIDTH  = 16; // Data width of Correlation Value
localparam MEMDATAWIDTH = 13; // Data width of one channel coefficient
localparam NPIPE        =  7; // Number of pipeline stages in Cordic Rot
localparam NCOMBSTG     =  2; // Number of combinatorial stages in Cordic Rot
localparam SCALING      =  1; // Scaling to [-4,4] range in Cordic Rot

// General Parameters
localparam    signed   [TDFOIPWIDTH-1:0]   CONST_ZERO_TDFOIPWIDTH = {{TDFOIPWIDTH}{1'b0}};
localparam    signed   [MEMDATAWIDTH-1:0] CONST_ZERO_MEMDATAWIDTH = {{MEMDATAWIDTH}{1'b0}};
localparam    signed   [TDFOCVWIDTH-1:0]   CONST_ZERO_TDFOCVWIDTH = {{TDFOCVWIDTH}{1'b0}};

//TDFO Mode
localparam COARSE = 1'b0;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                                   CorrEn;
wire                                   MovAvgEn;
wire                                   MovAvgClr;
wire                                   TDFOEstMode;
wire                                   TDFOCmpMode;
wire                                   TDFOLTFCmpMode;
wire                                   TDFOCmpDataMode;
wire                                   CorrValOutValid;
wire    signed   [TDFOCVWIDTH-1:0]     CorrValOutRe;
wire    signed   [TDFOCVWIDTH-1:0]     CorrValOutIm;
wire                                   CorrValAccEn;
wire                                   AbsCValAccEn;
wire                                   PlatFallLowDetEn;
wire                                   PlatFallHighDetEn;
wire                                   ModApproxCompEn;
wire                                   PlatThreshDoneP;
wire    signed   [TDFOCVWIDTH-1:0]     CorrValAccRe;
wire    signed   [TDFOCVWIDTH-1:0]     CorrValAccIm;
wire                                   CorrValAccValidP;
wire                                   CVectDataEnIn;
wire                                   CVectOngoing;
wire                                   CRotEnable;
wire                                   CRot0DataInValid;
wire                                   CRot1DataInValid;
wire    signed   [ANGLEWIDTH+3:0]      CRot0Angle;
wire    signed   [ANGLEWIDTH+3:0]      CRot1Angle;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ReDataIn0;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ImDataIn0;
wire    signed   [ANGLEWIDTH-1:0]      CVAngleOut;
wire                                   CVectDone;
wire                                   CRot0DataOutValid;
wire                                   CRot01DataOutValid;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ReDataOut0;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ImDataOut0;
wire    signed   [MEMDATAWIDTH-1:0]    CRot01ReDataOut0;
wire    signed   [MEMDATAWIDTH-1:0]    CRot01ImDataOut0;
wire                                   CorValFineDataEn;
wire                                   MemRdColOut;
wire    signed   [MEMDATAWIDTH-1:0]    SharedRndInRe0;
wire    signed   [MEMDATAWIDTH-1:0]    SharedRndInIm0;
wire    signed   [TDFOIPWIDTH-1:0]     SharedRndOutRe0;
wire    signed   [TDFOIPWIDTH-1:0]     SharedRndOutIm0;
wire    signed   [TDFOIPWIDTH-1:0]     SharedSatRe0;
wire    signed   [TDFOIPWIDTH-1:0]     SharedSatIm0;
wire                                   PlatFallPInt;
wire                                   PlatFallHighP;
`ifdef RW_NX_DERIV_PATH1
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ReDataIn1;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ImDataIn1;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ReDataOut1;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ImDataOut1;
wire    signed   [MEMDATAWIDTH-1:0]    CRot01ReDataOut1;
wire    signed   [MEMDATAWIDTH-1:0]    CRot01ImDataOut1;
wire    signed   [MEMDATAWIDTH-1:0]    RxDataInRe1D2;
wire    signed   [MEMDATAWIDTH-1:0]    RxDataInIm1D2;
wire    signed   [MEMDATAWIDTH-1:0]    SharedRndInRe1;
wire    signed   [MEMDATAWIDTH-1:0]    SharedRndInIm1;
wire    signed   [TDFOIPWIDTH-1:0]     SharedRndOutRe1;
wire    signed   [TDFOIPWIDTH-1:0]     SharedRndOutIm1;
wire    signed   [TDFOIPWIDTH-1:0]     SharedSatRe1;
wire    signed   [TDFOIPWIDTH-1:0]     SharedSatIm1;
wire    signed   [TDFOIPWIDTH-1:0]     CVDataInReRx1Inter;
wire    signed   [TDFOIPWIDTH-1:0]     CVDataInImRx1Inter;
wire    signed   [TDFOIPWIDTH-1:0]     CVDataDBufReRx1Inter;
wire    signed   [TDFOIPWIDTH-1:0]     CVDataDBufImRx1Inter;
`endif
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ReDataIn20S;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ImDataIn20S;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ReDataOut20S;
wire    signed   [MEMDATAWIDTH-1:0]    CRot0ImDataOut20S;
wire    signed   [MEMDATAWIDTH-1:0]    CRot01ReDataOut20S;
wire    signed   [MEMDATAWIDTH-1:0]    CRot01ImDataOut20S;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                   CVDataValid;
reg     signed     [TDFOIPWIDTH-1:0]  CVDataInReRx0;
reg     signed     [TDFOIPWIDTH-1:0]  CVDataInImRx0;
reg     signed     [TDFOCVWIDTH-1:0]  CVReDataIn; 
reg     signed     [TDFOCVWIDTH-1:0]  CVImDataIn; 
reg     signed     [TDFOIPWIDTH-1:0]  CVDataDBufReRx0;
reg     signed     [TDFOIPWIDTH-1:0]  CVDataDBufImRx0;
reg     signed     [MEMDATAWIDTH-1:0] CRot1ReDataIn0;
reg     signed     [MEMDATAWIDTH-1:0] CRot1ImDataIn0;
reg     signed     [MEMDATAWIDTH-1:0] CRot0MuxReDataIn0;
reg     signed     [MEMDATAWIDTH-1:0] CRot0MuxImDataIn0;
reg                                   RxDataValid1D;
reg                                   RxDataValid2D;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInRe0D1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInIm0D1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInRe0D2;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInIm0D2;
reg                                   TDFOEstModeD;
reg     signed     [TDFOIPWIDTH-1:0]  CRot0ReDataOut0D1;
reg     signed     [TDFOIPWIDTH-1:0]  CRot0ImDataOut0D1;
`ifdef RW_NX_DERIV_PATH1
reg     signed     [TDFOIPWIDTH-1:0]  CVDataInReRx1;
reg     signed     [TDFOIPWIDTH-1:0]  CVDataInImRx1;
reg     signed     [TDFOIPWIDTH-1:0]  CVDataDBufReRx1;
reg     signed     [TDFOIPWIDTH-1:0]  CVDataDBufImRx1;
reg     signed     [MEMDATAWIDTH-1:0] CRot1ReDataIn1;
reg     signed     [MEMDATAWIDTH-1:0] CRot1ImDataIn1;
reg     signed     [MEMDATAWIDTH-1:0] CRot0MuxReDataIn1;
reg     signed     [MEMDATAWIDTH-1:0] CRot0MuxImDataIn1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInRe1D1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInIm1D1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInRe1D2;
reg     signed     [MEMDATAWIDTH-1:0] RxDataInIm1D2;
reg     signed     [TDFOIPWIDTH-1:0]  CRot0ReDataOut1D1;
reg     signed     [TDFOIPWIDTH-1:0]  CRot0ImDataOut1D1;
`endif
reg     signed     [MEMDATAWIDTH-1:0] CRot1ReDataIn20S;
reg     signed     [MEMDATAWIDTH-1:0] CRot1ImDataIn20S;
reg     signed     [MEMDATAWIDTH-1:0] CRot0MuxReDataIn20S;
reg     signed     [MEMDATAWIDTH-1:0] CRot0MuxImDataIn20S;
reg     signed     [MEMDATAWIDTH-1:0] RxDataIn20SReD1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataIn20SImD1;
reg     signed     [MEMDATAWIDTH-1:0] RxDataIn20SReD2;
reg     signed     [MEMDATAWIDTH-1:0] RxDataIn20SImD2;

`ifdef RW_NX_TDFO_NSR_EN
//////////////////////////////////////////////////////////////////////////////
// NSR computation
//////////////////////////////////////////////////////////////////////////////
wire    signed     [TDFOIPWIDTH*2-1:0]  corrLTFsReRx0;
wire    signed     [TDFOIPWIDTH*2-1:0]  corrLTFsImRx0;
reg     signed     [TDFOIPWIDTH*2-1:0]  corrLTFsRe;
reg     signed     [TDFOIPWIDTH*2-1:0]  corrLTFsIm;
wire    signed     [TDFOIPWIDTH*2:0]    ExtCorrLTFsRe;
wire    signed     [TDFOIPWIDTH*2:0]    ExtCorrLTFsIm;
reg                                     CorrEnD;
reg                                     CordicMagEn;
wire                                    ModulusValid;
reg                                     ModulusValidD;
wire               [TDFOIPWIDTH*2+1:0]  Modulus;
`ifdef RW_NX_DERIV_PATH1
wire    signed     [TDFOIPWIDTH*2-1:0]  corrLTFsReRx1;
wire    signed     [TDFOIPWIDTH*2-1:0]  corrLTFsImRx1;
reg                                     CordicMagRx0Done;
reg                [TDFOIPWIDTH*2+1:0]  SumSignalPower;
`else
  wire             [TDFOIPWIDTH*2+1:0]  SumSignalPower;
`endif

wire               [21:0]               totalPower;
reg                [22:0]               noisePower;
wire               [22:0]               MaxNoisePower1;
wire               [4:0]                noisePowerExp;
wire               [9:0]                noisePowerMant;
wire               [4:0]                signalPowerExp;
wire               [10:0]               signalPowerMant;
wire               [11:0]               ExtNoisePowerMant;
wire               [11:0]               ExtSignalPowerMant;

reg                                     rValueEquEn;
wire                                    SRTClear;
wire                                    SRTEn;
wire               [11:0]               Qf;
wire                                    QfValid;
wire               [6:0]                NextRatioShift;
`endif

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module is the top level for the TDFO estimation and compensation blocks
// It instantiates TDFOCorrVal, TDFOPlatDet, TDFOCntrl and TDFOCmp  modules, 
// as well as the logic needed for SNR computation.
// This module also controls the functioning of the H Memory by muxing signals
// coming from MIMO Equalizer, Channel Estimation and the TDFOCntrl.

// Instantiate TDFOCorrVal
TDFOCorrVal # (
               // Parameters
               .TDFOIPWIDTH(TDFOIPWIDTH),
               .TDFOCVWIDTH(TDFOCVWIDTH)
              ) U_CorrVal (

                           // Inputs
                           .PhyClk(TDFOEstGClk),
                           .nPhyRst(nPhyRst),
                           .CorrEn(CorrEn),
                           .MovAvgEn(MovAvgEn),
                           .TDFOEstMode(TDFOEstMode),
                           .MovAvgClr(MovAvgClr),
                           .DataValid(CVDataValid),
                           .DataInReRx0(CVDataInReRx0),
                           .DataInImRx0(CVDataInImRx0),
`ifdef RW_NX_DERIV_PATH1
                           .DataInReRx1(CVDataInReRx1Inter),
                           .DataInImRx1(CVDataInImRx1Inter),
`endif
                           .DataDBufReRx0(CVDataDBufReRx0),
                           .DataDBufImRx0(CVDataDBufImRx0),
`ifdef RW_NX_DERIV_PATH1
                           .DataDBufReRx1(CVDataDBufReRx1Inter),
                           .DataDBufImRx1(CVDataDBufImRx1Inter),
`endif
                           // Outputs
`ifdef RW_NX_TDFO_NSR_EN
                           // NSR computation
                           .corrLTFsReRx0(corrLTFsReRx0),
                           .corrLTFsImRx0(corrLTFsImRx0),
`ifdef RW_NX_DERIV_PATH1
                           
                           .corrLTFsReRx1(corrLTFsReRx1),
                           .corrLTFsImRx1(corrLTFsImRx1),
`endif
                           .totalPower   (totalPower),
`endif
                           .CorrValOutValid(CorrValOutValid),
                           .CorrValOutRe(CorrValOutRe),
                           .CorrValOutIm(CorrValOutIm)
                           );

`ifdef RW_NX_TDFO_NSR_EN

// TDFOCordicMagRotTop for NSR computation
// The SqrtOf2Sqrs block must be enabled after Fine FO estimation, up to Done received.
// If NRX=2, it is enabled a second time to accumulate the power of the two antennas.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CordicMagEn_Blk
      if (nPhyRst == 1'b0) begin
         CorrEnD           <= 1'b0;
         CordicMagEn       <= 1'b0;
         ModulusValidD     <= 1'b0;
         corrLTFsRe        <= {(TDFOIPWIDTH*2){1'b0}};
         corrLTFsIm        <= {(TDFOIPWIDTH*2){1'b0}};
`ifdef RW_NX_DERIV_PATH1
         CordicMagRx0Done  <= 1'b0;
         SumSignalPower    <= {(TDFOIPWIDTH*2+1){1'b0}};
`endif
      end else begin
         CorrEnD           <= CorrEn;
         CordicMagEn       <= 1'b0;        // Pulse as only one value to compute
         ModulusValidD     <= ModulusValid;
         // Start for RX0: at end of fine FO estimation
         if ((!CorrEn && CorrEnD) && (TDFOEstModeD!=COARSE)) begin
            CordicMagEn    <= 1'b1;
            corrLTFsRe     <= corrLTFsReRx0;
            corrLTFsIm     <= corrLTFsImRx0;
`ifdef RW_NX_DERIV_PATH1
            CordicMagRx0Done <= 1'b0;
            SumSignalPower <= {(TDFOIPWIDTH*2+1){1'b0}};
         // Start for RX1: after Sqrt computation for RX path 0
         end else if ((!ModulusValid && ModulusValidD) && (CordicMagRx0Done==1'b0)) begin
            CordicMagRx0Done       <= 1'b1;
            CordicMagEn    <= 1'b1;
            corrLTFsRe     <= corrLTFsReRx1;
            corrLTFsIm     <= corrLTFsImRx1;
         end else if (ModulusValid == 1'b1) begin
            // SumSignalPower can have same number of bits as Modulus
            // because Cordic inputs were extended with 1 bit to match iteration number in matlab
            SumSignalPower <= SumSignalPower + Modulus;
`endif
         end
      end
   end //CordicMagEn_Blk


`ifndef RW_NX_DERIV_PATH1
// Only one antenna: no need to accumulate Modulus values
assign SumSignalPower = Modulus;
`endif

// Extend inputs to match matlab iteration number
assign ExtCorrLTFsRe = {corrLTFsRe[TDFOIPWIDTH*2-1],corrLTFsRe};
assign ExtCorrLTFsIm = {corrLTFsIm[TDFOIPWIDTH*2-1],corrLTFsIm};

// Compute the modulus using a Cordic
CordicMagTop # (               // Matlab: CordicMagnitudeVec
  .DATAWIDTH(TDFOIPWIDTH*2+1), // Matlab: NbitI 
  .ANGLEWIDTH(16),             // Matlab: NfracP+3
  .SCALING(1)
) U_TDFOCordicMagTop (
  .nPhyRst(    nPhyRst),
  .PhyClk(     PhyClk),
  .DataInEn(   CordicMagEn),
  .ReDataIn(   ExtCorrLTFsRe),
  .ImDataIn(   ExtCorrLTFsIm),
  .CordicDone( ModulusValid),
  .ModOut(     Modulus)
 );


// register noisePower and Start SRT block
always @ (posedge PhyClk or negedge nPhyRst)
   begin
      if (nPhyRst == 1'b0) begin
         noisePower     <= 23'd0;
         rValueEquEn    <= 1'b0;
      end else begin
         rValueEquEn    <= 1'b0; // pulse, only one value to compute
         if (!ModulusValid && ModulusValidD
`ifdef RW_NX_DERIV_PATH1
           && CordicMagRx0Done // 2 Rx paths: start SRT only after noise for both antennae is valid
`endif
            ) begin
            noisePower  <= {totalPower[21],totalPower} - {SumSignalPower[TDFOIPWIDTH*2+1],SumSignalPower}; // Updated after each antenna computation
            rValueEquEn <= 1'b1;
         end
      end
   end

// noisePower = max(noisePower,1);
assign MaxNoisePower1 = ((noisePower == 23'd0) || noisePower[22] ) ? 23'd1 : noisePower;

// Convert noise power to Mantissa + exponant format
rValueEqu # (
  .INPUTPOS    (1),
  .INBITS      (23),
  .NSATBITSIN  (22),
  .NSATBITSOUT (10)
) U_rValueEqu_Noise (
  .nPhyRst(nPhyRst),
  .PhyClk(PhyClk),
  .Data(MaxNoisePower1),
  .DataValidIn(rValueEquEn),
  .DataExp(noisePowerExp),   // belongs to [-(NSATBITSOUT-1), NSATBITSIN-(NSATBITSOUT-1)] i.e. [-9, 13]
  .DataMant(noisePowerMant),
  .DataValidOut(SRTEn)
);

// Convert signal power to Mantissa + exponant format
rValueEqu # (
  .INPUTPOS    (0),
  .INBITS      (22),
  .NSATBITSIN  (22),
  .NSATBITSOUT (11)
) U_rValueEqu_Signal (
  .nPhyRst(nPhyRst),
  .PhyClk(PhyClk),
  .Data(SumSignalPower),
  .DataValidIn(rValueEquEn),
  .DataExp(signalPowerExp),   // belongs to [-(NSATBITSOUT-1), NSATBITSIN-(NSATBITSOUT-1)] i.e. [-10, 12]
  .DataMant(signalPowerMant),
  .DataValidOut()
);



//Af    = noisePower_Mantis;
//Na    = nBitsNSRPrec;
//Bf    = signalPower_Mantis;
//Nb    = nBitsNSRPrec+1;
//Nq    = nBitsNSRPrec+1;
//NqINT = 1; 
//Nqn   = 2*Nq - 1; 
//Nrn   = 2*Nq - 1; 
//Qf    = DividerSRT(Af, Na, Bf, Nb, Nq, NqINT, Nqn, Nrn);

assign ExtSignalPowerMant = {signalPowerMant[10],signalPowerMant};
assign ExtNoisePowerMant  = {{2{noisePowerMant[9]}},noisePowerMant};

assign SRTClear           = ~TDFOEn;

SrtDivider #(
  .NBITINPUT               (12), // = Na+1 = Nb+1 (min 3)
  .NBITQUOTIENT            (12), // = Nq+1 (min 2)
  .NINTBITQUOTIENT         (1),  // = NqINT. 0 to min(NBITQUOTIENT-1, NBITINPUT-1)
  .NBITQUOTIENTSRT         (21), // = Nqn. Warning: Nq < Nqn < 2Nq
  .NBITPREMSRT             (21), // = Nrn. Warning: Nq < Nqn < 2Nq
  .SIZEBITSHIFTSRTQUOTIENT (4)   // size in bits of -NBITINPUT-NINTBITQUOTIENT+2
) U_SrtDivider (
  // Clock and reset
  .PhyClk       (PhyClk),
  .nPhyRst      (nPhyRst),
  //Data & Control
  .Clear        (SRTClear),
  .Dividend     (ExtNoisePowerMant),  // Af
  .Divisor      (ExtSignalPowerMant), // Bf
  .DataEnIn     (SRTEn),
  // Result
  .Quotient     (Qf),
  .DataEnOut    (QfValid)
);


// RxParameter.NoiseEst{AlgoParameter.iiAlgo}.nbits_right_shift  = (Nq-NqINT) - (noisePower_Expo-signalPower_Expo);
// belongs to [-13, 31]
assign NextRatioShift = {{2{signalPowerExp[4]}},signalPowerExp}
                      - {{2{noisePowerExp[4]}},noisePowerExp}
                      + 7'd10;

always @ (posedge PhyClk or negedge nPhyRst)
   begin
      if (nPhyRst == 1'b0) begin
         RatioNoiseSignal <= 12'd0;
         RatioShift       <= 6'd0;
      end else begin
         if (QfValid==1'b1) begin
            RatioNoiseSignal <= Qf;
            RatioShift       <= NextRatioShift[5:0];
         end
      end
   end
`endif // RW_NX_TDFO_NSR_EN

// Instantiate TDFOPlatDet
TDFOPlatDet #(
              //Parameters
              .TDFOCVWIDTH(TDFOCVWIDTH)
             ) U_PlatDet (
                          // Inputs
                          .PhyClk(TDFOEstGClk),
                          .nPhyRst(nPhyRst),
                          .AutoCorrCompareRatioLow(AutoCorrCompareRatioLow),
                          .AutoCorrCompareRatioHigh(AutoCorrCompareRatioHigh),
                          .AutoCorrFallCount(AutoCorrFallCount),
                          .CorrValValid(CorrValOutValid),
                          .CorrValRe(CorrValOutRe),
                          .CorrValIm(CorrValOutIm),
                          .CorrValAccEn(CorrValAccEn),
                          .CorrValAccClr(MovAvgClr),
                          .AbsCValAccEn(AbsCValAccEn),
                          .PlatFallLowDetEn(PlatFallLowDetEn),
                          .PlatFallHighDetEn(PlatFallHighDetEn),
                          .ModApproxCompEn(ModApproxCompEn),
                          // Outputs
                          .PlatFallLowP(PlatFallPInt),
                          .PlatFallHighP(PlatFallHighP),
                          .PlatThreshDoneP(PlatThreshDoneP),
                          .CorrValAccRe(CorrValAccRe),
                          .CorrValAccIm(CorrValAccIm),
                          .CorrValAccValidP(CorrValAccValidP)
                         );

// Instantiate TDFOCmp
TDFOCmp #(
          //Parameters
          .NPIPE(NPIPE),
          .NCOMBSTG(NCOMBSTG),
          .SCALING(SCALING),
          .CROTDATAWIDTH(MEMDATAWIDTH),
          .CROTANGLEWIDTH(ANGLEWIDTH+4),
          .CVECTANGLEWIDTH(ANGLEWIDTH),
          .CVECTDATAWIDTH(TDFOCVWIDTH)
         ) U_TDFOCmp (
                      .PhyClk(PhyClk),
                      .nPhyRst(nPhyRst),
                      .CVectDataEnIn(CVectDataEnIn),
                      .CVReDataIn(CVReDataIn),
                      .CVImDataIn(CVImDataIn),
                      .CVAngleOut(CVAngleOut),
                      .CVectDone(CVectDone),
                      //
                      .TDFOCmpMode(TDFOCmpMode),
                      .CRotEnable(CRotEnable),
`ifndef RW_NX_DERIV_CHBW20ONLY
                      // Secondary data 20MHz
                      .CRot0ReDataIn20S(CRot0ReDataIn20S),
                      .CRot0ImDataIn20S(CRot0ImDataIn20S),
                      .CRot0ReDataOut20S(CRot0ReDataOut20S),
                      .CRot0ImDataOut20S(CRot0ImDataOut20S),
                      .CRot1ReDataIn20S(CRot1ReDataIn20S),
                      .CRot1ImDataIn20S(CRot1ImDataIn20S),
                      .CRot01ReDataOut20S(CRot01ReDataOut20S),
                      .CRot01ImDataOut20S(CRot01ImDataOut20S),
`endif
                      // Primary data
                      .CRot0DataInValid(CRot0DataInValid),
                      .CRot1DataInValid(CRot1DataInValid),
                      .CRot0AngleIn(CRot0Angle),
                      .CRot0ReDataIn0(CRot0ReDataIn0),
                      .CRot0ImDataIn0(CRot0ImDataIn0),
`ifdef RW_NX_DERIV_PATH1
                      .CRot0ReDataIn1(CRot0ReDataIn1),
                      .CRot0ImDataIn1(CRot0ImDataIn1),
`endif
                      .CRot1AngleIn(CRot1Angle),
                      .CRot1ReDataIn0(CRot1ReDataIn0),
                      .CRot1ImDataIn0(CRot1ImDataIn0),
`ifdef RW_NX_DERIV_PATH1
                      .CRot1ReDataIn1(CRot1ReDataIn1),
                      .CRot1ImDataIn1(CRot1ImDataIn1),
`endif

                      .CRot0ReDataOut0(CRot0ReDataOut0),
                      .CRot0ImDataOut0(CRot0ImDataOut0),
`ifdef RW_NX_DERIV_PATH1
                      .CRot0ReDataOut1(CRot0ReDataOut1),
                      .CRot0ImDataOut1(CRot0ImDataOut1),
`endif
                      .CRot0DataOutValid(CRot0DataOutValid),
                      .CRot01ReDataOut0(CRot01ReDataOut0),
                      .CRot01ImDataOut0(CRot01ImDataOut0),
`ifdef RW_NX_DERIV_PATH1
                      .CRot01ReDataOut1(CRot01ReDataOut1),
                      .CRot01ImDataOut1(CRot01ImDataOut1),
`endif
                      .CRot01DataOutValid(CRot01DataOutValid)
                     );

// Instantiate TDFOCntrl
TDFOCntrl #(
           // Parameters
           .ANGLEWIDTH(ANGLEWIDTH)
           ) U_TDFOCntrl (
                          // Inputs
                          .PhyClk(PhyClk),
                          .nPhyRst(nPhyRst),
                          .CVectDone(CVectDone),
                          .PlatFallLowP(PlatFallPInt),
                          .PlatFallHighP(PlatFallHighP),
                          .PlatThreshDoneP(PlatThreshDoneP),
                          .CorrValAccValidP(CorrValAccValidP),
                          .DataValid(RxDataValid),
                          .Bandwidth(Bandwidth),
                          .TDFOStrtP(TDFOStrtP),
                          .Force20(Force20),
                          .Force40(Force40),
                          .TDFOEn(TDFOEn),
                          .CVAngleOut(CVAngleOut),
                          .DataCmpStrtP(DataCmpStrtP),
                          .FDCpeSlope(FDCpeSlope),
                          .TBECount(TBECount),
                          .TBECountValid(TBECountValid),
                          .CRot0DataOutValid(CRot0DataOutValid),
                          .CRot01DataOutValid(CRot01DataOutValid),
                          .AutoCorrPlatSumStart(AutoCorrPlatSumStart),
                          .AutoCorrPlatSumEnd(AutoCorrPlatSumEnd),
                          .AutoCorrPlatFallSearchStart(AutoCorrPlatFallSearchStart),
                          .AutoCorrPlatFallIndexOffset(AutoCorrPlatFallIndexOffset),
                          .AutoCorrPhSumStart(AutoCorrPhSumStart),
                          .AutoCorrPhSumEnd(AutoCorrPhSumEnd),
                          .TDFOCompEn(TDFOCompEn),
                          .TDFOFineEstEn(TDFOFineEstEn),
                          .TDFOCpeSlopeEn(TDFOCpeSlopeEn),

                          // Outputs
                          .CoarseFOAngle(CoarseFOAngle),
                          .FineFOAngle(FineFOAngle),
                          .FOAngle(FOAngle),
                          .CorValFineDataEn(CorValFineDataEn),
                          .CVectDataEnIn(CVectDataEnIn),
                          .CVectOngoing(CVectOngoing),
                          .CRotEnable(CRotEnable),
                          .CRot0DataInValid(CRot0DataInValid),
                          .CRot1DataInValid(CRot1DataInValid),
                          .CorrEn(CorrEn),
                          .MovAvgEn(MovAvgEn),
                          .TDFOEstMode(TDFOEstMode),
                          .TDFOCmpMode(TDFOCmpMode),
                          .TDFOLTFCmpMode(TDFOLTFCmpMode),
                          .TDFOCmpDataMode(TDFOCmpDataMode),
                          .RxLTFValid(RxLTFValid),
                          .MovAvgClr(MovAvgClr),
                          .CorrValAccEn(CorrValAccEn),
                          .AbsCValAccEn(AbsCValAccEn),
                          .PlatFallLowDetEn(PlatFallLowDetEn),
                          .PlatFallHighDetEn(PlatFallHighDetEn),
                          .ModApproxCompEn(ModApproxCompEn),
                          .CmpLTFDoneP(CmpLTFDoneP),
                          .FineFODone(FineFODone),
                          .CRot0Angle(CRot0Angle),
                          .CRot1Angle(CRot1Angle),
                          .FFTMemWrEn(FFTMemWrEn),
                          .FFTMemSel(FFTMemSel),
                          .MemWrEn(TDFOMemWrEn),
                          .MemWrSel(TDFOMemWrSel),
                          .MemRdEn(TDFOMemRdEn),
                          .MemRdAddr(TDFOMemRdAddr),
                          .MemWrAddr(TDFOMemWrAddr),
                          .MemRdColOut(MemRdColOut)
                          );

// Generate Inputs to TDFOCorrVal
// Input Data Valid
// During Coarse estimation, the data valid will be a 3 clock delayed version
// of the incoming RxDataValid. This is required because of the delay involved
// in reading the corresponding sample (for correlation) from the TDFO Memory.
// In case of Fine estimation, it will be a 2 clock delayed version of the
// Memory Read Enable signal
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CVDataValid_Blk
      if (nPhyRst == 1'b0) begin
         RxDataValid1D <= 1'b0;
         RxDataValid2D <= 1'b0;
         CVDataValid   <= 1'b0;
      end
      else begin
         RxDataValid1D <= RxDataValid;
         RxDataValid2D <= RxDataValid1D;
         if (TDFOEstMode == COARSE)
            CVDataValid <= RxDataValid2D;
         else
            CVDataValid <= CorValFineDataEn;
      end
   end // CVDataValid_Blk

// Data
// During Coarse estimation a 3 clock delayed version of the incoming data 
// and the registered version of the TBE delay line is sent 
// to the TDFOCorrVal block.
// During Fine estimation, the data coming from the TBE is sent to 
// the block after registering. Note that L1 needs to be conjugated inside, so
// the column corresponding to L1 goes to the buffered input of CorrVal 
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CVDataIn_Blk0
      if (nPhyRst == 1'b0) begin
         CVDataInReRx0   <= CONST_ZERO_TDFOIPWIDTH;
         CVDataInImRx0   <= CONST_ZERO_TDFOIPWIDTH;
         CVDataDBufReRx0 <= CONST_ZERO_TDFOIPWIDTH;
         CVDataDBufImRx0 <= CONST_ZERO_TDFOIPWIDTH;
      end
      else if (TDFOEstMode == COARSE) begin
         if (RxDataValid2D == 1'b1) begin
            CVDataInReRx0   <= SharedSatRe0;
            CVDataInImRx0   <= SharedSatIm0;
            CVDataDBufReRx0 <= TBE16DRxI0;
            CVDataDBufImRx0 <= TBE16DRxQ0;
         end // RxDataValid2D == 1'b1
      end // TDFOEstMode == COARSE
      else begin //TDFOEstMode == FINE
         if (CorValFineDataEn == 1'b1) begin
            // These inputs correspond to L1 and L2
            // L2 current inputs
            CVDataInReRx0 <= SharedSatRe0;
            CVDataInImRx0 <= SharedSatIm0;

            // L1 delayed inputs from delay line
            CVDataDBufReRx0 <= TBE64DRxI0;
            CVDataDBufImRx0 <= TBE64DRxQ0;
         end // CorValFineDataEn == 1'b1
      end // TDFOEstMode == FINE
   end // CVDataIn_Blk0

`ifdef RW_NX_DERIV_PATH1
     assign CVDataInReRx1Inter   = CVDataInReRx1;
     assign CVDataInImRx1Inter   = CVDataInImRx1;
     assign CVDataDBufReRx1Inter = CVDataDBufReRx1;
     assign CVDataDBufImRx1Inter = CVDataDBufImRx1;

     always @ (posedge PhyClk or negedge nPhyRst)
        begin: CVDataIn_Blk1
           if (nPhyRst == 1'b0) begin
              CVDataInReRx1   <= CONST_ZERO_TDFOIPWIDTH;
              CVDataInImRx1   <= CONST_ZERO_TDFOIPWIDTH;
              CVDataDBufReRx1 <= CONST_ZERO_TDFOIPWIDTH;
              CVDataDBufImRx1 <= CONST_ZERO_TDFOIPWIDTH;
           end
           else if (TDFOEstMode == COARSE) begin
              if (RxDataValid2D == 1'b1) begin
                 CVDataInReRx1   <= SharedSatRe1;
                 CVDataInImRx1   <= SharedSatIm1;
                 CVDataDBufReRx1 <= TBE16DRxI1;
                 CVDataDBufImRx1 <= TBE16DRxQ1;
              end // RxDataValid2D == 1'b1
           end // TDFOEstMode == COARSE
           else begin  // TDFOEstMode == FINE
              if (CorValFineDataEn == 1'b1) begin
                 // L2 current inputs
                 CVDataInReRx1 <= SharedSatRe1;
                 CVDataInImRx1 <= SharedSatIm1;

                 // L1 delayed inputs from delay line
                 CVDataDBufReRx1 <= TBE64DRxI1;
                 CVDataDBufImRx1 <= TBE64DRxQ1;
              end // CorValFineDataEn == 1'b1
           end // TDFOEstMode == 1'b1
        end // CVDataIn_Blk1
`endif

// Delayed version of RxDataIn
always @ (posedge PhyClk or negedge nPhyRst)
   begin: RxDataIn_0_Blk
      if (nPhyRst == 1'b0) begin
         RxDataInRe0D1   <= CONST_ZERO_MEMDATAWIDTH;
         RxDataInIm0D1   <= CONST_ZERO_MEMDATAWIDTH;
         RxDataInRe0D2   <= CONST_ZERO_MEMDATAWIDTH;
         RxDataInIm0D2   <= CONST_ZERO_MEMDATAWIDTH;
         RxDataIn20SReD2 <= CONST_ZERO_MEMDATAWIDTH;
         RxDataIn20SImD2 <= CONST_ZERO_MEMDATAWIDTH;
         RxDataIn20SReD1 <= CONST_ZERO_MEMDATAWIDTH;
         RxDataIn20SImD1 <= CONST_ZERO_MEMDATAWIDTH;
      end
      else begin
        if (RxDataValid) begin
           RxDataInRe0D1   <= RxDataInRe0;
           RxDataInIm0D1   <= RxDataInIm0; 
           RxDataIn20SReD1 <= RxDataIn20SRe;
           RxDataIn20SImD1 <= RxDataIn20SIm;
        end
        if (RxDataValid1D) begin
           RxDataInRe0D2   <= RxDataInRe0D1;
           RxDataInIm0D2   <= RxDataInIm0D1; 
           RxDataIn20SReD2 <= RxDataIn20SReD1;
           RxDataIn20SImD2 <= RxDataIn20SImD1;
        end
      end
   end //RxDataIn_0_Blk

`ifdef RW_NX_DERIV_PATH1
      always @ (posedge PhyClk or negedge nPhyRst)
         begin: RxDataIn_1_Blk
            if (nPhyRst == 1'b0) begin
               RxDataInRe1D1 <= CONST_ZERO_MEMDATAWIDTH;
               RxDataInIm1D1 <= CONST_ZERO_MEMDATAWIDTH;
               RxDataInRe1D2 <= CONST_ZERO_MEMDATAWIDTH;
               RxDataInIm1D2 <= CONST_ZERO_MEMDATAWIDTH;
            end
            else begin
              if (RxDataValid) begin
                 RxDataInRe1D1 <= RxDataInRe1;
                 RxDataInIm1D1 <= RxDataInIm1;
              end
              if (RxDataValid1D) begin
                 RxDataInRe1D2 <= RxDataInRe1D1;
                 RxDataInIm1D2 <= RxDataInIm1D1; 
              end
            end
         end // RxDataIn_1_Blk
`endif


// Write Data for Memory from TDFO Block
// Write Data for Port 1
// Memory is used for storing the long preamble samples which will be used
// for compensation during COARSE+FINE mode.
// All samples coming in from the beginning are stored until the end
// of L1 and L2 compensation.
// This is to make sure we have sufficient samples for estimation
// under all circumstances.
// Until the COARSE+FINE angle is available samples are compensated.
// TDFOMemWrData will be a 2 clock delayed version of the CRot0 Output to
// ensure that it is aligned correctly to the address.
// The same data can be duplicated on both columns (of every chain), because
// the write enable will ensure that it is written only to the correct column

assign TDFOMemWrDataRe0 = RxDataInRe0D2;
assign TDFOMemWrDataIm0 = RxDataInIm0D2;
`ifdef RW_NX_DERIV_PATH1
assign TDFOMemWrDataRe1 = RxDataInRe1D2;
assign TDFOMemWrDataIm1 = RxDataInIm1D2;
`endif

// Secondary data 20MHz
`ifndef RW_NX_DERIV_CHBW20ONLY
assign TDFOMemWrData20SRe = RxDataIn20SReD2;
assign TDFOMemWrData20SIm = RxDataIn20SImD2;
`else
assign TDFOMemWrData20SRe = CONST_ZERO_MEMDATAWIDTH;
assign TDFOMemWrData20SIm = CONST_ZERO_MEMDATAWIDTH;
assign CRot0ReDataOut20S  = CONST_ZERO_MEMDATAWIDTH;
assign CRot0ImDataOut20S  = CONST_ZERO_MEMDATAWIDTH;
assign CRot01ReDataOut20S = CONST_ZERO_MEMDATAWIDTH;
assign CRot01ImDataOut20S = CONST_ZERO_MEMDATAWIDTH;
`endif


// TBE Control Signals
assign RxLTFRe0 = CRot0ReDataOut0D1;
assign RxLTFIm0 = CRot0ImDataOut0D1;
`ifdef RW_NX_DERIV_PATH1
  assign RxLTFRe1 = CRot0ReDataOut1D1;
  assign RxLTFIm1 = CRot0ImDataOut1D1;
`endif


// Generate input data for CordicRot0
// Muxing CordicRot0 input as it is shared between coarse compensation and
// L1+L2 coarse+fine compensation
assign CRot0ReDataIn0 = (TDFOCmpMode == 1'b1)   ? CRot0MuxReDataIn0 : 
                                                  RxDataInRe0D2;
assign CRot0ImDataIn0 = (TDFOCmpMode == 1'b1)   ? CRot0MuxImDataIn0 : 
                                                  RxDataInIm0D2;
`ifdef RW_NX_DERIV_PATH1
assign CRot0ReDataIn1 = (TDFOCmpMode == 1'b1)   ? CRot0MuxReDataIn1 :
                                                  RxDataInRe1D2;
assign CRot0ImDataIn1 = (TDFOCmpMode == 1'b1)   ? CRot0MuxImDataIn1 :
                                                  RxDataInIm1D2;
`endif
assign CRot0ReDataIn20S = (TDFOCmpMode == 1'b1) ? CRot0MuxReDataIn20S : 
                                                  RxDataIn20SReD2;
assign CRot0ImDataIn20S = (TDFOCmpMode == 1'b1) ? CRot0MuxImDataIn20S : 
                                                  RxDataIn20SImD2;

// Generate input data for CordicRot1
always @ (posedge PhyClk or negedge nPhyRst)
   begin:CRot01DataIn_Blk
      if (nPhyRst == 1'b0) begin
         CRot1ReDataIn0      <= CONST_ZERO_MEMDATAWIDTH;
         CRot1ImDataIn0      <= CONST_ZERO_MEMDATAWIDTH;
         CRot0MuxReDataIn0   <= CONST_ZERO_MEMDATAWIDTH;
         CRot0MuxImDataIn0   <= CONST_ZERO_MEMDATAWIDTH;
`ifdef RW_NX_DERIV_PATH1
         CRot1ReDataIn1      <= CONST_ZERO_MEMDATAWIDTH;
         CRot1ImDataIn1      <= CONST_ZERO_MEMDATAWIDTH;
         CRot0MuxReDataIn1   <= CONST_ZERO_MEMDATAWIDTH;
         CRot0MuxImDataIn1   <= CONST_ZERO_MEMDATAWIDTH;
`endif
         CRot1ReDataIn20S    <= CONST_ZERO_MEMDATAWIDTH;
         CRot1ImDataIn20S    <= CONST_ZERO_MEMDATAWIDTH;
         CRot0MuxReDataIn20S <= CONST_ZERO_MEMDATAWIDTH;
         CRot0MuxImDataIn20S <= CONST_ZERO_MEMDATAWIDTH;
      end
      else if (MemRdColOut == 1'b1) begin
         // L2 belongs to first column, L1 belongs to second column
         CRot1ReDataIn0      <= TDFOMem1RdDataRe0;
         CRot1ImDataIn0      <= TDFOMem1RdDataIm0;
         CRot0MuxReDataIn0   <= TDFOMem0RdDataRe0;
         CRot0MuxImDataIn0   <= TDFOMem0RdDataIm0;
`ifdef RW_NX_DERIV_PATH1
         CRot1ReDataIn1      <= TDFOMem1RdDataRe1;
         CRot1ImDataIn1      <= TDFOMem1RdDataIm1;
         CRot0MuxReDataIn1   <= TDFOMem0RdDataRe1;
         CRot0MuxImDataIn1   <= TDFOMem0RdDataIm1;
`endif
         CRot1ReDataIn20S    <= TDFOMem1RdData20SRe;
         CRot1ImDataIn20S    <= TDFOMem1RdData20SIm;
         CRot0MuxReDataIn20S <= TDFOMem0RdData20SRe;
         CRot0MuxImDataIn20S <= TDFOMem0RdData20SIm;
      end // MemRdColOut == 1'b1
      else begin
         // L1 belongs to first column, L2 belongs to second column
         CRot1ReDataIn0      <= TDFOMem0RdDataRe0;
         CRot1ImDataIn0      <= TDFOMem0RdDataIm0;
         CRot0MuxReDataIn0   <= TDFOMem1RdDataRe0;
         CRot0MuxImDataIn0   <= TDFOMem1RdDataIm0;
`ifdef RW_NX_DERIV_PATH1
         CRot1ReDataIn1      <= TDFOMem0RdDataRe1;
         CRot1ImDataIn1      <= TDFOMem0RdDataIm1;
         CRot0MuxReDataIn1   <= TDFOMem1RdDataRe1;
         CRot0MuxImDataIn1   <= TDFOMem1RdDataIm1;
`endif
         CRot1ReDataIn20S    <= TDFOMem0RdData20SRe;
         CRot1ImDataIn20S    <= TDFOMem0RdData20SIm;
         CRot0MuxReDataIn20S <= TDFOMem1RdData20SRe;
         CRot0MuxImDataIn20S <= TDFOMem1RdData20SIm;
      end // MemRdColOut == 1'b0
   end // CRot01DataIn_Blk

// Generate CordicVect Inputs
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CVectIn_Blk
      if (nPhyRst == 1'b0) begin
         CVReDataIn <= CONST_ZERO_TDFOCVWIDTH;
         CVImDataIn <= CONST_ZERO_TDFOCVWIDTH;
      end
      else if (!CVectOngoing && PlatFallHighDetEn && (CorrValAccValidP == 1'b1 || PlatFallHighP == 1'b1)) begin
         CVReDataIn <= CorrValAccRe;
         CVImDataIn <= CorrValAccIm;
      end
      else if (TDFOEstMode == COARSE && TDFOEstModeD != COARSE) begin
         CVReDataIn <= CorrValOutRe;
         CVImDataIn <= CorrValOutIm;
      end
   end // CVectIn_Blk

// Delayed version of TDFOEstMode
always @ (posedge PhyClk or negedge nPhyRst)
   begin: TDFOEstModeD_Blk
      if (nPhyRst == 1'b0)
         TDFOEstModeD <= 1'b0;
      else
         TDFOEstModeD <= TDFOEstMode;
   end // TDFOEstModeD_Blk

// Resync versions of the CRot0 saturation output for TBE
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CRot0DataOut0_Blk
      if (nPhyRst == 1'b0) begin
         CRot0ReDataOut0D1 <= CONST_ZERO_TDFOIPWIDTH;
         CRot0ImDataOut0D1 <= CONST_ZERO_TDFOIPWIDTH;
      end
      else begin
         CRot0ReDataOut0D1 <= SharedSatRe0;
         CRot0ImDataOut0D1 <= SharedSatIm0;
      end
   end // CRot0DataOut0_Blk

`ifdef RW_NX_DERIV_PATH1
      always @ (posedge PhyClk or negedge nPhyRst)
         begin: CRot0DataOut1_Blk
            if (nPhyRst == 1'b0) begin
               CRot0ReDataOut1D1 <= CONST_ZERO_TDFOIPWIDTH;
               CRot0ImDataOut1D1 <= CONST_ZERO_TDFOIPWIDTH;
            end
            else begin
               CRot0ReDataOut1D1 <= SharedSatRe1;
               CRot0ImDataOut1D1 <= SharedSatIm1;
            end
         end // CRot0DataOut1_Blk

`endif

// Generate outputs for FFT Memory
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FFTDataOut_Blk
      if (nPhyRst == 1'b0) begin
         FFTMemWrDataRe0   <= {{MEMDATAWIDTH}{1'b0}};
         FFTMemWrDataIm0   <= {{MEMDATAWIDTH}{1'b0}};
`ifdef RW_NX_DERIV_PATH1
         FFTMemWrDataRe1   <= {{MEMDATAWIDTH}{1'b0}};
         FFTMemWrDataIm1   <= {{MEMDATAWIDTH}{1'b0}};
`endif
         FFTMemWrData20SRe <= {{MEMDATAWIDTH}{1'b0}};
         FFTMemWrData20SIm <= {{MEMDATAWIDTH}{1'b0}};
      end
      else if (TDFOCmpMode == 1'b1) begin
         FFTMemWrDataRe0   <= CRot01ReDataOut0;
         FFTMemWrDataIm0   <= CRot01ImDataOut0;
`ifdef RW_NX_DERIV_PATH1
         FFTMemWrDataRe1   <= CRot01ReDataOut1;
         FFTMemWrDataIm1   <= CRot01ImDataOut1;
`endif
         FFTMemWrData20SRe <= CRot01ReDataOut20S;
         FFTMemWrData20SIm <= CRot01ImDataOut20S;
      end
      else begin
         FFTMemWrDataRe0   <= CRot0ReDataOut0;
         FFTMemWrDataIm0   <= CRot0ImDataOut0;
`ifdef RW_NX_DERIV_PATH1
         FFTMemWrDataRe1   <= CRot0ReDataOut1;
         FFTMemWrDataIm1   <= CRot0ImDataOut1;
`endif
         FFTMemWrData20SRe <= CRot0ReDataOut20S;
         FFTMemWrData20SIm <= CRot0ImDataOut20S;
      end
   end // FFTDataOut_Blk

// Sharing Rounded values of inputs for Coarse/Fine Estimation
assign SharedRndInRe0 = (TDFOLTFCmpMode == 1'b0) ? RxDataInRe0D2 : CRot0ReDataOut0;
assign SharedRndInIm0 = (TDFOLTFCmpMode == 1'b0) ? RxDataInIm0D2 : CRot0ImDataOut0;

Round #(
        .INPUT_WIDTH(MEMDATAWIDTH),
        .OUTPUT_WIDTH(TDFOIPWIDTH)
       )
        U_ROUNDSHAREDRE0(
                   .InputData(SharedRndInRe0),
                   .RoundData(SharedRndOutRe0)
                   );

Round #(
        .INPUT_WIDTH(MEMDATAWIDTH),
        .OUTPUT_WIDTH(TDFOIPWIDTH)
       )
        U_ROUNDSHAREDIM0(
                   .InputData(SharedRndInIm0),
                   .RoundData(SharedRndOutIm0)
                   );

`ifdef RW_NX_DERIV_PATH1
      assign SharedRndInRe1 = (TDFOLTFCmpMode == 1'b0) ? RxDataInRe1D2 : CRot0ReDataOut1;
      assign SharedRndInIm1 = (TDFOLTFCmpMode == 1'b0) ? RxDataInIm1D2 : CRot0ImDataOut1;

      Round #(
              .INPUT_WIDTH(MEMDATAWIDTH),
              .OUTPUT_WIDTH(TDFOIPWIDTH)
             )
              U_ROUNDSHAREDRE1(
                         .InputData(SharedRndInRe1),
                         .RoundData(SharedRndOutRe1)
                         );

      Round #(
              .INPUT_WIDTH(MEMDATAWIDTH),
              .OUTPUT_WIDTH(TDFOIPWIDTH)
             )
              U_ROUNDSHAREDIM1(
                         .InputData(SharedRndInIm1),
                         .RoundData(SharedRndOutIm1)
                         );
`endif

// Sharing Symmetric Saturate of inputs to Coarse/Fine Estimation
SatEquSigned #(
               .INPUT_WIDTH(TDFOIPWIDTH),
               .OUTPUT_WIDTH(TDFOIPWIDTH)
              )
               U_SATSYMSHAREDRE0(
                        .InputData(SharedRndOutRe0),
                        .SatSymData(SharedSatRe0)
                        );

SatEquSigned #(
               .INPUT_WIDTH(TDFOIPWIDTH),
               .OUTPUT_WIDTH(TDFOIPWIDTH)
              )
               U_SATSYMSHAREDIM0(
                        .InputData(SharedRndOutIm0),
                        .SatSymData(SharedSatIm0)
                        );

`ifdef RW_NX_DERIV_PATH1
      SatEquSigned #(
                     .INPUT_WIDTH(TDFOIPWIDTH),
                     .OUTPUT_WIDTH(TDFOIPWIDTH)
                    )
                     U_SATSYMSHAREDRE1(
                              .InputData(SharedRndOutRe1),
                              .SatSymData(SharedSatRe1)
                              );

      SatEquSigned #(
                     .INPUT_WIDTH(TDFOIPWIDTH),
                     .OUTPUT_WIDTH(TDFOIPWIDTH)
                    )
                     U_SATSYMSHAREDIM1(
                              .InputData(SharedRndOutIm1),
                              .SatSymData(SharedSatIm1)
                              );
`endif

// Avoiding reading from output ports
assign PlatFallP = PlatFallPInt;

endmodule //TDFOTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
