//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 34361 $
// $Date: 2018-06-22 11:57:21 +0200 (Fri, 22 Jun 2018) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Converts a number into Mantissa + Exponent notation
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL:  $
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

// This module follows the rValueEqu function used in Matlab FineFO_FXPT.m
module rValueEqu # (
  parameter INPUTPOS    = 0,  // 1 if input is positive
  parameter INBITS      = 23,
  parameter NSATBITSIN  = 22,
  parameter NSATBITSOUT = 10  // NSATBITSOUT <= 15
)(

  //Clock and Reset
  input   wire                   nPhyRst, // Active LOW Reset
  input   wire                   PhyClk,  // PHY Clock

  //Data
  input   wire                   DataValidIn,
  input   wire [INBITS-1:0]      Data,

  output  reg                    DataValidOut,
  output  reg  [4:0]             DataExp, // belongs to [-(NSATBITSOUT-1), NSATBITSIN-(NSATBITSOUT-1)] 
  output  reg  [NSATBITSOUT-1:0] DataMant
);

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam [4:0] NSATBITSOUT_M2 = NSATBITSOUT-2; // Set bitwidth to constant for lint
localparam [4:0] NSATBITSOUT_M1 = NSATBITSOUT-1; // Set bitwidth to constant for lint

//////////////////////////////////////////////////////////////////////////////
//  Internal Signals Declarations
//////////////////////////////////////////////////////////////////////////////
wire                  DataSign;
wire [INBITS-1:0]     AbsData;
wire [NSATBITSIN-1:0] SatData;
reg  [NSATBITSIN-1:0] SatDataReg;
reg  [4:0]            DataFSB;
reg  [NSATBITSIN-1:0] DataShift;
wire [NSATBITSIN-1:0] NegDataShift;
reg                   DataValidInt;

integer   i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Take absolute value if needed
generate
if (INPUTPOS==0) begin : AbsGen
  wire [INBITS-1:0]     NegData;

  assign DataSign = Data[INBITS-1];
  assign NegData  = ~Data + {{(INBITS-1){1'b0}},1'b1};
  assign AbsData  = DataSign ? NegData : Data;

end else begin

  assign DataSign = 1'b0;
  assign AbsData  = Data;

end // AbsGen
endgenerate

// Saturate to NSATBITSIN bits if needed
generate
if (NSATBITSIN<INBITS) begin : SatGen

  SatSymSigned #(
     .INPUT_WIDTH(INBITS),
     .OUTPUT_WIDTH(NSATBITSIN)
  ) U_SATCRT0 (
     .InputData(AbsData),
     .SatSymData(SatData)
    );

end else begin

  assign SatData = (AbsData=={1'b1,{(INBITS-1){1'b0}}}) ? {1'b0,{(INBITS-1){1'b1}}} : AbsData;

end // SatGen
endgenerate

// register saturated absolute value
always @ (posedge PhyClk or negedge nPhyRst)
  begin
    if (nPhyRst == 1'b0) begin
      SatDataReg    <= {NSATBITSIN{1'b0}};
      DataValidInt  <= 1'b0;
    end else begin
      DataValidInt  <= DataValidIn;
      if (DataValidIn==1'b1)
        SatDataReg  <= SatData;
    end
  end

// calculate the position of the most significant '1' in the input vector.
always @(*)
begin
  DataFSB = 5'd0;
  for (i=NSATBITSIN-1; i>0; i=i-1)
    begin
      if ((SatDataReg[i] == 1'b1) && (DataFSB == 5'd0))
        begin
          DataFSB = i;
        end
    end
end

// Shift according to FSB to obtain Mantissa
always @(*)
begin
  if (DataFSB==NSATBITSOUT_M2)
    DataShift    = SatDataReg;
  else if (DataFSB<NSATBITSOUT_M2)
    DataShift    = SatDataReg << (NSATBITSOUT_M2-DataFSB);
  else
    DataShift    = (SatDataReg >> (DataFSB-NSATBITSOUT_M2)) + 
                   {{(NSATBITSIN-1){1'b0}},SatDataReg[DataFSB-NSATBITSOUT_M1]}; // rounding
end

assign NegDataShift = ~DataShift+{{(NSATBITSIN-1){1'b0}},1'b1};

always @ (posedge PhyClk or negedge nPhyRst)
  begin
    if (nPhyRst == 1'b0) begin
      DataExp      <= 5'd0;
      DataMant     <= {NSATBITSOUT{1'b0}};
      DataValidOut <= 1'b0;
    end else begin
      DataValidOut <= DataValidInt;
      if (DataValidInt==1'b1) begin
        if (SatDataReg=={NSATBITSIN{1'b0}}) begin            
          DataExp <= -(NSATBITSOUT-1);                       
          if (DataSign)                                      
            DataMant <= {1'b1,{(NSATBITSOUT-2){1'b0}},1'b1}; 
          else                                               
            DataMant <= {1'b0,{(NSATBITSOUT-1){1'b1}}};      
        end else begin                                       
          DataExp <= DataFSB-NSATBITSOUT_M2;                 
          if (DataSign)                                      
            DataMant <= NegDataShift[NSATBITSOUT-1:0];       
          else                                               
            DataMant <= DataShift[NSATBITSOUT-1:0];          
        end                                                  
      end                                                  
    end
  end

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
