//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: agrunert $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 36501 $
// $Date: 2018-12-04 14:58:03 +0100 (Tue, 04 Dec 2018) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : sfo_status module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module sfo_status
( 
  /*****************************************************************************
  * system                                             
  *****************************************************************************/
  input  wire                 nPhyRst,
  input  wire                 PhyClk,
  /*****************************************************************************
  * controls
  *****************************************************************************/
  input  wire                 Enable,
  input  wire                 CfgRegForce20,
  input  wire [ 1:0]          CfgGiType,
  input  wire [ 3:0]          CfgFormatMod,
  input  wire [14:0]          CfgInvCarrierFreq,
  input  wire                 CfgSFOTDEn,          // use Coarse/FineFOAngle when high
  input  wire                 CfgSFOFDPreambEn,    // use FDOPreamb when high
  input  wire                 CfgSFOFDDataEn,      // use FDOData when high
  /*****************************************************************************
  * data
  *****************************************************************************/
  input  wire                 FDDone,
  input  wire [20:0]          CoarseFOAngle,
  input  wire [20:0]          FineFOAngle,
  input  wire [21:0]          FDOPreamb,
  input  wire [21:0]          FDOData,

  output reg                  SFODone,
  output reg  [24:0]          dPhi,            // Phase correction for FO Cordic
  output reg  [26:0]          TsNormOut,       // normalized and saturated TsOut/TsIn for Resampler
  output reg  [26:0]          StatusTsNormOut, // normalized TsOut/TsIn for Status
  output reg                  StatusTsNormOOR  // High when StatusTsNormOut is out of range
  );


//////////////////////////////////////////////////////////////////////////////
// Parameters
//////////////////////////////////////////////////////////////////////////////
  localparam FORMATMOD_LEG   = 4'd0,
             FORMATMOD_DUP   = 4'd1,
             FORMATMOD_MM    = 4'd2,
             FORMATMOD_GF    = 4'd3,
             FORMATMOD_VHT   = 4'd4,
             FORMATMOD_HE_SU = 4'd5,
             FORMATMOD_HE_MU = 4'd6,
             FORMATMOD_HE_ER = 4'd7,
             FORMATMOD_HE_TB = 4'd8;
 
//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
reg         FDDoneD;
wire        FDDonePulse;
reg   [4:0] Cpt;
reg   [4:0] CptMax;
wire  [4:0] CptPlusOne;
wire  [4:0] CptNext;
wire  [4:0] CptStart;
reg  [29:0] MeasureAccu;
reg  [29:0] OffsetMHz;
wire [29:0] OffsetMHzNeg;
wire [44:0] ppmMult;
reg  [19:0] ppmNotRounded;
// Accu used in along MeasureAccu to compute /40, to avoid having a 38 bits accu.
reg  [11:0] Accu2;
reg  [26:0] TsNormOutSat;
reg         TsNormOutOOR;
wire        SFOFDPreambEn;
 
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This block computes MHz = (round(20*TD_20MHz*2^q_M) + round(20*FD_P_20MHz*2^q_M) + round(20*FD_D_20MHz*2^q_M));
// cf TBOffsetPrecomp.m and OffsetStatus.m

// simu - debug 
/*wire [28:0] TD_20MHz, FD_P_20MHz;
reg  [28:0] FD_D_20MHz;
wire [28:0] MeasureAccuDbg;
wire [28:0] Offsetx40dbg;
wire [17:0] ppm;
assign TD_20MHz = 20*({{8{CoarseFOAngle[20]}},CoarseFOAngle}+
                      {{8{FineFOAngle[20]}}, FineFOAngle});
assign FD_P_20MHz = FDOPreamb*64;


always @(*)
  begin
    case( {CfgHe, CfgGiType})
      {1'b0, 2'd0} : FD_D_20MHz = FDOData*64; // long gi, symbol duration = 4 us, done at step 10
      {1'b0, 2'd1} : FD_D_20MHz = FDOData*71;     // short gi, symbol duration = 3.6 us, need 7 steps after step 10
      {1'b1, 2'd1} : FD_D_20MHz = FDOData*19;     // HE with GI 0.8 us, symbol duration = 13.6 us, need 2 steps after step 10
      {1'b1, 2'd2} : FD_D_20MHz = FDOData*18;     // HE with GI 0.8 us, symbol duration = 14.4 us, need 1 steps after step 10
      {1'b1, 2'd3} : FD_D_20MHz = FDOData*16; // HE with GI 0.8 us, symbol duration = 16 us, done at step 10
      default      : FD_D_20MHz = 'd0;
    endcase
  end
assign MeasureAccuDbg = FD_P_20MHz + FD_D_20MHz*CfgSFOFDPreambEn - TD_20MHz;
assign Offsetx40dbg = 40*MeasureAccu;
assign ppm          = ppmNotRounded[18:1] + {17'd0,ppmNotRounded[0]};
// end simu - debug 
*/

// Mask CfgSFOFDPreambEn for legacy frames
assign SFOFDPreambEn = CfgSFOFDPreambEn & (CfgFormatMod > FORMATMOD_DUP);

// Compute  from FD and TD used in Tx precomp

assign FDDonePulse = FDDone & !FDDoneD;

always @(*)
  begin
    if (CfgSFOFDDataEn==1'b0) begin
      CptMax = 5'd9;
    end else begin
      case (CfgFormatMod)
        FORMATMOD_LEG,FORMATMOD_DUP : 
          CptMax = 5'd10;   // symbol duration = 4 us, done at step 10
        FORMATMOD_MM,FORMATMOD_VHT : begin
          if (CfgGiType==1'b0)
            CptMax = 5'd17; // short gi, symbol duration = 3.6 us, need 7 steps after step 10
          else
            CptMax = 5'd10; // long gi, symbol duration = 4 us, done at step 10
        end
        default : begin // HE
          case(CfgGiType)
            2'd1    : CptMax = 5'd13; // HE with GI 0.8 us, symbol duration = 13.6 us, need 3 steps after step 10
            2'd2    : CptMax = 5'd12; // HE with GI 0.8 us, symbol duration = 14.4 us, need 2 steps after step 10
            2'd3    : CptMax = 5'd10; // HE with GI 0.8 us, symbol duration = 16 us, done at step 10
            default : CptMax = 5'd0;
          endcase
        end
      endcase
    end
  end

// Multiplier
assign ppmMult = $signed({{15{MeasureAccu[29]}},MeasureAccu[29:0]}) * $signed({30'd0,CfgInvCarrierFreq});

assign CptPlusOne = Cpt + 5'd1;
assign CptNext    = (Cpt==CptMax) ? 5'd18 : CptPlusOne;
assign CptStart   = (CfgSFOTDEn==1'b1) ? 5'd1 : 5'd9; // skip TD part if CfgSFOTDEn=0

assign OffsetMHzNeg = ~OffsetMHz + 30'd1;


// Control
always @ (posedge PhyClk or negedge nPhyRst)
  begin: CordicMagEn_Blk
    if (nPhyRst == 1'b0) begin
      MeasureAccu   <= 30'd0;
      Cpt           <= 5'd0;
      FDDoneD       <= 1'd0;
      OffsetMHz     <= 30'd0;
      dPhi          <= 25'd0; 
      ppmNotRounded <= 20'd0;
      StatusTsNormOut <= 27'h4000000;
      StatusTsNormOOR <= 1'b0;
      TsNormOut     <= 27'h4000000;
      Accu2         <= 12'd0;
      SFODone       <= 1'b0;
    end else if (Enable == 1'b0) begin
      // Reset all except status and computed values
      MeasureAccu   <= 30'd0;
      Cpt           <= 5'd0;
      FDDoneD       <= 1'd0;
      OffsetMHz     <= 30'd0;
      ppmNotRounded <= 20'd0;
      Accu2         <= 12'd0;
      SFODone       <= 1'b0;
    end else begin
      FDDoneD   <= FDDone;

      case (Cpt)
        5'd0    : begin
          if (FDDonePulse==1'b1) begin // Start counting
            MeasureAccu <= 30'd0;
            Cpt         <= CptStart;
            SFODone     <= 1'b0;
          end
        end
        5'd1    : begin
          MeasureAccu   <= MeasureAccu - {{9{CoarseFOAngle[20]}},CoarseFOAngle};      //     -CoarseFOAngle, 21 bits
          Cpt           <= CptNext;
        end
        5'd2    : begin
          MeasureAccu   <= MeasureAccu - {{9{FineFOAngle[20]}},FineFOAngle};          //     -CoarseFOAngle-FineFOAngleNeg, 22 bits
          Cpt           <= CptNext;
        end
        5'd3, 5'd4, 5'd5, 5'd6 : begin                                                 //  2*(-CoarseFOAngle-FineFOAngleNeg), 23 bits
          MeasureAccu   <= MeasureAccu + MeasureAccu;                                  //  4*(-CoarseFOAngle-FineFOAngleNeg), 24 bits
          Cpt           <= CptNext;                                                    //  8*(-CoarseFOAngle-FineFOAngleNeg), 25 bits
        end                                                                            // 16*(-CoarseFOAngle-FineFOAngleNeg), 26 bits
        5'd7    : begin
          MeasureAccu   <= MeasureAccu - {{7{CoarseFOAngle[20]}},CoarseFOAngle,2'd0}; // 16*(-CoarseFOAngle-FineFOAngleNeg) + 4*(-CoarseFOAngle)
          Cpt           <= CptNext;
        end
        5'd8    : begin
          MeasureAccu   <= MeasureAccu - {{7{FineFOAngle[20]}},FineFOAngle,2'd0};     // 20*(-CoarseFOAngle-FineFOAngleNeg) = (20*TD_20MHz*2^25), 27 bits
          Cpt           <= CptNext;
        end
        5'd9    : begin
          MeasureAccu   <= MeasureAccu + ({30{SFOFDPreambEn}} & {{2{FDOPreamb[21]}},FDOPreamb,6'd0});   // (20*TD_20MHz*2^25)+(20*FD_P_20MHz*2^25), 28 bits
          Cpt           <= CptNext;
        end
        5'd10   : begin
          if (CfgFormatMod<FORMATMOD_HE_SU) // 4 us or 3.6 us data symbol duration
            MeasureAccu <= MeasureAccu + {{2{FDOData[21]}},FDOData,6'd0};             // (20*TD_20MHz*2^25)+(20*FD_P_20MHz*2^25) + 64*FDDataMeasure, 29 bits
          else
            MeasureAccu <= MeasureAccu + {{4{FDOData[21]}},FDOData,4'd0};             // (20*TD_20MHz*2^25)+(20*FD_P_20MHz*2^25) + 16*FDDataMeasure, 29 bits
          Cpt           <= CptNext;
        end
        default : begin // 11 to 17, variable count depending on he and gi
          MeasureAccu   <= MeasureAccu + {{8{FDOData[21]}},FDOData};                  // (20*TD_20MHz*2^25)+(20*FD_P_20MHz*2^25) + up to 71*FDDataMeasure, 30 bits
          Cpt           <= CptNext;
        end
        // compute TsNormOut =  round(2^26 + round( OffsetMHz * CfgInvCarrierFreq * 2^-25 )
        5'd18   : begin
          MeasureAccu   <=               {4'b01,25'd0,ppmMult[24]};                    // ppm rounding and adding 2^26
          Cpt           <= CptNext;
          OffsetMHz     <= MeasureAccu[29:0];
          ppmNotRounded <= ppmMult[44:25]; // register at multiplier output
        end
        5'd19   : begin
          MeasureAccu   <= MeasureAccu + {{10{ppmNotRounded[19]}},ppmNotRounded};      // adding ppm = OffsetMHz * CfgInvCarrierFreq on 20 bits to 2^26, result on 27 bits
          Cpt           <= CptNext;          
        end
        // compute dPhi = -OffsetMHz/40
        5'd20   : begin
          MeasureAccu   <=                                 OffsetMHz;                  // MHz * 1/64 with 38 bits precision, split on OffsetMHz and Accu2
          Cpt           <= CptNext;
          StatusTsNormOut <= MeasureAccu[26:0];                                        // 2^26 + ratio, on 27 bits unisgned
          Accu2         <= 12'd0;                                                      // Remaining 10 bits float precision of MHz * 1/64 computation
        end
        5'd21   : begin
          MeasureAccu   <= MeasureAccu + {{2{OffsetMHzNeg[29]}},OffsetMHzNeg[29:2]};   // MHz * (1/64 - 1/256)
          Cpt           <= CptNext;          
          Accu2         <= Accu2 + {2'd0,OffsetMHzNeg[1:0],8'd0};
        end
        5'd22   : begin
          MeasureAccu   <= MeasureAccu + {{4{OffsetMHz[29]}},OffsetMHz[29:4]};         // MHz * (1/64 - 1/256 + 1/1024)
          Cpt           <= CptNext;
          Accu2         <= Accu2 + {2'd0,OffsetMHz[3:0],6'd0};
        end
        5'd23   : begin
          MeasureAccu   <= MeasureAccu + {{6{OffsetMHzNeg[29]}},OffsetMHzNeg[29:6]};   // MHz * (1/64 - 1/256 + 1/1024 - 1/4096)
          Cpt           <= CptNext;
          Accu2         <= Accu2 + {2'd0,OffsetMHzNeg[5:0],4'd0};
        end
        5'd24   : begin
          MeasureAccu   <= MeasureAccu + {{8{OffsetMHz[29]}},OffsetMHz[29:8]};         // MHz * (1/64 - 1/256 + 1/1024 - 1/4096 + 1/16384)
          Cpt           <= CptNext;
          Accu2         <= Accu2 + {2'd0,OffsetMHz[7:0],2'd0};
        end
        5'd25   : begin
          MeasureAccu   <= MeasureAccu + {{10{OffsetMHzNeg[29]}},OffsetMHzNeg[29:10]}; // MHz * (1/64 - 1/256 + 1/1024 - 1/4096 + 1/16384 - 1/65536)
          Cpt           <= CptNext;
          Accu2         <= Accu2 + {2'd0,OffsetMHzNeg[9:0]};
        end
        5'd26   : begin
          MeasureAccu   <= MeasureAccu + {28'd0,Accu2[11:10]};                         // Report carry from 10 LSBs
          Cpt           <= CptNext;
        end
        5'd27   : begin
          if (CfgRegForce20==1'b1)
            MeasureAccu <= {{5{MeasureAccu[29]}},MeasureAccu[29:5]} + {29'd0,MeasureAccu[4]};  // 20 MHz: Cut 5 more bits with rounding
          else
            MeasureAccu <= {{6{MeasureAccu[29]}},MeasureAccu[29:6]} + {29'd0,MeasureAccu[5]};  // 40 MHz: Cut 6 more bits with rounding
          Cpt           <= CptNext;
        end
        5'd28   : begin
          Cpt           <= 5'd0;
          SFODone       <= 1'b1;
          dPhi          <= MeasureAccu[24:0]; // 30 bits, 5 or 6 cut: 25 bits left
          TsNormOut       <= TsNormOutSat;
          StatusTsNormOOR <= TsNormOutOOR;
        end
      endcase
    end
 end

 // Ensure that TsNormOut is within functional limits for resampler, else TxPath can get stuck
 always @(*)
 begin
   if (StatusTsNormOut<27'd67106180) begin // +40 ppm
     TsNormOutSat = 27'd67106180;
     TsNormOutOOR = 1'b1;
   end else if (StatusTsNormOut>27'd67111548) begin // -40ppm
     TsNormOutSat = 27'd67111548;
     TsNormOutOOR = 1'b1;
   end else begin
     TsNormOutSat = StatusTsNormOut;
     TsNormOutOOR = 1'b0;
   end
 end

//wire [37:0] dbg;
//assign dbg = {MeasureAccu,10'd0} + Accu2;

endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
