//------------------------------------------------------------------------------
// ldpcEnc.v
// 
// Description
//   Wrapper for the LDPC encoder core, including clock gating, memory && regs.
// 
// Inputs:
//   nReset         : Asynchronous reset.
//   clk            : Master clock.
// (From the processor)
//   blkSelect      : Active high select for the block.
//   writeNotRead   : Active high write enable.
//   addr           : Address bus from processor.
//   wrData         : Write data bus from processor.
// (From sender (e.g. DMA device)
//   inStrobe       : High when inData is valid.
//   inDataWord     : `IP_WIDTH samples of RX_BITS.
// (From downstream HW)
//   clrToSend      : Enables output of encoded data when high.
// Outputs:
// (To sender)
//   rdyToRcv       : Enables the sender to start sending. Stays high until
//                    the expected amount of data has been transferred.
// (To downstream hardware)
//   encodeComplete   : Pulse indicate completion of a block encode.
//   packetComplete   : Goes high after last encode.
//   opStrobe         : High when opDataWord is valid.
//   opDataWord       : Encoded output.
// (To the processor)
//   rdDataOut        : Read data bus.
//
// 20 Aug 2010 M. Rumsey. Created.
// 09 May 2011 M. Rumsey. Top level split into a wrapper && a core.
//
// (c) Copyright 2010-11, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcEnc.vh"

module ldpcEnc
  ( 
    input                           nReset,
    input                           clk,
    input                           scanMode,
    // Inputs from the processor
    input [`NUM_MU_BITS-1:0]        selUser,
    input [`BUS32:0]                writeEn,
    input                           readEn,
    input [`NUM_ADDR_BITS-1:0]      addr,
    input [`NUM_DATA_BITS-1:0]      wrData,
    // Inputs from sender             
    input                           inStrobe,
    input [`IP_WIDTH-1:0]           inDataWord,
    // Inputs from output interface   
    input                           clrToSend,
    // Outputs to sender
    output [`NUM_MU_BITS-1:0]       inUserOut,
    output                          rdyToRcvOut,
    // Outputs to downstream hardware
    output [`NUM_MU_BITS-1:0]       opUserOut,
    output                          encodeCompleteOut,
    output [`NUM_MU-1:0]            packetCompleteOut,
    output                          opStrobeOut,
    output [`OP_WIDTH-1:0]          opDataWordOut,
    output [numBits(`OP_WIDTH)-1:0] numOpBitsOut,
    output                          lastOpSampleOut,
    // Outputs to the processor. Not registered.
    output [`NUM_DATA_BITS-1:0]     rdDataOut);

`include "ldpcEncFuncs.vh"

  // Clocking
  wire ipClk;
  wire ipClkEn;
  wire opClk;
  wire opClkEn;
  wire ctrlClk;
  wire ctrlClkEn;
  wire cellClk;
  wire cellClkEn;
  wire encRam0Clk;
  wire encRam0ClkEn;
  wire encRam1Clk;
  wire encRam1ClkEn;
  wire regsClk;
  wire regsClkEn;
  wire shiftClk;
  wire shiftClkEn;
  wire rotClk;
  wire rotClkEn;
  wire encClk;
  wire encClkEn;
 
  //---------------------------------------------------------------------------
  // Regs
  //---------------------------------------------------------------------------

  wire [`NUM_MU-1:0]       enable;
  // To mem block to program the code cell information.
  wire                     cellAccess;
  wire                     cellRead;
  wire                     cellWrite;
  wire [`CELL_RAM_W-1:0]   cellWrData;
  // Code characteristics to control

  wire `zEnumArrayType           zEnum;
  wire `rEnumArrayType           rEnum;
  wire `bpsArrayType             bitsPerSymbol;
  wire `packetLenArrayType       packetLen;
  wire `nShrtFloorArrayType      nShrtFloor;
  wire `shrtModArrayType         shrtMod;
  wire `nPuncFloorArrayType      nPuncFloor;
  wire `puncModArrayType         puncMod;
  wire `nRepFloorArrayType       nRepFloor;
  wire `repModArrayType          repMod;
  wire `blkNumArrayType          curBlkNum;
  wire `opWidthArrayType         opWidth;
  wire `zEnumType                zEnumPeUser;
  wire `rEnumType                rEnumPeUser;
  
  wire                     enablePeUser;
  wire [`NUM_MU_BITS-1:0]  peUser;
  
  //---------------------------------------------------------------------------
  // RAMs
  //---------------------------------------------------------------------------

  wire cellRamSel;
  wire [`CELL_RAM_A-1:0]     cellRamAddr;
  wire [`CELL_RAM_W-1:0]     cellRamWrData;
  wire                       cellRamWe;
  wire [`CELL_RAM_W-1:0]     cellRamRdData;
  wire [numBits(`NROWS)-1:0] numRowsRom;
  
  // registered version of cellRamRdData for readback.
  wire [`CELL_RAM_W-1:0]     cellRdData;
  wire                       encRamSel0;
  wire [`ENC_RAM_A-1:0]      encRamAddr0;
  wire [`ENC_RAM_W-1:0]      encRamWrData0;
  wire [`ENC_RAM_WE-1:0]     encRamWe0;
  wire [`ENC_RAM_W-1:0]      encRamRdData0;
  wire                       encRamSel1;
  wire [`ENC_RAM_A-1:0]      encRamAddr1;
  wire [`ENC_RAM_W-1:0]      encRamWrData1;
  wire [`ENC_RAM_WE-1:0]     encRamWe1;
  wire [`ENC_RAM_W-1:0]      encRamRdData1;

  //---------------------------------------------------------------------------
  // Debug Port
  //---------------------------------------------------------------------------

  wire [`DBG_PORT_WIDTH-1:0] dbgIp1;
  wire [`DBG_PORT_WIDTH-1:0] dbgIp2;
  wire [`DBG_PORT_WIDTH-1:0] dbgOp1;
  wire [`DBG_PORT_WIDTH-1:0] dbgOp2;
  wire [`DBG_PORT_WIDTH-1:0] dbgPe1;
  wire [`DBG_PORT_WIDTH-1:0] dbgPe2;
  wire [`DBG_PORT_WIDTH-1:0] dbgState;


  // Clock gating
  ldpcEncClkGate clkGate1 (clk, scanMode, ipClkEn, nReset, ipClk);
  ldpcEncClkGate clkGate2 (clk, scanMode, opClkEn, nReset, opClk);
  ldpcEncClkGate clkGate3 (clk, scanMode, regsClkEn, nReset, regsClk);
  ldpcEncClkGate clkGate4 (clk, scanMode, cellClkEn, nReset, cellClk);
  ldpcEncClkGate clkGate5 (clk, scanMode, ctrlClkEn, nReset, ctrlClk);
  ldpcEncClkGate clkGate6 (clk, scanMode, encRam0ClkEn, nReset, encRam0Clk);
  generate
    if (`DOUBLE_BUFFER) begin: gDoubleBufClk
      ldpcEncClkGate clkGate7 (clk, scanMode, encRam1ClkEn, nReset, encRam1Clk);
    end
  endgenerate //  gDoubleBufClk
  ldpcEncClkGate clkGate8 (clk, scanMode, shiftClkEn, nReset, shiftClk);
  ldpcEncClkGate clkGate9 (clk, scanMode, rotClkEn, nReset, rotClk);
  ldpcEncClkGate clkGate10 (clk, scanMode, encClkEn, nReset, encClk);

  ldpcEncRegs regs
    (
     .nReset            (nReset),
     .clk               (regsClk),
     // Processor interface inputs.
     .selUser           (selUser),
     .writeEn           (writeEn),
     .readEn            (readEn),
     .addr              (addr),
     .wrData            (wrData),
     // Read only registers that may be monitored by processor.
     .curBlkNum         (curBlkNum),
     .cellRdData        (cellRdData), // Not used if `USE_CELL_ROM==1.
     .clkEnOut          (regsClkEn),
     .rdDataOut         (rdDataOut),
     // Master enable to control.
     .enableOut         (enable),
     // To mem block to program the code cell information.
     // Not used if `USE_CELL_ROM==1.
     .cellAccessOut     (cellAccess),
     .cellReadOut       (cellRead),
     .cellWriteOut      (cellWrite),
     .cellWrDataOut     (cellWrData),
     // Code characteristics to control.
     .zEnumOut          (zEnum),
     .rEnumOut          (rEnum),
     .bitsPerSymbolOut  (bitsPerSymbol),
     .packetLenOut      (packetLen),
     // Shortening etc parameters to control.
     .nShrtFloorOut     (nShrtFloor),
     .shrtModOut        (shrtMod),
     .nPuncFloorOut     (nPuncFloor),
     .puncModOut        (puncMod),
     .nRepFloorOut      (nRepFloor),
     .repModOut         (repMod),
     // other
     .opWidthOut        (opWidth));
  
   ldpcEncCore core
     (
      // Block in/out interfaces
      .nReset            (nReset),
      .inStrobe          (inStrobe),
      .inDataWord        (inDataWord),
      .clrToSend         (clrToSend),
      .inUserOut         (inUserOut),
      .rdyToRcvOut       (rdyToRcvOut),
      .opUserOut         (opUserOut),
      .encodeCompleteOut (encodeCompleteOut),
      .packetCompleteOut (packetCompleteOut),
      .opStrobeOut       (opStrobeOut),
      .opDataWordOut     (opDataWordOut),
      .numOpBitsOut      (numOpBitsOut),
      .lastOpSampleOut   (lastOpSampleOut),
      .cellRdDataOut     (cellRdData),
      //  Register values
      .enable            (enable),
      .opWidthP          (opWidth),
      .cellAccess        (cellAccess),
      .cellRead          (cellRead),
      .cellWrite         (cellWrite),
      .cellWrData        (cellWrData),
      .zEnumP            (zEnum),
      .rEnumP            (rEnum),
      .bitsPerSymbolP    (bitsPerSymbol),
      .packetLenP        (packetLen),
      .nShrtFloorP       (nShrtFloor),
      .shrtModP          (shrtMod),
      .nPuncFloorP       (nPuncFloor),
      .puncModP          (puncMod),
      .nRepFloorP        (nRepFloor),
      .repModP           (repMod),
      .curBlkNumOut      (curBlkNum),
      //  Memories
      .cellRamSelOut     (cellRamSel),
      .cellRamAddrOut    (cellRamAddr),
      .cellRamWrDataOut  (cellRamWrData),
      .cellRamWeOut      (cellRamWe),
      .cellRamRdData     (cellRamRdData),
      .numRowsRom        (numRowsRom),
      .encRamSel0Out     (encRamSel0),
      .encRamAddr0Out    (encRamAddr0),
      .encRamWrData0Out  (encRamWrData0),
      .encRamWe0Out      (encRamWe0),
      .encRamRdData0     (encRamRdData0),
      .encRamSel1Out     (encRamSel1),
      .encRamAddr1Out    (encRamAddr1),
      .encRamWrData1Out  (encRamWrData1),
      .encRamWe1Out      (encRamWe1),
      .encRamRdData1     (encRamRdData1),
      .peUserOut         (peUser),
      //  Clocks
      .ipClk             (ipClk),
      .opClk             (opClk),
      .ctrlClk           (ctrlClk),
      .cellClk           (cellClk),
      .shiftClk          (shiftClk),
      .rotClk            (rotClk),
      .encClk            (encClk),
      .ipClkEnOut        (ipClkEn),
      .opClkEnOut        (opClkEn),
      .ctrlClkEnOut      (ctrlClkEn),
      .cellClkEnOut      (cellClkEn),
      .encRam0ClkEnOut   (encRam0ClkEn),
      .encRam1ClkEnOut   (encRam1ClkEn),
      .shiftClkEnOut     (shiftClkEn),
      .rotClkEnOut       (rotClkEn),
      .encClkEnOut       (encClkEn),
      // Debug port
      .dbgIp1Out         (dbgIp1),
      .dbgIp2Out         (dbgIp2),
      .dbgOp1Out         (dbgOp1),
      .dbgOp2Out         (dbgOp2),
      .dbgPe1Out         (dbgPe1),
      .dbgPe2Out         (dbgPe2),
      .dbgStateOut       (dbgState));

  //---------------------------------------------------------------------------
  // RAMs
  //---------------------------------------------------------------------------

  generate
    if (`USE_CELL_ROM == 0) begin: gCellRom0
      // Stores information defining the code
      ldpcEncCellRam cellRam
        (
         .nReset (nReset),
         .ck   (cellClk),
         .sel  (cellRamSel),
         .addr (cellRamAddr),
         .dIn  (cellRamWrData),
         .we   (cellRamWe),
         .dOut (cellRamRdData));
    end
  endgenerate //  gCellRom0

  generate
    if (`USE_CELL_ROM == 1) begin: gCellRom1
    assign zEnumPeUser  = zEnum[peUser*2+:2];
    assign rEnumPeUser  = rEnum[peUser*2+:2];
    assign enablePeUser  = enable[peUser];    
      ldpcEncCellRom cellRom
        (
         .nReset         (nReset),
         .zEnum          (zEnumPeUser),
         .rEnum          (rEnumPeUser),
         .enable         (enablePeUser),
         .clk            (cellClk),
         .sel            (cellRamSel),
         .addr           (cellRamAddr),
         .dIn            (cellRamWrData),
         .we             (cellRamWe),
         .dOut           (cellRamRdData),
         .numRowsOut     (numRowsRom));
    end
  endgenerate //  gCellRom1

  // Memory buffer for input, encode and output
  ldpcEncEncRam EncRam0
    (
     // Load/read via regs and read Enc info.
     .nReset (nReset),
     .ck   (encRam0Clk),
     .sel  (encRamSel0),
     .addr (encRamAddr0),
     .dIn  (encRamWrData0),
     .we   (encRamWe0),
     .dOut (encRamRdData0));
  
  // Memory buffer for input, encode and output
  generate
    if (`DOUBLE_BUFFER) begin: gDoubleBufRam
      ldpcEncEncRam EncRam1
        (
         // Load/read via regs && read Enc info.
         .nReset (nReset),
         .ck   (encRam1Clk),
         .sel  (encRamSel1),
         .addr (encRamAddr1),
         .dIn  (encRamWrData1),
         .we   (encRamWe1),
         .dOut (encRamRdData1));
    end
  endgenerate //  gDoubleBufRam
  
endmodule
