//------------------------------------------------------------------------------
// ldpcEnc.v
// 
// Description
//   Constants and functions for the LDPC Encoder.        
//
// 17 Aug 2010 M. Rumsey
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`ifdef BRC_TIMESCALE
`timescale 1ns/1ps
`endif

`ifndef ldpcEnc_vh
`define ldpcEnc_vh

`default_nettype wire

// how to convert seconds to timeunits when reading vectors
`define TU 1e9
`define DBG_PORT_WIDTH  (32)

`define MIN(a,b) ((a<b)?a:b)
`define MAX(a,b) ((a>b)?a:b)
// Prepend 'bits' zeros before a.
`define PAD(a,bits) {{toUnsigned(bits) {1'b0}}, a}

  //---------------------------------------------------------------------------
// Packing/Unpacking
//---------------------------------------------------------------------------

// w0 is the packed width and w1 is the unpacked widths.
// 
// Generate form:
`define LENC_UNPACK(label, src, dest, w0, w1) generate for (idx1=0; idx1<(w1); idx1=idx1+1) begin:label assign dest[idx1] = src[(1+idx1)*(w0)-1 -: w0]; end  endgenerate
`define LENC_PACK(label, src, dest, w0, w1) generate for (idx1=0; idx1<(w1); idx1=idx1+1) begin: label assign dest[(1+idx1)*(w0)-1 : idx1*(w0)] = $unsigned(src[idx1]); end endgenerate

`define LENC_INIT(dest, init, w1) for (idx1=0; idx1<(w1); idx1=idx1+1) dest[idx1] = init
// Sequential process form
`define LENC_INITQ(dest, init, w1) for (idx1=0; idx1<(w1); idx1=idx1+1) dest[idx1] <= init

//---------------------------------------------------------------------------
// Types
//---------------------------------------------------------------------------
  
// These are mainly mem mapped registers that have to be stored per-user.

`define RCB_HI numBits(`BPS_MAX + `N_MAX + `REP_MAX)-1
`define BPS_HI numBits(`BPS_MAX)-1

`define Z_ENUM_BITS          numBits(`Z_ENUM_MAX)
`define R_ENUM_BITS          numBits(`R_ENUM_MAX)
`define PACKETLEN_BITS       numBits(`MAX_BLKNUM)
`define NSHRTFLOOR_BITS      numBits(`K_MAX)
`define MOD_BITS             numBits(`MAX_BLKNUM)
`define NPUNCFLOOR_BITS      numBits(`M_MAX)
`define NREPFLOOR_BITS       numBits(`REP_MAX)
`define BLKNUM_BITS          numBits(`MAX_BLKNUM)
`define OPWIDTH_BITS         numBits(`OP_WIDTH)  

`define zEnumType           [numBits(`Z_ENUM_MAX)-1:0]
`define rEnumType           [numBits(`R_ENUM_MAX)-1:0]
`define bpsType             [`BPS_HI:0]
`define runningBitCountType [`RCB_HI:0] 
`define packetLenType       [numBits(`MAX_BLKNUM)-1:0]
`define nShrtFloorType      [numBits(`K_MAX-1)-1:0]
`define shrtModType         [numBits(`MAX_BLKNUM)-1:0]
`define nPuncFloorType      [numBits(`M_MAX-1)-1:0]
`define puncModType         [numBits(`MAX_BLKNUM)-1:0]
`define nRepFloorType       [numBits(`REP_MAX-1)-1:0]
`define repModType          [numBits(`MAX_BLKNUM)-1:0]
`define blkNumType          [numBits(`MAX_BLKNUM)-1:0]
`define opWidthType         [numBits(`OP_WIDTH)-1:0]  

`define zEnumArrayType           [`NUM_MU*`Z_ENUM_BITS-1:0]
`define rEnumArrayType           [`NUM_MU*`R_ENUM_BITS-1:0]
`define bpsArrayType             [`NUM_MU*(`BPS_HI+1)-1:0]
`define runningBitCountArrayType [`NUM_MU*(`RCB_HI+1)-1:0]
`define packetLenArrayType       [`NUM_MU*`PACKETLEN_BITS-1:0]
`define nShrtFloorArrayType      [`NUM_MU*`NSHRTFLOOR_BITS-1:0]
`define shrtModArrayType         [`NUM_MU*`MOD_BITS-1:0]
`define nPuncFloorArrayType      [`NUM_MU*`NPUNCFLOOR_BITS-1:0]
`define puncModArrayType         [`NUM_MU*`MOD_BITS-1:0]
`define nRepFloorArrayType       [`NUM_MU*`NREPFLOOR_BITS-1:0]
`define repModArrayType          [`NUM_MU*`MOD_BITS-1:0]
`define blkNumArrayType          [`NUM_MU*`BLKNUM_BITS-1:0]
`define opWidthArrayType         [`NUM_MU*`OPWIDTH_BITS-1:0]

// Packed versions

//---------------------------------------------------------------------------
// User configurable.
//---------------------------------------------------------------------------

// Where a define is not in brackets the value is used as a literal in bitwidths
// and the value must be a literal - not an expression. Consider using ifdef
// constructs if you need to support multiple cases.

// This ASIC/FPGA switch gives an easy way to switch between what is
// optimal on an ASIC and an FPGA. You may not have to change any other
// options. See the 'sel' functions in constants below for the choices
// made in each case.
`define FPGA              (0)

// Select 32 bit or 16 wide accesses (BUS 32 = 1/0 respectively).
// In 32 bit mode WRITE_EN and READ_EN become 2 bits, which allows you to
// set the 16 bit words within the 32 bit word individually or together.
`define BUS32             (0)

// Number of users to support (11ac MU mode)
//`define NUM_MU_BITS       2
//`define NUM_MU            4
`define NUM_MU_BITS       1
`define NUM_MU            1

//`define CLK_GATING        (`FPGA ? 0 : 1)
`define CLK_GATING        0

// RESET_ALL: This is always true for Verilog. Only the VHDL version allows
// some register to not be reset.

// Uncomment the negedge line to get async resets, otherwise use the
// blank RESET_STR line option to get synchronous resets.
`define RESET_STR      or negedge(nReset)
//`define RESET_STR

// If your synthesis tool does not support recursive functions then comment this out
//`define ALLOW_RECURSIVE

// Use a hardcoded LUT for the code information, rather than programming the
// Cell Ram via the processor interface.
`define USE_CELL_ROM      1'b1

// With single buffering a memory buffer is passed from input to encode to
// output blocks and input may not begin again until the output complete.
// With double buffering a second memory buffer is available, which
// can follow behind the first once each hardware resource (input,
// encode and output) is free.
`define DOUBLE_BUFFER     1'b1
  
// Input Interface width and width of internal 'collection register'
//`define IP_WIDTH          16
//`define IP_WIDTH_BITS      5
//`define ENC_RAM_IP_WIDTH  27
//`define ENC_RAM_IP_WIDTH_BITS 5
`define IP_WIDTH          8
`define IP_WIDTH_BITS      4
`define ENC_RAM_IP_WIDTH  9
`define ENC_RAM_IP_WIDTH_BITS 4
// Barrel shifter LEFT bits. For ldpcEncIp and ldpcEncOp respectively.
`define CR_HI (`ENC_RAM_IP_WIDTH + 2*`IP_WIDTH -1)
`define CR_OP_HI (2*`MEM_FETCH_SIZE+`OP_WIDTH -1)

// Processor Interface
`define NUM_ADDR_BITS   (4)
`define NUM_DATA_BITS   ((`BUS32 == 1) ? 32 : 16)

// Output Interface width and width used to fetch data from RAM for output
// interface. Ideally divisible into Z but must be > OP_WIDTH.
// For these 8 and 9 respectively provide good performance. For higher
// output bandwidth go for 16 and 27.
// OP_WIDTH is the max output width. OP_WIDTH register may be used to select
// a lower width (actual used is the number of QAM bits recorded in the vectors).
//`define OP_WIDTH        8
//`define MEM_FETCH_SIZE  9
// 16 bit ASIC (& FPGA) configuration
`define OP_WIDTH        (16)
`define MEM_FETCH_SIZE  (27)

// Concatenation relates to the amount of input data for a block not being a
// multiple of the input width or the amount of output data not being a
// multiple of the output width. One option is for the encoder to ignore
// spare input bits and zero spare output bits, another is to allow the
// blocks of a packet to be concatenated such that an input or output word
// may straddle block boundaries.
`define CONCAT_OP    1'b0
`define CONCAT_IP    1'b1

// In ldpcEncPe, barrel shifter registering falls in the middle of the
// RotateDownZ function. The registering can be moved
// to optimize timing. RDZ_LEVEL can have values from -1 to 6. Level 0
// is the first and level 6 is the 7th, corresponding to the number of bitszEnum
// in the cyclical shift. -1 causes the register to preceed the rotator.
`define RDZ_LEVEL    (3)

// Number of Spatial streams.
// This is referred to 80MHz channels so use double for 160MHz.
//`define NSS            (4)
`ifdef RW_TXRX_1X1
  `define NSS            (1)
`endif  
`ifdef RW_TXRX_2X2
  `define NSS            (2)
`endif  

// Max number of blks in pkt. To avoid floats aMaxPPDUTime=5.484 s becomes 54840 and the OFDM symbol period in us is x10.
`ifdef RW_NX_DERIV_CHBW20ONLY
  `define NCBPSMAX_AX (2340)
  `define NCBPSMAX_AC (416)
`endif
`ifdef RW_NX_DERIV_CHBW4020ONLY
  `define NCBPSMAX_AX (4680)
  `define NCBPSMAX_AC (864)
`endif
`ifdef RW_NX_DERIV_CHBW804020ONLY
  `define NCBPSMAX_AX (9800)
  `define NCBPSMAX_AC (1872)
`endif
`define MAX_BLKNUM_AX (54840/136 * `NSS * `NCBPSMAX_AX / 1944)
`define MAX_BLKNUM_AC (54840/36 * `NSS * `NCBPSMAX_AC / 1944)
`define MAX_BLKNUM    (maximum(`MAX_BLKNUM_AX, `MAX_BLKNUM_AC))

//---------------------------------------------------------------------------
// Not User Configurable
//---------------------------------------------------------------------------

// Block version information.
`define BLK_ID         (8'hB2)
`define MAJ_VER        (2)
`define MIN_VER        (10)
`define BLK_VER        ((`MAJ_VER-1)*16 + `MIN_VER)
`define BLK_ID_VER     (`BLK_ID*256 + `BLK_VER)
// Z is the number of rows and cols in the macro cell. There are
// several sizes of Z and we enumerate the possibilities.
`define Z_ENUM_MAX     (2)
// R is the code rate and rEnum is the enumeration number for lookups.
`define R_ENUM_MAX     (3)

`define zSizeType [0:`Z_ENUM_MAX]`integer
`define rSizeType [0:`R_ENUM_MAX]`integer                         
`define kSizeType [0:`Z_ENUM_MAX]`rSizeType                     

// Lookup tables using zEnum and rEnum.

`define Z_MAX          (81)
`define N_MAX          (1944)
`define K_MAX          (`N_MAX*5/6)
`define M_MAX          (`N_MAX*1/2)
`define BPS_MAX        (1872*`NSS)
`define REP_MAX        (`BPS_MAX - 1944*1/6)
  
`define NCOLS          (24)
// These two needed for literals. Extend to number of bits in NCOLS.
`define NCOLS_LESS2    5'd22
`define P1_BASE        5'd23
`define NDCOLS         (20)
`define NROWS          (12)
`define NDCELLS        (81)
`define MEM_MUX_BITS   (numBits(ceilDiv(`Z_MAX, `MEM_FETCH_SIZE)-1))

`define BLK_VERSION_ADDR      (0)
`define CODE_ACCESS_ADDR      (1)
// byte0: enables
`define ENABLE_ADDR           (2)
`define PACKET_LEN_ADDR       (3)
// byte0: code size, byte 1 code rate.
`define CODE_ENUM_ADDR        (4)
`define BITS_PER_SYMBOL_ADDR  (5)

`define SHRT_FLOOR_ADDR     (6)
`define SHRT_MOD_ADDR       (7)
`define PUNC_FLOOR_ADDR     (8)
`define PUNC_MOD_ADDR       (9)
`define REP_FLOOR_ADDR      (10)
`define REP_MOD_ADDR        (11)
`define OP_WIDTH_ADDR       (12)

// Address map: status registers.
`define CUR_BLK_NUM_ADDR    (13)

// Input / Output ports (what you write may not be what you read!)
`define CODE_ADDR           (14)

//---------------------------------------------------------------------------
// Memory sizes.
//---------------------------------------------------------------------------
  
// Memories. Sizes D, A and W refer to depth, number
// of address bits, and data bus width.

// Code Cell memory. Width is sum of widths needed to store cols, the cyc
// shift and end row flag. 
`define CELL_RAM_D  (`NDCELLS)
`define CELL_RAM_A  (numBits(`CELL_RAM_D))
`define CELL_RAM_W  (numBits(`NDCOLS-1) + numBits(`Z_MAX-1) + 1)

// Encoder memory. Stores inputs and intermediate vector. The sum of these
// two is NCOLS by definition.
// is NDCOLS*Z, 
`define ENC_RAM_D   `NCOLS
`define ENC_RAM_A   (numBits(`ENC_RAM_D))
`define ENC_RAM_W   (`Z_MAX)
`define ENC_RAM_WE  (ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH))

 
`endif

