//------------------------------------------------------------------------------
// ldcpEncCellMem.v
// 
// Description:
//   Stores information defining the code. The code is defined in terms of
//   macro cells, hence the name of the memory.
//
//   cellClk             : Clock for Port A of CellMem.
// (From Regs)
//   cellAccess          : The Cell Info RAM is in exernal access mode.
//   cellRead            : Read the Cell Info RAM (this cycle).
//   cellWrite           : Write cellWrData to the Cell Info RAM.
//   cellWrData          : Code info to load into the Cell Info RAM.
// [From ctrl:fetch data for encode]
//   cellMemAddr         : Address into the cell memory.
//   cellMemRdSel        : Combined read and select for cell memory.
// (To ctrl)
//   endRow              : A flag bit from cellMem indicating end of row.
//   col                 : The column number from cellMem.
// (To Clocking)         
//   cellClkEn           : Clock enable for the cell memory.
// (To Ram)
//   Ram input/output signals.
//   
// 26 Aug 2010 M. Rumsey. Created.
// 09 May 2011 M. Rumsey. RAM now located at ldpcEnc top level.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcEnc.vh"

module ldpcEncCellMem
  ( 
    input                            nReset,
    input                            cellClk,
    input                            disabling, 
    // From regs to load Cell Info RAM
    input                            cellAccess,
    input                            cellRead,
    input                            cellWrite,
    input [`CELL_RAM_W-1:0]          cellWrData,
    // From ctrl to fetch Cell Info    
    input [numBits(`CELL_RAM_D)-1:0] cellMemAddr,
    input                            cellMemRdSel, 
    // To encPe                        
    output [numBits(`Z_MAX-1)-1:0]   cycOut_2,
    // To ctrl                         
    output                           endRowOut_2,
    output [numBits(`NDCOLS-1)-1:0]  colOut_2,
    output [numBits(`NROWS)-1:0]     numRowsOut,
    output                           cellClkEnOut,
    // To registers block (registered version of cellRamRdData)
    output reg [`CELL_RAM_W-1:0]     cellRdDataOut,
    // Port to RAM                     
    output                           cellRamSelOut,
    output [`CELL_RAM_A-1:0]         cellRamAddrOut,
    output [`CELL_RAM_W-1:0]         cellRamWrDataOut,
    output                           cellRamWeOut,
    input [`CELL_RAM_W-1:0]          cellRamRdData);
  

`include "ldpcEncFuncs.vh"

    // Signals for the Cell info memory.
  reg  [numBits(`NDCELLS)-1:0] cellLoadIdx;
  reg                          cellAccessActive;
  wire [`CELL_RAM_A-1:0]       cellAddr;
  wire                         cellRamEn;
  reg                          resettingAddr;
                          
  // Local version of outputs
  wire                         endRow_2;
  wire [numBits(`NDCOLS)-1:0]  col_2;
  wire [numBits(`Z_MAX-1)-1:0] cyc_2;
  reg [numBits(`NROWS)-1:0]    numRows; 
 
  assign cellClkEnOut = cellAccess | cellAccessActive | cellMemRdSel | disabling;
 
  //---------------------------------------------------------------------------
  // CellInfo Memory
  //---------------------------------------------------------------------------

  // This is loaded by the host processor via the Regs block && read throughout
  // encode using the same port.
  
  assign cellAddr = (cellAccess) ? cellLoadIdx : cellMemAddr; 
  assign cellRamEn = cellAccess | resettingAddr | cellMemRdSel;
               
  // Address generator when loading Cell Info. A cellLoad bit is set in a
  // control register to enable load. Writes || reads to CELL_INFO_ADDR
  // are signalled with cellWrite/Read && data is passed on CellWrData and
  // cellRdData. Reading is registered && two dummy reads are needed to
  // read data back on the processor bus.

  generate
    if (!`USE_CELL_ROM) begin: gCellRam
      always @(posedge(cellClk) `RESET_STR)
      begin : pCellLoadIdx
        reg  [numBits(`NDCELLS-1)-1:0] cellLoadIdxV;
        reg                            latchRdDataV;
        if (nReset == 1'b0) begin
          latchRdDataV = 1'b0;
          numRows <= `PAD(1'b0, numBits(`NROWS)-1);
          cellLoadIdx <= `PAD(1'b0, numBits(`NDCELLS)-1);
          resettingAddr <= 1'b0;
          cellAccessActive <= 1'b0;
          cellRdDataOut <= `PAD(1'b0, `CELL_RAM_W-1);
        end else begin
          if (disabling) begin
            latchRdDataV = 1'b0;
          end          
          if (~cellAccess) begin
            // After an access from the processor ensure that the address;
            // reset to 0 and a read done [to get endRow:a good state].
            if (cellAccessActive) begin
              resettingAddr <= 1'b1;
              cellLoadIdx <= `PAD(1'b0, numBits(`NDCELLS)-1);
            end
            // The active sig extends the clock to allow the sync clr.
            if (resettingAddr) begin
              cellAccessActive <= 1'b0;
              resettingAddr <= 1'b0;
            end      
          end else begin
            cellAccessActive <= 1'b1;       
            if (cellRead | cellWrite) begin
              cellLoadIdxV = cellLoadIdx;
              if (cellLoadIdxV < `NDCELLS-1) begin
                cellLoadIdxV = cellLoadIdx + `PAD(1'b1, numBits(`NDCELLS)-1);
              end
              cellLoadIdx <= cellLoadIdxV;
              // Capture the number of rows written. Bit 0 of the write data
              // is the end of row flag.
              if (cellWrite & cellWrData[0]) begin
                numRows <= numRows+`PAD(1'b1, numBits(`NROWS)-1);
              end
            end
            // Read data is registered because putting RAM output directly onto a
            // peripheral read bus will often cause timing issues.
            if (latchRdDataV) begin
              cellRdDataOut <= cellRamRdData;
            end
            latchRdDataV = cellRead;
          end
          // Initialise the row/col counters. This count when the block is being
          // configured (ie. enable==0), so the best way to clear them is at reset
          // and as the block is disabled.
          if (disabling) begin
            numRows <= `PAD(1'b0, numBits(`NROWS)-1);
            cellLoadIdx <= `PAD(1'b0, numBits(`NDCELLS)-1);
            resettingAddr <= 1'b0;
            cellAccessActive <= 1'b0;
            latchRdDataV = 1'b0;
          end          
        end    
      end //pCellLoadIdx
    end else begin
      wire cellAccessActiveDummy;
      assign cellAccessActiveDummy = 1'b0;
      always @(*) begin
        numRows = `PAD(1'b0, numBits(`NROWS)-1);
        // A trick to trigger the block.
        cellAccessActive = cellAccessActiveDummy;
        resettingAddr = 1'b0;
        cellRdDataOut = `PAD(1'b0, `CELL_RAM_W-1);
        cellLoadIdx = `PAD(1'b0, numBits(`NDCELLS)-1);
      end
    end
  endgenerate
  
  assign cellRamSelOut = cellRamEn;
  assign cellRamAddrOut = cellAddr;
  assign cellRamWrDataOut = cellWrData;
  assign cellRamWeOut = cellWrite;

  assign {col_2, cyc_2, endRow_2} = cellRamRdData;

  assign colOut_2 = col_2;
  assign cycOut_2 = cyc_2;
  assign endRowOut_2 = endRow_2;
  assign numRowsOut = numRows;
   
endmodule

    
     
    
