//------------------------------------------------------------------------------
// ldpcEncCore.v
// 
// Description
//   Top functional level of the LDPC encoder.
// 
// Inputs:
//   nReset         : Asynchronous reset.
// (From sender (e.g. DMA device)
//   inStrobe       : High when inData is valid.
//   inDataWord     : `IP_WIDTH samples of RX_BITS.
// (From downstream HW)
//   clrToSend      : Enables output of encoded data when high.
// Outputs:
// (To sender)
//   rdyToRcv       : Enables the sender to start sending. Stays high until
//                    the expected amount of data has been transferred.
// (To downstream hardware)
//   encodeComplete   : Pulse indicate completion of a block encode.
//   packetComplete   : Goes high after last encode.
//   opStrobe         : High when opDataWord is valid.
//   opDataWord       : Encoded output.
// (From the wrapper)
//   read/write register values
//   memory outputs [inputs:the core]
//   clocks
// (To the wrapper)
//   read only register values
//   memory inputs (outputs from the core)
//   Clock enables
//
// 9 May 2011 M. Rumsey. ldpcEnc split into a wrapper && a core.
//
// (c) Copyright 2011, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`default_nettype wire
`include "ldpcEnc.vh"

module ldpcEncCore
  (
   input                             nReset,
   // Inputs from sender                
   input                             inStrobe,
   input [`IP_WIDTH-1:0]             inDataWord,
   // Inputs from output interface      
   input                             clrToSend,
   // Outputs to sender
   output [`NUM_MU_BITS-1:0]         inUserOut, 
   output                            rdyToRcvOut,
   // Outputs to downstream hardware
   output [`NUM_MU_BITS-1:0]         opUserOut,
   output                            encodeCompleteOut,
   output [`NUM_MU-1:0]              packetCompleteOut,
   output                            opStrobeOut,
   output [`OP_WIDTH-1:0]            opDataWordOut,
   output [numBits(`OP_WIDTH)-1:0]   numOpBitsOut,
   output                            lastBlockOpSampleOut,
   output                            lastOpSampleOut,
   output [`CELL_RAM_W-1:0]          cellRdDataOut,
   //----------------------------       
   //  Register values                  
   //----------------------------       
   input [`NUM_MU-1:0]               enable,
   input `opWidthArrayType           opWidthP,
   // To mem block to program the code cell information.
   input                             cellAccess,
   input                             cellRead,
   input                             cellWrite,
   input [`CELL_RAM_W-1:0]           cellWrData,
   // Code characteristics to control.  
   input  `zEnumArrayType            zEnumP,
   input  `rEnumArrayType            rEnumP,
   input  `bpsArrayType              bitsPerSymbolP,
   input  `packetLenArrayType        packetLenP,
   input  `nShrtFloorArrayType       nShrtFloorP,
   input  `shrtModArrayType          shrtModP,
   input  `nPuncFloorArrayType       nPuncFloorP,
   input  `puncModArrayType          puncModP,
   input  `nRepFloorArrayType        nRepFloorP,
   input  `repModArrayType           repModP,
   output `blkNumArrayType           curBlkNumOut,
  
   //----------------------------       
   //  Memories                         
   //----------------------------       
   output                            cellRamSelOut,
   output [`CELL_RAM_A-1:0]          cellRamAddrOut,
   output [`CELL_RAM_W-1:0]          cellRamWrDataOut,
   output                            cellRamWeOut,
   input [`CELL_RAM_W-1:0]           cellRamRdData,
   input [numBits(`NROWS)-1:0]       numRowsRom,
   output                            encRamSel0Out,
   output [`ENC_RAM_A-1:0]           encRamAddr0Out,
   output [`ENC_RAM_W-1:0]           encRamWrData0Out,
   output [`ENC_RAM_WE-1:0]          encRamWe0Out,
   output                            encRamSel1Out,
   output [`ENC_RAM_A-1:0]           encRamAddr1Out,
   output [`ENC_RAM_W-1:0]           encRamWrData1Out,
   output [`ENC_RAM_WE-1:0]          encRamWe1Out,
   input [`ENC_RAM_W-1:0]            encRamRdData0,
   input [`ENC_RAM_W-1:0]            encRamRdData1,
   output [`NUM_MU_BITS-1:0]         peUserOut,
  
   //----------------------------       
   //  Clocks                           
   //----------------------------       
   input                             ipClk,
   input                             opClk,
   input                             ctrlClk,
   input                             cellClk,
   input                             shiftClk,
   input                             rotClk,
   input                             encClk,
   // Clock Enables                     
   output                            ipClkEnOut,
   output                            opClkEnOut,
   output                            ctrlClkEnOut,
   output                            cellClkEnOut,
   output                            encRam0ClkEnOut,
   output                            encRam1ClkEnOut,
   output                            shiftClkEnOut,
   output                            rotClkEnOut,
   output                            encClkEnOut,
   //----------------------------       
   //  Debug Port                       
   //----------------------------       
   output [`DBG_PORT_WIDTH-1:0]      dbgIp1Out,
   output [`DBG_PORT_WIDTH-1:0]      dbgIp2Out,
   output [`DBG_PORT_WIDTH-1:0]      dbgOp1Out,
   output [`DBG_PORT_WIDTH-1:0]      dbgOp2Out,
   output [`DBG_PORT_WIDTH-1:0]      dbgPe1Out,
   output [`DBG_PORT_WIDTH-1:0]      dbgPe2Out,
   output [`DBG_PORT_WIDTH-1:0]      dbgStateOut);

`include "ldpcEncFuncs.vh"

  localparam MAX_BLKNUM = `MAX_BLKNUM;
  localparam BLKNUM_BITS  = numBits(MAX_BLKNUM);
  localparam LAST_USER = `NUM_MU - 1;

  //---------------------------------------------------------------------------
  // Ctrl block outputs
  //---------------------------------------------------------------------------

  wire                               enablePulse;
  wire                               disabling;
  
  // To Ip
  reg                                setRdyToRcv;
  wire                               setRdyToRcvCtrl;
  wire                               setRdyToRcvOp;
  // To cellMem
  wire [numBits(`CELL_RAM_D)-1:0]    cellMemAddr;
  wire                               cellMemRdSel;
  // To encMem
  wire [numBits(`ENC_RAM_D-1)-1:0]   encMemAddr;
  wire                               encMemSel;
  wire                               encMemWe;
  // To Enc
  wire                               shiftClkEn;
  wire                               rotClkEn;
  wire                               encClkEn;
  wire                               cellClkEn;
  wire [2:0]                         encState;
  wire                               initVec;
  wire [numBits(2)-1:0]              shiftOverride;
  // To Op
  wire                               encDone;

  wire [`NUM_MU-1:0]                 packetCompleteArray;
  wire                               packetComplete;

  //---------------------------------------------------------------------------
  // Input block outputs
  //---------------------------------------------------------------------------

  // lint: this IS used. See 2 lines down.
  localparam WR_OFF_MAX = maximum(ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH)-1, 1);
  wire [`ENC_RAM_IP_WIDTH-1:0]       wrIpData;
  wire [numBits(WR_OFF_MAX)-1:0]     wrIpOffset;
  wire [numBits(`NCOLS-1)-1:0]       wrIpAddr;
  wire                               wrIpEnable;
  wire                               inputLoaded; 
  wire                               rdyToRcv;
  wire [numBits(`MAX_BLKNUM)-1:0]    ipBlkNum;

  //---------------------------------------------------------------------------
  // Output block outputs
  //---------------------------------------------------------------------------

  wire                               opDone;
  wire                               opBufferSel;
  wire                               lastBlockOpSample;
  wire                               lastOpSample;
  wire [`ENC_RAM_A-1:0]              opBufferAddr;    
  wire                               opStrobe;
  wire [`OP_WIDTH-1:0]               opDataWord;
  wire [numBits(`OP_WIDTH)-1:0]      numOpBits;
  
  //---------------------------------------------------------------------------
  // Mem block outputs
  //---------------------------------------------------------------------------

  wire [numBits(`NROWS)-1:0]         numRows;
  wire [numBits(`NROWS)-1:0]         numRowsRam;
  wire                               endRow_2;
  wire [numBits(`NDCOLS-1)-1:0]      col_2;
  wire [numBits(`Z_MAX-1)-1:0]       cyc_2;

  //---------------------------------------------------------------------------
  // Memory muxing
  //---------------------------------------------------------------------------

  // BufferStateType
  localparam [2:0] inputActive = 0;
  localparam [2:0] waitForEncode = 1;
  localparam [2:0] encodeActive = 2;
  localparam [2:0] waitForOutput = 3;
  localparam [2:0] outputActive = 4;
  localparam [2:0] waitForInput = 5;

  reg [2:0]                          buffer0State;
  reg [2:0]                          buffer1State;
  // encRam 0/1 will get muxed to Op and Pe blocks in ping pong fashion.
  wire [`ENC_RAM_W-1:0]              encRamRdDataOp;
  wire [`ENC_RAM_W-1:0]              encRamRdDataPe;
  wire                               pingPongIp;
  wire                               pingPongOp;
  wire                               pingPongPe;
  reg                                nextBlk;
  reg                                encStart;
  reg                                opStart;

  //---------------------------------------------------------------------------
  //  Pe block outputs
  //---------------------------------------------------------------------------
  
  wire [`Z_MAX-1:0]                  encMemWrData;
  
  //---------------------------------------------------------------------------
  // Values that are looked up based on ENUMS
  //---------------------------------------------------------------------------

  wire [numBits(ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH))-1:0] numWrPerZ;  
  wire [numBits(`NCOLS-1)-1:0] parityStartColInUser;
  wire [numBits(`K_MAX)-1:0] kInUser;

  wire [numBits(`NCOLS-1)-1:0] parityStartColOpUser;
  wire [numBits(`Z_MAX)-1:0] zOpUser;
  wire [numBits(`N_MAX)-1:0] nOpUser;
  wire [numBits(`K_MAX)-1:0] kOpUser;

  wire [numBits(`NCOLS-1)-1:0] parityStartColPeUser;

  //---------------------------------------------------------------------------
  // Packed 
  //---------------------------------------------------------------------------

  wire `opWidthType    opWidth       [`NUM_MU-1:0];
  wire `zEnumType      zEnum         [`NUM_MU-1:0];
  wire `rEnumType      rEnum         [`NUM_MU-1:0];
  wire `bpsType        bitsPerSymbol [`NUM_MU-1:0];
  wire `packetLenType  packetLen     [`NUM_MU-1:0];
  wire `nShrtFloorType nShrtFloor    [`NUM_MU-1:0];
  wire `shrtModType    shrtMod       [`NUM_MU-1:0];
  wire `nPuncFloorType nPuncFloor    [`NUM_MU-1:0];
  wire `puncModType    puncMod       [`NUM_MU-1:0];
  wire `nRepFloorType  nRepFloor     [`NUM_MU-1:0];
  wire `repModType     repMod        [`NUM_MU-1:0];

  //---------------------------------------------------------------------------
  // Multi-user copies of registers
  //---------------------------------------------------------------------------
  
  wire                       enableInUser;
  wire `packetLenType        packetLenInUser;
  wire `nShrtFloorType       nShrtFloorInUser;
  wire `shrtModType          shrtModInUser;
  wire `zEnumType            zEnumInUser;

  wire `bpsType              bitsPerSymbolInUser;
  wire `bpsType              bitsPerSymbolNextUser;
  wire `runningBitCountType  bitsPerSymbolInUserExt;
  wire `runningBitCountType  bitsPerSymbolNextUserExt;

  wire                       enableOpUser;
  wire `opWidthType          opWidthOpUser;
  wire `zEnumType            zEnumOpUser;
  wire `packetLenType        packetLenOpUser;
  wire `nShrtFloorType       nShrtFloorOpUser;
  wire `shrtModType          shrtModOpUser;
  wire `nPuncFloorType       nPuncFloorOpUser;
  wire `puncModType          puncModOpUser;
  wire `nRepFloorType        nRepFloorOpUser;
  wire `repModType           repModOpUser;
                             
  wire `zEnumType            zEnumPeUser;
                             
  wire                       enableAny;
  wire `blkNumArrayType      blkNumVecP;
  wire `blkNumType           blkNumVec   [`NUM_MU-1:0];
  
  reg  `runningBitCountType runningBitCount [`NUM_MU-1:0];
  reg  `runningBitCountType runningBitCountNxt;

  //reg  [numBits(`N_MAX + `REP_MAX)-1:0] n2;   //for debug
  reg  updateNextUser;
  reg  setRdyToRcvD1;
  reg  muCalcActive;

  //---------------------------------------------------------------------------
  // Multi-User features
  //---------------------------------------------------------------------------

  reg  [`NUM_MU_BITS-1:0] nextUser;
  reg  [`NUM_MU_BITS-1:0] inUser;
  reg  [`NUM_MU_BITS-1:0] opUser;
  reg  [`NUM_MU_BITS-1:0] peUser;  

  //---------------------------------------------------------------------------
  // Other
  //---------------------------------------------------------------------------

  reg [`NUM_MU-1:0] lastIpBlk;

  //---------------------------------------------------------------------------
  // Debug port
  //---------------------------------------------------------------------------

  wire [`MEM_MUX_BITS-1:0]                               dbgMemMux;
  wire [`MEM_FETCH_SIZE-1:0]                             dbgMemDataWord;
  reg [2:0]                                              stateDbg0;
  reg [2:0]                                              stateDbg1;
  reg [2:0]                                              stateDbgPe;

  //---------------------------------------------------------------------------
  // Unpack registers
  //---------------------------------------------------------------------------

  genvar idx1;
  localparam NUM_MU = `NUM_MU;
 
  `LENC_UNPACK(gZeP, zEnumP, zEnum, `Z_ENUM_BITS, NUM_MU)
  `LENC_UNPACK(gReP, rEnumP, rEnum, `R_ENUM_BITS, NUM_MU)
  `LENC_UNPACK(gBpsP, bitsPerSymbolP, bitsPerSymbol, BPS_HI+1, NUM_MU)
  `LENC_UNPACK(gShrtFP, nShrtFloorP, nShrtFloor, `NSHRTFLOOR_BITS, NUM_MU)
  `LENC_UNPACK(gPuncFP, nPuncFloorP, nPuncFloor, `NPUNCFLOOR_BITS, NUM_MU)
  `LENC_UNPACK(gRepFP, nRepFloorP, nRepFloor, `NREPFLOOR_BITS, NUM_MU)
  `LENC_UNPACK(gShrtMP, shrtModP, shrtMod, `MOD_BITS, NUM_MU)
  `LENC_UNPACK(gPuncMP, puncModP, puncMod, `MOD_BITS, NUM_MU)
  `LENC_UNPACK(gRepMP, repModP, repMod, `MOD_BITS, NUM_MU)
  `LENC_UNPACK(gOpWP, opWidthP, opWidth, `OPWIDTH_BITS, NUM_MU)
  `LENC_UNPACK(gPackLenP, packetLenP, packetLen, `PACKETLEN_BITS, NUM_MU)

  //---------------------------------------------------------------------------
  // Lookups to simplify register interface
  //---------------------------------------------------------------------------

  assign numWrPerZ = NUM_WR_PER_Z(zEnumInUser);

  assign parityStartColInUser = PARITY_START_COL(rEnum[inUser]);
  assign kInUser = K_SIZES(zEnum[inUser], rEnum[inUser]);
  assign enableInUser = enable[inUser];
  assign packetLenInUser = packetLen[inUser];
  assign nShrtFloorInUser = nShrtFloor[inUser];
  assign shrtModInUser = shrtMod[inUser];
  assign zEnumInUser = zEnum[inUser];
  assign bitsPerSymbolInUser = bitsPerSymbol[inUser];
  assign bitsPerSymbolNextUser = bitsPerSymbol[nextUser];
  
  assign parityStartColOpUser = PARITY_START_COL(rEnum[opUser]);
  assign zOpUser = Z_SIZES(zEnum[opUser]);
  assign nOpUser = N_SIZES(zEnum[opUser]);
  assign kOpUser = K_SIZES(zEnum[opUser], rEnum[opUser]);

  assign parityStartColPeUser = PARITY_START_COL(rEnum[peUser]);
  assign zEnumPeUser = zEnum[peUser];

  assign enableAny = |enable;

  // extend bitsPerSymbol to runningCount bits
  localparam W1 = (RCB_HI-BPS_HI);
  localparam W2 = (RCB_HI-BPS_HI);
  assign bitsPerSymbolInUserExt = `PAD(bitsPerSymbolInUser, W1);
  assign bitsPerSymbolNextUserExt = `PAD(bitsPerSymbolNextUser, W2);


/******************************************************************************/
/*                                                                            */
/*   Begin RW ADDED CODE! Do not overwrite in case of new LDPC release.       */
/*                                                                            */
/******************************************************************************/
  //----------------------------------------------------------------------------
  // When not selected hold the RAM read data at its last value.
  //----------------------------------------------------------------------------

  reg                                                    encRamSel0_dly;
  reg [`ENC_RAM_W-1:0]                                   encRamRdData0_buf;
  wire [`ENC_RAM_W-1:0]                                  encRamRdDataBuf0;
  reg                                                    encRamSel1_dly;
  reg [`ENC_RAM_W-1:0]                                   encRamRdData1_buf;
  wire [`ENC_RAM_W-1:0]                                  encRamRdDataBuf1;
  
  // When not selected hold the RAM read data at its last value.
  assign encRamRdDataBuf0 = encRamSel0_dly ? encRamRdData0 : encRamRdData0_buf;
  assign encRamRdDataBuf1 = encRamSel1_dly ? encRamRdData1 : encRamRdData1_buf;

  always @(posedge(ctrlClk) `RESET_STR)
  begin : pBufRam
    if (nReset == 1'b0) begin
      encRamRdData0_buf <= `PAD(1'b0, `ENC_RAM_W-1);
      encRamRdData1_buf <= `PAD(1'b0, `ENC_RAM_W-1);
      encRamSel0_dly <= 1'b0;      
      encRamSel1_dly <= 1'b0;      
    end else begin
      if (encRamSel0_dly == 1'b1) begin
        encRamRdData0_buf <= encRamRdData0;
      end
      if (encRamSel1_dly == 1'b1) begin
        encRamRdData1_buf <= encRamRdData1;
      end
      encRamSel0_dly <= encRamSel0Out;
      encRamSel1_dly <= encRamSel1Out;
      
    end
  end

/******************************************************************************/
/*                                                                            */
/*    End RW ADDED CODE.                                                      */
/*                                                                            */
/******************************************************************************/

  //---------------------------------------------------------------------------
  // Input/Output
  //---------------------------------------------------------------------------

   ldpcEncIp ip
     (
      .nReset         (nReset),
      .clk            (ipClk),
      .clkEnOut       (ipClkEnOut),
      // From control
      .enable         (enableInUser),
      .enablePulse    (enablePulse),
      .inUser         (inUser),
      .packetLen      (packetLenInUser),
      .nShrtFloor     (nShrtFloorInUser),
      .shrtMod        (shrtModInUser),
      .zEnum          (zEnumInUser),
      .k              (kInUser),
      .parityStartCol (parityStartColInUser),
      .numWrPerZ      (numWrPerZ),
      .disabling      (disabling),
      .setRdyToRcv    (setRdyToRcv),
      .nextBlk        (nextBlk),
      // From sender
      .inStrobe       (inStrobe),
      .inDataWord     (inDataWord),
      // To sender
      .rdyToRcvOut    (rdyToRcv),
      // To control
      .inputLoadedOut (inputLoaded),
      .blkNumVecOut   (blkNumVecP),     
      .blkNumOut      (ipBlkNum),
      // To memory
      .pingPongIpOut  (pingPongIp),
      .wrDataOut      (wrIpData),
      .wrOffsetOut    (wrIpOffset),
      .wrAddrOut      (wrIpAddr),
      .wrEnableOut    (wrIpEnable));

  `LENC_UNPACK(gBlkU, blkNumVecP, blkNumVec, `BLKNUM_BITS, `NUM_MU)

    // Debug: Primary input
  // lint warnings expected. Debug bits may be lost.
  assign dbgIp1Out = {wrIpAddr, wrIpOffset, wrIpEnable, inDataWord,
                      pingPongPe, pingPongOp, pingPongIp, rdyToRcv,
                      setRdyToRcv, enable, nextBlk, inputLoaded};

  // Debug: Output of input interface
  // lint warnings expected. Debug bits may be lost.
  assign dbgIp2Out = {wrIpAddr, wrIpData, wrIpOffset, wrIpEnable, nextBlk, inputLoaded};

  // !!!!!!!!! The following line has been modified by RW. Merge in case of new LDPC release !!!!!!!!!
  assign encRamRdDataOp = (pingPongOp == 1'b0) ? encRamRdDataBuf0 : encRamRdDataBuf1;

  assign enableOpUser = enable[opUser];
  assign zEnumOpUser = zEnum[opUser];
  assign nShrtFloorOpUser = nShrtFloor[opUser];
  assign shrtModOpUser = shrtMod[opUser];
  assign nPuncFloorOpUser = nPuncFloor[opUser];
  assign puncModOpUser = puncMod[opUser];
  assign nRepFloorOpUser = nRepFloor[opUser];
  assign repModOpUser = repMod[opUser];
  assign packetLenOpUser = packetLen[opUser];
  assign opWidthOpUser = opWidth[opUser];

  
  ldpcEncOp op
    (
     .nReset             (nReset),
     .opClk              (opClk),
     .disabling          (disabling),
    
     .enable             (enableOpUser),
     .opUser             (opUser),
     // Fm Regs         
     .zEnum              (zEnumOpUser),
     .k                  (kOpUser),
     .n                  (nOpUser),
     .z                  (zOpUser),
     .nShrtFloor         (nShrtFloorOpUser),
     .shrtMod            (shrtModOpUser),
     .nPuncFloor         (nPuncFloorOpUser),
     .puncMod            (puncModOpUser),
     .nRepFloor          (nRepFloorOpUser),
     .repMod             (repModOpUser),
     .packetLen          (packetLenOpUser),
     .opWidth            (opWidthOpUser),
     // Fm Top          
     .parityStartCol     (parityStartColOpUser),

     .clrToSend          (clrToSend),
     // Fm encMem       
     .opBufferData       (encRamRdDataOp),
     // Fm Ctrl         
     .opStart            (opStart),
     // to Top          
     .opStrobeOut        (opStrobe),
     .opDataWordOut      (opDataWord),
     .numOpBitsOut       (numOpBits),
     // To encMem       
     .pingPongOpOut      (pingPongOp),
     .opBufferSelOut     (opBufferSel),
     .opBufferAddrOut    (opBufferAddr),
     // To Regs/top     
     .curBlkNumOut       (curBlkNumOut),
     .packetCompleteOut  (packetCompleteArray),
    
     .setRdyToRcvOut     (setRdyToRcvOp),
     .lastBlockOpSampleOut(lastBlockOpSample),
     .lastOpSampleOut    (lastOpSample),
     // To Ctrl         
     .opClkEnOut         (opClkEnOut),
     .opDoneOut          (opDone),
     // To Debug
     .memMuxOut          (dbgMemMux),     
     .memDataWordOut     (dbgMemDataWord));

  // Debug: Input to output interface.
  // lint warnings expected. Debug bits may be lost.
  assign dbgOp1Out = {opDone, opBufferAddr, dbgMemDataWord, dbgMemMux,
                      opBufferSel, pingPongOp, opStart};

  // Debug: Output of output interface.
  // lint warnings expected. Debug bits may be lost.
  assign dbgOp2Out = {curBlkNumOut[`BLKNUM_BITS-1:0], packetComplete, opDone,lastOpSample, numOpBits,
                      opDataWord, setRdyToRcvOp, pingPongOp, opStart};
  
  //---------------------------------------------------------------------------
  // Encode Processing Element and related controller
  //---------------------------------------------------------------------------

  // The number of rows in the code is either known by parsing the info that;
  // dynamically loaded into the cell RAM or, when `USE_CELL_ROM==1 a LUT is used.
  assign numRows = (`USE_CELL_ROM == 0) ? numRowsRam : numRowsRom;
  
   ldpcEncCtrl ctrl
     (
      .ctrlClk          (ctrlClk),
      .nReset           (nReset),
      .enable           (enableAny),
      .endRow_2         (endRow_2),
      .col_2            (col_2),
      .numRows          (numRows),
      .numDataCols      (parityStartColPeUser),
      .encStart         (encStart),
      // To Input
      .setRdyToRcvOut   (setRdyToRcvCtrl),
      // To cellMem     
      .cellMemAddrOut   (cellMemAddr),
      .cellMemRdSelOut  (cellMemRdSel),
      // To encMem
      .pingPongPeOut    (pingPongPe),
      .encMemAddrOut    (encMemAddr),
      .encMemSelOut     (encMemSel),
      .encMemWeOut      (encMemWe),
      // To top
      .disablingOut     (disabling),
      .ctrlClkEnOut     (ctrlClkEnOut),
      .encodeCompleteOut (encodeCompleteOut),
      // To Ip
      .enablePulseOut   (enablePulse),
      // To Op
      .encDoneOut       (encDone),
      // To Enc         
      .shiftClkEnOut    (shiftClkEn),
      .rotClkEnOut      (rotClkEn),
      .encClkEnOut      (encClkEn),
      .encStateOut      (encState),
      .initVecOut       (initVec),
      .shiftOverrideOut (shiftOverride));


  // !!!!!!!!! The following line has been modified by RW. Merge in case of new LDPC release !!!!!!!!!
  assign encRamRdDataPe = (pingPongPe == 1'b0) ? encRamRdDataBuf0 : encRamRdDataBuf1;
  
   ldpcEncPe pe
(
      .shiftClk      (shiftClk),
      .rotClk        (rotClk),
      .encClk        (encClk),
      .shiftClkEn    (shiftClkEn),
      .rotClkEn      (rotClkEn),
      .encClkEn      (encClkEn),
      .nReset        (nReset),
      .enable        (enableAny),
      .memIn         (encRamRdDataPe),
      .encState      (encState),
      .initVec       (initVec),
      .cyc_2         (cyc_2),
      .shiftOverride (shiftOverride),
      .zEnum         (zEnumPeUser),
      .memOut        (encMemWrData));

  // Unequal length LINT warnings may occur. This is expected.
  assign dbgPe1Out = {encMemAddr, encRamRdDataPe[Z_SIZES(0)-1 : 0], encMemWe,
                      encMemSel};
  
  assign dbgPe2Out  = {encMemWrData[Z_SIZES(0)-1 : 0], shiftClkEn, rotClkEn,
                       encClkEn, encMemWe};

  always @(encState)
  begin : pPeStateDbg
    case (encState)
      inactive : begin
        stateDbgPe = 3'b000;
      end
      syndrome : begin
        stateDbgPe = 3'b001;        
      end
      firstZ : begin
        stateDbgPe = 3'b010;       
      end
      secondZ : begin
        stateDbgPe = 3'b011;      
      end
      remaining : begin
        stateDbgPe = 3'b100;      
      end
      default : begin
        stateDbgPe = 3'b111;        
      end
    endcase
  end //pPeStateDbg

  assign shiftClkEnOut = shiftClkEn;
  assign rotClkEnOut = rotClkEn;
  assign encClkEnOut = encClkEn;
  assign cellClkEnOut = cellClkEn | disabling;
  
  assign packetComplete = &(packetCompleteArray | ~enable);
  assign packetCompleteOut = packetCompleteArray | ~enable;

  //---------------------------------------------------------------------------
  // Cell Ram defined the LDPC code
  //---------------------------------------------------------------------------

   ldpcEncCellMem cellMem
     (
      .nReset            (nReset),
      .cellClk           (cellClk),
      .disabling         (disabling),
      .cellAccess        (cellAccess),
      .cellRead          (cellRead),
      .cellWrite         (cellWrite),
      .cellWrData        (cellWrData),
      .cellMemAddr       (cellMemAddr),
      .cellMemRdSel      (cellMemRdSel),
      .cycOut_2          (cyc_2),
      .endRowOut_2       (endRow_2),
      .colOut_2          (col_2),
      .numRowsOut        (numRowsRam),
      .cellClkEnOut      (cellClkEn),
      .cellRdDataOut     (cellRdDataOut),
      // Ram
      .cellRamSelOut     (cellRamSelOut),
      .cellRamAddrOut    (cellRamAddrOut),
      .cellRamWrDataOut  (cellRamWrDataOut),
      .cellRamWeOut      (cellRamWeOut),
      .cellRamRdData     (cellRamRdData));
  
  //---------------------------------------------------------------------------
  // Encode Memory controllers (actual Rams are at top level)
  //---------------------------------------------------------------------------

  ldpcEncMem encMem
    (
     // Select RAM to use
     .pingPongIp      (pingPongIp),
     .pingPongOp      (pingPongOp),
     .pingPongPe      (pingPongPe),
     // From Ip (load of input data)
     .wrData          (wrIpData),
     .wrOffset        (wrIpOffset),
     .wrAddr          (wrIpAddr),
     .wrEnable        (wrIpEnable),
     // From Ctrl (fetching of data for the enc block)
     .encMemAddr      (encMemAddr),
     .encMemSel       (encMemSel),
     .encMemWe        (encMemWe),
     // From encPe
     .encMemWrData    (encMemWrData),
     // From Op
     .opBufferSel     (opBufferSel),
     .opBufferAddr    (opBufferAddr),
     // To Top
     .encRam0ClkEnOut  (encRam0ClkEnOut),
     .encRam1ClkEnOut  (encRam1ClkEnOut),
     // Ram 0
     .encRamSel0Out    (encRamSel0Out),
     .encRamAddr0Out   (encRamAddr0Out),
     .encRamWrData0Out (encRamWrData0Out),
     .encRamWe0Out     (encRamWe0Out),
     // Ram 1 (not present when `DOUBLE_BUFFER false)
     .encRamSel1Out    (encRamSel1Out),
     .encRamAddr1Out   (encRamAddr1Out),
     .encRamWrData1Out (encRamWrData1Out),
     .encRamWe1Out     (encRamWe1Out));

  //---------------------------------------------------------------------------
  // Manage double buffers
  //---------------------------------------------------------------------------

  generate
    if (!`DOUBLE_BUFFER) begin: gDouble0
      always @(*) begin
        // Each HW resource triggers the next and input may only be done once
        // output is finished.
        nextBlk = opDone;
        encStart = inputLoaded;
        opStart = encDone;
        // Ctrl block generates rdyToRcv when current block has been outputted.
        setRdyToRcv = setRdyToRcvCtrl | setRdyToRcvOp;
        stateDbg0 = 3'b000;
        stateDbg1 = 3'b000;
        inUser  = nextUser;
        peUser  = nextUser;
        opUser  = nextUser;
      end
    end
  endgenerate // gDouble0

  generate
    if (`DOUBLE_BUFFER) begin: gDouble1
      
      always @(posedge(ctrlClk) `RESET_STR)
      begin : pDouble
        reg  [`NUM_MU-1:0] enableLastV;
        reg `runningBitCountType runningBitCountV;
        localparam N2MAX_LEFT = numBits(`N_MAX + `REP_MAX)-1;
        reg [N2MAX_LEFT:0] n2V;
        reg [N2MAX_LEFT:0] n2V1;
        integer            idx1;
        
        
        if (nReset == 1'b0) begin
          buffer0State <= inputActive;
          buffer1State <= waitForInput;
          nextBlk <= 1'b0;
          encStart <= 1'b0;
          opStart <= 1'b0;
          setRdyToRcv <= 1'b0;
          setRdyToRcvD1 <= 1'b0;
          enableLastV = `NUM_MU'b0;
          lastIpBlk <= `NUM_MU'b0;
          nextUser <= `NUM_MU_BITS'b0;
          inUser <= `NUM_MU_BITS'b0;
          opUser <= `NUM_MU_BITS'b0;
          peUser <= `NUM_MU_BITS'b0;
          `LENC_INITQ(runningBitCount, `PAD(1'b0, `RCB_HI), `NUM_MU);          
          runningBitCountNxt <= `PAD(1'b0, `RCB_HI);
          updateNextUser <= 1'b0;
          muCalcActive <= 1'b0;
          

        end else begin
          n2V1 = `PAD(1'b1, N2MAX_LEFT);
          nextBlk <= 1'b0;
          setRdyToRcv <= 1'b0;
          encStart <= 1'b0;
          opStart <= 1'b0;
          setRdyToRcvD1 <= setRdyToRcv;
         
          if (! (|enable)) begin
            
            buffer0State <= inputActive;
            buffer1State <= waitForInput;
            enableLastV = `NUM_MU'b0;
            lastIpBlk <= `NUM_MU'b0;
            nextUser <= `NUM_MU_BITS'b0;
            inUser <= `NUM_MU_BITS'b0;
            opUser <= `NUM_MU_BITS'b0;
            peUser <= `NUM_MU_BITS'b0;
            `LENC_INITQ(runningBitCount, `PAD(1'b0, `RCB_HI), `NUM_MU);          
            updateNextUser <= 1'b0;
            muCalcActive <= 1'b0;            
            
          end
          else begin

            // Update nextUser as directed on previous clock-cycle.
            if (updateNextUser) begin
              if (nextUser == LAST_USER) begin
                nextUser <= `NUM_MU_BITS'b0;
              end
              else begin
                nextUser <= nextUser + `NUM_MU_BITS'b1;
              end      
            end
            // For large `BPS: When a running bit count exceeds `BPS we know that it
            // is time to move on to the next user.
            // For small `BPS: One decode can fill several OFDM symbols.
            // If runningBitCount is still bigger than `BPS after the initial
            // decrement by `BPS (see setRdyTORcv below) begin it is necessary to
            // skip decoding opportunities. Consider User A has a data rate 
            // (in terms of symbols/LDPC block) of 3 && `B has a rate of 2.
            // The code will 'loop' as follows:
            //        RunningBits        encode/skip   RunningBits residue
            //          A     `B            A    `B         A      `B
            // Start    0     0           enc  enc       2BPS   `BPS
            //       2BPS   `BPS            -    -         `BPS     0
            //        `BPS     0            -   enc          0   `BPS
            //          0   `BPS           enc   -        2BPS     0
            //       2BPS     0            -   enc        `BPS   `BPS
            //        `BPS   `BPS            -    -           0     0
            //          0     0
            // You can see that the users are arbitrated in the desired 2:3 ratio.
            updateNextUser <= 1'b0;
            if (setRdyToRcvD1) begin
              // nextUser will have settled by now && inUser==nextUser.
              // Data for this user will now be loaded.
              // Update running bit count for this user && decide whether
              // to move on to another user.
              runningBitCountV = runningBitCountNxt;
              if (runningBitCountV >= bitsPerSymbolInUserExt) begin
                updateNextUser <= 1'b1;
                muCalcActive <= 1'b1;
                runningBitCountV = runningBitCountV - bitsPerSymbolInUserExt;
              end
              runningBitCount[inUser] <= runningBitCountV;
            end
            else if ((lastIpBlk[nextUser] || !enable[nextUser]) &&
              !updateNextUser) begin
              // If the user is disabled or finished begin skip it.
              // updateNextUser processing takes 2 cycles hence updateNextUser==0
              // check.
              updateNextUser <= 1'b1;
              muCalcActive <= 1'b1;
            end
            else if ((runningBitCount[nextUser] >= bitsPerSymbolNextUserExt) &&
                                                  !updateNextUser) begin
              // Small `BPS case. Skip this decoding opportunity.
              runningBitCount[nextUser] <= runningBitCount[nextUser] -
                                           bitsPerSymbolNextUserExt;              
              updateNextUser <= 1'b1;               
              muCalcActive <= 1'b1;
            end
            else if (!updateNextUser) begin
              // nextUser has settled.            
              muCalcActive <= 1'b0;
            end
            // Work out number of bits in the next block. This is a
            // pre-calculation for the above code to improve timing.

            n2V = `PAD(N_SIZES(zEnum[nextUser]), N2MAX_LEFT-(numBits(`N_MAX)-1)) -
                  `PAD(nShrtFloor[nextUser], N2MAX_LEFT-(numBits(`K_MAX-1)-1)) -
                  `PAD(nPuncFloor[nextUser], N2MAX_LEFT-(numBits(`M_MAX-1)-1)) +
                  `PAD(nRepFloor[nextUser], N2MAX_LEFT-(`NREPFLOOR_BITS-1));
            if (blkNumVec[nextUser] < shrtMod[nextUser]) begin
              n2V = n2V - n2V1;
            end
            if (blkNumVec[nextUser] < puncMod[nextUser]) begin
              n2V = n2V - n2V1;
            end
            if (blkNumVec[nextUser] < repMod[nextUser]) begin
              n2V = n2V + n2V1;
            end
            runningBitCountNxt <= runningBitCount[nextUser] + `PAD(n2V, RCB_HI-N2MAX_LEFT);
            
            // There are 3 HW resources in a line (input, encode and output) and each
            // memory buffer is passed along the line but is not allowed to
            // catch up with the other buffer. Each resource has a completion
            // signal && a control signal. This state machine monitors the
            // completion and generates the control signal. Completion of input
            // triggers start of encode, completion of encode triggers start of
            // output. Completion of input also triggers start of the next input
            // actual triggering will only take place once the buffer is in the
            // right state.
            case (buffer0State)

              // Each trigger signal is anded with pingPong signals to make sure
              // that the trigger applies to this particular buffer.
              
              inputActive : begin

                // Initial trigger to load at start of packet
                if (!enableLastV[inUser] && enableInUser) begin
                  setRdyToRcv <= 1'b1;
                end
                if (inputLoaded && !pingPongIp) begin
                  buffer0State <= waitForEncode;
                  // Free the input resource for use by the other buffer.
                  nextBlk <= 1'b1;
                  lastIpBlk[inUser] <= 1'b0;
                  if (ipBlkNum == packetLenInUser-`PAD(1'b1, BLKNUM_BITS-1)) begin
                    lastIpBlk[inUser] <= 1'b1;
                  end
                end
              end
              waitForEncode : begin
                if (buffer1State != encodeActive) begin
                  buffer0State <= encodeActive;
                  encStart <= 1'b1;
                  peUser <= inUser;
                end
              end
              encodeActive : begin
                if (encDone &&  !pingPongPe) begin
                  buffer0State <= waitForOutput;
                end
              end
              waitForOutput : begin
                if (buffer1State != outputActive) begin
                  buffer0State <= outputActive;
                  opStart <= 1'b1;
                  opUser <= peUser;
                end
              end
              outputActive : begin
                if (opDone && !pingPongOp) begin
                  buffer0State <= waitForInput;
                end
              end
              waitForInput : begin
                if ((buffer1State != inputActive) && !lastIpBlk[nextUser] && !pingPongIp) begin
                  if (!muCalcActive) begin
                    buffer0State <= inputActive;
                    setRdyToRcv <= 1'b1;
                    inUser <= nextUser;
                  end                   
                end
              end
              default : begin
                buffer0State <= inputActive;
                
              end
            endcase
            
           case (buffer1State)

             // Each trigger signal is anded with pingPong signals to make sure
             // that the trigger applies to this particular buffer.
             
             inputActive : begin
               if (inputLoaded && pingPongIp) begin
                 buffer1State <= waitForEncode;
                 // Free the input resource for use by the other buffer.
                 nextBlk <= 1'b1;
                 lastIpBlk[inUser] <= 1'b0;
                 if (ipBlkNum == packetLenInUser-`PAD(1'b1, BLKNUM_BITS-1)) begin
                   lastIpBlk[inUser] <= 1'b1;
                 end
               end
             end
             waitForEncode : begin
               if (buffer0State != encodeActive) begin
                 buffer1State <= encodeActive;
                 encStart <= 1'b1;
                 peUser <= inUser;                 
               end
             end
             encodeActive : begin
               if (encDone && pingPongPe) begin
                 buffer1State <= waitForOutput;
               end
             end
             waitForOutput : begin
               if (buffer0State != outputActive) begin
                 buffer1State <= outputActive;
                 opStart <= 1'b1;
                 opUser <= peUser;                  
               end
             end
             outputActive : begin
               if (opDone && pingPongOp) begin
                 buffer1State <= waitForInput;
               end
             end
             waitForInput : begin
               if ((buffer0State != inputActive) && ~lastIpBlk[nextUser] && pingPongIp) begin
                 if (!muCalcActive) begin
                   buffer1State <= inputActive;
                   setRdyToRcv <= 1'b1;
                   inUser <= nextUser;
                 end
               end
             end
             default : begin
               buffer1State <= waitForInput;
               
             end
           endcase
            
          end
          enableLastV = enable;
          
        end
      end //pDouble
      
      always @(buffer0State, buffer1State)
      begin : pStateDbg
        case (buffer0State)
          inputActive : begin
            stateDbg0 = 3'b000;
          end
          waitForEncode : begin
            stateDbg0 = 3'b001;
          end
          encodeActive : begin         
            stateDbg0 = 3'b010;
          end
          waitForOutput : begin
            stateDbg0 = 3'b011;
          end
          outputActive : begin
            stateDbg0 = 3'b100;
          end
          waitForInput : begin
            stateDbg0 = 3'b101;
          end
          default : begin     
            stateDbg0 = 3'b111;
          end
        endcase
        case (buffer1State)
          inputActive : begin
            stateDbg1 = 3'b000;
          end
          waitForEncode : begin
            stateDbg1 = 3'b001;
          end
          encodeActive : begin         
            stateDbg1 = 3'b010;
          end
          waitForOutput : begin
            stateDbg1 = 3'b011;
          end
          outputActive : begin
            stateDbg1 = 3'b100;
          end
          waitForInput : begin
            stateDbg1 = 3'b101;
          end
          default : begin     
            stateDbg1 = 3'b111;
          end
        endcase
      end //pStateDbg
      
    end
  endgenerate //  gDouble1

  // Linting warning expected. We rely on automatic resizing here
  // and accept some debug info may be lost.
  assign dbgStateOut = {stateDbg1, stateDbg0, stateDbgPe};

  assign rdyToRcvOut = rdyToRcv;
  assign lastBlockOpSampleOut = lastBlockOpSample;
  assign lastOpSampleOut = lastOpSample;
  assign opStrobeOut = opStrobe;
  assign opDataWordOut = opDataWord;
  assign numOpBitsOut  = numOpBits;
  assign peUserOut  = peUser;
  assign inUserOut  = inUser;
  assign opUserOut  = opUser;
  
endmodule
