//-----------------------------------------------------------------------------
// ldpcEncCtrl.v
// 
// This block controls all processing other than IO. When enabled
// the encoder waits for a block of input data to be loaded && begin proceeds
// to step ldpcEncPe through its states. At the same time it arranges for the
// necessary fetches of data from encMem and cellMem. Apart from the endRow
// flag the ctrl block doesn't read || write memory data directly - the data
// goes to the enc block.
// 
// Inputs:
//   enable         : Register bit indicating that the encoder regs have been
//                    configured so the decoder is ready to accept input.
//   endRow         : A flag bit from cellMem indicating end of row.
//   col            : The column number from cellMem.
//   numRows        : Number of macro rows in the code.
//   numDataCols    : Number of macro cols used for data (not parity).
//   encStart       : A high pulse signifying that a block of input has been loaded.
//   opDone         : Pulse when encoded data has all been outputted.
// Outputs:         
//  (to Ip)         
//   setRdyToRcv    : Pulse to set rdyToRcv in Ip block.
//  (To cellMem)    
//   cellMemAddr    : Address into the cell memory.
//   cellMemRdSel   : Combined read and select for cell memory.
//  (To encMem)     
//   encMemAddr     : Address into the enc memory.
//   encMemSel      : Enable enc memory rd || write.
//   encMemWe       : Enc Memory write not read signal.
//  (To Top)        
//   encodeComplete : Pulse indicate completion of a block encode.
//  (To Enc)        
//   shiftClkEn     : Enable calculation of shift amount.
//   rotClkEn       : Enable rotation of data.
//   encClkEn       : Enable ^ processing of v vector.
//   encState       : The state that the encoder should be in. The controller
//                    uses these same states as well.
//   initIv         : Flag to initialise the V register.
//   shiftOverride  : Override of left shift info used in 'secondZ' state.
// 
// 16 Aug 2010 M. Rumsey - Initial Version.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//-----------------------------------------------------------------------------

`include "ldpcEnc.vh"

module ldpcEncCtrl
  #(parameter
    CELL_MEM_ADDR_BITS = numBits(`CELL_RAM_D),
    ENC_MEM_ADDR_BITS = numBits(`ENC_RAM_D),
    ROW_COUNT_BITS = numBits(`NROWS+1),
    NROWS_BITS = numBits(`NROWS)
  )(
   input                              ctrlClk,
   input                              nReset,
   input                              enable,
   input                              endRow_2,
   input [numBits(`NDCOLS)-1:0]       col_2,
    input [NROWS_BITS-1:0]            numRows,
   input [numBits(`NDCOLS)-1:0]       numDataCols,
   input                              encStart,
   // To Ip
   output                             setRdyToRcvOut,
   // To cellMem
   output [numBits(`CELL_RAM_D)-1:0]  cellMemAddrOut,
   output                             cellMemRdSelOut,
   // To encMem
   output                             pingPongPeOut,
   output [numBits(`ENC_RAM_D-1)-1:0] encMemAddrOut,
   output                             encMemSelOut,
   output                             encMemWeOut,
   // To top
   output                             disablingOut,
   output                             ctrlClkEnOut,
   output reg                         encodeCompleteOut,
   // to Ip
   output                             enablePulseOut,
   // to Op
   output                             encDoneOut,
   // To Enc
   output                             shiftClkEnOut,
   output                             rotClkEnOut,
   output                             encClkEnOut,
   output [2:0]                       encStateOut,
   output                             initVecOut,
   output [1:0]                       shiftOverrideOut);
  
`include "ldpcEncFuncs.vh"

  reg                                 enableLast;
  reg                                 stateChange;
  reg                                 encDone;
  reg  [numBits(`NCOLS)-1:0]          vMemIdx;
  reg  [ROW_COUNT_BITS-1:0]           rowCount;
  reg                                 endRow_1;
  reg                                 endRow;
  reg                                 endRowD1;
  reg                                 endRowD2;
  reg                                 lastRow;
  
  // Local versions of outputs
  reg [2:0]                           encState;
  reg                                 initVec_1;
  reg                                 initVec;
  reg  [CELL_MEM_ADDR_BITS-1:0]       cellMemAddr;
  reg                                 cellMemRdSel;
  reg  [numBits(`ENC_RAM_D-1)-1:0]    encMemAddr;
  reg                                 encMemSel;
  reg                                 encMemWe;
  reg                                 shiftClkEn;
  reg                                 rotClkEn;
  reg                                 encClkEn;
  reg  [1:0]                          shiftOverride;
  reg                                 setRdyToRcv;
  reg                                 pingPongPe;

  wire                                disabling;

  assign disablingOut = disabling;
  assign disabling = enableLast & !enable;
    assign enablePulseOut = enable & !enableLast;

  always @(posedge(ctrlClk) `RESET_STR)
  begin : pCtrl
    if (nReset == 1'b0) begin
      
      enableLast <= 1'b0;
      encState <= inactive;
      initVec_1 <= 1'b1;
      initVec <= 1'b1;
      encDone <= 1'b0;
      cellMemAddr <= `PAD(1'b0,CELL_MEM_ADDR_BITS-1);
      cellMemRdSel <= 1'b0;
      stateChange <= 1'b0;
      vMemIdx <= `PAD(1'b0, numBits(`NCOLS)-1);
      endRow_1 <= 1'b0;
      endRow <= 1'b0;
      endRowD1 <= 1'b0;
      endRowD2 <= 1'b0;
      rowCount <= `PAD(1'b0, ROW_COUNT_BITS-1);
      shiftClkEn <= 1'b0;
      rotClkEn <= 1'b0;
      encClkEn <= 1'b0;
      encMemAddr <= `PAD(1'b0, numBits(`ENC_RAM_D-1)-1);
      encMemWe <= 1'b0;
      encMemSel <= 1'b0;
      lastRow <= 1'b0;       
      shiftOverride <= 2'b1;
      stateChange <= 1'b0;
      
    end else begin
      
      enableLast <= enable;
      if (enable == 1'b0) begin
        encState <= inactive;
        initVec_1 <= 1'b1;
        initVec <= 1'b1;
        encDone <= 1'b0;
        cellMemAddr <= `PAD(1'b0,CELL_MEM_ADDR_BITS-1);
        cellMemRdSel <= 1'b0;
        stateChange <= 1'b0;
        vMemIdx <= `PAD(1'b0, numBits(`NCOLS)-1);
        endRow_1 <= 1'b0;
        endRow <= 1'b0;
        endRowD1 <= 1'b0;
        endRowD2 <= 1'b0;
        rowCount <= `PAD(1'b0, ROW_COUNT_BITS-1);
        shiftClkEn <= 1'b0;
        rotClkEn <= 1'b0;
        encClkEn <= 1'b0;
        encMemAddr <= `PAD(1'b0, numBits(`ENC_RAM_D-1)-1);
        encMemSel <= 1'b0;
        encMemWe <= 1'b0;
        lastRow <= 1'b0;
        shiftOverride <= 2'b1;             // forces a shift of 0.
        stateChange <= 1'b0;
      end
      else begin
        
        endRow_1 <= endRow_2;
        endRow <= endRow_1;
        endRowD1 <= endRow;
        endRowD2 <= endRowD1;
        lastRow <= 1'b0;
        
        // Defaults
        stateChange <= 1'b0;
        encMemWe <= 1'b0;
        initVec <= 1'b0;
        encDone <= 1'b0;
        
        // The detailed timing of this state machine is very intricate. It can
        // be understood using timing diagrams from ldpcEncDiagrams.odp in
        // conjunction with the algo spec.
        case (encState)
          
          inactive : begin
            cellMemAddr <= `PAD(1'b0,CELL_MEM_ADDR_BITS-1);
            if (encStart == 1'b1) begin
              cellMemRdSel <= 1'b1;
            end
            else if (cellMemRdSel == 1'b1) begin
              initVec <= 1'b1;
              encState <= syndrome;
              shiftClkEn <= 1'b1;
              cellMemAddr <= `PAD(1'b1, CELL_MEM_ADDR_BITS-1);
              stateChange <= 1'b1;
              // Intermediate vector is stored after the input data.
              vMemIdx <= numDataCols;
              endRow_1 <= 1'b0;
              rowCount <= `PAD(1'b0, ROW_COUNT_BITS-1);
              shiftOverride <= 2'b0;
            end
          end
          //-----------------------------------------------------------------
          // Create Intermediate Vector
          //-----------------------------------------------------------------
          syndrome : begin
            encClkEn <= ~endRowD1;
            encMemSel <= ~endRow;
            cellMemRdSel <= ~endRow_2 & ~endRow_1;
            shiftOverride <= 2'b0;         // shift according to cyc shift.
            initVec_1 <= stateChange | endRowD1;
            initVec <= initVec_1;
            encMemAddr <= col_2;        // may be overridden for write
            if (cellMemRdSel == 1'b1) begin
              // After each cell read we normally increment the address, but
              // if this is the last cell (endRow is set for last row) begin
              // wrap the address to zero to be sure of getting endRow_2
              // cleared down.
              if (endRow_2 == 1'b1 && rowCount == numRows-`PAD(1'b1, ROW_COUNT_BITS-1)) begin
                cellMemAddr <= `PAD(1'b0, CELL_MEM_ADDR_BITS-1);
              end
              else begin
                cellMemAddr <= cellMemAddr + `PAD(1'b1, CELL_MEM_ADDR_BITS-1);
              end
            end
            if (endRowD1 == 1'b1) begin
              // At the end of row we store v
              encMemAddr <= vMemIdx;
              vMemIdx <= vMemIdx + `PAD(1'b1, ENC_MEM_ADDR_BITS-1);
              encMemWe <= 1'b1;
              rowCount <= rowCount + `PAD(1'b1, ROW_COUNT_BITS-1);
            end
            if (initVec == 1'b1 && rowCount == numRows) begin
              encState <= firstZ;
              stateChange <= 1'b1;
              // Next state will read through VMEM
              encMemAddr <= numDataCols;
              rowCount <= `PAD(1'b0, ROW_COUNT_BITS-1);
              vMemIdx <= `PAD(1'b0, numBits(`NCOLS)-1);
              shiftClkEn <= 1'b0;
              rotClkEn <= 1'b0;            
              encClkEn <= 1'b0;
              shiftOverride <= 2'd1;
            end
            // No change on cyc_2 at this point so no clock needed.
            shiftClkEn <= ~endRow;
            // Rotator to be held after endRow to retain memRot while the
            // encMem write is done.
            rotClkEn <= ~endRowD2 & ~endRowD1;
          end
          
          //-----------------------------------------------------------------
          // Generate first Z parity bits
          //-----------------------------------------------------------------
          firstZ : begin
            cellMemRdSel <= 1'b0;
            // In this mode v is used to accumulate what we logically call 'p1'.
            // v will have been zeroed in the init state. We stay in this state
            // numRows cycles and not over all rows of v. p1 will be saved in
            // the top slice of the v memory as this is no longer needed.
            encClkEn <= 1'b1;
            encMemSel <= 1'b1;
            if (rowCount == numRows) begin
              lastRow <= 1'b1;
              encMemAddr <= `P1_BASE;
              encMemWe <= 1'b1;
              encClkEn <= 1'b0;            
            end
            else begin
              rowCount <= rowCount + `PAD(1'b1, ROW_COUNT_BITS-1);
              if (encMemAddr < `P1_BASE) begin
                encMemAddr <= encMemAddr + `PAD(1'b1, ENC_MEM_ADDR_BITS-1);
              end
            end
            if (lastRow == 1'b1) begin
              encClkEn <= 1'b0;
              encMemWe <= 1'b0;
              encState <= secondZ;
              stateChange <= 1'b1;              
              encMemAddr <= `P1_BASE;
              rowCount <= `PAD(1'b0, ROW_COUNT_BITS-1);
              shiftClkEn <= 1'b1;
            end
          end
          //-----------------------------------------------------------------
          // Generate second Z parity bits
          //-----------------------------------------------------------------
          secondZ : begin
            // Here we update v using p1. The last slice of v never gets used
            // && so (based on the 802.11n parity matrix) only the middle v and
            // V0 get updated. rowCount is used here as a general counter for
            // control.
            rowCount <= rowCount + `PAD(1'b1, ROW_COUNT_BITS-1);
            if (rowCount == `PAD(1'b1, ROW_COUNT_BITS-1)) begin
              shiftClkEn <= 1'b1;
            end
            else begin
              shiftClkEn <= 1'b0;
            end
            rotClkEn <= shiftClkEn;
            encClkEn <= rotClkEn;
            encMemSel <= 1'b1;
            case (rowCount)
              4'd0 : begin
                encMemAddr <= numDataCols + {2'b0, numRows[NROWS_BITS-1:1]};
              end
              4'd1 : begin
                encMemAddr <= `P1_BASE;
                // Arrange for P1 to get left shifted 1.
                shiftOverride <= 2'd2;
              end
              4'd2 : begin
                shiftOverride <= 2'd1;
                encMemAddr <= numDataCols;
              end
              4'd3 : begin
                encMemAddr <= numDataCols + {2'b0, numRows[NROWS_BITS-1:1]};
                encMemWe <= 1'b1;
              end
              4'd4 : begin
                encMemAddr <= numDataCols;
                encMemWe <= 1'b1;
              end
              default : begin // 5
                stateChange <= 1'b1;
                encState <= remaining;
                encMemAddr <= numDataCols + `PAD(1'b1, ENC_MEM_ADDR_BITS-1);
              end
            endcase
          end
          
          //-----------------------------------------------------------------
          // Generate remaining Z parity bits
          //-----------------------------------------------------------------
          default : begin // remaining
            // Update the second slice of vMem onwards by xoring each slice with
            // the previous slice. From the previous state, v is assumed to
            // already hold the lowest slice of the new vMem so the mem is used
            // to fetch the following slice. 3 cycles are needed per V slice so
            // we re-use rowCount for cycle counter and vMemIdx for the V slice
            // number.
            encClkEn <= 1'b1;
            encMemSel <= 1'b1;
            
            // Alternate between reading and writing the memory. On entering the
            // state the first read is already done.
            encMemWe <= !encMemWe;
            if (encMemWe == 1'b1) begin
              if (vMemIdx == `PAD(numRows-`PAD(2'd2, NROWS_BITS-2),
                                  numBits(`NCOLS)-NROWS_BITS)) begin
                encDone <= 1'b1;
                encState <= inactive;
                encMemSel <= 1'b0;
                encClkEn <= 1'b0;        // Power saving
              end
              else begin
                encClkEn <= 1'b0;        // Hold v at its previous value
                encMemAddr <= encMemAddr + `PAD(1'b1, ENC_MEM_ADDR_BITS-1);
              end
            end else begin
              vMemIdx <= vMemIdx + `PAD(1'b1, ENC_MEM_ADDR_BITS-1);              
            end
          end
        endcase
        
      end
    end
  end //pCtrl

  //---------------------------------------------------------------------------
  // Block Counter and main control
  //---------------------------------------------------------------------------

  always @(posedge(ctrlClk) `RESET_STR)
  begin : pBlk
    if (nReset == 1'b0) begin
      encodeCompleteOut <= 1'b0;
      setRdyToRcv <= 1'b0;
      pingPongPe <= 1'b0;
    end else begin
      setRdyToRcv <= 1'b0;
      if (enable == 1'b0) begin
        encodeCompleteOut <= 1'b0;
        pingPongPe <= 1'b0;
      end
      else begin
        // Rdy to rcv on first cycle of enable.
        if (!`DOUBLE_BUFFER) begin
          setRdyToRcv <= !enableLast;
        end
        encodeCompleteOut <= 1'b0;
        if (encDone == 1'b1) begin
          encodeCompleteOut <= 1'b1;
          if (`DOUBLE_BUFFER) begin
            pingPongPe <= !pingPongPe;
          end
        end
      end
    end
  end //pBlk

  assign cellMemAddrOut = cellMemAddr;
  assign cellMemRdSelOut = cellMemRdSel;
  assign encMemAddrOut = encMemAddr;
  assign encMemSelOut = encMemSel;
  assign encMemWeOut = encMemWe;
  
  assign ctrlClkEnOut = enable | enableLast | disabling;
  assign shiftClkEnOut = shiftClkEn | disabling;
  assign rotClkEnOut = rotClkEn | disabling;
  assign encClkEnOut = encClkEn | disabling; // For abort by forcing enable=0.

  assign encStateOut = encState;
  assign initVecOut = initVec;
  assign shiftOverrideOut = shiftOverride;
  assign setRdyToRcvOut = setRdyToRcv;
  assign encDoneOut = encDone;
  assign pingPongPeOut = pingPongPe;
  
endmodule
