//------------------------------------------------------------------------------
// ldpcEncIp.v
// 
// Description
//   The input controller. This registers the ipData word and generates addresses
//   for where the data should be stored. Additional writes of zeros will be
//   generated automatically when shortening is required, however the last
//   ipDataWord that contained valid data should have the shortened part zeroed
//   by the sender.
//
// Inputs:
// (From control)
//   enable         : Encoder has been configured so is ready for encode.
//   packetLen      : Number of blocks in the packet.
//   nShrtFloor     : Floor of number of shortening bits.
//   shrtMod        : 1 when blkNum < nShrtMod.
//   zEnum          : The enumeration number of the Z size.
//   k              : Data-part size for the code.
//   parityStartCol : The macro column number (rel 0) where parity bits start.
//   setRdyToRcv    : Pulse at end of first encode || on first enable.
//   nextBlk        : Pulse to indicate that next block may be input.
// (From sender)
//   inStrobe       : High when inData is valid.
//   inDataWord     : `IP_WIDTH samples of 1 bit.
// (From Op block)
// Outputs:
// (To Op Block)
//   k2             : Number of data bits after shortening.
// (To sender)
//   rdyToRcv       : Enables the sender to start sending. Stays high until
//                    the expected amount of data has been transferred.
// (To control)
//   inputLoaded    : Pulse when a complete ip block has been loaded.
// (To memories)
//   wrData         : Write data of length $size(inData).
//   wrOffset       : Write is to the bits starting at wrOffset*'ENC_RAM_IP_WIDTH.
//   wrAddr         : Write address - actually the macro column number.
//   wrEnable       : Memory write enable.
//  
// 19 Aug 2010 M. Rumsey - First version.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcEnc.vh"

module ldpcEncIp
  #(parameter WOL = (numBits(`MAX(ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH)-1, 1))-1),
              NWZL = (numBits(ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH))-1))
  (
   // Clocking/reset
   input                                    nReset,
   input                                    clk,
   output                                   clkEnOut,
   // From control
   input                                    enable,
   input                                    enablePulse,
   input [`NUM_MU_BITS-1:0]                 inUser,
   input [numBits(`MAX_BLKNUM)-1:0]         packetLen,
   input [numBits(`K_MAX)-1:0]              nShrtFloor,
   input [numBits(`MAX_BLKNUM)-1:0]         shrtMod,
   input [numBits(`Z_ENUM_MAX)-1:0]         zEnum,
   input [numBits(`K_MAX)-1:0]              k,
   input [numBits(`NCOLS)-1:0]              parityStartCol,
   input [NWZL:0]                           numWrPerZ,
   input                                    disabling,
   input                                    setRdyToRcv,
   input                                    nextBlk,
   // From sender
   input                                    inStrobe,
   input [`IP_WIDTH-1:0]                    inDataWord, 
   // To sender
   output                                   rdyToRcvOut,
   // To control
   output                                   inputLoadedOut,
   output `blkNumArrayType                  blkNumVecOut, 
   output [numBits(`MAX_BLKNUM)-1:0]        blkNumOut,
   // To memory
   output                                   pingPongIpOut,
   output [`ENC_RAM_IP_WIDTH-1:0]           wrDataOut,
   output [WOL:0]                           wrOffsetOut,
   output [numBits(`NCOLS-1)-1:0]           wrAddrOut,
   output                                   wrEnableOut);

`include "ldpcEncFuncs.vh"

  // CollectionReg left bit. CollectionReg could have `ENC_RAM_IP_WIDTH-1 bits,
  // begin get another `IP_WIDTH bits, but only have 1 removed (say at end of
  // a Z row). It begin needs a further `IP_WIDTH to store new ip data.
  // LoadBase is where we put new data. Consider that the collectionReg has
  // `ENC_RAM_IP_WIDTH-1 bits [not enough to output] begin on the next cycle
  // we load `IP_WIDTH but only remove 1 bit [eg due to end of Z row quirk]
  // then loadBase goes to `ENC_RAM_IP_WIDTH-1 + (`IP_WIDTH-1). On the following
  // cycle we may add `IP_WIDTH before subtracting the number of removed samples
  // so to give headroom we need LoadBase of `ENC_RAM_IP_WIDTH-1 + (`IP_WIDTH-1)
  // + `IP_WIDTH.
  // Note CR_HI moved to ldpcEnc.vh. loadbase is a pointer into collection reg
  // so has a value as high as CR_HI.
  localparam LB_HI = numBits(`CR_HI)-1; 
  localparam BTT_BITS = numBits(`ENC_RAM_IP_WIDTH); // For bitsToTake
  // WriteOffset Hi
  //localparam WO_HI = maximum(ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH)-1, 1);
  localparam CR_BITS = `CR_HI+1;
  localparam ENC_RAM_IP_WIDTH_BITS = numBits(`ENC_RAM_IP_WIDTH);
  localparam COL_NUM_BITS = numBits(`NCOLS-1);

  // Input data registering && collection
  reg                                    inStrobeReg;
  reg  [`IP_WIDTH-1:0]                   inDataReg;
  reg  [`CR_HI:0]                        collectionReg;
  reg  [LB_HI:0]                         loadBase;
  reg  [numBits(`ENC_RAM_IP_WIDTH)-1:0]  bitsToTake;
  reg  [numBits(`K_MAX)-1:0]             numSamplesCarried [0:`NUM_MU-1];
  wire [numBits(`K_MAX)-1:0]             numSamplesCarried0;
  reg  [numBits(`IP_WIDTH)-1:0]          numBitsInIpWord;
  reg  [`IP_WIDTH-1:0]                   carriedSamples [0:`NUM_MU-1];
  reg                                    updateCarriedSamples;
  reg  [numBits(`Z_MAX)-1:0]             spaceOnRow;
  reg  [WOL:0]                           wrOffset;
  reg  [numBits(`NCOLS-1)-1:0]           wrAddr;
  reg                                    wrEnable;
  reg                                    capturing;
  wire                                   clkEn;
  reg                                    gotDataAlready;
  
  // Track how much data has been written
  wire [numBits(`Z_MAX)-1:0]             z;
  reg  [numBits(`K_MAX + `IP_WIDTH)-1:0] numIpSamples;
  reg  [numBits(`K_MAX)-1:0]             numOpSamples;
  reg  `blkNumType                       blkNumVec [0:`NUM_MU-1];
  wire [numBits(`MAX_BLKNUM)-1:0]        blkNum;
  reg  [numBits(`K_MAX)-1:0]             k2;
  wire [numBits(`K_MAX)-1:0]             shrtBlkMod;
  reg                                    shortenMode;
  reg                                    ipShortened;
  reg                                    endZRow;
  wire                                   gotDataAlreadyCmp;
  wire [NWZL:0]                          numWrPerZLess1;

  // Internal versions of outputs
  reg                                    rdyToRcv;
  reg                                    inputLoaded;
  reg                                    pingPongIp;

  genvar                                 idx1;

  localparam BLKNUM_BITS = numBits(`MAX_BLKNUM);

  assign clkEn  = capturing | setRdyToRcv | disabling | nextBlk | enablePulse;
  assign blkNum = blkNumVec[inUser];
  assign numWrPerZLess1 = numWrPerZ - `PAD(1'b1, NWZL);
  assign numSamplesCarried0 = `PAD(1'b0, numBits(`K_MAX)-1);
  
  //---------------------------------------------------------------------------
  // Work out shortening parameter
  //---------------------------------------------------------------------------

  assign z  = Z_SIZES(zEnum);

  assign shrtBlkMod = `PAD((blkNum < shrtMod ) ? 1'b1 : 1'b0, numBits(`K_MAX)-1);
  
  // This is registered as otherwise it is a timing critical path.
  always @(posedge(clk) `RESET_STR)
  begin : pPing
    integer idx1;
    if (nReset == 1'b0) begin
      pingPongIp <= 1'b0;
      `LENC_INITQ(blkNumVec, `PAD(1'b0, BLKNUM_BITS-1), `NUM_MU);
    end else begin
      if (!enable) begin
        `LENC_INITQ(blkNumVec, `PAD(1'b0, BLKNUM_BITS-1), `NUM_MU);
        pingPongIp <= 1'b0;
      end else if (nextBlk) begin
        blkNumVec[inUser] <= blkNumVec[inUser] + `PAD(1'b1, BLKNUM_BITS-1);
        if (`DOUBLE_BUFFER) begin
          pingPongIp <= ~pingPongIp;
        end
      end
    end 

  end //pPing  

  always @(k, nShrtFloor, shrtBlkMod)
  begin : pk2
    reg [numBits(`K_MAX)-1:0] shrtBitsV;
    shrtBitsV = nShrtFloor+shrtBlkMod;
    k2 = k-shrtBitsV;   
  end //pK2   
  
  assign gotDataAlreadyCmp = ((numSamplesCarried[inUser] >= k2) && `CONCAT_IP) ? 1'b1 : 1'b0;

  //---------------------------------------------------------------------------
  // Capture input data
  //---------------------------------------------------------------------------
  
  // We get `IP_WIDTH samples, which are first registered && begin moved into
  // a collection register that is `IP_WIDTH + `ENC_RAM_IP_WIDTH - 1 wide.
  // Data is taken out of the register from the bottom when there is more than
  // `ENC_RAM_IP_WIDTH bits present && any remaining data in the collection
  // register is shifted down `ENC_RAM_IP_WIDTH bits. When data is loaded into
  // the collection register it must be loaded at loadBase, which is the start
  // of free space above the remaining data. If `ENC_RAM_IP_WIDTH;
  // sufficiently large with respect to `IP_WIDTH begin the width difference will
  // compensate for the last write of each Z row where only
  // Z-ENC_RAM_IP_WIDTH*(Z/ENC_RAM_IP_WIDTH) samples are written.

  // Capture input data

  // Basic register of input data.
  always @(posedge(clk) `RESET_STR)
  begin : pCapture
    localparam NIS_HI = numBits(`K_MAX + `IP_WIDTH)-1;
    reg [NIS_HI:0]                        numIpSamplesV;
    reg [numBits(`K_MAX)-1:0]             numSamplesCarriedV;
    reg [numBits(`IP_WIDTH-1)-1:0]        shiftDownByV;
    integer                               idx1;
    
    // Asynchronous reset.
    if (nReset == 1'b0) begin
      inStrobeReg <= 1'b0;
      ipShortened <= 1'b0; 
      rdyToRcv <= 1'b0;
      `LENC_INITQ(numSamplesCarried, `PAD(1'b0, numBits(`K_MAX)-1), `NUM_MU);
      updateCarriedSamples <= 1'b0;
      // Different to VHDL. Always reset.
      // if (`RESET_ALL) begin
      numBitsInIpWord <= `PAD(1'b0, numBits(`IP_WIDTH)-1);
      numIpSamples <= `PAD(1'b0, NIS_HI);
      `LENC_INITQ(carriedSamples, `IP_WIDTH, `NUM_MU);      
      inDataReg <= `PAD(1'b0, `IP_WIDTH-1);
      // end
    end else if (clkEn == 1'b1) begin

      numBitsInIpWord <= `IP_WIDTH_BITS'd`IP_WIDTH;
      updateCarriedSamples <= 1'b0;
      // First set rdy to rcv.
      if (setRdyToRcv == 1'b1 && blkNum < packetLen) begin
        rdyToRcv <= !gotDataAlreadyCmp;
      end
      if (setRdyToRcv == 1'b1 || enablePulse == 1'b1) begin
        if (enablePulse == 1'b1 || !`CONCAT_IP) begin
          numIpSamples <= `PAD(1'b0, NIS_HI);
          numSamplesCarried[inUser] <= `PAD(1'b0, numBits(`K_MAX)-1);
        end
        else begin
          // Input concatentation mode. We may have already received some
          // bits for the current block at tail end of last block.
          numIpSamples <= numSamplesCarried[inUser];
        end
      end
      // Receive the data.        
      if (rdyToRcv == 1'b1) begin
        // retime the strobe
        inStrobeReg <= inStrobe;
        // register the data
        if (inStrobe == 1'b1) begin
          inDataReg <= inDataWord;
          numIpSamplesV = numIpSamples + `PAD(`IP_WIDTH_BITS'd`IP_WIDTH, NIS_HI-(`IP_WIDTH_BITS-1));
          numIpSamples <= numIpSamplesV;

          if (numIpSamplesV >= k2) begin
            // We have loaded the last input so set a flag to zero inDataReg
            // on the next cycle.
            ipShortened <= 1'b1;
            rdyToRcv <= 1'b0;
            numSamplesCarriedV = numIpSamplesV - k2;
            numIpSamplesV = k2;
            numSamplesCarried[inUser] <= numSamplesCarriedV;             
            numBitsInIpWord <= `IP_WIDTH_BITS'd`IP_WIDTH - numSamplesCarriedV[`IP_WIDTH_BITS-1:0];
            updateCarriedSamples <= 1'b1;
          end
        end
      end
      if (updateCarriedSamples && `CONCAT_IP) begin
        if (numSamplesCarried[inUser] > numSamplesCarried0) begin
          shiftDownByV = `IP_WIDTH_BITS'd`IP_WIDTH - numSamplesCarried[inUser][`IP_WIDTH_BITS-1:0];
          // lint: expect to have lost some bits.
          carriedSamples[inUser] <= shiftDownIp(inDataReg, numBits(`IP_WIDTH-1)-1,
                                      0, shiftDownByV);
        end
      end
      if (inputLoaded) begin
        // Cleardown signals at the end of the input for the code block.
        inStrobeReg <= 1'b0;
        ipShortened <= 1'b0;          
      end
      else if (ipShortened) begin
        // Zero the data in case of shortening. ipShortened relates to
        // the input sampling and is not to be confused with 
        // shortenMode that relates to the RAM load, which lags the input.
        inDataReg <= `PAD(1'b0, `IP_WIDTH-1);
        inStrobeReg <= 1'b1;
      end
      // Synchronous reset.
      if (!enable) begin
        rdyToRcv <= 1'b0;
        inStrobeReg <= 1'b0;
        ipShortened <= 1'b0;           
        `LENC_INITQ(numSamplesCarried, `PAD(1'b0, numBits(`K_MAX)-1), `NUM_MU);
      end
    end

  end //pCapture

  //---------------------------------------------------------------------------
  // Re-align to RAM word size
  //---------------------------------------------------------------------------
  
  // Complicated mapping of input data onto the collection register.
  always @(posedge(clk) `RESET_STR)
  begin : pCollect
    reg [LB_HI:0]                           inIdxV;
    reg [LB_HI:0]                           loadBaseV;
    reg [numBits(`K_MAX)-1:0]               numOpSamplesV;
    reg [numBits(`ENC_RAM_IP_WIDTH)-1:0]    bitsToTakeV;
    integer                                 idx1;
    integer                                 b;

    if (nReset == 1'b0) begin
      gotDataAlready <= 1'b0;
      endZRow <= 1'b0;
      inputLoaded <= 1'b0;
      wrEnable <= 1'b0;
      wrAddr <= `PAD(1'b0, numBits(`NCOLS-1)-1);
      wrOffset <= `PAD(1'b0, WOL);
      shortenMode <= 1'b0;
      numOpSamples <= `PAD(1'b0, numBits(`K_MAX)-1);
      capturing <= 1'b0;
      // Different to VHDL. Always reset.
      // if (`RESET_ALL) begin
        spaceOnRow <= `PAD(1'b0, numBits(`Z_MAX)-1);
        loadBase <= `PAD(1'b0, LB_HI);          
        collectionReg <= `PAD(1'b0, `CR_HI);
        bitsToTake <= `PAD(1'b0, numBits(`ENC_RAM_IP_WIDTH)-1);
      // end
    end else if (clkEn == 1'b1) begin
      // Defaults
      inputLoaded <= 1'b0;
      
      // Initialise
      if (setRdyToRcv == 1'b1) begin
        endZRow <= 1'b0;
        numOpSamples <= `PAD(1'b0, numBits(`K_MAX)-1);
        spaceOnRow <= z;
        wrEnable <= 1'b0;
        wrAddr <= `PAD(1'b0, numBits(`NCOLS-1)-1);
        wrOffset <= `PAD(1'b0, WOL);
        shortenMode <= 1'b0;
        capturing <= setRdyToRcv;
        loadBase <= `PAD(1'b0, LB_HI);          
        collectionReg <= `PAD(1'b0, `CR_HI);
        gotDataAlready <= 1'b0;
      end
      
      // In the event of input block concatentation, an input word may
      // contain data for two blocks && the saved data for the 2nd block
      // needs to be injected into the collection register.
      if (setRdyToRcv && (numSamplesCarried[inUser] > numSamplesCarried0) && `CONCAT_IP) begin
        loadBase <= numSamplesCarried[inUser][LB_HI:0];
        collectionReg <= `PAD(carriedSamples[inUser], `CR_HI+1-numBits(`K_MAX));
        // Detect special case where we already have enough input data (left over from
        // previous block) to complete the current block.
        gotDataAlready <= gotDataAlreadyCmp;
      end
      if (shortenMode == 1'b1) begin
        
        //-------------------
        // Shortening Mode
        //-------------------

        wrEnable <= 1'b1;
        collectionReg <= `PAD(1'b0, `CR_HI);
        if (wrOffset == numWrPerZLess1[WOL:0]) begin  
          wrOffset <= `PAD(1'b0, WOL);
          wrAddr <= wrAddr + `PAD(1'b1, COL_NUM_BITS-1);
       end
        else begin
          wrOffset <= wrOffset + `PAD(1'b1, WOL);
        end
        
      end else if ((inStrobeReg & rdyToRcv) ||
                   (ipShortened & ~inputLoaded) || gotDataAlready) begin
          
        wrEnable <= 1'b0;
        endZRow <= 1'b0;

        //-------------------
        // Normal Mode
        //-------------------
        
        // Loadbase is the index in collectionReg where the Ip word gets
        // put. The signal version is what we use. The signal version
        // prepares for the next load by adding `IP_WIDTH while removing
        // the number of samples will have been transferred to the RAM.

        // After the following, loadBaseV tells us how many samples will
        // reside in collectionReg if none are shifted out.
        loadBaseV = loadBase + `PAD(numBitsInIpWord, LB_HI-(numBits(`IP_WIDTH)-1));
       
        // Detect when there are enough samples to transfer && correct
        // loadBaseV if data is removed.
        if (loadBaseV >=
            `PAD(`ENC_RAM_IP_WIDTH_BITS'd`ENC_RAM_IP_WIDTH,
                 LB_HI-(`ENC_RAM_IP_WIDTH_BITS-1))) begin 
          wrEnable <= 1'b1;
          // Mostly we remove the full RAM_IP_WIDTH samples except when
          // there is less space on the RAM row || there is less data.
          if (spaceOnRow < `PAD(`ENC_RAM_IP_WIDTH_BITS'd`ENC_RAM_IP_WIDTH,
                                numBits(`Z_MAX)-`ENC_RAM_IP_WIDTH_BITS)) begin
            bitsToTakeV = spaceOnRow[BTT_BITS-1:0];
          end
          else begin
            bitsToTakeV = `ENC_RAM_IP_WIDTH_BITS'd`ENC_RAM_IP_WIDTH;
          end

          // Reduce the base for loading data if data will also be taken out.
          loadBaseV = loadBaseV - `PAD(bitsToTakeV, LB_HI-(BTT_BITS-1));
          bitsToTake <= bitsToTakeV;
          
          if (spaceOnRow <= `PAD(bitsToTakeV, numBits(`Z_MAX)-BTT_BITS)) begin
            spaceOnRow <= z;
            endZRow <= 1'b1;
          end
          else begin
            spaceOnRow <= spaceOnRow -
              `PAD(bitsToTakeV, numBits(`Z_MAX)-numBits(`ENC_RAM_IP_WIDTH));
          end
        end
        loadBase <= loadBaseV;
        
        // Manage the write addresses ready for next write.
        if (wrEnable == 1'b1 || gotDataAlready == 1'b1) begin
          // This is the number of output samples that will be written.
          numOpSamplesV = numOpSamples +
                                 `PAD(bitsToTake, numBits(`K_MAX)-BTT_BITS);
          if (numOpSamplesV >= k2) begin
            shortenMode <= 1'b1;
            wrEnable <= 1'b1;
          end
          numOpSamples <= numOpSamplesV;
          if (gotDataAlready == 1'b1) begin
            wrOffset <= `PAD(1'b0, WOL);            
          end
          else if (endZRow == 1'b1) begin
            wrAddr <= wrAddr + `PAD(1'b1, COL_NUM_BITS-1);
            wrOffset <= `PAD(1'b0, WOL);
          end else begin
            wrOffset <= wrOffset + `PAD(1'b1, WOL);
          end
        end
        // Shift down collection register if data has been taken out.
        if (wrEnable == 1'b1) begin
          collectionReg <= shiftDown(collectionReg, ENC_RAM_IP_WIDTH_BITS-1,
                                     0, bitsToTake);
        end
        // Load to the collection reg at loadBase
        for (b=0; b<CR_BITS; b=b+1) begin
          if (b[LB_HI+1:0] >= {1'b0, loadBase}) begin
            // Load inData into collection register
            inIdxV = b[LB_HI:0] - loadBase;
            if (inIdxV < `PAD($unsigned(`IP_WIDTH_BITS'd`IP_WIDTH),
                              LB_HI-(`IP_WIDTH_BITS-1))) begin
              collectionReg[b[LB_HI:0]] <= inDataReg[inIdxV];
            end
          end
        end
        gotDataAlready <= 1'b0;
      end
      //------------------------
      // Detect end of capture.
      //-----------------------
      if (capturing == 1'b1) begin
        if ((wrAddr == parityStartCol-`PAD(1'b1, numBits(`NCOLS-1)-1)) &&
            (wrOffset == numWrPerZLess1 && wrEnable == 1'b1)) begin
          inputLoaded <= 1'b1;
          shortenMode <= 1'b0;
          wrEnable <= 1'b0;
        end
        // Hold clock on for a cycle after inputLoaded to clear down process.
        if (inputLoaded == 1'b1) begin
          capturing <= 1'b0;
        end
      end

    if (enable == 1'b0) begin
      inputLoaded <= 1'b0;
      endZRow <= 1'b0;
      numOpSamples <= `PAD(1'b0, numBits(`K_MAX)-1);
      // spaceOnRow <= z;
      wrEnable <= 1'b0;
      wrAddr <= `PAD(1'b0, numBits(`NCOLS-1)-1);
      wrOffset <= `PAD(1'b0, WOL);
      shortenMode <= 1'b0;
      capturing <= setRdyToRcv;
      loadBase <= `PAD(1'b0, LB_HI);          
      // collectionReg <= `PAD(1'b0, `CR_HI);
      gotDataAlready <= 1'b0;
    end

    end


  end //pCollect

  assign inputLoadedOut = inputLoaded;
  assign rdyToRcvOut = rdyToRcv;
  assign wrEnableOut = wrEnable;
  assign wrAddrOut = wrAddr;
  assign wrOffsetOut = wrOffset;
  assign wrDataOut = collectionReg[`ENC_RAM_IP_WIDTH-1:0];
  assign clkEnOut = clkEn;
  `LENC_PACK(gBlk, blkNumVec, blkNumVecOut, BLKNUM_BITS, `NUM_MU)
  assign blkNumOut = blkNum;
  assign pingPongIpOut  = pingPongIp;
  
endmodule

