//------------------------------------------------------------------------------
// ldpcEncMem.v
// 
// Description:
//   Contains the memory used to store input data && also to generate the
//   parity bits.
//
//   
// 26 Aug 2010 M. Rumsey. Created.
//
// (c) Copyright 2010-2011, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcEnc.vh"

module ldpcEncMem
  (
   // Select RAM to use
   input                                                                 pingPongIp,
   input                                                                 pingPongOp,
   input                                                                 pingPongPe,
   // From Ip (load of input data)
   input [`ENC_RAM_IP_WIDTH-1:0]                                         wrData,
   input [numBits(maximum(ceilDiv(`Z_MAX, `ENC_RAM_IP_WIDTH)-1, 1))-1:0] wrOffset,
   input [numBits(`ENC_RAM_D-1)-1:0]                                     wrAddr,
   input                                                                 wrEnable,
   // From Ctrl (fetching of data for the enc block)
   input [numBits(`ENC_RAM_D-1)-1:0]                                     encMemAddr,
   input                                                                 encMemSel,
   input                                                                 encMemWe,
   // From encPe
   input [`Z_MAX-1:0]                                                    encMemWrData,
   // From Op
   input                                                                 opBufferSel,
   input [`ENC_RAM_A-1:0]                                                opBufferAddr,
   // To Top
   output                                                                encRam0ClkEnOut,
   output                                                                encRam1ClkEnOut,
   // Port to RAM
   output                                                                encRamSel0Out,
   output [`ENC_RAM_A-1:0]                                               encRamAddr0Out,
   output [`ENC_RAM_W-1:0]                                               encRamWrData0Out,
   output [`ENC_RAM_WE-1:0]                                              encRamWe0Out,
   output                                                                encRamSel1Out,
   output [`ENC_RAM_A-1:0]                                               encRamAddr1Out,
   output [`ENC_RAM_W-1:0]                                               encRamWrData1Out,
   output [`ENC_RAM_WE-1:0]                                              encRamWe1Out);   

`include "ldpcEncFuncs.vh"

  localparam ERW = `ENC_RAM_IP_WIDTH;  // for code legibility
  localparam ERW1 = ceilDiv(`Z_MAX, ERW);
  
  wire                                   muxEnable0;
  wire [`ENC_RAM_A-1:0]                  muxAddr0;
  wire [`Z_MAX-1:0]                      muxWrData0;
  reg [ERW1-1:0]                         muxMemWe0;
  wire                                   muxEnable1;
  wire [`ENC_RAM_A-1:0]                  muxAddr1;
  wire [`Z_MAX-1:0]                      muxWrData1;
  reg [ERW1-1:0]                         muxMemWe1;
  wire [`Z_MAX-1:0]                      aliasedWrData;
  
  // Access the RAM as specified by the ctrl block || by a load of input
  // data through the ip interface || output through op interface.
  assign encRam0ClkEnOut = muxEnable0;
  assign encRam1ClkEnOut = muxEnable1;

  //---------------------------------------------------------------------------
  // Alias the input data across the RAM's full width
  //---------------------------------------------------------------------------

  // For input data, the RAM takes its input from the wrData bus. wrData 
  // is steered in a hardcoded way so that it is repeated across the width
  // of the RAM.
  //AG synthesis pb
  //AG Error:  ./ldpcEncCore_syn.v:1001: Non-constant index expression on left hand side of continuous assignment 'minimum'. (VER-327)
  //AG ENC_RAM_W=Z_MAX=81
  //AG ERW = `ENC_RAM_IP_WIDTH = 9 in our configuration
  //AG ERW1=ceilDiv(`Z_MAX, ERW)=ceilDiv(81, 9);=9
  //AG i = 0:ERW1-1 = 0:8
  //AG minimum(`ENC_RAM_W,(i+1)ERW) => ERW to 9*ERW = 9 to 81 so min =(i+1)ERW 
  generate 
    genvar i;    
    for (i = 0; i<= ERW1-1; i = i+1) begin: genWrAlias   
      assign aliasedWrData[`MIN(`ENC_RAM_W,(i+1)*ERW)-1 : i*ERW] = 
         wrData[`MIN(`ENC_RAM_W, (i+1)*ERW)-1 - i*ERW : 0];
// AG      assign aliasedWrData[minimum(`ENC_RAM_W,(i+1)*ERW)-1 : i*ERW] = 
// AG              wrData[minimum(`ENC_RAM_W, (i+1)*ERW)-1 - i*ERW : 0];
//     assign aliasedWrData[(i+1)*ERW-1 : i*ERW] = wrData[(i+1)*ERW-1 - i*ERW : 0]; // AG
    end
  endgenerate // ipMap
                           
  always @(wrOffset, wrEnable, encMemWe, pingPongIp, pingPongPe)
  begin : pIpWe0
    if (wrEnable && !pingPongIp) begin
      // Access from ip block
      muxMemWe0 = `PAD(1'b0, ERW1-1);
      muxMemWe0[wrOffset] = wrEnable;
    end
    else begin
      // access from encoder PE.
      muxMemWe0 = {ERW1{encMemWe & ~pingPongPe}};
    end
  end //pIpWe0

  always @(wrOffset, wrEnable, encMemWe, pingPongIp, pingPongPe)
  begin : pIpWe1
    if (wrEnable && pingPongIp) begin
      // Access from ip block
      muxMemWe1 = `PAD(1'b0, ERW1-1);
      muxMemWe1[wrOffset] = wrEnable;
    end
    else begin
      // access from encoder PE.
      muxMemWe1 = {ERW1{encMemWe & pingPongPe}};
    end
  end //pIpWe1
  
  //---------------------------------------------------------------------------
  // Mux the control signals to two encode buffers (0 & 1)
  //---------------------------------------------------------------------------
  
  // For single buffer mode only one of the select signals will be high at a
  // time so it is most efficient to OR them. Once gated with the pingPong signals,
  // the same is true for double buffer mode.
  assign muxEnable0 = (encMemSel & ~pingPongPe) | (wrEnable & !pingPongIp) |
                      (opBufferSel & !pingPongOp);
  assign muxEnable1 = (encMemSel & pingPongPe) | (wrEnable & pingPongIp) |
                      (opBufferSel & pingPongOp);
  
  // Mux the input. The sources should never be simultaneous.
  
  assign muxAddr0 = (wrEnable && !pingPongIp) ? wrAddr :
                    (encMemSel && !pingPongPe) ? encMemAddr : opBufferAddr;
  assign muxAddr1 = (wrEnable && pingPongIp) ? wrAddr :
                    (encMemSel && pingPongPe) ? encMemAddr : opBufferAddr;
  
  assign muxWrData0 = (wrEnable && !pingPongIp) ? aliasedWrData : encMemWrData;
  assign muxWrData1 = (wrEnable && pingPongIp) ? aliasedWrData : encMemWrData;
  
  assign encRamSel0Out = muxEnable0;
  assign encRamAddr0Out = muxAddr0;
  assign encRamWrData0Out = muxWrData0;
  assign encRamWe0Out = muxMemWe0;
  
  assign encRamSel1Out = muxEnable1;
  assign encRamAddr1Out = muxAddr1;
  assign encRamWrData1Out = muxWrData1;
  assign encRamWe1Out = muxMemWe1;
   
endmodule

    
     
    
