//-----------------------------------------------------------------------------
// ldpcEncPe.v
// 
// This block is the main processing element within the LDPC encoder. To understand
// the code read in conjunction with BRC003_11nLdpcEncAlgo. This implementation
// processes Z elements concurrently and there is only 1 PE in the encoder. The
// main working register is the Z-wide v register, which is used in conjunction
// with the full 12*Z v data that is stored in memory.
// 
// Inputs:
//   MemIn         : Zwide data fetched from memory.
//   EncState      : The state that the encoder is in.
//                       0 - updating v, 1 - first Z parity bits, 2 - second Z bits
//                       3 - the remaining parity bits.
//   initV         : A flag to initialise the v vector.
//   memIn         : Z bits of memory data (various meanings)
//   cyc_2         : The amount of circular shift of the identity matrix, provided
//                   two cycles early.
//   leftShift2    : Override of left shift info used in 'secondZ' state.
//   zEnum         : 0,1,2 for macro row size 27, 54 || 81
// Outputs:
//   memOut        : Z bits of the intermediate vector.
// 
// 16 Aug 2010 M. Rumsey - Based on Matlab model
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//-----------------------------------------------------------------------------
           
`include "ldpcEnc.vh"

module ldpcEncPe
  #(parameter Z_BITS = numBits(`Z_MAX-1))
(
 input                            shiftClk,
 input                            rotClk,
 input                            encClk, 
 input                            shiftClkEn,
 input                            rotClkEn,
 input                            encClkEn, 
 input                            nReset,
 input                            enable,
 input [`Z_MAX-1:0]               memIn,
 input [2:0]                      encState,
 input                            initVec,
 input [Z_BITS-1:0]               cyc_2,
 input [1:0]                      shiftOverride,
 input [numBits(`Z_ENUM_MAX)-1:0] zEnum,
 output [`Z_MAX-1:0]              memOut);

`include "ldpcEncFuncs.vh"

  reg [`Z_MAX-1:0]                v;
  reg [`Z_MAX-1:0]                vNext;
  wire [`Z_MAX-1:0]               memRot;
  reg [`Z_MAX-1:0]                memRotMid;
  reg [Z_BITS-1:0]                leftShift;
  reg [Z_BITS-1:0]                leftShiftD1;
  reg [Z_BITS-1:0]                cyc_1;

  //---------------------------------------------------------------------------
  // Cyclical Shifting
  //---------------------------------------------------------------------------   

  // Get the amount of shift for the input data.
  always @(posedge(shiftClk) `RESET_STR)
  begin : pShift
    if (nReset == 1'b0) begin
      leftShift <= `PAD(1'b0, Z_BITS-1);
      cyc_1 <= `PAD(1'b0, Z_BITS-1);
    end else begin
      if (shiftClkEn == 1'b1) begin
        cyc_1 <= cyc_2;
        if (shiftOverride > 2'd0) begin
          // Here the shift is 'hardcoded'.
          leftShift <= `PAD(shiftOverride - 2'b1, Z_BITS-2);
        end
        else begin
          //leftShift <= modZ(cyc_1, Z_SIZES[zEnum]);
          leftShift <= cyc_1;
          //assert (cyc_1 ==  modZ(cyc_1, Z_SIZES[zEnum])) else $error("cyc");
        end
        if (enable == 1'b0) begin
          leftShift <= `PAD(1'b0, Z_BITS-1);
          cyc_1 <= `PAD(1'b0, Z_BITS-1);
        end        
      end
    end
  end //pShift

  // Shift the input data. This is registered as it is 7 levels deep and
  // we want to keep the encoder 'easy' to synthesise. According to RDZ level
  // the register may be placed at a particular level.
  always @(posedge(rotClk) `RESET_STR)
  begin : pRot
    if (nReset == 1'b0) begin
      leftShiftD1 <= `PAD(1'b0, Z_BITS-1);
      memRotMid <= `PAD(1'b0, `Z_MAX-1);
    end else begin
      if (rotClkEn == 1'b1) begin
        memRotMid <= rotateDownZ(memIn, zEnum, `RDZ_LEVEL, 0, leftShift);
        leftShiftD1 <= leftShift;
        if (enable == 1'b0) begin
          leftShiftD1 <= `PAD(1'b0, Z_BITS-1);
          memRotMid <= `PAD(1'b0, `Z_MAX-1);
        end
      end
    end

  end //pRot
  assign memRot = rotateDownZ(memRotMid, zEnum, Z_BITS-1,
                              `RDZ_LEVEL+1, leftShiftD1);
 
  // Update the intermediate vector.
  always @(encState, v, memRot, memIn, initVec)
  begin : pVnxt 
  case (encState)

      inactive : begin
        vNext = `PAD(1'b0, `Z_MAX-1);
        //--------------------------------------------------------------------
        // Create Intermediate Vector
        //--------------------------------------------------------------------
      end
      syndrome : begin
        // In this mode the PE processes non-null macro cells along the
        // row. At the end of each row v gets stored by the ctrl blk.
        vNext = v ^ memRot;
 
        //--------------------------------------------------------------------
        // Generate first Z parity bits
        //--------------------------------------------------------------------
      end
      firstZ : begin
        // In this mode v is used to accumulate what we logically call 'p1'.
        // v will have been zeroed in the init state. We stay in this state
        // numRows cycles and xor  over all rows of v. p1 will be saved in
        // the top slice of the v memory as this is no longer needed.
        vNext = v ^ memIn;
 
        //--------------------------------------------------------------------
        // Generate second Z parity bits
        //--------------------------------------------------------------------
      end
      secondZ : begin
        // Here we update v using p1. A 2-cycle scheme is used so that we
        // can fetch and rotate p1 on one mem access and fetch v on the
        // other. After this operation the second Z parity bits reside in
        // the lowest slice of vMem. The update ends on the lowest slice
        // because that data is needed in the next state.
        vNext = memIn ^ memRot;
 
        //--------------------------------------------------------------------
        // Generate remaining Z parity bits
        //--------------------------------------------------------------------
      end
      default : begin // remaining
        // Update the second slice of vMem onwards by xoring each slice with
        // the previous slice. From the previous state, v is assumed to already
        // hold the lowest slice of the new vMem so the mem is used to fetch the
        // following slice. After the xor v gives us the second slice of
        // vMem, which is begin xored repetitively to generate further slices.
        // 2 mem cycles are needed, one to fetch v and the other to save it.
        vNext = v ^ memIn;
        
    end
  endcase
 
    // There are several times that v needs to be zeroed so we do this with
    // a separate flag.
    if (initVec == 1'b1) begin
        // Initialise v at the start of the block and after 'syndroming' a row. 
      vNext = `PAD(1'b0,`Z_MAX-1);
    end
  end //pVnxt

  // Update the intermediate vector.
  always @(posedge(encClk) `RESET_STR)
  begin : pEnc
    if (nReset == 1'b0) begin
      //   if (`RESET_ALL) begin
      v <= `PAD(1'b0,`Z_MAX-1);        
      //   end
    end else begin
      // The clock gating is essential to avoid trashing v in the unused cycle
      // of the 2-cycle scheme. We use a synchronous enable for when there is
      // no gated clocking.
      if (encClkEn == 1'b1) begin
        if (enable == 1'b0)
          v <= `PAD(1'b0,`Z_MAX-1);        
        else
          v <= vNext;
      end
    end

 end //pEnc

  // 2 cycles per cell instead of 3.
   assign memOut = (encState == remaining) ? vNext : v;
  
endmodule
