//-----------------------------------------------------------------------------
// ldpcEncRegs.v
// 
// Description
//   "Memory mapped" registers for the LDPC encoder. A basic single cycle
//   interface is provided. This provides raw access to the registers and
//   may be easily interfaced to a more substantial bus protocol with
//   external (or internal) adaption logic.
// 
// Inputs:
//   nReset           : Asynchronous reset.
//   clk              : Master clock.
// (From the processor)
//   writeEn          : Active high write enable.
//   readEn           : Active high read enable.
//   addr             : Address bus from processor.
//   wrData           : Write data bus from processor.
// (Fm CellMem)
//   cellRdData       : Readback data from cell memory (debug access)
// (Read-only registers)
//   curBlkNum        : Block number within the packet that is being encoded.
//  
// Outputs:
// (To the processor)
//   rdDataOut        : Read data bus.
// [Control registers:ctrl && elsewhere]
//   enable           : Master enable. Blip this low between encodes to init.
//   packetLen        : Number of blocks in the packet.
//   zEnum            : Select from `Z_SIZES in ldpcEncPkg.
//   rEnum            : Select from `Z_SIZES in ldpcEncPkg.
//   nShrtFloor       : Floor of number of shorting bits.
//   shrtMod          : 1 when blkNum < nShrtMod.
//   nPuncFloor       : Floor of number of shorting bits.
//   puncMod          : 1 when blkNum < nShrtMod.
//   nRepFloor        : Floor of number of repetition bits.
//   repMod           : 1 when blkNum < nRepMod.
// (To cellMem)
//   cellAccess       : Cell memory is enabled for access (rather than encode)
//   cellRead         : Cell memory read access
//   cellWrite        : Cell memory read access
//   cellWrData       : Cell memory write data
//   
// 
// 21 Apr 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcEnc.vh"

module ldpcEncRegs
  (
   input                             nReset,
   input                             clk,
   // Processor interface inputs.
   input [`NUM_MU_BITS-1:0]          selUser,  
   input [`BUS32:0]                  writeEn,
   input                             readEn,
   input [`NUM_ADDR_BITS-1:0]        addr,
   input [`NUM_DATA_BITS-1:0]        wrData,

   // Read only registers that may be monitored by processor.
   input `blkNumArrayType            curBlkNum,
   input [`CELL_RAM_W-1:0]           cellRdData,

   output                            clkEnOut,
   output [`NUM_DATA_BITS-1:0]       rdDataOut,
  
   // Control registers.
  
   // Master enable to control.
   output [`NUM_MU-1:0]             enableOut,
   // To mem block to program the code cell information.
   output                           cellAccessOut,
   output                           cellReadOut,
   output                           cellWriteOut,
   output reg [`CELL_RAM_W-1:0]     cellWrDataOut,
   // Code characteristics to control.
   output `zEnumArrayType           zEnumOut,
   output `rEnumArrayType           rEnumOut,
   output `bpsArrayType             bitsPerSymbolOut,
   output `packetLenArrayType       packetLenOut,
   // Shortening etc parameters to control.
   output `nShrtFloorArrayType      nShrtFloorOut,
   output `shrtModArrayType         shrtModOut,
   output `nPuncFloorArrayType      nPuncFloorOut,
   output `puncModArrayType         puncModOut,
   output `nRepFloorArrayType       nRepFloorOut,
   output `repModArrayType          repModOut,
   // Other                          
   output `opWidthArrayType         opWidthOut);
  
`include "ldpcEncFuncs.vh"

  localparam NUM_MU = `NUM_MU;

  reg [NUM_MU-1:0]        enable;
  reg `packetLenType       packetLen     [NUM_MU-1:0];
  reg `zEnumType           zEnum         [NUM_MU-1:0];
  reg `rEnumType           rEnum         [NUM_MU-1:0];
  reg `bpsType             bitsPerSymbol [NUM_MU-1:0];
  reg `nShrtFloorType      nShrtFloor    [NUM_MU-1:0];
  reg `shrtModType         shrtMod       [NUM_MU-1:0];
  reg `nPuncFloorType      nPuncFloor    [NUM_MU-1:0];
  reg `puncModType         puncMod       [NUM_MU-1:0];                        
  reg `nRepFloorType       nRepFloor     [NUM_MU-1:0];
  reg `repModType          repMod        [NUM_MU-1:0];                        
  reg `opWidthType         opWidth       [NUM_MU-1:0];
  wire `blkNumType         curBlkNumU    [NUM_MU-1:0];

  wire [`NUM_ADDR_BITS-2:0] addr2;
  wire                      rdSel0;
  wire                      rdSel1;
  wire                      writeEn0;
  wire                      writeEn1;
  wire [15:0]               wrData0;
  wire [15:0]               wrData1;
  reg  [15:0]               rdData0;
  reg  [15:0]               rdData1;
                                                    
  reg                       cellAccess;
  reg                       cellWrite;

  genvar                    idx1;

  `LENC_UNPACK(gBlkU, curBlkNum, curBlkNumU, `BLKNUM_BITS, NUM_MU)

  generate
    if (!`BUS32) begin: gBus16
      // Support 16-bit. Odd && Even addresses are both written from
      // writeData[15:0]. LSB of adddress selects between odd/even.
      assign writeEn0 = writeEn[0] & !addr[0];
      assign writeEn1 = writeEn[0] & addr[0];  
      assign wrData0  = wrData[15:0];
      assign wrData1  = wrData[15:0];
      // Select odd/even read words according to address.
      assign rdSel0   = readEn & !addr[0];
      assign rdSel1   = readEn & addr[0];
      assign rdDataOut = (addr[0]) ? rdData1 : rdData0;
    end
  endgenerate // gBus16
  
  generate
    if (`BUS32) begin: gBus32
      // Support for 16-bit plus 32-bit. Odd && Even addresses can
      // be written simultanously. LSB of address is ignored. 
      assign writeEn0 = writeEn[0];
      assign writeEn1 = writeEn[`BUS32];  
      assign wrData0  = wrData[15:0];
      assign wrData1  = wrData[15+`BUS32*16:`BUS32*16];
      // Both words are presented
      assign rdSel0   = readEn;
      assign rdSel1   = readEn;
      assign rdDataOut = {rdData1, rdData0};
    end
  endgenerate // gBus32
  
  assign addr2 = addr[`NUM_ADDR_BITS-1:1];  
  assign clkEnOut  = (| writeEn) | cellWrite;


  //---------------------------------------------------------------------------
  // Write registers
  //---------------------------------------------------------------------------

  always @(posedge(clk) `RESET_STR)
  begin : pWrite
    integer idx1;
    localparam [`OPWIDTH_BITS-1:0] OP_WIDTH  = `OP_WIDTH;
    if (nReset == 1'b0) begin
      // register bank reset states.
      enable           <= {NUM_MU {1'b0}};
      cellAccess       <= 1'b0;
      cellWrDataOut    <= `PAD(1'b0, `CELL_RAM_W-1);
      cellWrite        <= 1'b0;
      `LENC_INITQ(packetLen, `PAD(1'b0, `PACKETLEN_BITS-1), NUM_MU);
      `LENC_INITQ(zEnum, `PAD(1'b0, `Z_ENUM_BITS-1), NUM_MU);
      `LENC_INITQ(rEnum, `PAD(1'b0, `R_ENUM_BITS-1), NUM_MU);
      `LENC_INITQ(bitsPerSymbol, `PAD(1'b0, BPS_HI), NUM_MU);
      `LENC_INITQ(nShrtFloor, `PAD(1'b0, `NSHRTFLOOR_BITS-1), NUM_MU);
      `LENC_INITQ(shrtMod, `PAD(1'b0, `MOD_BITS-1), NUM_MU);
      `LENC_INITQ(nPuncFloor, `PAD(1'b0, `NPUNCFLOOR_BITS-1), NUM_MU);
      `LENC_INITQ(puncMod, `PAD(1'b0, `MOD_BITS-1), NUM_MU);   
      `LENC_INITQ(nRepFloor, `PAD(1'b0, `NREPFLOOR_BITS-1), NUM_MU);
      `LENC_INITQ(repMod, `PAD(1'b0, `MOD_BITS-1), NUM_MU);
      `LENC_INITQ(opWidth, OP_WIDTH, NUM_MU);
      
    end else begin
      cellWrite <= 1'b0;               

      if (writeEn0) begin
        case (addr2)
          `ENABLE_ADDR/2 : begin
            enable <= wrData0[NUM_MU-1:0];
          end
          `CODE_ENUM_ADDR/2 : begin
            zEnum[selUser] <= wrData0[1:0];
            rEnum[selUser] <= wrData0[10:8];
          end
          `SHRT_FLOOR_ADDR/2 : begin nShrtFloor[selUser] <= wrData0; 
          end
          `PUNC_FLOOR_ADDR/2 : begin nPuncFloor[selUser] <= wrData0; 
          end
          `REP_FLOOR_ADDR/2 : begin nRepFloor[selUser] <= wrData0; 
          end
          `OP_WIDTH_ADDR/2 : begin opWidth[selUser] <= wrData0; 
          end
          `CODE_ADDR/2 : begin
            cellWrDataOut <= wrData0[`CELL_RAM_W-1:0];
            cellWrite <= cellAccess;
          end
          default : ;
        endcase
      end
      if (writeEn1) begin
        case (addr2)
          `CODE_ACCESS_ADDR/2 : begin
            cellAccess <= wrData1[0];
          end
          `PACKET_LEN_ADDR/2 : begin packetLen[selUser] <= wrData1; 
          end
          `BITS_PER_SYMBOL_ADDR/2 : begin bitsPerSymbol[selUser] <= wrData1; 
          end
          `SHRT_MOD_ADDR/2 : begin shrtMod[selUser] <= wrData1; 
          end
          `PUNC_MOD_ADDR/2 : begin puncMod[selUser] <= wrData1; 
          end
          `REP_MOD_ADDR/2 : begin repMod[selUser] <= wrData1; 
          end
          default : ;
        endcase
      end
    end
  end
  
  //---------------------------------------------------------------------------
  // Read Mux
  //---------------------------------------------------------------------------

  always @(*) begin
    rdData0 <= `PAD(1'b0, `CELL_RAM_W-1);
    if (rdSel0) begin
      case (addr2) 
        `BLK_VERSION_ADDR/2   : rdData0 <= `BLK_ID_VER;
        `ENABLE_ADDR/2        : rdData0 <= enable;
        `CODE_ENUM_ADDR/2     : rdData0 <= {rEnum[selUser], 6'b000000, zEnum[selUser]};
        `SHRT_FLOOR_ADDR/2    : rdData0 <= nShrtFloor[selUser];
        `PUNC_FLOOR_ADDR/2    : rdData0 <= nPuncFloor[selUser];
        `REP_FLOOR_ADDR/2     : rdData0 <= nRepFloor[selUser];
        `OP_WIDTH_ADDR/2      : rdData0 <= opWidth[selUser];        
        `CODE_ADDR/2          : rdData0 <= cellRdData;
        default : ;
        
      endcase
    end
  end

  always @(*) begin
    rdData1 <= `PAD(1'b0, `CELL_RAM_W-1);
    if (rdSel1) begin
      case (addr2)
        `CODE_ACCESS_ADDR/2     : rdData1 <= cellAccess;
        `PACKET_LEN_ADDR/2      : rdData1 <= packetLen[selUser];
        `BITS_PER_SYMBOL_ADDR/2 : rdData1 <= bitsPerSymbol[selUser];
        `SHRT_MOD_ADDR/2        : rdData1 <= shrtMod[selUser];
        `PUNC_MOD_ADDR/2        : rdData1 <= puncMod[selUser];
        `REP_MOD_ADDR/2         : rdData1 <= repMod[selUser];
        // Status registers     
        `CUR_BLK_NUM_ADDR/2     : rdData1 <= curBlkNumU[selUser];
        default : ;
        
      endcase
    end
  end
  
  // Create a pulse when the code cell information is read. This is used to
  // increment the read address.
  assign cellReadOut = cellAccess & readEn;

  assign enableOut = enable;

  // Synopsys gets confused so force the width parameters to constants
  localparam Z_ENUM_BITS = `Z_ENUM_BITS;
  localparam R_ENUM_BITS = `R_ENUM_BITS;
  localparam BPS_HI1 = BPS_HI + 1;
  localparam NSHRTFLOOR_BITS = `NSHRTFLOOR_BITS;
  localparam NPUNCFLOOR_BITS = `NPUNCFLOOR_BITS;
  localparam NREPFLOOR_BITS = `NREPFLOOR_BITS;
  localparam MOD_BITS = `MOD_BITS;
  localparam OPWIDTH_BITS = `OPWIDTH_BITS;
  localparam PACKETLEN_BITS = `PACKETLEN_BITS;
  
 `LENC_PACK(gZeP, zEnum, zEnumOut, Z_ENUM_BITS, NUM_MU)
 `LENC_PACK(gReP, rEnum, rEnumOut, R_ENUM_BITS, NUM_MU)
 `LENC_PACK(gBpsP, bitsPerSymbol, bitsPerSymbolOut, BPS_HI1, NUM_MU)
 `LENC_PACK(gShrtFP, nShrtFloor, nShrtFloorOut, NSHRTFLOOR_BITS, NUM_MU)
 `LENC_PACK(gPuncFP, nPuncFloor, nPuncFloorOut, NPUNCFLOOR_BITS, NUM_MU)
 `LENC_PACK(gRepFP, nRepFloor, nRepFloorOut, NREPFLOOR_BITS, NUM_MU)
 `LENC_PACK(gShrtMP, shrtMod, shrtModOut, MOD_BITS, NUM_MU)
 `LENC_PACK(gPuncMP, puncMod, puncModOut, MOD_BITS, NUM_MU)
 `LENC_PACK(gRepMP, repMod, repModOut, MOD_BITS, NUM_MU)
 `LENC_PACK(gOpWP, opWidth, opWidthOut, OPWIDTH_BITS, NUM_MU)
 `LENC_PACK(gPackLenP, packetLen, packetLenOut, PACKETLEN_BITS, NUM_MU)
  
  assign cellAccessOut = cellAccess;
  assign cellWriteOut = cellWrite;

endmodule
