/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module tx_bd_data_gen
(
  /*****************************************************************************                                                                                                                                     
  * system                                                                                                                                                                           
  *****************************************************************************/                                                                                                                                     
  input  wire         rst_n,                                                                                  
  input  wire         clk,
  
  /*****************************************************************************                                                                                                                                     
  * control
  *****************************************************************************/                                                                                                                                     
  input wire          enable,
  
  input wire          frame_fec,
  input wire  [ 3:0]  frame_format,
  input wire  [15:0]  frame_service,
  input wire  [23:0]  frame_psdulen,
  input wire  [15:0]  frame_npad,
  
  /*****************************************************************************
  * MAC IF
  *****************************************************************************/
  output wire         in_ready,                                                                           
  input  wire [ 7:0]  in_data,                                                                                    
  input  wire         in_last,
  input  wire         in_valid,

  /*****************************************************************************
  * BIT PARSER IF
  *****************************************************************************/
  input  wire         out_ready,
  output reg  [7:0]   out_data,
  output reg  [7:0]   out_tail,
  output reg  [3:0]   out_len,
  output reg          out_last,
  output reg          out_valid
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  /* constants */
  localparam S_IDLE     = 3'd0,
             S_SERVICE  = 3'd1,
             S_PAYLOAD  = 3'd2,
             S_PSDUFILL = 3'd3,
             S_PAD      = 3'd4,
             S_TAIL     = 3'd5,
             S_DONE     = 3'd6;

  localparam NON_HT     = 4'd0,
             NON_HT_DUP = 4'd1,
             HT_MM      =4'd2,
             HT_GF      =4'd3,
             VHT        =4'd4,
             HE_SU      =4'd5,
             HE_MU      =4'd6,
             HE_ER_SU   =4'd7,
             HE_TB      =4'd8;

  /* input mux */
  reg  [ 3:0]  smux_len;
  reg  [ 7:0]  smux_tail,smux_data;
  reg          smux_last;
  
  /* fsm */
  reg  [ 2:0]  state;
  reg          count_dir;
  reg  [23:0]  count;
  wire [24:0]  n_count_incr;
  wire [23:0]  n_count;
  wire [ 3:0]  count_max8;
  wire         n_count_eq_psdulen;
  wire         count_eq_0;
  
  /* fifo */
  wire        fifo_wren,  fifo_rden;
  wire [ 3:0] fifo_wrlen, fifo_rdlen;
  wire [ 4:0] fifo_in_shift;
  wire [ 4:0] n_fifo_count;
  reg  [ 4:0] fifo_count;
  wire        fifo_last;
  
  wire [23:0] fifo_data_in_sh1,fifo_data_in_sh2,fifo_data_in_sh4,fifo_data_in_sh8,fifo_data_in_sh16;
  wire [23:0] fifo_data_out;
  wire [23:0] n_fifo_data;
  reg  [23:0] fifo_data;
  
  wire [23:0] fifo_tail_in_sh1,fifo_tail_in_sh2,fifo_tail_in_sh4,fifo_tail_in_sh8,fifo_tail_in_sh16;
  wire [23:0] fifo_tail_out;
  wire [23:0] n_fifo_tail;
  reg  [23:0] fifo_tail;

  /* output slot */
  reg [ 7:0] n_eof;
 
  /*****************************************************************************
  * counter update
  *****************************************************************************/
  assign n_count_incr       = {1'b0,count} + (count_dir?25'h1fffff8:25'h1); /* +1 or -8  */
  assign n_count            = n_count_incr[24]?24'd0:n_count_incr[23:0];    /* clip to 0 */
  
  assign count_eq_0         = count == 24'd0;
  assign count_max8         = (count <= 24'd8)?count[3:0]:4'd8;
  assign n_count_eq_psdulen = n_count == frame_psdulen;
  
  /*****************************************************************************
  * SOURCES MUX
  *****************************************************************************/
  always @(*)
    case(state)
      S_SERVICE:
      begin
        smux_len  = 4'd8;
        smux_tail = 8'b0;
        if(!count[0])
          {smux_last, smux_data} = {1'b0, frame_service[ 7:0]};
        else
          {smux_last, smux_data} = {1'b1, frame_service[15:8]};
      end
      S_PAYLOAD:
      begin
        smux_len  = {in_valid,3'b0};
        smux_tail = 8'b0;
        smux_data = in_data;
        smux_last = in_last;
      end 
      S_PSDUFILL:
      begin
        smux_len  = 4'd8;
        smux_tail = 8'b0;
        smux_data = 8'd0;
        smux_last = 1'b0;
      end 
      S_PAD:
      begin
        smux_len  = count_max8;
        smux_tail = 8'b0;
        smux_data = 8'd0;
        smux_last = count_eq_0;
      end 
      S_TAIL:
      begin
        smux_len = count_max8;
        case(count_max8)
          4'd1:    smux_tail = 8'b00000001;
          4'd2:    smux_tail = 8'b00000011;
          4'd3:    smux_tail = 8'b00000111;
          4'd4:    smux_tail = 8'b00001111;
          4'd5:    smux_tail = 8'b00011111;
          4'd6:    smux_tail = 8'b00111111;
          4'd7:    smux_tail = 8'b01111111;
          4'd8:    smux_tail = 8'b11111111;
          default: smux_tail = 8'b00000000;
        endcase
        smux_data = 8'd0;
        smux_last = count_eq_0;
      end 
      default:
      begin
        smux_len  = 4'd0;
        smux_tail = 8'b0;
        {smux_last, smux_data} = {1'b0,  8'd0};
      end
    endcase
  
  /*****************************************************************************
  * SOURCE FSM
  *****************************************************************************/
  wire  is_nonht_ht;
  
  assign is_nonht_ht = frame_format==NON_HT     |
                       frame_format==NON_HT_DUP |
                       frame_format==HT_MM      |
                       frame_format==HT_GF;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      count_dir <= 1'b0;
      count     <= 24'd0;
      state     <= S_IDLE;
    end
    else if(!enable)
    begin
      count_dir <= 1'b0;
      count     <= 24'd0;
      state     <= S_IDLE;
    end
    else 
    begin
      case(state)
        /*********************************************************************  
        * IDLE                                                                  
        *********************************************************************/  
        S_IDLE:                                                                   
        begin                                                                   
          count_dir <= 1'b0;
          count     <= 24'd0;
          state     <= S_SERVICE;                                                  
        end                                                                     
      
        /*********************************************************************  
        * SERVICE                                                                   
        *********************************************************************/  
        S_SERVICE:                                                               
        begin                                                                   
          if(fifo_wren)
          begin
            count <= n_count;
            if(smux_last)
            begin
              count <= 24'd0;
              state <= S_PAYLOAD;
            end
          end
        end                                                                     
        
        /*********************************************************************  
        * PAYLOAD                                                                   
        *********************************************************************/  
//      S_PAYLOAD:                                                                   
//      begin
//        if(fifo_wren)
//        begin
//          count <= n_count;
//          if(smux_last)
//          begin
//            if(!n_count_eq_psdulen)
//            begin
//              /* continue PSDU filling */
//              state <= S_PSDUFILL;
//            end
//            else
//            begin
//              /* PSDU done */
//              count_dir <= 1'b1;
//              if(symsel==4'd0)
//              begin
//                /* NONHT/HT */
//                count <= (!fec)?4'd6:4'd0;
//                state <= S_TAIL;
//              end
//              else
//              begin
//                count <= {8'b0,npad};
//                state <= S_PAD;
//              end
//            end
//          end
//        end 
//      end  
        S_PAYLOAD:                                                                   
        begin
          if(fifo_wren)
          begin
            count <= n_count;
            if(smux_last)
            begin
              if(is_nonht_ht)
              begin
                /* NONHT/HT */
                count_dir <= 1'b1;
                count <= (!frame_fec)?24'd6:24'd0;
                state <= S_TAIL;
              end
              else
              begin
                /* VHT/HE */
                if(!n_count_eq_psdulen)
                begin
                  /* continue PSDU filling */
                  state <= S_PSDUFILL;
                end
                else
                begin
                  count_dir <= 1'b1;
                  count <= {8'b0,frame_npad};
                  state <= S_PAD;
                end
              end
            end
          end
        end
        
        /*********************************************************************  
        * PSDUFILL (MAC BYTE PADDING)                                                                  
        *********************************************************************/  
        S_PSDUFILL:                                                                   
        begin
          if(fifo_wren)
          begin
            count <= n_count;
            if(n_count_eq_psdulen)
            begin
              count_dir <= 1'b1;
              if(is_nonht_ht)
              begin
                count <= (!frame_fec)?24'd6:24'd0;
                state <= S_TAIL;
              end
              else
              begin
                count <= {8'b0,frame_npad};
                state <= S_PAD;
              end
            end
          end
        end                                                                     
        
        /*********************************************************************  
        * PAD (PHY BIT PADDING)                                                                      
        *********************************************************************/  
        S_PAD:                                                                
        begin
          if(count_eq_0)
          begin
            if(is_nonht_ht)
            begin
              /* NONHT/HT */
              count <= 24'd0;
              state <= S_DONE;
            end
            else
            begin
              /* VHT/HE */
              count <= (!frame_fec)?24'd6:24'd0;
              state <= S_TAIL;
            end
          end
          else
          begin
            if(fifo_wren)
              count <= n_count;
          end
        end                                                                     
       
        /*********************************************************************  
        * TAIL                                                                   
        *********************************************************************/  
        S_TAIL:                                                                
        begin 
          if(count_eq_0)
          begin
            if(is_nonht_ht)
            begin
              count <= {8'b0,frame_npad};
              state <= S_PAD;
            end
            else
            begin
              count <= 24'd0;
              state <= S_DONE;
            end
          end
          else
          begin
            if(fifo_wren)
              count <= n_count;
          end
        end                                                                     
        
        /*********************************************************************  
        * DONE                                                                   
        *********************************************************************/  
        default: 
          count <= 24'd0;                                                               
      endcase                                                                   
    end
  end
  
  /*****************************************************************************
  * FIFO 24 entries of 2 bits {tail,data}
  *****************************************************************************/
  /* control */
  assign in_ready          = state==S_PAYLOAD && fifo_count<=5'd16;
 
  assign fifo_wren         = fifo_count<=5'd16 && smux_len!=4'd0;
  assign fifo_wrlen        = smux_len;
 
  /* note: fifo_count > 8, (strictly above 8) ensures that we always have at least 8 bit in the fifo
           this prevents to push the ultimate byte into the write slote before detecting it is the last one */ 
  assign fifo_rden         = |fifo_rdlen && (out_ready || out_len==4'd0);
 
  assign {fifo_last,fifo_rdlen} = (fifo_count>5'd8)                 ? {1'b0,4'd8}:
                                  (state==S_DONE&&fifo_count!=5'd0) ? {1'b1,fifo_count[3:0]}: 
                                                                      {1'b0,4'd0};
  
  assign fifo_in_shift     = fifo_count    - {1'b0, {4{fifo_rden}} & fifo_rdlen};
  assign n_fifo_count      = fifo_in_shift + {1'b0, {4{fifo_wren}} & fifo_wrlen};
 
  /* data */
  assign fifo_data_in_sh1  = fifo_in_shift[0]?{   15'b0,smux_data,  1'b0}:{16'b0,smux_data};
  assign fifo_data_in_sh2  = fifo_in_shift[1]?{ fifo_data_in_sh1[21:0],  2'b0}:fifo_data_in_sh1;
  assign fifo_data_in_sh4  = fifo_in_shift[2]?{ fifo_data_in_sh2[19:0],  4'b0}:fifo_data_in_sh2;
  assign fifo_data_in_sh8  = fifo_in_shift[3]?{ fifo_data_in_sh4[15:0],  8'b0}:fifo_data_in_sh4;
  assign fifo_data_in_sh16 = fifo_in_shift[4]?{ fifo_data_in_sh8[ 7:0], 16'b0}:fifo_data_in_sh8;
  assign fifo_data_out     = fifo_rden?{8'b0,fifo_data[23:8]}:fifo_data;
  assign n_fifo_data       = fifo_data_out | {24{fifo_wren}}&fifo_data_in_sh16;
  /* tail */
  assign fifo_tail_in_sh1  = fifo_in_shift[0]?{   15'b0,smux_tail,  1'b0}:{16'b0,smux_tail};
  assign fifo_tail_in_sh2  = fifo_in_shift[1]?{ fifo_tail_in_sh1[21:0],  2'b0}:fifo_tail_in_sh1;
  assign fifo_tail_in_sh4  = fifo_in_shift[2]?{ fifo_tail_in_sh2[19:0],  4'b0}:fifo_tail_in_sh2;
  assign fifo_tail_in_sh8  = fifo_in_shift[3]?{ fifo_tail_in_sh4[15:0],  8'b0}:fifo_tail_in_sh4;
  assign fifo_tail_in_sh16 = fifo_in_shift[4]?{ fifo_tail_in_sh8[ 7:0], 16'b0}:fifo_tail_in_sh8;
  assign fifo_tail_out     = fifo_rden?{8'b0,fifo_tail[23:8]}:fifo_tail;
  assign n_fifo_tail       = fifo_tail_out | {24{fifo_wren}}&fifo_tail_in_sh16;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      fifo_data  <= 24'd0;
      fifo_tail  <= 24'd0;
      fifo_count <= 5'd0;
    end
    else if(!enable)
    begin
      fifo_data  <= 24'd0;
      fifo_tail  <= 24'd0;
      fifo_count <= 5'd0;
    end
    else 
    begin
      fifo_data  <= n_fifo_data;
      fifo_tail  <= n_fifo_tail;
      fifo_count <= n_fifo_count;
    end
  end
 
  /*****************************************************************************
  * OUTPUT SLOT
  *****************************************************************************/
  always @(*)
  begin
    if(state==S_DONE)
      case(fifo_count)
        5'd1:    n_eof = 8'b00000001;
        5'd2:    n_eof = 8'b00000010;
        5'd3:    n_eof = 8'b00000100;
        5'd4:    n_eof = 8'b00001000;
        5'd5:    n_eof = 8'b00010000;
        5'd6:    n_eof = 8'b00100000;
        5'd7:    n_eof = 8'b01000000;
        5'd8:    n_eof = 8'b10000000;
        default: n_eof = 8'b00000000;
      endcase
    else
      n_eof = 8'b0;
  end
    
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      out_data  <= 8'd0;
      out_tail  <= 8'd0;
      out_len   <= 4'd0;
      out_last  <= 1'd0;
      out_valid <= 1'b0;
    end
    else if(!enable)
    begin
      out_data  <= 8'd0;
      out_tail  <= 8'd0;
      out_len   <= 4'd0;
      out_last  <= 1'd0;
      out_valid <= 1'b0;
    end
    else 
    begin
      /* slot consumed */
      if(out_ready)
      begin
        out_data  <= 8'd0;
        out_tail  <= 8'd0;
        out_len   <= 4'd0;
        out_last  <= 1'd0;
        out_valid <= 1'b0;
      end
      
      /* slot written */
      if(fifo_rden)
      begin
        out_data  <= fifo_data[7:0];
        out_tail  <= fifo_tail[7:0];
        out_len   <= fifo_rdlen;
        out_last  <= fifo_last;
        out_valid <= 1'b1;
      end
    end
  end
  
`ifdef RW_SIMU_ON
  reg [16*8-1:0] state_str;
  always @(*)
    case(state)
      S_IDLE:     state_str = "S_IDLE";
      S_SERVICE:  state_str = "S_SERVICE";
      S_PAYLOAD:  state_str = "S_PAYLOAD";
      S_PSDUFILL: state_str = "S_PSDUFILL";
      S_PAD:      state_str = "S_PAD";
      S_TAIL:     state_str = "S_TAIL";
      S_DONE:     state_str = "S_DONE";
      default:    state_str = "UNDEFINED";
    endcase 
`endif
  
endmodule
`default_nettype wire
