/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     : 
* Description      : tx_fd_interleaver
* Simulation Notes : 
* Synthesis Notes  : 
* Application Note : 
* Simulator        : 
* Parameters       : 
* Terms & concepts : 
* Bugs             : 
* Open issues and future enhancements : 
* References       : 
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module tx_bd_fifo
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire          clk,
  input wire          rst_n,
  
  /*****************************************************************************
  * 
  *****************************************************************************/
  input  wire         enable,
  
  /*****************************************************************************
  * parameters
  *****************************************************************************/
  input  wire [ 1:0]  mdmcfg_conf_bw,
  input  wire         mem_released,
  
  /*****************************************************************************
  * input interface
  *****************************************************************************/
  output wire         in_ready,
  input  wire [15:0]  in_data,
  input  wire [ 4:0]  in_len,
  input  wire         in_last,
  input  wire         in_valid,
 
  /*****************************************************************************
  * output interface
  *****************************************************************************/
  input  wire         out_ready,
  output wire [15:0]  out_data,
  output wire [ 4:0]  out_len,
  output wire         out_last,
  output wire         out_valid,

  /*****************************************************************************
  * sram interface
  *****************************************************************************/
  output wire         wen,
  output wire [ 9:0]  waddr,
  output wire [29:0]  wdata,
 
  output wire         ren,
  output wire [ 9:0]  raddr,
  input  wire [29:0]  rdata
  
);
  /*****************************************************************************
  * declarations
  *****************************************************************************/
  /* STATIC */
  reg           static_mem_released;
  
  /* IFIFO */
  reg  [  59:0] ififo;
  wire [  59:0] n_ififo;
  reg  [   5:0] ififo_count; 
  wire [   5:0] n_ififo_count; 
  wire [   4:0] ififo_wrlen; /* 0-16 */
  wire [   4:0] ififo_rdlen; /* 0-30 */
  wire [  15:0] ififo_in;
  wire [  59:0] ififo_in_sh1,ififo_in_sh2,ififo_in_sh3,ififo_in_sh4,ififo_in_sh5,ififo_in_sh6;
  wire [  59:0] ififo_out_sh1,ififo_out_sh2,ififo_out_sh3,ififo_out_sh4,ififo_out_sh5;
  wire [   5:0] ififo_in_shsel;
  wire [   4:0] ififo_out_shsel;
  reg           ififo_last;
  wire          n_ififo_last;
 
  /* MFIFO WP */
  reg [ 4:0]    mfifo_wrlen;
  reg           mfifo_last;
 
  reg [ 9:0]    w0_ptr;
  reg           w0_en;
  reg [ 9:0]    w0_addr;
  reg [29:0]    w0_data;
  reg           w0_mfifo_ready;
  reg  [  14:0] w0_mfifo_count;
  wire [  14:0] n_w0_mfifo_count;
  reg  [   4:0] w0_mfifo_wrlen;
  reg           w0_mfifo_last;
  
  reg  [   4:0] w1_mfifo_wrlen;
  reg           w1_mfifo_last;
  
  reg  [  14:0] w2_mfifo_count;
  wire [  14:0] n_w2_mfifo_count;
  reg           w2_mfifo_last;
  
  /* MFIFO RP */
  reg  [ 9:0]   r0_ptr;
  reg           r0_en;
  reg  [ 9:0]   r0_addr;
  reg  [ 4:0]   r0_ofifo_rqlen;
  reg           n_r0_ofifo_last;
  reg  [ 6:0]   r0_ofifo_count;
  wire [ 6:0]   n_r0_ofifo_count;
  reg           r0_ofifo_last;
  
  reg  [ 4:0]   r1_ofifo_rqlen;
  reg           r1_ofifo_last;
  
  reg  [ 4:0]   r2_ofifo_rqlen;
  reg           r2_ofifo_last;
  
  reg  [ 4:0]   r3_ofifo_rqlen;
  reg           r3_ofifo_last;
  reg  [29:0]   r3_data;
  
  /* OFIFO */
  reg  [ 119:0] ofifo;
  wire [ 119:0] n_ofifo;
  reg  [   6:0] ofifo_count; 
  wire [   6:0] n_ofifo_count; 
  reg  [   4:0] ofifo_wrlen;
  reg  [   4:0] ofifo_rqlen;
  wire [   4:0] ofifo_rdlen;
  wire [  29:0] ofifo_in;
  wire [ 119:0] ofifo_in_sh1,ofifo_in_sh2,ofifo_in_sh3,ofifo_in_sh4,ofifo_in_sh5,ofifo_in_sh6,ofifo_in_sh7;
  wire [ 119:0] ofifo_out_sh1,ofifo_out_sh2,ofifo_out_sh3,ofifo_out_sh4,ofifo_out_sh5;
  wire [   6:0] ofifo_in_shsel;
  wire [   4:0] ofifo_out_shsel;
  reg           ofifo_last;
  reg           n_ofifo_last;
  
  /*****************************************************************************
  * STATIC PARAMETERS
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      static_mem_released <= 1'b0;
    end
    else if(!enable)
    begin
      static_mem_released <= 1'b0;
    end
    else
    begin
      static_mem_released <= mem_released;
    end
  end
  
  /*****************************************************************************
  * IFIFO 60 bits / IN 0-16 bits / OUT 0-30 bits
  *****************************************************************************/
  assign in_ready        = ififo_count <= 6'd44;
  assign ififo_wrlen     = in_len&{5{in_valid & in_ready}};
  assign ififo_rdlen     = (!static_mem_released)?ofifo_wrlen:mfifo_wrlen;
 
  assign ififo_in_shsel  = ififo_count - {1'b0,ififo_rdlen};
  assign ififo_out_shsel = ififo_rdlen;
  assign ififo_in        = in_data & {16{in_valid & in_ready}};
  assign ififo_in_sh1    = ififo_in_shsel[0]?{    43'b0,ififo_in,  1'b0}:{44'b0,ififo_in};
  assign ififo_in_sh2    = ififo_in_shsel[1]?{ififo_in_sh1[57:0],  2'b0}:ififo_in_sh1;
  assign ififo_in_sh3    = ififo_in_shsel[2]?{ififo_in_sh2[55:0],  4'b0}:ififo_in_sh2;
  assign ififo_in_sh4    = ififo_in_shsel[3]?{ififo_in_sh3[51:0],  8'b0}:ififo_in_sh3;
  assign ififo_in_sh5    = ififo_in_shsel[4]?{ififo_in_sh4[43:0], 16'b0}:ififo_in_sh4;
  assign ififo_in_sh6    = ififo_in_shsel[5]?{ififo_in_sh5[27:0], 32'b0}:ififo_in_sh5;

  assign ififo_out_sh1   = ififo_out_shsel[0]?{ 1'b0,        ififo[59: 1]}:ififo;
  assign ififo_out_sh2   = ififo_out_shsel[1]?{ 2'b0,ififo_out_sh1[59: 2]}:ififo_out_sh1;
  assign ififo_out_sh3   = ififo_out_shsel[2]?{ 4'b0,ififo_out_sh2[59: 4]}:ififo_out_sh2;
  assign ififo_out_sh4   = ififo_out_shsel[3]?{ 8'b0,ififo_out_sh3[59: 8]}:ififo_out_sh3;
  assign ififo_out_sh5   = ififo_out_shsel[4]?{16'b0,ififo_out_sh4[59:16]}:ififo_out_sh4;
  assign n_ififo         = ififo_in_sh6 | ififo_out_sh5;
  assign n_ififo_count   = ififo_count + {1'b0,ififo_wrlen} - {1'b0,ififo_rdlen};
  assign n_ififo_last    = ififo_last | (in_valid & in_ready & in_last);
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      ififo       <= 60'b0;
      ififo_count <= 6'b0;
      ififo_last  <= 1'b0;
    end
    else if(!enable)
    begin
      ififo       <= 60'b0;
      ififo_count <= 6'b0;
      ififo_last  <= 1'b0;
    end
    else
    begin
      ififo       <= n_ififo;      
      ififo_count <= n_ififo_count;
      ififo_last  <= n_ififo_last;
    end
  end

  /*****************************************************************************
  * MFIFO WRITE PIPE
  *****************************************************************************/
  always @(*)
  begin
    if(static_mem_released)
    begin
      case(mdmcfg_conf_bw)
        2'd0:    w0_mfifo_ready = w0_mfifo_count<=15'd7650;   /*  256*30-30 */
        2'd1:    w0_mfifo_ready = w0_mfifo_count<=15'd15330;  /*  512*30-30 */
        default: w0_mfifo_ready = w0_mfifo_count<=15'd30690;  /* 1024*30-30 */
      endcase
    end
    else
    begin
      w0_mfifo_ready = 1'b0;
    end
  end
  
  always @(*)
  begin
    if(static_mem_released)
    begin
      if(w0_mfifo_ready)
      begin
        if(ififo_last && ififo_count <=5'd30)
        begin
          mfifo_wrlen = ififo_count[4:0];
          mfifo_last  = 1'b1;
        end
        else if(ififo_count>=6'd30)
        begin
          mfifo_wrlen = 5'd30;
          mfifo_last  = w0_mfifo_last;
        end
        else
        begin
          mfifo_wrlen = 5'd0;
          mfifo_last  = w0_mfifo_last;
        end 
      end
      else
      begin
        mfifo_wrlen = 5'd0;
        mfifo_last  = w0_mfifo_last;
      end
    end
    else
    begin
      mfifo_wrlen = 5'd0;
      mfifo_last  = w0_mfifo_last;
    end
  end
  
  assign n_w0_mfifo_count = w0_mfifo_count + {10'b0,   mfifo_wrlen} - {10'b0,r1_ofifo_rqlen};
  assign n_w2_mfifo_count = w2_mfifo_count + {10'b0,w1_mfifo_wrlen} - {10'b0,   ofifo_rqlen};

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      w0_en           <= 1'b0;
      w0_addr         <= 10'b0; 
      w0_data         <= 30'b0;
      w0_ptr          <= 10'd0;
      w0_mfifo_count  <= 15'b0;
      w0_mfifo_wrlen  <= 5'd0;
      w0_mfifo_last   <= 1'b0;
      w1_mfifo_wrlen  <= 5'd0;
      w1_mfifo_last   <= 1'b0;
      w2_mfifo_count  <= 15'b0;
      w2_mfifo_last   <= 1'b0;
    end
    else if(!enable)
    begin
      w0_en           <= 1'b0;
      w0_addr         <= 10'b0; 
      w0_data         <= 30'b0;
      w0_ptr          <= 10'd0;
      w0_mfifo_count  <= 15'b0;
      w0_mfifo_wrlen  <= 5'd0;
      w0_mfifo_last   <= 1'b0;
      w1_mfifo_wrlen  <= 5'd0;
      w1_mfifo_last   <= 1'b0;
      w2_mfifo_count  <= 15'b0;
      w2_mfifo_last   <= 1'b0;
    end
    else
    begin
      if(mfifo_wrlen!=5'd0)
      begin
        w0_en   <= 1'b1;
        w0_addr <= w0_ptr; 
        w0_data <= ififo[29:0];
        w0_ptr  <= w0_ptr + 10'd1;
      end
      else
      begin
        w0_en   <= 1'b0;
      end
      w0_mfifo_count  <= n_w0_mfifo_count;
      w0_mfifo_wrlen  <= mfifo_wrlen;
      w0_mfifo_last   <= mfifo_last;
      w1_mfifo_wrlen  <= w0_mfifo_wrlen;
      w1_mfifo_last   <= w0_mfifo_last;
      w2_mfifo_count  <= n_w2_mfifo_count;
      w2_mfifo_last   <= w1_mfifo_last;
    end
  end
   
  assign wen   = w0_en;
  assign waddr = w0_addr;
  assign wdata = w0_data;
  
  /*****************************************************************************
  * MFIFO READ PIPE
  *****************************************************************************/
  always @(*)
  begin
    if(static_mem_released)
    begin
      if(r0_ofifo_count<=7'd90)
      begin
        if(w2_mfifo_last && w2_mfifo_count<=5'd30)
        begin
          ofifo_rqlen     = w2_mfifo_count[4:0];
          n_r0_ofifo_last = 1'b1;
        end
        else if(w2_mfifo_count>=5'd30)
        begin
          ofifo_rqlen     = 5'd30;
          n_r0_ofifo_last = r0_ofifo_last;
        end
        else
        begin
          ofifo_rqlen     = 5'd0;
          n_r0_ofifo_last = r0_ofifo_last;
        end
      end
      else
      begin
        ofifo_rqlen     = 5'd0;
        n_r0_ofifo_last = r0_ofifo_last;
      end
    end
    else
    begin
      ofifo_rqlen     = 5'd0;
      n_r0_ofifo_last = r0_ofifo_last;
    end
  end
 
  assign n_r0_ofifo_count = r0_ofifo_count + {2'b0,static_mem_released?ofifo_rqlen:ofifo_wrlen} - {2'b0,ofifo_rdlen};
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin  
      r0_ptr         <= 10'd0;
      r0_en          <= 1'b0;
      r0_addr        <= 10'b0;
      r0_ofifo_count <= 7'd0;
      r0_ofifo_rqlen <= 5'd0;
      r0_ofifo_last  <= 1'b0;
      r1_ofifo_rqlen <= 5'd0;
      r1_ofifo_last  <= 1'b0;
      r2_ofifo_rqlen <= 5'd0;
      r2_ofifo_last  <= 1'b0;
      r3_ofifo_rqlen <= 5'd0;
      r3_ofifo_last  <= 1'b0;
      r3_data        <= 30'd0;
    end
    else if(!enable)
    begin
      r0_ptr         <= 10'd0;
      r0_en          <= 1'b0;
      r0_addr        <= 10'b0;
      r0_ofifo_count <= 7'd0;
      r0_ofifo_rqlen <= 5'd0;
      r0_ofifo_last  <= 1'b0;
      r1_ofifo_rqlen <= 5'd0;
      r1_ofifo_last  <= 1'b0;
      r2_ofifo_rqlen <= 5'd0;
      r2_ofifo_last  <= 1'b0;
      r3_ofifo_rqlen <= 5'd0;
      r3_ofifo_last  <= 1'b0;
      /* note: no enable on r3_rdata */
    end
    else
    begin
      r0_ofifo_count <= n_r0_ofifo_count;
      r0_ofifo_last  <= n_r0_ofifo_last;
      if(ofifo_rqlen!=5'd0)
      begin
        r0_en          <= 1'b1;
        r0_addr        <= r0_ptr;
        r0_ofifo_rqlen <= ofifo_rqlen;
        r0_ptr         <= r0_ptr + 10'd1;
      end
      else
      begin
        r0_en          <= 1'b0;
        r0_ofifo_rqlen <= 5'd0;
      end
      
      r1_ofifo_rqlen <= r0_ofifo_rqlen;
      r1_ofifo_last  <= r0_ofifo_last;
      r2_ofifo_rqlen <= r1_ofifo_rqlen;
      r2_ofifo_last  <= r1_ofifo_last;
      r3_ofifo_rqlen <= r2_ofifo_rqlen;
      r3_ofifo_last  <= r2_ofifo_last;
      r3_data        <= rdata;
    end
  end

  assign ren   = r0_en;
  assign raddr = r0_addr;  

  /*****************************************************************************
  * OFIFO 120 bits / IN 0-30 bits / OUT 0-16 bits
  *****************************************************************************/
  always @(*)
  begin
    if(ofifo_count<=7'd90)
    begin
      if(!static_mem_released)
      begin
        /* IFIFO */
        if(ififo_count>6'd30)
        begin
          ofifo_wrlen  = 5'd30;  
          n_ofifo_last = ofifo_last;              
        end
        else
        begin
          ofifo_wrlen  = ififo_count[4:0];  
          n_ofifo_last = ofifo_last | ififo_last;              
        end
      end
      else
      begin
        /* MFIFO */
        if(r3_ofifo_rqlen!=5'd0)   
        begin                                  
          ofifo_wrlen  = r3_ofifo_rqlen;  
          n_ofifo_last = r3_ofifo_last;              
        end                                    
        else                                   
        begin                                  
          ofifo_wrlen  = 5'd0;  
          n_ofifo_last = ofifo_last;              
        end                                    
      end
    end
    else
    begin
      ofifo_wrlen  = 5'd0;  
      n_ofifo_last = ofifo_last;              
    end
  end
  
  assign ofifo_rdlen     = {5{           out_ready}} & out_len;
  
  assign ofifo_in_shsel  = ofifo_count - {2'b0,ofifo_rdlen};
  assign ofifo_in        = (!static_mem_released)?ififo[29:0]:{r3_data & {30{r3_ofifo_rqlen!=5'd0}}};
  assign ofifo_in_sh1    = ofifo_in_shsel[0]?{     89'b0,ofifo_in,  1'b0}:{90'b0,ofifo_in};
  assign ofifo_in_sh2    = ofifo_in_shsel[1]?{ofifo_in_sh1[117:0],  2'b0}:ofifo_in_sh1;
  assign ofifo_in_sh3    = ofifo_in_shsel[2]?{ofifo_in_sh2[115:0],  4'b0}:ofifo_in_sh2;
  assign ofifo_in_sh4    = ofifo_in_shsel[3]?{ofifo_in_sh3[111:0],  8'b0}:ofifo_in_sh3;
  assign ofifo_in_sh5    = ofifo_in_shsel[4]?{ofifo_in_sh4[103:0], 16'b0}:ofifo_in_sh4;
  assign ofifo_in_sh6    = ofifo_in_shsel[5]?{ofifo_in_sh5[ 87:0], 32'b0}:ofifo_in_sh5;
  assign ofifo_in_sh7    = ofifo_in_shsel[6]?{ofifo_in_sh6[ 55:0], 64'b0}:ofifo_in_sh6;

  assign ofifo_out_shsel = ofifo_rdlen;
  assign ofifo_out_sh1   = ofifo_out_shsel[0]?{ 1'b0,        ofifo[119: 1]}:ofifo;
  assign ofifo_out_sh2   = ofifo_out_shsel[1]?{ 2'b0,ofifo_out_sh1[119: 2]}:ofifo_out_sh1;
  assign ofifo_out_sh3   = ofifo_out_shsel[2]?{ 4'b0,ofifo_out_sh2[119: 4]}:ofifo_out_sh2;
  assign ofifo_out_sh4   = ofifo_out_shsel[3]?{ 8'b0,ofifo_out_sh3[119: 8]}:ofifo_out_sh3;
  assign ofifo_out_sh5   = ofifo_out_shsel[4]?{16'b0,ofifo_out_sh4[119:16]}:ofifo_out_sh4;

  assign n_ofifo         = ofifo_in_sh7 |  ofifo_out_sh5;
  assign n_ofifo_count   = ofifo_count    + {2'b0,ofifo_wrlen} - {2'b0,ofifo_rdlen};

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      ofifo          <= 120'b0;
      ofifo_count    <= 7'b0;
      ofifo_last     <= 1'b0;
    end
    else if(!enable)
    begin
      ofifo          <= 120'b0;
      ofifo_count    <= 7'b0;
      ofifo_last     <= 1'b0;
    end
    else
    begin
      ofifo          <= n_ofifo;      
      ofifo_count    <= n_ofifo_count;
      ofifo_last     <= n_ofifo_last;
    end
  end
 
  assign out_data  = ofifo[15:0]; 
  assign out_last  = ofifo_last & (ofifo_count<=7'd16);
  assign out_len   = (ofifo_count>7'd16)?5'd16:ofifo_count[4:0];
  assign out_valid = ofifo_count!=7'd0;
    
endmodule
`default_nettype wire
