/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     : 
* Description      : tx_fd_interleaver
* Simulation Notes : 
* Synthesis Notes  : 
* Application Note : 
* Simulator        : 
* Parameters       : 
* Terms & concepts : 
* Bugs             : 
* Open issues and future enhancements : 
* References       : 
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module tx_fast_interleaver
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire          clk,
  input wire          rst_n,
  
  /*****************************************************************************
  * 
  *****************************************************************************/
  input  wire         enable, 
  input  wire         start,
  output reg          busy,
  
  /*****************************************************************************
  * parameters/control
  *****************************************************************************/
  /* field */
  input  wire         fec,
  input  wire [ 1:0]  nsd,
  input  wire         ruen,
  input  wire [ 2:0]  rulen,
  input  wire [ 2:0]  nbpsc,
  input  wire         dcm,
  
  /*****************************************************************************
  * BD interface
  *****************************************************************************/
  output wire         bd_ready,
  input  wire [15:0]  bd_data,
  input  wire [ 4:0]  bd_len,
  input  wire         bd_valid,
 
  /*****************************************************************************
  * FD interface
  *****************************************************************************/
  input  wire         fd_ready,
  output reg  [9:0]   fd_data,
  output reg          fd_last,
  output reg          fd_valid
);
  /*****************************************************************************
  * declarations
  *****************************************************************************/
  /* constants */
  localparam  NSD_48=2'd0, 
              NSD_52=2'd1, 
              NSD_108=2'd2, 
              NSD_234=2'd3;
   
  localparam  RU_26=3'd0,
              RU_52=3'd1,
              RU_106=3'd2,
              RU_242=3'd3,
              RU_484=3'd4,
              RU_996=3'd5;
              
  localparam  NBPSC_1=3'd0,
              NBPSC_2=3'd1,
              NBPSC_4=3'd2,
              NBPSC_6=3'd3,
              NBPSC_8=3'd4,
              NBPSC_10=3'd5;
  
  /* parameters re-registering */
  reg  [ 3:0]  dec_nbpsc;
  reg  [ 8:0]  dec_nsd;
  reg  [ 5:0]  ncol;
  reg  [ 3:0]  nrow;
  
  /* fifo row_max * nbpsc_max +16  = 26*8+16 = 224 bits */
  wire   [  4:0] fifo_wrlen;
  wire   [  4:0] fifo_rdlen;
  
  wire   [ 15:0] fifo_in;
  wire   [223:0] fifo_in_sh1,fifo_in_sh2,fifo_in_sh3,fifo_in_sh4;
  wire   [223:0] fifo_in_sh5,fifo_in_sh6,fifo_in_sh7,fifo_in_sh8;
  wire   [223:0] fifo_out_sh1,fifo_out_sh2,fifo_out_sh3,fifo_out_sh4,fifo_out_sh5;  
  
  wire   [  7:0] fifo_in_shsel;
  wire   [  4:0] fifo_out_shsel;
  
  wire   [223:0] n_fifo;
  reg    [223:0] fifo;
  wire   [  7:0] n_fifo_count;
  reg    [  7:0] fifo_count;
  
  /* ncol*nbpsc transpose */
  wire [  7:0]   ncol_x_nbpsc;
  /* row transpose */
  wire           c4_n2, c4_n4, c4_n6, c4_n8;
  wire           c8_n2, c8_n4, c8_n6, c8_n8;
  wire           c13_n2,c13_n4,c13_n6,c13_n8;
  wire           c16_n2,c16_n4,c16_n6,c16_n8;
  wire           c17_n2,c17_n4,c17_n6,c17_n8;
  wire           c18_n2,c18_n4,c18_n6,c18_n8;
  wire           c26_n2,c26_n4,c26_n6,c26_n8;

  wire [207:0]   t_c4_n2, t_c4_n4, t_c4_n6, t_c4_n8;
  wire [207:0]   t_c8_n2, t_c8_n4, t_c8_n6, t_c8_n8;
  wire [207:0]   t_c13_n2,t_c13_n4,t_c13_n6,t_c13_n8;
  wire [207:0]   t_c16_n2,t_c16_n4,t_c16_n6,t_c16_n8;
  wire [207:0]   t_c17_n2,t_c17_n4,t_c17_n6,t_c17_n8;
  wire [207:0]   t_c18_n2,t_c18_n4,t_c18_n6,t_c18_n8;
  wire [207:0]   t_c26_n2,t_c26_n4,t_c26_n6,t_c26_n8;
  wire [207:0]   transpose_out;
  wire           en_cx_nx;
  wire           n_transposed;
  reg            transposed;
  wire [ 7:0]    n_transposed_count;
  reg  [ 7:0]    transposed_count;
 
  /* fd slot */  
  reg            done;
  reg  [ 5:0]    fd_col;
  reg  [ 3:0]    fd_row;
  wire [ 5:0]    n_fd_col;
  wire [ 3:0]    n_fd_row;
  wire           fd_col_last;
  wire           fd_row_last;
  wire           padding;
  
  /*****************************************************************************
  * parameters decoding
  *****************************************************************************/
  always @(*)
  begin
    /* nsd */
    if(!ruen)
    begin
      if(!dcm)
      begin
        case(nsd)
          NSD_48:  {dec_nsd,ncol,nrow} = { 9'd48, 6'd16, 4'd3};
          NSD_52:  {dec_nsd,ncol,nrow} = { 9'd52, 6'd13, 4'd4};
          NSD_108: {dec_nsd,ncol,nrow} = {9'd108, 6'd18, 4'd6};
          default: {dec_nsd,ncol,nrow} = {9'd234, 6'd26, 4'd9};
        endcase
      end
      else
      begin
        {dec_nsd,ncol,nrow} = { 9'd26, 6'd13, 4'd2};
      end
    end
    else
    begin
      if(!dcm)
      begin
        case(rulen) 
          RU_26:  {dec_nsd,ncol,nrow} = { 9'd24,  6'd8,  4'd3};
          RU_52:  {dec_nsd,ncol,nrow} = { 9'd48, 6'd16,  4'd3};
          RU_106: {dec_nsd,ncol,nrow} = {9'd102, 6'd17,  4'd6};
          RU_242: {dec_nsd,ncol,nrow} = {9'd234, 6'd26,  4'd9};
          default:{dec_nsd,ncol,nrow} = {9'd468, 6'd39, 4'd12};
        endcase
      end
      else
      begin
        case(rulen) 
          RU_26:  {dec_nsd,ncol,nrow} = { 9'd12,  6'd4,  4'd3};
          RU_52:  {dec_nsd,ncol,nrow} = { 9'd24,  6'd8,  4'd3};
          RU_106: {dec_nsd,ncol,nrow} = { 9'd51, 6'd17,  4'd3};
          RU_242: {dec_nsd,ncol,nrow} = {9'd117, 6'd13,  4'd9};
          default:{dec_nsd,ncol,nrow} = {9'd234, 6'd26,  4'd9};
        endcase
      end
    end
    
    /* nbpsc */
    case(nbpsc)
      NBPSC_1: dec_nbpsc = 4'd1;
      NBPSC_2: dec_nbpsc = 4'd2;
      NBPSC_4: dec_nbpsc = 4'd4;
      NBPSC_6: dec_nbpsc = 4'd6;
      NBPSC_8: dec_nbpsc = 4'd8;
      default: dec_nbpsc = 4'd10;
    endcase
  end

  /* transpose row length (bcc only,  max 26*8= 208 8 bits) */
  assign ncol_x_nbpsc =
    ({2'b0,ncol     } & {8{dec_nbpsc[0]}}) +
    ({1'b0,ncol,1'b0} & {8{dec_nbpsc[1]}}) +
    ({     ncol,2'b0} & {8{dec_nbpsc[2]}}) +
    ({ncol[4:0],3'b0} & {8{dec_nbpsc[3]}});
 
  /* transpose pattern sel */
  assign c4_n2  = ncol==6'd4  && nbpsc==NBPSC_2;
  assign c4_n4  = ncol==6'd4  && nbpsc==NBPSC_4;
  assign c4_n6  = ncol==6'd4  && nbpsc==NBPSC_6;
  assign c4_n8  = ncol==6'd4  && nbpsc==NBPSC_8;
  assign c8_n2  = ncol==6'd8  && nbpsc==NBPSC_2;
  assign c8_n4  = ncol==6'd8  && nbpsc==NBPSC_4;
  assign c8_n6  = ncol==6'd8  && nbpsc==NBPSC_6;
  assign c8_n8  = ncol==6'd8  && nbpsc==NBPSC_8;
  assign c13_n2 = ncol==6'd13 && nbpsc==NBPSC_2;
  assign c13_n4 = ncol==6'd13 && nbpsc==NBPSC_4;
  assign c13_n6 = ncol==6'd13 && nbpsc==NBPSC_6;
  assign c13_n8 = ncol==6'd13 && nbpsc==NBPSC_8;
  assign c16_n2 = ncol==6'd16 && nbpsc==NBPSC_2;
  assign c16_n4 = ncol==6'd16 && nbpsc==NBPSC_4;
  assign c16_n6 = ncol==6'd16 && nbpsc==NBPSC_6;
  assign c16_n8 = ncol==6'd16 && nbpsc==NBPSC_8;
  assign c17_n2 = ncol==6'd17 && nbpsc==NBPSC_2;
  assign c17_n4 = ncol==6'd17 && nbpsc==NBPSC_4;
  assign c17_n6 = ncol==6'd17 && nbpsc==NBPSC_6;
  assign c17_n8 = ncol==6'd17 && nbpsc==NBPSC_8;
  assign c18_n2 = ncol==6'd18 && nbpsc==NBPSC_2;
  assign c18_n4 = ncol==6'd18 && nbpsc==NBPSC_4;
  assign c18_n6 = ncol==6'd18 && nbpsc==NBPSC_6;
  assign c18_n8 = ncol==6'd18 && nbpsc==NBPSC_8;
  assign c26_n2 = ncol==6'd26 && nbpsc==NBPSC_2;
  assign c26_n4 = ncol==6'd26 && nbpsc==NBPSC_4;
  assign c26_n6 = ncol==6'd26 && nbpsc==NBPSC_6;
  assign c26_n8 = ncol==6'd26 && nbpsc==NBPSC_8;

  /*****************************************************************************
  * fifo 228 bits
  *****************************************************************************/
  wire  [15:0] data_mask_sh1,data_mask_sh2,data_mask_sh3,data_mask_sh4,data_mask_sh5;
  wire  [ 4:0] data_mask_shsel;

  assign bd_ready        = (fifo_count <= 8'd208);
  
  assign data_mask_shsel = 5'd16 - bd_len;
  assign data_mask_sh1   = data_mask_shsel[0]?16'h7fff:16'hffff;
  assign data_mask_sh2   = data_mask_shsel[1]?{2'b0,data_mask_sh1[15:2]}:data_mask_sh1;
  assign data_mask_sh3   = data_mask_shsel[2]?{4'b0,data_mask_sh2[15:4]}:data_mask_sh2;
  assign data_mask_sh4   = data_mask_shsel[3]?{8'b0,data_mask_sh3[15:8]}:data_mask_sh3;
  assign data_mask_sh5   = data_mask_shsel[4]?16'h0000:data_mask_sh4;
  
  assign fifo_wrlen      = bd_len&{5{bd_valid & bd_ready}};
  assign fifo_rdlen      = {1'b0,dec_nbpsc} & {5{(transposed_count>={4'b0,dec_nbpsc}) & (~fd_valid | fd_ready) & ~done & ~padding}};
  assign fifo_in_shsel   = fifo_count - {3'b0,fifo_rdlen};
  assign fifo_out_shsel  = fifo_rdlen;
  
  assign fifo_in        = bd_data & data_mask_sh5;
  assign fifo_in_sh1    = fifo_in_shsel[0]?{    207'b0,fifo_in,  1'b0}:{208'b0,fifo_in};
  assign fifo_in_sh2    = fifo_in_shsel[1]?{fifo_in_sh1[221:0],  2'b0}:fifo_in_sh1;
  assign fifo_in_sh3    = fifo_in_shsel[2]?{fifo_in_sh2[219:0],  4'b0}:fifo_in_sh2;
  assign fifo_in_sh4    = fifo_in_shsel[3]?{fifo_in_sh3[215:0],  8'b0}:fifo_in_sh3;
  assign fifo_in_sh5    = fifo_in_shsel[4]?{fifo_in_sh4[207:0], 16'b0}:fifo_in_sh4;
  assign fifo_in_sh6    = fifo_in_shsel[5]?{fifo_in_sh5[191:0], 32'b0}:fifo_in_sh5;
  assign fifo_in_sh7    = fifo_in_shsel[6]?{fifo_in_sh6[159:0], 64'b0}:fifo_in_sh6;
  assign fifo_in_sh8    = fifo_in_shsel[7]?{fifo_in_sh7[ 95:0],128'b0}:fifo_in_sh7;

  assign fifo_out_sh1   = fifo_out_shsel[0]?{ 1'b0,        fifo[223: 1]}:fifo;
  assign fifo_out_sh2   = fifo_out_shsel[1]?{ 2'b0,fifo_out_sh1[223: 2]}:fifo_out_sh1;
  assign fifo_out_sh3   = fifo_out_shsel[2]?{ 4'b0,fifo_out_sh2[223: 4]}:fifo_out_sh2;
  assign fifo_out_sh4   = fifo_out_shsel[3]?{ 8'b0,fifo_out_sh3[223: 8]}:fifo_out_sh3;
  assign fifo_out_sh5   = fifo_out_shsel[4]?{16'b0,fifo_out_sh4[223:16]}:fifo_out_sh4;

  assign n_fifo         = fifo_in_sh8                     | 
                          fifo_out_sh5 & {224{!en_cx_nx}} | 
                          {fifo[223:208],transpose_out} & {224{en_cx_nx}};

  assign n_fifo_count   = fifo_count + {3'b0,fifo_wrlen} - {3'b0,fifo_rdlen};
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      fifo       <= 224'd0;
      fifo_count <= 8'd0;
    end
    else if(!enable)
    begin
      fifo       <= 224'd0;
      fifo_count <= 8'd0;
    end
    else
    begin
      fifo       <= n_fifo;
      fifo_count <= n_fifo_count;
    end
  end

  /* mask fifo output */
  reg  [ 9:0] fifo_data;
  always @(*)
  begin
    case(nbpsc)                                 
      NBPSC_1: fifo_data = { 9'b0, fifo[  0]};  
      NBPSC_2: fifo_data = { 8'b0, fifo[1:0]};  
      NBPSC_4: fifo_data = { 6'b0, fifo[3:0]};  
      NBPSC_6: fifo_data = { 4'b0, fifo[5:0]};  
      NBPSC_8: fifo_data = { 2'b0, fifo[7:0]};  
      default: fifo_data = fifo[9:0];           
    endcase                                     
  end
 
  /*****************************************************************************
  * transpose control
  *****************************************************************************/
  wire  [ 7:0]  transposed_count_decr;
  
  assign en_cx_nx  = ~transposed & (fifo_count>=ncol_x_nbpsc);
  
  assign transposed_count_decr = transposed_count - {3'b0,fifo_rdlen};
  assign n_transposed          = en_cx_nx | transposed & (n_transposed_count!=8'd0);
  assign n_transposed_count    = en_cx_nx?ncol_x_nbpsc:transposed_count_decr;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      transposed       <= 1'd0;
      transposed_count <= 8'd0;
    end
    else if(!enable)
    begin
      transposed       <= 1'd0;
      transposed_count <= 8'd0;
    end
    else
    begin
      if(nbpsc==NBPSC_1 || fec)
      begin
        transposed       <= 1'b1;
        transposed_count <= n_fifo_count;
      end
      else
      begin
        transposed       <= n_transposed;
        transposed_count <= n_transposed_count;
      end
    end
  end
 
  /*****************************************************************************
  * output slot
  *****************************************************************************/
  /* sub-carrier position */
  assign   n_fd_col    = fd_col + 6'd1;
  assign   n_fd_row    = fd_row + 4'd1;
  assign   fd_col_last = n_fd_col==ncol;
  assign   fd_row_last = n_fd_row==nrow;
  
  /* cases of odd ncbps */
  assign   padding     = fd_col_last && fd_row_last && 
                         dcm && nbpsc==NBPSC_1 && ~fec && ruen && (rulen==RU_106 || rulen==RU_242); 
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      busy     <= 1'b0;
      done     <= 1'b0;
      fd_col   <= 6'b0;
      fd_row   <= 4'b0;
      fd_valid <= 1'b0;
      fd_data  <= 10'd0;
      fd_last  <= 1'b0;
    end
    else if(!enable)
    begin
      busy     <= 1'b0;
      done     <= 1'b0;
      fd_col   <= 6'b0;
      fd_row   <= 4'b0;
      fd_valid <= 1'b0;
      fd_data  <= 10'd0;
      fd_last  <= 1'b0;
    end
    else
    begin
      if(!busy)
      begin
        if(start)
        begin
          busy <= 1'b1;
          done <= 1'b0;
        end
      end
      else
      begin
        if(fd_ready && fd_valid && fd_last)
        begin
          fd_valid  <= 1'b0;
          fd_data   <= 10'd0;
          fd_last   <= 1'b0;
          busy      <= 1'b0;
        end
        else
        begin
          if(fd_ready)
          begin
            fd_valid  <= 1'b0;
            fd_last   <= 1'b0;
          end
          
          if(fifo_rdlen!=0 || padding & (~fd_valid | fd_ready))
          begin
            fd_valid <= 1'b1;
            if(!padding)
              fd_data <= fifo_data;
            else
              fd_data <= 10'b0; /* padding odd ncbps */
            
            if(fd_col_last)
            begin
              fd_col <= 6'd0;
              if(fd_row_last)
              begin
                fd_row  <= 4'd0;
                fd_last <= 1'b1;
                done    <= 1'b1;
              end
              else
              begin
                fd_row  <= n_fd_row;
              end
            end
            else
            begin
              fd_col <= n_fd_col;
            end
          end
        end
      end
    end
  end
  
  /*****************************************************************************
  * transpose mux
  *****************************************************************************/
  assign transpose_out  =  
    t_c4_n2  | t_c4_n4  | t_c4_n6  | t_c4_n8  |
    t_c8_n2  | t_c8_n4  | t_c8_n6  | t_c8_n8  |
    t_c13_n2 | t_c13_n4 | t_c13_n6 | t_c13_n8 | 
    t_c16_n2 | t_c16_n4 | t_c16_n6 | t_c16_n8 | 
    t_c17_n2 | t_c17_n4 | t_c17_n6 | t_c17_n8 | 
    t_c18_n2 | t_c18_n4 | t_c18_n6 | t_c18_n8 | 
    t_c26_n2 | t_c26_n4 | t_c26_n6 | t_c26_n8;

  /*****************************************************************************
  * row transpose and sub-carrier component rotation
  *****************************************************************************/
  assign t_c4_n2 = {208{c4_n2}} & { fifo[207:  8], 
    fifo[  7],fifo[  3],
    fifo[  6],fifo[  2],
    fifo[  5],fifo[  1],
    fifo[  4],fifo[  0]
    };
  assign t_c4_n4 = {208{c4_n4}} & { fifo[207: 16], 
    fifo[ 11],fifo[ 15],fifo[  3],fifo[  7],
    fifo[ 14],fifo[ 10],fifo[  6],fifo[  2],
    fifo[  9],fifo[ 13],fifo[  1],fifo[  5],
    fifo[ 12],fifo[  8],fifo[  4],fifo[  0]
    };
  assign t_c4_n6 = {208{c4_n6}} & { fifo[207: 24], 
    fifo[ 23],fifo[ 19],fifo[ 15],fifo[ 11],fifo[  7],fifo[  3],
    fifo[ 18],fifo[ 14],fifo[ 22],fifo[  6],fifo[  2],fifo[ 10],
    fifo[ 13],fifo[ 21],fifo[ 17],fifo[  1],fifo[  9],fifo[  5],
    fifo[ 20],fifo[ 16],fifo[ 12],fifo[  8],fifo[  4],fifo[  0]
    };
  assign t_c4_n8 = {208{c4_n8}} & { fifo[207: 32], 
    fifo[ 27],fifo[ 23],fifo[ 19],fifo[ 31],fifo[ 11],fifo[  7],fifo[  3],fifo[ 15],
    fifo[ 22],fifo[ 18],fifo[ 30],fifo[ 26],fifo[  6],fifo[  2],fifo[ 14],fifo[ 10],
    fifo[ 17],fifo[ 29],fifo[ 25],fifo[ 21],fifo[  1],fifo[ 13],fifo[  9],fifo[  5],
    fifo[ 28],fifo[ 24],fifo[ 20],fifo[ 16],fifo[ 12],fifo[  8],fifo[  4],fifo[  0]
    };
  assign t_c8_n2 = {208{c8_n2}} & { fifo[207: 16], 
    fifo[ 15],fifo[  7],
    fifo[ 14],fifo[  6],
    fifo[ 13],fifo[  5],
    fifo[ 12],fifo[  4],
    fifo[ 11],fifo[  3],
    fifo[ 10],fifo[  2],
    fifo[  9],fifo[  1],
    fifo[  8],fifo[  0]
    };
  assign t_c8_n4 = {208{c8_n4}} & { fifo[207: 32], 
    fifo[ 23],fifo[ 31],fifo[  7],fifo[ 15],
    fifo[ 30],fifo[ 22],fifo[ 14],fifo[  6],
    fifo[ 21],fifo[ 29],fifo[  5],fifo[ 13],
    fifo[ 28],fifo[ 20],fifo[ 12],fifo[  4],
    fifo[ 19],fifo[ 27],fifo[  3],fifo[ 11],
    fifo[ 26],fifo[ 18],fifo[ 10],fifo[  2],
    fifo[ 17],fifo[ 25],fifo[  1],fifo[  9],
    fifo[ 24],fifo[ 16],fifo[  8],fifo[  0]
    };
  assign t_c8_n6 = {208{c8_n6}} & { fifo[207: 48], 
    fifo[ 31],fifo[ 47],fifo[ 39],fifo[  7],fifo[ 23],fifo[ 15],
    fifo[ 46],fifo[ 38],fifo[ 30],fifo[ 22],fifo[ 14],fifo[  6],
    fifo[ 37],fifo[ 29],fifo[ 45],fifo[ 13],fifo[  5],fifo[ 21],
    fifo[ 28],fifo[ 44],fifo[ 36],fifo[  4],fifo[ 20],fifo[ 12],
    fifo[ 43],fifo[ 35],fifo[ 27],fifo[ 19],fifo[ 11],fifo[  3],
    fifo[ 34],fifo[ 26],fifo[ 42],fifo[ 10],fifo[  2],fifo[ 18],
    fifo[ 25],fifo[ 41],fifo[ 33],fifo[  1],fifo[ 17],fifo[  9],
    fifo[ 40],fifo[ 32],fifo[ 24],fifo[ 16],fifo[  8],fifo[  0]
    };
  assign t_c8_n8 = {208{c8_n8}} & { fifo[207: 64], 
    fifo[ 55],fifo[ 47],fifo[ 39],fifo[ 63],fifo[ 23],fifo[ 15],fifo[  7],fifo[ 31],
    fifo[ 46],fifo[ 38],fifo[ 62],fifo[ 54],fifo[ 14],fifo[  6],fifo[ 30],fifo[ 22],
    fifo[ 37],fifo[ 61],fifo[ 53],fifo[ 45],fifo[  5],fifo[ 29],fifo[ 21],fifo[ 13],
    fifo[ 60],fifo[ 52],fifo[ 44],fifo[ 36],fifo[ 28],fifo[ 20],fifo[ 12],fifo[  4],
    fifo[ 51],fifo[ 43],fifo[ 35],fifo[ 59],fifo[ 19],fifo[ 11],fifo[  3],fifo[ 27],
    fifo[ 42],fifo[ 34],fifo[ 58],fifo[ 50],fifo[ 10],fifo[  2],fifo[ 26],fifo[ 18],
    fifo[ 33],fifo[ 57],fifo[ 49],fifo[ 41],fifo[  1],fifo[ 25],fifo[ 17],fifo[  9],
    fifo[ 56],fifo[ 48],fifo[ 40],fifo[ 32],fifo[ 24],fifo[ 16],fifo[  8],fifo[  0]
    };
  assign t_c13_n2 = {208{c13_n2}} & { fifo[207: 26], 
    fifo[ 25],fifo[ 12],
    fifo[ 24],fifo[ 11],
    fifo[ 23],fifo[ 10],
    fifo[ 22],fifo[  9],
    fifo[ 21],fifo[  8],
    fifo[ 20],fifo[  7],
    fifo[ 19],fifo[  6],
    fifo[ 18],fifo[  5],
    fifo[ 17],fifo[  4],
    fifo[ 16],fifo[  3],
    fifo[ 15],fifo[  2],
    fifo[ 14],fifo[  1],
    fifo[ 13],fifo[  0]
    };
  assign t_c13_n4 = {208{c13_n4}} & { fifo[207: 52], 
    fifo[ 51],fifo[ 38],fifo[ 25],fifo[ 12],
    fifo[ 37],fifo[ 50],fifo[ 11],fifo[ 24],
    fifo[ 49],fifo[ 36],fifo[ 23],fifo[ 10],
    fifo[ 35],fifo[ 48],fifo[  9],fifo[ 22],
    fifo[ 47],fifo[ 34],fifo[ 21],fifo[  8],
    fifo[ 33],fifo[ 46],fifo[  7],fifo[ 20],
    fifo[ 45],fifo[ 32],fifo[ 19],fifo[  6],
    fifo[ 31],fifo[ 44],fifo[  5],fifo[ 18],
    fifo[ 43],fifo[ 30],fifo[ 17],fifo[  4],
    fifo[ 29],fifo[ 42],fifo[  3],fifo[ 16],
    fifo[ 41],fifo[ 28],fifo[ 15],fifo[  2],
    fifo[ 27],fifo[ 40],fifo[  1],fifo[ 14],
    fifo[ 39],fifo[ 26],fifo[ 13],fifo[  0]
    };
  assign t_c13_n6 = {208{c13_n6}} & { fifo[207: 78], 
    fifo[ 77],fifo[ 64],fifo[ 51],fifo[ 38],fifo[ 25],fifo[ 12],
    fifo[ 63],fifo[ 50],fifo[ 76],fifo[ 24],fifo[ 11],fifo[ 37],
    fifo[ 49],fifo[ 75],fifo[ 62],fifo[ 10],fifo[ 36],fifo[ 23],
    fifo[ 74],fifo[ 61],fifo[ 48],fifo[ 35],fifo[ 22],fifo[  9],
    fifo[ 60],fifo[ 47],fifo[ 73],fifo[ 21],fifo[  8],fifo[ 34],
    fifo[ 46],fifo[ 72],fifo[ 59],fifo[  7],fifo[ 33],fifo[ 20],
    fifo[ 71],fifo[ 58],fifo[ 45],fifo[ 32],fifo[ 19],fifo[  6],
    fifo[ 57],fifo[ 44],fifo[ 70],fifo[ 18],fifo[  5],fifo[ 31],
    fifo[ 43],fifo[ 69],fifo[ 56],fifo[  4],fifo[ 30],fifo[ 17],
    fifo[ 68],fifo[ 55],fifo[ 42],fifo[ 29],fifo[ 16],fifo[  3],
    fifo[ 54],fifo[ 41],fifo[ 67],fifo[ 15],fifo[  2],fifo[ 28],
    fifo[ 40],fifo[ 66],fifo[ 53],fifo[  1],fifo[ 27],fifo[ 14],
    fifo[ 65],fifo[ 52],fifo[ 39],fifo[ 26],fifo[ 13],fifo[  0]
    };
  assign t_c13_n8 = {208{c13_n8}} & { fifo[207:104], 
    fifo[103],fifo[ 90],fifo[ 77],fifo[ 64],fifo[ 51],fifo[ 38],fifo[ 25],fifo[ 12],
    fifo[ 89],fifo[ 76],fifo[ 63],fifo[102],fifo[ 37],fifo[ 24],fifo[ 11],fifo[ 50],
    fifo[ 75],fifo[ 62],fifo[101],fifo[ 88],fifo[ 23],fifo[ 10],fifo[ 49],fifo[ 36],
    fifo[ 61],fifo[100],fifo[ 87],fifo[ 74],fifo[  9],fifo[ 48],fifo[ 35],fifo[ 22],
    fifo[ 99],fifo[ 86],fifo[ 73],fifo[ 60],fifo[ 47],fifo[ 34],fifo[ 21],fifo[  8],
    fifo[ 85],fifo[ 72],fifo[ 59],fifo[ 98],fifo[ 33],fifo[ 20],fifo[  7],fifo[ 46],
    fifo[ 71],fifo[ 58],fifo[ 97],fifo[ 84],fifo[ 19],fifo[  6],fifo[ 45],fifo[ 32],
    fifo[ 57],fifo[ 96],fifo[ 83],fifo[ 70],fifo[  5],fifo[ 44],fifo[ 31],fifo[ 18],
    fifo[ 95],fifo[ 82],fifo[ 69],fifo[ 56],fifo[ 43],fifo[ 30],fifo[ 17],fifo[  4],
    fifo[ 81],fifo[ 68],fifo[ 55],fifo[ 94],fifo[ 29],fifo[ 16],fifo[  3],fifo[ 42],
    fifo[ 67],fifo[ 54],fifo[ 93],fifo[ 80],fifo[ 15],fifo[  2],fifo[ 41],fifo[ 28],
    fifo[ 53],fifo[ 92],fifo[ 79],fifo[ 66],fifo[  1],fifo[ 40],fifo[ 27],fifo[ 14],
    fifo[ 91],fifo[ 78],fifo[ 65],fifo[ 52],fifo[ 39],fifo[ 26],fifo[ 13],fifo[  0]
    };
  assign t_c16_n2 = {208{c16_n2}} & { fifo[207: 32], 
    fifo[ 31],fifo[ 15],
    fifo[ 30],fifo[ 14],
    fifo[ 29],fifo[ 13],
    fifo[ 28],fifo[ 12],
    fifo[ 27],fifo[ 11],
    fifo[ 26],fifo[ 10],
    fifo[ 25],fifo[  9],
    fifo[ 24],fifo[  8],
    fifo[ 23],fifo[  7],
    fifo[ 22],fifo[  6],
    fifo[ 21],fifo[  5],
    fifo[ 20],fifo[  4],
    fifo[ 19],fifo[  3],
    fifo[ 18],fifo[  2],
    fifo[ 17],fifo[  1],
    fifo[ 16],fifo[  0]
    };
  assign t_c16_n4 = {208{c16_n4}} & { fifo[207: 64], 
    fifo[ 47],fifo[ 63],fifo[ 15],fifo[ 31],
    fifo[ 62],fifo[ 46],fifo[ 30],fifo[ 14],
    fifo[ 45],fifo[ 61],fifo[ 13],fifo[ 29],
    fifo[ 60],fifo[ 44],fifo[ 28],fifo[ 12],
    fifo[ 43],fifo[ 59],fifo[ 11],fifo[ 27],
    fifo[ 58],fifo[ 42],fifo[ 26],fifo[ 10],
    fifo[ 41],fifo[ 57],fifo[  9],fifo[ 25],
    fifo[ 56],fifo[ 40],fifo[ 24],fifo[  8],
    fifo[ 39],fifo[ 55],fifo[  7],fifo[ 23],
    fifo[ 54],fifo[ 38],fifo[ 22],fifo[  6],
    fifo[ 37],fifo[ 53],fifo[  5],fifo[ 21],
    fifo[ 52],fifo[ 36],fifo[ 20],fifo[  4],
    fifo[ 35],fifo[ 51],fifo[  3],fifo[ 19],
    fifo[ 50],fifo[ 34],fifo[ 18],fifo[  2],
    fifo[ 33],fifo[ 49],fifo[  1],fifo[ 17],
    fifo[ 48],fifo[ 32],fifo[ 16],fifo[  0]
    };
  assign t_c16_n6 = {208{c16_n6}} & { fifo[207: 96], 
    fifo[ 95],fifo[ 79],fifo[ 63],fifo[ 47],fifo[ 31],fifo[ 15],
    fifo[ 78],fifo[ 62],fifo[ 94],fifo[ 30],fifo[ 14],fifo[ 46],
    fifo[ 61],fifo[ 93],fifo[ 77],fifo[ 13],fifo[ 45],fifo[ 29],
    fifo[ 92],fifo[ 76],fifo[ 60],fifo[ 44],fifo[ 28],fifo[ 12],
    fifo[ 75],fifo[ 59],fifo[ 91],fifo[ 27],fifo[ 11],fifo[ 43],
    fifo[ 58],fifo[ 90],fifo[ 74],fifo[ 10],fifo[ 42],fifo[ 26],
    fifo[ 89],fifo[ 73],fifo[ 57],fifo[ 41],fifo[ 25],fifo[  9],
    fifo[ 72],fifo[ 56],fifo[ 88],fifo[ 24],fifo[  8],fifo[ 40],
    fifo[ 55],fifo[ 87],fifo[ 71],fifo[  7],fifo[ 39],fifo[ 23],
    fifo[ 86],fifo[ 70],fifo[ 54],fifo[ 38],fifo[ 22],fifo[  6],
    fifo[ 69],fifo[ 53],fifo[ 85],fifo[ 21],fifo[  5],fifo[ 37],
    fifo[ 52],fifo[ 84],fifo[ 68],fifo[  4],fifo[ 36],fifo[ 20],
    fifo[ 83],fifo[ 67],fifo[ 51],fifo[ 35],fifo[ 19],fifo[  3],
    fifo[ 66],fifo[ 50],fifo[ 82],fifo[ 18],fifo[  2],fifo[ 34],
    fifo[ 49],fifo[ 81],fifo[ 65],fifo[  1],fifo[ 33],fifo[ 17],
    fifo[ 80],fifo[ 64],fifo[ 48],fifo[ 32],fifo[ 16],fifo[  0]
    };
  assign t_c16_n8 = {208{c16_n8}} & { fifo[207:128], 
    fifo[111],fifo[ 95],fifo[ 79],fifo[127],fifo[ 47],fifo[ 31],fifo[ 15],fifo[ 63],
    fifo[ 94],fifo[ 78],fifo[126],fifo[110],fifo[ 30],fifo[ 14],fifo[ 62],fifo[ 46],
    fifo[ 77],fifo[125],fifo[109],fifo[ 93],fifo[ 13],fifo[ 61],fifo[ 45],fifo[ 29],
    fifo[124],fifo[108],fifo[ 92],fifo[ 76],fifo[ 60],fifo[ 44],fifo[ 28],fifo[ 12],
    fifo[107],fifo[ 91],fifo[ 75],fifo[123],fifo[ 43],fifo[ 27],fifo[ 11],fifo[ 59],
    fifo[ 90],fifo[ 74],fifo[122],fifo[106],fifo[ 26],fifo[ 10],fifo[ 58],fifo[ 42],
    fifo[ 73],fifo[121],fifo[105],fifo[ 89],fifo[  9],fifo[ 57],fifo[ 41],fifo[ 25],
    fifo[120],fifo[104],fifo[ 88],fifo[ 72],fifo[ 56],fifo[ 40],fifo[ 24],fifo[  8],
    fifo[103],fifo[ 87],fifo[ 71],fifo[119],fifo[ 39],fifo[ 23],fifo[  7],fifo[ 55],
    fifo[ 86],fifo[ 70],fifo[118],fifo[102],fifo[ 22],fifo[  6],fifo[ 54],fifo[ 38],
    fifo[ 69],fifo[117],fifo[101],fifo[ 85],fifo[  5],fifo[ 53],fifo[ 37],fifo[ 21],
    fifo[116],fifo[100],fifo[ 84],fifo[ 68],fifo[ 52],fifo[ 36],fifo[ 20],fifo[  4],
    fifo[ 99],fifo[ 83],fifo[ 67],fifo[115],fifo[ 35],fifo[ 19],fifo[  3],fifo[ 51],
    fifo[ 82],fifo[ 66],fifo[114],fifo[ 98],fifo[ 18],fifo[  2],fifo[ 50],fifo[ 34],
    fifo[ 65],fifo[113],fifo[ 97],fifo[ 81],fifo[  1],fifo[ 49],fifo[ 33],fifo[ 17],
    fifo[112],fifo[ 96],fifo[ 80],fifo[ 64],fifo[ 48],fifo[ 32],fifo[ 16],fifo[  0]
    };
  assign t_c17_n2 = {208{c17_n2}} & { fifo[207: 34], 
    fifo[ 33],fifo[ 16],
    fifo[ 32],fifo[ 15],
    fifo[ 31],fifo[ 14],
    fifo[ 30],fifo[ 13],
    fifo[ 29],fifo[ 12],
    fifo[ 28],fifo[ 11],
    fifo[ 27],fifo[ 10],
    fifo[ 26],fifo[  9],
    fifo[ 25],fifo[  8],
    fifo[ 24],fifo[  7],
    fifo[ 23],fifo[  6],
    fifo[ 22],fifo[  5],
    fifo[ 21],fifo[  4],
    fifo[ 20],fifo[  3],
    fifo[ 19],fifo[  2],
    fifo[ 18],fifo[  1],
    fifo[ 17],fifo[  0]
    };
  assign t_c17_n4 = {208{c17_n4}} & { fifo[207: 68], 
    fifo[ 67],fifo[ 50],fifo[ 33],fifo[ 16],
    fifo[ 49],fifo[ 66],fifo[ 15],fifo[ 32],
    fifo[ 65],fifo[ 48],fifo[ 31],fifo[ 14],
    fifo[ 47],fifo[ 64],fifo[ 13],fifo[ 30],
    fifo[ 63],fifo[ 46],fifo[ 29],fifo[ 12],
    fifo[ 45],fifo[ 62],fifo[ 11],fifo[ 28],
    fifo[ 61],fifo[ 44],fifo[ 27],fifo[ 10],
    fifo[ 43],fifo[ 60],fifo[  9],fifo[ 26],
    fifo[ 59],fifo[ 42],fifo[ 25],fifo[  8],
    fifo[ 41],fifo[ 58],fifo[  7],fifo[ 24],
    fifo[ 57],fifo[ 40],fifo[ 23],fifo[  6],
    fifo[ 39],fifo[ 56],fifo[  5],fifo[ 22],
    fifo[ 55],fifo[ 38],fifo[ 21],fifo[  4],
    fifo[ 37],fifo[ 54],fifo[  3],fifo[ 20],
    fifo[ 53],fifo[ 36],fifo[ 19],fifo[  2],
    fifo[ 35],fifo[ 52],fifo[  1],fifo[ 18],
    fifo[ 51],fifo[ 34],fifo[ 17],fifo[  0]
    };
  assign t_c17_n6 = {208{c17_n6}} & { fifo[207:102], 
    fifo[ 67],fifo[101],fifo[ 84],fifo[ 16],fifo[ 50],fifo[ 33],
    fifo[100],fifo[ 83],fifo[ 66],fifo[ 49],fifo[ 32],fifo[ 15],
    fifo[ 82],fifo[ 65],fifo[ 99],fifo[ 31],fifo[ 14],fifo[ 48],
    fifo[ 64],fifo[ 98],fifo[ 81],fifo[ 13],fifo[ 47],fifo[ 30],
    fifo[ 97],fifo[ 80],fifo[ 63],fifo[ 46],fifo[ 29],fifo[ 12],
    fifo[ 79],fifo[ 62],fifo[ 96],fifo[ 28],fifo[ 11],fifo[ 45],
    fifo[ 61],fifo[ 95],fifo[ 78],fifo[ 10],fifo[ 44],fifo[ 27],
    fifo[ 94],fifo[ 77],fifo[ 60],fifo[ 43],fifo[ 26],fifo[  9],
    fifo[ 76],fifo[ 59],fifo[ 93],fifo[ 25],fifo[  8],fifo[ 42],
    fifo[ 58],fifo[ 92],fifo[ 75],fifo[  7],fifo[ 41],fifo[ 24],
    fifo[ 91],fifo[ 74],fifo[ 57],fifo[ 40],fifo[ 23],fifo[  6],
    fifo[ 73],fifo[ 56],fifo[ 90],fifo[ 22],fifo[  5],fifo[ 39],
    fifo[ 55],fifo[ 89],fifo[ 72],fifo[  4],fifo[ 38],fifo[ 21],
    fifo[ 88],fifo[ 71],fifo[ 54],fifo[ 37],fifo[ 20],fifo[  3],
    fifo[ 70],fifo[ 53],fifo[ 87],fifo[ 19],fifo[  2],fifo[ 36],
    fifo[ 52],fifo[ 86],fifo[ 69],fifo[  1],fifo[ 35],fifo[ 18],
    fifo[ 85],fifo[ 68],fifo[ 51],fifo[ 34],fifo[ 17],fifo[  0]
    };
  assign t_c17_n8 = {208{c17_n8}} & { fifo[207:136], 
    fifo[135],fifo[118],fifo[101],fifo[ 84],fifo[ 67],fifo[ 50],fifo[ 33],fifo[ 16],
    fifo[117],fifo[100],fifo[ 83],fifo[134],fifo[ 49],fifo[ 32],fifo[ 15],fifo[ 66],
    fifo[ 99],fifo[ 82],fifo[133],fifo[116],fifo[ 31],fifo[ 14],fifo[ 65],fifo[ 48],
    fifo[ 81],fifo[132],fifo[115],fifo[ 98],fifo[ 13],fifo[ 64],fifo[ 47],fifo[ 30],
    fifo[131],fifo[114],fifo[ 97],fifo[ 80],fifo[ 63],fifo[ 46],fifo[ 29],fifo[ 12],
    fifo[113],fifo[ 96],fifo[ 79],fifo[130],fifo[ 45],fifo[ 28],fifo[ 11],fifo[ 62],
    fifo[ 95],fifo[ 78],fifo[129],fifo[112],fifo[ 27],fifo[ 10],fifo[ 61],fifo[ 44],
    fifo[ 77],fifo[128],fifo[111],fifo[ 94],fifo[  9],fifo[ 60],fifo[ 43],fifo[ 26],
    fifo[127],fifo[110],fifo[ 93],fifo[ 76],fifo[ 59],fifo[ 42],fifo[ 25],fifo[  8],
    fifo[109],fifo[ 92],fifo[ 75],fifo[126],fifo[ 41],fifo[ 24],fifo[  7],fifo[ 58],
    fifo[ 91],fifo[ 74],fifo[125],fifo[108],fifo[ 23],fifo[  6],fifo[ 57],fifo[ 40],
    fifo[ 73],fifo[124],fifo[107],fifo[ 90],fifo[  5],fifo[ 56],fifo[ 39],fifo[ 22],
    fifo[123],fifo[106],fifo[ 89],fifo[ 72],fifo[ 55],fifo[ 38],fifo[ 21],fifo[  4],
    fifo[105],fifo[ 88],fifo[ 71],fifo[122],fifo[ 37],fifo[ 20],fifo[  3],fifo[ 54],
    fifo[ 87],fifo[ 70],fifo[121],fifo[104],fifo[ 19],fifo[  2],fifo[ 53],fifo[ 36],
    fifo[ 69],fifo[120],fifo[103],fifo[ 86],fifo[  1],fifo[ 52],fifo[ 35],fifo[ 18],
    fifo[119],fifo[102],fifo[ 85],fifo[ 68],fifo[ 51],fifo[ 34],fifo[ 17],fifo[  0]
    };
  assign t_c18_n2 = {208{c18_n2}} & { fifo[207: 36], 
    fifo[ 35],fifo[ 17],
    fifo[ 34],fifo[ 16],
    fifo[ 33],fifo[ 15],
    fifo[ 32],fifo[ 14],
    fifo[ 31],fifo[ 13],
    fifo[ 30],fifo[ 12],
    fifo[ 29],fifo[ 11],
    fifo[ 28],fifo[ 10],
    fifo[ 27],fifo[  9],
    fifo[ 26],fifo[  8],
    fifo[ 25],fifo[  7],
    fifo[ 24],fifo[  6],
    fifo[ 23],fifo[  5],
    fifo[ 22],fifo[  4],
    fifo[ 21],fifo[  3],
    fifo[ 20],fifo[  2],
    fifo[ 19],fifo[  1],
    fifo[ 18],fifo[  0]
    };
  assign t_c18_n4 = {208{c18_n4}} & { fifo[207: 72], 
    fifo[ 53],fifo[ 71],fifo[ 17],fifo[ 35],
    fifo[ 70],fifo[ 52],fifo[ 34],fifo[ 16],
    fifo[ 51],fifo[ 69],fifo[ 15],fifo[ 33],
    fifo[ 68],fifo[ 50],fifo[ 32],fifo[ 14],
    fifo[ 49],fifo[ 67],fifo[ 13],fifo[ 31],
    fifo[ 66],fifo[ 48],fifo[ 30],fifo[ 12],
    fifo[ 47],fifo[ 65],fifo[ 11],fifo[ 29],
    fifo[ 64],fifo[ 46],fifo[ 28],fifo[ 10],
    fifo[ 45],fifo[ 63],fifo[  9],fifo[ 27],
    fifo[ 62],fifo[ 44],fifo[ 26],fifo[  8],
    fifo[ 43],fifo[ 61],fifo[  7],fifo[ 25],
    fifo[ 60],fifo[ 42],fifo[ 24],fifo[  6],
    fifo[ 41],fifo[ 59],fifo[  5],fifo[ 23],
    fifo[ 58],fifo[ 40],fifo[ 22],fifo[  4],
    fifo[ 39],fifo[ 57],fifo[  3],fifo[ 21],
    fifo[ 56],fifo[ 38],fifo[ 20],fifo[  2],
    fifo[ 37],fifo[ 55],fifo[  1],fifo[ 19],
    fifo[ 54],fifo[ 36],fifo[ 18],fifo[  0]
    };
  assign t_c18_n6 = {208{c18_n6}} & { fifo[207:108], 
    fifo[ 89],fifo[ 71],fifo[107],fifo[ 35],fifo[ 17],fifo[ 53],
    fifo[ 70],fifo[106],fifo[ 88],fifo[ 16],fifo[ 52],fifo[ 34],
    fifo[105],fifo[ 87],fifo[ 69],fifo[ 51],fifo[ 33],fifo[ 15],
    fifo[ 86],fifo[ 68],fifo[104],fifo[ 32],fifo[ 14],fifo[ 50],
    fifo[ 67],fifo[103],fifo[ 85],fifo[ 13],fifo[ 49],fifo[ 31],
    fifo[102],fifo[ 84],fifo[ 66],fifo[ 48],fifo[ 30],fifo[ 12],
    fifo[ 83],fifo[ 65],fifo[101],fifo[ 29],fifo[ 11],fifo[ 47],
    fifo[ 64],fifo[100],fifo[ 82],fifo[ 10],fifo[ 46],fifo[ 28],
    fifo[ 99],fifo[ 81],fifo[ 63],fifo[ 45],fifo[ 27],fifo[  9],
    fifo[ 80],fifo[ 62],fifo[ 98],fifo[ 26],fifo[  8],fifo[ 44],
    fifo[ 61],fifo[ 97],fifo[ 79],fifo[  7],fifo[ 43],fifo[ 25],
    fifo[ 96],fifo[ 78],fifo[ 60],fifo[ 42],fifo[ 24],fifo[  6],
    fifo[ 77],fifo[ 59],fifo[ 95],fifo[ 23],fifo[  5],fifo[ 41],
    fifo[ 58],fifo[ 94],fifo[ 76],fifo[  4],fifo[ 40],fifo[ 22],
    fifo[ 93],fifo[ 75],fifo[ 57],fifo[ 39],fifo[ 21],fifo[  3],
    fifo[ 74],fifo[ 56],fifo[ 92],fifo[ 20],fifo[  2],fifo[ 38],
    fifo[ 55],fifo[ 91],fifo[ 73],fifo[  1],fifo[ 37],fifo[ 19],
    fifo[ 90],fifo[ 72],fifo[ 54],fifo[ 36],fifo[ 18],fifo[  0]
    };
  assign t_c18_n8 = {208{c18_n8}} & { fifo[207:144], 
    fifo[ 89],fifo[143],fifo[125],fifo[107],fifo[ 17],fifo[ 71],fifo[ 53],fifo[ 35],
    fifo[142],fifo[124],fifo[106],fifo[ 88],fifo[ 70],fifo[ 52],fifo[ 34],fifo[ 16],
    fifo[123],fifo[105],fifo[ 87],fifo[141],fifo[ 51],fifo[ 33],fifo[ 15],fifo[ 69],
    fifo[104],fifo[ 86],fifo[140],fifo[122],fifo[ 32],fifo[ 14],fifo[ 68],fifo[ 50],
    fifo[ 85],fifo[139],fifo[121],fifo[103],fifo[ 13],fifo[ 67],fifo[ 49],fifo[ 31],
    fifo[138],fifo[120],fifo[102],fifo[ 84],fifo[ 66],fifo[ 48],fifo[ 30],fifo[ 12],
    fifo[119],fifo[101],fifo[ 83],fifo[137],fifo[ 47],fifo[ 29],fifo[ 11],fifo[ 65],
    fifo[100],fifo[ 82],fifo[136],fifo[118],fifo[ 28],fifo[ 10],fifo[ 64],fifo[ 46],
    fifo[ 81],fifo[135],fifo[117],fifo[ 99],fifo[  9],fifo[ 63],fifo[ 45],fifo[ 27],
    fifo[134],fifo[116],fifo[ 98],fifo[ 80],fifo[ 62],fifo[ 44],fifo[ 26],fifo[  8],
    fifo[115],fifo[ 97],fifo[ 79],fifo[133],fifo[ 43],fifo[ 25],fifo[  7],fifo[ 61],
    fifo[ 96],fifo[ 78],fifo[132],fifo[114],fifo[ 24],fifo[  6],fifo[ 60],fifo[ 42],
    fifo[ 77],fifo[131],fifo[113],fifo[ 95],fifo[  5],fifo[ 59],fifo[ 41],fifo[ 23],
    fifo[130],fifo[112],fifo[ 94],fifo[ 76],fifo[ 58],fifo[ 40],fifo[ 22],fifo[  4],
    fifo[111],fifo[ 93],fifo[ 75],fifo[129],fifo[ 39],fifo[ 21],fifo[  3],fifo[ 57],
    fifo[ 92],fifo[ 74],fifo[128],fifo[110],fifo[ 20],fifo[  2],fifo[ 56],fifo[ 38],
    fifo[ 73],fifo[127],fifo[109],fifo[ 91],fifo[  1],fifo[ 55],fifo[ 37],fifo[ 19],
    fifo[126],fifo[108],fifo[ 90],fifo[ 72],fifo[ 54],fifo[ 36],fifo[ 18],fifo[  0]
    };
  assign t_c26_n2 = {208{c26_n2}} & { fifo[207: 52], 
    fifo[ 51],fifo[ 25],
    fifo[ 50],fifo[ 24],
    fifo[ 49],fifo[ 23],
    fifo[ 48],fifo[ 22],
    fifo[ 47],fifo[ 21],
    fifo[ 46],fifo[ 20],
    fifo[ 45],fifo[ 19],
    fifo[ 44],fifo[ 18],
    fifo[ 43],fifo[ 17],
    fifo[ 42],fifo[ 16],
    fifo[ 41],fifo[ 15],
    fifo[ 40],fifo[ 14],
    fifo[ 39],fifo[ 13],
    fifo[ 38],fifo[ 12],
    fifo[ 37],fifo[ 11],
    fifo[ 36],fifo[ 10],
    fifo[ 35],fifo[  9],
    fifo[ 34],fifo[  8],
    fifo[ 33],fifo[  7],
    fifo[ 32],fifo[  6],
    fifo[ 31],fifo[  5],
    fifo[ 30],fifo[  4],
    fifo[ 29],fifo[  3],
    fifo[ 28],fifo[  2],
    fifo[ 27],fifo[  1],
    fifo[ 26],fifo[  0]
    };
  assign t_c26_n4 = {208{c26_n4}} & { fifo[207:104], 
    fifo[ 77],fifo[103],fifo[ 25],fifo[ 51],
    fifo[102],fifo[ 76],fifo[ 50],fifo[ 24],
    fifo[ 75],fifo[101],fifo[ 23],fifo[ 49],
    fifo[100],fifo[ 74],fifo[ 48],fifo[ 22],
    fifo[ 73],fifo[ 99],fifo[ 21],fifo[ 47],
    fifo[ 98],fifo[ 72],fifo[ 46],fifo[ 20],
    fifo[ 71],fifo[ 97],fifo[ 19],fifo[ 45],
    fifo[ 96],fifo[ 70],fifo[ 44],fifo[ 18],
    fifo[ 69],fifo[ 95],fifo[ 17],fifo[ 43],
    fifo[ 94],fifo[ 68],fifo[ 42],fifo[ 16],
    fifo[ 67],fifo[ 93],fifo[ 15],fifo[ 41],
    fifo[ 92],fifo[ 66],fifo[ 40],fifo[ 14],
    fifo[ 65],fifo[ 91],fifo[ 13],fifo[ 39],
    fifo[ 90],fifo[ 64],fifo[ 38],fifo[ 12],
    fifo[ 63],fifo[ 89],fifo[ 11],fifo[ 37],
    fifo[ 88],fifo[ 62],fifo[ 36],fifo[ 10],
    fifo[ 61],fifo[ 87],fifo[  9],fifo[ 35],
    fifo[ 86],fifo[ 60],fifo[ 34],fifo[  8],
    fifo[ 59],fifo[ 85],fifo[  7],fifo[ 33],
    fifo[ 84],fifo[ 58],fifo[ 32],fifo[  6],
    fifo[ 57],fifo[ 83],fifo[  5],fifo[ 31],
    fifo[ 82],fifo[ 56],fifo[ 30],fifo[  4],
    fifo[ 55],fifo[ 81],fifo[  3],fifo[ 29],
    fifo[ 80],fifo[ 54],fifo[ 28],fifo[  2],
    fifo[ 53],fifo[ 79],fifo[  1],fifo[ 27],
    fifo[ 78],fifo[ 52],fifo[ 26],fifo[  0]
    };
  assign t_c26_n6 = {208{c26_n6}} & { fifo[207:156], 
    fifo[103],fifo[155],fifo[129],fifo[ 25],fifo[ 77],fifo[ 51],
    fifo[154],fifo[128],fifo[102],fifo[ 76],fifo[ 50],fifo[ 24],
    fifo[127],fifo[101],fifo[153],fifo[ 49],fifo[ 23],fifo[ 75],
    fifo[100],fifo[152],fifo[126],fifo[ 22],fifo[ 74],fifo[ 48],
    fifo[151],fifo[125],fifo[ 99],fifo[ 73],fifo[ 47],fifo[ 21],
    fifo[124],fifo[ 98],fifo[150],fifo[ 46],fifo[ 20],fifo[ 72],
    fifo[ 97],fifo[149],fifo[123],fifo[ 19],fifo[ 71],fifo[ 45],
    fifo[148],fifo[122],fifo[ 96],fifo[ 70],fifo[ 44],fifo[ 18],
    fifo[121],fifo[ 95],fifo[147],fifo[ 43],fifo[ 17],fifo[ 69],
    fifo[ 94],fifo[146],fifo[120],fifo[ 16],fifo[ 68],fifo[ 42],
    fifo[145],fifo[119],fifo[ 93],fifo[ 67],fifo[ 41],fifo[ 15],
    fifo[118],fifo[ 92],fifo[144],fifo[ 40],fifo[ 14],fifo[ 66],
    fifo[ 91],fifo[143],fifo[117],fifo[ 13],fifo[ 65],fifo[ 39],
    fifo[142],fifo[116],fifo[ 90],fifo[ 64],fifo[ 38],fifo[ 12],
    fifo[115],fifo[ 89],fifo[141],fifo[ 37],fifo[ 11],fifo[ 63],
    fifo[ 88],fifo[140],fifo[114],fifo[ 10],fifo[ 62],fifo[ 36],
    fifo[139],fifo[113],fifo[ 87],fifo[ 61],fifo[ 35],fifo[  9],
    fifo[112],fifo[ 86],fifo[138],fifo[ 34],fifo[  8],fifo[ 60],
    fifo[ 85],fifo[137],fifo[111],fifo[  7],fifo[ 59],fifo[ 33],
    fifo[136],fifo[110],fifo[ 84],fifo[ 58],fifo[ 32],fifo[  6],
    fifo[109],fifo[ 83],fifo[135],fifo[ 31],fifo[  5],fifo[ 57],
    fifo[ 82],fifo[134],fifo[108],fifo[  4],fifo[ 56],fifo[ 30],
    fifo[133],fifo[107],fifo[ 81],fifo[ 55],fifo[ 29],fifo[  3],
    fifo[106],fifo[ 80],fifo[132],fifo[ 28],fifo[  2],fifo[ 54],
    fifo[ 79],fifo[131],fifo[105],fifo[  1],fifo[ 53],fifo[ 27],
    fifo[130],fifo[104],fifo[ 78],fifo[ 52],fifo[ 26],fifo[  0]
    };
  assign t_c26_n8 = {208{c26_n8}} & { 
    fifo[129],fifo[207],fifo[181],fifo[155],fifo[ 25],fifo[103],fifo[ 77],fifo[ 51],
    fifo[206],fifo[180],fifo[154],fifo[128],fifo[102],fifo[ 76],fifo[ 50],fifo[ 24],
    fifo[179],fifo[153],fifo[127],fifo[205],fifo[ 75],fifo[ 49],fifo[ 23],fifo[101],
    fifo[152],fifo[126],fifo[204],fifo[178],fifo[ 48],fifo[ 22],fifo[100],fifo[ 74],
    fifo[125],fifo[203],fifo[177],fifo[151],fifo[ 21],fifo[ 99],fifo[ 73],fifo[ 47],
    fifo[202],fifo[176],fifo[150],fifo[124],fifo[ 98],fifo[ 72],fifo[ 46],fifo[ 20],
    fifo[175],fifo[149],fifo[123],fifo[201],fifo[ 71],fifo[ 45],fifo[ 19],fifo[ 97],
    fifo[148],fifo[122],fifo[200],fifo[174],fifo[ 44],fifo[ 18],fifo[ 96],fifo[ 70],
    fifo[121],fifo[199],fifo[173],fifo[147],fifo[ 17],fifo[ 95],fifo[ 69],fifo[ 43],
    fifo[198],fifo[172],fifo[146],fifo[120],fifo[ 94],fifo[ 68],fifo[ 42],fifo[ 16],
    fifo[171],fifo[145],fifo[119],fifo[197],fifo[ 67],fifo[ 41],fifo[ 15],fifo[ 93],
    fifo[144],fifo[118],fifo[196],fifo[170],fifo[ 40],fifo[ 14],fifo[ 92],fifo[ 66],
    fifo[117],fifo[195],fifo[169],fifo[143],fifo[ 13],fifo[ 91],fifo[ 65],fifo[ 39],
    fifo[194],fifo[168],fifo[142],fifo[116],fifo[ 90],fifo[ 64],fifo[ 38],fifo[ 12],
    fifo[167],fifo[141],fifo[115],fifo[193],fifo[ 63],fifo[ 37],fifo[ 11],fifo[ 89],
    fifo[140],fifo[114],fifo[192],fifo[166],fifo[ 36],fifo[ 10],fifo[ 88],fifo[ 62],
    fifo[113],fifo[191],fifo[165],fifo[139],fifo[  9],fifo[ 87],fifo[ 61],fifo[ 35],
    fifo[190],fifo[164],fifo[138],fifo[112],fifo[ 86],fifo[ 60],fifo[ 34],fifo[  8],
    fifo[163],fifo[137],fifo[111],fifo[189],fifo[ 59],fifo[ 33],fifo[  7],fifo[ 85],
    fifo[136],fifo[110],fifo[188],fifo[162],fifo[ 32],fifo[  6],fifo[ 84],fifo[ 58],
    fifo[109],fifo[187],fifo[161],fifo[135],fifo[  5],fifo[ 83],fifo[ 57],fifo[ 31],
    fifo[186],fifo[160],fifo[134],fifo[108],fifo[ 82],fifo[ 56],fifo[ 30],fifo[  4],
    fifo[159],fifo[133],fifo[107],fifo[185],fifo[ 55],fifo[ 29],fifo[  3],fifo[ 81],
    fifo[132],fifo[106],fifo[184],fifo[158],fifo[ 28],fifo[  2],fifo[ 80],fifo[ 54],
    fifo[105],fifo[183],fifo[157],fifo[131],fifo[  1],fifo[ 79],fifo[ 53],fifo[ 27],
    fifo[182],fifo[156],fifo[130],fifo[104],fifo[ 78],fifo[ 52],fifo[ 26],fifo[  0]
    };

endmodule
`default_nettype wire
