/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     : 
* Description      : tx_fd_dup
* Simulation Notes : 
* Synthesis Notes  : 
* Application Note : 
* Simulator        : 
* Parameters       : 
* Terms & concepts : 
* Bugs             : 
* Open issues and future enhancements : 
* References       : 
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module tx_fd_dup
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire          clk,
  input  wire          rst_n,
  
  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire          enable,
  output reg           done,

  /*****************************************************************************
  * phy config
  *****************************************************************************/
  input  wire  [ 1:0]  conf_bandwidth,
  input  wire  [ 2:0]  conf_primary,
  
  /*****************************************************************************
  * frame parameters
  *****************************************************************************/
  input  wire  [ 3:0]  frame_format,
  input  wire  [ 1:0]  frame_bandwidth,
  input  wire  [ 1:0]  frame_heltf_type,
  input  wire          frame_beam_change,
  input  wire          frame_htdup,
  input  wire  [ 3:0]  frame_hesubchan20,
  
  /*****************************************************************************
  * symbol parameters
  *****************************************************************************/
  input  wire  [ 5:0]  symbol,
  
  /*****************************************************************************
  * input
  *****************************************************************************/
  output wire         ready,
  input  wire  [12:0] i0,
  input  wire  [12:0] q0,
  input  wire  [ 9:0] index,
  input  wire         last,
  input  wire         valid,

  /*****************************************************************************
  * output
  *****************************************************************************/
  output wire         fft_en,
  output wire  [ 9:0] fft_index,
  output wire  [ 2:0] fft_len,
  output wire  [ 3:0] fft_scale,
  output wire  [12:0] fft_i0,
  output wire  [12:0] fft_q0
);
  
  /*****************************************************************************
  * declarations
  ******************************************************************************/
  localparam NON_HT=4'd0,
             NON_HT_DUP=4'd1,
             HT_MM=4'd2,
             HT_GF=4'd3,
             VHT=4'd4,
             HE_SU=4'd5,
             HE_MU=4'd6,
             HE_ER_SU=4'd7,
             HE_TB=4'd8;
  
  localparam GAMMA_P1=2'd0,
             GAMMA_PJ=2'd1,
             GAMMA_M1=2'd2;

  localparam BW_20=2'd0,
             BW_40=2'd1,
             BW_80=2'd2;
             
  localparam  IDLE       = 6'd0,
              LSTF       = 6'd1,
              LLTF       = 6'd2,
              LSIG       = 6'd3,
              LDATA      = 6'd4,
              HTGFSTF    = 6'd5,
              HTGFLTF    = 6'd6,
              HTGFSIG1   = 6'd7,
              HTGFSIG2   = 6'd8,
              HTMMSIG1   = 6'd9,
              HTMMSIG2   = 6'd10,
              HTMMSTF    = 6'd11,
              HTDLTF     = 6'd12,
              HTELTF     = 6'd13,
              HTDATA     = 6'd14,
              VHTSIGA1   = 6'd15,
              VHTSIGA2   = 6'd16,
              VHTSTF     = 6'd17,
              VHTLTF     = 6'd18,
              VHTSIGB    = 6'd19,
              VHTDATA    = 6'd20,
              HELSIG     = 6'd21,
              HESIGA1    = 6'd22,
              HESIGA2    = 6'd23,
              HESIGB     = 6'd24,
              HESTF      = 6'd25,
              HELTF      = 6'd26,
              HEDATA     = 6'd27,
              HEPE       = 6'd28,
              DONE       = 6'd63;
 
  localparam HELTF_1X=2'd0,
             HELTF_2X=2'd1,
             HELTF_4X=2'd2;

  localparam FFT64=3'd0,
             FFT128=3'd1,
             FFT256=3'd2,
             FFT512=3'd3,
             FFT1024=3'd4,
             FFT2048=3'd5;
  
  /*****************************************************************************
  * S0: duplication and gamma
  *****************************************************************************/
  reg  [ 1:0] n_s0_state;
  reg  [ 1:0] s0_state;
  reg  [ 9:0] s0_index;
  reg         s0_last;
  reg  [12:0] s0_i0,s0_q0;
  reg         s0_valid;
  
  wire [ 9:0] s0_index_m96,s0_index_m32,s0_index_p32,s0_index_p96;
  reg  [ 9:0] s0_dup_index;
  reg         s0_dup_last;
  
  wire [12:0] s0_neg_i0, s0_neg_q0;
  reg  [12:0] s0_gamma_i0,s0_gamma_q0;
    
  /* duplication */
  wire is_duplicable;
  assign is_duplicable = symbol==LSTF     | symbol==LLTF     | symbol==LSIG     | symbol==LDATA |
                         symbol==HTGFSTF  | symbol==HTGFSIG1 | symbol==HTGFSIG2 | 
                         symbol==HTMMSIG1 | symbol==HTMMSIG2 | symbol==HTMMSTF  | symbol==HTDATA & frame_htdup |
                         symbol==VHTSIGA1 | symbol==VHTSIGA2 | symbol==VHTSTF   | 
                         symbol==HELSIG   | symbol==HESIGA1  | symbol==HESIGA2;
  
  wire is_he_modulated;
  assign is_he_modulated  = symbol==HESTF | symbol==HELTF | symbol==HEDATA | symbol==HEPE; 
  
  wire is_he;
  assign is_he            = frame_format==HE_SU | frame_format==HE_MU | frame_format==HE_ER_SU | frame_format==HE_TB;
  
  /* duplication */
  assign ready = s0_state==2'd0;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s0_i0    <= 13'd0;
      s0_q0    <= 13'd0; 
      s0_index <= 10'd0;
      s0_last  <= 1'b0;
      s0_valid <= 1'b0;
      s0_state <= 2'b0;
    end
    else if(!enable)
    begin
      s0_i0    <= 13'd0;
      s0_q0    <= 13'd0; 
      s0_index <= 10'd0;
      s0_last  <= 1'b0;
      s0_valid <= 1'b0;
      s0_state <= 2'b0;
    end
    else
    begin
      if(s0_state==2'd0)
      begin
        if(valid)
        begin
          s0_i0    <= i0;
          s0_q0    <= q0;
          s0_index <= index;
          s0_last  <= last;
          s0_valid <= 1'b1;
          s0_state <= n_s0_state;
        end
        else
        begin
          s0_valid <= 1'b0;
        end
      end
      else
      begin
        s0_state <= n_s0_state;
      end
    end
  end
  
  assign s0_index_m96 = s0_index - 10'd96;
  assign s0_index_m32 = s0_index - 10'd32;
  assign s0_index_p32 = s0_index + 10'd32;
  assign s0_index_p96 = s0_index + 10'd96;
  
  always @(*)
  begin
    if(is_he)
    begin
      /* HE context */
      if(!is_he_modulated)
      begin
        /* preamble, NON-HE modulated */
        case(frame_bandwidth)
          BW_20:
          begin
           { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index, s0_last};
          end
          BW_40:
          begin
            case(frame_hesubchan20)
              4'b1000,4'b0010: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m32, s0_last};
              4'b0100,4'b0001: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_p32, s0_last};
              default:
              begin
                case(s0_state)
                  2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m32,    1'b0};
                  default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_p32, s0_last};
                endcase
              end
            endcase
          end
          default: /* BW_80 */
          begin
            case(frame_hesubchan20)
              4'b1000:     { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m96, s0_last};
              4'b0100:     { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m32, s0_last};
              4'b0010:     { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_p32, s0_last};
              4'b0001:     { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_p96, s0_last};
              4'b1100:
                case(s0_state)
                  2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m96,    1'b0};
                  default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_m32, s0_last};
                endcase
              4'b0110:
                case(s0_state)
                  2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m32,    1'b0};
                  default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_p32, s0_last};
                endcase
              4'b0011:
                case(s0_state)
                  2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_p32,    1'b0};
                  default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_p96, s0_last};
                endcase
              default: /* 4'b1111 */
                case(s0_state)
                  2'd3:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd2, s0_index_m96,    1'b0};
                  2'd2:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_m32,    1'b0};
                  2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_p32,    1'b0};
                  default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd3, s0_index_p96, s0_last};
                endcase
            endcase
          end
        endcase
      end
      else
      begin
        /* HE modulated */
        { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index,   s0_last};
      end
    end
    else
    begin
      if(is_duplicable)
      begin
        case(frame_bandwidth)
          BW_20:   { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index, s0_last};
          BW_40:
            case(s0_state)
              2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_m32,    1'b0};
              default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_p32, s0_last};
            endcase
          default:
            case(s0_state)
              2'd3:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd2, s0_index_m96,    1'b0};
              2'd2:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd1, s0_index_m32,    1'b0};
              2'd1:    { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index_p32,    1'b0};
              default: { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd3, s0_index_p96, s0_last};
            endcase
        endcase
      end
      else
      begin
        { n_s0_state, s0_dup_index, s0_dup_last} = { 2'd0, s0_index, s0_last};
      end
    end
  end
  
  assign s0_neg_i0 = ~s0_i0 + 13'd1;
  assign s0_neg_q0 = ~s0_q0 + 13'd1;
  
  /* gamma */
  always @(*)
  begin
    if(!is_he_modulated && frame_beam_change || !is_he)
    begin
      /* gamma applied */
      case(frame_bandwidth)
        BW_20: 
          { s0_gamma_i0, s0_gamma_q0}  = { s0_i0, s0_q0}; /* +1 */
        
        BW_40: 
          case(s0_dup_index[9:7]) /* < 0 */
            3'b111:  {s0_gamma_i0, s0_gamma_q0} = {     s0_i0,     s0_q0}; /* +1 */
            default: {s0_gamma_i0, s0_gamma_q0} = { s0_neg_q0,     s0_i0}; /* +i */
          endcase
       
        default: /* CBW80 */
          case(s0_dup_index[9:6]) /* < -64 */
            4'b1110: {s0_gamma_i0, s0_gamma_q0} = {     s0_i0,     s0_q0}; /* +1 */
            default: {s0_gamma_i0, s0_gamma_q0} = { s0_neg_i0, s0_neg_q0}; /* -1 */
          endcase
      endcase
    end
    else
    begin
      /* no gamma applied */
      {s0_gamma_i0, s0_gamma_q0} = { s0_i0, s0_q0}; /* +1 */
    end
  end
  
  /*****************************************************************************
  * S1: FFT OFFSET
  ******************************************************************************
  * o if frame_bandwidth < conf_bandwidth then shift to primary    (all formats)
  *     
  *     the shift is indicated by the *conf_primary* parameter
  *
  * o if frame_bandwidth > conf_bandwidth then shift to fft center (HE_TB only)
  *     
  *     the shift is indicated by the *frame_hesubchan20* parameter
  *
  ******************************************************************************
  * conf_primary definition
  *
  *   config_bandwidth=40M
  *
  *       0  [ 20p 20s ] 
  *       1  [ 20s 20p ] 
  *
  *   config_bandwidth=80M
  *
  *       0  [ 20p 20s 40s] 
  *       1  [ 20s 20p 40s] 
  *       2  [ 40s 20p 20s] 
  *       3  [ 40s 20s 20p] 
  *
  *   config_bandwidth=160M
  *
  *       0  [ 20p 20s 40s 80s] 
  *       1  [ 20s 20p 40s 80s] 
  *       2  [ 40s 20p 20s 80s] 
  *       3  [ 40s 20s 20p 80s] 
  *       4  [ 80s 20p 20s 40s] 
  *       5  [ 80s 20s 20p 40s] 
  *       6  [ 80s 40s 20p 20s] 
  *       7  [ 80s 40s 20s 20p] 
  *
  *****************************************************************************/
  /* shift to center/primary */
  reg [ 9:0] s0_fft_offset;
  
  always @(*)
  begin:b_fft_offset
  
    reg [7:0] v_offset;
  
    case({conf_bandwidth,frame_bandwidth})
      {BW_20,BW_20}:  
        v_offset =  8'd0;
     
      {BW_20,BW_40}:
        case(frame_hesubchan20)
          4'b1000,4'b0010: v_offset =  8'd32;
          default:         v_offset = -8'd32;
        endcase  
     
      {BW_20,BW_80}:
        case(frame_hesubchan20)
          4'b1000: v_offset = +8'd96;
          4'b0100: v_offset = +8'd32;
          4'b0010: v_offset = -8'd32;
          default: v_offset = -8'd96;
        endcase
     
      {BW_40,BW_20}:
        case(conf_primary)
          3'd0,3'd2,3'd4,3'd6: v_offset = -8'd32;
          default:             v_offset = +8'd32;
        endcase
      
      {BW_40,BW_40}:
        v_offset =  8'd0;
      
      {BW_40,BW_80}:
        case(frame_hesubchan20)
          4'b1000,4'b1100,4'b0100: v_offset = +8'd64;
          default:                 v_offset = -8'd64;
        endcase
     
      {BW_80,BW_20}:
        case(conf_primary)
          3'd0,3'd4: v_offset = -8'd96;
          3'd1,3'd5: v_offset = -8'd32; 
          3'd2,3'd6: v_offset = +8'd32; 
          default:   v_offset = +8'd96;
        endcase 
          
      {BW_80,BW_40}:
        case(conf_primary)
          3'd0,3'd1,3'd4,3'd5: v_offset = -8'd64;
          default:             v_offset = +8'd64;
        endcase 
      default: /* {BW_80,BW_40} */
        v_offset = 8'd0;
    endcase
    
    if(!is_he_modulated)
      s0_fft_offset = {{2{v_offset[7]}},v_offset};
    else
      s0_fft_offset = {v_offset,2'b0};
    
  end
  
  reg [ 2:0] s0_fft_len;
  reg [ 1:0] s0_fft_shift;
  reg [ 3:0] s0_fft_scale;
  
  always @(*)
  begin
    if(!is_he_modulated)
    begin
      if(!is_he)
      begin
        case({conf_bandwidth,frame_bandwidth})
          {BW_20,BW_20}: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT128,2'd0,4'd2};
          {BW_40,BW_20}: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd0,4'd2};
          {BW_40,BW_40}: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd0,4'd3};
          {BW_80,BW_20}: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd0,4'd2};
          {BW_80,BW_40}: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd0,4'd3};
          default:       {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd0,4'd4};
        endcase
      end
      else
      begin
        case(conf_bandwidth)
          BW_20: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT128,2'd0,4'd2};
          BW_40:
            case(frame_hesubchan20)
              4'b1000,4'b0100,4'b0010,4'b0001: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd0,4'd2};
              default:                         {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd0,4'd3};
            endcase
          default: /* BW_80 */
            case(frame_hesubchan20)
              4'b1000,4'b0100,4'b0010,4'b0001: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT128,2'd0,4'd2};
              4'b1100,4'b0110,4'b0011:         {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd0,4'd3};
              default:                         {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd0,4'd4}; /* 1111 */
            endcase
        endcase
      end
    end
    else
    begin
      case(symbol)
        HESTF:
        begin
          case(conf_bandwidth)
            BW_20:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT128,2'd2,4'd2};
            BW_40:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd2,4'd2};
            default: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd2,4'd2};
          endcase
        end
        HELTF:
        begin
          case(frame_heltf_type)
            HELTF_1X:
            begin
              case(conf_bandwidth)
                BW_20:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT128,2'd2,4'd2};
                BW_40:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd2,(frame_bandwidth==BW_40)?4'd3:4'd2};
                default: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd2,4'd4};
              endcase
            end
            HELTF_2X:
            begin
              case(conf_bandwidth)
                BW_20:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = { FFT256,2'd1,4'd2};
                BW_40:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = { FFT512,2'd1,(frame_bandwidth==BW_40)?4'd3:4'd2};
                default: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT1024,2'd1,4'd4};
              endcase
            end
            default: /* HELTF_4X */
            begin
              case(conf_bandwidth)
                BW_20:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = { FFT512,2'd0,4'd2};
                BW_40:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT1024,2'd0,(frame_bandwidth==BW_40)?4'd3:4'd2};
                default: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT2048,2'd0,4'd4};
              endcase
            end
          endcase
        end
        
        HEDATA:
        begin
          case(conf_bandwidth)
            BW_20:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = { FFT512,2'd0,4'd2};
            BW_40:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT1024,2'd0,(frame_bandwidth==BW_40)?4'd3:4'd2};
            default: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT2048,2'd0,4'd4};
          endcase
        end
       
        default: /* HEPE */
        begin
          case(conf_bandwidth)              
            BW_20:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT128,2'd2,4'd2};  
            BW_40:  {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT256,2'd2,4'd2};  
            default: {s0_fft_len,s0_fft_shift,s0_fft_scale} = {FFT512,2'd2,4'd2};  
          endcase                            
        end
        
      
      endcase
    end
  end

  wire [ 9:0] s0_preshift_index;
  assign s0_preshift_index = s0_dup_index + s0_fft_offset;
  
  reg  [ 9:0] s0_fft_index;
  always @(*)
  begin
    case(s0_fft_shift)
      2'd0:    s0_fft_index = s0_preshift_index;
      2'd1:    s0_fft_index = {{1{s0_preshift_index[9]}},s0_preshift_index[9:1]};
      default: s0_fft_index = {{2{s0_preshift_index[9]}},s0_preshift_index[9:2]};
    endcase
  end

  reg  [ 9:0] s1_index;
  reg  [ 2:0] s1_len;
  reg  [ 3:0] s1_scale;
  reg  [12:0] s1_i0;
  reg  [12:0] s1_q0;
  reg         s1_last;
  reg         s1_valid;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      done      <= 1'b0;
      s1_valid  <= 1'b0;    
      s1_index  <= 10'd0;
      s1_len    <= 3'd0;
      s1_scale  <= 4'd0;
      s1_i0     <= 13'd0;
      s1_q0     <= 13'd0;
      s1_last   <= 1'b0;   
    end
    else if(!enable)
    begin
      done      <= 1'b0;
      s1_valid  <= 1'b0;    
      s1_index  <= 10'd0;
      s1_len    <= 3'd0;
      s1_scale  <= 4'd0;
      s1_i0     <= 13'd0;
      s1_q0     <= 13'd0;   
      s1_last   <= 1'b0;   
    end
    else
    begin
      done      <= s1_valid && s1_last;
      s1_i0     <= s0_gamma_i0;
      s1_q0     <= s0_gamma_q0;
      s1_index  <= s0_fft_index;
      s1_len    <= s0_fft_len;
      s1_scale  <= s0_fft_scale;
      s1_last   <= s0_dup_last;
      s1_valid  <= s0_valid;
    end
  end   
  
  assign fft_en    = s1_valid;
  assign fft_i0    = s1_i0;
  assign fft_q0    = s1_q0;
  assign fft_index = s1_index;
  assign fft_len   = s1_len;
  assign fft_scale = s1_scale;
  
endmodule
`default_nettype wire
