`default_nettype none
module tx_scheduler
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire          clk,
  input wire          rst_n,

  /*****************************************************************************
  * 
  *****************************************************************************/
  input wire  [ 1:0]  mdmcfg_conf_bw,
  
  /*****************************************************************************
  * 
  *****************************************************************************/
  input  wire         enable,
  output reg          mem_released,
  
  input  wire         framep1_update,
  input  wire [ 3:0]  framep1_format,
  input  wire         framep1_fec_coding,
  input  wire [ 2:0]  framep1_nsts,
  input  wire [ 1:0]  framep1_ness,
  input  wire [ 2:0]  framep1_nheltf,
  input  wire [ 1:0]  framep1_heltf_type,
  input  wire         framep1_doppler,
  input  wire         framep1_midamble,
  input  wire         framep1_continuoustx,
  input  wire         framep2_update,
  input  wire [15:0]  framep2_nsym,
  input  wire [ 2:0]  framep2_tpe,
  input  wire [ 8:0]  framep2_nma,
  
  /*****************************************************************************
  * 
  *****************************************************************************/
  /* BD ISSUE */
  output wire        bd_valid,
  output reg  [ 5:0] bd_symbol,
  output reg         bd_last,
  input  wire        bd_done,

  /* FD ISSUE*/
  output wire        fd_valid,
  output reg  [ 5:0] fd_symbol,
  output reg  [ 2:0] fd_lo_sel,
  output reg  [ 2:0] fd_hi_sel,
  output reg         fd_offset,
  input  wire        fd_done,
  
  /* FFT ISSUE*/
  output wire        fft_valid,
  output reg  [ 5:0] fft_symbol,
  output reg  [ 2:0] fft_hi_sel,
  output reg  [ 2:0] fft_lo_sel,
  output reg         fft_offset,
  input  wire        fft_done,
  
  /* TD ISSUE*/
  output wire        td_valid,
  output reg  [ 5:0] td_symbol,
  output reg         td_last,
  output reg  [ 2:0] td_hi_sel,
  output reg  [ 2:0] td_lo_sel,
  output reg         td_offset,
  input  wire        td_done
  
);
  /*****************************************************************************
  * symbol generation
  *****************************************************************************/
  localparam  IDLE       = 6'd0,
              LSTF       = 6'd1,
              LLTF       = 6'd2,
              LSIG       = 6'd3,
              LDATA      = 6'd4,
              HTGFSTF    = 6'd5,
              HTGFLTF    = 6'd6,
              HTGFSIG1   = 6'd7,
              HTGFSIG2   = 6'd8,
              HTMMSIG1   = 6'd9,
              HTMMSIG2   = 6'd10,
              HTMMSTF    = 6'd11,
              HTDLTF     = 6'd12,
              HTELTF     = 6'd13,
              HTDATA     = 6'd14,
              VHTSIGA1   = 6'd15,
              VHTSIGA2   = 6'd16,
              VHTSTF     = 6'd17,
              VHTLTF     = 6'd18,
              VHTSIGB    = 6'd19,
              VHTDATA    = 6'd20,
              HELSIG     = 6'd21,
              HESIGA1    = 6'd22,
              HESIGA2    = 6'd23,
              HESIGB     = 6'd24,
              HESTF      = 6'd25,
              HELTF      = 6'd26,
              HEDATA     = 6'd27,
              HEPE       = 6'd28,
              DONE       = 6'd63;
  
  localparam  NON_HT     = 4'd0,
              NON_HT_DUP = 4'd1,
              HT_MM      = 4'd2,
              HT_GF      = 4'd3,
              VHT        = 4'd4,
              HE_SU      = 4'd5,
              HE_MU      = 4'd6,
              HE_ER_SU   = 4'd7,
              HE_TB      = 4'd8;

  localparam  NSTS_1     = 3'd0,
              NSTS_2     = 3'd1,
              NSTS_3     = 3'd2,
              NSTS_4     = 3'd3,
              NSTS_5     = 3'd4,
              NSTS_6     = 3'd5,
              NSTS_7     = 3'd6,
              NSTS_8     = 3'd7;
  
  /* SYMBOL GENERATOR */
  reg  [ 5:0]  n_symbol;
  reg  [ 2:0]  nhtdltf;
  reg  [ 2:0]  nhteltf;
  reg  [ 3:0]  nvhtltf;
  reg  [ 3:0]  nheltf;
  wire [ 4:0]  mma;
  
  wire [15:0]  count_sym_incr;
  wire [ 4:0]  count_mma_incr;   
  wire [ 8:0]  count_nma_incr;   
  wire [ 2:0]  count_htdltf_incr;
  wire [ 2:0]  count_hteltf_incr;
  wire [ 3:0]  count_vhtltf_incr;
  wire [ 3:0]  count_heltf_incr;   
  
  reg  [15:0]  n_count_sym;
  reg  [ 4:0]  n_count_mma;           
  reg  [ 8:0]  n_count_nma;           
  reg  [ 2:0]  n_count_htdltf; // 0-3 
  reg  [ 2:0]  n_count_hteltf; // 0-3 
  reg  [ 3:0]  n_count_vhtltf; // 1-8 
  reg  [ 3:0]  n_count_heltf;  // 1-8 
  
  reg  [15:0]  count_sym;
  reg  [ 4:0]  count_mma;
  reg  [ 8:0]  count_nma;
  reg  [ 2:0]  count_htdltf; // 0-3
  reg  [ 2:0]  count_hteltf; // 0-3
  reg  [ 3:0]  count_vhtltf; // 1-8
  reg  [ 3:0]  count_heltf;  // 1-8
  reg          symbol_valid;
  reg  [ 5:0]  symbol;       // symbol name
  reg          symbol_data;  // symbol contains data sub-carrier (SIG and DATA)
  reg          symbol_last;  // symbol is the last one
  reg          preamble_done;
  wire         symbol_ready;

  /* BD FIFO */
  wire         bd_fifo_full,bd_fifo_empty;
  wire         bd_fifo_wren, bd_fifo_rden;
  wire [ 2:0]  bd_fifo_in_sh;
  wire [ 3:0]  n_bd_fifo_count;
  wire [ 6:0]  n_bd_fifo_in;
  wire [55:0]  n_bd_fifo_sh1,n_bd_fifo_sh2,n_bd_fifo_sh4,n_bd_fifo_out,n_bd_fifo;
  reg  [ 3:0]  bd_fifo_count;
  reg  [55:0]  bd_fifo;
  
  /* BD ISSUE  */
  reg          bd_assigned;

  /* FD FIFO */
  wire         fd_fifo_full,fd_fifo_empty;
  wire         fd_fifo_wren, fd_fifo_rden;
  wire [ 2:0]  fd_fifo_in_sh;
  wire [ 3:0]  n_fd_fifo_count;
  wire [ 8:0]  n_fd_fifo_in;
  wire [71:0]  n_fd_fifo_sh1,n_fd_fifo_sh2,n_fd_fifo_sh4,n_fd_fifo_out,n_fd_fifo;
  reg  [ 3:0]  fd_fifo_count;
  reg  [71:0]  fd_fifo;
  
  /* FD ALLOCATION */
  reg  [ 1:0]  fd_memreq_alloc;
  reg          fd_memreq_ready;
  reg  [ 2:0]  fd_memreq_lo_sel,fd_memreq_hi_sel;
  reg          fd_memreq_offset;
  
  /* FD ISSUE */
  reg          fd_assigned;
  reg          fd_allocated;
  reg  [ 2:0]  fd_prev_sel;
  reg  [ 1:0]  fd_alloc;
  reg          fd_heltf_odd;
  reg          fd_last;
  
  /* FFT FIFO */
  wire         fft_fifo_full,fft_fifo_empty;
  wire         fft_fifo_wren, fft_fifo_rden;
  wire [  2:0] fft_fifo_in_sh;
  wire [  3:0] n_fft_fifo_count;
  wire [ 14:0] n_fft_fifo_in;
  wire [119:0] n_fft_fifo_sh1,n_fft_fifo_sh2,n_fft_fifo_sh4,n_fft_fifo_out,n_fft_fifo;
  reg  [  3:0] fft_fifo_count;
  reg  [119:0] fft_fifo;
  
  /* FFT ISSUE */
  reg          fft_assigned;
  reg          fft_releasable;
  reg          fft_last;
  
  /* TD FIFO */
  wire         td_fifo_full,td_fifo_empty;
  wire         td_fifo_wren, td_fifo_rden;
  wire [  2:0] td_fifo_in_sh;
  wire [  3:0] n_td_fifo_count;
  wire [ 14:0] n_td_fifo_in;
  wire [119:0] n_td_fifo_sh1,n_td_fifo_sh2,n_td_fifo_sh4,n_td_fifo_out,n_td_fifo;
  reg  [  3:0] td_fifo_count;
  reg  [119:0] td_fifo;
  
  /* TD ISSUE */
  reg          td_assigned;
  reg          td_releasable;
  
  /* SCOREBOARD1 FD/FFT/TD MEM */
  reg          scr_fd0,scr_fft0,scr_td0;
  reg          scr_fd1,scr_fft1,scr_td1;
  reg          scr_fd2,scr_fft2,scr_td2;
  reg          scr_fd3,scr_fft3,scr_td3;
  reg          scr_fd4,scr_fft4,scr_td4;
  
  wire         scr_lock0;
  wire         scr_lock1;
  wire         scr_lock2;
  wire         scr_lock3;
  wire         scr_lock4;
  
  reg          scr_free_lo_valid;
  reg  [ 2:0]  scr_free_lo;
  reg          scr_free_hi_valid;
  reg  [ 2:0]  scr_free_hi;

  /*****************************************************************************
  * CONSTANTS
  *****************************************************************************/
  always @(*)
  begin
    if(framep1_format==HT_MM || framep1_format==HT_GF)
    begin
      case(framep1_nsts)
        NSTS_1:  nhtdltf = 3'd1;
        NSTS_2:  nhtdltf = 3'd2;
        NSTS_3:  nhtdltf = 3'd4;
        default: nhtdltf = 3'd4;
      endcase
      case(framep1_ness)
        2'd0:    nhteltf = 3'd0;
        2'd1:    nhteltf = 3'd1;
        2'd2:    nhteltf = 3'd2;
        default: nhteltf = 3'd4;
      endcase
    end
    else
    begin
      nhtdltf = 3'd0;
      nhteltf = 3'd0;
    end
  end
  
  always @(*)
  begin
    if(framep1_format==VHT)
      case(framep1_nsts)
        NSTS_1:  nvhtltf = 4'd1;
        NSTS_2:  nvhtltf = 4'd2;
        NSTS_3:  nvhtltf = 4'd4;
        NSTS_4:  nvhtltf = 4'd4;
        NSTS_5:  nvhtltf = 4'd6;
        NSTS_6:  nvhtltf = 4'd6;
        NSTS_7:  nvhtltf = 4'd8;
        default: nvhtltf = 4'd8;
      endcase
    else
      nvhtltf = 4'd0;
  end

  always @(*)
  begin
    case(framep1_nheltf)
      3'd0:    nheltf = 4'd1;
      3'd1:    nheltf = 4'd2;
      3'd2:    nheltf = 4'd3;
      3'd3:    nheltf = 4'd4;
      3'd4:    nheltf = 4'd5;
      3'd5:    nheltf = 4'd6;
      3'd6:    nheltf = 4'd7;
      default: nheltf = 4'd8;
    endcase
  end
  
  assign mma = (!framep1_midamble)?5'd10:5'd20;
  
  /*****************************************************************************
  * SYMBOL GENERATION
  *****************************************************************************/
  assign symbol_ready      = ~bd_fifo_full & ~fd_fifo_full;
  
  assign count_sym_incr    = count_sym    + 16'd1;
  assign count_mma_incr    = count_mma    +  5'd1;
  assign count_nma_incr    = count_nma    +  9'd1;
  assign count_htdltf_incr = count_htdltf +  3'd1;
  assign count_hteltf_incr = count_hteltf +  3'd1;
  assign count_vhtltf_incr = count_vhtltf +  4'd1;
  assign count_heltf_incr  = count_heltf  +  4'd1;

  /* SYMBOL */
  always @(*)
  begin:b_symbol_decoder
  
    reg       v,l,d;
    reg [5:0] s;
    
    case(symbol)                                                                       
      IDLE:                                                                            
      begin                                                                            
        if(framep1_update)                                                             
        begin                                                                          
          if(framep1_format==HT_GF)                                                    
            {v,l,d,s} = {3'b000,HTGFSTF};                                                         
          else                                                                 
            {v,l,d,s} = {3'b000,LSTF};                                                            
        end
        else
        begin
          {v,l,d,s} = {3'b000,IDLE};                                                            
        end
      end                                                                              
      LSTF: {v,l,d,s} = {3'b100,LLTF};                                                            
      LLTF:
      begin
        case(framep1_format)
          HE_SU,HE_TB: {v,l,d,s} = {3'b100,HELSIG};  
          default:     {v,l,d,s} = {3'b100,LSIG};                                                                
        endcase
      end
      LSIG:                                                                            
      begin                                                                            
        case(framep1_format)                                                           
          NON_HT,NON_HT_DUP: {v,l,d,s} = {3'b101,LDATA};                                          
          HT_MM:             {v,l,d,s} = {3'b101,HTMMSIG1};                                       
          VHT:               {v,l,d,s} = {3'b101,VHTSIGA1};                                       
          default:           {v,l,d,s} = {3'b111,DONE};                                           
        endcase                                                                        
      end 
      HTGFSTF:  {v,l,d,s} = {3'b100,HTGFLTF};                                                     
      HTGFLTF:  {v,l,d,s} = {3'b100,HTGFSIG1};                                                    
      HTGFSIG1: {v,l,d,s} = {3'b101,HTGFSIG2};                                                    
      HTGFSIG2:                                                                        
      begin                                                                            
        if(nhtdltf!=3'd1)                                                              
          {v,l,d,s} = {3'b101,HTDLTF};                                                            
        else if(nhteltf!=3'd0)                                                         
          {v,l,d,s} = {3'b101,HTELTF};                                                            
        else if(!framep2_update)
          {v,l,d,s} = {3'b000,symbol};                                                              
        else if(framep2_nsym!=16'd0)                                                   
          {v,l,d,s} = {3'b101,HTDATA};                                                       
        else                                                                           
          {v,l,d,s} = {3'b111,DONE};                                                              
      end                                                                              
      HTMMSIG1: {v,l,d,s} = {3'b101,HTMMSIG2};                                                    
      HTMMSIG2: {v,l,d,s} = {3'b101,HTMMSTF};    
      HTMMSTF:  {v,l,d,s} = {3'b100,HTDLTF};                                                      
      HTDLTF:                                                                          
      begin                                                                            
        if(n_count_htdltf!=nhtdltf)                                                    
          {v,l,d,s} = {3'b100,HTDLTF};                                                      
        else if(nhteltf!=3'd0)                                                         
          {v,l,d,s} = {3'b100,HTELTF};                                                      
        else if(!framep2_update)
          {v,l,d,s} = {3'b000,symbol};                                                         
        else if(framep2_nsym!=16'd0)                                                   
          {v,l,d,s} = {3'b100,HTDATA};                                                         
        else                                                                           
          {v,l,d,s} = {3'b110,DONE};                                                              
      end                                                                              
      HTELTF:                                                                          
      begin                                                                            
        if(n_count_hteltf!=nhteltf)                                                     
          {v,l,d,s} = {3'b100,HTELTF};                                                      
        else if(!framep2_update)
          {v,l,d,s} = {3'b000,symbol};                                                         
        else if(framep2_nsym!=16'd0)                                                   
          {v,l,d,s} = {3'b100,HTDATA};                                                         
        else                                                                           
          {v,l,d,s} = {3'b110,DONE};                                                              
      end                                                                              
      VHTSIGA1: {v,l,d,s} = {3'b101,VHTSIGA2};                                                    
      VHTSIGA2: {v,l,d,s} = {3'b101,VHTSTF};                                                      
      VHTSTF:   {v,l,d,s} = {3'b100,VHTLTF};                                                      
      VHTLTF:                                                                          
      begin                                                                            
        if(n_count_vhtltf!=nvhtltf)                                                    
          {v,l,d,s} = {3'b100,VHTLTF};                                                            
        else                                                                   
          {v,l,d,s} = {3'b100,VHTSIGB};                                                           
      end                                                                              
      VHTSIGB:                                                                         
      begin                                                                            
        if(!framep2_update)
          {v,l,d,s} = {3'b000,symbol};                                                              
        else if(framep2_nsym==16'd0)                                                        
          {v,l,d,s} = {3'b111,DONE};                                                              
        else                                                                   
          {v,l,d,s} = {3'b101,VHTDATA};                                                           
      end                                                                              
      HELSIG:  {v,l,d,s} = {3'b101,HESIGA1};                                                      
      HESIGA1: {v,l,d,s} = {3'b101,HESIGA2};                                                      
      HESIGA2: {v,l,d,s} = {3'b101,HESTF};                                                        
      HESIGB:  {v,l,d,s} = {3'b111,DONE};
      HESTF:   {v,l,d,s} = {3'b100,HELTF};                                                        
      HELTF:                                                                           
      begin                                                                            
        if(n_count_heltf!=nheltf)                                                      
          {v,l,d,s} = {3'b100,HELTF};                                                             
        else if(!framep2_update) 
          {v,l,d,s} = {3'b000,symbol};                                                              
        else if(framep2_nsym!=16'd0)
          {v,l,d,s} = {3'b100,HEDATA};                                                            
        else
          {v,l,d,s} = {3'b110,DONE};    
      end                                                                              
      HEDATA:                                                                          
      begin                                                                            
        if(!framep2_update)
          {v,l,d,s} = {3'b000,symbol};                                                        
        else if(framep1_doppler && n_count_mma==mma && count_nma!=framep2_nma)              
          {v,l,d,s} = {3'b101,HELTF};                                                        
        else if(n_count_sym==framep2_nsym && !framep1_continuoustx)  
        begin                                                                          
          if(framep2_tpe==3'd0)                                                        
            {v,l,d,s} = {3'b111,DONE};                                                            
          else                                                                         
            {v,l,d,s} = {3'b101,HEPE};                                                            
        end
        else
          {v,l,d,s} = {3'b101,HEDATA};                                                        
      end                                                                              
      HEPE: {v,l,d,s} = {3'b110,DONE};                                                            
      LDATA,VHTDATA,HTDATA:                                                            
      begin                                                                            
        if(!framep2_update)
          {v,l,d,s} = {3'b000,symbol};
        else if(n_count_sym==framep2_nsym)                         
          {v,l,d,s} = {3'b111,DONE};
        else
          {v,l,d,s} = {3'b101,symbol};
      end                                                                              
      default: {v,l,d,s} = {3'b000,DONE};
    endcase
    {symbol_valid,symbol_last,symbol_data,n_symbol} = {v,l,d,s};
  end
  
  /* COUNTERS */
  always @(*)
  begin
    n_count_sym    = count_sym;
    n_count_mma    = count_mma;
    n_count_nma    = count_nma;
    n_count_htdltf = count_htdltf;
    n_count_hteltf = count_hteltf;
    n_count_vhtltf = count_vhtltf;
    n_count_heltf  = count_heltf;
    case(symbol)
      HTGFLTF:  n_count_htdltf = 3'd1;
      HTDLTF:   n_count_htdltf = count_htdltf_incr;
      HTELTF:   n_count_hteltf = count_hteltf_incr;
      VHTLTF:   n_count_vhtltf = count_vhtltf_incr;
      HELTF:    
      begin
        n_count_mma    = 9'd0;
        n_count_heltf  = count_heltf_incr;
      end
      HEDATA:   
      begin
        n_count_heltf  = 4'd0;
        n_count_sym    = count_sym_incr;
        if(framep1_doppler)
          n_count_mma = count_mma_incr;
        if(n_symbol==HELTF && preamble_done)
          n_count_nma = count_nma_incr;
      end
      LDATA,HTDATA,VHTDATA: n_count_sym = count_sym_incr;
      default: ;
    endcase
  end

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      preamble_done <= 1'b0;
      count_sym     <= 16'd0; 
      count_mma     <= 5'd0;  
      count_nma     <= 10'd0; 
      count_htdltf  <= 3'd0;  
      count_hteltf  <= 3'd0;  
      count_vhtltf  <= 4'd0;  
      count_heltf   <= 4'd0; 
      symbol        <= IDLE;
    end
    else if(!enable)
    begin
      preamble_done <= 1'b0;
      count_sym     <= 16'd0; 
      count_mma     <= 5'd0;  
      count_nma     <= 10'd0; 
      count_htdltf  <= 3'd0;  
      count_hteltf  <= 3'd0;  
      count_vhtltf  <= 4'd0;  
      count_heltf   <= 4'd0; 
      symbol        <= IDLE;
    end
    else
    begin
      if(symbol_ready)
      begin
        if(symbol_valid)
        begin
          preamble_done   <= n_symbol==HEDATA;
          count_sym       <= n_count_sym;    
          count_mma       <= n_count_mma;    
          count_nma       <= n_count_nma;    
          count_htdltf    <= n_count_htdltf; 
          count_hteltf    <= n_count_hteltf; 
          count_vhtltf    <= n_count_vhtltf; 
          count_heltf     <= n_count_heltf;
        end  
        symbol          <= n_symbol;
      end
    end
  end
  
  /*****************************************************************************
  * BD FIFO  8 x 7 {0,symbol[5:0]}
  *****************************************************************************/
  assign bd_fifo_full    = bd_fifo_count==4'd8;
  assign bd_fifo_empty   = bd_fifo_count==4'd0;
  assign bd_fifo_wren    = symbol_ready & symbol_valid & symbol_data;
  assign bd_fifo_rden    = ~bd_fifo_empty & ~bd_assigned;
  assign bd_fifo_in_sh   = bd_fifo_count[2:0] - {2'b0,bd_fifo_rden};
  assign n_bd_fifo_count = bd_fifo_count - {3'b0,bd_fifo_rden} + {3'b0,bd_fifo_wren};
  
  assign n_bd_fifo_in    = {symbol_last,symbol} & {7{bd_fifo_wren}};
  assign n_bd_fifo_sh1   = bd_fifo_in_sh[0]?{ 42'b0,n_bd_fifo_in, 7'b0}:{49'b0,n_bd_fifo_in};
  assign n_bd_fifo_sh2   = bd_fifo_in_sh[1]?{n_bd_fifo_sh1[41:0],14'b0}:n_bd_fifo_sh1;
  assign n_bd_fifo_sh4   = bd_fifo_in_sh[2]?{n_bd_fifo_sh2[27:0],28'b0}:n_bd_fifo_sh2;
  assign n_bd_fifo_out   = bd_fifo_rden?{7'b0,bd_fifo[55:7]}:bd_fifo;
  assign n_bd_fifo       = n_bd_fifo_out | n_bd_fifo_sh4;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      bd_fifo_count <= 4'd0;
      bd_fifo       <= 56'd0;
    end
    else if(!enable)
    begin
      bd_fifo_count <= 4'd0;
      bd_fifo       <= 56'd0;
    end
    else
    begin
      bd_fifo_count <= n_bd_fifo_count;
      bd_fifo       <= n_bd_fifo;
    end
  end
 
  /*****************************************************************************
  * BD ISSUE
  *****************************************************************************/
  assign bd_valid = bd_assigned;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      bd_assigned <= 1'b0;
      bd_symbol   <= IDLE;
      bd_last     <= 1'b0;
    end
    else if(!enable)
    begin
      bd_assigned <= 1'b0;
      bd_symbol   <= IDLE;
      bd_last     <= 1'b0;
    end
    else
    begin
      if(!bd_assigned)
      begin
        if(bd_fifo_rden)
        begin
          bd_assigned <= 1'b1;
          bd_symbol   <= bd_fifo[5:0];
          bd_last     <= bd_fifo[6];
        end
      end
      else
      begin
        if(bd_done)
        begin
          bd_assigned  <= 1'b0;
        end
      end
    end
  end
  
  /*****************************************************************************
  * FD FIFO  8 x 9 {last,heltf_odd,0,symbol[5:0]}
  *****************************************************************************/
  assign fd_fifo_full    = fd_fifo_count==4'd8;
  assign fd_fifo_empty   = fd_fifo_count==4'd0;
  assign fd_fifo_wren    = symbol_ready & symbol_valid;
  assign fd_fifo_rden    = ~fd_fifo_empty & ~fd_assigned;
  assign fd_fifo_in_sh   = fd_fifo_count[2:0] - {2'b0,fd_fifo_rden};
  assign n_fd_fifo_count = fd_fifo_count - {3'b0,fd_fifo_rden} + {3'b0,fd_fifo_wren};
  
  assign n_fd_fifo_in    = {symbol_last,count_heltf[0],1'b0,symbol} & {9{fd_fifo_wren}};
  assign n_fd_fifo_sh1   = fd_fifo_in_sh[0]?{54'b0, n_fd_fifo_in, 9'b0}:{63'b0,n_fd_fifo_in};
  assign n_fd_fifo_sh2   = fd_fifo_in_sh[1]?{n_fd_fifo_sh1[53:0],18'b0}:n_fd_fifo_sh1;
  assign n_fd_fifo_sh4   = fd_fifo_in_sh[2]?{n_fd_fifo_sh2[35:0],36'b0}:n_fd_fifo_sh2;
 
  assign n_fd_fifo_out   = fd_fifo_rden?{9'b0,fd_fifo[71:9]}:fd_fifo;
 
  assign n_fd_fifo       = n_fd_fifo_out | n_fd_fifo_sh4;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      fd_fifo_count <= 4'd0;
      fd_fifo       <= 72'd0;
    end
    else if(!enable)
    begin
      fd_fifo_count <= 4'd0;
      fd_fifo       <= 72'd0;
    end
    else
    begin
      fd_fifo_count <= n_fd_fifo_count;
      fd_fifo       <= n_fd_fifo;
    end
  end
  
  /*****************************************************************************
  * FD ISSUE
  *****************************************************************************/
  /* ALLOCATION 
  *
  * ALLOC_ONE     the symbol requires 1 new memory
  * ALLOC_LOWER   the symbol requires 1 new memory, only lower half used, not release by TD
  * ALLOC_UPPER   the symbol requires the upper half memory of previous allocation, full memory released by TD
  * ALLOC_TWO     the symbol requires 2 new memories
  *
  */
  localparam ALLOC_ONE   = 2'd0,
             ALLOC_LOWER = 2'd1,
             ALLOC_UPPER = 2'd2,
             ALLOC_TWO   = 2'd3;
  
  
  always @(*)
  begin
    case(fd_symbol)
      /* STF */
      LSTF:     fd_memreq_alloc = ALLOC_ONE;
      HTGFSTF:  fd_memreq_alloc = ALLOC_ONE;
      HTMMSTF:  fd_memreq_alloc = ALLOC_ONE;
      VHTSTF:   fd_memreq_alloc = ALLOC_ONE;
      HESTF:    fd_memreq_alloc = ALLOC_ONE;
      /* LTF */
      LLTF:     fd_memreq_alloc = ALLOC_LOWER;
      HTGFLTF:  fd_memreq_alloc = ALLOC_LOWER;
      HTDLTF:   fd_memreq_alloc = ALLOC_ONE;
      HTELTF:   fd_memreq_alloc = ALLOC_ONE;
      VHTLTF:   fd_memreq_alloc = ALLOC_ONE;
      HELTF:
      begin    
        case(framep1_heltf_type)
          2'd0: 
            if(framep1_nheltf==3'd0)
              fd_memreq_alloc = ALLOC_ONE;
            else if(!fd_heltf_odd)
              fd_memreq_alloc = ALLOC_LOWER;
            else
              fd_memreq_alloc = ALLOC_UPPER;
          2'd1:    fd_memreq_alloc = ALLOC_ONE;
          default: fd_memreq_alloc = ALLOC_TWO;
        endcase
      end
      /* SIG */
      LSIG:     fd_memreq_alloc = ALLOC_UPPER;
      HELSIG:   fd_memreq_alloc = ALLOC_UPPER;
      HTGFSIG1: fd_memreq_alloc = ALLOC_UPPER;
      HTGFSIG2: fd_memreq_alloc = ALLOC_ONE;
      HTMMSIG1: fd_memreq_alloc = ALLOC_LOWER;
      HTMMSIG2: fd_memreq_alloc = ALLOC_UPPER;
      VHTSIGA1: fd_memreq_alloc = ALLOC_LOWER;
      VHTSIGA2: fd_memreq_alloc = ALLOC_UPPER;
      VHTSIGB:  fd_memreq_alloc = ALLOC_ONE;
      HESIGA1:  fd_memreq_alloc = ALLOC_LOWER;
      HESIGA2:  fd_memreq_alloc = ALLOC_UPPER;
      HESIGB:   fd_memreq_alloc = ALLOC_ONE;
      /*DATA */
      LDATA:    fd_memreq_alloc = ALLOC_ONE;
      HTDATA:   fd_memreq_alloc = ALLOC_ONE;
      VHTDATA:  fd_memreq_alloc = ALLOC_ONE;
      HEDATA:
      begin
        case(mdmcfg_conf_bw)
          2'd0:    fd_memreq_alloc = ALLOC_ONE;
          default: fd_memreq_alloc = ALLOC_TWO;
        endcase
      end
      /* HEPE */
      HEPE:     fd_memreq_alloc = ALLOC_ONE;
      default:  fd_memreq_alloc = ALLOC_ONE;
    endcase
  end
  
  always @(*)
  begin
    case(fd_memreq_alloc)
    
      ALLOC_UPPER:
      begin
        /* half buffer can be written if neither used by fd nor fft */
        case(fd_prev_sel)
          3'd0:    fd_memreq_ready = ~scr_fd0 & ~scr_fft0; 
          3'd1:    fd_memreq_ready = ~scr_fd1 & ~scr_fft1; 
          3'd2:    fd_memreq_ready = ~scr_fd2 & ~scr_fft2; 
          3'd3:    fd_memreq_ready = ~scr_fd3 & ~scr_fft3; 
          default: fd_memreq_ready = ~scr_fd4 & ~scr_fft4; 
        endcase
        fd_memreq_lo_sel = fd_prev_sel;
        fd_memreq_hi_sel = 3'd7;
        fd_memreq_offset = 1'b1; 
      end
      
      ALLOC_ONE,ALLOC_LOWER:
      begin
        if(scr_free_lo_valid)
        begin
          fd_memreq_ready  = 1'b1;
          fd_memreq_lo_sel = scr_free_lo;
          fd_memreq_hi_sel = 3'd7;
          fd_memreq_offset = 1'b0; 
        end
        else
        begin
          fd_memreq_ready  = 1'b0;
          fd_memreq_lo_sel = 3'd7;
          fd_memreq_hi_sel = 3'd7;
          fd_memreq_offset = 1'b0; 
        end
      end
      
      default: /* ALLOC_TWO */
      begin
        if(scr_free_lo_valid && scr_free_hi_valid)
        begin
          fd_memreq_ready  = 1'b1;
          fd_memreq_lo_sel = scr_free_lo;
          fd_memreq_hi_sel = scr_free_hi;
          fd_memreq_offset = 1'b0; 
        end
        else
        begin
          fd_memreq_ready  = 1'b0;
          fd_memreq_lo_sel = 3'd7;
          fd_memreq_hi_sel = 3'd7;
          fd_memreq_offset = 1'b0; 
        end
      end
    endcase
  end
  
  /* FD ISSUE SLOT 
  *
  *  fd_assigned,fd_allocated
  *
  *  0,0  : slot is free and ready to access a symbol from fifo
  *  1,0  : slot contains a symbol but memories not yet available
  *  1,1  : slot contains a symbol with allocated memories
  *  0,1  : illegal
  */
  assign fd_valid = fd_allocated;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      fd_assigned  <= 1'b0;
      fd_allocated <= 1'b0;
      fd_symbol    <= IDLE;
      fd_lo_sel    <= 3'd0;
      fd_hi_sel    <= 3'd0;
      fd_offset    <= 1'b0;
      fd_alloc     <= 2'b0;
      fd_prev_sel  <= 3'd0;
      fd_heltf_odd <= 1'b0;
      fd_last      <= 1'b0;
      
      scr_fd0      <= 1'b0;
      scr_fd1      <= 1'b0;
      scr_fd2      <= 1'b0;
      scr_fd3      <= 1'b0;
      scr_fd4      <= 1'b0;
    end
    else if(!enable)
    begin
      fd_assigned  <= 1'b0;
      fd_allocated <= 1'b0;
      fd_symbol    <= IDLE;
      fd_lo_sel    <= 3'd0;
      fd_hi_sel    <= 3'd0;
      fd_offset    <= 1'b0;
      fd_alloc     <= 2'b0;
      fd_prev_sel  <= 3'd0;
      fd_heltf_odd <= 1'b0;
      fd_last      <= 1'b0;
 
      scr_fd0      <= 1'b0;
      scr_fd1      <= 1'b0;
      scr_fd2      <= 1'b0;
      scr_fd3      <= 1'b0;
      scr_fd4      <= 1'b0;
    end
    else
    begin
      if(!fd_assigned)
      begin
        if(fd_fifo_rden)
        begin
          fd_symbol    <= fd_fifo[5:0];
          fd_heltf_odd <= fd_fifo[7];
          fd_last      <= fd_fifo[8];
          fd_assigned  <= 1'b1;
        end
      end
      else
      begin
        if(!fd_allocated)
        begin
          if(fd_memreq_ready && !fft_fifo_full)
          begin
            fd_allocated <= 1'b1;
            fd_alloc     <= fd_memreq_alloc;
            fd_lo_sel    <= fd_memreq_lo_sel;
            fd_hi_sel    <= fd_memreq_hi_sel;
            fd_offset    <= fd_memreq_offset;
            case(fd_memreq_lo_sel)
              3'd0:    scr_fd0 <= 1'b1;
              3'd1:    scr_fd1 <= 1'b1;
              3'd2:    scr_fd2 <= 1'b1;
              3'd3:    scr_fd3 <= 1'b1;
              3'd4:    scr_fd4 <= 1'b1;
              default: ;
            endcase
            case(fd_memreq_hi_sel)
              3'd0:    scr_fd0 <= 1'b1;
              3'd1:    scr_fd1 <= 1'b1;
              3'd2:    scr_fd2 <= 1'b1;
              3'd3:    scr_fd3 <= 1'b1;
              3'd4:    scr_fd4 <= 1'b1;
              default: ;
            endcase
          end
        end
        else
        begin
          if(fd_done)
          begin
            fd_assigned  <= 1'b0;
            fd_allocated <= 1'b0;
            fd_prev_sel  <= fd_lo_sel;
            case(fd_lo_sel)
              3'd0:    scr_fd0 <= 1'b0;
              3'd1:    scr_fd1 <= 1'b0;
              3'd2:    scr_fd2 <= 1'b0;
              3'd3:    scr_fd3 <= 1'b0;
              3'd4:    scr_fd4 <= 1'b0;
              default: ;
            endcase
            case(fd_hi_sel)
              3'd0:    scr_fd0 <= 1'b0;
              3'd1:    scr_fd1 <= 1'b0;
              3'd2:    scr_fd2 <= 1'b0;
              3'd3:    scr_fd3 <= 1'b0;
              3'd4:    scr_fd4 <= 1'b0;
              default: ;
            endcase
          end
        end
      end
    end
  end
  
  /*****************************************************************************
  * FFT FIFO 8 x 15 {last,releasable,hi_sel[2:0],lo_sel[2:0],offset,symbol[5:0]}
  *****************************************************************************/
  assign fft_fifo_full    = fft_fifo_count==4'd8;
  assign fft_fifo_empty   = fft_fifo_count==4'd0;
  assign fft_fifo_wren    = fd_valid && fd_done;
  assign fft_fifo_rden    = ~fft_fifo_empty & ~fft_assigned;
  assign fft_fifo_in_sh   = fft_fifo_count[2:0] - {2'b0,fft_fifo_rden};
  assign n_fft_fifo_count = fft_fifo_count - {3'b0,fft_fifo_rden} + {3'b0,fft_fifo_wren};
  
  assign n_fft_fifo_in    = {fd_last,fd_alloc!=ALLOC_LOWER,fd_hi_sel,fd_lo_sel,fd_offset,fd_symbol} & {15{fft_fifo_wren}};
  assign n_fft_fifo_sh1   = fft_fifo_in_sh[0]?{90'b0, n_fft_fifo_in,15'b0}:{105'b0,n_fft_fifo_in};
  assign n_fft_fifo_sh2   = fft_fifo_in_sh[1]?{n_fft_fifo_sh1[89:0],30'b0}:n_fft_fifo_sh1;
  assign n_fft_fifo_sh4   = fft_fifo_in_sh[2]?{n_fft_fifo_sh2[59:0],60'b0}:n_fft_fifo_sh2;
 
  assign n_fft_fifo_out   = fft_fifo_rden?{15'b0,fft_fifo[119:15]}:fft_fifo;
 
  assign n_fft_fifo       = n_fft_fifo_out | n_fft_fifo_sh4;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      fft_fifo_count <= 4'd0;
      fft_fifo       <= 120'd0;
    end
    else if(!enable)
    begin
      fft_fifo_count <= 4'd0;
      fft_fifo       <= 120'd0;
    end
    else
    begin
      fft_fifo_count <= n_fft_fifo_count;
      fft_fifo       <= n_fft_fifo;
    end
  end
  
  /*****************************************************************************
  * FFT ISSUE
  *****************************************************************************/
  assign fft_valid = fft_assigned;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      fft_assigned   <= 1'b0;
      fft_symbol     <= IDLE;  
      fft_lo_sel     <= 3'd0;  
      fft_hi_sel     <= 3'd0;  
      fft_offset     <= 1'b0;  
      fft_last       <= 1'b0;  
      fft_releasable <= 1'b0;
      scr_fft0       <= 1'b0;
      scr_fft1       <= 1'b0;
      scr_fft2       <= 1'b0;
      scr_fft3       <= 1'b0;
      scr_fft4       <= 1'b0;
    end
    else if(!enable)
    begin
      fft_assigned   <= 1'b0;
      fft_symbol     <= IDLE;  
      fft_lo_sel     <= 3'd0;  
      fft_hi_sel     <= 3'd0;  
      fft_offset     <= 1'b0;  
      fft_last       <= 1'b0;  
      fft_releasable <= 1'b0;
      scr_fft0       <= 1'b0;
      scr_fft1       <= 1'b0;
      scr_fft2       <= 1'b0;
      scr_fft3       <= 1'b0;
      scr_fft4       <= 1'b0;
    end
    else
    begin
      if(!fft_assigned)
      begin
        if(fft_fifo_rden)
        begin
          fft_assigned   <= 1'b1;
          fft_last       <= fft_fifo[14]; 
          fft_releasable <= fft_fifo[13]; 
          fft_hi_sel     <= fft_fifo[12:10]; 
          fft_lo_sel     <= fft_fifo[9:7]; 
          fft_offset     <= fft_fifo[6]; 
          fft_symbol     <= fft_fifo[5:0];
        end
      end
      else
      begin
        if(fft_done)
        begin
          fft_assigned <= 1'b0;
          case(fft_lo_sel)
            3'd0:    scr_fft0 <= 1'b0;
            3'd1:    scr_fft1 <= 1'b0;
            3'd2:    scr_fft2 <= 1'b0;
            3'd3:    scr_fft3 <= 1'b0;
            3'd4:    scr_fft4 <= 1'b0;
            default: ;
          endcase
          case(fft_hi_sel)
            3'd0:    scr_fft0 <= 1'b0;
            3'd1:    scr_fft1 <= 1'b0;
            3'd2:    scr_fft2 <= 1'b0;
            3'd3:    scr_fft3 <= 1'b0;
            3'd4:    scr_fft4 <= 1'b0;
            default: ;
          endcase
        end
      end
      
      if(fd_allocated && fd_done)
      begin
        case(fd_lo_sel)
          3'd0:    scr_fft0 <= 1'b1;
          3'd1:    scr_fft1 <= 1'b1;
          3'd2:    scr_fft2 <= 1'b1;
          3'd3:    scr_fft3 <= 1'b1;
          3'd4:    scr_fft4 <= 1'b1;
          default: ;
        endcase
        case(fd_hi_sel)
          3'd0:    scr_fft0 <= 1'b1;
          3'd1:    scr_fft1 <= 1'b1;
          3'd2:    scr_fft2 <= 1'b1;
          3'd3:    scr_fft3 <= 1'b1;
          3'd4:    scr_fft4 <= 1'b1;
          default: ;
        endcase
      end
    end
  end
  
  /*****************************************************************************
  * TD FIFO 8 x 15 {last,releasable,hi_sel[2:0],lo_sel[2:0],offset,symbol[5:0]}
  *****************************************************************************/
  assign td_fifo_full    = td_fifo_count==4'd8;
  assign td_fifo_empty   = td_fifo_count==4'd0;
  assign td_fifo_wren    = fft_assigned && fft_done;
  assign td_fifo_rden    = ~td_fifo_empty & ~td_assigned;
  assign td_fifo_in_sh   = td_fifo_count[2:0] - {2'b0,td_fifo_rden};
  assign n_td_fifo_count = td_fifo_count - {3'b0,td_fifo_rden} + {3'b0,td_fifo_wren};
  
  assign n_td_fifo_in    = {fft_last,fft_releasable,fft_hi_sel,fft_lo_sel,fft_offset,fft_symbol} & {15{td_fifo_wren}};
  assign n_td_fifo_sh1   = td_fifo_in_sh[0]?{90'b0, n_td_fifo_in,15'b0}:{105'b0,n_td_fifo_in};
  assign n_td_fifo_sh2   = td_fifo_in_sh[1]?{n_td_fifo_sh1[89:0],30'b0}:n_td_fifo_sh1;
  assign n_td_fifo_sh4   = td_fifo_in_sh[2]?{n_td_fifo_sh2[59:0],60'b0}:n_td_fifo_sh2;
 
  assign n_td_fifo_out   = td_fifo_rden?{15'b0,td_fifo[119:15]}:td_fifo;
 
  assign n_td_fifo       = n_td_fifo_out | n_td_fifo_sh4;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      td_fifo_count <= 4'd0;
      td_fifo       <= 120'd0;
    end
    else if(!enable)
    begin
      td_fifo_count <= 4'd0;
      td_fifo       <= 120'd0;
    end
    else
    begin
      td_fifo_count <= n_td_fifo_count;
      td_fifo       <= n_td_fifo;
    end
  end
  
  /*****************************************************************************
  * TD ISSUE
  *****************************************************************************/
  assign td_valid = td_assigned;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      td_assigned   <= 1'b0;  
      td_symbol     <= IDLE;  
      td_lo_sel     <= 3'd0;  
      td_hi_sel     <= 3'd0;  
      td_offset     <= 1'b0;  
      td_last       <= 1'b0;  
      td_releasable <= 1'b0;
      scr_td0       <= 1'b0;
      scr_td1       <= 1'b0;
      scr_td2       <= 1'b0;
      scr_td3       <= 1'b0;
      scr_td4       <= 1'b0;
    end
    else if(!enable)
    begin
      td_assigned   <= 1'b0;  
      td_symbol     <= IDLE;  
      td_lo_sel     <= 3'd0;  
      td_hi_sel     <= 3'd0;  
      td_offset     <= 1'b0;  
      td_last       <= 1'b0;  
      td_releasable <= 1'b0;
      scr_td0       <= 1'b0;
      scr_td1       <= 1'b0;
      scr_td2       <= 1'b0;
      scr_td3       <= 1'b0;
      scr_td4       <= 1'b0;
    end
    else
    begin
      if(!td_assigned)
      begin
        if(td_fifo_rden)
        begin
          td_assigned   <= 1'b1;
          td_last       <= td_fifo[14]; 
          td_releasable <= td_fifo[13]; 
          td_hi_sel     <= td_fifo[12:10]; 
          td_lo_sel     <= td_fifo[9:7]; 
          td_offset     <= td_fifo[6]; 
          td_symbol     <= td_fifo[5:0];
        end
      end
      else
      begin
        if(td_done)
        begin
          td_assigned <= 1'b0;
          if(td_releasable)
          begin
            case(td_lo_sel)
              3'd0:    scr_td0 <= 1'b0;
              3'd1:    scr_td1 <= 1'b0;
              3'd2:    scr_td2 <= 1'b0;
              3'd3:    scr_td3 <= 1'b0;
              3'd4:    scr_td4 <= 1'b0;
              default: ;
            endcase
            case(td_hi_sel)
              3'd0:    scr_td0 <= 1'b0;
              3'd1:    scr_td1 <= 1'b0;
              3'd2:    scr_td2 <= 1'b0;
              3'd3:    scr_td3 <= 1'b0;
              3'd4:    scr_td4 <= 1'b0;
              default: ;
            endcase
          end
        end
      end
      
      if(fft_assigned && fft_done)
      begin
        case(fft_lo_sel)
          3'd0:    scr_td0 <= 1'b1;
          3'd1:    scr_td1 <= 1'b1;
          3'd2:    scr_td2 <= 1'b1;
          3'd3:    scr_td3 <= 1'b1;
          3'd4:    scr_td4 <= 1'b1;
          default: ;
        endcase
        case(fft_hi_sel)
          3'd0:    scr_td0 <= 1'b1;
          3'd1:    scr_td1 <= 1'b1;
          3'd2:    scr_td2 <= 1'b1;
          3'd3:    scr_td3 <= 1'b1;
          3'd4:    scr_td4 <= 1'b1;
          default: ;
        endcase
      end
    end
  end
  
  /*****************************************************************************
  * FD/FFT/TD MEMORY SCOREBOARD (applies to mem0 up to mem4)
  *****************************************************************************/
  /* free entries */
  assign scr_lock0 = scr_fd0 | scr_fft0 | scr_td0;
  assign scr_lock1 = scr_fd1 | scr_fft1 | scr_td1;
  assign scr_lock2 = scr_fd2 | scr_fft2 | scr_td2 | ~mem_released;
  assign scr_lock3 = scr_fd3 | scr_fft3 | scr_td3 | ~mem_released;
  assign scr_lock4 = scr_fd4 | scr_fft4 | scr_td4 | ~mem_released;
  
  always @(*)
  begin
    /* first free memory */
    if(!scr_lock0)
    begin
      scr_free_lo_valid = 1'b1;
      scr_free_lo       = 3'd0;
    end
    else if(!scr_lock1)
    begin
      scr_free_lo_valid = 1'b1;
      scr_free_lo       = 3'd1;
    end
    else if(!scr_lock2)
    begin
      scr_free_lo_valid = 1'b1;
      scr_free_lo       = 3'd2;
    end
    else if(!scr_lock3)
    begin
      scr_free_lo_valid = 1'b1;
      scr_free_lo       = 3'd3;
    end
    else if(!scr_lock4)
    begin
      scr_free_lo_valid = 1'b1;
      scr_free_lo       = 3'd4;
    end
    else
    begin
      scr_free_lo_valid = 1'b0;
      scr_free_lo       = 3'd0;
    end
    
    /* second free memory */
    if(!scr_lock0 && scr_free_lo!=3'd0)
    begin
      scr_free_hi_valid = 1'b1;
      scr_free_hi       = 3'd0;
    end
    else if(!scr_lock1 && scr_free_lo!=3'd1)
    begin
      scr_free_hi_valid = 1'b1;
      scr_free_hi       = 3'd1;
    end
    else if(!scr_lock2 && scr_free_lo!=3'd2)
    begin
      scr_free_hi_valid = 1'b1;
      scr_free_hi       = 3'd2;
    end
    else if(!scr_lock3 && scr_free_lo!=3'd3)
    begin
      scr_free_hi_valid = 1'b1;
      scr_free_hi       = 3'd3;
    end
    else if(!scr_lock4 && scr_free_lo!=3'd4)
    begin
      scr_free_hi_valid = 1'b1;
      scr_free_hi       = 3'd4;
    end
    else
    begin
      scr_free_hi_valid = 1'b0;
      scr_free_hi       = 3'd0;
    end
  end

  /*****************************************************************************
  * mem_released timer
  *****************************************************************************/
  reg [12:0]  mem_timer;
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      mem_released <= 1'b0;
      mem_timer    <= 13'b0;
    end
    else if(!framep1_update)
    begin
      mem_released <= 1'b0;
      //mem_timer    <= 13'd2760; /* 23 us after txreq */
      mem_timer    <= 13'd2400; /* 20 us after txreq */
    end
    else if(mem_timer!=13'd0)
    begin
      mem_timer    <= mem_timer - 13'd1;
    end
    else
    begin
      mem_released <= 1'b1;
    end
  end

`ifdef RW_SIMU_ON
  wire [32*8-1:0] str_symbol[0:63];
  assign str_symbol[IDLE]     = "IDLE";
  assign str_symbol[LSTF]     = "LSTF";
  assign str_symbol[LLTF]     = "LLTF";
  assign str_symbol[LSIG]     = "LSIG";
  assign str_symbol[LDATA]    = "LDATA";
  assign str_symbol[HTGFSTF]  = "HTGFSTF";
  assign str_symbol[HTGFLTF]  = "HTGFLTF";
  assign str_symbol[HTGFSIG1] = "HTGFSIG1";
  assign str_symbol[HTGFSIG2] = "HTGFSIG2";
  assign str_symbol[HTMMSIG1] = "HTMMSIG1";
  assign str_symbol[HTMMSIG2] = "HTMMSIG2";
  assign str_symbol[HTMMSTF]  = "HTMMSTF";
  assign str_symbol[HTDLTF]   = "HTDLTF";
  assign str_symbol[HTELTF]   = "HTELTF";
  assign str_symbol[HTDATA]   = "HTDATA";
  assign str_symbol[VHTSIGA1] = "VHTSIGA1";
  assign str_symbol[VHTSIGA2] = "VHTSIGA2";
  assign str_symbol[VHTSTF]   = "VHTSTF";
  assign str_symbol[VHTLTF]   = "VHTLTF";
  assign str_symbol[VHTSIGB]  = "VHTSIGB";
  assign str_symbol[VHTDATA]  = "VHTDATA";
  assign str_symbol[HELSIG]   = "HELSIG";
  assign str_symbol[HESIGA1]  = "HESIGA1";
  assign str_symbol[HESIGA2]  = "HESIGA2";
  assign str_symbol[HESIGB]   = "HESIGB";
  assign str_symbol[HESTF]    = "HESTF";
  assign str_symbol[HELTF]    = "HELTF";
  assign str_symbol[HEDATA]   = "HEDATA";
  assign str_symbol[HEPE]     = "HEPE";
  assign str_symbol[DONE]     = "DONE";
 
  wire [32*8-1:0] s_symbol,s_bd_symbol,s_fd_symbol,s_fft_symbol,s_td_symbol;
  assign s_symbol     = str_symbol[symbol];
  assign s_bd_symbol  = str_symbol[bd_symbol];
  assign s_fd_symbol  = str_symbol[fd_symbol];
  assign s_fft_symbol = str_symbol[fft_symbol];
  assign s_td_symbol  = str_symbol[td_symbol];
 
`endif  
  
endmodule
`default_nettype wire
