/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module tx_td_windowing
(
  /* system */         
  input  wire         clk,
  input  wire         rst_n,
  
  /* phy config  */
  input  wire [ 1:0]  mdmcfg_conf_bw,
  
  /* frame parameters */      
  input  wire         frame_trigbase,
  input  wire [ 1:0]  frame_gi_type,
  input  wire [ 1:0]  frame_heltf_type,
  input  wire [ 2:0]  frame_tpe,
  
  /* block control */
  input  wire         tx_enable,
  output wire         td_done,
  
  /* scheduler */
  input  wire         sched_valid,
  input  wire [ 5:0]  sched_symbol,
  input  wire         sched_last,
  output reg          sched_done,
  
  /* fft read ports */  
  output wire         fft0_ren,
  output wire [ 9:0]  fft0_rindex,           
  input  wire [12:0]  fft0_ri,           
  input  wire [12:0]  fft0_rq,           
                       
  /* frontend ports */  
  input  wire         out_ready,
  output wire         out0_enable,
  output wire [12:0]  out0_i,
  output wire [12:0]  out0_q,
  output wire         out_last,
  output wire         out_valid
  
);
  /*****************************************************************************
  * declaration
  *****************************************************************************/
  localparam  IDLE       = 6'd0,
              LSTF       = 6'd1,
              LLTF       = 6'd2,
              LSIG       = 6'd3,
              LDATA      = 6'd4,
              HTGFSTF    = 6'd5,
              HTGFLTF    = 6'd6,
              HTGFSIG1   = 6'd7,
              HTGFSIG2   = 6'd8,
              HTMMSIG1   = 6'd9,
              HTMMSIG2   = 6'd10,
              HTMMSTF    = 6'd11,
              HTDLTF     = 6'd12,
              HTELTF     = 6'd13,
              HTDATA     = 6'd14,
              VHTSIGA1   = 6'd15,
              VHTSIGA2   = 6'd16,
              VHTSTF     = 6'd17,
              VHTLTF     = 6'd18,
              VHTSIGB    = 6'd19,
              VHTDATA    = 6'd20,
              HELSIG     = 6'd21,
              HESIGA1    = 6'd22,
              HESIGA2    = 6'd23,
              HESIGB     = 6'd24,
              HESTF      = 6'd25,
              HELTF      = 6'd26,
              HEDATA     = 6'd27,
              HEPE       = 6'd28,
              DONE       = 6'd63;
 
  localparam  BW_20=2'd0,
              BW_40=2'd1,
              BW_80=2'd2,
              BW_160=2'd3;
 
  localparam  HELTF_1X=2'd0,
              HELTF_2X=2'd1,
              HELTF_4X=2'd2;
 
  localparam  S_WAIT_BUFFER=2'd0,
              S_READ_BUFFER=2'd1,
              S_WAIT_RELEASE=2'd2,
              S_DONE=2'd3;

  wire        ready;

  reg  [ 1:0] s0_state;
  reg  [ 5:0] s0_sym;
  reg         s0_valid;
  reg         s0_last;
  reg  [ 3:0] s0_rdptr;
  reg  [10:0] s0_count;
  reg         s0_prefix;
  reg         s0_replayable;
  
  reg  [ 9:0] s0_index0; 
  wire [10:0] s0_index0_incr; 
  reg  [10:0] s0_gi_offset20;
  reg  [10:0] s0_gi_offset40;
  reg  [10:0] s0_gi_offset;
  
  reg  [10:0] count_sample_max20;
  reg  [10:0] count_sample_max40;
  reg  [10:0] count_sample_max;

  reg         s1_prefix;
  reg         s1_suffix;
  reg         s1_last;
  reg         s1_valid;
  reg         s1_release;
  reg [ 9:0]  s1_index0;
  reg         s1_ren;
  
  reg         s2_prefix;
  reg         s2_suffix;
  reg         s2_last;
  reg         s2_valid;
  reg         s2_release;
  
  reg         s3_prefix;
  reg         s3_suffix;
  reg         s3_last;
  reg         s3_valid;
  
  reg         s4_prefix;
  reg         s4_suffix;
  reg         s4_last;
  reg         s4_valid;
  reg [12:0]  s4_i0,s4_q0;
  
  reg [144:0] s5_fifo;
  reg [ 2:0]  s5_fifo_count;
  wire        s5_prefix;
  wire        s5_suffix;
  wire        s5_last;
  wire        s5_valid;
  wire [12:0] s5_i0,s5_q0;
  
  wire [16:0] n_s6_x8_i0,   n_s6_x8_q0;
  wire [16:0] n_s6_x15_i0,  n_s6_x15_q0;
  
  reg  [16:0] n_s6_mult_i0, n_s6_mult_q0;
  wire [17:0] n_s6_add_i0,  n_s6_add_q0;
  wire [12:0] n_s6_win_i0,  n_s6_win_q0;
  
  reg         s6_en0;
  reg         s6_last;
  reg         s6_valid;
  reg         s6_done;
  reg  [ 2:0] s6_flush;
  reg  [16:0] s6_tap0_i0, s6_tap0_q0;
  reg  [16:0] s6_tap1_i0, s6_tap1_q0;
  reg  [16:0] s6_tap2_i0, s6_tap2_q0;
  reg  [12:0] s6_i0,s6_q0;
  reg         s6_prefix,s6_prefix_1t;
  reg         s6_suffix,s6_suffix_1t;
 
  /*****************************************************************************
  * STALL
  *****************************************************************************/
  assign ready = out_ready | ~out_valid;
  
  /*****************************************************************************
  * S0 stage (frame generation, sample counting)
  *****************************************************************************/

  /*****************************************************************************
  * symbol      gi (us)  duration (us)   fft         cycles (Ts)
  * 
  * NONHT20              4.0             128          160
  * HT20/VHT20 gi=0.8    4.0             128          160
  * HT20/VHT20 gi=0.4    3.6             128          144
  *
  * LSTF/HTSTF           8.0             128          320
  * LLTF/HTLTF           8.0             128          320
  *
  * HESTF                4.0             128          160
  *
  * HELTF2X    gi=0.8    7.2             256          288
  * HELTF2X    gi=1.6    8.0             256          320
  * HELTF4X    gi=3.2   16.0             512          640
  * HELTF1X    gi=1.6    4.8             128          192
  * HELTF1X    gi=0.8    4.0             128          160
  * HELTF4X    gi=0.8   13.6             512          544
  *
  * HEDATA     gi=0.8   13.6             512          544
  * HEDATA     gi=1.6   14.4             512          576
  * HEDATA     gi=3.2   16               512          640
  *
  * HEPE       gi=3.2    4               128          160
  * HEPE       gi=3.2    8               128          320
  * HEPE       gi=3.2   12               128          480 
  * HEPE       gi=3.2   16               128          640
  *
  *****************************************************************************/
  always @(*)  
  begin 
    case(s0_sym)
      
      LSTF,LLTF,HTGFSTF,HTGFLTF:
      begin
        count_sample_max20 = 11'd320;
      end
      
      HTDATA,VHTDATA:
      begin
        if(frame_gi_type[0]==1'b0)
          count_sample_max20 = 11'd160;
        else
          count_sample_max20 = 11'd144;
      end
      
      HESTF:
      begin
        if(frame_trigbase)
          count_sample_max20 = 11'd320;
        else
          count_sample_max20 = 11'd160;
      end
      
      HELTF:
      begin
        case({frame_heltf_type,frame_gi_type})
          {HELTF_2X,2'd0}: count_sample_max20 = 11'd288;
          {HELTF_2X,2'd1}: count_sample_max20 = 11'd320;
          {HELTF_4X,2'd2}: count_sample_max20 = 11'd640;
          {HELTF_1X,2'd1}: count_sample_max20 = 11'd192;
          {HELTF_1X,2'd0}: count_sample_max20 = 11'd160;
          default:         count_sample_max20 = 11'd544;
        endcase
      end
      
      HEDATA:
      begin
        case(frame_gi_type)
          2'd0:    count_sample_max20 = 11'd544; /* 12.8 + 0.8 us */
          2'd1:    count_sample_max20 = 11'd576; /* 12.8 + 1.6 us */
          default: count_sample_max20 = 11'd640; /* 12.8 + 3.2 us */  
        endcase
      end
      
      HEPE:
      begin
        case(frame_tpe)
          3'd1:    count_sample_max20 = 11'd160; /*  4 us */
          3'd2:    count_sample_max20 = 11'd320; /*  8 us */
          3'd3:    count_sample_max20 = 11'd480; /* 12 us */
          default: count_sample_max20 = 11'd640; /* 16 us */
        endcase
      end
      
      default:
      begin
        count_sample_max20 = 11'd160;
      end
      
    endcase
    
    count_sample_max40 =  {count_sample_max20[9:0],1'b0};
    
    case(mdmcfg_conf_bw)
      BW_20:   count_sample_max = count_sample_max20;
      default: count_sample_max = count_sample_max40;
    endcase
    
  end
  
  /* registers */
  wire [10:0] n_s0_count;
  assign n_s0_count = s0_count + 11'd1;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      sched_done     <= 1'b0;
      s0_valid       <= 1'b0;
      s0_rdptr       <= 4'b0;
      s0_count       <= 11'b0;
      s0_state       <= S_WAIT_BUFFER;
      s0_sym         <= IDLE;
      s0_prefix      <= 1'b0;
      s0_last        <= 1'b0;
      s0_replayable  <= 1'b0;
    end
    else if(!tx_enable)
    begin
      sched_done     <= 1'b0;
      s0_valid       <= 1'b0;
      s0_rdptr       <= 4'b0;
      s0_count       <= 11'b0;
      s0_state       <= S_WAIT_BUFFER;
      s0_sym         <= IDLE;
      s0_prefix      <= 1'b0;
      s0_last        <= 1'b0;
      s0_replayable  <= 1'b0;
    end
    else
    begin
      /* RTZ */
      sched_done   <= 1'b0;
      
      case(s0_state)
        /***********************************************************************
        * S_WAIT_BUFFER
        ***********************************************************************/
        S_WAIT_BUFFER:
        begin
          if(sched_valid && !sched_done)
          begin
            s0_state      <= S_READ_BUFFER;
            s0_valid      <= 1'b1;
            s0_count      <= 11'd0;
            s0_prefix     <= 1'b1; 
            s0_replayable <= sched_symbol==HELSIG;
            s0_sym        <= sched_symbol;
            s0_last       <= sched_last;
          end
        end
        
        /***********************************************************************
        * S_WAIT_BUFFER
        ***********************************************************************/
        S_READ_BUFFER:
        begin
          if(ready)
          begin
            s0_prefix <= 1'b0; 
            if(n_s0_count==count_sample_max)                                         
            begin                                                               
              /* symbol done */                                                 
              if(s0_replayable)
              begin
                /* replay HELSIG as HERLSIG */
                s0_count      <= 11'd0;
                s0_prefix     <= 1'b1;
                s0_replayable <= 1'b0;
              end
              else
              begin 
                s0_valid      <= 1'b0;                                                 
                s0_state      <= S_WAIT_RELEASE;
              end
            end                                                                 
            else                                                                
            begin                                                               
              s0_valid   <= 1'b1;                                                 
              s0_count   <= n_s0_count;                                           
            end
          end                                                                 
        end
        
        S_WAIT_RELEASE:
        begin
          if(s2_release)
          begin
            /* last sample read from mem so the mem can be released */
            sched_done <= 1'b1;
            if(s2_last)
              s0_state <= S_DONE;
            else
              s0_state <= S_WAIT_BUFFER;
          end
        end
        
        /* DONE */
        default: ;
      endcase
    end
  end
 
  /* guard interval */
  always @(*)
  begin
    case(s0_sym)
      /* long GI 800 ns */
      LSIG,LDATA,
      HTMMSIG1,HTMMSIG2,HTGFSIG1,HTGFSIG2,
      VHTSIGA1,VHTSIGA2,VHTSIGB,HTDLTF,HTELTF,VHTLTF,
      HELSIG,HESIGA1,HESIGA2,HESIGB:
        s0_gi_offset20 = 11'd32;
      
      /* short or long GI */
      HTDATA,VHTDATA:
        if(frame_gi_type[0]==1'b0)
          s0_gi_offset20 = 11'd32;
        else
          s0_gi_offset20 = 11'd16;
      
      HEDATA,HELTF:
      begin
        case(frame_gi_type)
          2'd0:    s0_gi_offset20 = 11'd32;
          2'd1:    s0_gi_offset20 = 11'd64;
          default: s0_gi_offset20 = 11'd128;
        endcase
      end
     
      /* LLTF,HTGFLTF                        GI2 1600 ns*/
      LLTF,HTGFLTF:  
        s0_gi_offset20 = 11'd64;

      /* no GI LLSTF,HTGFSTF,HTMMSTF,VHTSTF,HEPE */ 
      default:  
        s0_gi_offset20 = 11'd0;

    endcase 
    
    s0_gi_offset40 = {s0_gi_offset20[9:0],1'b0};
    
    case(mdmcfg_conf_bw)
      BW_20:   s0_gi_offset = s0_gi_offset20;
      default: s0_gi_offset = s0_gi_offset40;
    endcase
    
  end 
  
  /* fft address generation */
  assign s0_index0_incr = s0_count - s0_gi_offset;
 
  /* fft length address wrap */
  always @(*)
  begin
    case(mdmcfg_conf_bw)
      BW_20:
        case(s0_sym)
          HELTF:
            case(frame_heltf_type)
              HELTF_1X: s0_index0 = {3'b0,s0_index0_incr[6:0]};
              HELTF_2X: s0_index0 = {2'b0,s0_index0_incr[7:0]};
              default:  s0_index0 = {1'b0,s0_index0_incr[8:0]};
            endcase
          HEDATA:  s0_index0 = {1'b0,s0_index0_incr[8:0]};
          default: s0_index0 = {3'b0,s0_index0_incr[6:0]};
        endcase
      default:
        case(s0_sym)
          HELTF:
            case(frame_heltf_type)
              HELTF_1X: s0_index0 = {2'b0,s0_index0_incr[7:0]};
              HELTF_2X: s0_index0 = {1'b0,s0_index0_incr[8:0]};
              default:  s0_index0 = s0_index0_incr[9:0];
            endcase
          HEDATA:  s0_index0 = s0_index0_incr[9:0];
          default: s0_index0 = {2'b0,s0_index0_incr[7:0]};
        endcase
    endcase
  end

  /*****************************************************************************
  * S1 stage (fft sram address generation)
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1_prefix  <= 1'b0;
      s1_suffix  <= 1'b0;
      s1_last    <= 1'b0;
      s1_valid   <= 1'b0;
      s1_release <= 1'b0;
      s1_ren     <= 1'b0;
      s1_index0  <= 10'b0;
    end
    else if(!tx_enable)
    begin
      s1_prefix  <= 1'b0;
      s1_suffix  <= 1'b0;
      s1_last    <= 1'b0;
      s1_valid   <= 1'b0;
      s1_release <= 1'b0;
      s1_ren     <= 1'b0;
      s1_index0  <= 10'b0;
    end
    else
    begin
      if(ready)
      begin
        if(s0_state==S_READ_BUFFER)
        begin
          s1_valid   <= s0_valid; 
          s1_prefix  <= s0_prefix;
          s1_suffix  <= s0_index0==10'd0 && !s0_prefix;
          s1_last    <= s0_last && (n_s0_count==count_sample_max);
          s1_release <= !s0_replayable && n_s0_count==count_sample_max;
          s1_index0  <= s0_index0;
        end
        else
        begin
        s1_valid   <= 1'b0; 
        s1_prefix  <= 1'b0;
        s1_suffix  <= 1'b0;
        s1_last    <= 1'b0;
        s1_release <= 1'b0;
        
        
        end
      end
      else
      begin
        s1_valid   <= 1'b0; 
        s1_prefix  <= 1'b0;
        s1_suffix  <= 1'b0;
        s1_last    <= 1'b0;
        s1_release <= 1'b0;
      end
    end
  end

  /* fft address ports */
  assign fft0_ren    = s1_valid;
  assign fft0_rindex = s1_index0;

  /*****************************************************************************
  * S2 stage (fft sram access cycle)
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_prefix  <= 1'b0;
      s2_suffix  <= 1'b0;
      s2_last    <= 1'b0;
      s2_valid   <= 1'b0;
      s2_release <= 1'b0;
    end
    else if(!tx_enable)
    begin
      s2_prefix  <= 1'b0;
      s2_suffix  <= 1'b0;
      s2_last    <= 1'b0;
      s2_valid   <= 1'b0;
      s2_release <= 1'b0;
    end
    else
    begin
      s2_prefix  <= s1_prefix;
      s2_suffix  <= s1_suffix;
      s2_last    <= s1_last;
      s2_valid   <= s1_valid;
      s2_release <= s1_release;
    end
  end
  
  /*****************************************************************************
  * S3 stage (fft memmux latency)
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s3_prefix <= 1'b0;
      s3_suffix <= 1'b0;
      s3_last   <= 1'b0;
      s3_valid  <= 1'b0;
    end
    else if(!tx_enable)
    begin
      s3_prefix <= 1'b0;
      s3_suffix <= 1'b0;
      s3_last   <= 1'b0;
      s3_valid  <= 1'b0;
    end
    else
    begin
      s3_prefix <= s2_prefix;
      s3_suffix <= s2_suffix;
      s3_last   <= s2_last;
      s3_valid  <= s2_valid;
     end
   end

  /*****************************************************************************
  * S4 stage data capture
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s4_prefix     <= 1'b0;
      s4_suffix     <= 1'b0;
      s4_last       <= 1'b0;
      s4_valid      <= 1'b0;
      {s4_q0,s4_i0} <= 26'b0;
    end
    else if(!tx_enable)
    begin
      s4_prefix     <= 1'b0;
      s4_suffix     <= 1'b0;
      s4_last       <= 1'b0;
      s4_valid      <= 1'b0;
      {s4_q0,s4_i0} <= 26'b0;
    end
    else
    begin
      s4_prefix     <= s3_prefix;
      s4_suffix     <= s3_suffix;
      s4_last       <= s3_last;
      s4_valid      <= s3_valid;
      {s4_q0,s4_i0} <= {fft0_rq,fft0_ri};
    end
  end
  
  wire [28:0] s4_packed;
  assign s4_packed = {s4_prefix,s4_suffix,s4_last,s4_q0,s4_i0} & {29{s4_valid}};
  
  /*****************************************************************************
  * S5 stage stall buffer (5 entries of 29 bits)
  *****************************************************************************/
  wire [  2:0] n_s5_fifo_shsel,n_s5_fifo_count;
  wire [144:0] n_s5_fifo_sh1,n_s5_fifo_sh2,n_s5_fifo_sh4,n_s5_fifo_out,n_s5_fifo;
  
  assign n_s5_fifo_shsel = s5_fifo_count - {2'b0, s5_fifo_count!=3'd0 && ready};  
  assign n_s5_fifo_sh1   = n_s5_fifo_shsel[0]?{ 87'b0,    s4_packed,  29'b0}:{116'b0,s4_packed};
  assign n_s5_fifo_sh2   = n_s5_fifo_shsel[1]?{ n_s5_fifo_sh1[86:0],  58'b0}:n_s5_fifo_sh1;
  assign n_s5_fifo_sh4   = n_s5_fifo_shsel[2]?{ n_s5_fifo_sh2[28:0], 116'b0}:n_s5_fifo_sh2;
  
  assign n_s5_fifo_out   = (ready)?{29'b0,s5_fifo[144:29]}:s5_fifo;
  
  assign n_s5_fifo       = n_s5_fifo_out | n_s5_fifo_sh4;
  assign n_s5_fifo_count = n_s5_fifo_shsel + {2'b0,s4_valid};  
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s5_fifo_count <= 3'd0;
      s5_fifo       <= 145'b0;
    end
    else if(!tx_enable)
    begin
      s5_fifo_count <= 3'd0;
      s5_fifo       <= 145'b0;
    end
    else
    begin
      s5_fifo_count <= n_s5_fifo_count;
      s5_fifo       <= n_s5_fifo;
    end
  end
  
  assign s5_prefix     = s5_fifo[28];
  assign s5_suffix     = s5_fifo[27];
  assign s5_last       = s5_fifo[26];
  assign {s5_q0,s5_i0} = s5_fifo[25:0];
  assign s5_valid      = s5_fifo_count!=3'd0;
  
  /*****************************************************************************
  * S6 stage (windowing)
  *****************************************************************************/
  /* multiply by 1,4,7 */
  
  assign n_s6_x8_i0   = {{1{s5_i0[12]}},s5_i0, 3'b0};
  assign n_s6_x8_q0   = {{1{s5_q0[12]}},s5_q0, 3'b0};
 
  assign n_s6_x15_i0  = {s5_i0, 4'b0} - {{4{s5_i0[12]}}, s5_i0};
  assign n_s6_x15_q0  = {s5_q0, 4'b0} - {{4{s5_q0[12]}}, s5_q0};
     
  always @(*)
  begin
    if(s6_prefix || s6_suffix)
    begin
      /* x8 */
      n_s6_mult_i0 = n_s6_x8_i0;
      n_s6_mult_q0 = n_s6_x8_q0;
    end
    else if(s6_prefix_1t || s5_suffix)
    begin
      /* x15 */
      n_s6_mult_i0 = n_s6_x15_i0;
      n_s6_mult_q0 = n_s6_x15_q0;
    end
    else
    begin
      /* x1*/
      n_s6_mult_i0 = {{4{s5_i0[12]}},s5_i0};
      n_s6_mult_q0 = {{4{s5_q0[12]}},s5_q0};
    end
  end
  
  /* add */
  assign n_s6_add_i0  = {n_s6_mult_i0[16],n_s6_mult_i0} + {s6_tap2_i0[16],s6_tap2_i0};
  assign n_s6_add_q0  = {n_s6_mult_q0[16],n_s6_mult_q0} + {s6_tap2_q0[16],s6_tap2_q0};

  /* trunc */
  assign n_s6_win_i0  = n_s6_add_i0[16:4];
  assign n_s6_win_q0  = n_s6_add_q0[16:4];

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s6_en0       <= 1'b0;
      s6_last      <= 1'b0;
      s6_tap0_i0   <= 17'b0;
      s6_tap0_q0   <= 17'b0;
      s6_tap1_i0   <= 17'b0;
      s6_tap1_q0   <= 17'b0;
      s6_tap2_i0   <= 17'b0;
      s6_tap2_q0   <= 17'b0;
      s6_i0        <= 17'b0;
      s6_q0        <= 17'b0;
      s6_valid     <= 1'b0;
      s6_done      <= 1'b0;
      s6_prefix    <= 1'b0;
      s6_prefix_1t <= 1'b0;
      s6_suffix    <= 1'b0;
      s6_suffix_1t <= 1'b0;
      s6_flush     <= 3'b0;
    end
    else if(!tx_enable)
    begin
      s6_en0       <= 1'b0;
      s6_last      <= 1'b0;
      s6_tap0_i0   <= 17'b0;
      s6_tap0_q0   <= 17'b0;
      s6_tap1_i0   <= 17'b0;
      s6_tap1_q0   <= 17'b0;
      s6_tap2_i0   <= 17'b0;
      s6_tap2_q0   <= 17'b0;
      s6_i0        <= 17'b0;
      s6_q0        <= 17'b0;
      s6_valid     <= 1'b0;
      s6_done      <= 1'b0;
      s6_prefix    <= 1'b0;
      s6_prefix_1t <= 1'b0;
      s6_suffix    <= 1'b0;
      s6_suffix_1t <= 1'b0;
      s6_flush     <= 3'b0;
    end
    else if(!s6_done)
    begin
      /* last sample consummed */
      if(out_ready && s6_valid && s6_last)
      begin
        s6_en0     <= 1'b0;
        s6_last    <= 1'b0;
        s6_i0      <= 13'b0;
        s6_q0      <= 13'b0;
        s6_valid   <= 1'b0;
        s6_done    <= 1'b1;
      end
      else
      begin
        /* slot read */
        if(out_ready)
        begin
          s6_valid <= 1'b0;
        end
      
        /* slot write */
        if(out_ready || !s6_valid)
        begin
          if(s5_valid || s6_flush!=3'd0)
          begin
            s6_en0       <= 1'b1;
            s6_prefix    <= s5_prefix;
            s6_prefix_1t <= s6_prefix;
            s6_suffix    <= s5_suffix;
            s6_suffix_1t <= s6_suffix;
            /* tap registers */
            if(s5_suffix || s6_suffix || s6_suffix_1t)
            begin
              s6_tap0_i0 <= n_s6_mult_i0;
              s6_tap0_q0 <= n_s6_mult_q0;
              s6_tap1_i0 <= s6_tap0_i0;
              s6_tap1_q0 <= s6_tap0_q0;
              s6_tap2_i0 <= s6_tap1_i0;
              s6_tap2_q0 <= s6_tap1_q0;
            end
            else if(s5_prefix || s6_prefix || s6_prefix_1t || s6_flush!=3'd0)
            begin
              s6_tap0_i0 <= 17'd0;
              s6_tap0_q0 <= 17'd0;
              s6_tap1_i0 <= s6_tap0_i0;
              s6_tap1_q0 <= s6_tap0_q0;
              s6_tap2_i0 <= s6_tap1_i0;
              s6_tap2_q0 <= s6_tap1_q0;
            end
          
            /* output slot mux */
            if(s5_prefix || s6_prefix || s6_prefix_1t || s6_flush!=3'd0)
            begin
              s6_i0      <= n_s6_win_i0;
              s6_q0      <= n_s6_win_q0;
            end
            else
            begin
              s6_i0      <= s5_i0;
              s6_q0      <= s5_q0;
            end
            
            /* flush */
            if(s5_last)
              s6_flush <= 3'b111;
            else
              s6_flush <= {1'b0,s6_flush[2:1]};
            
            if(s6_flush==3'b001)
              s6_last <= 1'b1;
           
            s6_valid <= 1'b1;
          end
          else
          begin
            s6_valid <= 1'b0;
          end
        end
      end
    end
  end
    
  assign out_valid       = s6_valid;    
  assign out0_enable     = s6_en0; 
  assign {out0_q,out0_i} = {s6_q0,s6_i0};
  assign out_last        = s6_last;
  assign td_done         = s6_done;

`ifdef RW_SIMU_ON
  // pragma coverage block = off, expr = off, toggle = off
  reg  [16*8-1:0] s0_state_str;
  reg  [16*8-1:0] s0_sym_str;
  always @(*)
  begin
    case(s0_sym)
      LSTF:      s0_sym_str="LSTF";  
      LLTF:      s0_sym_str="LLTF";  
      LSIG:      s0_sym_str="LSIG";
      LDATA:     s0_sym_str="LDATA";
      HTGFSTF:   s0_sym_str="HTGFSTF";
      HTGFLTF:   s0_sym_str="HTGFLTF";
      HTGFSIG1:  s0_sym_str="HTGFSIG1";
      HTGFSIG2:  s0_sym_str="HTGFSIG2";
      HTMMSIG1:  s0_sym_str="HTMMSIG1";
      HTMMSIG2:  s0_sym_str="HTMMSIG2";
      HTMMSTF:   s0_sym_str="HTMMSTF";
      HTDLTF:    s0_sym_str="HTDLTF";
      HTELTF:    s0_sym_str="HTELTF";
      HTDATA:    s0_sym_str="HTDATA";
      VHTSIGA1:  s0_sym_str="VHTSIGA1";
      VHTSIGA2:  s0_sym_str="VHTSIGA2";
      VHTSTF:    s0_sym_str="VHTSTF";
      VHTLTF:    s0_sym_str="VHTLTF";
      VHTSIGB:   s0_sym_str="VHTSIGB";
      VHTDATA:   s0_sym_str="VHTDATA";
      HELSIG:    s0_sym_str="HELSIG";
      HESIGA1:   s0_sym_str="HESIGA1";
      HESIGA2:   s0_sym_str="HESIGA2";
      HESIGB:    s0_sym_str="HESIGB"; 
      HESTF:     s0_sym_str="HESTF";  
      HELTF:     s0_sym_str="HELTF"; 
      HEDATA:    s0_sym_str="HEDATA"; 
      HEPE:      s0_sym_str="HEPE";   
      DONE:      s0_sym_str="DONE";  
      IDLE:      s0_sym_str="IDLE";   
      default:   s0_sym_str="XXXXXXX";   
    endcase
  
    case(s0_state)
      S_WAIT_BUFFER:  s0_state_str="S_WAIT_BUFFER";   
      S_READ_BUFFER:  s0_state_str="S_READ_BUFFER";   
      S_WAIT_RELEASE: s0_state_str="S_WAIT_RELEASE";   
      S_DONE:         s0_state_str="S_DONE";   
      default:        s0_state_str="XXXXXXX";   
    endcase
  end
  // pragma coverage block = on, expr = on, toggle = on
`endif  
endmodule
`default_nettype wire
