`default_nettype none
module txvecdec_ucpu
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire            clk,
  input wire            rst_n,

  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire           clear,
  input  wire           enable,
  input  wire   [ 3:0]  subprog,
  output wire           done,

  /*****************************************************************************
  * input parameters
  *****************************************************************************/
  /* txvector */
  input  wire           er,
  input  wire           nss,
  input  wire           fec,
  input  wire           stbc,
  input  wire   [19:0]  length,
  input  wire   [11:0]  leglengthi,
  input  wire   [ 1:0]  gi_type,
  input  wire   [ 1:0]  heltf_type,
  input  wire   [ 2:0]  nheltf,
  input  wire           extrai,
  input  wire   [ 2:0]  packet_extension,
  input  wire           doppler,
  input  wire           mma,
  input  wire   [ 1:0]  nsd,
  input  wire           dcm,
  input  wire   [ 2:0]  rulen,
  input  wire   [ 2:0]  nbpsc,
  input  wire   [ 1:0]  cr,
  
  /*****************************************************************************
  * output results
  *****************************************************************************/
  output reg    [15:0]  ncbps,
  output reg    [15:0]  ndbps,
  output reg    [15:0]  nsym,
  output reg    [ 2:0]  a,
 
  output reg    [23:0]  psdulen,
  output reg    [15:0]  npad,
  
  output reg            extrao,
  output reg            pedisambo,
  output reg    [11:0]  leglengtho,
  output reg    [ 2:0]  tpe,
  output reg            sgidisamb,
  output reg    [20:0]  sigblen,
  output reg    [ 8:0]  nma,
  output reg            inconsistent,
  
  output reg    [11:0]  ncw,
  output reg    [ 1:0]  lldpc,
  output reg    [11:0]  nshrtq,
  output reg    [11:0]  nshrtr,
  output reg    [11:0]  npuncq,
  output reg    [11:0]  npuncr,
  output reg    [11:0]  nrepq,
  output reg    [11:0]  nrepr,
  
  
  /* for debug */
  output reg    [23:0]  npld,
  output reg    [23:0]  navbits,
  output reg    [22:0]  nshrt,
  output reg    [23:0]  npunc,
  output reg    [15:0]  ncbps_short,
  output reg    [15:0]  ndbps_short,
  output reg    [15:0]  ncbps_last,
  output reg    [15:0]  ndbps_last,
  output reg    [15:0]  nsyminit,
  output reg    [ 2:0]  ainit
);
  
  /*****************************************************************************
  * constants
  *****************************************************************************/
  wire  [63:0] cc;
  wire  [ 8:0] pr_addr;
  wire  [15:0] pr_data;
  wire  [ 5:0] rb_addr;
  reg   [31:0] rb_data;
  wire  [ 5:0] wb_addr;
  wire  [31:0] wb_x,wb_y;
  reg   [31:0] r0,r1,r2;
  reg   [ 2:0] format;
  
  localparam  NBPSC_1=3'd0, 
              NBPSC_2=3'd1,
              NBPSC_4=3'd2,
              NBPSC_6=3'd3,
              NBPSC_8=3'd4,
              NBPSC_10=3'd5;
              
  localparam  NSD_48=2'd0, 
              NSD_52=2'd1, 
              NSD_108=2'd2, 
              NSD_234=2'd3;

  localparam  RU_26=3'd0,
              RU_52=3'd1,
              RU_106=3'd2,
              RU_242=3'd3,
              RU_484=3'd4,
              RU_996=3'd5,
              RU_2X996=3'd6;
  
  localparam  CR_12=2'd0,
              CR_23=2'd1,
              CR_34=2'd2,
              CR_56=2'd3;

  localparam  HELTF_1X=2'd0,
              HELTF_2X=2'd1,
              HELTF_4X=2'd2;
  
  localparam  NON_HT=3'd0,
              HT=3'd1,
              VHT=3'd2,
              HE_SU=3'd3,
              HE_TB_FRAME=3'd4,
              HE_TB_TRS=3'd5;

  /*****************************************************************************
  * rw_ucpu
  *****************************************************************************/
  rw_ucpu u_ucpu
  (
    /* system */
    .rst_n(       rst_n),
    .clk(         clk),
  
    /* control */
    .enable(      enable),
    .done(        done),
  
    /* prog */
    .pr_addr(     pr_addr),
    .pr_data(     pr_data),

    /* cc */
    .cc(          cc),
  
    /* rb */
    .rb_addr(     rb_addr),
    .rb_data(     rb_data),
  
    /* wb */
    .wb_addr(     wb_addr),
    .wb_x(        wb_x),
    .wb_y(        wb_y)
  );
 
  /*****************************************************************************
  * 
  *****************************************************************************/
  reg pedisamb;
  always @(*)
  begin
    pedisamb = 1'b0;
    if(format==HE_TB_TRS)
    begin
      if(tpe==3'd4) /* tpe==16us */
      begin
        pedisamb = 1'b1;
      end
      else if(tpe==3'd3 && heltf_type==2'd1) /* tpe==12us && heltftype==2X*/
      begin
       if(leglengthi==12'd49)  pedisamb=1'b1; 
       if(leglengthi==12'd70)  pedisamb=1'b1;  
       if(leglengthi==12'd103) pedisamb=1'b1;  
       if(leglengthi==12'd124) pedisamb=1'b1;  
       if(leglengthi==12'd157) pedisamb=1'b1;  
       if(leglengthi==12'd178) pedisamb=1'b1;  
       if(leglengthi==12'd211) pedisamb=1'b1;  
       if(leglengthi==12'd232) pedisamb=1'b1;  
       if(leglengthi==12'd265) pedisamb=1'b1;  
       if(leglengthi==12'd286) pedisamb=1'b1;  
       if(leglengthi==12'd319) pedisamb=1'b1;  
       if(leglengthi==12'd340) pedisamb=1'b1;  
       if(leglengthi==12'd373) pedisamb=1'b1;        
      end
    end
    else
    begin
      pedisamb = packet_extension[2];
    end
  end
 
  always @(*)
  begin
    case(subprog)
      4'd1,4'd2:      format = HT;
      4'd3,4'd4:      format = VHT;
      4'd5,4'd6,4'd7: format = HE_SU;
      4'd8:           format = HE_TB_FRAME;
      4'd13:          format = HE_TB_TRS;
      default:        format = NON_HT;
    endcase
  end

  /*****************************************************************************
  * condition codes
  *****************************************************************************/
  localparam CC_TRUE           = 6'd0,    
             CC_X_ZERO         = 6'd1,
             CC_EQ             = 6'd2,
             CC_LE             = 6'd3,
             CC_LT             = 6'd4,
             CC_GE             = 6'd5,
             CC_GT             = 6'd6,
             CC_AINIT_NE_3     = 6'd7,
             CC_AINIT_NE_4     = 6'd8,
             CC_A_EQ_1         = 6'd9,
             CC_NOT_FEC_EXTRA  = 6'd10,
             CC_NMA_TEST       = 6'd11,
             CC_NOT_PEDIS      = 6'd12,
             CC_SIGBLEN_NOREM  = 6'd13,
             CC_NOT_SGI        = 6'd14,
             CC_NOT_SGIDISAMB  = 6'd15,
             CC_SUBPROG0       = 6'd16,
             CC_SUBPROG1       = 6'd17,
             CC_SUBPROG2       = 6'd18,
             CC_SUBPROG3       = 6'd19,
             CC_SUBPROG4       = 6'd20,
             CC_SUBPROG5       = 6'd21,
             CC_SUBPROG6       = 6'd22,
             CC_SUBPROG7       = 6'd23, 
             CC_SUBPROG8       = 6'd24,     
             CC_SUBPROG9       = 6'd25,    
             CC_SUBPROG10      = 6'd26,    
             CC_SUBPROG11      = 6'd27,   
             CC_SUBPROG12      = 6'd28,
             CC_SUBPROG13      = 6'd29;
  
  assign cc[CC_TRUE]           = 1'b1;
  assign cc[CC_X_ZERO]         = wb_x[31:0]==32'd0;
  assign cc[CC_EQ]             = wb_y[31:0]==32'd0;          
  assign cc[CC_LE]             = wb_y[31:0]==32'd0 || wb_y[31]; 
  assign cc[CC_LT]             = wb_y[31]; 
  assign cc[CC_GE]             = !wb_y[31]; 
  assign cc[CC_GT]             = !wb_y[31] && wb_y[30:0]!=31'd0; 
  assign cc[CC_AINIT_NE_3]     = ainit!=3'd3; 
  assign cc[CC_AINIT_NE_4]     = ainit!=3'd4; 
  assign cc[CC_A_EQ_1]         = a==2'd1;
  assign cc[CC_NOT_FEC_EXTRA]  = !(extrai && fec); 
  assign cc[CC_NMA_TEST]       = !wb_y[31] && doppler;
  assign cc[CC_NOT_PEDIS]      = !pedisamb;
  assign cc[CC_SIGBLEN_NOREM]  = wb_y[1:0]==2'b0;
  assign cc[CC_NOT_SGI]        = gi_type[0]==1'b0;
  assign cc[CC_NOT_SGIDISAMB]  = sgidisamb==1'b0;
  assign cc[CC_SUBPROG0]       = subprog==4'd0;
  assign cc[CC_SUBPROG1]       = subprog==4'd1;
  assign cc[CC_SUBPROG2]       = subprog==4'd2;
  assign cc[CC_SUBPROG3]       = subprog==4'd3;
  assign cc[CC_SUBPROG4]       = subprog==4'd4;
  assign cc[CC_SUBPROG5]       = subprog==4'd5;
  assign cc[CC_SUBPROG6]       = subprog==4'd6;
  assign cc[CC_SUBPROG7]       = subprog==4'd7;
  assign cc[CC_SUBPROG8]       = subprog==4'd8;
  assign cc[CC_SUBPROG9]       = subprog==4'd9;
  assign cc[CC_SUBPROG10]      = subprog==4'd10;
  assign cc[CC_SUBPROG11]      = subprog==4'd11;
  assign cc[CC_SUBPROG12]      = subprog==4'd12;
  assign cc[CC_SUBPROG13]      = subprog==4'd13;
  assign cc[63:30]             = 41'd0;

  /*****************************************************************************
  * operands
  *****************************************************************************/
  localparam  R_NULL           = 6'd0,
              R_1              = 6'd1,
              R_3              = 6'd2,
              R_4              = 6'd3,
              R_3PM            = 6'd4,
              R_10             = 6'd5,
              R_120            = 6'd6,
              R_40             = 6'd7,
              R_600            = 6'd8,
              R_160_240        = 6'd9,
              R_NBPSC          = 6'd10,
              R_NSD            = 6'd11,
              R_NSDSHRT        = 6'd12,
              R_CRN            = 6'd13,
              R_CRD            = 6'd14,
              R_LENGTHX8       = 6'd15,
              R_ST             = 6'd16,
              R_MXNDBPS        = 6'd17,
              R_MXNCBPS        = 6'd18,
              R_MXNDBPSSHRT    = 6'd19,
              R_MXNCBPSSHRT    = 6'd20,
              R_MXNDBPSLAST    = 6'd21,
              R_MXNCBPSLAST    = 6'd22,
              R_NSYM           = 6'd23,
              R_NSYMINIT       = 6'd24,
              R_NCBPS          = 6'd25,
              R_NCBPSSHRT      = 6'd26,
              R_NDBPS          = 6'd27,
              R_NDBPSSHRT      = 6'd28,
              R_NPLD           = 6'd29,
              R_NAVBITS        = 6'd30,
              R_NCW            = 6'd31,
              R_LLDPC          = 6'd32,
              R_648            = 6'd33,
              R_1944XCR        = 6'd34,
              R_2916DIVCRD     = 6'd35,
              R_1464DIVCRD     = 6'd36,
              R_912DIVCRD      = 6'd37,
              R_12XCRN         = 6'd38,
              R_NPUNC          = 6'd39,
              R_NSHRT          = 6'd40,
              R_R0             = 6'd41,
              R_R1             = 6'd42,
              R_R2             = 6'd43,
              R_MSTBC          = 6'd44,
              R_AINIT          = 6'd45,
              R_TPE            = 6'd46,
              R_LLENGTH        = 6'd47,
              R_MMA            = 6'd48,
              R_THELTFSYM      = 6'd49,
              R_THELTFSYMX3    = 6'd50,
              R_NHELTF         = 6'd51,
              R_TSYM           = 6'd52,
              R_TSYMX3         = 6'd53,
              R_TVHTPREAMBLE   = 6'd54,
              R_LENGTH         = 6'd55,
              R_PSDULEN        = 6'd56;
                                   
  always @(*)
  begin
    case(rb_addr)
      R_1:           rb_data = 32'd1;
      R_3:           rb_data = 32'd3;
      R_4:           rb_data = 32'd4;
      R_3PM:         rb_data = (!er)?32'd5:32'd4;
      R_10:          rb_data = 32'd10;
      R_40:          rb_data = 32'd40;
      R_120:         rb_data = 32'd120;
      R_160_240:     rb_data = (!er)?32'd160:32'd240;
      R_600:         rb_data = 32'd600;
      R_NBPSC: 
      begin
        if(!nss || format==NON_HT)
          case(nbpsc)
            NBPSC_1: rb_data = 32'd1;
            NBPSC_2: rb_data = 32'd2;
            NBPSC_4: rb_data = 32'd4;
            NBPSC_6: rb_data = 32'd6;
            NBPSC_8: rb_data = 32'd8;
            default: rb_data = 32'd10;
          endcase
        else
          case(nbpsc)
            NBPSC_1: rb_data = 32'd2;
            NBPSC_2: rb_data = 32'd4;
            NBPSC_4: rb_data = 32'd8;
            NBPSC_6: rb_data = 32'd12;
            NBPSC_8: rb_data = 32'd16;
            default: rb_data = 32'd20;
          endcase
      end
      R_NSD:
      begin
        case(format)
          NON_HT:
          begin
            rb_data = 32'd48;
          end
          HT,VHT:
          begin
            case(nsd)
              NSD_48:  rb_data = 32'd48;
              NSD_52:  rb_data = 32'd52;
              NSD_108: rb_data = 32'd108;
              default: rb_data = 32'd234;
            endcase
          end
          default:
          begin
            if(!dcm)
            begin
              case(rulen)
                RU_26:    rb_data = 32'd24;
                RU_52:    rb_data = 32'd48;
                RU_106:   rb_data = 32'd102;
                RU_242:   rb_data = 32'd234;
                RU_484:   rb_data = 32'd468;
                RU_996:   rb_data = 32'd980;
                default:  rb_data = 32'd1960;
              endcase
            end
            else
            begin
              case(rulen)
                RU_26:    rb_data = 32'd12;
                RU_52:    rb_data = 32'd24;
                RU_106:   rb_data = 32'd51;
                RU_242:   rb_data = 32'd117;
                RU_484:   rb_data = 32'd234;
                RU_996:   rb_data = 32'd490;
                default:  rb_data = 32'd980;
              endcase
            end
          end
        endcase
      end
      R_NSDSHRT:
      begin
        if(!dcm)
        begin
          case(rulen)
            RU_26:    rb_data = 32'd6;
            RU_52:    rb_data = 32'd12;
            RU_106:   rb_data = 32'd24;
            RU_242:   rb_data = 32'd60;
            RU_484:   rb_data = 32'd120;
            RU_996:   rb_data = 32'd240;
            default:  rb_data = 32'd492;
          endcase
        end
        else
        begin
          case(rulen)
            RU_26:    rb_data = 32'd2;
            RU_52:    rb_data = 32'd6;
            RU_106:   rb_data = 32'd12;
            RU_242:   rb_data = 32'd30;
            RU_484:   rb_data = 32'd60;
            RU_996:   rb_data = 32'd120;
            default:  rb_data = 32'd246;
          endcase
        end
      end
      R_CRN:
      begin
        case(cr)                                 
          CR_12:    rb_data = 32'd1;             
          CR_23:    rb_data = 32'd2;             
          CR_34:    rb_data = 32'd3;             
          default   rb_data = 32'd5;             
        endcase
      end                                  
      R_CRD:
      begin
        case(cr)                                 
          CR_12:    rb_data = 32'd2;             
          CR_23:    rb_data = 32'd3;             
          CR_34:    rb_data = 32'd4;             
          default   rb_data = 32'd6;             
        endcase 
      end                                 
      R_LENGTHX8:    rb_data = (format==NON_HT)?{17'b0,leglengthi,3'b0}:{9'd0,length,3'b0};
      R_ST:
      begin
        case(format)
          NON_HT:  rb_data = 32'd22;
          VHT: 
          begin
            if(fec)
              rb_data = 32'd16;
            else if(nss && nsd==NSD_234 && (nbpsc==NBPSC_8 || (nbpsc==NBPSC_6 && cr==CR_56)))
              rb_data = 32'd28;
            else
              rb_data = 32'd22;
          end
          default: rb_data = (fec)?32'd16:32'd22;
        endcase         
      end
      R_MXNDBPS:     rb_data = (stbc)?{15'b0,       ndbps, 1'b0}:{16'b0,       ndbps};
      R_MXNCBPS:     rb_data = (stbc)?{15'd0,       ncbps, 1'b0}:{16'd0,       ncbps};
      R_MXNDBPSSHRT: rb_data = (stbc)?{15'b0, ndbps_short, 1'b0}:{16'b0, ndbps_short};
      R_MXNCBPSSHRT: rb_data = (stbc)?{15'b0, ncbps_short, 1'b0}:{16'b0, ncbps_short};
      R_MXNDBPSLAST: rb_data = (stbc)?{15'b0,  ndbps_last, 1'b0}:{16'b0,  ndbps_last};
      R_MXNCBPSLAST: rb_data = (stbc)?{15'b0,  ncbps_last, 1'b0}:{16'b0,  ncbps_last};
      R_NSYM:        rb_data = {16'b0,nsym};
      R_NSYMINIT:    rb_data = {16'b0,nsyminit};
      R_NCBPS:       rb_data = {16'b0,ncbps};
      R_NCBPSSHRT:   rb_data = {16'b0,ncbps_short};
      R_NDBPS:       rb_data = {16'b0,ndbps};
      R_NDBPSSHRT:   rb_data = {16'b0,ndbps_short};
      R_NPLD:        rb_data = {8'd0,npld};
      R_NAVBITS:     rb_data = {8'd0,navbits};
      R_NCW:         rb_data = {20'd0,ncw};
      R_LLDPC:     
      begin
        case(lldpc)
          2'd0:    rb_data = 32'd648;
          2'd1:    rb_data = 32'd1296;
          default: rb_data = 32'd1944;
        endcase
      end
      R_648:       rb_data = 32'd648;
      R_1944XCR:
      begin     
        case(cr)
          CR_12:   rb_data = 32'd972;   // 1944 * 1/2
          CR_23:   rb_data = 32'd1296;  // 1944 * 2/3
          CR_34:   rb_data = 32'd1458;  // 1944 * 3/4 
          default: rb_data = 32'd1620;  // 1944 * 5/6 
        endcase
      end
      R_2916DIVCRD:
      begin  
        case(cr)                                   
          CR_12:   rb_data = 32'd1458;             
          CR_23:   rb_data = 32'd972;              
          CR_34:   rb_data = 32'd729;              
          default: rb_data = 32'd486;              
        endcase
      end
      R_1464DIVCRD:
      begin 
        case(cr)           
          CR_12:   rb_data = 32'd732;              
          CR_23:   rb_data = 32'd488;              
          CR_34:   rb_data = 32'd366;              
          default: rb_data = 32'd244;              
        endcase
      end
      R_912DIVCRD: 
      begin
        case(cr)           
          CR_12:   rb_data = 32'd456; 
          CR_23:   rb_data = 32'd304; 
          CR_34:   rb_data = 32'd228; 
          default: rb_data = 32'd152; 
        endcase 
      end
      R_12XCRN:
      begin 
        case(cr)
          CR_12:   rb_data = 32'd12;
          CR_23:   rb_data = 32'd24;
          CR_34:   rb_data = 32'd36;
          default: rb_data = 32'd60;
        endcase 
      end
      R_NPUNC:       rb_data = {8'd0,npunc};
      R_NSHRT:       rb_data = {9'd0,nshrt};
      R_R0:          rb_data = r0;
      R_R1:          rb_data = r1;
      R_R2:          rb_data = r2;
      R_MSTBC:       rb_data = (stbc)?32'd2:32'd1;
      R_AINIT:
      begin
        case(ainit)
          3'd1:   rb_data = 32'd1;
          3'd2:   rb_data = 32'd2;
          3'd3:   rb_data = 32'd3;
          default:rb_data = 32'd4;
        endcase
      end
      R_TPE:
      begin
        case(tpe)
          3'd0:    rb_data = 32'd0;   /*  0 us */      
          3'd1:    rb_data = 32'd40;  /*  4 us */      
          3'd2:    rb_data = 32'd80;  /*  8 us */      
          3'd3:    rb_data = 32'd120; /* 12 us */      
          default: rb_data = 32'd160; /* 16 us */      
        endcase
      end
    
      R_LLENGTH:   rb_data = {20'b0,leglengthi};               
      R_MMA:
      begin
        if(doppler)
          rb_data = (!mma)?32'd10:32'd20;
        else
          rb_data = 32'd0;
      end
      R_THELTFSYM:
      begin
        case({gi_type,heltf_type})
          {2'd0,HELTF_1X}: rb_data = 32'd40;  /* 0.8 1X SU    */
          {2'd1,HELTF_1X}: rb_data = 32'd48;  /* 1.6 1X    TB */
          {2'd0,HELTF_2X}: rb_data = 32'd72;  /* 0.8 2X SU    */
          {2'd1,HELTF_2X}: rb_data = 32'd80;  /* 1.6 2X SU TB */
          {2'd0,HELTF_4X}: rb_data = 32'd136; /* 0.8 4X SU    */
          {2'd2,HELTF_4X}: rb_data = 32'd160; /* 3.2 4X SU TB */
          default:         rb_data = 32'd0;
        endcase
      end
      R_THELTFSYMX3:
      begin
        case({gi_type,heltf_type})
          {2'd0,HELTF_1X}: rb_data = 32'd120; /* 0.8 1X SU    */
          {2'd1,HELTF_1X}: rb_data = 32'd144; /* 1.6 1X    TB */
          {2'd0,HELTF_2X}: rb_data = 32'd216; /* 0.8 2X SU    */
          {2'd1,HELTF_2X}: rb_data = 32'd240; /* 1.6 2X SU TB */
          {2'd0,HELTF_4X}: rb_data = 32'd408; /* 0.8 4X SU    */
          {2'd2,HELTF_4X}: rb_data = 32'd480; /* 3.2 4X SU TB */
          default:         rb_data = 32'd0;
        endcase
      end
      R_NHELTF:
      begin
        case(format)
          HE_SU: rb_data = (!nss)?32'd1:32'd2;
          default: /* HE_TB */
            case(nheltf)
              3'd0:    rb_data = 32'd1;
              3'd1:    rb_data = 32'd2;
              3'd3:    rb_data = 32'd4;
              3'd5:    rb_data = 32'd6;
              default: rb_data = 32'd8;
            endcase
        endcase
      end
      R_TSYM:
      begin
        case(gi_type)
          2'd0:    rb_data = 32'd136;  /* 13.6 SU    */
          2'd1:    rb_data = 32'd144;  /* 14.4    TB */
          default: rb_data = 32'd160;  /* 16.0 SU    */
        endcase
      end
      R_TSYMX3:
      begin
        case(gi_type)
          2'd0:    rb_data = 32'd408;  /* 3*13.6 SU    */
          2'd1:    rb_data = 32'd432;  /* 3*14.4    TB */
          default: rb_data = 32'd480;  /* 3*16.0 SU    */
        endcase
      end
      R_TVHTPREAMBLE: rb_data = (nss || stbc)?32'd44:32'd40; /* 36us+nvhtlf*4us*/
      R_LENGTH:       rb_data = {12'b0, length};
      R_PSDULEN:      rb_data = { 8'b0, psdulen};
      default:        rb_data = 32'd0;
    endcase
  end

  /*****************************************************************************
  * register file
  *****************************************************************************/
  localparam W_NULL          = 6'd0,      
             W_NCBPS         = 6'd1,      
             W_NCBPSSHRT     = 6'd2,      
             W_NCBPSLAST     = 6'd3,    
             W_NDBPS         = 6'd4,   
             W_NDBPSSHRT     = 6'd5,   
             W_NDBPSLAST     = 6'd6,   
             W_MXNSYMINIT    = 6'd7,   
             W_NSYM          = 6'd8,    
             W_NSYMINIT      = 6'd9,    
             W_R0            = 6'd10,  
             W_R0_X          = 6'd11,  
             W_R1            = 6'd12,  
             W_R2            = 6'd13,  
             W_MXR2          = 6'd14,     
             W_TPE           = 6'd15,  
             W_PSDULEN_NPAD  = 6'd16,  
             W_NPLD          = 6'd17,    
             W_NAVBITS       = 6'd18,     
             W_LLENGTH       = 6'd19,     
             W_AINIT_NEXCESS = 6'd20,  
             W_DISAMB        = 6'd21,   
             W_SIGBLEN       = 6'd22,  
             W_EXTRA         = 6'd23,   
             W_AEQAINITP1    = 6'd24,  
             W_AINITEQ4      = 6'd25,  
             W_AINITEQAM1    = 6'd26,  
             W_TBINIT        = 6'd27,  
             W_LLDPC1296     = 6'd28,    
             W_LLDPC1944     = 6'd29,    
             W_LLDPC648      = 6'd30,    
             W_NCW           = 6'd31,    
             W_NCW1          = 6'd32,     
             W_NCW2          = 6'd33,    
             W_NPUNC         = 6'd34,    
             W_NPUNCQR       = 6'd35,    
             W_NREPQR        = 6'd36,    
             W_NSHRT         = 6'd37,    
             W_NSHRTQR       = 6'd38,
             W_INCONS_LT     = 6'd39,
             W_INCONS_LE     = 6'd40,
             W_NMA           = 6'd41;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      ncbps            <= 16'b0;
      ndbps            <= 16'b0;
      nsym             <= 16'd0;   
      nsyminit         <= 16'd0;   
      npad             <= 16'd0;
      ainit            <= 3'b0;
      a                <= 3'd0;
      nma              <= 9'd0;
      r0               <= 32'd0;        
      r1               <= 32'd0;        
      r2               <= 32'd0;        
      npld             <= 24'd0;         
      navbits          <= 24'd0;           
      ncw              <= 12'd0;
      lldpc            <= 2'b0;
      nshrt            <= 23'b0;
      npunc            <= 24'b0;           
      {nshrtq,nshrtr}  <= {12'b0,12'b0};
      {npuncq,npuncr}  <= {12'b0,12'b0};
      { nrepq, nrepr}  <= {12'b0,12'b0};
      extrao           <= 1'b0;
      ncbps_short      <= 16'b0;
      ndbps_short      <= 16'b0;
      ncbps_last       <= 16'b0;
      ndbps_last       <= 16'b0;
      psdulen          <= 24'b0;
      pedisambo        <= 1'b0;
      leglengtho       <= 12'b0;
      sgidisamb        <= 1'b0;
      tpe              <= 3'b0;
      sigblen          <= 21'd0;
      inconsistent     <= 1'b0;
    end
    else if(clear)
    begin
       ncbps           <= 16'b0;
       ndbps           <= 16'b0;
       nsym            <= 16'd0;            
       nsyminit        <= 16'd0;            
       ainit           <= 3'b0;
       a               <= 3'd0;
       nma             <= 9'd0;
       npad            <= 16'd0;         
       r0              <= 32'd0;         
       r1              <= 32'd0;         
       r2              <= 32'd0;         
       npld            <= 24'd0;          
       navbits         <= 24'd0;            
       ncw             <= 12'd0;
       lldpc           <= 2'b0;
       nshrt           <= 23'b0;
      {nshrtq,nshrtr}  <= {12'b0,12'b0};
       npunc           <= 24'b0;            
      {npuncq,npuncr}  <= {12'b0,12'b0};
      { nrepq, nrepr}  <= {12'b0,12'b0};
      extrao           <= 1'b0;
      ncbps_short      <= 16'b0;
      ndbps_short      <= 16'b0;
      ncbps_last       <= 16'b0;
      ndbps_last       <= 16'b0;
      psdulen          <= 24'b0;
      pedisambo        <= 1'b0;
      leglengtho       <= 12'b0;
      sgidisamb        <= 1'b0;
      tpe              <= 3'b0;
      sigblen          <= 21'd0;
      inconsistent     <= 1'b0;
    end
    else
    begin
      case(wb_addr)
        W_NCBPS:       ncbps       <= wb_y[15:0];
        W_NCBPSSHRT:   ncbps_short <=  wb_y[15:0];
        W_NCBPSLAST:   ncbps_last  <=  wb_y[15:0];
        W_NDBPS:       ndbps       <= wb_y[15:0];
        W_NDBPSSHRT:   ndbps_short <=  wb_y[15:0];
        W_NDBPSLAST:   ndbps_last  <=  wb_y[15:0];
        W_MXNSYMINIT:
        begin
          if(!stbc)
          begin
            inconsistent <= inconsistent | wb_y[31:16]!=16'd0;
            nsyminit     <= wb_y[15:0];
          end
          else
          begin
            inconsistent <= inconsistent | wb_y[31:15]!=17'd0;
            nsyminit     <= {wb_y[14:0],1'b0};
          end
        end
        W_NSYM:
        begin
          inconsistent <= inconsistent | wb_y[31:16]!=16'd0;
          nsym         <= wb_y[15:0];
        end
        W_NSYMINIT:
        begin
          inconsistent <= inconsistent | wb_y[31:16]!=16'd0;
          nsyminit     <= wb_y[15:0];
        end
        W_R0:          r0          <= wb_y;
        W_R0_X:        r0          <= wb_x;
        W_R1:          r1          <= wb_y;
        W_R2:          r2          <= wb_y;
        W_MXR2:        r2          <= (stbc)?{wb_y[30:0],1'b0}:wb_y;
        W_TPE:
        begin
          case(format)
            HE_SU:
            begin
              /* HE_SU */
              case({a,packet_extension})
                {3'd4,3'd2}: tpe <= 3'd2; /*  8 us */
                {3'd3,3'd2}: tpe <= 3'd1; /*  4 us */
                {3'd1,3'd4}: tpe <= 3'd1; /*  4 us */
                {3'd2,3'd4}: tpe <= 3'd2; /*  8 us */
                {3'd3,3'd4}: tpe <= 3'd3; /* 12 us */
                {3'd4,3'd4}: tpe <= 3'd4; /* 16 us */
                default:     tpe <= 3'd0; /*  0 us */
              endcase
            end
            HE_TB_FRAME:
            begin
              inconsistent <= inconsistent | wb_y[2:0]>=3'd5;
              tpe          <= wb_y[2:0];
            end
            default: /* HE_TB_TRS */
            begin
              tpe <= packet_extension[2:0];
            end
          endcase
        end
        W_PSDULEN_NPAD:
        begin
          case(format)
            NON_HT,HT:
            begin
              psdulen <= 24'b0;
              npad    <= wb_y[15:0];
            end
            default:
            begin
              inconsistent <= inconsistent | wb_y[31] | wb_y[31:3]==29'd0;
              psdulen      <= wb_y[26:3];
              npad         <= {13'b0,wb_y[ 2:0]};
            end
          endcase
        end
        W_NPLD:        npld    <= wb_y[23:0];         
        W_NAVBITS:     navbits <= wb_y[23:0];   
        W_LLENGTH:  
        begin
          inconsistent <= inconsistent | (wb_y[31:12]!=20'd0);
          leglengtho   <= wb_y[11:0];
        end
        W_AINIT_NEXCESS:
        begin
          case(wb_y[16:0])                    
            17'd1:   {a,ainit} <= {3'd1,3'd1};     
            17'd2:   {a,ainit} <= {3'd2,3'd2};     
            17'd3:   {a,ainit} <= {3'd3,3'd3};     
            default: {a,ainit} <= {3'd4,3'd4};  /* for y=0 or y>=4 */  
          endcase
        end
        W_DISAMB:
        begin
          if(format==VHT)
            sgidisamb <= (gi_type[0] && (wb_x[3:0]==4'd9));
          else
            pedisambo <= (wb_y[31])?1'b0:1'b1;
        end
        W_SIGBLEN: sigblen <= wb_y[22:2]; 
        W_EXTRA:   extrao  <= 1'b1;
        W_AEQAINITP1:
        begin
          case(ainit)                    
            3'd1:    a <= 3'd2;     
            3'd2:    a <= 3'd3;     
            3'd3:    a <= 3'd4;     
            default: a <= 3'd1;     
          endcase
        end
        W_AINITEQ4:  ainit <= 3'd4;
        W_AINITEQAM1:
        begin
          case(packet_extension[1:0])
            2'd0:    ainit <= 3'd3;
            2'd1:    ainit <= 3'd0;
            2'd2:    ainit <= 3'd1;
            default: ainit <= 3'd2;
          endcase
        end
        W_TBINIT:
        begin
          nsyminit <= nsym;
          if(format==HE_TB_FRAME)
          begin
            /* FRAME method */
            case(packet_extension[1:0])
              2'd0:    {a,ainit} <= {3'd4,3'd4};
              2'd1:    {a,ainit} <= {3'd1,3'd1};
              2'd2:    {a,ainit} <= {3'd2,3'd2};
              default: {a,ainit} <= {3'd3,3'd3};
            endcase
          end
          else
          begin
            /* TRS method */
            a <= 3'd4;
            if(!fec)
            begin
              ainit  <= 3'd4;
              extrao <= 1'b0;
            end
            else
            begin
              ainit  <= 3'd3;
              extrao <= 1'b1;
            end
          end
        end
        W_LLDPC1944:   lldpc   <= 2'd2;
        W_LLDPC1296:   lldpc   <= 2'd1;
        W_LLDPC648:    lldpc   <= 2'd0;
        W_NCW:         ncw     <= wb_y[11:0];
        W_NCW1:        ncw     <= 12'd1;
        W_NCW2:        ncw     <= 12'd2;
        W_NPUNC:       npunc           <= wb_y[23:0];
        W_NPUNCQR:     {npuncq,npuncr} <= {wb_y[11:0],wb_x[11:0]};
        W_NREPQR:      { nrepq, nrepr} <= {wb_y[11:0],wb_x[11:0]};
        W_NSHRT:       nshrt           <= wb_y[22:0];
        W_NSHRTQR:     {nshrtq,nshrtr} <= {wb_y[11:0],wb_x[11:0]};
        W_INCONS_LT:   inconsistent    <= inconsistent | wb_y[31];
        W_INCONS_LE:   inconsistent    <= inconsistent | wb_y[31] | wb_y[31:0]==32'd0;
        W_NMA:         nma             <= wb_y[8:0];
        default: ; /* do nothing */
      endcase
    end
  end
 
  /*****************************************************************************
  * program
  *****************************************************************************/
  localparam   NOP  = 3'd0,  CLRXY  = 3'd1,
               ADD  = 3'd2,  SUB    = 3'd3,
               MUL  = 3'd4,  DIV    = 3'd5,
               TERM = 3'd7;
  
  localparam   I=1'b0, J=1'b1;
  
  wire [15:0]  p[0:427];

  localparam  NON_HT0          = 9'd14,
              NON_HT0_0        = 9'd26,
              HT0              = 9'd30,
              HT0_0            = 9'd42,
              HT1_VHT2         = 9'd52,
              VHT0             = 9'd59,
              VHT0_0           = 9'd71,
              VHT0_1           = 9'd83,
              VHT1             = 9'd84,
              VHT1_1           = 9'd96,
              VHT1_2           = 9'd108,
              HE_SU0           = 9'd112,
              HE_SU0_0         = 9'd131,
              HE_SU0_1         = 9'd136,
              HE_SU0_2         = 9'd143,
              HE_SU0_3         = 9'd149,
              HE_SU1           = 9'd161,
              HE_SU1_1         = 9'd172,
              HE_SU1_2         = 9'd176,
              HE_SU1_4         = 9'd181,
              HE_SU2           = 9'd182,
              HE_SU2_0         = 9'd191,
              HE_SU_1          = 9'd194,
              HE_SU2_2         = 9'd206,
              HE_SU2_3         = 9'd218,
              HE_TB_FRAME0     = 9'd222,
              HE_TB_FRAME0_0   = 9'd241,
              HE_TB_FRAME0_1   = 9'd244,
              HE_TB_FRAME0_2   = 9'd246,
              HE_TB_FRAME0_3   = 9'd256,
              HE_TB_FRAME0_X   = 9'd271,
              HE_TB_FRAME0_4   = 9'd276,
              HE_TB_FRAME0_5   = 9'd294,
              HE_TB_FRAME0_6   = 9'd300,
              HE_TB_TRS0       = 9'd315,
              HE_TB_TRS1       = 9'd324,
              LDPC0            = 9'd327,
              NCW_4            = 9'd337,
              NCW_41           = 9'd342,
              NCW_3            = 9'd344,
              NCW_2            = 9'd351,
              NCW_1            = 9'd354,
              NCW_0            = 9'd361,
              L_NSHR           = 9'd367,
              NSHR_0           = 9'd378,
              LDPC2            = 9'd379,
              LDPC2_1          = 9'd385,
              LDPC1            = 9'd386,
              LDPC1_1          = 9'd407,
              LDPC1_2          = 9'd408,
              LDPC3            = 9'd409,
              LDPC3_1          = 9'd415;

  /***************************************************************************
  * select the subprogram
  ***************************************************************************/
  assign p[              0] = {J,               CC_SUBPROG0,        NON_HT0}; // ncbps,ndbps,nsym,npad
  assign p[              1] = {J,               CC_SUBPROG1,            HT0}; // ncbps,ndbps,npld,nsym_init,npad,navbits
  assign p[              2] = {J,               CC_SUBPROG2,       HT1_VHT2}; // (ldpc) update navbits,nsym
  assign p[              3] = {J,               CC_SUBPROG3,           VHT0}; // ncbps,ndbps,npld,nsym_init,npad,navbits,psdulen,sigblen
  assign p[              4] = {J,               CC_SUBPROG4,           VHT1}; // sgidisamb,l_length
  assign p[              5] = {J,               CC_SUBPROG5,         HE_SU0}; // ncbps,ndbps,nsym_init,a_init,npld,navbits,psdulen,npad
  assign p[              6] = {J,               CC_SUBPROG6,         HE_SU1}; // (ldpc) update navbits,nsym,a
  assign p[              7] = {J,               CC_SUBPROG7,         HE_SU2}; // nma,l_length,pe_disamb
  assign p[              8] = {J,               CC_SUBPROG8,   HE_TB_FRAME0}; // ncbps,ndbps,tpe,nsym_init,a_init,npld,navbits,psdulen,npad
  assign p[              9] = {J,               CC_SUBPROG9,          LDPC0}; // ncw,lldpc,npunc
  assign p[             10] = {J,              CC_SUBPROG10,          LDPC1}; // evaluate extra
  assign p[             11] = {J,              CC_SUBPROG11,          LDPC2}; // recompute npunc
  assign p[             12] = {J,              CC_SUBPROG12,          LDPC3}; // nrepq,nrepr,npuncq,npuncr,nshrtq,nshrtr
  assign p[             13] = {J,              CC_SUBPROG13,     HE_TB_TRS0}; //
  /***************************************************************************
  * nonht0 
  ****************************************************************************
  * ncbps     = nbpsc*nsd                          
  * ndbps     = ncbps*crn/crd                           
  * nsym      = ceil((8*l_length+16+6)/ndbps)
  * npad      = ndbps - ( (8*l_length+16+6) % ndbps)
  ***************************************************************************/
  assign p[        NON_HT0] = {I,        W_NULL,      CLRXY,        R_NBPSC};             
  assign p[             15] = {I,        W_NULL,        ADD,          R_NSD}; 
  assign p[             16] = {I,        W_NULL,        MUL,         R_NULL}; 
  assign p[             17] = {I,       W_NCBPS,        NOP,          R_CRN}; // ncbps=nbpsc*nsd
  assign p[             18] = {I,        W_NULL,        MUL,          R_CRD};
  assign p[             19] = {I,        W_NULL,        DIV,         R_NULL}; // ndbps=ncbps*crn/crd
  assign p[             20] = {I,       W_NDBPS,      CLRXY,     R_LENGTHX8};
  assign p[             21] = {I,        W_NULL,        ADD,           R_ST};
  assign p[             22] = {I,        W_NULL,        ADD,        R_NDBPS}; // r0 = 8*l_length+16+6
  assign p[             23] = {I,          W_R0,        DIV,            R_1};
  assign p[             24] = {J,                 CC_X_ZERO,      NON_HT0_0};
  assign p[             25] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[      NON_HT0_0] = {I,        W_NSYM,        NOP,        R_NDBPS}; // nsym=ceil( (8*l_length+16+6) / ndbps )
  assign p[             27] = {I,        W_NULL,        MUL,           R_R0};
  assign p[             28] = {I,        W_NULL,        SUB,         R_NULL};
  assign p[             29] = {I,W_PSDULEN_NPAD,       TERM,         R_NULL}; // npad=nsym*ndbps-(8*l_length+16+6)
  /***************************************************************************
  * ht0       
  ****************************************************************************
  * ncbps     = nbpsc*nsd*nss                           
  * ndbps     = ncbps*crn/crd                           
  * npld      = 8*length+16+6*nes
  * nsym_init = mstbc*ceil(npld/(mstbc*ndbps))
  * npad      = nsym_init*ndbps - (8*length+16+6*nes)
  * navbits   = nsym_init*ncbps
  ***************************************************************************/
  assign p[            HT0] = {I,        W_NULL,      CLRXY,        R_NBPSC};             
  assign p[             31] = {I,        W_NULL,        ADD,          R_NSD}; 
  assign p[             32] = {I,        W_NULL,        MUL,         R_NULL}; 
  assign p[             33] = {I,       W_NCBPS,        NOP,          R_CRN}; // ncbps=nbpsc*nsd*nss
  assign p[             34] = {I,        W_NULL,        MUL,          R_CRD};
  assign p[             35] = {I,        W_NULL,        DIV,         R_NULL}; // ndbps=ncbps*crn/crd
  assign p[             36] = {I,       W_NDBPS,      CLRXY,     R_LENGTHX8};
  assign p[             37] = {I,        W_NULL,        ADD,           R_ST};
  assign p[             38] = {I,        W_NULL,        ADD,      R_MXNDBPS}; 
  assign p[             39] = {I,        W_NPLD,        DIV,            R_1}; // npld = 8*length+16+6*nes
  assign p[             40] = {J,                 CC_X_ZERO,          HT0_0};
  assign p[             41] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[          HT0_0] = {I,  W_MXNSYMINIT,        NOP,         R_NULL}; // nsym_init=mstbc*ceil( (8*length-16-6*nes) / (mstbc*ndbps) )
  assign p[             43] = {I,        W_NULL,      CLRXY,     R_NSYMINIT}; 
  assign p[             44] = {I,        W_NULL,        ADD,        R_NDBPS}; 
  assign p[             45] = {I,        W_NSYM,        MUL,     R_LENGTHX8}; // nsym = nsym_init
  assign p[             46] = {I,        W_NULL,        SUB,           R_ST};
  assign p[             47] = {I,        W_NULL,        SUB,         R_NULL};
  assign p[             48] = {I,W_PSDULEN_NPAD,      CLRXY,     R_NSYMINIT}; // npad = nsym_init*ndbps - (8*length+16+6*nes)
  assign p[             49] = {I,        W_NULL,        ADD,        R_NCBPS}; 
  assign p[             50] = {I,        W_NULL,        MUL,         R_NULL}; 
  assign p[             51] = {I,     W_NAVBITS,       TERM,         R_NULL}; // navbits   = nsym_init*ncbps
  /***************************************************************************
  * ht1,vht2           update navbits, nsym
  ***************************************************************************/
  assign p[       HT1_VHT2] = {I,        W_NULL,      CLRXY,      R_NAVBITS}; 
  assign p[             53] = {I,        W_NULL,        ADD,      R_MXNCBPS}; 
  assign p[             54] = {I,        W_NULL,        ADD,         R_NULL};      
  assign p[             55] = {I,     W_NAVBITS,      CLRXY,     R_NSYMINIT}; // navbits = navbits + mstbc*ncbps                                
  assign p[             56] = {I,        W_NULL,        ADD,        R_MSTBC};          
  assign p[             57] = {I,        W_NULL,        ADD,         R_NULL};           
  assign p[             58] = {I,        W_NSYM,       TERM,         R_NULL}; // nsym    = nsym_init + mstbc                  
  /***************************************************************************
  * vht0               ncbps,ndbps,nsym, npad
  ****************************************************************************
  * ncbps      = nbpsc*nsd*nss                             
  * ndbps      = ncbps*crn/crd                             
  * nsym_init  = mstbc*ceil((8*apep_length+16+6*nes)/(mstbc*ndbps))  
  * npld       = nsym_init*ndbps
  * navbits    = nsym_init*ncbps
  * psdulen    = floor( (nsym_init*ndbps - (16+6*nes)) / 8)
  * npad       = (nsym_init*ndbps - (16+6*nes)) % 8
  * sigblen    = ceil( apep_length / 4)
  ***************************************************************************/
  assign p[           VHT0] = {I,        W_NULL,      CLRXY,        R_NBPSC};             
  assign p[             60] = {I,        W_NULL,        ADD,          R_NSD}; 
  assign p[             61] = {I,        W_NULL,        MUL,         R_NULL}; 
  assign p[             62] = {I,       W_NCBPS,        NOP,          R_CRN}; // ncbps=nbpsc*nsd*nss
  assign p[             63] = {I,        W_NULL,        MUL,          R_CRD};
  assign p[             64] = {I,        W_NULL,        DIV,         R_NULL}; // ndbps=ncbps*crn/crd
  assign p[             65] = {I,       W_NDBPS,      CLRXY,     R_LENGTHX8};
  assign p[             66] = {I,        W_NULL,        ADD,           R_ST};
  assign p[             67] = {I,        W_NULL,        ADD,      R_MXNDBPS};
  assign p[             68] = {I,        W_NULL,        DIV,            R_1};
  assign p[             69] = {J,                 CC_X_ZERO,         VHT0_0};
  assign p[             70] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[         VHT0_0] = {I,  W_MXNSYMINIT,        NOP,         R_NULL}; // nsym_init=mstbc*ceil( (8*apep_length-16-6*nes) / (mstbc*ndbps) )
  assign p[             72] = {I,        W_NULL,      CLRXY,     R_NSYMINIT}; 
  assign p[             73] = {I,        W_NULL,        ADD,        R_NCBPS}; 
  assign p[             74] = {I,        W_NSYM,        MUL,         R_NULL}; // nsym = nsym_init
  assign p[             75] = {I,     W_NAVBITS,      CLRXY,     R_NSYMINIT}; // navbits = nsym_init*ncbps
  assign p[             76] = {I,        W_NULL,        ADD,        R_NDBPS};
  assign p[             77] = {I,        W_NULL,        MUL,           R_ST};
  assign p[             78] = {I,        W_NPLD,        SUB,         R_NULL}; // npld = nsym_init*ndbps
  assign p[             79] = {I,W_PSDULEN_NPAD,      CLRXY,       R_LENGTH}; // psdulen = floor( (nsym_init*ndbps - (16+6*nes)) / 8) ; npad = (nsym_init*ndbps - (16+6*nes)) % 8
  assign p[             80] = {I,        W_NULL,        ADD,            R_4};
  assign p[             81] = {J,          CC_SIGBLEN_NOREM,         VHT0_1};
  assign p[             82] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[         VHT0_1] = {I,     W_SIGBLEN,       TERM,         R_NULL}; // sigblen = ceil( apep_length / 4)
  /***************************************************************************
  * vht1               sgidisamb, l_length
  ****************************************************************************
  * sgidisamb      = (sgi==1) ? (nsym%10)==9 : 0 
  * txtime (long)  = 36 + nvhtltf*4 + t_syml*nsym 
  * txtime (short) = 36 + nvhtltf*4 + t_syml*ceil( nsym*t_syms/t_syml)
  * l_length       = ceil((txtime-20)/4)*3-3
  ***************************************************************************/
  assign p[           VHT1] = {I,        W_NULL,      CLRXY,         R_NSYM};
  assign p[             85] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[             86] = {J,                CC_NOT_SGI,         VHT1_1}; 
  assign p[             87] = {I,        W_NULL,        NOP,           R_10}; 
  assign p[             88] = {I,        W_NULL,        DIV,         R_NULL};
  assign p[             89] = {I,      W_DISAMB,      CLRXY,           R_10}; // sgidisamb  = (sgi==1) ? (nsym%10)==9 : 0 
  assign p[             90] = {I,        W_NULL,        ADD,            R_1};
  assign p[             91] = {I,        W_NULL,        SUB,         R_NSYM};
  assign p[             92] = {I,        W_NULL,        MUL,           R_10};
  assign p[             93] = {I,        W_NULL,        DIV,            R_1};
  assign p[             94] = {J,                 CC_X_ZERO,         VHT1_1}; 
  assign p[             95] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[         VHT1_1] = {I,          W_R0,      CLRXY,         R_NULL}; // r0 = (sgi==0) ? nsym : ceil(9*nsym/10)
  assign p[             97] = {I,        W_NULL,        NOP,           R_R0};
  assign p[             98] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[             99] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[            100] = {I,        W_NULL,        ADD,         R_NULL}; 
  assign p[            101] = {I,        W_NULL,        ADD, R_TVHTPREAMBLE}; // y=4*r0 
  assign p[            102] = {I,        W_NULL,        ADD,           R_10};
  assign p[            103] = {I,        W_NULL,        SUB,         R_NULL};
  assign p[            104] = {I,        W_NULL,        SUB,            R_4};
  assign p[            105] = {I,        W_NULL,        DIV,            R_1}; 
  assign p[            106] = {J,                 CC_X_ZERO,         VHT1_2}; 
  assign p[            107] = {I,        W_NULL,        ADD,         R_NULL};
  assign p[         VHT1_2] = {I,        W_NULL,        NOP,            R_3}; 
  assign p[            109] = {I,        W_NULL,        MUL,            R_3}; 
  assign p[            110] = {I,        W_NULL,        SUB,         R_NULL}; 
  assign p[            111] = {I,     W_LLENGTH,       TERM,         R_NULL}; // l_length = ceil((txtime-20)/4)*3-3
  /***************************************************************************
  * he_su0             ncbps,ndbps,nsym_init,nexcess,a_init,npld,navbits
  *      
  ****************************************************************************
  * ncbps           = nbpsc*nsd*nss
  * ncbps_short     = nbpsc*nsd_short*nss
  * ndbps           = ncbps*crn/crd
  * ndbps_short     = ncbps_short*crn/crd
  * nsym_init       = mstbc*ceil( (8*apep_length+6*nes+16) / (mstbc*ndbps))
  * nexcess         = (8*apep_length+6*nes+16) % (mstbc*ndbps)
  * a_init          = (nexcess==0) ? 4 : min( 4, ceil(nexcess / (mstbc*ndbps_short)))
  * ncbps_last_init = (a_init==4) ? ncbps : (a_init*ncbps_short)
  * ndbps_last_init = (a_init==4) ? ndbps : (a_init*ndbps_short)
  * npld            = (nsym_init - mstbc)*ndbps + mstbc*ndbps_last_init
  * navbits         = (nsym_init - mstbc)*ncbps + mstbc*ncbps_last_init
  * psdulen         = floor( (npld - (16+6*nes)) / 8)
  * npad            = (npld - (16+6*nes)) % 8
  ***************************************************************************/
  assign p[         HE_SU0] = {I,         W_NULL,     CLRXY,        R_NBPSC};       
  assign p[            113] = {I,         W_NULL,       ADD,          R_NSD};
  assign p[            114] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            115] = {I,        W_NCBPS,       NOP,          R_CRN}; // ncbps = nbpsc*nsd
  assign p[            116] = {I,         W_NULL,       MUL,          R_CRD};
  assign p[            117] = {I,         W_NULL,       DIV,         R_NULL};
  assign p[            118] = {I,        W_NDBPS,     CLRXY,        R_NBPSC}; // ndbps = ncbps*crn/crd
  assign p[            119] = {I,         W_NULL,       ADD,      R_NSDSHRT};
  assign p[            120] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            121] = {I,    W_NCBPSSHRT,       NOP,          R_CRN}; // ncbps_short = nsd_short*nbpsc
  assign p[            122] = {I,         W_NULL,       MUL,          R_CRD};
  assign p[            123] = {I,         W_NULL,       DIV,         R_NULL};
  assign p[            124] = {I,    W_NDBPSSHRT,     CLRXY,     R_LENGTHX8}; // ndbps_short = ncbps_short*crn/crd
  assign p[            125] = {I,         W_NULL,       ADD,           R_ST};
  assign p[            126] = {I,         W_NULL,       ADD,      R_MXNDBPS};
  assign p[            127] = {I,         W_NULL,       DIV,         R_NULL};
  assign p[            128] = {I,         W_R0_X,       NOP,            R_1}; // r0 = nexcess = (8*lenth+6*nes+16) % (mstbc*ndbps)
  assign p[            129] = {J,                 CC_X_ZERO,       HE_SU0_0};
  assign p[            130] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[       HE_SU0_0] = {I,   W_MXNSYMINIT,     CLRXY,           R_R0}; // nsym_init = mstbc*ceil( (8*lenth+6*nes+16) / (mstbc*ndbps))
  assign p[            132] = {I,         W_NULL,       ADD,  R_MXNDBPSSHRT};
  assign p[            133] = {I,         W_NULL,       DIV,            R_1};
  assign p[            134] = {J,                 CC_X_ZERO,       HE_SU0_1};
  assign p[            135] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[       HE_SU0_1] = {I,W_AINIT_NEXCESS,     CLRXY,         R_NULL}; // a_init = (nexcess==0) ? 0 : min(ceil(nexcess/(mstbc*ndbps_short)),4)
  assign p[            137] = {I,         W_NULL,       NOP,    R_NCBPSSHRT}; 
  assign p[            138] = {I,         W_NULL,       ADD,        R_AINIT}; 
  assign p[            139] = {I,         W_NULL,       MUL,         R_NULL}; 
  assign p[            140] = {J,             CC_AINIT_NE_4,       HE_SU0_2}; 
  assign p[            141] = {I,         W_NULL,     CLRXY,        R_NCBPS};
  assign p[            142] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[       HE_SU0_2] = {I,    W_NCBPSLAST,     CLRXY,    R_NDBPSSHRT}; // ncbps_last_init = (a_init==4) ? ncbps : a_init*ncbps_short
  assign p[            144] = {I,         W_NULL,       ADD,        R_AINIT}; 
  assign p[            145] = {I,         W_NULL,       MUL,         R_NULL}; 
  assign p[            146] = {J,             CC_AINIT_NE_4,       HE_SU0_3};       
  assign p[            147] = {I,         W_NULL,     CLRXY,        R_NDBPS};
  assign p[            148] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[       HE_SU0_3] = {I,    W_NDBPSLAST,     CLRXY,     R_NSYMINIT}; // ndbps_last_init = (a_init==4) ? ndbps : a_init*ndbps_short
  assign p[            150] = {I,         W_NULL,       ADD,        R_NDBPS}; 
  assign p[            151] = {I,         W_NSYM,       MUL,      R_MXNDBPS}; // nsym = nsym_init
  assign p[            152] = {I,         W_NULL,       SUB,  R_MXNDBPSLAST};
  assign p[            153] = {I,         W_NULL,       ADD,           R_ST};
  assign p[            154] = {I,         W_NPLD,       SUB,         R_NULL}; // npld = (nsym_init-mstbc)*ndbps+mstbc*ndbps_last_init
  assign p[            155] = {I, W_PSDULEN_NPAD,     CLRXY,     R_NSYMINIT}; // psdulen = (npld-16-6*nes) / 8 ; npad = (npld-16-6*nes) % 8
  assign p[            156] = {I,         W_NULL,       ADD,        R_NCBPS};
  assign p[            157] = {I,         W_NULL,       MUL,      R_MXNCBPS};
  assign p[            158] = {I,         W_NULL,       SUB,  R_MXNCBPSLAST};
  assign p[            159] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[            160] = {I,      W_NAVBITS,      TERM,         R_NULL}; // navbits = (nsym_init-mstbc)*ndbps + mstbc*ncbps_last_init
  /***************************************************************************
  * HESU1     update navbits,nsym
  ****************************************************************************
  * navbits         = (a_init!=3) ? navbits + mstbc*ncbps_short
  *                                 navbits + mstbc*(ncbps-3*ncbps_short) : 
  * nsym            = (a_init!=4) ? nsym_init  : nsym_init + mstbc
  * a               = (a_init!=4) ? (a_init+1) : 1 
  ***************************************************************************/
  assign p[         HE_SU1] = {J,             CC_AINIT_NE_3,       HE_SU1_1}; // if(a_init!=3) goto he_su1_1
  assign p[            162] = {I,         W_NULL,     CLRXY,        R_NCBPS}; 
  assign p[            163] = {I,         W_NULL,       ADD,    R_NCBPSSHRT};
  assign p[            164] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            165] = {I,         W_NULL,       SUB,         R_NULL}; 
  assign p[            166] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            167] = {I,         W_MXR2,     CLRXY,      R_NAVBITS}; // r2 = mstbc*(ncbps-3*ncbps_short)
  assign p[            168] = {I,         W_NULL,       ADD,           R_R2}; 
  assign p[            169] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[            170] = {I,      W_NAVBITS,       NOP,         R_NULL}; // navbits = navbits + mstbc*(ncbps-3*ncbps_short)
  assign p[            171] = {J,                   CC_TRUE,       HE_SU1_2}; 
  assign p[       HE_SU1_1] = {I,         W_NULL,     CLRXY,  R_MXNCBPSSHRT}; 
  assign p[            173] = {I,         W_NULL,       ADD,      R_NAVBITS}; 
  assign p[            174] = {I,         W_NULL,       ADD,         R_NULL}; 
  assign p[            175] = {I,      W_NAVBITS,       NOP,         R_NULL}; // navbits = navbits + mstbc*ncbps_short
  assign p[       HE_SU1_2] = {J,             CC_AINIT_NE_4,       HE_SU1_4};
  /* AINIT==4 NSYM=NSYMINIT, A=AINIT+1 */
  assign p[            177] = {I,         W_NULL,     CLRXY,     R_NSYMINIT}; 
  assign p[            178] = {I,         W_NULL,       ADD,        R_MSTBC}; 
  assign p[            179] = {I,         W_NULL,       ADD,         R_NULL}; 
  assign p[            180] = {I,         W_NSYM,       NOP,         R_NULL}; 
  assign p[       HE_SU1_4] = {I,   W_AEQAINITP1,      TERM,         R_NULL}; // a = (a_init!=4) ? (a_init+1) : 1 
  /***************************************************************************
  * HE_SU2    nma, l_length, pe_disamb
  ****************************************************************************
  * nma      = (doppler==1) ? max(0, ceil( (nsym-1) / mma) -1) : 0
  * l_length = ceil( (txtime-20us) /4)*3-(3+m)
  * p_disamb = 
  ***************************************************************************/
  assign p[         HE_SU2] = {I,          W_TPE,     CLRXY,    R_THELTFSYM}; // tpe=table_28.43(nominal_packet_padding,a)  see table 28-4
  assign p[            183] = {I,         W_NULL,       ADD,       R_NHELTF};
  assign p[            184] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            185] = {I,           W_R2,     CLRXY,         R_NSYM}; // r2 = nheltf*t_heltf_sym
  assign p[            186] = {I,         W_NULL,       ADD,            R_1};
  assign p[            187] = {I,         W_NULL,       SUB,          R_MMA};
  assign p[            188] = {I,         W_NULL,       DIV,            R_1};
  assign p[            189] = {J,                 CC_X_ZERO,       HE_SU2_0};
  assign p[            190] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[       HE_SU2_0] = {I,         W_NULL,       SUB,            R_1};
  assign p[            192] = {J,               CC_NMA_TEST,        HE_SU_1};
  assign p[            193] = {I,         W_NULL,     CLRXY,         R_NULL};
  assign p[        HE_SU_1] = {I,          W_NMA,       ADD,           R_R2}; // y = nma = (doppler==1) ? max(0, ceil( (nsym-1) / mma) -1) : 0
  assign p[            195] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            196] = {I,           W_R0,     CLRXY,         R_NSYM}; // r0 = (1+nma) * nheltf * t_heltf_sym 
  assign p[            197] = {I,         W_NULL,       ADD,         R_TSYM}; 
  assign p[            198] = {I,         W_NULL,       MUL,           R_R0}; 
  assign p[            199] = {I,         W_NULL,       ADD,      R_160_240}; 
  assign p[            200] = {I,         W_NULL,       ADD,          R_TPE}; 
  assign p[            201] = {I,         W_NULL,       ADD,           R_40}; // y = 16/24 us + (1+nma)*t_heltf_sym*nheltf + nsym*tsym + tpe ( = txtime - 20us)
  assign p[            202] = {I,         W_NULL,       DIV,            R_1};
  assign p[            203] = {I,         W_R0_X,       NOP,         R_NULL}; // y = (txtime-20us) / 4us ; r0 = (txtime-20us) % 4us , r0 used for pe_dis afterwards
  assign p[            204] = {J,                 CC_X_ZERO,       HE_SU2_2}; 
  assign p[            205] = {I,         W_NULL,       ADD,         R_NULL};  
  assign p[       HE_SU2_2] = {I,           W_R2,       NOP,         R_NULL}; // r2 = ceil((txtime-20us)/4us)
  assign p[            207] = {I,         W_NULL,       NOP,           R_R2}; 
  assign p[            208] = {I,         W_NULL,       ADD,           R_R2};
  assign p[            209] = {I,         W_NULL,       ADD,          R_3PM}; 
  assign p[            210] = {I,         W_NULL,       SUB,         R_NULL}; 
  assign p[            211] = {I,      W_LLENGTH,       NOP,         R_NULL}; // l_length = ceil((txtime-20us)/4us)*3-(3+m)
  assign p[            212] = {I,         W_NULL,     CLRXY,           R_R0};
  assign p[            213] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[            214] = {J,                     CC_EQ,       HE_SU2_3};
  assign p[            215] = {I,         W_NULL,     CLRXY,           R_40};
  assign p[            216] = {I,         W_NULL,       ADD,           R_R0};
  assign p[            217] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[       HE_SU2_3] = {I,         W_NULL,       NOP,          R_TPE};
  assign p[            219] = {I,         W_NULL,       ADD,         R_TSYM};
  assign p[            220] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            221] = {I,       W_DISAMB,      TERM,         R_NULL}; // pe_disamb
  /***************************************************************************
  * he_tb_frame0:   nma, nsym, t_pe, a_init, nsym_init, 
  ****************************************************************************
  * t_ma          = mma*t_sym + nheltf*t_heltf_sym
  * t_he_preamble = 20us + nheltf*t_heltf_sym
  * nma    = (doppler==1) ? max(0,floor[( (l_length+3+m)/3*4-t_he_preamble-(pedis+2)*t_sym )/t_ma) : 0
  * nsym   = floor[( (l_length+3+m)/3*4-t_he_preamble-nma*nheltf*t_heltf_sym)/t_sym]-pedis
  * t_pe   = floor[( (l_length+3+m)/3*4-t_he_preamble-nma*nheltf*t_heltf_sym-nsym*t_sym)/4]*4
  *
  * if(fec==1 && extra==1)
  *   a_init    = (a==1) ? 4            : (a-1)
  *   nsym_init = (a==1) ? (nsym-mstbc) : nsym 
  * else
  *   a_init    = a
  *   nsym_init = nsym
  * end
  *
  * ncbps            = nbpsc*nsd
  * ncbps_short      = nbpsc*nsd_short
  * ncbps_last_init  = (a_init==4) ? ncbps : (a_init*ncbps_short)
  * ndbps            = ncbps*crn/crd
  * ndbps_short      = ncbps_short*crn/crd
  * ndbps_last_init  = (a_init==4) ? ndbps : (a_init*ndbps_short)
  * npld             = (nsym_init-mstbc)*ndbps + mstbc*ndbps_last_init
  * navbits          = (nsym_init-mstbc)*ncbps + mstbc*ncbps_last_init
  * psdulen          = floor( (npld-16-6*nes) / 8)
  * npad             = (npld-16-6*nes) % 8
  ***************************************************************************/
  assign p[   HE_TB_FRAME0] = {I,         W_NULL,     CLRXY,  R_THELTFSYMX3};
  assign p[            223] = {I,         W_NULL,       ADD,       R_NHELTF};
  assign p[            224] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            225] = {I,           W_R2,     CLRXY,      R_LLENGTH}; // r2 = 3*nheltf*t_heltf_sym
  assign p[            226] = {I,         W_NULL,       ADD,          R_3PM};
  assign p[            227] = {I,         W_NULL,       ADD,           R_40};
  assign p[            228] = {I,         W_NULL,       MUL,           R_R2}; 
  assign p[            229] = {I,         W_NULL,       SUB,          R_600};
  assign p[            230] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            231] = {I,           W_R1,     CLRXY,          R_MMA}; // r1 = (l_length+3+m)*40-3*t_he_preamble
  assign p[            232] = {I,         W_NULL,       ADD,       R_TSYMX3};
  assign p[            233] = {I,         W_NULL,       MUL,           R_R2};
  assign p[            234] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[            235] = {I,           W_R0,     CLRXY,           R_R1}; // r0 = 3*t_ma = 3*nheltf*t_heltf_sym + 3*mma*t_sym
  assign p[            236] = {I,         W_NULL,       ADD,       R_TSYMX3};
  assign p[            237] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            238] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            239] = {J,              CC_NOT_PEDIS, HE_TB_FRAME0_0};
  assign p[            240] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[ HE_TB_FRAME0_0] = {J,               CC_NMA_TEST, HE_TB_FRAME0_1}; // nma=0 if !doppler or (l_length+3+m)*40-3*t_he_preamble-3*(pedis+2)*t_sym )<0 
  assign p[            242] = {I,         W_NULL,     CLRXY,         R_NULL};
  assign p[            243] = {J,                   CC_TRUE, HE_TB_FRAME0_2};
  assign p[ HE_TB_FRAME0_1] = {I,         W_NULL,       NOP,           R_R0};
  assign p[            245] = {I,         W_NULL,       DIV,         R_NULL};
  assign p[ HE_TB_FRAME0_2] = {I,          W_NMA,       NOP,           R_R2}; // y = nma = (doppler)?max(0 , floor(  (l_length+3+m)*40-3*t_he_preamble-3*(pedis+2)*t_sym )/(3*t_ma)))) :0
  assign p[            247] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            248] = {I,           W_R0,     CLRXY,           R_R1}; // r0 = 3*nma*nheltf*t_heltf_sym
  assign p[            249] = {I,         W_NULL,       ADD,           R_R0};
  assign p[            250] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            251] = {I,    W_INCONS_LE,       NOP,       R_TSYMX3};
  assign p[            252] = {I,           W_R0,       DIV,            R_1}; // r0 = (l_length+3+m)*40-3*t_he_preamble-3*nma*nheltf*t_heltf_sym
  assign p[            253] = {J,              CC_NOT_PEDIS, HE_TB_FRAME0_3};
  assign p[            254] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            255] = {I,    W_INCONS_LE,       NOP,         R_NULL};
  assign p[ HE_TB_FRAME0_3] = {I,         W_NSYM,     CLRXY,       R_TSYMX3}; // nsym = floor[( (l_length+3+m)*40-3*t_he_preamble-3*nma*nheltf*t_heltf_sym)/(3*t_sym)]-pedis
  assign p[            257] = {I,         W_NULL,       ADD,         R_NSYM};
  assign p[            258] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            259] = {I,           W_R1,     CLRXY,           R_R0}; // r1 = 3*nsym*t_sym
  assign p[            260] = {I,         W_NULL,       ADD,           R_R1}; 
  assign p[            261] = {I,         W_NULL,       SUB,          R_120}; 
  assign p[            262] = {I,    W_INCONS_LT,       DIV,         R_NULL}; // r1 = (l_length+3+m)*40-3*t_he_preamble-3*nma*nheltf*t_heltf_sym-3*nsym*t_sym
  assign p[            263] = {I,          W_TPE,       NOP,         R_NULL}; // tpe = floor[( (l_length+3+m)/3*4-t_he_preamble-nma*nheltf*t_heltf_sym-nsym*t_sym)/4]*4
  assign p[            264] = {I,       W_TBINIT,       NOP,         R_NULL}; // a_init = a; nsym_init = nsym
  assign p[            265] = {J,          CC_NOT_FEC_EXTRA, HE_TB_FRAME0_4};
  assign p[            266] = {J,                 CC_A_EQ_1, HE_TB_FRAME0_X};
  assign p[            267] = {I,         W_NULL,     CLRXY,         R_NSYM};
  assign p[            268] = {I,   W_AINITEQAM1,       ADD,         R_NULL};
  assign p[            269] = {I,     W_NSYMINIT,       NOP,         R_NULL}; // a_init = a -1 ; nsym_init = nsym
  assign p[            270] = {J,                   CC_TRUE, HE_TB_FRAME0_4};
  assign p[ HE_TB_FRAME0_X] = {I,     W_AINITEQ4,     CLRXY,         R_NSYM};
  assign p[            272] = {I,         W_NULL,       ADD,        R_MSTBC};
  assign p[            273] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            274] = {I,    W_INCONS_LE,       NOP,         R_NULL}; 
  assign p[            275] = {I,     W_NSYMINIT,       NOP,         R_NULL}; // a_init = 4 ; nsym_init= nsym-mstbc
  assign p[ HE_TB_FRAME0_4] = {I,         W_NULL,     CLRXY,        R_NBPSC}; // inconsistent=1 if nsym_init<=0      
  assign p[            277] = {I,         W_NULL,       ADD,          R_NSD};
  assign p[            278] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            279] = {I,        W_NCBPS,       NOP,          R_CRN}; // ncbps = nbpsc*nsd
  assign p[            280] = {I,         W_NULL,       MUL,          R_CRD};
  assign p[            281] = {I,         W_NULL,       DIV,         R_NULL};
  assign p[            282] = {I,        W_NDBPS,     CLRXY,        R_NBPSC}; // ndbps = ncbps*crn/crd
  assign p[            283] = {I,         W_NULL,       ADD,      R_NSDSHRT};
  assign p[            284] = {I,         W_NULL,       MUL,         R_NULL};
  assign p[            285] = {I,    W_NCBPSSHRT,       NOP,          R_CRN}; // ncbps_short = nsd_short*nbpsc
  assign p[            286] = {I,         W_NULL,       MUL,          R_CRD};
  assign p[            287] = {I,         W_NULL,       DIV,         R_NULL};
  assign p[            288] = {I,    W_NDBPSSHRT,     CLRXY,    R_NCBPSSHRT}; // ndbps_short = ncbps_short*crn/crd
  assign p[            289] = {I,         W_NULL,       ADD,        R_AINIT}; 
  assign p[            290] = {I,         W_NULL,       MUL,         R_NULL}; 
  assign p[            291] = {J,             CC_AINIT_NE_4, HE_TB_FRAME0_5}; 
  assign p[            292] = {I,         W_NULL,     CLRXY,        R_NCBPS};
  assign p[            293] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[ HE_TB_FRAME0_5] = {I,    W_NCBPSLAST,     CLRXY,    R_NDBPSSHRT}; // ncbps_last_init = (a_init==4) ? ncbps : a_init*ncbps_short
  assign p[            295] = {I,         W_NULL,       ADD,        R_AINIT}; 
  assign p[            296] = {I,         W_NULL,       MUL,         R_NULL}; 
  assign p[            297] = {J,             CC_AINIT_NE_4, HE_TB_FRAME0_6};       
  assign p[            298] = {I,         W_NULL,     CLRXY,        R_NDBPS};
  assign p[            299] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[ HE_TB_FRAME0_6] = {I,    W_NDBPSLAST,     CLRXY,     R_NSYMINIT}; // ndbps_last_init = (a_init==4) ? ndbps : a_init*ndbps_short
  assign p[            301] = {I,         W_NULL,       ADD,        R_NDBPS}; 
  assign p[            302] = {I,         W_NULL,       MUL,      R_MXNDBPS};
  assign p[            303] = {I,         W_NULL,       SUB,  R_MXNDBPSLAST};
  assign p[            304] = {I,         W_NULL,       ADD,           R_ST};
  assign p[            305] = {I,         W_NPLD,       SUB,         R_NULL}; // npld = (nsym_init-mstbc)*ndbps+mstbc*ndbps_last_init
  assign p[            306] = {I, W_PSDULEN_NPAD,     CLRXY,     R_NSYMINIT}; // psdulen = (npld-16-6*nes) / 8 ; npad = (npld-16-6*nes) % 8
  assign p[            307] = {I,         W_NULL,       ADD,        R_NCBPS};
  assign p[            308] = {I,         W_NULL,       MUL,      R_MXNCBPS};
  assign p[            309] = {I,         W_NULL,       SUB,  R_MXNCBPSLAST};
  assign p[            310] = {I,         W_NULL,       ADD,         R_NULL};
  assign p[            311] = {I,      W_NAVBITS,     CLRXY,      R_PSDULEN}; // navbits = (nsym_init-mstbc)*ndbps + mstbc*ncbps_last_init
  assign p[            312] = {I,         W_NULL,       ADD,       R_LENGTH};
  assign p[            313] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[            314] = {I,    W_INCONS_LT,      TERM,         R_NULL}; // inconsistent=1 if(apep_length>psdulen)
  /***************************************************************************
  * he_tb_trs0:   nma, nsym, t_pe, a_init, nsym_init, 
  ****************************************************************************
  * note: by design, the machw does not provide the nsym (fval field) parameter
  *       so it is mandatory to recompute nsym like in the 'frame' method.
  *       all ambiguous cases are identified and stored in a dedicated lut to 
  *       generate 'pedis'
  ****************************************************************************
  * t_he_preamble = 20us + 1*t_heltf_sym
  * nsym   = floor[( (l_length+3+m)/3*4-t_he_preamble)/t_sym]-pedis
  * nsym_init = nsym 
  *
  * if(fec==1)
  *   extra     = 1 
  *   a_init    = 3
  * else
  *   a_init    = 4
  * end
  *
  * ncbps            = nbpsc*nsd
  * ncbps_short      = nbpsc*nsd_short
  * ncbps_last_init  = (a_init==4) ? ncbps : (a_init*ncbps_short)
  * ndbps            = ncbps*crn/crd
  * ndbps_short      = ncbps_short*crn/crd
  * ndbps_last_init  = (a_init==4) ? ndbps : (a_init*ndbps_short)
  * npld             = (nsym_init-mstbc)*ndbps + mstbc*ndbps_last_init
  * navbits          = (nsym_init-mstbc)*ncbps + mstbc*ncbps_last_init
  * psdulen          = floor( (npld-16-6*nes) / 8)
  * npad             = (npld-16-6*nes) % 8
  ***************************************************************************/
  assign p[     HE_TB_TRS0] = {I,         W_TPE,      CLRXY,      R_LLENGTH};
  assign p[            316] = {I,         W_NULL,       ADD,          R_3PM};
  assign p[            317] = {I,         W_NULL,       ADD,           R_40};
  assign p[            318] = {I,         W_NULL,       MUL,  R_THELTFSYMX3}; 
  assign p[            319] = {I,         W_NULL,       SUB,          R_600};
  assign p[            320] = {I,         W_NULL,       SUB,       R_TSYMX3}; 
  assign p[            321] = {I,         W_NULL,       DIV,            R_1};
  assign p[            322] = {J,              CC_NOT_PEDIS,     HE_TB_TRS1}; // dependency with tpe, l_length, heltf_type
  assign p[            323] = {I,         W_NULL,       SUB,         R_NULL};
  assign p[     HE_TB_TRS1] = {I,         W_NSYM,       NOP,         R_NULL}; // nsym = floor[((l_length+3+m)*40-3*t_he_preamble)/t_sym]-pedis
  assign p[            325] = {I,       W_TBINIT,       NOP,         R_NULL}; // initialize a,ainit,nsyminit
  assign p[            326] = {J,                   CC_TRUE, HE_TB_FRAME0_4}; // others compuation are identical to he_tb_frame
  /***************************************************************************
  * LDPC0            ncw,lldpc,nshrt,npunc
  ****************************************************************************
  * ncw   = see 802.11-2012 table 20-16
  * lldpc = see 802.11-2012 table 20-16
  * nshrt = max(0,(ncw*lldpc*cr)-npld)
  * npunc = max(0,(ncw*lldpc)-navbits-nshrt)
  * r0    = ncw*lldpc
  * r1    = ncw*lldpc*(1-cr)
  * r2    = ncw*lldpc*cr
  ***************************************************************************/
  /* NCW, LLDPC */
  assign p[          LDPC0] = {I,         W_NULL,     CLRXY,      R_NAVBITS}; 
  assign p[            328] = {I,         W_NULL,       ADD,          R_648}; // Y=NAVBITS-648  
  assign p[            329] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-648  
  assign p[            330] = {J,                     CC_LE,          NCW_0}; // JMP NCW0 IF NAVBITS<=648
  assign p[            331] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-1296    
  assign p[            332] = {J,                     CC_LE,          NCW_1}; // JMP NCW1 IF NAVBITS<=1296
  assign p[            333] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-1944     
  assign p[            334] = {J,                     CC_LE,          NCW_2}; // JMP NCW2 IF NAVBITS<=1944
  assign p[            335] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-2592      
  assign p[            336] = {J,                     CC_LE,          NCW_3}; // JMP NCW3 IF NAVBITS<=2592
  /* IF 2592 < NAVBITS */
  assign p[          NCW_4] = {I,    W_LLDPC1944,     CLRXY,         R_NPLD}; // LLDPC=1944 ;   X=Y=0
  assign p[            338] = {I,         W_NULL,       ADD,      R_1944XCR}; // Y=NPLD;        B=1944*CR
  assign p[            339] = {I,         W_NULL,       DIV,            R_1}; // Y=NPLD/1944/R; B=1
  assign p[            340] = {J,                 CC_X_ZERO,         NCW_41}; // JMP  IF X==0
  assign p[            341] = {I,         W_NULL,       ADD,         R_NULL}; // Y=NPLD/1944/R+1 
  assign p[         NCW_41] = {I,          W_NCW,       NOP,         R_NULL}; // NCW=Y;
  assign p[            343] = {J,                   CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF 1944 < NAVBITS <= 2592 */
  assign p[          NCW_3] = {I,         W_NCW2,     CLRXY,      R_NAVBITS}; // NCW=2; X=Y=0 ; B=NAVBITS
  assign p[            345] = {I,    W_LLDPC1944,       ADD,         R_NPLD}; // LLDPC=1944; Y=NAVBITS;  B=NPLD
  assign p[            346] = {I,         W_NULL,       SUB,   R_2916DIVCRD}; // Y=NAVBITS-NPLD;         B=2916*(1-CR)
  assign p[            347] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-NPLD-2616*(1-CR)
  assign p[            348] = {J,                     CC_GE,         L_NSHR}; // JMP L_NSHR IF NAVBITS >= (NPLD+2616*(1-CR))     
  assign p[            349] = {I,    W_LLDPC1296,       NOP,         R_NULL}; // LLDPC=1296  
  assign p[            350] = {J,                   CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF 1296 < NAVBITS <= 1944 */
  assign p[          NCW_2] = {I,         W_NCW1,       NOP,         R_NULL}; // NCW=1 ;
  assign p[            352] = {I,    W_LLDPC1944,       NOP,         R_NULL}; // LLDPC=2 [1944] ;
  assign p[            353] = {J,                   CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF 648  < NAVBITS <= 1296 */
  assign p[          NCW_1] = {I,         W_NCW1,     CLRXY,      R_NAVBITS}; // NCW=1 ; X=Y=0 ; B=NAVBITS
  assign p[            355] = {I,    W_LLDPC1944,       ADD,         R_NPLD}; // LLDPC=1944; Y=NAVBITS; B=NPLD 
  assign p[            356] = {I,         W_NULL,       SUB,   R_1464DIVCRD}; // Y=NAVBITS-NPLD;        B=1464*(1-CR)
  assign p[            357] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-NPLD-1464*(1-R)    
  assign p[            358] = {J,                     CC_GE,         L_NSHR}; // JMP L_NSHR IF NAVBITS >= (NPLD+1464*(1-CR)) 
  assign p[            359] = {I,    W_LLDPC1296,       NOP,         R_NULL}; // LLDPC=1296   
  assign p[            360] = {J,                   CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF NAVBITS <= 648  */
  assign p[          NCW_0] = {I,         W_NCW1,     CLRXY,      R_NAVBITS}; // NCW=1 ; X=Y=0 ;          B=NAVBITS
  assign p[            362] = {I,    W_LLDPC1296,       ADD,         R_NPLD}; // LLDPC=1296; Y=NAVBITS;   B=NPLD
  assign p[            363] = {I,         W_NULL,       SUB,    R_912DIVCRD}; // Y=NAVBITS-NPLD;          B=912*(1-CR)
  assign p[            364] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NAVBITS-NPLD-912*(1-CR)
  assign p[            365] = {J,                     CC_GE,         L_NSHR}; // JMP L_NSHR   
  assign p[            366] = {I,     W_LLDPC648,       NOP,         R_NULL}; // LLDPC=648  
  /* NSHRT */
  assign p[         L_NSHR] = {I,         W_NULL,     CLRXY,          R_NCW}; // X=Y=0;      B=NCW
  assign p[            368] = {I,         W_NULL,       ADD,        R_LLDPC}; // Y=NCW;      B=LLDPC
  assign p[            369] = {I,         W_NULL,       MUL,         R_NULL}; // Y=NCW*LLDPC
  /*    R0=NCW*LLDPC                        [ used several times afterwards ] */
  assign p[            370] = {I,           W_R0,       NOP,          R_CRD}; // R0=Y; B=CRD
  /*    R1=NCW*LLDPC*(1-CR)=NCW*LLDPC/CRD   [ used several times afterwards ] */
  assign p[            371] = {I,         W_NULL,       DIV,         R_NULL}; // Y=NCW*LLDPC/CRD
  assign p[            372] = {I,           W_R1,       NOP,          R_CRN}; // R1=Y; B=CRN
  /*    R2=NCW*LLDPC*CR=NCW*LLDPC/CRD*CRN   [ used several times afterwards ] */
  /* NSHRT=MAX(0,NCW*LLDPC*CR-NPLD) */
  assign p[            373] = {I,         W_NULL,       MUL,         R_NULL}; // Y=NCW*LLDPC/CRD
  assign p[            374] = {I,         W_NULL,       NOP,         R_NPLD}; // Y=NCW*LLDPC/CRD*CRN;  B=NPLD
  assign p[            375] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NCW*LLDPC/CRD*CRN-NPLD
  assign p[            376] = {J,                     CC_GE,         NSHR_0}; // JMP LA_9 NCW*LLDPC/CRD*CRN-NPLD >= 0
  assign p[            377] = {I,         W_NULL,     CLRXY,         R_NULL}; // X=Y=0
  assign p[         NSHR_0] = {I,        W_NSHRT,       NOP,         R_NULL}; // NSHRT=Y, continuing to ldpc2 (compute npunc)
  /***************************************************************************
  * LDPC2           [re]compute npunc
  ****************************************************************************
  * (in) r0    = ncw*lldpc
  ****************************************************************************
  * npunc      = max(0, ncw*lldpc-navbits-nshrt)
  ***************************************************************************/
  /* NPUNC=MAX(0,NCW*LLDPC-NAVBITS-NSHRT) */
  assign p[          LDPC2] = {I,         W_NULL,     CLRXY,           R_R0}; // (in) r0 = ncw*lldpc                      
  assign p[            380] = {I,         W_NULL,       ADD,      R_NAVBITS};                                             
  assign p[            381] = {I,         W_NULL,       SUB,        R_NSHRT};                                             
  assign p[            382] = {I,         W_NULL,       SUB,         R_NULL};                                             
  assign p[            383] = {J,                     CC_GE,        LDPC2_1};                                             
  assign p[            384] = {I,         W_NULL,     CLRXY,         R_NULL};                                             
  assign p[        LDPC2_1] = {I,        W_NPUNC,      TERM,         R_NULL}; // npunc = max(0, ncw*lldpc-navbits-nshrt)  
  /***************************************************************************
  * LDPC1           evaluate extra
  ****************************************************************************
  * (in) r1 = ncw*lldpc*(1-cr)
  * (in) r2 = ncw*lldpc*cr
  ****************************************************************************
  *
  ***************************************************************************/
  assign p[          LDPC1] = {I,         W_NULL,     CLRXY,        R_NPUNC}; // R2=10*NPUNC 
  assign p[            387] = {I,         W_NULL,       ADD,           R_10}; //  B=10 ;
  /* IF NPUNC > 0.3*NCW*LLDPC*(1-CR) THEN ADD EXTRAU */
  assign p[            388] = {I,         W_NULL,       MUL,         R_NULL}; // Y=NPUNC*10;
  assign p[            389] = {I,           W_R2,     CLRXY,           R_R1}; // R3=10*NPUNC ; X=Y=0; B=NCW*LLDPC*(1-CR) 
  assign p[            390] = {I,         W_NULL,       ADD,         R_NULL}; // Y=NCW*LLDPC*(1-CR)
  assign p[            391] = {I,         W_NULL,       ADD,         R_NULL}; // Y=2*NCW*LLDPC*(1-CR)
  assign p[            392] = {I,         W_NULL,       ADD,           R_R2}; // Y=2*NCW*LLDPC*(1-CR); B=10*NPUNC
  assign p[            393] = {I,         W_NULL,       SUB,         R_NULL}; // Y=3*NCW*LLDPC*(1-CR)-10*NPUNC
  assign p[            394] = {J,                     CC_LT,        LDPC1_2}; // JMP ADDEXT IF 10*NPUNC > 3*NCW*LLDPC*(1-CR)
  /* IF !(NPUNC > 0.1*NCW*LLDPC(1-CR)) THEN NO EXTRAU  IS REWRITTEN AS */
  /* IF NPUNC <= 0.1*NCW*LLDPC(1-CR) THEN NO EXTRAU   */
  assign p[            395] = {I,         W_NULL,     CLRXY,           R_R1}; // X=Y=0;  B=NCW*LLDPC*(1-CR)
  assign p[            396] = {I,         W_NULL,       ADD,           R_R2}; // Y=NCW*LLDPC*(1-CR); B=10*NPUNC  
  assign p[            397] = {I,         W_NULL,       SUB,         R_NULL}; // Y=NCW*LLDPC*(1-CR)-10*NPUNC
  assign p[            398] = {J,                     CC_GE,        LDPC1_1}; // JMP NOEXT [NOEXTRAU]
  assign p[            399] = {I,         W_NULL,     CLRXY,        R_NSHRT}; // X=Y=0 ; B=NSHRT 
  assign p[            400] = {I,         W_NULL,       ADD,           R_10}; // Y=NSHRT ; B=10
  assign p[            401] = {I,         W_NULL,       MUL,         R_NULL}; // Y=NSHRT*10;
  /*   R3 = 10*NSHRT   */
  assign p[            402] = {I,           W_R2,     CLRXY,        R_NPUNC}; // R3=10*NSHRT ; X=Y=0; B=NPUNC
  assign p[            403] = {I,         W_NULL,       ADD,       R_12XCRN}; // Y=NPUNC ; B=12*CR/(1-CR)
  assign p[            404] = {I,         W_NULL,       MUL,           R_R2}; // Y=12*NPUNC*R/(1-R) ; B=10*NSHRT
  assign p[            405] = {I,         W_NULL,       SUB,         R_NULL}; // Y=12*NPUNC*R/(1-R)-10*NSHRT
  /* IF NSHRT < 1.2*NPUNC*R/(1-R) THEN ADD EXTRAU  */
  assign p[            406] = {J,                     CC_GT,        LDPC1_2}; // JMP ADDEXT
  /* ELSE */
  /* NO EXTRA */
  assign p[        LDPC1_1] = {I,         W_NULL,      TERM,         R_NULL}; // NSYMU=Y;
  /* ADD EXTRA */
  assign p[        LDPC1_2] = {I,        W_EXTRA,      TERM,         R_NULL}; // NSYMU=Y;
  /***************************************************************************
  * LDPC3    nrep,    (nrepq,nrepr), (npuncq,npuncr), (nshrtq,nshrtr)
  ****************************************************************************
  * (in) r1 = ncw*lldpc*(1-cr)
  *  nrep   = max(0, navbits-ncw*lldpc*(1-cr)-npld)
  *  nrepq  = floor(nrep  / ncw)
  *  npuncq = floor(npunc / ncw)
  *  nshrtq = floor(nshrt / ncw)
  *  nrepr  = nrep  % ncw
  *  npuncr = npunc % ncw
  *  nshrtr = nshrt % ncw
  ***************************************************************************/
  assign p[          LDPC3] = {I,      W_NULL,        CLRXY,      R_NAVBITS}; 
  assign p[            410] = {I,      W_NULL,          ADD,           R_R1}; 
  assign p[            411] = {I,      W_NULL,          SUB,         R_NPLD}; 
  assign p[            412] = {I,      W_NULL,          SUB,         R_NULL}; 
  assign p[            413] = {J,                     CC_GE,        LDPC3_1};
  assign p[            414] = {I,      W_NULL,        CLRXY,         R_NULL};
  assign p[        LDPC3_1] = {I,      W_NULL,          NOP,          R_NCW}; // y = max(0, navbits-ncw*lldpc*(1-cr)-npld)
  assign p[            416] = {I,      W_NULL,          DIV,         R_NULL};
  /* NREPQ,  NREPR */
  assign p[            417] = {I,    W_NREPQR,        CLRXY,        R_NPUNC};
  assign p[            418] = {I,      W_NULL,          ADD,          R_NCW};
  assign p[            419] = {I,      W_NULL,          DIV,         R_NULL};
  /* NPUNCQ, NPUNCR */
  assign p[            420] = {I,   W_NPUNCQR,        CLRXY,        R_NSHRT};
  assign p[            421] = {I,      W_NULL,          ADD,          R_NCW};
  assign p[            422] = {I,      W_NULL,          DIV,         R_NULL};
  /* NSHRTQ, NSHRTR */
  assign p[            423] = {I,   W_NSHRTQR,        CLRXY,         R_NULL};
  assign p[            424] = {I,      W_NULL,         TERM,         R_NULL};
  /* */
  assign p[            425] = {I,      W_NULL,         TERM,         R_NULL};
  /***************************************************************************
  * NULL line because of prefecth
  ***************************************************************************/
  assign p[            426] = {I,     W_NULL,   TERM,          R_NULL}; //
  assign p[            427] = {I,     W_NULL,   TERM,          R_NULL}; //


  assign pr_data  = p[pr_addr];
 
`ifdef RW_SIMU_ON
  /*****************************************************************************
  * for debug/development
  *****************************************************************************/
  // pragma coverage block = off, expr = off, toggle = off
  reg [8:0] op_pc_s;
  reg       op_drop_s;

  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
      op_pc_s   <= 'd0;
      op_drop_s <= 'b0;
    end
    else if(!u_ucpu.enable || u_ucpu.done)
    begin
      op_pc_s   <= 'd0;
      op_drop_s <= 'b0;
    end
    else
    begin
      op_drop_s <= u_ucpu.op_jmp && u_ucpu.cc_flag;
      op_pc_s   <= u_ucpu.pc;
    end

  reg [32*8-1:0] op_wb_s;
  reg [32*8-1:0] op_rb_s;
  reg [32*8-1:0] op_alu_s;
  reg [32*8-1:0] op_cond_s;
  reg [ 8:0]     op_target_s;

  always @(*)
  begin
    op_wb_s     = "XXXXXXXX";
    op_alu_s    = "XXXXXXXX";
    op_rb_s     = "XXXXXXXX";
    op_cond_s   = "XXXXXXXX";
    op_target_s = 0;

    if(!u_ucpu.done)
    begin
      if(op_drop_s)
      begin
        op_wb_s     = " ";
        op_alu_s    = "DROP";
        op_rb_s     = " ";
        op_target_s = 0;
        op_cond_s   = " ";
      end
      else if(u_ucpu.op_jmp)
      begin
        op_wb_s     = " ";
        op_alu_s    = " ";
        op_rb_s     = " ";
        op_target_s = u_ucpu.op_target;
        case(u_ucpu.op_cond)
          CC_LE: op_cond_s = "CC_LE";
          CC_X_ZERO: op_cond_s = "CC_X_ZERO";
          CC_NOT_PEDIS: op_cond_s = "CC_NOT_PEDIS";
          CC_SUBPROG5: op_cond_s = "CC_SUBPROG5";
          CC_SUBPROG10: op_cond_s = "CC_SUBPROG10";
          CC_EQ: op_cond_s = "CC_EQ";
          CC_SUBPROG12: op_cond_s = "CC_SUBPROG12";
          CC_SUBPROG1: op_cond_s = "CC_SUBPROG1";
          CC_SUBPROG11: op_cond_s = "CC_SUBPROG11";
          CC_SUBPROG8: op_cond_s = "CC_SUBPROG8";
          CC_NMA_TEST: op_cond_s = "CC_NMA_TEST";
          CC_SUBPROG7: op_cond_s = "CC_SUBPROG7";
          CC_SUBPROG0: op_cond_s = "CC_SUBPROG0";
          CC_SUBPROG4: op_cond_s = "CC_SUBPROG4";
          CC_SUBPROG6: op_cond_s = "CC_SUBPROG6";
          CC_NOT_SGI: op_cond_s = "CC_NOT_SGI";
          CC_SIGBLEN_NOREM: op_cond_s = "CC_SIGBLEN_NOREM";
          CC_AINIT_NE_4: op_cond_s = "CC_AINIT_NE_4";
          CC_GT: op_cond_s = "CC_GT";
          CC_SUBPROG9: op_cond_s = "CC_SUBPROG9";
          CC_NOT_FEC_EXTRA: op_cond_s = "CC_NOT_FEC_EXTRA";
          CC_TRUE: op_cond_s = "CC_TRUE";
          CC_SUBPROG2: op_cond_s = "CC_SUBPROG2";
          CC_A_EQ_1: op_cond_s = "CC_A_EQ_1";
          CC_AINIT_NE_3: op_cond_s = "CC_AINIT_NE_3";
          CC_SUBPROG13: op_cond_s = "CC_SUBPROG13";
          CC_LT: op_cond_s = "CC_LT";
          CC_GE: op_cond_s = "CC_GE";
          CC_SUBPROG3: op_cond_s = "CC_SUBPROG3";
          default: op_cond_s = "XXXXXX";
        endcase
      end
      else
      begin
        op_cond_s = " ";
        case (u_ucpu.op_wb)
          W_R2: op_wb_s = "W_R2";
          W_LLDPC648: op_wb_s = "W_LLDPC648";
          W_R1: op_wb_s = "W_R1";
          W_AINITEQAM1: op_wb_s = "W_AINITEQAM1";
          W_NREPQR: op_wb_s = "W_NREPQR";
          W_PSDULEN_NPAD: op_wb_s = "W_PSDULEN_NPAD";
          W_NPUNC: op_wb_s = "W_NPUNC";
          W_NSHRTQR: op_wb_s = "W_NSHRTQR";
          W_NDBPS: op_wb_s = "W_NDBPS";
          W_NSHRT: op_wb_s = "W_NSHRT";
          W_INCONS_LT: op_wb_s = "W_INCONS_LT";
          W_NPLD: op_wb_s = "W_NPLD";
          W_SIGBLEN: op_wb_s = "W_SIGBLEN";
          W_NCBPSSHRT: op_wb_s = "W_NCBPSSHRT";
          W_NCW1: op_wb_s = "W_NCW1";
          W_R0: op_wb_s = "W_R0";
          W_MXR2: op_wb_s = "W_MXR2";
          W_NCBPSLAST: op_wb_s = "W_NCBPSLAST";
          W_NCBPS: op_wb_s = "W_NCBPS";
          W_NAVBITS: op_wb_s = "W_NAVBITS";
          W_LLDPC1296: op_wb_s = "W_LLDPC1296";
          W_MXNSYMINIT: op_wb_s = "W_MXNSYMINIT";
          W_TPE: op_wb_s = "W_TPE";
          W_NMA: op_wb_s = "W_NMA";
          W_NULL: op_wb_s = "W_NULL";
          W_AINIT_NEXCESS: op_wb_s = "W_AINIT_NEXCESS";
          W_NCW: op_wb_s = "W_NCW";
          W_NPUNCQR: op_wb_s = "W_NPUNCQR";
          W_EXTRA: op_wb_s = "W_EXTRA";
          W_AINITEQ4: op_wb_s = "W_AINITEQ4";
          W_NDBPSSHRT: op_wb_s = "W_NDBPSSHRT";
          W_NDBPSLAST: op_wb_s = "W_NDBPSLAST";
          W_NSYM: op_wb_s = "W_NSYM";
          W_LLENGTH: op_wb_s = "W_LLENGTH";
          W_TBINIT: op_wb_s = "W_TBINIT";
          W_AEQAINITP1: op_wb_s = "W_AEQAINITP1";
          W_LLDPC1944: op_wb_s = "W_LLDPC1944";
          W_NSYMINIT: op_wb_s = "W_NSYMINIT";
          W_NCW2: op_wb_s = "W_NCW2";
          W_DISAMB: op_wb_s = "W_DISAMB";
          W_INCONS_LE: op_wb_s = "W_INCONS_LE";
          W_R0_X: op_wb_s = "W_R0_X";
          default:             op_wb_s = "XXXXXX";
        endcase
        case(u_ucpu.op_rb)
          R_NSYM: op_rb_s = "R_NSYM";
          R_CRD: op_rb_s = "R_CRD";
          R_PSDULEN: op_rb_s = "R_PSDULEN";
          R_912DIVCRD: op_rb_s = "R_912DIVCRD";
          R_MSTBC: op_rb_s = "R_MSTBC";
          R_648: op_rb_s = "R_648";
          R_1: op_rb_s = "R_1";
          R_10: op_rb_s = "R_10";
          R_MMA: op_rb_s = "R_MMA";
          R_2916DIVCRD: op_rb_s = "R_2916DIVCRD";
          R_NSDSHRT: op_rb_s = "R_NSDSHRT";
          R_MXNCBPS: op_rb_s = "R_MXNCBPS";
          R_MXNDBPS: op_rb_s = "R_MXNDBPS";
          R_CRN: op_rb_s = "R_CRN";
          R_3PM: op_rb_s = "R_3PM";
          R_MXNDBPSSHRT: op_rb_s = "R_MXNDBPSSHRT";
          R_R1: op_rb_s = "R_R1";
          R_NSHRT: op_rb_s = "R_NSHRT";
          R_THELTFSYM: op_rb_s = "R_THELTFSYM";
          R_LLDPC: op_rb_s = "R_LLDPC";
          R_MXNCBPSLAST: op_rb_s = "R_MXNCBPSLAST";
          R_MXNDBPSLAST: op_rb_s = "R_MXNDBPSLAST";
          R_NULL: op_rb_s = "R_NULL";
          R_NSD: op_rb_s = "R_NSD";
          R_1464DIVCRD: op_rb_s = "R_1464DIVCRD";
          R_TVHTPREAMBLE: op_rb_s = "R_TVHTPREAMBLE";
          R_MXNCBPSSHRT: op_rb_s = "R_MXNCBPSSHRT";
          R_NCW: op_rb_s = "R_NCW";
          R_NPLD: op_rb_s = "R_NPLD";
          R_600: op_rb_s = "R_600";
          R_TPE: op_rb_s = "R_TPE";
          R_R0: op_rb_s = "R_R0";
          R_NAVBITS: op_rb_s = "R_NAVBITS";
          R_NSYMINIT: op_rb_s = "R_NSYMINIT";
          R_NCBPSSHRT: op_rb_s = "R_NCBPSSHRT";
          R_R2: op_rb_s = "R_R2";
          R_4: op_rb_s = "R_4";
          R_40: op_rb_s = "R_40";
          R_AINIT: op_rb_s = "R_AINIT";
          R_NDBPSSHRT: op_rb_s = "R_NDBPSSHRT";
          R_TSYMX3: op_rb_s = "R_TSYMX3";
          R_THELTFSYMX3: op_rb_s = "R_THELTFSYMX3";
          R_NCBPS: op_rb_s = "R_NCBPS";
          R_TSYM: op_rb_s = "R_TSYM";
          R_LENGTHX8: op_rb_s = "R_LENGTHX8";
          R_NHELTF: op_rb_s = "R_NHELTF";
          R_ST: op_rb_s = "R_ST";
          R_12XCRN: op_rb_s = "R_12XCRN";
          R_NBPSC: op_rb_s = "R_NBPSC";
          R_NPUNC: op_rb_s = "R_NPUNC";
          R_LLENGTH: op_rb_s = "R_LLENGTH";
          R_1944XCR: op_rb_s = "R_1944XCR";
          R_120: op_rb_s = "R_120";
          R_160_240: op_rb_s = "R_160_240";
          R_3: op_rb_s = "R_3";
          R_LENGTH: op_rb_s = "R_LENGTH";
          R_NDBPS: op_rb_s = "R_NDBPS";
          default:        op_rb_s = "XXXXXX";
        endcase
        case (u_ucpu.op_alu)
          NOP:     op_alu_s = "NOP";
          CLRXY:   op_alu_s = "CLRXY";
          ADD:     op_alu_s = "ADD";
          SUB:     op_alu_s = "SUB";
          MUL:     op_alu_s = "MUL";
          DIV:     op_alu_s = "DIV";
          TERM:    op_alu_s = "TERM";
        endcase
      end
    end
  end
  // pragma coverage block = on, expr = on, toggle = on
`endif
endmodule
