`default_nettype none
/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :                                       
* Description      : FFT engine 
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
********************************************************************************
*  (fft_addr,fft_data) |  2X       |  4X
*  --------------------+-----------+---------------
*  (20M)   FFT512      |  (7,120)  |  (6,240)
*  (40M)   FFT1024     |  (8,120)  |  (7,240)
*  (80M)   FFT2048     |  (9,120)  |  (8,240)
********************************************************************************
* len
*
*  0   fft64
*  1   fft128
*  2   fft256
*  3   fft512
*  4   fft1024
*  5   fft2048
*  6   fft64 x 2
*  7   no fft
*
*******************************************************************************/
module fft1024x4_memmux
(
  /*****************************************************************************
  * system  
  *****************************************************************************/
  input  wire          rst_n,
  input  wire          clk,
  
  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire          enable,
  input  wire          start,
  input  wire          dir,
  input  wire  [  2:0] hisel,
  input  wire  [  2:0] losel,
  input  wire          offset,
  output reg           done,
  output reg           clken,
 
  /*****************************************************************************
  * TX
  *****************************************************************************/
  /* bd write port */
  input  wire          txbd_wen,
  input  wire  [  8:0] txbd_windex,
  input  wire  [ 29:0] txbd_wdata,

  /* bd read port */
  input  wire          txbd_ren,
  input  wire  [  8:0] txbd_rindex,
  output wire  [ 29:0] txbd_rdata,
  output wire          txbd_rvalid,
 
  /* fd write port */
  input  wire          txfd_wen,
  input  wire  [  2:0] txfd_whisel, 
  input  wire  [  2:0] txfd_wlosel, 
  input  wire          txfd_woffset, 
  input  wire  [  2:0] txfd_wlen,
  input  wire  [  3:0] txfd_wscale,
  input  wire  [  9:0] txfd_windex,
  input  wire  [ 12:0] txfd_wi,
  input  wire  [ 12:0] txfd_wq,
  
  /* td read port */
  input  wire          txtd_ren,
  input  wire  [  2:0] txtd_rhisel,
  input  wire  [  2:0] txtd_rlosel,
  input  wire          txtd_roffset,
  input  wire  [  9:0] txtd_rindex,
  output wire  [ 12:0] txtd_ri,
  output wire  [ 12:0] txtd_rq,
  output wire          txtd_rvalid,

  /*****************************************************************************
  * RX
  *****************************************************************************/
  /* bdfd write port */
  input  wire          bdfd_wen,
  input  wire  [ 8:0]  bdfd_windex,
  input  wire  [29:0]  bdfd_wdata,
  
  /* bdfd read port */
  input  wire          bdfd_ren,
  input  wire  [ 8:0]  bdfd_rindex,
  output wire  [29:0]  bdfd_rdata,
  output wire          bdfd_rvalid,
  
  /* td write port */
  input  wire          rxtd_wsel,
  input  wire  [  2:0] rxtd_wlen, 
  input  wire  [  3:0] rxtd_wscale,
  input  wire          rxtd_wen,
  input  wire  [  8:0] rxtd_windex,
  input  wire  [ 12:0] rxtd_wpi,
  input  wire  [ 12:0] rxtd_wpq,
  input  wire  [ 12:0] rxtd_wsi,
  input  wire  [ 12:0] rxtd_wsq,
  
  /* td restricted read port, only buffers 0 and 1 */
  input  wire          rxtd_ren,
  input  wire  [  6:0] rxtd_rindex,
  output wire  [ 12:0] rxtd_rpi0,
  output wire  [ 12:0] rxtd_rpq0,
  output wire  [ 12:0] rxtd_rsi0,
  output wire  [ 12:0] rxtd_rsq0,
  output wire  [ 12:0] rxtd_rpi1,
  output wire  [ 12:0] rxtd_rpq1,
  output wire  [ 12:0] rxtd_rsi1,
  output wire  [ 12:0] rxtd_rsq1,
  output wire          rxtd_rvalid,

  /* read port equ/fdo/chest */
  input  wire          rxfd_rsel,
  input  wire          rxfd_rps,   /* 0=primary,1=secondary for 2x20 only, MUST be 0 for others fft modes */
  input  wire          rxfd_ren,
  input  wire  [  8:0] rxfd_rindex,
  output wire  [ 12:0] rxfd_ri,
  output wire  [ 12:0] rxfd_rq,
  output wire          rxfd_rvalid,

  /* s write port */
  input  wire          s_wen,
  input  wire  [  5:0] s_windex,
  input  wire  [ 12:0] s_wpi,
  input  wire  [ 12:0] s_wpq,
  input  wire  [ 12:0] s_wsi,
  input  wire  [ 12:0] s_wsq,

  /* s restricted read port */
  input  wire          s_ren,
  input  wire  [  5:0] s_rindex,
  output wire  [ 12:0] s_rpi,
  output wire  [ 12:0] s_rpq,
  output wire  [ 12:0] s_rsi,
  output wire  [ 12:0] s_rsq,
  output wire          s_rvalid,

  /* h write port */
  input  wire  [  8:0] h_windex,
  input  wire          h_weni0,
  input  wire          h_wenq0,
  input  wire  [ 12:0] h_wi0,
  input  wire  [ 12:0] h_wq0,
  input  wire          h_weni1,
  input  wire          h_wenq1,
  input  wire  [ 12:0] h_wi1,
  input  wire  [ 12:0] h_wq1,
  input  wire          h_weni2,
  input  wire          h_wenq2,
  input  wire  [ 12:0] h_wi2,
  input  wire  [ 12:0] h_wq2,
  input  wire          h_weni3,
  input  wire          h_wenq3,
  input  wire  [ 12:0] h_wi3,
  input  wire  [ 12:0] h_wq3,
 
  /* h read port */
  input  wire  [  8:0] h_rindex,
  input  wire          h_ren0,
  output wire  [ 12:0] h_ri0,
  output wire  [ 12:0] h_rq0,
  input  wire          h_ren1,
  output wire  [ 12:0] h_ri1,
  output wire  [ 12:0] h_rq1,
  input  wire          h_ren2,
  output wire  [ 12:0] h_ri2,
  output wire  [ 12:0] h_rq2,
  input  wire          h_ren3,
  output wire  [ 12:0] h_ri3,
  output wire  [ 12:0] h_rq3,
  output wire          h_rvalid,
  
  /* stbc write port */
  input  wire          stbc_wen,
  input  wire  [  8:0] stbc_windex,
  input  wire  [ 75:0] stbc_wdata,
  
  /* stbc read port */
  input  wire          stbc_ren,
  input  wire  [  8:0] stbc_rindex,
  output wire  [ 75:0] stbc_rdata,
  output wire          stbc_rvalid,
 
  /*****************************************************************************
  * FFT
  *****************************************************************************/
  output  reg          fft_enable,
  output  reg  [  3:0] fft_scale,
  output  reg  [  2:0] fft_len,
  input   wire         fft_done,
  
  input   wire         fft_wen,
  input   wire [  6:0] fft_waddr,
  input   wire [239:0] fft_wdata,
  
  input   wire         fft_ren,
  input   wire [  6:0] fft_raddr,
  output  wire [239:0] fft_rdata,
  output  reg          fft_rdirty,
  
  /*****************************************************************************
  * MEMORIES
  *****************************************************************************/
  output wire  [ 15:0] mem0_wen,
  output wire  [  5:0] mem0_waddr,
  output wire  [239:0] mem0_wdata,
  output wire          mem0_ren,
  output wire  [  5:0] mem0_raddr,
  input  wire  [239:0] mem0_rdata,

  output wire  [ 15:0] mem1_wen,
  output wire  [  5:0] mem1_waddr,
  output wire  [239:0] mem1_wdata,
  output wire          mem1_ren,
  output wire  [  5:0] mem1_raddr,
  input  wire  [239:0] mem1_rdata,

  output wire  [ 15:0] mem2_wen,
  output wire  [  5:0] mem2_waddr,
  output wire  [239:0] mem2_wdata,
  output wire          mem2_ren,
  output wire  [  5:0] mem2_raddr,
  input  wire  [239:0] mem2_rdata,

  output wire  [ 15:0] mem3_wen,
  output wire  [  5:0] mem3_waddr,
  output wire  [239:0] mem3_wdata,
  output wire          mem3_ren,
  output wire  [  5:0] mem3_raddr,
  input  wire  [239:0] mem3_rdata,
  
  output wire  [ 15:0] mem4_wen,
  output wire  [  5:0] mem4_waddr,
  output wire  [303:0] mem4_wdata,
  output wire          mem4_ren,
  output wire  [  5:0] mem4_raddr,
  input  wire  [303:0] mem4_rdata,

  output wire  [ 15:0] mem5_wen,
  output wire  [  5:0] mem5_waddr,
  output wire  [303:0] mem5_wdata,
  output wire          mem5_ren,
  output wire  [  5:0] mem5_raddr,
  input  wire  [303:0] mem5_rdata
);
 
  /*****************************************************************************
  * ports matrix
  ******************************************************************************
         mem0       mem1       mem2       mem3       mem4       mem5
  
  tx     Y          Y          Y          Y          Y          Y
  rx     Y          Y          
  h                            Y          Y          Y          Y
  s                                                             Y
  
  *****************************************************************************/
 
  /*****************************************************************************
  * declaration
  *****************************************************************************/
  /* fft control */
  reg          state;
  reg [3:0]    clken_count;
  reg [2:0]    fft_hisel;
  reg [2:0]    fft_losel;
  reg          fft_offset;
  reg          fft_sec20_sel;
  reg          fft_sec20_state;
  
  /* sub-carrier write port */
  reg  [  9:0] windex_rev;
  reg  [  7:0] wp_lane;
  reg  [ 15:0] wp_wen;
  reg  [  4:0] wp_sel;
  reg  [  5:0] wp_addr;
  reg  [ 12:0] wp_pq,wp_pi;
  reg  [ 12:0] wp_sq,wp_si;
  wire [239:0] wp_data;
  reg  [  2:0] wp0_len,wp1_len,wp2_len,wp3_len,wp4_len;
  reg  [  3:0] wp0_scale,wp1_scale,wp2_scale,wp3_scale,wp4_scale;
  reg  [ 63:0] wp0_dirty,wp1_dirty,wp2_dirty,wp3_dirty,wp4_dirty;
  reg  [  2:0] wp0_addr_mask,wp1_addr_mask;
  
  /* fft memory select */
  wire [4:0] fwp_sel;
  wire [5:0] fwp_addr;
  wire [4:0] frp_sel;
  wire [5:0] frp_addr;

  /*****************************************************************************
  * static / memory rdata re-registering
  *****************************************************************************/
  reg         static_dir;
  reg [239:0] mem0_rdata_2t,mem1_rdata_2t,mem2_rdata_2t,mem3_rdata_2t;
  reg [303:0] mem4_rdata_2t,mem5_rdata_2t;
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      static_dir    <= 1'b0;
      mem0_rdata_2t <= 240'b0;
      mem1_rdata_2t <= 240'b0;
      mem2_rdata_2t <= 240'b0;
      mem3_rdata_2t <= 240'b0;
      mem4_rdata_2t <= 304'b0;
      mem5_rdata_2t <= 304'b0;
    end
    else
    begin
      static_dir    <= dir;
      mem0_rdata_2t <= mem0_rdata;
      mem1_rdata_2t <= mem1_rdata;
      mem2_rdata_2t <= mem2_rdata;
      mem3_rdata_2t <= mem3_rdata;
      mem4_rdata_2t <= mem4_rdata;
      mem5_rdata_2t <= mem5_rdata;
    end
  end
    
  /*****************************************************************************
  * fft control
  *****************************************************************************/
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      clken           <= 1'b0;
      clken_count     <= 4'd0;
      fft_hisel       <= 3'd0;
      fft_losel       <= 3'd0;
      fft_offset      <= 1'd0;
      fft_sec20_sel   <= 1'b0;
      fft_sec20_state <= 1'b0;
      fft_len         <= 3'b0;
      fft_scale       <= 4'b0;
      fft_enable      <= 1'b0;
      state           <= 1'b0;
      done            <= 1'b0;
    end
    else if(!enable)
    begin
      clken           <= 1'b0;
      clken_count     <= 4'd0;
      fft_hisel       <= 3'd0;
      fft_losel       <= 3'd0;
      fft_offset      <= 1'd0;
      fft_sec20_sel   <= 1'b0;
      fft_sec20_state <= 1'b0;
      fft_len         <= 3'b0;
      fft_scale       <= 4'b0;
      fft_enable      <= 1'b0;
      state           <= 1'b0;
      done            <= 1'b0;
    end
    else
    begin
      /* rtz */
      done <= 2'b0;
      
      if(!state)
      begin
        if(start)
        begin
          state           <= 1'b1;
          clken           <= 1'b1;
          clken_count     <= 4'd0;
          fft_sec20_sel   <= 1'b0;
          fft_sec20_state <= 1'b0;
          fft_enable      <= 1'b0;
        end
      end
      else
      begin
        if(!fft_sec20_state)
        begin
          if(!fft_enable)
          begin
            if(clken_count==4'd6)
            begin
              fft_enable    <= 1'b1;
              fft_hisel     <= hisel;
              fft_losel     <= losel;
              fft_offset    <= offset;
              fft_sec20_sel <= 1'b0;
              case(losel)
                3'd0:    {fft_sec20_state,fft_len,fft_scale} <= { (wp0_len==3'd6)?{1'b1,3'd0}:{1'b0,wp0_len} ,wp0_scale};
                3'd1:    {fft_sec20_state,fft_len,fft_scale} <= { (wp1_len==3'd6)?{1'b1,3'd0}:{1'b0,wp1_len} ,wp1_scale};
                3'd2:    {fft_sec20_state,fft_len,fft_scale} <= { (wp2_len==3'd6)?{1'b1,3'd0}:{1'b0,wp2_len} ,wp2_scale};
                3'd3:    {fft_sec20_state,fft_len,fft_scale} <= { (wp3_len==3'd6)?{1'b1,3'd0}:{1'b0,wp3_len} ,wp3_scale};
                default: {fft_sec20_state,fft_len,fft_scale} <= { (wp4_len==3'd6)?{1'b1,3'd0}:{1'b0,wp4_len} ,wp4_scale};
              endcase
            end
            else
            begin
              clken_count <= clken_count + 4'd1;
            end
          end
          else
          begin
            if(fft_done)
            begin
              clken <= 1'b0;       
              done  <= 1'd1;      
            
              fft_enable <= 1'b0;  
              state      <= 1'b0; 
            end
          end
        end
        else
        begin
          if(!fft_enable)
          begin
            fft_sec20_state <= 1'b0; 
            fft_sec20_sel   <= 1'b1;
            fft_enable      <= 1'b1;
          end
          else
          begin
            if(fft_done)
              fft_enable <= 1'b0;
          end
        end
      end
    end
  end

  /*****************************************************************************
  * rxtd primary/secondary dispatch (second access generator)
  *****************************************************************************/
  reg        rxtd_wen_1t;   
  reg        rxtd_wsel_1t;   
  reg [ 3:0] rxtd_wscale_1t; 
  reg [ 5:0] rxtd_windex_1t; 
  reg [12:0] rxtd_wsi_1t;    
  reg [12:0] rxtd_wsq_1t;      
    
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      rxtd_wen_1t    <= 1'b0;
      rxtd_wsel_1t   <= 1'b0;
      rxtd_wscale_1t <= 4'b0;
      rxtd_windex_1t <= 6'b0;
      rxtd_wsi_1t    <= 13'b0;
      rxtd_wsq_1t    <= 13'b0;
    end
    else if(!enable)
    begin
      rxtd_wen_1t    <= 1'b0;
      rxtd_wsel_1t   <= 1'b0;
      rxtd_wscale_1t <= 4'b0;
      rxtd_windex_1t <= 6'b0;
      rxtd_wsi_1t    <= 13'b0;
      rxtd_wsq_1t    <= 13'b0;
    end
    else
    begin
      if(rxtd_wen && rxtd_wlen==3'd6)
      begin
        rxtd_wen_1t    <= 1'b1;   
        rxtd_wsel_1t   <= rxtd_wsel; 
        rxtd_wscale_1t <= rxtd_wscale;
        rxtd_windex_1t <= rxtd_windex[5:0];
        rxtd_wsi_1t    <= rxtd_wsi;  
        rxtd_wsq_1t    <= rxtd_wsq;
      end
      else
      begin
        rxtd_wen_1t    <= 1'b0;
        rxtd_wsel_1t   <= 1'b0;
        rxtd_wscale_1t <= 4'b0;
        rxtd_windex_1t <= 6'b0;
        rxtd_wsi_1t    <= 13'b0;
        rxtd_wsq_1t    <= 13'b0;
      end
    end
  end
  
  
  /*****************************************************************************
  * fft-able write port 
  *****************************************************************************/
  reg          wen;
  reg   [ 2:0] whisel;
  reg   [ 2:0] wlosel;
  reg          woffset;
  reg   [ 2:0] wlen;
  reg   [ 3:0] wscale;
  reg   [ 9:0] windex;
  reg   [12:0] wi,wq;
  
  always @(*)
  begin
    if(static_dir)
    begin
      /* TX */
      wen    = txfd_wen;
      whisel = txfd_whisel;
      wlosel = txfd_wlosel;
      wlen   = txfd_wlen;
      wscale = txfd_wscale;
      windex = txfd_windex;
      woffset= txfd_woffset;
      wi     = txfd_wi;
      wq     = txfd_wq;
    end
    else
    begin
      /* RX */
      woffset = 1'b0;
      if(!rxtd_wen_1t)
      begin
        /* write primary 20 */
        wen     =  rxtd_wen;
        whisel  =  3'd7;
        wlosel  =  {2'b0,rxtd_wsel}; 
        wlen    =  rxtd_wlen;
        wscale  =  rxtd_wscale;
        windex  =  {1'b0,rxtd_windex};
        wi      =  rxtd_wpi;
        wq      =  rxtd_wpq;      
      end
      else
      begin
        /* write secondary 20 */
        wen     =  1'b1;
        whisel  =  3'd7;
        wlosel  =  {1'b0,1'b0, rxtd_wsel_1t}; 
        wlen    =  3'd6;
        wscale  =  rxtd_wscale_1t;
        windex  =  {3'b0,1'b1,rxtd_windex_1t[5:0]};
        wi      =  rxtd_wsi_1t;
        wq      =  rxtd_wsq_1t;      
      end
    end
  end
  
  /* index mirroring */
  always @(*)
  begin
    case(wlen)  
      3'd0:    windex_rev = {     1'b0,     1'b0,     1'b0,     1'b0,windex[0],windex[1],windex[2],windex[3],windex[4],windex[5]}; /* FFT64     */
      3'd1:    windex_rev = {     1'b0,     1'b0,     1'b0,windex[0],windex[1],windex[2],windex[3],windex[4],windex[5],windex[6]}; /* FFT128    */
      3'd2:    windex_rev = {     1'b0,     1'b0,windex[0],windex[1],windex[2],windex[3],windex[4],windex[5],windex[6],windex[7]}; /* FFT256    */
      3'd3:    windex_rev = {     1'b0,windex[0],windex[1],windex[2],windex[3],windex[4],windex[5],windex[6],windex[7],windex[8]}; /* FFT512    */
      3'd4:    windex_rev = {windex[0],windex[1],windex[2],windex[3],windex[4],windex[5],windex[6],windex[7],windex[8],windex[9]}; /* FFT1024   */
      3'd6:    windex_rev = {     1'b0,     1'b0,     1'b0,windex[6],windex[0],windex[1],windex[2],windex[3],windex[4],windex[5]}; /* FFT64 X2  */
      default: windex_rev = {     1'b0,     1'b0,     1'b0,windex[6],windex[5],windex[4],windex[3],windex[2],windex[1],windex[0]}; /* NO FFT    */                                                                                        /* NO FFT    */
    endcase
  end

  /* fft-able write port re-registering */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      wp_sel                        <= 5'b0;
      wp_wen                        <= 16'b0;
      wp_lane                       <= 8'b0;
      wp_addr                       <= 6'b0;
      {wp_pq,wp_pi}                 <= 26'b0;
      {wp_sq,wp_si}                 <= 26'b0;
      {wp0_scale,wp0_len,wp0_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp1_scale,wp1_len,wp1_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp2_scale,wp2_len,wp2_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp3_scale,wp3_len,wp3_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp4_scale,wp4_len,wp4_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      wp0_addr_mask                 <= 3'b0;
      wp1_addr_mask                 <= 3'b0;
    end
    else if(enable)
    begin
      
      /*************************************************************************
      * RTZ
      *************************************************************************/
      wp_sel   <= 5'b0;
      wp_lane  <= 8'b0;
      wp_addr  <= 6'b0;
      
      /*************************************************************************
      * CLEAR DIRTY TAGS
      *************************************************************************/
      if(fft_done)
      begin
        if(fft_losel==3'd0 || fft_hisel==3'd0)
          wp0_dirty <= {64{1'b1}};
       
        if(fft_losel==3'd1 || fft_hisel==3'd1)
          wp1_dirty <= {64{1'b1}};
      
        if(fft_losel==3'd2 || fft_hisel==3'd2)
          wp2_dirty <= {64{1'b1}};
      
        if(fft_losel==3'd3 || fft_hisel==3'd3)
          wp3_dirty <= {64{1'b1}};
      
        if(fft_losel==3'd4 || fft_hisel==3'd4)
          wp4_dirty <= {64{1'b1}};
      end
      
      /*************************************************************************
      * WRITE ACCESS
      *************************************************************************/
      if(wen)
      begin
        /* WRITE PORT */
        if(wlen!=3'd7)
        begin
          /* FFT */
          wp_addr       <= windex_rev[8:3] | {woffset,5'b0};
          {wp_pq,wp_pi} <= {      wq,       wi};
          {wp_sq,wp_si} <= {      wq,       wi};
          case(windex_rev[2:0])
            3'd0:    {wp_wen,wp_lane} <= {16'b0000000000000011,8'b00000001};
            3'd1:    {wp_wen,wp_lane} <= {16'b0000000000001100,8'b00000010}; 
            3'd2:    {wp_wen,wp_lane} <= {16'b0000000000110000,8'b00000100};  
            3'd3:    {wp_wen,wp_lane} <= {16'b0000000011000000,8'b00001000};  
            3'd4:    {wp_wen,wp_lane} <= {16'b0000001100000000,8'b00010000};  
            3'd5:    {wp_wen,wp_lane} <= {16'b0000110000000000,8'b00100000};  
            3'd6:    {wp_wen,wp_lane} <= {16'b0011000000000000,8'b01000000};  
            default: {wp_wen,wp_lane} <= {16'b1100000000000000,8'b10000000};  
          endcase
        end
        else
        begin
          /* NO FFT */
          wp_addr       <= {1'b0,windex_rev[6:2]};
          {wp_pq,wp_pi} <= {      wq,       wi};
          {wp_sq,wp_si} <= {rxtd_wsq, rxtd_wsi};
          case(windex_rev[1:0])
            2'd0:    {wp_wen,wp_lane} <= {16'b0000000000001111,8'b00000011};
            2'd1:    {wp_wen,wp_lane} <= {16'b0000000011110000,8'b00001100}; 
            2'd2:    {wp_wen,wp_lane} <= {16'b0000111100000000,8'b00110000};  
            default: {wp_wen,wp_lane} <= {16'b1111000000000000,8'b11000000};  
          endcase
        end
        
        /* MEM0 ATTRIBUTES */
        if((wlosel==3'd0 && !windex_rev[9] || whisel==3'd0 && windex_rev[9]) )
        begin
          wp_sel                     <= 5'b00001;
          wp0_len                    <= wlen;
          case(wlen)
            3'd0,3'd6: wp0_addr_mask <= 3'b000;
            3'd1:      wp0_addr_mask <= 3'b001;
            3'd2:      wp0_addr_mask <= 3'b011;
            default:   wp0_addr_mask <= 3'b111;
          endcase
          wp0_scale                  <= wscale;
          wp0_dirty[windex_rev[8:3]] <= 1'b0;
          if(wp0_dirty[windex_rev[8:3]])
            wp_wen <= {16{1'b1}};
        end

        /* MEM1 ATTRIBUTES */
        if((wlosel==3'd1 && !windex_rev[9] || whisel==3'd1 && windex_rev[9]) )
        begin
          wp_sel                     <= 5'b00010;
          wp1_len                    <= wlen;
          case(wlen)
            3'd0,3'd6: wp1_addr_mask <= 3'b000;
            3'd1:      wp1_addr_mask <= 3'b001;
            3'd2:      wp1_addr_mask <= 3'b011;
            default:   wp1_addr_mask <= 3'b111;
          endcase
          wp1_scale                  <= wscale;
          wp1_dirty[windex_rev[8:3]] <= 1'b0;
          if(wp1_dirty[windex_rev[8:3]])
            wp_wen <= {16{1'b1}};
        end

        /* MEM2 ATTRIBUTES */
        if((wlosel==3'd2 && !windex_rev[9] || whisel==3'd2 && windex_rev[9]) )
        begin
          wp_sel                     <= 5'b00100;
          wp2_len                    <= wlen;
          wp2_scale                  <= wscale;
          wp2_dirty[windex_rev[8:3]] <= 1'b0;
          if(wp2_dirty[windex_rev[8:3]])
            wp_wen <= {16{1'b1}};
        end
        
        /* MEM3 ATTRIBUTES */
        if((wlosel==3'd3 && !windex_rev[9] || whisel==3'd3 && windex_rev[9]) )
        begin
          wp_sel                     <= 5'b01000;
          wp3_len                    <= wlen;
          wp3_scale                  <= wscale;
          wp3_dirty[windex_rev[8:3]] <= 1'b0;
          if(wp3_dirty[windex_rev[8:3]])
            wp_wen <= {16{1'b1}};
        end

        /* MEM4 ATTRIBUTES */
        if((wlosel==3'd4 && !windex_rev[9] || whisel==3'd4 && windex_rev[9]) )
        begin
          wp_sel                     <= 5'b10000;
          wp4_len                    <= wlen;
          wp4_scale                  <= wscale;
          wp4_dirty[windex_rev[8:3]] <= 1'b0;
          if(wp4_dirty[windex_rev[8:3]])
            wp_wen <= {16{1'b1}};
        end
      end
      else
      begin
        wp_sel  <= 5'b0;
        wp_lane <= 8'b0;
        wp_wen  <= 16'b0;
      end
    end
    else
    begin
      wp_sel                        <= 5'b0;
      wp_addr                       <= 6'b0;
      wp_lane                       <= 8'b0;
      wp_wen                        <= 16'b0;
      {wp_pi,wp_pq}                 <= 26'b0;
      {wp_si,wp_sq}                 <= 26'b0;
      {wp0_scale,wp0_len,wp0_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp1_scale,wp1_len,wp1_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp2_scale,wp2_len,wp2_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp3_scale,wp3_len,wp3_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      {wp4_scale,wp4_len,wp4_dirty} <= {4'b0,3'b0,{64{1'b1}}}; 
      wp0_addr_mask                 <= 3'b0;
      wp1_addr_mask                 <= 3'b0;
    end
  end
  
  assign wp_data = { 4'b0,wp_sq,wp_si, 4'b0,wp_pq,wp_pi, 4'b0,wp_sq,wp_si, 4'b0,wp_pq,wp_pi,4'b0,wp_sq,wp_si, 4'b0,wp_pq,wp_pi, 4'b0,wp_sq,wp_si, 4'b0,wp_pq,wp_pi} &
                   { {30{wp_lane[7]}}, {30{wp_lane[6]}}, {30{wp_lane[5]}}, {30{wp_lane[4]}},{30{wp_lane[3]}}, {30{wp_lane[2]}}, {30{wp_lane[1]}}, {30{wp_lane[0]}}};
  
  /*****************************************************************************
  * tx read port
  *****************************************************************************/
  wire  [4:0] tx_rp_sel;
  assign tx_rp_sel[0] = txtd_ren & (txtd_rindex[9]?(txtd_rhisel==3'd0):(txtd_rlosel==3'd0));
  assign tx_rp_sel[1] = txtd_ren & (txtd_rindex[9]?(txtd_rhisel==3'd1):(txtd_rlosel==3'd1));
  assign tx_rp_sel[2] = txtd_ren & (txtd_rindex[9]?(txtd_rhisel==3'd2):(txtd_rlosel==3'd2));
  assign tx_rp_sel[3] = txtd_ren & (txtd_rindex[9]?(txtd_rhisel==3'd3):(txtd_rlosel==3'd3));
  assign tx_rp_sel[4] = txtd_ren & (txtd_rindex[9]?(txtd_rhisel==3'd4):(txtd_rlosel==3'd4));
  
  reg  [   4:0] tx_rp_sel_1t,tx_rp_sel_2t;
  reg  [   7:0] tx_rp_lane_1t,tx_rp_lane_2t;
  reg           tx_rp_valid_2t;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      tx_rp_sel_1t   <= 5'b0;
      tx_rp_lane_1t  <= 8'b0;
      tx_rp_sel_2t   <= 5'b0;
      tx_rp_lane_2t  <= 8'b0;
      tx_rp_valid_2t <= 1'b0;
    end
    else
    begin
      case(txtd_rindex[2:0])
        3'd0:    tx_rp_lane_1t <= 8'b00000001;
        3'd1:    tx_rp_lane_1t <= 8'b00000010;
        3'd2:    tx_rp_lane_1t <= 8'b00000100;
        3'd3:    tx_rp_lane_1t <= 8'b00001000;
        3'd4:    tx_rp_lane_1t <= 8'b00010000;
        3'd5:    tx_rp_lane_1t <= 8'b00100000;
        3'd6:    tx_rp_lane_1t <= 8'b01000000;
        default: tx_rp_lane_1t <= 8'b10000000;
      endcase
      tx_rp_lane_2t  <= tx_rp_lane_1t;
      tx_rp_sel_1t   <= tx_rp_sel;
      tx_rp_sel_2t   <= tx_rp_sel_1t;
      tx_rp_valid_2t <= |tx_rp_sel_1t;
    end
  end
  
  wire [ 239:0] tx_rp_data_2t;
  assign tx_rp_data_2t = mem0_rdata_2t & {240{tx_rp_sel_2t[0]}} |
                         mem1_rdata_2t & {240{tx_rp_sel_2t[1]}} |
                         mem2_rdata_2t & {240{tx_rp_sel_2t[2]}} |
                         mem3_rdata_2t & {240{tx_rp_sel_2t[3]}} |
                         {mem4_rdata_2t[295:266],mem4_rdata_2t[257:228],mem4_rdata_2t[219:190],mem4_rdata_2t[181:152],mem4_rdata_2t[143:114],mem4_rdata_2t[105: 76],mem4_rdata_2t[ 67: 38],mem4_rdata_2t[ 29:  0]} & {240{tx_rp_sel_2t[4]}};

  
  assign {txtd_rq,txtd_ri} = {tx_rp_data_2t[ 25:  0]} & {26{tx_rp_lane_2t[0]}} |
                             {tx_rp_data_2t[ 55: 30]} & {26{tx_rp_lane_2t[1]}} |
                             {tx_rp_data_2t[ 85: 60]} & {26{tx_rp_lane_2t[2]}} |
                             {tx_rp_data_2t[115: 90]} & {26{tx_rp_lane_2t[3]}} |
                             {tx_rp_data_2t[145:120]} & {26{tx_rp_lane_2t[4]}} |
                             {tx_rp_data_2t[175:150]} & {26{tx_rp_lane_2t[5]}} |
                             {tx_rp_data_2t[205:180]} & {26{tx_rp_lane_2t[6]}} |
                             {tx_rp_data_2t[235:210]} & {26{tx_rp_lane_2t[7]}};

  assign txtd_rvalid = tx_rp_valid_2t;

  /*****************************************************************************
  * rxfd read port
  *****************************************************************************/
  wire  [1:0] rxfd_rp_sel;
  assign rxfd_rp_sel[0] = rxfd_ren & (rxfd_rsel==1'b0);
  assign rxfd_rp_sel[1] = rxfd_ren & (rxfd_rsel==1'b1);
  
  reg  [   1:0] rxfd_rp_sel_1t,rxfd_rp_sel_2t;
  reg  [   7:0] rxfd_rp_lane_1t,rxfd_rp_lane_2t;
  reg           rxfd_rp_valid_2t;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      rxfd_rp_sel_1t   <= 2'b0;
      rxfd_rp_lane_1t  <= 8'b0;
      rxfd_rp_sel_2t   <= 2'b0;
      rxfd_rp_lane_2t  <= 8'b0;
      rxfd_rp_valid_2t <= 1'b0;
    end
    else
    begin
      case(rxfd_rindex[2:0])
        3'd0:    rxfd_rp_lane_1t <= 8'b00000001;
        3'd1:    rxfd_rp_lane_1t <= 8'b00000010;
        3'd2:    rxfd_rp_lane_1t <= 8'b00000100;
        3'd3:    rxfd_rp_lane_1t <= 8'b00001000;
        3'd4:    rxfd_rp_lane_1t <= 8'b00010000;
        3'd5:    rxfd_rp_lane_1t <= 8'b00100000;
        3'd6:    rxfd_rp_lane_1t <= 8'b01000000;
        default: rxfd_rp_lane_1t <= 8'b10000000;
      endcase
      rxfd_rp_lane_2t  <= rxfd_rp_lane_1t;
      rxfd_rp_sel_1t   <= rxfd_rp_sel;
      rxfd_rp_sel_2t   <= rxfd_rp_sel_1t;
      rxfd_rp_valid_2t <= |rxfd_rp_sel_1t;
    end
  end
  
  wire [ 239:0] rxfd_rp_data_2t;
  assign rxfd_rp_data_2t   = mem0_rdata_2t & {240{rxfd_rp_sel_2t[0]}} |
                             mem1_rdata_2t & {240{rxfd_rp_sel_2t[1]}};
  
  assign {rxfd_rq,rxfd_ri} = {rxfd_rp_data_2t[ 25:  0]} & {26{rxfd_rp_lane_2t[0]}} |
                             {rxfd_rp_data_2t[ 55: 30]} & {26{rxfd_rp_lane_2t[1]}} |
                             {rxfd_rp_data_2t[ 85: 60]} & {26{rxfd_rp_lane_2t[2]}} |
                             {rxfd_rp_data_2t[115: 90]} & {26{rxfd_rp_lane_2t[3]}} |
                             {rxfd_rp_data_2t[145:120]} & {26{rxfd_rp_lane_2t[4]}} |
                             {rxfd_rp_data_2t[175:150]} & {26{rxfd_rp_lane_2t[5]}} |
                             {rxfd_rp_data_2t[205:180]} & {26{rxfd_rp_lane_2t[6]}} |
                             {rxfd_rp_data_2t[235:210]} & {26{rxfd_rp_lane_2t[7]}};

  assign rxfd_rvalid = rxfd_rp_valid_2t;
 
  /*****************************************************************************
  * rxtd read port
  *****************************************************************************/
  reg           rxtd_rp_en_1t,rxtd_rp_en_2t;
  reg  [   3:0] rxtd_rp_lane_1t,rxtd_rp_lane_2t;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      rxtd_rp_en_1t    <= 1'b0;
      rxtd_rp_lane_1t  <= 4'b0;
      rxtd_rp_en_2t    <= 1'b0;
      rxtd_rp_lane_2t  <= 4'b0;
    end
    else
    begin
      case(rxtd_rindex[1:0])
        2'd0:    rxtd_rp_lane_1t <= 4'b0001;
        2'd1:    rxtd_rp_lane_1t <= 4'b0010;
        2'd2:    rxtd_rp_lane_1t <= 4'b0100;
        default: rxtd_rp_lane_1t <= 4'b1000;
      endcase
      rxtd_rp_en_1t    <= rxtd_ren;
      rxtd_rp_en_2t    <= rxtd_rp_en_1t;
      rxtd_rp_lane_2t  <= rxtd_rp_lane_1t;
    end
  end
  
  assign {rxtd_rsq0,rxtd_rsi0,rxtd_rpq0,rxtd_rpi0} = {mem0_rdata_2t[ 55: 30],mem0_rdata_2t[ 25:  0]} & {52{rxtd_rp_lane_2t[0]}} |
                                                     {mem0_rdata_2t[115: 90],mem0_rdata_2t[ 85: 60]} & {52{rxtd_rp_lane_2t[1]}} |
                                                     {mem0_rdata_2t[175:150],mem0_rdata_2t[145:120]} & {52{rxtd_rp_lane_2t[2]}} |
                                                     {mem0_rdata_2t[235:210],mem0_rdata_2t[205:180]} & {52{rxtd_rp_lane_2t[3]}};
 
  assign {rxtd_rsq1,rxtd_rsi1,rxtd_rpq1,rxtd_rpi1} = {mem1_rdata_2t[ 55: 30],mem1_rdata_2t[ 25:  0]} & {52{rxtd_rp_lane_2t[0]}} |
                                                     {mem1_rdata_2t[115: 90],mem1_rdata_2t[ 85: 60]} & {52{rxtd_rp_lane_2t[1]}} |
                                                     {mem1_rdata_2t[175:150],mem1_rdata_2t[145:120]} & {52{rxtd_rp_lane_2t[2]}} |
                                                     {mem1_rdata_2t[235:210],mem1_rdata_2t[205:180]} & {52{rxtd_rp_lane_2t[3]}};
  assign rxtd_rvalid = rxtd_rp_en_2t;

  /*****************************************************************************
  * h write ports
  *****************************************************************************/
  reg [ 3:0]  hwp_sel;
  reg [15:0]  h0wp_lane,h1wp_lane,h2wp_lane,h3wp_lane;
  reg [ 5:0]  hwp_addr;
  reg [12:0]  h0wp_q,h0wp_i;
  reg [12:0]  h1wp_q,h1wp_i;
  reg [12:0]  h2wp_q,h2wp_i;
  reg [12:0]  h3wp_q,h3wp_i;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      hwp_addr        <= 6'b0; 
      hwp_sel         <= 4'b0;
      h0wp_lane       <= 16'b0;
      h1wp_lane       <= 16'b0;
      h2wp_lane       <= 16'b0;
      h3wp_lane       <= 16'b0;
      {h0wp_q,h0wp_i} <= 26'b0;
      {h1wp_q,h1wp_i} <= 26'b0;
      {h2wp_q,h2wp_i} <= 26'b0;
      {h3wp_q,h3wp_i} <= 26'b0;    
    end
    else
    begin
      hwp_addr  <= h_windex[8:3];
      hwp_sel   <= {h_wenq3|h_weni3,h_wenq2|h_weni2,h_wenq1|h_weni1,h_wenq0|h_weni0};
      case(h_windex[2:0])
        3'd0:
        begin
          h0wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq0,h_weni0};   
          h1wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq1,h_weni1};   
          h2wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq2,h_weni2};   
          h3wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq3,h_weni3};   
        end
        3'd1:
        begin
          h0wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq0,h_weni0,            2'b0};   
          h1wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq1,h_weni1,            2'b0};   
          h2wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq2,h_weni2,            2'b0};   
          h3wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq3,h_weni3,            2'b0};   
        end
        3'd2:
        begin
          h0wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq0,h_weni0,            2'b0,            2'b0};   
          h1wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq1,h_weni1,            2'b0,            2'b0};   
          h2wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq2,h_weni2,            2'b0,            2'b0};   
          h3wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0,            2'b0, h_wenq3,h_weni3,            2'b0,            2'b0};   
        end
        3'd3:
        begin
          h0wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0, h_wenq0,h_weni0,            2'b0,            2'b0,            2'b0};   
          h1wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0, h_wenq1,h_weni1,            2'b0,            2'b0,            2'b0};   
          h2wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0, h_wenq2,h_weni2,            2'b0,            2'b0,            2'b0};   
          h3wp_lane <= {            2'b0,            2'b0,            2'b0,            2'b0, h_wenq3,h_weni3,            2'b0,            2'b0,            2'b0};   
        end
        3'd4:
        begin
          h0wp_lane <= {            2'b0,            2'b0,            2'b0, h_wenq0,h_weni0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h1wp_lane <= {            2'b0,            2'b0,            2'b0, h_wenq1,h_weni1,            2'b0,            2'b0,            2'b0,            2'b0};   
          h2wp_lane <= {            2'b0,            2'b0,            2'b0, h_wenq2,h_weni2,            2'b0,            2'b0,            2'b0,            2'b0};   
          h3wp_lane <= {            2'b0,            2'b0,            2'b0, h_wenq3,h_weni3,            2'b0,            2'b0,            2'b0,            2'b0};   
        end
        3'd5:
        begin
          h0wp_lane <= {            2'b0,            2'b0, h_wenq0,h_weni0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h1wp_lane <= {            2'b0,            2'b0, h_wenq1,h_weni1,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h2wp_lane <= {            2'b0,            2'b0, h_wenq2,h_weni2,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h3wp_lane <= {            2'b0,            2'b0, h_wenq3,h_weni3,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
        end
        3'd6:
        begin
          h0wp_lane <= {            2'b0, h_wenq0,h_weni0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h1wp_lane <= {            2'b0, h_wenq1,h_weni1,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h2wp_lane <= {            2'b0, h_wenq2,h_weni2,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h3wp_lane <= {            2'b0, h_wenq3,h_weni3,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
        end
        default:
        begin
          h0wp_lane <= { h_wenq0,h_weni0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h1wp_lane <= { h_wenq1,h_weni1,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h2wp_lane <= { h_wenq2,h_weni2,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
          h3wp_lane <= { h_wenq3,h_weni3,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0,            2'b0};   
        end      
      endcase
      
      {h0wp_q,h0wp_i} <= {h_wq0,h_wi0};
      {h1wp_q,h1wp_i} <= {h_wq1,h_wi1};
      {h2wp_q,h2wp_i} <= {h_wq2,h_wi2};
      {h3wp_q,h3wp_i} <= {h_wq3,h_wi3};
      
    end
  end
  
  wire [239:0]  h0wp_data,h1wp_data; /* mem2 and mem3 have 15 bits wide lane */
  assign h0wp_data = {  2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i,
                        2'b0, h0wp_q,  2'b0, h0wp_i};
                        
  assign h1wp_data = {  2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i,
                        2'b0, h1wp_q,  2'b0, h1wp_i};
  
  
  wire [303:0]  h2wp_data,h3wp_data; /* mem4 and mem5 have 19 bits wide lane */
  assign h2wp_data = {  6'b0, h2wp_q,  6'b0, h2wp_i,
                        6'b0, h2wp_q,  6'b0, h2wp_i,
                        6'b0, h2wp_q,  6'b0, h2wp_i,
                        6'b0, h2wp_q,  6'b0, h2wp_i,
                        6'b0, h2wp_q,  6'b0, h2wp_i,      
                        6'b0, h2wp_q,  6'b0, h2wp_i,      
                        6'b0, h2wp_q,  6'b0, h2wp_i,      
                        6'b0, h2wp_q,  6'b0, h2wp_i};
                        
  assign h3wp_data = {  6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i,
                        6'b0, h3wp_q,  6'b0, h3wp_i};
 
  /*****************************************************************************
  * h read ports
  *****************************************************************************/
  reg  [ 7:0] h_lane_1t,h_lane_2t;
  reg  [ 3:0] h_ren_1t,h_ren_2t;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      h_lane_1t  <= 8'b0;
      h_lane_2t  <= 8'b0; 
      h_ren_1t   <= 4'b0; 
      h_ren_2t   <= 4'b0; 
    end
    else
    begin
      case(h_rindex[2:0])
        3'd0:    h_lane_1t <= 8'b00000001;
        3'd1:    h_lane_1t <= 8'b00000010;
        3'd2:    h_lane_1t <= 8'b00000100;
        3'd3:    h_lane_1t <= 8'b00001000;
        3'd4:    h_lane_1t <= 8'b00010000;
        3'd5:    h_lane_1t <= 8'b00100000;
        3'd6:    h_lane_1t <= 8'b01000000;
        default: h_lane_1t <= 8'b10000000;
      endcase
      h_lane_2t <= h_lane_1t;
      h_ren_1t  <= {h_ren3,h_ren2,h_ren1,h_ren0};
      h_ren_2t  <= h_ren_1t;
    end
  end
 
  assign h_rvalid = |h_ren_2t;
 
  assign {h_rq3,h_ri3,h_rq2,h_ri2,h_rq1,h_ri1,h_rq0,h_ri0} =
         {mem5_rdata_2t[ 31: 19],mem5_rdata_2t[ 12:  0],mem4_rdata_2t[ 31: 19],mem4_rdata_2t[ 12:  0],mem3_rdata_2t[ 27: 15],mem3_rdata_2t[ 12:  0],mem2_rdata_2t[ 27: 15],mem2_rdata_2t[ 12:  0]} & {104{h_lane_2t[0]}} |
         {mem5_rdata_2t[ 69: 57],mem5_rdata_2t[ 50: 38],mem4_rdata_2t[ 69: 57],mem4_rdata_2t[ 50: 38],mem3_rdata_2t[ 57: 45],mem3_rdata_2t[ 42: 30],mem2_rdata_2t[ 57: 45],mem2_rdata_2t[ 42: 30]} & {104{h_lane_2t[1]}} |
         {mem5_rdata_2t[107: 95],mem5_rdata_2t[ 88: 76],mem4_rdata_2t[107: 95],mem4_rdata_2t[ 88: 76],mem3_rdata_2t[ 87: 75],mem3_rdata_2t[ 72: 60],mem2_rdata_2t[ 87: 75],mem2_rdata_2t[ 72: 60]} & {104{h_lane_2t[2]}} |
         {mem5_rdata_2t[145:133],mem5_rdata_2t[126:114],mem4_rdata_2t[145:133],mem4_rdata_2t[126:114],mem3_rdata_2t[117:105],mem3_rdata_2t[102: 90],mem2_rdata_2t[117:105],mem2_rdata_2t[102: 90]} & {104{h_lane_2t[3]}} |
         {mem5_rdata_2t[183:171],mem5_rdata_2t[164:152],mem4_rdata_2t[183:171],mem4_rdata_2t[164:152],mem3_rdata_2t[147:135],mem3_rdata_2t[132:120],mem2_rdata_2t[147:135],mem2_rdata_2t[132:120]} & {104{h_lane_2t[4]}} |
         {mem5_rdata_2t[221:209],mem5_rdata_2t[202:190],mem4_rdata_2t[221:209],mem4_rdata_2t[202:190],mem3_rdata_2t[177:165],mem3_rdata_2t[162:150],mem2_rdata_2t[177:165],mem2_rdata_2t[162:150]} & {104{h_lane_2t[5]}} |
         {mem5_rdata_2t[259:247],mem5_rdata_2t[240:228],mem4_rdata_2t[259:247],mem4_rdata_2t[240:228],mem3_rdata_2t[207:195],mem3_rdata_2t[192:180],mem2_rdata_2t[207:195],mem2_rdata_2t[192:180]} & {104{h_lane_2t[6]}} |
         {mem5_rdata_2t[297:285],mem5_rdata_2t[278:266],mem4_rdata_2t[297:285],mem4_rdata_2t[278:266],mem3_rdata_2t[237:225],mem3_rdata_2t[222:210],mem2_rdata_2t[237:225],mem2_rdata_2t[222:210]} & {104{h_lane_2t[7]}};

  /*****************************************************************************
  * s write port
  *****************************************************************************/
  reg         swp_sel;
  reg [15:0]  swp_lane;
  reg [ 3:0]  swp_addr;
  reg [25:0]  s_wp,s_ws;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      swp_sel  <= 1'b0;
      swp_addr <= 4'b0; 
      swp_lane <= 16'b0;
      s_wp     <= 26'b0; 
      s_ws     <= 26'b0; 
    end
    else
    begin
      swp_sel   <= s_wen;
      swp_addr  <= s_windex[5:2];
      case(s_windex[1:0])
        2'd0:    swp_lane <= 16'b0000000000001111;
        2'd1:    swp_lane <= 16'b0000000011110000;
        2'd2:    swp_lane <= 16'b0000111100000000;
        default: swp_lane <= 16'b1111000000000000;
      endcase
      s_wp  <= {s_wpq,s_wpi};
      s_ws  <= {s_wsq,s_wsi};
   end
  end

  wire [239:0]  swp_data;
  
  assign swp_data = { 4'b0,s_ws, 4'b0,s_wp, 4'b0,s_ws, 4'b0,s_wp, 4'b0,s_ws, 4'b0,s_wp, 4'b0,s_ws, 4'b0,s_wp};

  /*****************************************************************************
  * s read port
  *****************************************************************************/
  reg  [ 3:0] s_lane_1t,s_lane_2t;
  reg         s_ren_1t,s_ren_2t;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s_lane_1t  <= 4'b0;
      s_lane_2t  <= 4'b0; 
      s_ren_1t   <= 1'b0; 
      s_ren_2t   <= 1'b0; 
    end
    else
    begin
      case(s_rindex[1:0])
        2'd0:    s_lane_1t <= 4'b0001;
        2'd1:    s_lane_1t <= 4'b0010;
        2'd2:    s_lane_1t <= 4'b0100;
        default: s_lane_1t <= 4'b1000;
      endcase
      s_lane_2t <= s_lane_1t;
      s_ren_1t  <= s_ren;
      s_ren_2t  <= s_ren_1t;
    end
  end
 
  assign s_rvalid = s_ren_2t;
 
  assign {s_rsq,s_rsi,s_rpq,s_rpi} = {mem5_rdata_2t[ 63: 38],mem5_rdata_2t[ 25:  0]} & {52{s_lane_2t[0]}} |
                                     {mem5_rdata_2t[139:114],mem5_rdata_2t[101: 76]} & {52{s_lane_2t[1]}} |
                                     {mem5_rdata_2t[215:190],mem5_rdata_2t[177:152]} & {52{s_lane_2t[2]}} |
                                     {mem5_rdata_2t[291:266],mem5_rdata_2t[253:228]} & {52{s_lane_2t[3]}};

  /*****************************************************************************
  * txbd write port
  *****************************************************************************/
  reg         txbdwp_sel;
  reg [15:0]  txbdwp_lane;
  reg [ 5:0]  txbdwp_addr;
  reg [29:0]  txbdwp_ldata;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      txbdwp_sel   <= 1'b0;
      txbdwp_addr  <= 6'b0; 
      txbdwp_lane  <= 16'b0;
      txbdwp_ldata <= 30'b0; 
    end
    else
    begin
      txbdwp_sel   <= txbd_wen;
      txbdwp_addr  <= txbd_windex[8:3];
      case(txbd_windex[2:0])
        3'd0:    txbdwp_lane <= 16'b0000000000000011;
        3'd1:    txbdwp_lane <= 16'b0000000000001100;
        3'd2:    txbdwp_lane <= 16'b0000000000110000;
        3'd3:    txbdwp_lane <= 16'b0000000011000000;
        3'd4:    txbdwp_lane <= 16'b0000001100000000;
        3'd5:    txbdwp_lane <= 16'b0000110000000000;
        3'd6:    txbdwp_lane <= 16'b0011000000000000;
        default: txbdwp_lane <= 16'b1100000000000000;
      endcase
      txbdwp_ldata <= txbd_wdata;
   end
  end

  wire [239:0]  txbdwp_data;
  assign txbdwp_data = {txbdwp_ldata,txbdwp_ldata,txbdwp_ldata,txbdwp_ldata,txbdwp_ldata,txbdwp_ldata,txbdwp_ldata,txbdwp_ldata};

  /*****************************************************************************
  * txbd read port
  *****************************************************************************/
  reg  [ 7:0] txbd_lane_1t,txbd_lane_2t;
  reg         txbd_ren_1t,txbd_ren_2t;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      txbd_lane_1t  <= 8'b0;
      txbd_lane_2t  <= 8'b0; 
      txbd_ren_1t   <= 1'b0; 
      txbd_ren_2t   <= 1'b0; 
    end
    else
    begin
      case(txbd_rindex[2:0])
        3'd0:    txbd_lane_1t <= 8'b00000001;
        3'd1:    txbd_lane_1t <= 8'b00000010;
        3'd2:    txbd_lane_1t <= 8'b00000100;
        3'd3:    txbd_lane_1t <= 8'b00001000;
        3'd4:    txbd_lane_1t <= 8'b00010000;
        3'd5:    txbd_lane_1t <= 8'b00100000;
        3'd6:    txbd_lane_1t <= 8'b01000000;
        default: txbd_lane_1t <= 8'b10000000;
      endcase
      txbd_lane_2t <= txbd_lane_1t;
      txbd_ren_1t  <= txbd_ren;
      txbd_ren_2t  <= txbd_ren_1t;
    end
  end
 
  assign txbd_rvalid = txbd_ren_2t;
 
  assign txbd_rdata  = mem5_rdata_2t[ 29:  0] & {30{txbd_lane_2t[0]}} |
                       mem5_rdata_2t[ 67: 38] & {30{txbd_lane_2t[1]}} |
                       mem5_rdata_2t[105: 76] & {30{txbd_lane_2t[2]}} |
                       mem5_rdata_2t[143:114] & {30{txbd_lane_2t[3]}} |
                       mem5_rdata_2t[181:152] & {30{txbd_lane_2t[4]}} |
                       mem5_rdata_2t[219:190] & {30{txbd_lane_2t[5]}} |
                       mem5_rdata_2t[257:228] & {30{txbd_lane_2t[6]}} |
                       mem5_rdata_2t[295:266] & {30{txbd_lane_2t[7]}};

  /*****************************************************************************
  * bdfd write port (can address mem4,mem3 via windex[8])
  *****************************************************************************/
  reg [ 1:0]  bdfdwp_sel;
  reg [15:0]  bdfdwp_lane;
  reg [ 4:0]  bdfdwp_addr;
  reg [29:0]  bdfdwp_data;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      bdfdwp_sel  <= 2'b0;
      bdfdwp_addr <= 5'b0; 
      bdfdwp_lane <= 16'b0;
      bdfdwp_data <= 30'b0; 
    end
    else
    begin
      bdfdwp_sel[0] <=  ~bdfd_windex[8] & bdfd_wen;
      bdfdwp_sel[1] <=   bdfd_windex[8] & bdfd_wen;
      bdfdwp_addr   <=   bdfd_windex[7:3];
      case(bdfd_windex[2:0])
        3'd0:    bdfdwp_lane <= 16'b0000000000000011;
        3'd1:    bdfdwp_lane <= 16'b0000000000001100;
        3'd2:    bdfdwp_lane <= 16'b0000000000110000;
        3'd3:    bdfdwp_lane <= 16'b0000000011000000;
        3'd4:    bdfdwp_lane <= 16'b0000001100000000;
        3'd5:    bdfdwp_lane <= 16'b0000110000000000;
        3'd6:    bdfdwp_lane <= 16'b0011000000000000;
        default: bdfdwp_lane <= 16'b1100000000000000;
      endcase
      bdfdwp_data <= bdfd_wdata;
   end
  end

  wire [239:0] bdfdwp_ldata;
  assign bdfdwp_ldata = {bdfdwp_data,bdfdwp_data,bdfdwp_data,bdfdwp_data,bdfdwp_data,bdfdwp_data,bdfdwp_data,bdfdwp_data};

  /*****************************************************************************
  * bdfd read port
  *****************************************************************************/
  reg  [ 7:0] bdfdrp_lane_1t,bdfdrp_lane_2t;
  wire [ 1:0] bdfdrp_sel;
  reg  [ 1:0] bdfdrp_sel_1t,bdfdrp_sel_2t;
  
  assign bdfdrp_sel = {2{bdfd_ren}} & { bdfd_rindex[8], ~bdfd_rindex[8] };
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      bdfdrp_lane_1t  <= 8'b0;
      bdfdrp_lane_2t  <= 8'b0; 
      bdfdrp_sel_1t   <= 2'b0; 
      bdfdrp_sel_2t   <= 2'b0; 
    end
    else
    begin
      case(bdfd_rindex[2:0])
        3'd0:    bdfdrp_lane_1t <= 8'b00000001;
        3'd1:    bdfdrp_lane_1t <= 8'b00000010;
        3'd2:    bdfdrp_lane_1t <= 8'b00000100;
        3'd3:    bdfdrp_lane_1t <= 8'b00001000;
        3'd4:    bdfdrp_lane_1t <= 8'b00010000;
        3'd5:    bdfdrp_lane_1t <= 8'b00100000;
        3'd6:    bdfdrp_lane_1t <= 8'b01000000;
        default: bdfdrp_lane_1t <= 8'b10000000;
      endcase
      bdfdrp_lane_2t <= bdfdrp_lane_1t;
      bdfdrp_sel_1t  <= bdfdrp_sel;
      bdfdrp_sel_2t  <= bdfdrp_sel_1t;
    end
  end
 
  assign bdfd_rvalid = |bdfdrp_sel_2t;
 
  assign bdfd_rdata  = mem4_rdata_2t[ 29:  0] & {30{bdfdrp_lane_2t[0] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[ 67: 38] & {30{bdfdrp_lane_2t[1] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[105: 76] & {30{bdfdrp_lane_2t[2] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[143:114] & {30{bdfdrp_lane_2t[3] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[181:152] & {30{bdfdrp_lane_2t[4] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[219:190] & {30{bdfdrp_lane_2t[5] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[257:228] & {30{bdfdrp_lane_2t[6] & bdfdrp_sel_2t[0]}} |
                       mem4_rdata_2t[295:266] & {30{bdfdrp_lane_2t[7] & bdfdrp_sel_2t[0]}} |    
                       
                       mem3_rdata_2t[ 29:  0] & {30{bdfdrp_lane_2t[0] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[ 59: 30] & {30{bdfdrp_lane_2t[1] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[ 89: 60] & {30{bdfdrp_lane_2t[2] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[119: 90] & {30{bdfdrp_lane_2t[3] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[149:120] & {30{bdfdrp_lane_2t[4] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[179:150] & {30{bdfdrp_lane_2t[5] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[209:180] & {30{bdfdrp_lane_2t[6] & bdfdrp_sel_2t[1]}} |
                       mem3_rdata_2t[239:210] & {30{bdfdrp_lane_2t[7] & bdfdrp_sel_2t[1]}};

  /*****************************************************************************
  * stbc write port (can address mem4,mem5 via windex[8])
  *****************************************************************************/
  reg [ 1:0]  stbcwp_sel;
  reg [15:0]  stbcwp_lane;
  reg [ 5:0]  stbcwp_addr;
  reg [75:0]  stbcwp_data;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      stbcwp_sel  <= 2'b0;
      stbcwp_addr <= 6'b0; 
      stbcwp_lane <= 16'b0;
      stbcwp_data <= 76'b0; 
    end
    else
    begin
      stbcwp_sel[0] <= ~stbc_windex[8] & stbc_wen;
      stbcwp_sel[1] <=  stbc_windex[8] & stbc_wen;
      stbcwp_addr   <=  stbc_windex[7:2];
      case(stbc_windex[1:0])
        2'd0:    stbcwp_lane <= 16'b0000000000001111;
        2'd1:    stbcwp_lane <= 16'b0000000011110000;
        2'd2:    stbcwp_lane <= 16'b0000111100000000;
        default: stbcwp_lane <= 16'b1111000000000000;
      endcase
      stbcwp_data <= stbc_wdata;
   end
  end

  wire [303:0] stbcwp_ldata;
  assign stbcwp_ldata = {stbcwp_data,stbcwp_data,stbcwp_data,stbcwp_data};
  
  /*****************************************************************************
  * stbc read port
  *****************************************************************************/
  reg  [ 3:0] stbcrp_lane_1t,stbcrp_lane_2t;
  wire [ 1:0] stbcrp_sel;
  reg  [ 1:0] stbcrp_sel_1t,stbcrp_sel_2t;
  
  assign stbcrp_sel = {2{stbc_ren}} & { stbc_rindex[8], ~stbc_rindex[8]};
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      stbcrp_lane_1t  <= 8'b0;
      stbcrp_lane_2t  <= 8'b0; 
      stbcrp_sel_1t   <= 2'b0; 
      stbcrp_sel_2t   <= 2'b0; 
    end
    else
    begin
      case(stbc_rindex[1:0])
        2'd0:    stbcrp_lane_1t <= 4'b0001;
        2'd1:    stbcrp_lane_1t <= 4'b0010;
        2'd2:    stbcrp_lane_1t <= 4'b0100;
        default: stbcrp_lane_1t <= 4'b1000;
      endcase
      stbcrp_lane_2t <= stbcrp_lane_1t;
      stbcrp_sel_1t  <= stbcrp_sel;
      stbcrp_sel_2t  <= stbcrp_sel_1t;
    end
  end
 
  assign stbc_rvalid = |stbcrp_sel_2t;
 
  assign stbc_rdata  = mem4_rdata_2t[ 75:  0] & {76{stbcrp_lane_2t[0] & stbcrp_sel_2t[0]}} |
                       mem4_rdata_2t[151: 76] & {76{stbcrp_lane_2t[1] & stbcrp_sel_2t[0]}} |
                       mem4_rdata_2t[227:152] & {76{stbcrp_lane_2t[2] & stbcrp_sel_2t[0]}} |
                       mem4_rdata_2t[303:228] & {76{stbcrp_lane_2t[3] & stbcrp_sel_2t[0]}} |
                       
                       mem5_rdata_2t[ 75:  0] & {76{stbcrp_lane_2t[0] & stbcrp_sel_2t[1]}} |
                       mem5_rdata_2t[151: 76] & {76{stbcrp_lane_2t[1] & stbcrp_sel_2t[1]}} |
                       mem5_rdata_2t[227:152] & {76{stbcrp_lane_2t[2] & stbcrp_sel_2t[1]}} |
                       mem5_rdata_2t[303:228] & {76{stbcrp_lane_2t[3] & stbcrp_sel_2t[1]}};

  /*****************************************************************************
  * memory muxes
  *****************************************************************************/
  /*  read port */
  assign {mem0_ren,mem0_raddr}            = { 1'b1, { 2'b0,rxfd_rps,3'b0} | rxfd_rindex[8:3]&{wp0_addr_mask,3'b111}} & {7{   rxfd_rp_sel[0]}} | 
                                            { 1'b1,                        {txtd_roffset,5'b0} |   txtd_rindex[8:3]} & {7{     tx_rp_sel[0]}} |
                                            { 1'b1,                                                        frp_addr} & {7{       frp_sel[0]}} |
                                            { 1'b1,                                           1'b0,rxtd_rindex[6:2]} & {7{         rxtd_ren}};

  assign {mem1_ren,mem1_raddr}            = { 1'b1, { 2'b0,rxfd_rps,3'b0} | rxfd_rindex[8:3]&{wp1_addr_mask,3'b111}} & {7{   rxfd_rp_sel[1]}} | 
                                            { 1'b1,                        {txtd_roffset,5'b0} |   txtd_rindex[8:3]} & {7{     tx_rp_sel[1]}} |
                                            { 1'b1,                                                        frp_addr} & {7{       frp_sel[1]}} |
                                            { 1'b1,                                           1'b0,rxtd_rindex[6:2]} & {7{         rxtd_ren}}; 
                                            
  assign {mem2_ren,mem2_raddr}            = { 1'b1,                                                   h_rindex[8:3]} & {7{           h_ren0}} | 
                                            { 1'b1,                        {txtd_roffset,5'b0} |   txtd_rindex[8:3]} & {7{     tx_rp_sel[2]}} |
                                            { 1'b1,                                                        frp_addr} & {7{       frp_sel[2]}}; 

  assign {mem3_ren,mem3_raddr}            = { 1'b1,                                                   h_rindex[8:3]} & {7{           h_ren1}} | 
                                            { 1'b1,                        {txtd_roffset,5'b0} |   txtd_rindex[8:3]} & {7{     tx_rp_sel[3]}} |
                                            { 1'b1,                                                        frp_addr} & {7{       frp_sel[3]}} |
                                            { 1'b1,                                         {1'b0,bdfd_rindex[7:3]}} & {7{    bdfdrp_sel[1]}};

  assign {mem4_ren,mem4_raddr}            = { 1'b1,                                                   h_rindex[8:3]} & {7{           h_ren2}} | 
                                            { 1'b1,                        {txtd_roffset,5'b0} |   txtd_rindex[8:3]} & {7{     tx_rp_sel[4]}} |
                                            { 1'b1,                                                        frp_addr} & {7{       frp_sel[4]}} | 
                                            { 1'b1,                                         {1'b0,bdfd_rindex[7:3]}} & {7{    bdfdrp_sel[0]}} |
                                            { 1'b1,                                                stbc_rindex[7:2]} & {7{    stbcrp_sel[0]}};

  assign {mem5_ren,mem5_raddr}            = { 1'b1,                                                   h_rindex[8:3]} & {7{           h_ren3}} | 
                                            { 1'b1,                                                txbd_rindex[8:3]} & {7{         txbd_ren}} | 
                                            { 1'b1,                                              2'b0,s_rindex[5:2]} & {7{            s_ren}} |
                                            { 1'b1,                                                stbc_rindex[7:2]} & {7{    stbcrp_sel[1]}};
 
  /* write port */
  assign {mem0_wen,mem0_waddr,mem0_wdata} = {     wp_wen,                                      wp_addr,     wp_data} & {262{    wp_sel[0]}} |
                                            { {16{1'b1}},                                     fwp_addr,   fft_wdata} & {262{   fwp_sel[0]}};

  assign {mem1_wen,mem1_waddr,mem1_wdata} = {     wp_wen,                                      wp_addr,     wp_data} & {262{    wp_sel[1]}} |
                                            { {16{1'b1}},                                     fwp_addr,   fft_wdata} & {262{   fwp_sel[1]}};

  assign {mem2_wen,mem2_waddr,mem2_wdata} = {  h0wp_lane,                                     hwp_addr,   h0wp_data} & {262{   hwp_sel[0]}} |
                                            {     wp_wen,                                      wp_addr,     wp_data} & {262{    wp_sel[2]}} |
                                            { {16{1'b1}},                                     fwp_addr,   fft_wdata} & {262{   fwp_sel[2]}};

  assign {mem3_wen,mem3_waddr,mem3_wdata} = {  h1wp_lane,                                     hwp_addr,   h1wp_data} & {262{   hwp_sel[1]}} |
                                            {     wp_wen,                                      wp_addr,     wp_data} & {262{    wp_sel[3]}} |
                                            { {16{1'b1}},                                     fwp_addr,   fft_wdata} & {262{   fwp_sel[3]}} |
                                            {bdfdwp_lane,                           {1'b0,bdfdwp_addr},bdfdwp_ldata} & {262{bdfdwp_sel[1]}};

  assign {mem4_wen,mem4_waddr,mem4_wdata} = {  h2wp_lane,           hwp_addr,   h2wp_data} & {326{   hwp_sel[2]}} |       
                                            {     wp_wen,            wp_addr,   {8'b0,     wp_data[239:210],8'b0,     wp_data[209:180],8'b0,     wp_data[179:150],8'b0,     wp_data[149:120],8'b0,     wp_data[119: 90],8'b0,     wp_data[ 89: 60],8'b0,     wp_data[ 59: 30],8'b0,     wp_data[ 29:  0]}} & {326{    wp_sel[4]}} |       
                                            { {16{1'b1}},           fwp_addr,   {8'b0,   fft_wdata[239:210],8'b0,   fft_wdata[209:180],8'b0,   fft_wdata[179:150],8'b0,   fft_wdata[149:120],8'b0,   fft_wdata[119: 90],8'b0,   fft_wdata[ 89: 60],8'b0,   fft_wdata[ 59: 30],8'b0,   fft_wdata[ 29:  0]}} & {326{   fwp_sel[4]}} |
                                            {bdfdwp_lane,{1'b0, bdfdwp_addr},   {8'b0,bdfdwp_ldata[239:210],8'b0,bdfdwp_ldata[209:180],8'b0,bdfdwp_ldata[179:150],8'b0,bdfdwp_ldata[149:120],8'b0,bdfdwp_ldata[119: 90],8'b0,bdfdwp_ldata[ 89: 60],8'b0,bdfdwp_ldata[ 59: 30],8'b0,bdfdwp_ldata[ 29:  0]}} & {326{bdfdwp_sel[0]}} |                    
                                            {stbcwp_lane,        stbcwp_addr,   stbcwp_ldata} & {326{stbcwp_sel[0]}};                    

  assign {mem5_wen,mem5_waddr,mem5_wdata} = {   swp_lane,      2'b0,swp_addr,   {8'b0,    swp_data[239:210],8'b0,    swp_data[209:180],8'b0,    swp_data[179:150],8'b0,    swp_data[149:120],8'b0,    swp_data[119: 90],8'b0,    swp_data[ 89: 60],8'b0,    swp_data[ 59: 30],8'b0,    swp_data[ 29:  0]}} & {326{      swp_sel}} |       
                                            {  h3wp_lane,           hwp_addr,   h3wp_data} & {326{   hwp_sel[3]}} |       
                                            {txbdwp_lane,        txbdwp_addr,   {8'b0, txbdwp_data[239:210],8'b0, txbdwp_data[209:180],8'b0, txbdwp_data[179:150],8'b0, txbdwp_data[149:120],8'b0, txbdwp_data[119: 90],8'b0, txbdwp_data[ 89: 60],8'b0, txbdwp_data[ 59: 30],8'b0, txbdwp_data[ 29:  0]}} & {326{   txbdwp_sel}} |     
                                            {stbcwp_lane,        stbcwp_addr,   stbcwp_ldata} & {326{stbcwp_sel[1]}};                    

  /*****************************************************************************
  * fft muxes
  *****************************************************************************/
  /* fft write port */
  assign fwp_addr   = fft_waddr[5:0] | {2'b0,fft_sec20_sel,3'b0} | {fft_offset,5'b0};
  assign fwp_sel[0] = fft_wen & (fft_waddr[6]?(fft_hisel==3'd0):(fft_losel==3'd0));
  assign fwp_sel[1] = fft_wen & (fft_waddr[6]?(fft_hisel==3'd1):(fft_losel==3'd1));
  assign fwp_sel[2] = fft_wen & (fft_waddr[6]?(fft_hisel==3'd2):(fft_losel==3'd2));
  assign fwp_sel[3] = fft_wen & (fft_waddr[6]?(fft_hisel==3'd3):(fft_losel==3'd3));
  assign fwp_sel[4] = fft_wen & (fft_waddr[6]?(fft_hisel==3'd4):(fft_losel==3'd4));
  
  /* fft read port */
  assign frp_addr   = fft_raddr[5:0] | {2'b0,fft_sec20_sel,3'b0} | {fft_offset,5'b0};
  assign frp_sel[0] = fft_ren & (fft_raddr[6]?(fft_hisel==3'd0):(fft_losel==3'd0));
  assign frp_sel[1] = fft_ren & (fft_raddr[6]?(fft_hisel==3'd1):(fft_losel==3'd1));
  assign frp_sel[2] = fft_ren & (fft_raddr[6]?(fft_hisel==3'd2):(fft_losel==3'd2));
  assign frp_sel[3] = fft_ren & (fft_raddr[6]?(fft_hisel==3'd3):(fft_losel==3'd3));
  assign frp_sel[4] = fft_ren & (fft_raddr[6]?(fft_hisel==3'd4):(fft_losel==3'd4));
  
  /* fft read port re-registering */
  reg  [  4:0] frp_sel_1t,frp_sel_2t;
  reg  [  5:0] frp_addr_1t;
  
  assign fft_rdata = mem0_rdata_2t & {240{frp_sel_2t[0]}} |
                     mem1_rdata_2t & {240{frp_sel_2t[1]}} |
                     mem2_rdata_2t & {240{frp_sel_2t[2]}} |
                     mem3_rdata_2t & {240{frp_sel_2t[3]}} |
                     {mem4_rdata_2t[295:266],mem4_rdata_2t[257:228],mem4_rdata_2t[219:190],mem4_rdata_2t[181:152],mem4_rdata_2t[143:114],mem4_rdata_2t[105: 76],mem4_rdata_2t[ 67: 38],mem4_rdata_2t[ 29:  0]} & {240{frp_sel_2t[4]}};

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      frp_sel_1t  <= 5'b0;
      frp_sel_2t  <= 5'b0;
      frp_addr_1t <= 6'b0;
      fft_rdirty  <= 1'b0;
    end
    else
    begin
      frp_sel_1t  <= frp_sel;
      frp_sel_2t  <= frp_sel_1t;
     
      if(dir)
      begin
        frp_addr_1t <= fft_raddr[5:0];
        case(frp_sel_1t)
          5'b00001: fft_rdirty <= wp0_dirty[frp_addr_1t];
          5'b00010: fft_rdirty <= wp1_dirty[frp_addr_1t];
          5'b00100: fft_rdirty <= wp2_dirty[frp_addr_1t];
          5'b01000: fft_rdirty <= wp3_dirty[frp_addr_1t];
          5'b10000: fft_rdirty <= wp4_dirty[frp_addr_1t];
          default:  fft_rdirty <= 1'b0;
        endcase
      end
      else
      begin
        fft_rdirty <= 1'b0;
      end
    end
  end

`ifdef RW_SIMU_ON
  /*****************************************************************************
  * SIMU ONLY
  *****************************************************************************/
  // pragma coverage block = off, expr = off, toggle = off
  reg [ 2:0] rp_collision0,wp_collision0;
  reg [ 2:0] rp_collision1,wp_collision1;
  reg [ 2:0] rp_collision2,wp_collision2;
  reg [ 2:0] rp_collision3,wp_collision3;
  reg [ 2:0] rp_collision4,wp_collision4;
  reg [ 2:0] rp_collision5,wp_collision5;
  
  always @(posedge clk)
  begin
    rp_collision0 =  {2'b0,frp_sel[0]} + {2'b0,tx_rp_sel[0]} + {2'b0,rxfd_rp_sel[0]} + {2'b0,     rxtd_ren}; 
    rp_collision1 =  {2'b0,frp_sel[1]} + {2'b0,tx_rp_sel[1]} + {2'b0,rxfd_rp_sel[1]} + {2'b0,     rxtd_ren}; 
    rp_collision2 =  {2'b0,frp_sel[2]} + {2'b0,tx_rp_sel[2]} + {2'b0,        h_ren0};
    rp_collision3 =  {2'b0,frp_sel[3]} + {2'b0,tx_rp_sel[3]} + {2'b0,        h_ren1} + {2'b0,bdfdrp_sel[1]};
    rp_collision4 =  {2'b0,frp_sel[4]} + {2'b0,tx_rp_sel[4]} + {2'b0,        h_ren2} + {2'b0,bdfdrp_sel[0]} + {2'b0,stbcrp_sel[0]};
    rp_collision5 =  {2'b0,    h_ren3} + {2'b0,       s_ren} + {2'b0,      txbd_ren} + {2'b0,stbcrp_sel[1]};
    
    if(rp_collision0>3'd1) $write("assert @%t:%m:collision on mem0 read port !\n",$time);
    if(rp_collision1>3'd1) $write("assert @%t:%m:collision on mem1 read port !\n",$time);
    if(rp_collision2>3'd1) $write("assert @%t:%m:collision on mem2 read port !\n",$time);
    if(rp_collision3>3'd1) $write("assert @%t:%m:collision on mem3 read port !\n",$time);
    if(rp_collision4>3'd1) $write("assert @%t:%m:collision on mem4 read port !\n",$time);
    if(rp_collision5>3'd1) $write("assert @%t:%m:collision on mem5 read port !\n",$time);
      
    wp_collision0 = {2'b0, wp_sel[0]} + {2'b0,fwp_sel[0]};
    wp_collision1 = {2'b0, wp_sel[1]} + {2'b0,fwp_sel[1]};
    wp_collision2 = {2'b0, wp_sel[2]} + {2'b0,fwp_sel[2]} + {2'b0, hwp_sel[0]};
    wp_collision3 = {2'b0, wp_sel[3]} + {2'b0,fwp_sel[3]} + {2'b0, hwp_sel[1]} + {2'b0, bdfdwp_sel[1]};
    wp_collision4 = {2'b0, wp_sel[4]} + {2'b0,fwp_sel[4]} + {2'b0, hwp_sel[2]} + {2'b0, bdfdwp_sel[0]} + {2'b0, stbcwp_sel[0]};
    wp_collision5 = {2'b0,hwp_sel[3]} + {2'b0,   swp_sel} + {2'b0, txbdwp_sel} + {2'b0, stbcwp_sel[1]};
    
    if(wp_collision0>3'd1) $write("assert @%t:%m:collision on mem0 write port !\n",$time);               
    if(wp_collision1>3'd1) $write("assert @%t:%m:collision on mem1 write port !\n",$time);               
    if(wp_collision2>3'd1) $write("assert @%t:%m:collision on mem2 write port !\n",$time);               
    if(wp_collision3>3'd1) $write("assert @%t:%m:collision on mem3 write port !\n",$time);               
    if(wp_collision4>3'd1) $write("assert @%t:%m:collision on mem4 write port !\n",$time);               
    if(wp_collision5>3'd1) $write("assert @%t:%m:collision on mem5 write port !\n",$time); 
    
    if(rxtd_wen && rxtd_wen_1t) $write("assert @%t:%m: rxtd idle state violation !\n",$time);    
    
  end
  // pragma coverage block = on, expr = on, toggle = on
`endif

endmodule
`default_nettype wire
