/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
*--------------------------------------------------------------------------
* $Author: $
* Company          : RivieraWaves
*--------------------------------------------------------------------------
* $Revision: $
* $Date: $
* -------------------------------------------------------------------------
* Dependencies     :                                       
* Description      : FFT engine 
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
* -------------------------------------------------------------------------
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module fftx4
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire           rst_n,
  input  wire           clk,
  
  /*****************************************************************************
  * control
  *
  * dir 0=FFT, 1=IFFT
  * len 0=64, 1=128, 2= 56, 3=512, 4=1024, 5=2048
  *****************************************************************************/
  input   wire          dir,    
  input   wire [  2:0]  len,
  input   wire [  2:0]  scale,
  input   wire          enable,    
  output  reg           done,   
      
  /*****************************************************************************
  * SRAM interface
  *****************************************************************************/
  output  wire          wen,
  output  wire [  7:0]  waddr,
  output  wire [239:0]  wdata,
  
  output  wire          ren,
  output  wire [  7:0]  raddr,
  input   wire [239:0]  rdata,
  input   wire          rdirty
);

  /*****************************************************************************
  * R0
  *****************************************************************************/
  reg          static_dir;             
  reg  [  2:0] static_len; 
  reg  [  2:0] static_scale; 
  
  wire         r0_col_last, r0_row_last;  
  wire [  7:0] n_r0_col;  
  wire [  3:0] n_r0_row;  
  reg  [  7:0] n_r0_addr;
  wire [  3:0] n_r0_bubble_insert, n_r0_bubble_decr;
  wire         r0_en;
  wire         r0_last;
 
  reg  [  7:0] r0_col;  /* 0 to 255 */               
  reg  [  3:0] r0_row;  /* 0 (FFT8) to 8 (FFT2048) */
  reg  [  7:0] r0_addr;                
  reg  [  3:0] r0_bubble;
  reg          r0_state;
  reg          r0_done;
  
  assign r0_col_last = (static_len==3'd0) & (r0_col==8'd7)  | 
                       (static_len==3'd1) & (r0_col==8'd15) | 
                       (static_len==3'd2) & (r0_col==8'd31) |
                       (static_len==3'd3) & (r0_col==8'd63) |
                       (static_len==3'd4) & (r0_col==8'd127)|
                       (static_len==3'd5) & (r0_col==8'd255);

  assign r0_row_last = (static_len==3'd0) & (r0_row==4'd3)  | 
                       (static_len==3'd1) & (r0_row==4'd4)  | 
                       (static_len==3'd2) & (r0_row==4'd5)  |
                       (static_len==3'd3) & (r0_row==4'd6)  |
                       (static_len==3'd4) & (r0_row==4'd7)  |
                       (static_len==3'd5) & (r0_row==4'd8);
  
  assign n_r0_col         = {8{~r0_col_last}} & (r0_col + 8'd1);
  assign n_r0_row         = r0_row + {3'b0,r0_col_last};
  assign n_r0_bubble_decr = r0_bubble + 4'b1111;
  
  /* read-after-write hazard, see scheduling.ods file */
  assign n_r0_bubble_insert = (4'd3  & {4{(static_len==3'd0) & r0_col_last & (r0_row==4'd0)}}) | 
                              (4'd5  & {4{(static_len==3'd0) & r0_col_last & (r0_row==4'd1)}}) | 
                              (4'd7  & {4{(static_len==3'd0) & r0_col_last & (r0_row==4'd2)}}) | 
                              (4'd3  & {4{(static_len==3'd1) & r0_col_last & (r0_row==4'd3)}}); 
 
  assign r0_en   = r0_state & (r0_bubble==4'd0);
  assign r0_last = r0_col_last & r0_row_last;
  
  /* address decoder*/
  always @(*)
  begin
    case(n_r0_row)
      4'd0:    n_r0_addr =  n_r0_col[7:0];
      4'd1:    n_r0_addr =  n_r0_col[7:0];
      4'd2:    n_r0_addr = {n_r0_col[7:2],   n_r0_col[0],n_r0_col[  1]};
      4'd3:    n_r0_addr = {n_r0_col[7:3],   n_r0_col[0],n_r0_col[2:1]};
      4'd4:    n_r0_addr = {n_r0_col[7:4],   n_r0_col[0],n_r0_col[3:1]};
      4'd5:    n_r0_addr = {n_r0_col[7:5],   n_r0_col[0],n_r0_col[4:1]};
      4'd6:    n_r0_addr = {n_r0_col[7:6],   n_r0_col[0],n_r0_col[5:1]};
      4'd7:    n_r0_addr = {n_r0_col[  7],   n_r0_col[0],n_r0_col[6:1]};
      default: n_r0_addr = {                 n_r0_col[0],n_r0_col[7:1]};
    endcase
  end
  
  /* registers */
  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
      static_dir   <= 1'b0;
      static_len   <= 3'b0;
      static_scale <= 3'd0;
      r0_col       <= 8'b0;
      r0_row       <= 4'b0;
      r0_addr      <= 8'b0;
      r0_bubble    <= 4'b0;
      r0_done      <= 1'b0;
      r0_state     <= 1'b0;
    end
    else if(!enable)
    begin
      static_dir   <= 1'b0;
      static_len   <= 3'b0;
      static_scale <= 3'd0;
      r0_col       <= 8'b0;
      r0_row       <= 4'b0;
      r0_addr      <= 8'b0;
      r0_bubble    <= 4'b0;
      r0_done      <= 1'b0;
      r0_state     <= 1'b0;
    end
    else
    begin
      if(!r0_state)
      begin
        static_dir   <= dir;
        static_len   <= len;
        static_scale <= scale;
        r0_col       <= 8'b0;
        r0_row       <= 4'b0;
        r0_addr      <= 8'b0;
        if(!r0_done)
          r0_state     <= 1'b1;
      end
      else
      begin
        /* hazard */
        if(r0_bubble==4'd0)
        begin
          r0_addr   <= n_r0_addr;
          r0_col    <= n_r0_col;
          r0_row    <= n_r0_row;
          r0_bubble <= n_r0_bubble_insert;
        end
        else
        begin
          r0_bubble <= n_r0_bubble_decr;
        end
        
        /* r0 stage done */
        if(r0_col_last & r0_row_last)
        begin
          r0_state <= 1'b0;
          r0_col   <= 8'b0;
          r0_row   <= 4'b0;
          r0_addr  <= 8'b0;
          r0_done  <= 1'b1;    
        end
      end
    end
  
  assign ren   = r0_en;
  assign raddr = r0_addr;
 
  /*****************************************************************************
  * R1
  *****************************************************************************/
  reg          r1_en;
  reg [ 7:0]   r1_col;
  reg [ 3:0]   r1_row;
  reg          r1_last;

  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
      r1_en   <= 1'b0;
      r1_col  <= 8'b0;
      r1_row  <= 4'b0;
      r1_last <= 1'b0;
    end
    else
    begin
      r1_en   <= r0_en & enable;
      r1_col  <= r0_col;
      r1_row  <= r0_row;
      r1_last <= r0_last;
    end
  
  /*****************************************************************************
  * R2
  *****************************************************************************/
  reg          r2_en;
  reg [ 7:0]   r2_col;
  reg [ 3:0]   r2_row;
  reg          r2_last;

  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
      r2_en   <= 1'b0;
      r2_col  <= 8'b0;
      r2_row  <= 4'b0;
      r2_last <= 1'b0;
    end
    else
    begin
      r2_en   <= r1_en & enable;
      r2_col  <= r1_col;
      r2_row  <= r1_row;
      r2_last <= r1_last;
    end
  
  /*****************************************************************************
  * C3 / C4
  *****************************************************************************/
  reg         c3_en;                                            
  reg  [ 7:0] c3_col;                                            
  reg  [ 3:0] c3_row;                                            
  reg         c3_last;                                            
  reg         c3_dirty;  
 
  reg  [ 3:0] c3_precln_e0; 
  reg  [12:0] c3_precln_i0,c3_precln_q0;   
  reg  [ 3:0] c3_precln_e1; 
  reg  [12:0] c3_precln_i1,c3_precln_q1;   
  reg  [ 3:0] c3_precln_e2; 
  reg  [12:0] c3_precln_i2,c3_precln_q2;   
  reg  [ 3:0] c3_precln_e3; 
  reg  [12:0] c3_precln_i3,c3_precln_q3;   
  reg  [ 3:0] c3_precln_e4; 
  reg  [12:0] c3_precln_i4,c3_precln_q4;   
  reg  [ 3:0] c3_precln_e5; 
  reg  [12:0] c3_precln_i5,c3_precln_q5;   
  reg  [ 3:0] c3_precln_e6; 
  reg  [12:0] c3_precln_i6,c3_precln_q6;   
  reg  [ 3:0] c3_precln_e7; 
  reg  [12:0] c3_precln_i7,c3_precln_q7;   
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      c3_en                                    <= 1'b0;
      c3_col                                   <= 8'b0;
      c3_row                                   <= 4'b0;
      c3_last                                  <= 1'b0;
      c3_dirty                                 <= 1'b0;
      {c3_precln_e0,c3_precln_q0,c3_precln_i0} <= 30'b0;
      {c3_precln_e1,c3_precln_q1,c3_precln_i1} <= 30'b0;
      {c3_precln_e2,c3_precln_q2,c3_precln_i2} <= 30'b0;
      {c3_precln_e3,c3_precln_q3,c3_precln_i3} <= 30'b0;
      {c3_precln_e4,c3_precln_q4,c3_precln_i4} <= 30'b0;
      {c3_precln_e5,c3_precln_q5,c3_precln_i5} <= 30'b0;
      {c3_precln_e6,c3_precln_q6,c3_precln_i6} <= 30'b0;
      {c3_precln_e7,c3_precln_q7,c3_precln_i7} <= 30'b0;
    end
    else
    begin
      c3_en                                    <= r2_en & enable;
      c3_col                                   <= r2_col;
      c3_row                                   <= r2_row;
      c3_last                                  <= r2_last;
      c3_dirty                                 <= rdirty;
      {c3_precln_e0,c3_precln_q0,c3_precln_i0} <= rdata[ 29:  0];
      {c3_precln_e1,c3_precln_q1,c3_precln_i1} <= rdata[ 59: 30];
      {c3_precln_e2,c3_precln_q2,c3_precln_i2} <= rdata[ 89: 60];
      {c3_precln_e3,c3_precln_q3,c3_precln_i3} <= rdata[119: 90];
      {c3_precln_e4,c3_precln_q4,c3_precln_i4} <= rdata[149:120];
      {c3_precln_e5,c3_precln_q5,c3_precln_i5} <= rdata[179:150];
      {c3_precln_e6,c3_precln_q6,c3_precln_i6} <= rdata[209:180];
      {c3_precln_e7,c3_precln_q7,c3_precln_i7} <= rdata[239:210];
    end
  end
  
  reg  [ 3:0]  c3_e0;
  reg  [12:0]  c3_i0,c3_q0;
  reg  [ 3:0]  c3_e1;
  reg  [12:0]  c3_i1,c3_q1;
  reg  [ 3:0]  c3_e2;
  reg  [12:0]  c3_i2,c3_q2;
  reg  [ 3:0]  c3_e3;
  reg  [12:0]  c3_i3,c3_q3;
  reg  [ 3:0]  c3_e4;
  reg  [12:0]  c3_i4,c3_q4;
  reg  [ 3:0]  c3_e5;
  reg  [12:0]  c3_i5,c3_q5;
  reg  [ 3:0]  c3_e6;
  reg  [12:0]  c3_i6,c3_q6;
  reg  [ 3:0]  c3_e7;
  reg  [12:0]  c3_i7,c3_q7;
  
  always @(*)
  begin
    if(c3_row==3'd0 && c3_dirty)
    begin
      {c3_e0,c3_q0,c3_i0} = 30'b0;
      {c3_e1,c3_q1,c3_i1} = 30'b0;
      {c3_e2,c3_q2,c3_i2} = 30'b0;
      {c3_e3,c3_q3,c3_i3} = 30'b0;
      {c3_e4,c3_q4,c3_i4} = 30'b0;
      {c3_e5,c3_q5,c3_i5} = 30'b0;
      {c3_e6,c3_q6,c3_i6} = 30'b0;
      {c3_e7,c3_q7,c3_i7} = 30'b0;
    end
    else
    begin
      {c3_e0,c3_q0,c3_i0} = {c3_precln_e0,c3_precln_q0,c3_precln_i0};
      {c3_e1,c3_q1,c3_i1} = {c3_precln_e1,c3_precln_q1,c3_precln_i1};
      {c3_e2,c3_q2,c3_i2} = {c3_precln_e2,c3_precln_q2,c3_precln_i2};
      {c3_e3,c3_q3,c3_i3} = {c3_precln_e3,c3_precln_q3,c3_precln_i3};
      {c3_e4,c3_q4,c3_i4} = {c3_precln_e4,c3_precln_q4,c3_precln_i4};
      {c3_e5,c3_q5,c3_i5} = {c3_precln_e5,c3_precln_q5,c3_precln_i5};
      {c3_e6,c3_q6,c3_i6} = {c3_precln_e6,c3_precln_q6,c3_precln_i6};
      {c3_e7,c3_q7,c3_i7} = {c3_precln_e7,c3_precln_q7,c3_precln_i7};
    end
  end
  
  wire  [13:0]  c3_fft2_i0,c3_fft2_q0;
  wire  [13:0]  c3_fft2_i1,c3_fft2_q1;
  wire  [13:0]  c3_fft2_i2,c3_fft2_q2;
  wire  [13:0]  c3_fft2_i3,c3_fft2_q3;
  wire  [13:0]  c3_fft2_i4,c3_fft2_q4;
  wire  [13:0]  c3_fft2_i5,c3_fft2_q5;
  wire  [13:0]  c3_fft2_i6,c3_fft2_q6;
  wire  [13:0]  c3_fft2_i7,c3_fft2_q7;
  
  fft_fft2 u_fft2_0
  (
    .a_i(        c3_i0),  
    .a_q(        c3_q0),  
    .b_i(        c3_i1), 
    .b_q(        c3_q1),
    .x_i(        c3_fft2_i0), 
    .x_q(        c3_fft2_q0), 
    .y_i(        c3_fft2_i1), 
    .y_q(        c3_fft2_q1) 
  );
 
  fft_fft2 u_fft2_1
  (
    .a_i(        c3_i2),  
    .a_q(        c3_q2),  
    .b_i(        c3_i3), 
    .b_q(        c3_q3),
    .x_i(        c3_fft2_i2), 
    .x_q(        c3_fft2_q2), 
    .y_i(        c3_fft2_i3), 
    .y_q(        c3_fft2_q3) 
  );
 
  fft_fft2 u_fft2_2
  (
    .a_i(        c3_i4),  
    .a_q(        c3_q4),  
    .b_i(        c3_i5), 
    .b_q(        c3_q5),
    .x_i(        c3_fft2_i4), 
    .x_q(        c3_fft2_q4), 
    .y_i(        c3_fft2_i5), 
    .y_q(        c3_fft2_q5) 
  );
 
  fft_fft2 u_fft2_3
  (
    .a_i(        c3_i6),  
    .a_q(        c3_q6),  
    .b_i(        c3_i7), 
    .b_q(        c3_q7),
    .x_i(        c3_fft2_i6), 
    .x_q(        c3_fft2_q6), 
    .y_i(        c3_fft2_i7), 
    .y_q(        c3_fft2_q7) 
  );

  /*****************************************************************************
  * C4
  *****************************************************************************/
  reg          c4_en;
  reg  [7:0]   c4_col;
  reg  [3:0]   c4_row;
  reg          c4_last;

  reg  [13:0]  c4_fft2_i0,c4_fft2_q0;
  reg  [13:0]  c4_fft2_i1,c4_fft2_q1;
  reg  [13:0]  c4_fft2_i2,c4_fft2_q2;
  reg  [13:0]  c4_fft2_i3,c4_fft2_q3;
  reg  [13:0]  c4_fft2_i4,c4_fft2_q4;
  reg  [13:0]  c4_fft2_i5,c4_fft2_q5;
  reg  [13:0]  c4_fft2_i6,c4_fft2_q6;
  reg  [13:0]  c4_fft2_i7,c4_fft2_q7;
  
  reg  [ 3:0]  c4_e0;
  reg  [12:0]  c4_i0,c4_q0;
  reg  [ 3:0]  c4_e1;
  reg  [12:0]  c4_i1,c4_q1;
  reg  [ 3:0]  c4_e2;
  reg  [12:0]  c4_i2,c4_q2;
  reg  [ 3:0]  c4_e3;
  reg  [12:0]  c4_i3,c4_q3;
  reg  [ 3:0]  c4_e4;
  reg  [12:0]  c4_i4,c4_q4;
  reg  [ 3:0]  c4_e5;
  reg  [12:0]  c4_i5,c4_q5;
  reg  [ 3:0]  c4_e6;
  reg  [12:0]  c4_i6,c4_q6;
  reg  [ 3:0]  c4_e7;
  reg  [12:0]  c4_i7,c4_q7;

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      c4_en                    <= 1'b0;
      c4_col                   <= 8'b0;
      c4_row                   <= 4'b0;
      c4_last                  <= 1'b0;
      {c4_fft2_q0,c4_fft2_i0}  <= 28'b0;
      {c4_fft2_q1,c4_fft2_i1}  <= 28'b0;
      {c4_fft2_q2,c4_fft2_i2}  <= 28'b0;
      {c4_fft2_q3,c4_fft2_i3}  <= 28'b0;
      {c4_fft2_q4,c4_fft2_i4}  <= 28'b0;
      {c4_fft2_q5,c4_fft2_i5}  <= 28'b0;
      {c4_fft2_q6,c4_fft2_i6}  <= 28'b0;
      {c4_fft2_q7,c4_fft2_i7}  <= 28'b0;
      {c4_e0,c4_q0,c4_i0}      <= 30'b0;
      {c4_e1,c4_q1,c4_i1}      <= 30'b0;
      {c4_e2,c4_q2,c4_i2}      <= 30'b0;
      {c4_e3,c4_q3,c4_i3}      <= 30'b0;
      {c4_e4,c4_q4,c4_i4}      <= 30'b0;
      {c4_e5,c4_q5,c4_i5}      <= 30'b0;
      {c4_e6,c4_q6,c4_i6}      <= 30'b0;
      {c4_e7,c4_q7,c4_i7}      <= 30'b0;
    end
    else
    begin
      c4_en                   <= c3_en & enable;      
      c4_col                  <= c3_col;     
      c4_row                  <= c3_row;     
      c4_last                 <= c3_last; 
      if(c3_row==4'd0)
      begin
        {c4_fft2_q0,c4_fft2_i0} <= {c3_fft2_q0,c3_fft2_i0};
        {c4_fft2_q1,c4_fft2_i1} <= {c3_fft2_q1,c3_fft2_i1};
        {c4_fft2_q2,c4_fft2_i2} <= {c3_fft2_q2,c3_fft2_i2};
        {c4_fft2_q3,c4_fft2_i3} <= {c3_fft2_q3,c3_fft2_i3};
        {c4_fft2_q4,c4_fft2_i4} <= {c3_fft2_q4,c3_fft2_i4};
        {c4_fft2_q5,c4_fft2_i5} <= {c3_fft2_q5,c3_fft2_i5};
        {c4_fft2_q6,c4_fft2_i6} <= {c3_fft2_q6,c3_fft2_i6};
        {c4_fft2_q7,c4_fft2_i7} <= {c3_fft2_q7,c3_fft2_i7};
      end
      else
      begin
        if(!c3_col[0])
        begin
          {c4_e0,c4_q0,c4_i0} <= {c3_e0,c3_q0,c3_i0};
          {c4_e1,c4_q1,c4_i1} <= {c3_e1,c3_q1,c3_i1};
          {c4_e2,c4_q2,c4_i2} <= {c3_e2,c3_q2,c3_i2};
          {c4_e3,c4_q3,c4_i3} <= {c3_e3,c3_q3,c3_i3};
          {c4_e4,c4_q4,c4_i4} <= {c3_e4,c3_q4,c3_i4};
          {c4_e5,c4_q5,c4_i5} <= {c3_e5,c3_q5,c3_i5};
          {c4_e6,c4_q6,c4_i6} <= {c3_e6,c3_q6,c3_i6};
          {c4_e7,c4_q7,c4_i7} <= {c3_e7,c3_q7,c3_i7};
        end
        else
        begin
          {c4_e0,c4_q0,c4_i0} <= {c4_e4,c4_q4,c4_i4};
          {c4_e1,c4_q1,c4_i1} <= {c4_e5,c4_q5,c4_i5};
          {c4_e2,c4_q2,c4_i2} <= {c4_e6,c4_q6,c4_i6};
          {c4_e3,c4_q3,c4_i3} <= {c4_e7,c4_q7,c4_i7};
          {c4_e4,c4_q4,c4_i4} <= {c3_e4,c3_q4,c3_i4};
          {c4_e5,c4_q5,c4_i5} <= {c3_e5,c3_q5,c3_i5};
          {c4_e6,c4_q6,c4_i6} <= {c3_e6,c3_q6,c3_i6};
          {c4_e7,c4_q7,c4_i7} <= {c3_e7,c3_q7,c3_i7};
        end
      end
    end
  end

  wire [ 1:0] c4_fft4_e0;
  wire [12:0] c4_fft4_i0, c4_fft4_q0;
  wire [ 1:0] c4_fft4_e1;
  wire [12:0] c4_fft4_i1, c4_fft4_q1;
  wire [ 1:0] c4_fft4_e2;
  wire [12:0] c4_fft4_i2, c4_fft4_q2;
  wire [ 1:0] c4_fft4_e3;
  wire [12:0] c4_fft4_i3, c4_fft4_q3;
  wire [ 1:0] c4_fft4_e4;
  wire [12:0] c4_fft4_i4, c4_fft4_q4;
  wire [ 1:0] c4_fft4_e5;
  wire [12:0] c4_fft4_i5, c4_fft4_q5;
  wire [ 1:0] c4_fft4_e6;
  wire [12:0] c4_fft4_i6, c4_fft4_q6;
  wire [ 1:0] c4_fft4_e7;
  wire [12:0] c4_fft4_i7, c4_fft4_q7;

  fft_fft4 u_fft4_0
  (
    .dir(   static_dir),
    .a_i0(  c4_fft2_i0),  
    .a_q0(  c4_fft2_q0),  
    .a_i1(  c4_fft2_i1),
    .a_q1(  c4_fft2_q1),
    .b_i0(  c4_fft2_i2), 
    .b_q0(  c4_fft2_q2), 
    .b_i1(  c4_fft2_i3), 
    .b_q1(  c4_fft2_q3), 
    .x_e0(  c4_fft4_e0),
    .x_i0(  c4_fft4_i0),
    .x_q0(  c4_fft4_q0),
    .x_e1(  c4_fft4_e1),
    .x_i1(  c4_fft4_i1),
    .x_q1(  c4_fft4_q1),
    .y_e0(  c4_fft4_e2),
    .y_i0(  c4_fft4_i2),
    .y_q0(  c4_fft4_q2),
    .y_e1(  c4_fft4_e3),
    .y_i1(  c4_fft4_i3),
    .y_q1(  c4_fft4_q3)
  );

  fft_fft4 u_fft4_1
  (
    .dir(   static_dir),
    .a_i0(  c4_fft2_i4),  
    .a_q0(  c4_fft2_q4),  
    .a_i1(  c4_fft2_i5),
    .a_q1(  c4_fft2_q5),
    .b_i0(  c4_fft2_i6), 
    .b_q0(  c4_fft2_q6), 
    .b_i1(  c4_fft2_i7), 
    .b_q1(  c4_fft2_q7), 
    .x_e0(  c4_fft4_e4),
    .x_i0(  c4_fft4_i4),
    .x_q0(  c4_fft4_q4),
    .x_e1(  c4_fft4_e5),
    .x_i1(  c4_fft4_i5),
    .x_q1(  c4_fft4_q5),
    .y_e0(  c4_fft4_e6),
    .y_i0(  c4_fft4_i6),
    .y_q0(  c4_fft4_q6),
    .y_e1(  c4_fft4_e7),
    .y_i1(  c4_fft4_i7),
    .y_q1(  c4_fft4_q7)
  );

  reg   [ 9:0] angle0,angle1,angle2,angle3;
  always @(*)
  begin  
    case(c4_row)
      4'd0:    angle0 = {            10'b0000000000}; /*   8  */
      4'd1:    angle0 = { c4_col[  0], 9'b000000000}; /*   16 */
      4'd2:    angle0 = { c4_col[1:0],  8'b00000000}; /*   32 */
      4'd3:    angle0 = { c4_col[2:0],   7'b0000000}; /*   64 */ 
      4'd4:    angle0 = { c4_col[3:0],    6'b000000}; /*  128 */ 
      4'd5:    angle0 = { c4_col[4:0],     5'b00000}; /*  256 */  
      4'd6:    angle0 = { c4_col[5:0],      4'b0000}; /*  512 */  
      4'd7:    angle0 = { c4_col[6:0],       3'b000}; /* 1024 */  
      default: angle0 = { c4_col[7:0],        2'b00}; /* 2048 */  
    endcase
    case(c4_row)
      4'd0:    angle1 = {            10'b0100000000}; /*   8  */
      4'd1:    angle1 = { c4_col[  0], 9'b010000000}; /*   16 */
      4'd2:    angle1 = { c4_col[1:0],  8'b01000000}; /*   32 */
      4'd3:    angle1 = { c4_col[2:0],   7'b0100000}; /*   64 */ 
      4'd4:    angle1 = { c4_col[3:0],    6'b010000}; /*  128 */ 
      4'd5:    angle1 = { c4_col[4:0],     5'b01000}; /*  256 */  
      4'd6:    angle1 = { c4_col[5:0],      4'b0100}; /*  512 */  
      4'd7:    angle1 = { c4_col[6:0],       3'b010}; /* 1024 */  
      default: angle1 = { c4_col[7:0],        2'b01}; /* 2048 */  
    endcase
    case(c4_row)
      4'd0:    angle2 = {            10'b1000000000}; /*   8  */
      4'd1:    angle2 = { c4_col[  0], 9'b100000000}; /*   16 */
      4'd2:    angle2 = { c4_col[1:0],  8'b10000000}; /*   32 */
      4'd3:    angle2 = { c4_col[2:0],   7'b1000000}; /*   64 */ 
      4'd4:    angle2 = { c4_col[3:0],    6'b100000}; /*  128 */ 
      4'd5:    angle2 = { c4_col[4:0],     5'b10000}; /*  256 */  
      4'd6:    angle2 = { c4_col[5:0],      4'b1000}; /*  512 */  
      4'd7:    angle2 = { c4_col[6:0],       3'b100}; /* 1024 */  
      default: angle2 = { c4_col[7:0],        2'b10}; /* 2048 */  
    endcase
    case(c4_row)
      4'd0:    angle3 = {            10'b1100000000}; /*   8  */
      4'd1:    angle3 = { c4_col[  0], 9'b110000000}; /*   16 */
      4'd2:    angle3 = { c4_col[1:0],  8'b11000000}; /*   32 */
      4'd3:    angle3 = { c4_col[2:0],   7'b1100000}; /*   64 */ 
      4'd4:    angle3 = { c4_col[3:0],    6'b110000}; /*  128 */ 
      4'd5:    angle3 = { c4_col[4:0],     5'b11000}; /*  256 */  
      4'd6:    angle3 = { c4_col[5:0],      4'b1100}; /*  512 */  
      4'd7:    angle3 = { c4_col[6:0],       3'b110}; /* 1024 */  
      default: angle3 = { c4_col[7:0],        2'b11}; /* 2048 */  
    endcase
  end
 
  wire  [10:0] c4_angle0, c4_angle1, c4_angle2, c4_angle3;
 
  assign c4_angle0 = (static_dir)?{1'b0,angle0}:{~{1'b0,angle0}+11'b1};
  assign c4_angle1 = (static_dir)?{1'b0,angle1}:{~{1'b0,angle1}+11'b1};
  assign c4_angle2 = (static_dir)?{1'b0,angle2}:{~{1'b0,angle2}+11'b1};
  assign c4_angle3 = (static_dir)?{1'b0,angle3}:{~{1'b0,angle3}+11'b1};
 
  wire  [15:0] c4_cos0, c4_sin0;
  wire  [15:0] c4_cos1, c4_sin1;
  wire  [15:0] c4_cos2, c4_sin2;
  wire  [15:0] c4_cos3, c4_sin3;
 
  fft_cos_lut u_fft_cos_lut_0           
  (                            
    .angle(   c4_angle0), 
    .lut_cos( c4_cos0),   
    .lut_sin( c4_sin0)  
  );                           

  fft_cos_lut u_fft_cos_lut_1          
  (                            
    .angle(   c4_angle1), 
    .lut_cos( c4_cos1),   
    .lut_sin( c4_sin1)  
  );                           

  fft_cos_lut u_fft_cos_lut_2          
  (                            
    .angle(   c4_angle2), 
    .lut_cos( c4_cos2),   
    .lut_sin( c4_sin2)  
  );                           

  fft_cos_lut u_fft_cos_lut_3         
  (                            
    .angle(   c4_angle3), 
    .lut_cos( c4_cos3),   
    .lut_sin( c4_sin3)  
  );                           

  /*****************************************************************************
  * A5 / B5
  *****************************************************************************/
  reg           c5_en;
  reg [3:0]     c5_row;
  reg [7:0]     c5_col;
  reg           c5_last;
 
  reg [ 3:0]    a5_e0;
  reg [12:0]    a5_i0,a5_q0;
  reg [ 3:0]    a5_e1;
  reg [12:0]    a5_i1,a5_q1;
  reg [ 3:0]    a5_e2;
  reg [12:0]    a5_i2,a5_q2;
  reg [ 3:0]    a5_e3;
  reg [12:0]    a5_i3,a5_q3;
  
  reg [ 3:0]    b5_e0;
  reg [12:0]    b5_i0,b5_q0;
  reg [ 3:0]    b5_e1;
  reg [12:0]    b5_i1,b5_q1;
  reg [ 3:0]    b5_e2;
  reg [12:0]    b5_i2,b5_q2;
  reg [ 3:0]    b5_e3;
  reg [12:0]    b5_i3,b5_q3;
  
  reg [15:0]    b5_cos0,b5_sin0;
  reg [15:0]    b5_cos1,b5_sin1;
  reg [15:0]    b5_cos2,b5_sin2;
  reg [15:0]    b5_cos3,b5_sin3;
 
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      c5_en                 <= 1'b0;
      c5_col                <= 8'b0;
      c5_row                <= 4'b0;
      c5_last               <= 1'b0;
      {a5_e0,a5_q0,a5_i0}   <= 30'b0;
      {a5_e1,a5_q1,a5_i1}   <= 30'b0;
      {a5_e2,a5_q2,a5_i2}   <= 30'b0;
      {a5_e3,a5_q3,a5_i3}   <= 30'b0;
      {b5_e0,b5_q0,b5_i0}   <= 30'b0;
      {b5_e1,b5_q1,b5_i1}   <= 30'b0;
      {b5_e2,b5_q2,b5_i2}   <= 30'b0;
      {b5_e3,b5_q3,b5_i3}   <= 30'b0;
      {b5_cos0,b5_sin0}     <= 32'b0;
      {b5_cos1,b5_sin1}     <= 32'b0;
      {b5_cos2,b5_sin2}     <= 32'b0;
      {b5_cos3,b5_sin3}     <= 32'b0;    
    end
    else
    begin
      c5_en                 <= c4_en & enable;
      c5_col                <= c4_col;
      c5_row                <= c4_row;
      c5_last               <= c4_last;
      
      {b5_cos0,b5_sin0}     <= {c4_cos0,c4_sin0};
      {b5_cos1,b5_sin1}     <= {c4_cos1,c4_sin1};
      {b5_cos2,b5_sin2}     <= {c4_cos2,c4_sin2};
      {b5_cos3,b5_sin3}     <= {c4_cos3,c4_sin3};    
      
      if(c4_row==4'd0)
      begin
        {a5_e0,a5_q0,a5_i0} <= {2'b0,c4_fft4_e0,c4_fft4_q0,c4_fft4_i0};
        {a5_e1,a5_q1,a5_i1} <= {2'b0,c4_fft4_e1,c4_fft4_q1,c4_fft4_i1};
        {a5_e2,a5_q2,a5_i2} <= {2'b0,c4_fft4_e2,c4_fft4_q2,c4_fft4_i2};
        {a5_e3,a5_q3,a5_i3} <= {2'b0,c4_fft4_e3,c4_fft4_q3,c4_fft4_i3};
        {b5_e0,b5_q0,b5_i0} <= {2'b0,c4_fft4_e4,c4_fft4_q4,c4_fft4_i4};
        {b5_e1,b5_q1,b5_i1} <= {2'b0,c4_fft4_e5,c4_fft4_q5,c4_fft4_i5};
        {b5_e2,b5_q2,b5_i2} <= {2'b0,c4_fft4_e6,c4_fft4_q6,c4_fft4_i6};
        {b5_e3,b5_q3,b5_i3} <= {2'b0,c4_fft4_e7,c4_fft4_q7,c4_fft4_i7};
      end
      else
      begin
        if(!c4_col[0])
        begin
          {a5_e0,a5_q0,a5_i0} <= {c4_e0,c4_q0,c4_i0};
          {a5_e1,a5_q1,a5_i1} <= {c4_e1,c4_q1,c4_i1};
          {a5_e2,a5_q2,a5_i2} <= {c4_e2,c4_q2,c4_i2};
          {a5_e3,a5_q3,a5_i3} <= {c4_e3,c4_q3,c4_i3};
          {b5_e0,b5_q0,b5_i0} <= {c3_e0,c3_q0,c3_i0};
          {b5_e1,b5_q1,b5_i1} <= {c3_e1,c3_q1,c3_i1};
          {b5_e2,b5_q2,b5_i2} <= {c3_e2,c3_q2,c3_i2};
          {b5_e3,b5_q3,b5_i3} <= {c3_e3,c3_q3,c3_i3};
        end
        else
        begin
          {a5_e0,a5_q0,a5_i0} <= {c4_e0,c4_q0,c4_i0};
          {a5_e1,a5_q1,a5_i1} <= {c4_e1,c4_q1,c4_i1};
          {a5_e2,a5_q2,a5_i2} <= {c4_e2,c4_q2,c4_i2};
          {a5_e3,a5_q3,a5_i3} <= {c4_e3,c4_q3,c4_i3};
          {b5_e0,b5_q0,b5_i0} <= {c4_e4,c4_q4,c4_i4};
          {b5_e1,b5_q1,b5_i1} <= {c4_e5,c4_q5,c4_i5};
          {b5_e2,b5_q2,b5_i2} <= {c4_e6,c4_q6,c4_i6};
          {b5_e3,b5_q3,b5_i3} <= {c4_e7,c4_q7,c4_i7};
        end
      end
    end
  end

  wire [13:0]    b7_i0,b7_q0;
  wire [13:0]    b7_i1,b7_q1;
  wire [13:0]    b7_i2,b7_q2;
  wire [13:0]    b7_i3,b7_q3;
  
  fft_cmult u_fft_cmult_0
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .a(        b5_i0),
    .b(        b5_q0),
    .c(        b5_cos0),
    .d(        b5_sin0),
    .x(        b7_i0),
    .y(        b7_q0)
  );
  
  fft_cmult u_fft_cmult_1
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .a(        b5_i1),
    .b(        b5_q1),
    .c(        b5_cos1),
    .d(        b5_sin1),
    .x(        b7_i1),
    .y(        b7_q1)
  );
  
  fft_cmult u_fft_cmult_2
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .a(        b5_i2),
    .b(        b5_q2),
    .c(        b5_cos2),
    .d(        b5_sin2),
    .x(        b7_i2),
    .y(        b7_q2)
  );

  fft_cmult u_fft_cmult_3
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .a(        b5_i3),
    .b(        b5_q3),
    .c(        b5_cos3),
    .d(        b5_sin3),
    .x(        b7_i3),
    .y(        b7_q3)
  );
  
  /*****************************************************************************
  * C6-C7
  *****************************************************************************/
  reg        c6_en; 
  reg [ 7:0] c6_col;
  reg [ 3:0] c6_row;
  reg        c6_last;
  reg [ 3:0] a6_e0;  
  reg [12:0] a6_i0,a6_q0;  
  reg [ 3:0] a6_e1;  
  reg [12:0] a6_i1,a6_q1;  
  reg [ 3:0] a6_e2;  
  reg [12:0] a6_i2,a6_q2;  
  reg [ 3:0] a6_e3;  
  reg [12:0] a6_i3,a6_q3;  
  reg [ 3:0] b6_e0;  
  reg [ 3:0] b6_e1;  
  reg [ 3:0] b6_e2;  
  reg [ 3:0] b6_e3;  

  reg        c7_en; 
  reg [ 7:0] c7_col;
  reg [ 3:0] c7_row;
  reg        c7_last;
  reg [ 3:0] a7_e0;  
  reg [12:0] a7_i0,a7_q0;  
  reg [ 3:0] a7_e1;  
  reg [12:0] a7_i1,a7_q1;  
  reg [ 3:0] a7_e2;  
  reg [12:0] a7_i2,a7_q2;  
  reg [ 3:0] a7_e3;  
  reg [12:0] a7_i3,a7_q3;  
  reg [ 3:0] b7_e0;  
  reg [ 3:0] b7_e1;  
  reg [ 3:0] b7_e2;  
  reg [ 3:0] b7_e3;  
   
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      c6_en               <= 2'b0;
      c6_col              <= 8'b0;
      c6_row              <= 4'b0;
      c6_last             <= 2'b0;
      {a6_e0,a6_q0,a6_i0} <= 30'b0;
      {a6_e1,a6_q1,a6_i1} <= 30'b0;
      {a6_e2,a6_q2,a6_i2} <= 30'b0;
      {a6_e3,a6_q3,a6_i3} <= 30'b0;
      b6_e0               <= 4'b0;
      b6_e1               <= 4'b0;
      b6_e2               <= 4'b0;
      b6_e3               <= 4'b0;
      c7_en               <= 2'b0;
      c7_col              <= 8'b0;
      c7_row              <= 4'b0;
      c7_last             <= 2'b0;
      {a7_e0,a7_q0,a7_i0} <= 30'b0;
      {a7_e1,a7_q1,a7_i1} <= 30'b0;
      {a7_e2,a7_q2,a7_i2} <= 30'b0;
      {a7_e3,a7_q3,a7_i3} <= 30'b0;
      b7_e0               <= 4'b0;
      b7_e1               <= 4'b0;
      b7_e2               <= 4'b0;
      b7_e3               <= 4'b0;
    end
    else
    begin
      c6_en               <= c5_en & enable;              
      c6_col              <= c5_col;            
      c6_row              <= c5_row;             
      c6_last             <= c5_last;            
      {a6_e0,a6_q0,a6_i0} <= {a5_e0,a5_q0,a5_i0};
      {a6_e1,a6_q1,a6_i1} <= {a5_e1,a5_q1,a5_i1};
      {a6_e2,a6_q2,a6_i2} <= {a5_e2,a5_q2,a5_i2};
      {a6_e3,a6_q3,a6_i3} <= {a5_e3,a5_q3,a5_i3};
      b6_e0               <= b5_e0;
      b6_e1               <= b5_e1;
      b6_e2               <= b5_e2;
      b6_e3               <= b5_e3;
      c7_en               <= c6_en;              
      c7_col              <= c6_col;             
      c7_row              <= c6_row;             
      c7_last             <= c6_last;            
      {a7_e0,a7_q0,a7_i0} <= {a6_e0,a6_q0,a6_i0};
      {a7_e1,a7_q1,a7_i1} <= {a6_e1,a6_q1,a6_i1};
      {a7_e2,a7_q2,a7_i2} <= {a6_e2,a6_q2,a6_i2};
      {a7_e3,a7_q3,a7_i3} <= {a6_e3,a6_q3,a6_i3};
      b7_e0               <= b6_e0;
      b7_e1               <= b6_e1;
      b7_e2               <= b6_e2;
      b7_e3               <= b6_e3;
    end 
  end

  wire [ 3:0]   b7_fp_e0;
  wire [12:0]   b7_fp_i0, b7_fp_q0;
  wire [ 3:0]   b7_fp_e1;
  wire [12:0]   b7_fp_i1, b7_fp_q1;
  wire [ 3:0]   b7_fp_e2;
  wire [12:0]   b7_fp_i2, b7_fp_q2;
  wire [ 3:0]   b7_fp_e3;
  wire [12:0]   b7_fp_i3, b7_fp_q3;
  
  fft_fp u_b7_fp_0
  (
    .in_e(    b7_e0),
    .in_i(    b7_i0),  
    .in_q(    b7_q0),  
    .out_e(   b7_fp_e0),
    .out_i(   b7_fp_i0), 
    .out_q(   b7_fp_q0) 
  );

  fft_fp u_b7_fp_1
  (
    .in_e(    b7_e1),
    .in_i(    b7_i1),  
    .in_q(    b7_q1),  
    .out_e(   b7_fp_e1),
    .out_i(   b7_fp_i1), 
    .out_q(   b7_fp_q1) 
  );
 
  fft_fp u_b7_fp_2
  (
    .in_e(    b7_e2),
    .in_i(    b7_i2),  
    .in_q(    b7_q2),  
    .out_e(   b7_fp_e2),
    .out_i(   b7_fp_i2), 
    .out_q(   b7_fp_q2) 
  );

  fft_fp u_b7_fp_3
  (
    .in_e(    b7_e3),
    .in_i(    b7_i3),  
    .in_q(    b7_q3),  
    .out_e(   b7_fp_e3),
    .out_i(   b7_fp_i3), 
    .out_q(   b7_fp_q3) 
  );

  wire [ 3:0]   ab7_agn_e0;
  wire [13:0]   a7_agn_i0,  a7_agn_q0;
  wire [13:0]   b7_agn_i0,  b7_agn_q0;
 
  wire [ 3:0]   ab7_agn_e1;
  wire [13:0]   a7_agn_i1,  a7_agn_q1;
  wire [13:0]   b7_agn_i1,  b7_agn_q1;
  
  wire [ 3:0]   ab7_agn_e2;
  wire [13:0]   a7_agn_i2,  a7_agn_q2;
  wire [13:0]   b7_agn_i2,  b7_agn_q2;
  
  wire [ 3:0]   ab7_agn_e3;
  wire [13:0]   a7_agn_i3,  a7_agn_q3;
  wire [13:0]   b7_agn_i3,  b7_agn_q3;

  fft_align u_c7_align_0
  (
    .in_e0(   a7_e0),
    .in_i0(   a7_i0),  
    .in_q0(   a7_q0),  
    .in_e1(   b7_fp_e0),
    .in_i1(   b7_fp_i0),  
    .in_q1(   b7_fp_q0),  
    .out_e(   ab7_agn_e0),
    .out_i0(  a7_agn_i0),
    .out_q0(  a7_agn_q0), 
    .out_i1(  b7_agn_i0), 
    .out_q1(  b7_agn_q0)
  );

  fft_align u_c7_align_1
  (
    .in_e0(   a7_e1),
    .in_i0(   a7_i1),  
    .in_q0(   a7_q1),  
    .in_e1(   b7_fp_e1),
    .in_i1(   b7_fp_i1),  
    .in_q1(   b7_fp_q1),  
    .out_e(   ab7_agn_e1),
    .out_i0(  a7_agn_i1),
    .out_q0(  a7_agn_q1), 
    .out_i1(  b7_agn_i1), 
    .out_q1(  b7_agn_q1)
  );
  
  fft_align u_c7_align_2
  (
    .in_e0(   a7_e2),
    .in_i0(   a7_i2),  
    .in_q0(   a7_q2),  
    .in_e1(   b7_fp_e2),
    .in_i1(   b7_fp_i2),  
    .in_q1(   b7_fp_q2),  
    .out_e(   ab7_agn_e2),
    .out_i0(  a7_agn_i2),
    .out_q0(  a7_agn_q2), 
    .out_i1(  b7_agn_i2), 
    .out_q1(  b7_agn_q2)
  );

  fft_align u_c7_align_3
  (
    .in_e0(   a7_e3),
    .in_i0(   a7_i3),  
    .in_q0(   a7_q3),  
    .in_e1(   b7_fp_e3),
    .in_i1(   b7_fp_i3),  
    .in_q1(   b7_fp_q3),  
    .out_e(   ab7_agn_e3),
    .out_i0(  a7_agn_i3),
    .out_q0(  a7_agn_q3), 
    .out_i1(  b7_agn_i3), 
    .out_q1(  b7_agn_q3)
  );
 
  wire [13:0]   x7_i0,  x7_q0;
  wire [13:0]   x7_i1,  x7_q1;
  wire [13:0]   x7_i2,  x7_q2;
  wire [13:0]   x7_i3,  x7_q3;
  wire [13:0]   y7_i0,  y7_q0;
  wire [13:0]   y7_i1,  y7_q1;
  wire [13:0]   y7_i2,  y7_q2;
  wire [13:0]   y7_i3,  y7_q3;
  
  assign x7_i0 = {a7_agn_i0[13],a7_agn_i0[13:1]} + {b7_agn_i0[13],b7_agn_i0[13:1]} + {              13'b0,a7_agn_i0[0]|b7_agn_i0[0]};
  assign x7_q0 = {a7_agn_q0[13],a7_agn_q0[13:1]} + {b7_agn_q0[13],b7_agn_q0[13:1]} + {              13'b0,a7_agn_q0[0]|b7_agn_q0[0]};
  assign y7_i0 = {a7_agn_i0[13],a7_agn_i0[13:1]} - {b7_agn_i0[13],b7_agn_i0[13:1]} + {{13{b7_agn_i0[0]}},a7_agn_i0[0]|b7_agn_i0[0]};
  assign y7_q0 = {a7_agn_q0[13],a7_agn_q0[13:1]} - {b7_agn_q0[13],b7_agn_q0[13:1]} + {{13{b7_agn_q0[0]}},a7_agn_q0[0]|b7_agn_q0[0]};
 
  assign x7_i1 = {a7_agn_i1[13],a7_agn_i1[13:1]} + {b7_agn_i1[13],b7_agn_i1[13:1]} + {              13'b0,a7_agn_i1[0]|b7_agn_i1[0]};
  assign x7_q1 = {a7_agn_q1[13],a7_agn_q1[13:1]} + {b7_agn_q1[13],b7_agn_q1[13:1]} + {              13'b0,a7_agn_q1[0]|b7_agn_q1[0]};
  assign y7_i1 = {a7_agn_i1[13],a7_agn_i1[13:1]} - {b7_agn_i1[13],b7_agn_i1[13:1]} + {{13{b7_agn_i1[0]}},a7_agn_i1[0]|b7_agn_i1[0]};
  assign y7_q1 = {a7_agn_q1[13],a7_agn_q1[13:1]} - {b7_agn_q1[13],b7_agn_q1[13:1]} + {{13{b7_agn_q1[0]}},a7_agn_q1[0]|b7_agn_q1[0]};

  assign x7_i2 = {a7_agn_i2[13],a7_agn_i2[13:1]} + {b7_agn_i2[13],b7_agn_i2[13:1]} + {              13'b0,a7_agn_i2[0]|b7_agn_i2[0]};
  assign x7_q2 = {a7_agn_q2[13],a7_agn_q2[13:1]} + {b7_agn_q2[13],b7_agn_q2[13:1]} + {              13'b0,a7_agn_q2[0]|b7_agn_q2[0]};
  assign y7_i2 = {a7_agn_i2[13],a7_agn_i2[13:1]} - {b7_agn_i2[13],b7_agn_i2[13:1]} + {{13{b7_agn_i2[0]}},a7_agn_i2[0]|b7_agn_i2[0]};
  assign y7_q2 = {a7_agn_q2[13],a7_agn_q2[13:1]} - {b7_agn_q2[13],b7_agn_q2[13:1]} + {{13{b7_agn_q2[0]}},a7_agn_q2[0]|b7_agn_q2[0]};
 
  assign x7_i3 = {a7_agn_i3[13],a7_agn_i3[13:1]} + {b7_agn_i3[13],b7_agn_i3[13:1]} + {              13'b0,a7_agn_i3[0]|b7_agn_i3[0]};
  assign x7_q3 = {a7_agn_q3[13],a7_agn_q3[13:1]} + {b7_agn_q3[13],b7_agn_q3[13:1]} + {              13'b0,a7_agn_q3[0]|b7_agn_q3[0]};
  assign y7_i3 = {a7_agn_i3[13],a7_agn_i3[13:1]} - {b7_agn_i3[13],b7_agn_i3[13:1]} + {{13{b7_agn_i3[0]}},a7_agn_i3[0]|b7_agn_i3[0]};
  assign y7_q3 = {a7_agn_q3[13],a7_agn_q3[13:1]} - {b7_agn_q3[13],b7_agn_q3[13:1]} + {{13{b7_agn_q3[0]}},a7_agn_q3[0]|b7_agn_q3[0]};
 
  /*****************************************************************************
  * X8 / Y8
  *****************************************************************************/
  reg           c8_en;
  reg [ 3:0]    c8_row;
  reg [ 7:0]    c8_col;
  reg           c8_last;
  reg           c8_saturate;
  reg [ 2:0]    c8_scale;
  
  reg [ 3:0]    xy8_e0;
  reg [13:0]    x8_i0,   x8_q0;
  reg [13:0]    y8_i0,   y8_q0;
  reg [ 3:0]    xy8_e1;
  reg [13:0]    x8_i1,   x8_q1;
  reg [13:0]    y8_i1,   y8_q1;
  reg [ 3:0]    xy8_e2;
  reg [13:0]    x8_i2,   x8_q2;
  reg [13:0]    y8_i2,   y8_q2;
  reg [ 3:0]    xy8_e3;
  reg [13:0]    x8_i3,   x8_q3;
  reg [13:0]    y8_i3,   y8_q3;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      c8_en         <= 1'b0;
      c8_col        <= 8'b0;
      c8_row        <= 4'b0;
      c8_last       <= 1'b0;
      c8_saturate   <= 1'b0;
      c8_scale      <= 3'b0;
      xy8_e0        <= 4'b0;
      {x8_i0,x8_q0} <= 28'b0;
      {y8_i0,y8_q0} <= 28'b0;
      xy8_e1        <= 4'b0;
      {x8_i1,x8_q1} <= 28'b0;
      {y8_i1,y8_q1} <= 28'b0;
      xy8_e2        <= 4'b0;
      {x8_i2,x8_q2} <= 28'b0;
      {y8_i2,y8_q2} <= 28'b0;
      xy8_e3        <= 4'b0;
      {x8_i3,x8_q3} <= 28'b0;
      {y8_i3,y8_q3} <= 28'b0;
    end
    else
    begin
      c8_en                     <= c7_en & enable;
      c8_col                    <= c7_col;
      c8_row                    <= c7_row;
      c8_last                   <= c7_last;
      case(static_len)
        3'd0:    {c8_saturate,c8_scale} <= (c7_row==3)?{1'b1,static_scale}:4'd0;
        3'd1:    {c8_saturate,c8_scale} <= (c7_row==4)?{1'b1,static_scale}:4'd0;
        3'd2:    {c8_saturate,c8_scale} <= (c7_row==5)?{1'b1,static_scale}:4'd0;
        3'd3:    {c8_saturate,c8_scale} <= (c7_row==6)?{1'b1,static_scale}:4'd0;
        3'd4:    {c8_saturate,c8_scale} <= (c7_row==7)?{1'b1,static_scale}:4'd0;
        default: {c8_saturate,c8_scale} <= (c7_row==8)?{1'b1,static_scale}:4'd0;
      endcase
      xy8_e0        <= ab7_agn_e0;
      {x8_i0,x8_q0} <= {x7_i0,x7_q0};
      {y8_i0,y8_q0} <= {y7_i0,y7_q0};
      xy8_e1        <= ab7_agn_e1;
      {x8_i1,x8_q1} <= {x7_i1,x7_q1};
      {y8_i1,y8_q1} <= {y7_i1,y7_q1};
      xy8_e2        <= ab7_agn_e2;
      {x8_i2,x8_q2} <= {x7_i2,x7_q2};
      {y8_i2,y8_q2} <= {y7_i2,y7_q2};
      xy8_e3        <= ab7_agn_e3;
      {x8_i3,x8_q3} <= {x7_i3,x7_q3};
      {y8_i3,y8_q3} <= {y7_i3,y7_q3};
    end
  end
  /*****************************************************************************
  * X9 / Y9
  *****************************************************************************/
  reg           c9_en;
  reg [ 3:0]    c9_row;
  reg [ 7:0]    c9_col;
  reg           c9_last;
  
  wire  [ 3:0]  x9_e0;
  wire  [12:0]  x9_i0, x9_q0;
  wire  [ 3:0]  x9_e1;
  wire  [12:0]  x9_i1, x9_q1;
  wire  [ 3:0]  x9_e2;
  wire  [12:0]  x9_i2, x9_q2;
  wire  [ 3:0]  x9_e3;
  wire  [12:0]  x9_i3, x9_q3;
 
  wire  [ 3:0]  y9_e0;
  wire  [12:0]  y9_i0, y9_q0;
  wire  [ 3:0]  y9_e1;
  wire  [12:0]  y9_i1, y9_q1;
  wire  [ 3:0]  y9_e2;
  wire  [12:0]  y9_i2, y9_q2;
  wire  [ 3:0]  y9_e3;
  wire  [12:0]  y9_i3, y9_q3;

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      c9_en   <= 1'b0;
      c9_col  <= 8'b0;
      c9_row  <= 4'b0;
      c9_last <= 1'b0;
    end
    else
    begin
      c9_en   <= c8_en & enable;
      c9_col  <= c8_col;
      c9_row  <= c8_row;
      c9_last <= c8_last;
    end
  end
  
  fft_fp_sqrt2 u_x9_fp_0
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e0),
    .in_i(     x8_i0),  
    .in_q(     x8_q0),  
    .out_e(    x9_e0),
    .out_i(    x9_i0), 
    .out_q(    x9_q0) 
  );
  
  fft_fp_sqrt2 u_y9_fp_0
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e0),
    .in_i(     y8_i0),  
    .in_q(     y8_q0),  
    .out_e(    y9_e0),
    .out_i(    y9_i0), 
    .out_q(    y9_q0) 
  );
  
  fft_fp_sqrt2 u_x9_fp_1
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e1),
    .in_i(     x8_i1),  
    .in_q(     x8_q1),  
    .out_e(    x9_e1),
    .out_i(    x9_i1), 
    .out_q(    x9_q1) 
  );
  
  fft_fp_sqrt2 u_y9_fp_1
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e1),
    .in_i(     y8_i1),  
    .in_q(     y8_q1),  
    .out_e(    y9_e1),
    .out_i(    y9_i1), 
    .out_q(    y9_q1) 
  );
 
  fft_fp_sqrt2 u_x9_fp_2
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e2),
    .in_i(     x8_i2),  
    .in_q(     x8_q2),  
    .out_e(    x9_e2),
    .out_i(    x9_i2), 
    .out_q(    x9_q2) 
  );
  
  fft_fp_sqrt2 u_y9_fp_2
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e2),
    .in_i(     y8_i2),  
    .in_q(     y8_q2),  
    .out_e(    y9_e2),
    .out_i(    y9_i2), 
    .out_q(    y9_q2) 
  );
  
  fft_fp_sqrt2 u_x9_fp_3
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e3),
    .in_i(     x8_i3),  
    .in_q(     x8_q3),  
    .out_e(    x9_e3),
    .out_i(    x9_i3), 
    .out_q(    x9_q3) 
  );
  
  fft_fp_sqrt2 u_y9_fp_3
  (
    .rst_n(    rst_n),
    .clk(      clk),
    .saturate( c8_saturate),
    .scale(    c8_scale),
    .in_e(     xy8_e3),
    .in_i(     y8_i3),  
    .in_q(     y8_q3),  
    .out_e(    y9_e3),
    .out_i(    y9_i3), 
    .out_q(    y9_q3) 
  );

  /*****************************************************************************
  * X10 / Y10
  *****************************************************************************/
  reg           c10_en;
  reg [ 3:0]    c10_row;
  reg [ 7:0]    c10_col;
  reg           c10_last;
  
  reg   [ 3:0] x10_e0;
  reg   [12:0] x10_i0,  x10_q0;
  reg   [ 3:0] x10_e1;
  reg   [12:0] x10_i1,  x10_q1;
  reg   [ 3:0] x10_e2;
  reg   [12:0] x10_i2,  x10_q2;
  reg   [ 3:0] x10_e3;
  reg   [12:0] x10_i3,  x10_q3;

  reg   [ 3:0] y10_e0;
  reg   [12:0] y10_i0,  y10_q0;
  reg   [ 3:0] y10_e1;
  reg   [12:0] y10_i1,  y10_q1;
  reg   [ 3:0] y10_e2;
  reg   [12:0] y10_i2,  y10_q2;
  reg   [ 3:0] y10_e3;
  reg   [12:0] y10_i3,  y10_q3;

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      c10_en                 <= 1'b0;
      c10_col                <= 8'b0;
      c10_row                <= 4'b0;
      c10_last               <= 1'b0;
      {x10_e0,x10_i0,x10_q0} <= 30'b0;
      {y10_e0,y10_i0,y10_q0} <= 30'b0;
      {x10_e1,x10_i1,x10_q1} <= 30'b0;
      {y10_e1,y10_i1,y10_q1} <= 30'b0;
      {x10_e2,x10_i2,x10_q2} <= 30'b0;
      {y10_e2,y10_i2,y10_q2} <= 30'b0;
      {x10_e3,x10_i3,x10_q3} <= 30'b0;
      {y10_e3,y10_i3,y10_q3} <= 30'b0;
    end
    else
    begin
      if(c9_row!=4'd0)
      begin
        c10_en                 <= c9_en & enable;
        c10_col                <= c9_col;
        c10_row                <= c9_row;
        c10_last               <= c9_last ;
        {x10_e0,x10_i0,x10_q0} <= {x9_e0,x9_i0,x9_q0};
        {y10_e0,y10_i0,y10_q0} <= {y9_e0,y9_i0,y9_q0};
        {x10_e1,x10_i1,x10_q1} <= {x9_e1,x9_i1,x9_q1};
        {y10_e1,y10_i1,y10_q1} <= {y9_e1,y9_i1,y9_q1};
        {x10_e2,x10_i2,x10_q2} <= {x9_e2,x9_i2,x9_q2};
        {y10_e2,y10_i2,y10_q2} <= {y9_e2,y9_i2,y9_q2};
        {x10_e3,x10_i3,x10_q3} <= {x9_e3,x9_i3,x9_q3};
        {y10_e3,y10_i3,y10_q3} <= {y9_e3,y9_i3,y9_q3};
      end
      else
      begin
        c10_en               <= 1'b0;
        c10_col              <= 8'd0;
        c10_row              <= 4'd0;
        c10_last             <= 1'b0;
      end
    end
  end

  reg [ 7:0] c10_addr;
  always @(*)
  begin
    case(c10_row)
      4'd0:    c10_addr =  c10_col[7:0];
      4'd1:    c10_addr =  c10_col[7:0];
      4'd2:    c10_addr = {c10_col[7:2], c10_col[0], c10_col[  1]};
      4'd3:    c10_addr = {c10_col[7:3], c10_col[0], c10_col[2:1]};
      4'd4:    c10_addr = {c10_col[7:4], c10_col[0], c10_col[3:1]};
      4'd5:    c10_addr = {c10_col[7:5], c10_col[0], c10_col[4:1]};
      4'd6:    c10_addr = {c10_col[7:6], c10_col[0], c10_col[5:1]};
      4'd7:    c10_addr = {c10_col[  7], c10_col[0], c10_col[6:1]};
      default: c10_addr = {              c10_col[0], c10_col[7:1]};
    endcase
  end
  
  /*****************************************************************************
  * Y11
  *****************************************************************************/
  reg [ 3:0]    y11_e0;
  reg [12:0]    y11_i0, y11_q0;
  reg [ 3:0]    y11_e1;
  reg [12:0]    y11_i1, y11_q1;
  reg [ 3:0]    y11_e2;
  reg [12:0]    y11_i2, y11_q2;
  reg [ 3:0]    y11_e3;
  reg [12:0]    y11_i3, y11_q3;

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      {y11_e0,y11_i0,y11_q0} <= 30'b0;
      {y11_e1,y11_i1,y11_q1} <= 30'b0;
      {y11_e2,y11_i2,y11_q2} <= 30'b0;
      {y11_e3,y11_i3,y11_q3} <= 30'b0;
    end
    else
    begin
      {y11_e0,y11_i0,y11_q0} <= {y10_e0,y10_i0,y10_q0};
      {y11_e1,y11_i1,y11_q1} <= {y10_e1,y10_i1,y10_q1};
      {y11_e2,y11_i2,y11_q2} <= {y10_e2,y10_i2,y10_q2};
      {y11_e3,y11_i3,y11_q3} <= {y10_e3,y10_i3,y10_q3};
    end
  end
 
  /*****************************************************************************
  * W11
  *****************************************************************************/
  reg         c12_en;
  reg [ 7:0]  c12_addr;
  reg         c12_last;
  
  reg [ 3:0]  c12_row;
  reg [ 7:0]  c12_col;
  
  reg [ 3:0]  c12_e0;
  reg [12:0]  c12_i0, c12_q0;
  reg [ 3:0]  c12_e1;
  reg [12:0]  c12_i1, c12_q1;
  reg [ 3:0]  c12_e2;
  reg [12:0]  c12_i2, c12_q2;
  reg [ 3:0]  c12_e3;
  reg [12:0]  c12_i3, c12_q3;
  reg [ 3:0]  c12_e4;
  reg [12:0]  c12_i4, c12_q4;
  reg [ 3:0]  c12_e5;
  reg [12:0]  c12_i5, c12_q5;
  reg [ 3:0]  c12_e6;
  reg [12:0]  c12_i6, c12_q6;
  reg [ 3:0]  c12_e7;
  reg [12:0]  c12_i7, c12_q7;

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      done                   <= 1'b0;
      c12_en                 <= 1'b0;
      c12_row                <= 4'b0;
      c12_col                <= 8'b0;
      c12_addr               <= 8'b0;
      c12_last               <= 1'b0;
      {c12_e0,c12_i0,c12_q0} <= 30'b0;
      {c12_e1,c12_i1,c12_q1} <= 30'b0;
      {c12_e2,c12_i2,c12_q2} <= 30'b0;
      {c12_e3,c12_i3,c12_q3} <= 30'b0;
      {c12_e4,c12_i4,c12_q4} <= 30'b0;
      {c12_e5,c12_i5,c12_q5} <= 30'b0;
      {c12_e6,c12_i6,c12_q6} <= 30'b0;
      {c12_e7,c12_i7,c12_q7} <= 30'b0;
    end
    else if(!enable)
    begin
      done                   <= 1'b0;
      c12_en                 <= 1'b0;
      c12_last               <= 1'b0;
    end
    else
    begin
      if(c9_en && c9_row==4'd0)
      begin
        c12_en                 <= c9_en;
        c12_row                <= 4'b0;
        c12_col                <= c9_col;
        c12_addr               <= c9_col;
        c12_last               <= 1'b0;
        {c12_e0,c12_i0,c12_q0} <= {x9_e0,x9_i0,x9_q0};
        {c12_e1,c12_i1,c12_q1} <= {x9_e1,x9_i1,x9_q1};
        {c12_e2,c12_i2,c12_q2} <= {x9_e2,x9_i2,x9_q2};
        {c12_e3,c12_i3,c12_q3} <= {x9_e3,x9_i3,x9_q3};
        {c12_e4,c12_i4,c12_q4} <= {y9_e0,y9_i0,y9_q0};
        {c12_e5,c12_i5,c12_q5} <= {y9_e1,y9_i1,y9_q1};
        {c12_e6,c12_i6,c12_q6} <= {y9_e2,y9_i2,y9_q2};
        {c12_e7,c12_i7,c12_q7} <= {y9_e3,y9_i3,y9_q3};
      end
      else if(c10_en)
      begin
        c12_en    <= 1'b1;
        c12_addr  <= c10_addr;
        c12_row   <= c10_row;
        c12_col   <= c10_col;
        if(c10_col[0]==1'b0)
        begin
          c12_last               <= 1'b0;
          {c12_e0,c12_i0,c12_q0} <= {x10_e0,x10_i0,x10_q0};
          {c12_e1,c12_i1,c12_q1} <= {x10_e1,x10_i1,x10_q1};
          {c12_e2,c12_i2,c12_q2} <= {x10_e2,x10_i2,x10_q2};
          {c12_e3,c12_i3,c12_q3} <= {x10_e3,x10_i3,x10_q3};
          {c12_e4,c12_i4,c12_q4} <= {x9_e0,x9_i0,x9_q0};
          {c12_e5,c12_i5,c12_q5} <= {x9_e1,x9_i1,x9_q1};
          {c12_e6,c12_i6,c12_q6} <= {x9_e2,x9_i2,x9_q2};
          {c12_e7,c12_i7,c12_q7} <= {x9_e3,x9_i3,x9_q3};
        end
        else
        begin
          c12_last               <= c10_last;
          {c12_e0,c12_i0,c12_q0} <= {y11_e0,y11_i0,y11_q0};
          {c12_e1,c12_i1,c12_q1} <= {y11_e1,y11_i1,y11_q1};
          {c12_e2,c12_i2,c12_q2} <= {y11_e2,y11_i2,y11_q2};
          {c12_e3,c12_i3,c12_q3} <= {y11_e3,y11_i3,y11_q3};
          {c12_e4,c12_i4,c12_q4} <= {y10_e0,y10_i0,y10_q0};
          {c12_e5,c12_i5,c12_q5} <= {y10_e1,y10_i1,y10_q1};
          {c12_e6,c12_i6,c12_q6} <= {y10_e2,y10_i2,y10_q2};
          {c12_e7,c12_i7,c12_q7} <= {y10_e3,y10_i3,y10_q3};
        end
      end
      else
      begin
        c12_en                 <= 1'b0;
        c12_row                <= 4'b0;
        c12_col                <= 8'b0;
        c12_addr               <= 8'b0;
        c12_last               <= 1'b0;
        {c12_e0,c12_i0,c12_q0} <= 30'b0;
        {c12_e1,c12_i1,c12_q1} <= 30'b0;
        {c12_e2,c12_i2,c12_q2} <= 30'b0;
        {c12_e3,c12_i3,c12_q3} <= 30'b0;
        {c12_e4,c12_i4,c12_q4} <= 30'b0;
        {c12_e5,c12_i5,c12_q5} <= 30'b0;
        {c12_e6,c12_i6,c12_q6} <= 30'b0;
        {c12_e7,c12_i7,c12_q7} <= 30'b0;
      end
      
      if(c12_en && c12_last)
        done <= 1'b1;
      
    end
  end
  
  assign wen   = c12_en;
  assign waddr = c12_addr;
  assign wdata = {c12_e7,c12_q7,c12_i7,
                  c12_e6,c12_q6,c12_i6,
                  c12_e5,c12_q5,c12_i5,
                  c12_e4,c12_q4,c12_i4,
                  c12_e3,c12_q3,c12_i3,
                  c12_e2,c12_q2,c12_i2,
                  c12_e1,c12_q1,c12_i1,
                  c12_e0,c12_q0,c12_i0};
  
endmodule
`default_nettype wire
