////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -------------------------------------------------------------------------
// Dependencies     : 
// Description      : Main State Machine is the coordinating state machine,
//                    which controls the state of the whole modem.
//                    This block controls also the multiplexers which switch
//                    between the OFDM modem and the DSSS/CCK modem.
// Simulation Notes : 
// Synthesis Notes  : 
// Application Note : 
// Simulator        : 
// Parameters       : 
// Terms & concepts : 
// Bugs             : 
// Open issues and future enhancements : 
// References       : 
// Revision History : 
// -------------------------------------------------------------------------
// 
// $HeadURL: $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

`default_nettype none
module mfsm
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        clk,
  input  wire        rst_n,
  
  /*****************************************************************************
  * registers
  *****************************************************************************/
  input  wire [ 1:0] mdmcfg_conf_bw,
  
  input  wire [ 9:0] mdmcfg_rxfeofdm20_delay,
  input  wire [ 9:0] mdmcfg_rxfeofdm40_delay,
  input  wire [ 9:0] mdmcfg_rxfeofdm4020_delay,
  input  wire [12:0] mdmcfg_rxsifs_delay,
  
  input  wire [ 7:0] mdmcfg_txfeofdm_delay,
  input  wire [ 7:0] mdmcfg_txfedsss_delay,
  input  wire [ 9:0] mdmcfg_txstart_delay,
  input  wire        mdmcfg_rxdsss_en,

  /*****************************************************************************
  * mpif
  *****************************************************************************/
  input  wire        mpif_txv_ofdm_update,
  input  wire        mpif_txv_dsss_update,
  input  wire [ 2:0] mpif_rxv_ch_bandwidth,
  
  input  wire        mpif_keeprfon,
  output reg         mpif_phyerr,
  
  input  wire        mpif_txreq,
  input  wire        mpif_txabort,
  output reg         mpif_txend,
  
  input  wire        mpif_rxreq,
  output reg         mpif_sendrxv1,
  output reg         mpif_sendrxv2,
  output reg         mpif_rxendfortiming,
  input  wire        mpif_rxend,
  output reg         mpif_rxerr,
  input  wire        mpif_rxvector1_done,
  
 
  /*****************************************************************************
  * rc
  *****************************************************************************/
  output reg         rc_tx_req,
  input  wire        rc_tx_ack,
  output reg         rc_rx_req,
  input  wire        rc_rx_ack,
 
  /*****************************************************************************
  * agc
  *****************************************************************************/
  input  wire        agc_ofdm_lock,
  input  wire        agc_rxrifsdetected,
  input  wire        agc_dsss_lock,
  //
  output reg         agc_on,                // agc on
  output reg         agc_demod_run,         // demodulation is running
  output reg         agc_rxendtiming,
  output reg         agc_lsig_valid,

  /*****************************************************************************
  * front-end
  *****************************************************************************/
  output reg         rx_dsss_fe_en,

  /*****************************************************************************
  * ofdm tx
  *****************************************************************************/
  input  wire        ofdm_tx_error,
  output reg         ofdm_tx_enable,
  output reg         ofdm_tx_riu_release,
  input  wire        ofdm_tx_riu_done,

  /*****************************************************************************
  * ofdm rx
  *****************************************************************************/
  input  wire        ofdm_rx_error,
  input  wire        ofdm_rx_frame_unsupported,
  input  wire        ofdm_rx_frame_violation,
  input  wire        ofdm_rx_td_done,
  input  wire        ofdm_rx_fd_done,
  input  wire        ofdm_rx_sfo_done,
  input  wire        ofdm_rxv_update,
  input  wire        ofdm_agc_lsig_valid,
  output reg         ofdm_rx_td_enable,
  output reg         ofdm_rx_fd_enable,
  output reg         ofdm_rx_bd_enable,

  /*****************************************************************************
  * dsss tx
  *****************************************************************************/
  input  wire        dsss_tx_error,
  input  wire        dsss_tx_done,
  output reg         dsss_tx_enable,
  output reg         dsss_tx_start,

  /*****************************************************************************
  * dsss rx
  *****************************************************************************/
  input  wire [ 1:0] dsss_rx_error,
  input  wire        dsss_rx_done,
  input  wire        dsss_rxv_update,
  output reg         dsss_rx_enable,

  /*****************************************************************************
  * CRM
  *****************************************************************************/
  output wire        tx_clken,             // Tx clock enable
  output wire        rx_td_clken,          // Rx OFDM TD clock enable
  //
  output wire        dsss_clken,
  
  /*****************************************************************************
  * debug port
  *****************************************************************************/
  output reg  [3:0]  mfsm_state
);
 
//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
  localparam TX_ACK_TIMEOUT = 10'd480; // 4us @120MHz
  
  localparam IDLE           = 4'd0,
             TX_WAIT_SIFS   = 4'd1,
             TX_CHECK_TXACK = 4'd2,
             TX_ONGOING     = 4'd3,
             TX_END_DELAY   = 4'd4,
             TX_CLOSE       = 4'd5,
             RX_LISTEN      = 4'd6,
             RX_DEMOD       = 4'd7,
             RX_END         = 4'd8,
             RX_CLOSE       = 4'd9;
    
//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
  reg  [ 9:0] counter;
  wire [ 9:0] n_counter_decr;

  reg         ofdm_tx_enable_1t;
  reg         ofdm_tx_enable_2t;
  reg         ofdm_tx_enable_3t;
  reg         ofdm_tx_enable_4t;
  reg         ofdm_rx_td_enable_1t;
  reg         ofdm_rx_td_enable_2t;
  reg         ofdm_rx_td_enable_3t;
  reg         ofdm_rx_td_enable_4t;

  reg         dsss_clken_1t;
  reg         dsss_clken_2t;
  reg         dsss_clken_3t;
  reg         dsss_clken_4t;
  reg         dsss_clken_5t;
  reg         dsss_clken_6t;

  /*****************************************************************************
  * SIFS counter
  *****************************************************************************/
  reg          ofdm_rx_td_done_1t;
  reg  [12:0]  counter_sifs;
  wire [12:0]  n_counter_sifs;
  wire         sifs_expired;
  wire [12:0]  sifs_minus_txstart_delay;
  
  assign n_counter_sifs           = counter_sifs + 13'd1;
  assign sifs_minus_txstart_delay = mdmcfg_rxsifs_delay  - {3'b0,mdmcfg_txstart_delay};
  assign sifs_expired             = counter_sifs >= sifs_minus_txstart_delay;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      ofdm_rx_td_done_1t <= 1'b0;
      counter_sifs       <=13'h1fff;
    end
    else
    begin
      ofdm_rx_td_done_1t <= ofdm_rx_td_done; /* ofdm_rx_td_done & ~ofdm_rx_error */
     
      if(ofdm_rx_td_done && !ofdm_rx_td_done_1t)
      begin
        case(mdmcfg_conf_bw)
          2'd0:    /* 20M */
            counter_sifs <= {3'b0,mdmcfg_rxfeofdm20_delay};
          default: /* 40M */
            if(mpif_rxv_ch_bandwidth==3'd0)
              counter_sifs <= {3'b0,mdmcfg_rxfeofdm4020_delay};
            else
              counter_sifs <= {3'b0,mdmcfg_rxfeofdm40_delay};
        endcase
      end
      else if(counter_sifs!=13'h1fff)
        counter_sifs <= n_counter_sifs;
    end
  end
  
  /*****************************************************************************
  * OFDM TX RIU release counter
  *****************************************************************************/
  reg  [ 9:0] counter_release;
  wire [ 9:0] n_counter_release;
  wire [ 9:0] txstart_minus_fe1_delay;
  
  assign n_counter_release        = counter_release + 10'd1;
  assign txstart_minus_fe1_delay  = mdmcfg_txstart_delay - {2'b0,mdmcfg_txfeofdm_delay};
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      counter_release     <= 10'd0;
      ofdm_tx_riu_release <= 1'b0;
    end
    else if(rc_tx_req)
    begin
      if((counter_release>=txstart_minus_fe1_delay) || ofdm_tx_riu_release)
      begin
        ofdm_tx_riu_release <= 1'b1;
      end
      else if(counter_release!=10'h3ff)
      begin
        counter_release <= n_counter_release;
      end
    end
    else
    begin
      counter_release     <= 10'd0;
      ofdm_tx_riu_release <= 1'b0;
    end
  end
    
  /*****************************************************************************
  * MFSM
  *****************************************************************************/
  assign n_counter_decr = counter - 10'd1;
 
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      mfsm_state          <= IDLE;
      mpif_txend          <= 1'b0;
      mpif_phyerr         <= 1'b0;
      mpif_sendrxv1       <= 1'b0;
      mpif_sendrxv2       <= 1'b0;
      mpif_rxendfortiming <= 1'b0;
      mpif_rxerr          <= 1'b0;
      //
      rc_tx_req           <= 1'b0;
      rc_rx_req           <= 1'b0;
      //
      rx_dsss_fe_en       <= 1'b0;
      //
      ofdm_tx_enable      <= 1'b0;
      ofdm_rx_td_enable   <= 1'b0;
      ofdm_rx_fd_enable   <= 1'b0;
      ofdm_rx_bd_enable   <= 1'b0;
      //
      dsss_tx_enable      <= 1'b0;
      dsss_tx_start       <= 1'b0;
      dsss_rx_enable      <= 1'b0;
      //
      agc_on              <= 1'b0;
      agc_demod_run       <= 1'b0;
      agc_rxendtiming     <= 1'b0;
      agc_lsig_valid      <= 1'b0;
      //
      counter             <= 10'd0;
    end
    else
    begin
      // AGC specific signals
      agc_rxendtiming     <= mpif_rxendfortiming;
      agc_lsig_valid      <= (dsss_rxv_update & (dsss_rx_error == 2'b0)) 
                             || ofdm_agc_lsig_valid;
      
      // FE specific signals
      rx_dsss_fe_en       <= dsss_rx_enable;
      
      case(mfsm_state)
        // Go to Tx or Listen at MAC request
        IDLE:
        begin
          rc_rx_req         <= 1'b0;
          ofdm_rx_td_enable <= 1'b0;
          ofdm_rx_fd_enable <= 1'b0;
          ofdm_rx_bd_enable <= 1'b0;
          dsss_rx_enable    <= 1'b0;
          //
          agc_on            <= 1'b0;
          agc_demod_run     <= 1'b0;
          
          if(mpif_rxreq)
          begin
            if (rc_rx_ack && rc_rx_req)
            begin
              mfsm_state    <= RX_LISTEN;
              rc_rx_req     <= 1'b1;
            end
            else
              rc_rx_req     <= ~rc_rx_ack; // Safeguard on rc_rx_ack in idle before rc_rx_req generation
          end
          else if(mpif_txreq)
          begin
            if(sifs_expired)
            begin
              mfsm_state      <= TX_CHECK_TXACK;
              rc_tx_req       <= 1'b1;
              counter         <= TX_ACK_TIMEOUT;
            end
            else
            begin
              mfsm_state      <= TX_WAIT_SIFS;
            end
          end
        end
        
        // only for OFDM, ensure that (SIFS - TXSTARTDELAY) delay is expired
        TX_WAIT_SIFS:
        begin
          if(mpif_txabort)
          begin
            mfsm_state     <= TX_CLOSE;
          end
          else if(sifs_expired || mpif_txv_dsss_update)
          begin
            mfsm_state      <= TX_CHECK_TXACK;
            rc_tx_req       <= 1'b1;
            counter         <= TX_ACK_TIMEOUT;
          end
        end
        
        // Safeguard against wrong RF programming:
        // Send an error if rc_tx_ack is not received after 4 us
        TX_CHECK_TXACK:
        begin
          counter        <= n_counter_decr;
          //
          ofdm_tx_enable <= mpif_txv_ofdm_update;
          dsss_tx_enable <= mpif_txv_dsss_update;
          //
          
          if(mpif_txabort)
          begin
            mfsm_state     <= TX_CLOSE;
            ofdm_tx_enable <= 1'b0;
            rc_tx_req      <= 1'b0;
          end
          else if(ofdm_tx_error || dsss_tx_error)
          begin
            mfsm_state     <= TX_CLOSE;
            ofdm_tx_enable <= 1'b0;
            dsss_tx_enable <= 1'b0;
            dsss_tx_start  <= 1'b0;
            rc_tx_req      <= 1'b0;
            mpif_phyerr    <= 1'b1;
          end
          else if(mpif_txv_dsss_update && counter==(TX_ACK_TIMEOUT-mdmcfg_txstart_delay+{2'd0,mdmcfg_txfedsss_delay}))
          begin
            dsss_tx_start <= 1'b1;
          end
          else if(counter==10'd0)
          begin
            if(rc_tx_ack)
            begin
              mfsm_state    <= TX_ONGOING;
            end
            else
            begin
              mfsm_state     <= TX_CLOSE;
              ofdm_tx_enable <= 1'b0;
              dsss_tx_enable <= 1'b0;
              dsss_tx_start  <= 1'b0;
              rc_tx_req      <= 1'b0;
              mpif_phyerr    <= 1'b1;
            end
          end
        end
        
        TX_ONGOING:
        begin
          if(mpif_txabort)
          begin
            mfsm_state     <= TX_CLOSE;
            ofdm_tx_enable <= 1'b0;
            dsss_tx_enable <= 1'b0;
            dsss_tx_start  <= 1'b0;
            rc_tx_req      <= 1'b0;
          end
          else if(ofdm_tx_error || dsss_tx_error)
          begin
            mfsm_state     <= TX_CLOSE;
            ofdm_tx_enable <= 1'b0;
            dsss_tx_enable <= 1'b0;
            dsss_tx_start  <= 1'b0;
            rc_tx_req      <= 1'b0;
            mpif_phyerr    <= 1'b1;
          end
          else if(ofdm_tx_riu_done)
          begin
            mfsm_state     <= TX_END_DELAY;
            counter        <= {2'd0,mdmcfg_txfeofdm_delay};
          end
          else if(dsss_tx_done)
          begin
            mfsm_state     <= TX_END_DELAY;
            counter        <= {2'd0,mdmcfg_txfedsss_delay};
          end
        end
        
        TX_END_DELAY:
        begin
          counter <= n_counter_decr;
          if(mpif_txabort || (counter==10'd0))
          begin
            mfsm_state     <= TX_CLOSE;
            ofdm_tx_enable <= 1'b0;
            dsss_tx_enable <= 1'b0;
            dsss_tx_start  <= 1'b0;
            rc_tx_req      <= 1'b0;
          end
          else if(ofdm_tx_error || dsss_tx_error)
          begin
            mfsm_state     <= TX_CLOSE;
            ofdm_tx_enable <= 1'b0;
            dsss_tx_enable <= 1'b0;
            dsss_tx_start  <= 1'b0;
            rc_tx_req      <= 1'b0;
            mpif_phyerr    <= 1'b1;
          end
        end
        
        TX_CLOSE:
        begin
          mpif_phyerr    <= 1'b0;
          if(!rc_tx_ack)
          begin
            mpif_txend <= 1'b1;
            if(!mpif_txreq)
            begin
              mpif_txend <= 1'b0;
              mfsm_state <= IDLE;
            end
          end
        end

        RX_LISTEN:
        begin
          agc_on              <= 1'b1;
          agc_demod_run       <= 1'b0;
          dsss_rx_enable      <= 1'b0;
          if(!mpif_rxreq)
          begin
            mfsm_state        <= RX_CLOSE;
            rc_rx_req         <= 1'b0;
          end
          else if(agc_ofdm_lock && !ofdm_rxv_update) // Add check on rxv_update as safeguard due to resynchro
          begin
            mfsm_state        <= RX_DEMOD;
            ofdm_rx_td_enable <= 1'b1;
            ofdm_rx_fd_enable <= 1'b1;
            ofdm_rx_bd_enable <= 1'b1;
            agc_demod_run     <= 1'b1;
          end
          else if(agc_dsss_lock && !dsss_rxv_update && mdmcfg_rxdsss_en) // Add check on rxv_update as safeguard due to resynchro
          begin
            mfsm_state     <= RX_DEMOD;
            dsss_rx_enable <= 1'b1;
            agc_demod_run  <= 1'b1;
          end
        end
        
        RX_DEMOD:
        begin
          mpif_sendrxv1       <= mpif_sendrxv1       | (agc_ofdm_lock & ofdm_rxv_update & (~ofdm_rx_frame_violation | ofdm_rx_frame_unsupported)) |
                                                       (agc_dsss_lock & dsss_rxv_update & (dsss_rx_error == 2'b0));
          mpif_rxendfortiming <= mpif_rxendfortiming | (ofdm_rx_td_done & ~ofdm_rx_error) |
                                                       dsss_rx_done;
          //
          if(!mpif_rxreq)
          begin
            mfsm_state          <= RX_CLOSE;
            mpif_sendrxv1       <= 1'b0;
            mpif_sendrxv2       <= 1'b0;
            mpif_rxendfortiming <= 1'b0;
            mpif_rxerr          <= 1'b0;
            ofdm_rx_td_enable   <= 1'b0;
            ofdm_rx_fd_enable   <= 1'b0;
            ofdm_rx_bd_enable   <= 1'b0;
            rc_rx_req           <= 1'b0;
          end
          else if( (|dsss_rx_error)                     ||
                     ofdm_rx_error                      ||
                     ofdm_rx_frame_violation            ||
                     ofdm_rx_frame_unsupported          ||
                     (!agc_dsss_lock && dsss_rx_enable) ||
                     (!agc_ofdm_lock && ((ofdm_rx_fd_enable && ofdm_rx_td_enable) || !agc_demod_run)))
          begin
            if (mpif_sendrxv1)
            begin
              if (mpif_rxvector1_done)
              begin
                mfsm_state        <= RX_END; // Move state only after rx vector 1 has been sent to MAC
                ofdm_rx_td_enable <= 1'b0;
                ofdm_rx_fd_enable <= 1'b0;
              end
              mpif_sendrxv2       <= 1'b1;
              mpif_rxerr          <= 1'b1;
              mpif_rxendfortiming <= 1'b1;
            end
            else if (!agc_dsss_lock && !agc_ofdm_lock)
            begin
              mfsm_state          <= RX_LISTEN;
              mpif_rxendfortiming <= 1'b0;
              ofdm_rx_td_enable   <= 1'b0;
              ofdm_rx_fd_enable   <= 1'b0;
              ofdm_rx_bd_enable   <= 1'b0;
            end
            else if (ofdm_rx_frame_violation || ofdm_rx_error)
            begin
              agc_demod_run       <= 1'b0; // Stop AGC on frame violation, might be a false alarm
              ofdm_rx_td_enable   <= 1'b0;
              ofdm_rx_fd_enable   <= 1'b0;
              ofdm_rx_bd_enable   <= 1'b0;
            end
          end
          else if(ofdm_rx_td_done)
          begin
            ofdm_rx_td_enable   <= 1'b0;
          end
          else if(((ofdm_rx_fd_done && ofdm_rx_sfo_done) || dsss_rx_done) && mpif_rxendfortiming && mpif_rxvector1_done)
          begin
            mfsm_state        <= RX_END;
            mpif_sendrxv2     <= 1'b1;
          end
        end
        
        RX_END:
        begin
          agc_demod_run         <= 1'b0;
          if(!mpif_rxreq)
          begin
            mfsm_state          <= RX_CLOSE;
            mpif_sendrxv1       <= 1'b0;
            mpif_sendrxv2       <= 1'b0;
            mpif_rxendfortiming <= 1'b0;
            mpif_rxerr          <= 1'b0;
            ofdm_rx_td_enable   <= 1'b0;
            ofdm_rx_fd_enable   <= 1'b0;
            ofdm_rx_bd_enable   <= 1'b0;
            rc_rx_req           <= 1'b0;
          end
          else if(mpif_rxend)
          begin
            ofdm_rx_td_enable   <= 1'b0;
            ofdm_rx_fd_enable   <= 1'b0;
            ofdm_rx_bd_enable   <= 1'b0;
            if((!agc_ofdm_lock || agc_rxrifsdetected) && !agc_dsss_lock) 
            begin
              // Wait for AGC unlock before moving back to listen
              mfsm_state          <= RX_LISTEN;
              mpif_sendrxv1       <= 1'b0;
              mpif_sendrxv2       <= 1'b0;
              mpif_rxendfortiming <= 1'b0;
              mpif_rxerr          <= 1'b0;
            end
          end
        end
        
        RX_CLOSE:
        begin
          agc_on              <= 1'b0;
          agc_demod_run       <= 1'b0;
          dsss_rx_enable      <= 1'b0;
          if(!rc_rx_ack)
          begin
            mfsm_state <= IDLE;
          end
        end

        // pragma coverage block = off
        default: ;
        // pragma coverage block = on
        
      endcase
    end
  end
  
// Clock enable
always@(posedge clk or negedge rst_n)
begin
  if(rst_n==1'b0)
  begin
    ofdm_tx_enable_1t    <= 1'b0;
    ofdm_tx_enable_2t    <= 1'b0;
    ofdm_tx_enable_3t    <= 1'b0;
    ofdm_tx_enable_4t    <= 1'b0;
    //
    ofdm_rx_td_enable_1t <= 1'b0;
    ofdm_rx_td_enable_2t <= 1'b0;
    ofdm_rx_td_enable_3t <= 1'b0;
    ofdm_rx_td_enable_4t <= 1'b0;
    //
    dsss_clken_1t        <= 1'b0;
    dsss_clken_2t        <= 1'b0;
    dsss_clken_3t        <= 1'b0;
    dsss_clken_4t        <= 1'b0;
    dsss_clken_5t        <= 1'b0;
    dsss_clken_6t        <= 1'b0;
  end
  else
  begin
    ofdm_tx_enable_1t <= ofdm_tx_enable;
    ofdm_tx_enable_2t <= ofdm_tx_enable_1t;
    ofdm_tx_enable_3t <= ofdm_tx_enable_2t;
    ofdm_tx_enable_4t <= ofdm_tx_enable_3t;
    //
    ofdm_rx_td_enable_1t <= ofdm_rx_td_enable;
    ofdm_rx_td_enable_2t <= ofdm_rx_td_enable_1t;
    ofdm_rx_td_enable_3t <= ofdm_rx_td_enable_2t;
    ofdm_rx_td_enable_4t <= ofdm_rx_td_enable_3t;
    //
    dsss_clken_1t        <= dsss_tx_enable | dsss_rx_enable;
    dsss_clken_2t        <= dsss_clken_1t;
    dsss_clken_3t        <= dsss_clken_2t;
    dsss_clken_4t        <= dsss_clken_3t;
    dsss_clken_5t        <= dsss_clken_4t;
    dsss_clken_6t        <= dsss_clken_5t;
  end
end

assign tx_clken     = ofdm_tx_enable | ofdm_tx_enable_4t;
assign rx_td_clken  = ofdm_rx_td_enable | ofdm_rx_td_enable_4t;
//
assign dsss_clken   = dsss_tx_enable | dsss_rx_enable | dsss_clken_6t;
  
  /*****************************************************************************
  * SIMU
  *****************************************************************************/
  // pragma coverage block = off
`ifdef RW_SIMU_ON
  reg [32*8-1:0] str_mfsm_state;

  always @(*)
  begin
    case(mfsm_state)
      IDLE:           str_mfsm_state = "IDLE";
      TX_WAIT_SIFS:   str_mfsm_state = "TX_WAIT_SIFS";
      TX_CHECK_TXACK: str_mfsm_state = "TX_CHECK_TXACK";
      TX_ONGOING:     str_mfsm_state = "TX_ONGOING";
      TX_END_DELAY:   str_mfsm_state = "TX_END_DELAY";
      TX_CLOSE:       str_mfsm_state = "TX_CLOSE";
      RX_LISTEN:      str_mfsm_state = "RX_LISTEN";
      RX_DEMOD:       str_mfsm_state = "RX_DEMOD";
      RX_END:         str_mfsm_state = "RX_END";
      RX_CLOSE:       str_mfsm_state = "RX_CLOSE";
      default:        str_mfsm_state = "UNKNOWN";
    endcase
  end
`endif
  // pragma coverage block = on
  
endmodule
`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
