//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: 2012-07-12 11:17:35 +0200 (Thu, 12 Jul 2012) $
// ---------------------------------------------------------------------------
// Dependencies     :
// Description      : Complex multiplication module
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
// $HeadURL :  $
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module ComplexMult #(parameter INPUT_WIDTH = 13,
                     parameter MUL_TYPE = 0
                    )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                   PhyClk,
            input   wire                                   nPhyRst,
            //Data
            input   wire   signed     [INPUT_WIDTH-1:0]    ARe, // Real part of A
            input   wire   signed     [INPUT_WIDTH-1:0]    AIm, // Imaginary part of A
            input   wire   signed     [INPUT_WIDTH-1:0]    BRe, // Real part of B
            input   wire   signed     [INPUT_WIDTH-1:0]    BIm, // Imaginary part of B
            input   wire                                   DataInEn, // Data Enable

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output  wire   signed     [2*INPUT_WIDTH:0]    CRe,
            output  wire   signed     [2*INPUT_WIDTH:0]    CIm
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam    signed   [INPUT_WIDTH-1:0]     CONST_ZERO_SIG_INPUT_WIDTH    = {{INPUT_WIDTH}{1'b0}};
localparam    signed   [INPUT_WIDTH:0]       CONST_ZERO_SIG_INPUT_WIDTH_P1 = {{(INPUT_WIDTH+1)}{1'b0}};
localparam    signed   [2*INPUT_WIDTH-1:0]   CONST_ZERO_INPUT_WIDTH_X2     = {{(2*INPUT_WIDTH)}{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire  signed      [INPUT_WIDTH-1:0]    a ;
wire  signed      [INPUT_WIDTH-1:0]    b ;
wire  signed      [INPUT_WIDTH-1:0]    c ;
wire  signed      [INPUT_WIDTH-1:0]    d ;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg   signed      [INPUT_WIDTH  :0]    MultIn00 ;
reg   signed      [INPUT_WIDTH  :0]    MultIn01 ;
reg   signed      [INPUT_WIDTH  :0]    MultIn10 ;
reg   signed      [INPUT_WIDTH  :0]    MultIn11 ;
reg   signed      [INPUT_WIDTH  :0]    MultIn20 ;
reg   signed      [INPUT_WIDTH  :0]    MultIn21 ;
reg   signed      [2*INPUT_WIDTH-1:0]  MultOut0 ;
reg   signed      [2*INPUT_WIDTH-1:0]  MultOut1 ;
reg   signed      [2*INPUT_WIDTH-1:0]  MultOut2 ;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Values are input to the module only if the DataInEn is set
assign a = DataInEn ? ARe : CONST_ZERO_SIG_INPUT_WIDTH;
assign b = DataInEn ? AIm : CONST_ZERO_SIG_INPUT_WIDTH;
assign c = DataInEn ? BRe : CONST_ZERO_SIG_INPUT_WIDTH;
assign d = DataInEn ? BIm : CONST_ZERO_SIG_INPUT_WIDTH;

generate
  if(MUL_TYPE == 1) begin : MUL_TYPE1_GEN

    always @(posedge PhyClk or negedge nPhyRst) begin
      if(nPhyRst == 1'b0 ) begin
        MultIn00 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn01 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn10 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn11 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn20 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn21 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultOut0 <= CONST_ZERO_INPUT_WIDTH_X2;
        MultOut1 <= CONST_ZERO_INPUT_WIDTH_X2;
        MultOut2 <= CONST_ZERO_INPUT_WIDTH_X2;
      end
      else begin
        // If MUL_TYPE == 1,it means that
        // b will always be zero, so we need not use the
        // optimised multiplier. Rather, we can do the
        // traditional way , which uses only 2 multipliers
        MultIn00 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn01 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn10 <= -a;
        MultIn11 <= d ;
        MultIn20 <= -a;
        MultIn21 <= c ;
        MultOut0 <= CONST_ZERO_INPUT_WIDTH_X2;
        MultOut1 <= MultIn10 * MultIn11;       // c(a-b)
        MultOut2 <= MultIn20 * MultIn21;       // d(a+b)
      end
    end

  end
endgenerate

generate
  if(MUL_TYPE == 0) begin : MUL_TYPE0_GEN

    always @(posedge PhyClk or negedge nPhyRst) begin
      if(nPhyRst == 1'b0 ) begin
        MultIn00 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn01 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn10 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn11 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn20 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultIn21 <= CONST_ZERO_SIG_INPUT_WIDTH_P1;
        MultOut0 <= CONST_ZERO_INPUT_WIDTH_X2;
        MultOut1 <= CONST_ZERO_INPUT_WIDTH_X2;
        MultOut2 <= CONST_ZERO_INPUT_WIDTH_X2;
      end
      else begin
        // When MUL_TYPE = 0, this implements the optimised complex multiplier.
        // The following identity is used:
        // (a +jb) * (c +jd)  = [(a(c+d)-d(a+b)] + j[a(c+d) - c(a-b)]
        //                      --- CRe  ------  + ---  CIm ---------
        // which uses lesser number of multipliers.
        MultIn00 <= a;
        MultIn01 <= c + d;
        MultIn10 <= a - b;
        MultIn11 <= c ;
        MultIn20 <= a + b;
        MultIn21 <= d ;
        MultOut0 <= MultIn00 * MultIn01;  // a(c+d)
        MultOut1 <= MultIn10 * MultIn11;  // c(a-b)
        MultOut2 <= MultIn20 * MultIn21;  // d(a+b)
      end
    end

  end
endgenerate

assign CRe = MultOut0 - MultOut2;
assign CIm = MultOut0 - MultOut1;

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
