//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 7840 $
// $Date: 2013-05-15 15:05:40 +0200 (Wed, 15 May 2013) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Combinatorial Rotation Stage
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       : cordic_combstage module from RW_WLAN RefIP
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/Common/DSP/verilog/CordicCombStage.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CordicCombStage #(parameter NCOMBSTG    =  4, //Number of Combinatorial stages
                         parameter CCDATAWIDTH = 18, //Width of Input Data
                         parameter ANGLEWIDTH  = 16, //Width of Input Angle
                         parameter NIP         =  1, //Number of Inputs (1 to 4)
                         parameter STARTSTAGE  =  0 //NCOMBSTG * n (n = 0 to NPIPE)
                        )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                          nPhyRst, //Active LOW Reset
            input    wire                          PhyClk,  //PHY Clock
            //Control Signals
            input    wire                          Enable,  //Block Enable 
            input    wire      [ANGLEWIDTH-1:0]    AngleIn, //Angle for Rotation
            //Data
            input    wire      [CCDATAWIDTH-1:0]   ReDataIn0, //Real Comp of Data 0
            input    wire      [CCDATAWIDTH-1:0]   ImDataIn0, //Imag Comp of Data 0
            input    wire      [CCDATAWIDTH-1:0]   ReDataIn1, //Real Comp of Data 1
            input    wire      [CCDATAWIDTH-1:0]   ImDataIn1, //Imag Comp of Data 1
            input    wire      [CCDATAWIDTH-1:0]   ReDataIn2, //Real Comp of Data 2
            input    wire      [CCDATAWIDTH-1:0]   ImDataIn2, //Imag Comp of Data 2
            input    wire      [CCDATAWIDTH-1:0]   ReDataIn3, //Real Comp of Data 3
            input    wire      [CCDATAWIDTH-1:0]   ImDataIn3, //Imag Comp of Data 3
            //ArcTan Values
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit31, //Reference Bit 31
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit30, //Reference Bit 30
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit29, //Reference Bit 29
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit28, //Reference Bit 28
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit27, //Reference Bit 27
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit26, //Reference Bit 26 
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit25, //Reference Bit 25
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit24, //Reference Bit 24
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit23, //Reference Bit 23
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit22, //Reference Bit 22
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit21, //Reference Bit 21
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit20, //Reference Bit 20
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit19, //Reference Bit 19
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit18, //Reference Bit 18
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit17, //Reference Bit 17
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit16, //Reference Bit 16
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit15, //Reference Bit 15
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit14, //Reference Bit 14
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit13, //Reference Bit 13
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit12, //Reference Bit 12
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit11, //Reference Bit 11
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit10, //Reference Bit 10
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit9,  //Reference Bit  9
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit8,  //Reference Bit  8
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit7,  //Reference Bit  7
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit6,  //Reference Bit  6
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit5,  //Reference Bit  5
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit4,  //Reference Bit  4
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit3,  //Reference Bit  3
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit2,  //Reference Bit  2
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit1,  //Reference Bit  1
            input    wire      [NCOMBSTG-1:0]      ArcTanArrRefBit0,  //Reference Bit  0

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Data
            output    reg      [CCDATAWIDTH-1:0]   ReDataOut0, //Real Comp of Data 0
            output    reg      [CCDATAWIDTH-1:0]   ImDataOut0, //Imag Comp of Data 0
            output    reg      [CCDATAWIDTH-1:0]   ReDataOut1, //Real Comp of Data 1
            output    reg      [CCDATAWIDTH-1:0]   ImDataOut1, //Imag Comp of Data 1
            output    reg      [CCDATAWIDTH-1:0]   ReDataOut2, //Real Comp of Data 2
            output    reg      [CCDATAWIDTH-1:0]   ImDataOut2, //Imag Comp of Data 2
            output    reg      [CCDATAWIDTH-1:0]   ReDataOut3, //Real Comp of Data 3
            output    reg      [CCDATAWIDTH-1:0]   ImDataOut3, //Imag Comp of Data 3
            //Angle
            output    reg      [ANGLEWIDTH-1:0]    AngleOut   //Remaining Angle
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam     [ANGLEWIDTH-1:0]   CONST_ZERO_ANGLEWIDTH   = {ANGLEWIDTH{1'b0}};
localparam     [CCDATAWIDTH-1:0]  CONST_ZERO_CCDATAWIDTH  = {CCDATAWIDTH{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
wire               [NCOMBSTG-1:0]      ZSign;
wire               [CCDATAWIDTH-1:0]   X0Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   Y0Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   X1Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   Y1Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   X2Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   Y2Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   X3Int[NCOMBSTG:0];
wire               [CCDATAWIDTH-1:0]   Y3Int[NCOMBSTG:0];
wire               [ANGLEWIDTH-1:0]    ZInt[NCOMBSTG:0];
wire               [ANGLEWIDTH-1:0]    ArcTanArray[NCOMBSTG-1:0];
wire               [31:0]              ArcTanArrRef[NCOMBSTG-1:0];

//Genvars
genvar i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign ZInt[0]  = AngleIn;
assign X0Int[0] = ReDataIn0;
assign Y0Int[0] = ImDataIn0;
assign X1Int[0] = ReDataIn1;
assign Y1Int[0] = ImDataIn1;
assign X2Int[0] = ReDataIn2;
assign Y2Int[0] = ImDataIn2;
assign X3Int[0] = ReDataIn3;
assign Y3Int[0] = ImDataIn3;

always @(negedge nPhyRst or posedge PhyClk)
   begin: Outputs_Blk
      if (nPhyRst == 1'b0) begin
         AngleOut   <= CONST_ZERO_ANGLEWIDTH;
         ReDataOut0 <= CONST_ZERO_CCDATAWIDTH;
         ImDataOut0 <= CONST_ZERO_CCDATAWIDTH;
         if (NIP >= 32'd2) begin
            ReDataOut1 <= CONST_ZERO_CCDATAWIDTH;
            ImDataOut1 <= CONST_ZERO_CCDATAWIDTH;
         end
         if (NIP >= 32'd3) begin
            ReDataOut2 <= CONST_ZERO_CCDATAWIDTH;
            ImDataOut2 <= CONST_ZERO_CCDATAWIDTH;
         end
         if (NIP == 32'd4) begin
            ReDataOut3 <= CONST_ZERO_CCDATAWIDTH;
            ImDataOut3 <= CONST_ZERO_CCDATAWIDTH;
         end
      end //nPhyRst == 1'b0
      else if (Enable == 1'b1) begin
         AngleOut   <= ZInt[NCOMBSTG];
         ReDataOut0 <= X0Int[NCOMBSTG];
         ImDataOut0 <= Y0Int[NCOMBSTG];
         if (NIP >= 32'd2) begin
            ReDataOut1 <= X1Int[NCOMBSTG];
            ImDataOut1 <= Y1Int[NCOMBSTG];
         end
         if (NIP >= 32'd3) begin
            ReDataOut2 <= X2Int[NCOMBSTG];
            ImDataOut2 <= Y2Int[NCOMBSTG];
         end
         if (NIP == 32'd4) begin
            ReDataOut3 <= X3Int[NCOMBSTG];
            ImDataOut3 <= Y3Int[NCOMBSTG];
         end
      end //Enable == 1'b1

   end //Outputs_Blk
  
//ArcTan Array Rearrangement
generate
   for(i = 0; i < NCOMBSTG; i = i+1) begin: ArcTanArrRef_Blk
      assign ArcTanArrRef[i] = {ArcTanArrRefBit31[i], 
                                ArcTanArrRefBit30[i],
                                ArcTanArrRefBit29[i],
                                ArcTanArrRefBit28[i],
                                ArcTanArrRefBit27[i],
                                ArcTanArrRefBit26[i],
                                ArcTanArrRefBit25[i],
                                ArcTanArrRefBit24[i],
                                ArcTanArrRefBit23[i],
                                ArcTanArrRefBit22[i],
                                ArcTanArrRefBit21[i],
                                ArcTanArrRefBit20[i],
                                ArcTanArrRefBit19[i],
                                ArcTanArrRefBit18[i],
                                ArcTanArrRefBit17[i],
                                ArcTanArrRefBit16[i],
                                ArcTanArrRefBit15[i],
                                ArcTanArrRefBit14[i],
                                ArcTanArrRefBit13[i],
                                ArcTanArrRefBit12[i],
                                ArcTanArrRefBit11[i],
                                ArcTanArrRefBit10[i],
                                ArcTanArrRefBit9[i],
                                ArcTanArrRefBit8[i],
                                ArcTanArrRefBit7[i],
                                ArcTanArrRefBit6[i],
                                ArcTanArrRefBit5[i],
                                ArcTanArrRefBit4[i],
                                ArcTanArrRefBit3[i],
                                ArcTanArrRefBit2[i],
                                ArcTanArrRefBit1[i],
                                ArcTanArrRefBit0[i]};

      assign ZSign[i] = ZInt[i][ANGLEWIDTH-1];
      assign ArcTanArray[i] = (ZSign[i]) ? ({3'b000,
                              ArcTanArrRef[i][31:(31-ANGLEWIDTH+4)]}):
                              (~({3'b000,
                              ArcTanArrRef[i][31:(31-ANGLEWIDTH+4)]})
                              + {{(ANGLEWIDTH-1){1'b0}}, 1'b1});
      assign ZInt[i+1] = ZInt[i] + ArcTanArray[i];

      //Microrotation Stages
      //Shift Adder 0
      ShiftAdder #(
                   //Parameters
                   .SADATAWIDTH(CCDATAWIDTH),
                   .STAGE(i+STARTSTAGE)
                  ) U_ShiftAdder0 (
                                   .ZSign(ZSign[i]),
                                   .XIn(X0Int[i]),
                                   .YIn(Y0Int[i]),
                                   .XOut(X0Int[i+1]),
                                   .YOut(Y0Int[i+1])
                                  );

      //Shift Adder 1
      if (NIP >= 32'd2) begin
         ShiftAdder #(
                      //Parameters
                      .SADATAWIDTH(CCDATAWIDTH),
                      .STAGE(i+STARTSTAGE)
                     ) U_ShiftAdder1 (
                                      .ZSign(ZSign[i]),
                                      .XIn(X1Int[i]),
                                      .YIn(Y1Int[i]),
                                      .XOut(X1Int[i+1]),
                                      .YOut(Y1Int[i+1])
                                     );
      end //NIP >= 2

      //Shift Adder 2
      if (NIP >= 32'd3) begin
         ShiftAdder #(
                      //Parameters
                      .SADATAWIDTH(CCDATAWIDTH),
                      .STAGE(i+STARTSTAGE)
                     ) U_ShiftAdder2 (
                                      .ZSign(ZSign[i]),
                                      .XIn(X2Int[i]),
                                      .YIn(Y2Int[i]),
                                      .XOut(X2Int[i+1]),
                                      .YOut(Y2Int[i+1])
                                     );
      end //NIP >= 3

      //Shift Adder 3
      if (NIP == 32'd4) begin
         ShiftAdder #(
                      //Parameters
                      .SADATAWIDTH(CCDATAWIDTH),
                      .STAGE(i+STARTSTAGE)
                     ) U_ShiftAdder3 (
                                      .ZSign(ZSign[i]),
                                      .XIn(X3Int[i]),
                                      .YIn(Y3Int[i]),
                                      .XOut(X3Int[i+1]),
                                      .YOut(Y3Int[i+1])
                                     );
      end //NIP >= 4

   end //ArcTanArrRef_Blk
endgenerate


endmodule  //CordicCombStage

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
