//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 16168 $
// $Date: 2014-09-25 14:30:37 +0200 (Thu, 25 Sep 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : General Purpose Cordic for Phase Calculation
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL:  $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CordicMag #( parameter DATAWIDTH    = 16, //Input Data Width
                    parameter ANGLEWIDTH   = 21, //Internal Angle Width
                    parameter SCALING      =  1  //Scaling for ArcTan Table
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, // Active LOW Reset
            input    wire                                  PhyClk,  // PHY Clock
            //Control Signals
            input    wire                                  Load, // Load input values.
            //Data
            input    wire      [DATAWIDTH-1:0]             XIn, //Real part
            input    wire      [DATAWIDTH-1:0]             YIn, //Imaginary part

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output     reg                                 ModOutValid,
            output     reg     [DATAWIDTH+1:0]             ModOut
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters
//////////////////////////////////////////////////////////////////////////////
localparam [DATAWIDTH-1:0]  CONST_ZERO        = {DATAWIDTH{1'b0}};
localparam                  CONST_NDECP       = 3;
localparam                  NBITER            = DATAWIDTH;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [ANGLEWIDTH-1:0]    ArcTan; 
wire               [ANGLEWIDTH:0]      ArcTanExt;
wire               [ANGLEWIDTH:0]      Zn1;
wire               [ANGLEWIDTH:0]      ZnStep;
wire               [DATAWIDTH+1:0]     XnStep;
wire               [DATAWIDTH+1:0]     YnStep;
wire               [DATAWIDTH+1:0]     XnShiftArray[NBITER-1:0];
wire               [DATAWIDTH+1:0]     YnShiftArray[NBITER-1:0];
wire               [DATAWIDTH+1:0]     TempSynX;
wire               [DATAWIDTH+1:0]     TempSynY;
wire               [DATAWIDTH+1:0]     Xn1;
wire               [DATAWIDTH+1:0]     Yn1;
wire                                   Sn;
wire               [DATAWIDTH+1:0]     XnShift;
wire               [DATAWIDTH+1:0]     YnShift;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [DATAWIDTH+1:0]     Xn;
reg                [DATAWIDTH+1:0]     Yn;
reg                [ANGLEWIDTH:0]      Zn;
reg                [4:0]               Index;
reg                                    ModRdy;
reg                                    ModRdyD;
reg                                    Busy;
wire                                   ResultIsReal;

//Genvars
genvar g;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Detect when result becomes real to abort iterations
assign ResultIsReal = (Yn1 == CONST_ZERO);

//This block increases the Index from 0 to NBITER-1.
//The value in the index is the iteration number ('n').
always @(posedge PhyClk or negedge nPhyRst)
   begin: Control_Blk
      if (nPhyRst == 1'b0) begin
         Index       <= NBITER-1;
         ModOut      <= {(DATAWIDTH+2){1'b0}};
         ModOutValid <= 1'b0;
         ModRdy      <= 1'b0;
         ModRdyD     <= 1'b0;
         Busy        <= 1'b0;
      end
      else begin
         ModRdyD     <= ModRdy ;
         ModOut      <= Xn1;
         ModOutValid <= ModRdy & ~ModRdyD; 
         if (Load == 1'b1) begin //Reset Index
            ModRdy <= 1'b0;
            Index  <= 5'b0; // This trigs Index count up
            Busy   <= 1'b1;
         end
         else begin
            if (Index == NBITER - 5'd2) begin
               ModRdy <= 1'b1;
               Busy   <= 1'b0;
            end

            if (Index <= NBITER - 5'd2) begin
              if (ResultIsReal==1'b1) begin // Stop computation if result is real
                 ModRdy <= 1'b1; // Stop computation
                 Index  <= NBITER-1;
                 Busy   <= 1'b0;
              end else begin
                 Index <= Index + 5'b1;
              end
            end
         end
      end
   end //Control_Blk

//Remaining real part sign, to determine direction of next microrotation.
assign Sn = Yn[DATAWIDTH];

//Extend sign bit of ArcTan look-up table value.
assign ArcTanExt = {{(CONST_NDECP+1){1'b0}}, ArcTan[ANGLEWIDTH-1:CONST_NDECP]};

//Calculate next angle microrotation:
//Zn1 = Zn - Sn*ArcTan(2^-n)  with  Sn = -1  if Zn <  0; 1  if Zn >= 0
assign ZnStep = (Sn == 1'b0) ? ArcTanExt : ~(ArcTanExt) + {{{ANGLEWIDTH}{1'b0}},1'b1};
assign Zn1    = Zn + ZnStep;

//Load input value or store microrotation result.
always @(posedge PhyClk or negedge nPhyRst)
   begin: AccBlk
      if (nPhyRst == 1'b0) begin
         Xn <= {(DATAWIDTH+2){1'b0}};
         Yn <= {(DATAWIDTH+2){1'b0}};
         Zn <= {(ANGLEWIDTH+1){1'b0}};
      end
      else begin
         if (Load == 1'b1) begin
            // Load angle init value (positive).
            Xn <= {2'b0, XIn};
            Yn <= {2'b0, YIn};
            Zn <= {(ANGLEWIDTH+1){1'b0}};
         end
         else if (Busy==1'b1) begin
            // Store microrotation result.
            Xn <= Xn1;
            Yn <= Yn1;
            Zn <= Zn1; // Angle microrotation.
         end
      end
   end //AccBlk

//Microrotation to calculate Yn1 uses the value XnShift = Xn*2^-n.
assign XnShift = XnShiftArray[Index];

//Microrotation to calculate Xn1 uses the value YnShift = Yn*2^-n.
assign YnShift = YnShiftArray[Index];

//Calculate next X value:
//Xn1 = Xn - Sn*(2^-n)*Yn.
assign XnStep = (Sn == 1'b0) ? YnShift : ~(YnShift) + {{{DATAWIDTH+1}{1'b0}},1'b1};
assign Xn1    = Xn + XnStep;

//Calculate next Y value:
//Yn1 = Yn + Sn*(2^-n)*Xn.
assign YnStep = (Sn == 1'b1) ? XnShift : ~(XnShift) + {{{DATAWIDTH+1}{1'b0}},1'b1};
assign Yn1 = Yn + YnStep;

assign XnShiftArray[0] = Xn;
assign YnShiftArray[0] = Yn;

//Use TempSynY signal for Synopsys work-around.
assign TempSynY = {(DATAWIDTH+2){Yn[DATAWIDTH+1]}};
assign TempSynX = {(DATAWIDTH+2){Xn[DATAWIDTH+1]}};

generate

   for(g = 1; g < NBITER ; g = g + 1) begin : SAR_Blk
      assign XnShiftArray[g][DATAWIDTH+1:DATAWIDTH-g+2]
                         = TempSynX[DATAWIDTH+1:DATAWIDTH-g+2];
      assign XnShiftArray[g][DATAWIDTH+1-g:0] = Xn[DATAWIDTH+1:g];

      assign YnShiftArray[g][DATAWIDTH+1:DATAWIDTH-g+2] 
                         = TempSynY[DATAWIDTH+1:DATAWIDTH-g+2];
      assign YnShiftArray[g][DATAWIDTH+1-g:0] = Yn[DATAWIDTH+1:g];
   end //SAR_Blk

endgenerate


//Instantiate ArcTanLUT
ArcTanLUT # (
             //Parameters
             .ANGLEWIDTH(ANGLEWIDTH),
             .SCALING(SCALING)
            )U_ATLUT0 (
                       //Input
                       .Index(Index),
                       //Output
                       .ArcTan(ArcTan)
                      );

endmodule //CordicVect

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
