//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 17048 $
// $Date: 2014-11-26 13:44:24 +0100 (Wed, 26 Nov 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Cordic Vector Top Level
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/IPs/HW/Modem/Src/MODEMCORE/OFDMCORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/CordicVectTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CordicMagTop #(parameter DATAWIDTH    = 21, //Input Data Width
                      parameter ANGLEWIDTH   = 16, //Internal Angle Width
                      parameter SCALING      =  1  //Scaling for ArcTan Table
                     )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                   nPhyRst, // Active LOW Reset
            input   wire                                   PhyClk,  // PHY Clock
            //Control Signals
            input   wire                                   DataInEn, // Qualifies input values.
            //Data
            input   wire  signed  [DATAWIDTH-1:0]          ReDataIn, //Real part
            input   wire  signed  [DATAWIDTH-1:0]          ImDataIn, //Imaginary part

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signal
            output   reg                                   CordicDone, //Modulus Computation Done
            //Modulus
            output   reg           [DATAWIDTH:0]           ModOut
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed [DATAWIDTH+1:0]  CONST_1_DATAWIDTH_P2  = {{{(DATAWIDTH+1)}{1'b0}},1'b1};
localparam signed [DATAWIDTH+1:0]  CONST_2_DATAWIDTH_P2  = 'd2;
localparam signed [DATAWIDTH+1:0]  CONST_4_DATAWIDTH_P2  = 'd4;
localparam signed [DATAWIDTH+1:0]  CONST_5_DATAWIDTH_P2  = 'd5;
localparam signed [DATAWIDTH+1:0]  CONST_7_DATAWIDTH_P2  = 'd7;
localparam signed [DATAWIDTH+1:0]  CONST_8_DATAWIDTH_P2  = 'd8;
localparam signed [DATAWIDTH+1:0]  CONST_11_DATAWIDTH_P2 = 'd11;
localparam signed [DATAWIDTH+1:0]  CONST_12_DATAWIDTH_P2 = 'd12;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [DATAWIDTH-1:0]     NegReDataIn;
wire               [DATAWIDTH-1:0]     NegImDataIn;
wire               [DATAWIDTH-1:0]     AbsReDataIn;
wire               [DATAWIDTH-1:0]     AbsImDataIn;
wire                                   ModOutValid;
wire               [DATAWIDTH+1:0]     ModOutInt;
wire               [DATAWIDTH+1:0]     ModOutNorm;
wire               [DATAWIDTH:0]       ModOutSat;
wire                                   InputIsReal;
wire                                   Load;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
assign InputIsReal = (ImDataIn == {DATAWIDTH{1'b0}});
assign Load = DataInEn && ~InputIsReal; // Do not start Cordic if Input is real

// Take absolute value of inputs
assign NegReDataIn = ~ReDataIn + {{(DATAWIDTH-1){1'b0}},1'b1};
assign AbsReDataIn = ReDataIn[DATAWIDTH-1] ? NegReDataIn : ReDataIn;
assign NegImDataIn = ~ImDataIn + {{(DATAWIDTH-1){1'b0}},1'b1};
assign AbsImDataIn = ImDataIn[DATAWIDTH-1] ? NegImDataIn : ImDataIn;


//Instantiate CordicMag module
CordicMag # (
   //Parameters
   .DATAWIDTH(DATAWIDTH),
   .ANGLEWIDTH(ANGLEWIDTH),
   .SCALING(SCALING)
) U_CordicMag (
   //Inputs
   .PhyClk(PhyClk),
   .nPhyRst(nPhyRst),
   .XIn(AbsReDataIn),
   .YIn(AbsImDataIn),
   .Load(Load),
   //Outputs
   .ModOut(ModOutInt),
   .ModOutValid(ModOutValid)
  );

// Normalize the outputs
assign ModOutNorm = (DataInEn && InputIsReal) ? NormalizeFct({2'd0,AbsReDataIn}) :
                                                NormalizeFct(ModOutInt);

// Saturate the Normalized Outputs
SatUnsigned #(
   .INPUT_WIDTH(DATAWIDTH+2),
   .OUTPUT_WIDTH(DATAWIDTH+1)
) U_SATCRT0 (
   .InputData(ModOutNorm),
   .SatData(ModOutSat)
  );


//Register the final output
always @ (posedge PhyClk or negedge nPhyRst)
   begin: OpReg_Blk
      if (nPhyRst == 1'b0) begin
         CordicDone <= 1'b0;
         ModOut     <= {(DATAWIDTH+1){1'b0}};
      end else begin
         if ((DataInEn && InputIsReal) || (ModOutValid == 1'b1)) begin
            CordicDone <= 1'b1;
            ModOut     <= ModOutSat;
         end else begin
            CordicDone <= 1'b0;
         end
      end

   end //OpReg_Blk

//Function to generate the Normalized Value of the output
//Multiply by normalization factor 2487/4096

function           [DATAWIDTH+1:0]                NormalizeFct;

//Inputs
input    signed    [DATAWIDTH+1:0]                DataIn;

//Internal Registers
reg      signed    [DATAWIDTH+13:0]               Normal;

   begin
      //Multiply by 2487
      //2487 = 2048 + 256 + 128 + 32 + 16 + 4 + 2 + 1
      Normal =   (DataIn <<< CONST_11_DATAWIDTH_P2) + (DataIn <<< CONST_8_DATAWIDTH_P2) + (DataIn <<< CONST_7_DATAWIDTH_P2)
               + (DataIn <<< CONST_5_DATAWIDTH_P2)  + (DataIn <<< CONST_4_DATAWIDTH_P2) + (DataIn <<< CONST_2_DATAWIDTH_P2)
               + (DataIn <<< CONST_1_DATAWIDTH_P2)  + (DataIn);

      // Divide by 4096
      NormalizeFct = $signed(Normal[DATAWIDTH+13:12]);

   end

endfunction  //NormalizeFct;

endmodule //CordicMagTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
