//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 16168 $
// $Date: 2014-09-25 14:30:37 +0200 (Thu, 25 Sep 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : General Purpose Cordic for Phase Calculation
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/IPs/HW/Modem/Src/MODEMCORE/OFDMCORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/CordicVect.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CordicVect #(parameter DATAWIDTH    = 16, //Input Data Width
                    parameter ANGLEWIDTH   = 21, //Output Angle Width
                    parameter SCALING      =  1 //Scaling for ArcTan Table
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, // Active LOW Reset
            input    wire                                  PhyClk,  // PHY Clock
            //Control Signals
            input    wire                                  Load, // Load input values.
            //Data
            input    wire      [DATAWIDTH-1:0]             XIn, //Real part
            input    wire      [DATAWIDTH-1:0]             YIn, //Imaginary part

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signal
            output     wire                                CordicRdy, //Ready
            //Angle
            output     reg     [ANGLEWIDTH-1:0]            AngleOut //Angle out
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters
//////////////////////////////////////////////////////////////////////////////
localparam [DATAWIDTH:0]    CONST_ZERO     = {(DATAWIDTH+1){1'b0}};
localparam [ANGLEWIDTH-2:0] CONST_PI2_SCALED  = {(ANGLEWIDTH-1){1'b1}};
localparam [ANGLEWIDTH-2:0] CONST_MPI2_SCALED = {(ANGLEWIDTH-1){1'b0}};
localparam                  CONST_NDECP          = 3;
//Define the min of DATAWIDTH/ANGLEWIDTH, which will be the number of
//iterations. Indeed, no need to iterate more, when ANGLEWIDTH < DATAWIDTH,
//as only zeros are added during the last iterations.
localparam [4:0] MINSIZE = (DATAWIDTH < ANGLEWIDTH) ? DATAWIDTH : ANGLEWIDTH; 


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [ANGLEWIDTH-1:0]    ArcTan; 
wire               [ANGLEWIDTH:0]      ArcTanExt;
wire               [ANGLEWIDTH:0]      Zn1;
wire               [ANGLEWIDTH:0]      Zn1Sat;
wire               [ANGLEWIDTH:0]      ZnStep;
wire               [DATAWIDTH:0]       XnStep;
wire               [DATAWIDTH:0]       YnStep;
wire               [DATAWIDTH:0]       XnShiftArray[DATAWIDTH:0];
wire               [DATAWIDTH:0]       YnShiftArray[DATAWIDTH:0];
wire               [DATAWIDTH:0]       TempSynY;
wire               [DATAWIDTH:0]       Xn1;
wire               [DATAWIDTH:0]       Yn1;
wire                                   Sn;
wire               [DATAWIDTH:0]       XnShift;
wire               [DATAWIDTH:0]       YnShift;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [DATAWIDTH:0]       Xn;
reg                [DATAWIDTH:0]       Yn;
reg                [ANGLEWIDTH:0]      Zn;
reg                [4:0]               Index;
reg                                    AngleRdy;
reg                                    AngleRdyD;
reg                                    InputEqZero;

//Genvars
genvar g;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//When the cordic is scaled, there is a risk of saturation, with low value
//at pi/2, (eg :  [0,13] can give 99 deg). If it occurs, the result is
//saturated at 90 deg(-1) , which is more accurate.
//more than  90 deg => 90 deg
//less than -90 deg => -90 deg

generate
   if (SCALING == 1)
     begin : gen_sc_eq_1
      assign Zn1Sat = (!(Zn1[ANGLEWIDTH]) && (Zn1[ANGLEWIDTH - 1])) ?
                        {2'b00, CONST_PI2_SCALED} :
                        ((Zn1[ANGLEWIDTH]) && !(Zn1[ANGLEWIDTH - 1])) ?
                        {2'b11, CONST_MPI2_SCALED} : Zn1;
     end
   else
     begin : gen_sc_neq_1
      assign Zn1Sat = Zn1;
     end
endgenerate

//Assign output port.
//angle(0, 0) = 0 for compatibility with MATLAB
always @(*)
   begin: Output_Blk
      if (InputEqZero)
         AngleOut = {(ANGLEWIDTH){1'b0}};
      else
         AngleOut = Zn1Sat[ANGLEWIDTH-1:0];

   end //Output_Blk

//This block increases the Index from 0 to DATAWIDTH-1.
//The value in the index is the iteration number ('n').
always @(posedge PhyClk or negedge nPhyRst)
   begin: Control_Blk
      if (nPhyRst == 1'b0) begin
         Index     <= MINSIZE;
         AngleRdy  <= 1'b0;
         AngleRdyD <= 1'b0;
      end
      else begin
         AngleRdyD <= AngleRdy ;
         if (Load == 1'b1) begin //Reset Index
            AngleRdy <= 1'b0;
            Index    <=  5'b0;
         end
         else begin
            if (Index == MINSIZE - 5'b1)
               AngleRdy <= 1'b1;

            if (Index <= MINSIZE - 5'b1)
               Index <= Index + 5'b1;
         end
      end
   end //Control_Blk

//CordicRdy
assign CordicRdy = AngleRdy & ~AngleRdyD; 

//Remaining angle sign, to determine direction of next microrotation.
assign Sn = Yn[DATAWIDTH];

//Extend sign bit of ArcTan look-up table value.
assign ArcTanExt = {{(CONST_NDECP+1){1'b0}}, ArcTan[ANGLEWIDTH-1:CONST_NDECP]};

//Calculate next angle microrotation:
//Zn1 = Zn - Sn*ArcTan(2^-n)  with  Sn = -1  if Zn <  0; 1  if Zn >= 0
assign ZnStep = (Sn == 1'b0) ? ArcTanExt : ~(ArcTanExt) + {{{ANGLEWIDTH}{1'b0}},1'b1};
assign Zn1    = Zn + ZnStep;

//Load input value or store microrotation result.
always @(posedge PhyClk or negedge nPhyRst)
   begin: AccBlk
      if (nPhyRst == 1'b0) begin
         Xn <= {(DATAWIDTH + 1){1'b0}};
         Yn <= {(DATAWIDTH + 1){1'b0}};
         Zn <= {(ANGLEWIDTH + 1){1'b0}};
         InputEqZero <= 1'b0; 
      end
      else begin
         if (Load == 1'b1) begin
            // Load angle init value.
            Xn <= {1'b0, XIn};
            Yn <= {YIn[DATAWIDTH-1], YIn};
            Zn <= {(ANGLEWIDTH+1){1'b0}};
            //Check for inputs equal to zero
            if ((XIn == CONST_ZERO[DATAWIDTH - 1:0]) &&
                (YIn == CONST_ZERO[DATAWIDTH - 1:0]))
               InputEqZero <= 1'b1;
            else
               InputEqZero <= 1'b0;
         end
         else begin
            // Store microrotation result.
            Xn <= Xn1;
            Yn <= Yn1;
            Zn <= Zn1; // Angle microrotation.
         end
      end
   end //AccBlk

//Microrotation to calculate Yn1 uses the value XnShift = Xn*2^-n.
assign XnShift = XnShiftArray[Index];

//Microrotation to calculate Xn1 uses the value YnShift = Yn*2^-n.
assign YnShift = YnShiftArray[Index];

//Calculate next X value:
//Xn1 = Xn - Sn*(2^-n)*Yn.
assign XnStep = (Sn == 1'b0) ? YnShift : ~(YnShift) + {{{DATAWIDTH}{1'b0}},1'b1};
assign Xn1    = Xn + XnStep;

//Calculate next Y value:
//Yn1 = Yn + Sn*(2^-n)*Xn.
assign YnStep = (Sn == 1'b1) ? XnShift : ~(XnShift) + {{{DATAWIDTH}{1'b0}},1'b1};
assign Yn1 = Yn + YnStep;

assign XnShiftArray[0] = Xn;
assign YnShiftArray[0] = Yn;

//Use TempSynY signal for Synopsys work-around.
assign TempSynY = {(DATAWIDTH+1){Yn[DATAWIDTH]}};

generate

   for(g = 1; g < DATAWIDTH+1 ; g = g + 1) begin : SAR_Blk
      assign XnShiftArray[g][DATAWIDTH:DATAWIDTH-g+1]
                          = CONST_ZERO[DATAWIDTH:DATAWIDTH-g+1];
      assign XnShiftArray[g][DATAWIDTH-g:0] = Xn[DATAWIDTH:g];

      assign YnShiftArray[g][DATAWIDTH:DATAWIDTH-g+1] 
                         = TempSynY[DATAWIDTH:DATAWIDTH-g+1];
      assign YnShiftArray[g][DATAWIDTH-g:0] = Yn[DATAWIDTH:g];
   end //SAR_Blk

endgenerate


//Instantiate ArcTanLUT
ArcTanLUT # (
             //Parameters
             .ANGLEWIDTH(ANGLEWIDTH),
             .SCALING(SCALING)
            )U_ATLUT0 (
                       //Input
                       .Index(Index),
                       //Output
                       .ArcTan(ArcTan)
                      );

endmodule //CordicVect

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
