//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 17048 $
// $Date: 2014-11-26 13:44:24 +0100 (Wed, 26 Nov 2014) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Cordic Vector Top Level
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/IPs/HW/Modem/Src/MODEMCORE/OFDMCORE/OFDMRXCORE/OFDMRXTD/TDFO/verilog/rtl/CordicVectTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CordicVectTop #(parameter DATAWIDTH    = 16, //Input Data Width
                       parameter ANGLEWIDTH   = 21, //Input Angle Width
                       parameter SCALING      =  1 //Scaling for ArcTan Table
                      )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                   nPhyRst, // Active LOW Reset
            input   wire                                   PhyClk,  // PHY Clock
            //Control Signals
            input   wire                                   DataInEn, // Qualifies input values.
            //Data
            input   wire  signed  [DATAWIDTH-1:0]          ReDataIn, //Real part
            input   wire  signed  [DATAWIDTH-1:0]          ImDataIn, //Imaginary part

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Control Signal
            output   reg                                   CordicDone, //Angle Computation Done
            //Angle
            output   reg  signed   [ANGLEWIDTH-1:0]        AngleOut   //Angle out
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed  [ANGLEWIDTH-1:0]    CONST_PI      = 4*(2**(ANGLEWIDTH-3));
localparam signed                      CONST_ZERO    = 'd0;
localparam signed  [DATAWIDTH-1:0]     CONST_1       = 'b1;
localparam                             CONST_SCALING = 32'd1;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [ANGLEWIDTH-1:0]    AngleOutInt;
wire                                   CordicRdy;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [DATAWIDTH-1:0]     XIn;
reg                [DATAWIDTH-1:0]     YIn;
reg                [1:0]               Flag;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module instantiates the CordicVect module. It also ensures that the
//values going in to the CordicVect are repositioned to be in the -pi/2 to
//pi/2 range. After the angle is calculated by CordicVect, this module again
//translates it back to the -pi to pi range if required.

//Input to CordicVect - Reposition to [-pi/2, pi/2]
always @ (*)
   begin: Input_Blk
      if (DataInEn == 1'b1 && ReDataIn < $signed(CONST_ZERO[DATAWIDTH-1:0])) begin
         XIn = -ReDataIn;
         YIn = -ImDataIn;
      end
      else begin
         XIn = ReDataIn;
         YIn = ImDataIn;
      end
   end //Input_Blk

//Flag to signify quadrant repositioning
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Flag_Blk
      if (nPhyRst == 1'b0)
         Flag <= 2'b0;
      else if (CordicRdy == 1'b1)
         Flag <= 2'b0;
      else if ((DataInEn == 1'b1) && ((ReDataIn < $signed(CONST_ZERO[DATAWIDTH-1:0])) && (ImDataIn >= $signed(CONST_ZERO[DATAWIDTH-1:0]))))
         Flag <= 2'b01;
      else if ((DataInEn == 1'b1) && ((ReDataIn < $signed(CONST_ZERO[DATAWIDTH-1:0])) && (ImDataIn < $signed(CONST_ZERO[DATAWIDTH-1:0]))))
         Flag <= 2'b10;

   end //Flag_Blk

//Output realignment
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Output_Blk
      if (nPhyRst == 1'b0)
         AngleOut <= $signed(CONST_ZERO[ANGLEWIDTH-1:0]);
      else if (CordicRdy == 1'b1) begin
        if (Flag == 2'b01 && (SCALING == CONST_SCALING))
            AngleOut <= AngleOutInt + CONST_PI;
        else if (Flag == 2'b10 && (SCALING == CONST_SCALING))
            AngleOut <= AngleOutInt - CONST_PI;
        else
            AngleOut <= AngleOutInt;
      end

   end //Output_Blk

//Delay CordicRdy
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CordicDone_Blk
      if (nPhyRst == 1'b0)
         CordicDone <= 1'b0;
      else
         CordicDone <= CordicRdy;

   end //CordicDone_Blk

//Instantiate CordicVect module
CordicVect #(
             //Parameters
             .DATAWIDTH(DATAWIDTH),
             .ANGLEWIDTH(ANGLEWIDTH),
             .SCALING(SCALING)
             )U_CV(
                   //Inputs
                   .PhyClk(PhyClk),
                   .nPhyRst(nPhyRst),
                   .XIn(XIn),
                   .YIn(YIn),
                   .Load(DataInEn),

                   //Outputs
                   .AngleOut(AngleOutInt),
                   .CordicRdy(CordicRdy)
                  );

endmodule //CordicVectTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
