////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39422 $
// $Date: 2019-07-12 12:05:19 +0200 (Fri, 12 Jul 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Pipelined divider for the SRT divider
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON/DSP/verilog/rtl/DivPipe.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
module DivPipe #(parameter NBITINPUTSRT    = 9,  // Number of bits of SRT inputs
                 parameter NBSTEPSRT       = 10, // Number of SRT iterations
                 // The following parameter should be equal to 2*NBITINPUTSRT+1 to cover all
                 // SRT input range! Can be reduced for specific applications.
                 parameter NBITPREMSRT     = 13, // Number of bits of the reminder > NBITINPUTSRT
                 parameter NBITQUOTIENTSRT = 12  // Number of bits of the SRT quotient > NBITINPUTSRT
                )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            // Clock and reset
            input  wire                       nPhyRst, // Active low reset
            input  wire                       PhyClk,  // PHY Clock

            //Data & Control
            input  wire    [NBITINPUTSRT-1:0] Dividend,
            input  wire    [NBITINPUTSRT-1:0] Divisor,
            input  wire                       DataEnIn,
            input   wire                      Clear,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output wire                       DataEnOut,
            output wire [NBITQUOTIENTSRT-1:0] Quotient
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
// Constant for 2^(Nq-1) = 2^(nbit_quotient_g-2) = 2^(nb_step_srt_g-2)
// Max and min values for the reminder

localparam signed [NBITPREMSRT-1:0] MAX = {{NBITPREMSRT-NBSTEPSRT+1{1'b0}}, 1'b1, {NBSTEPSRT-2{1'b0}}};
localparam signed [NBITPREMSRT-1:0] MIN = {{NBITPREMSRT-NBSTEPSRT+1{1'b1}}, 1'b1, {NBSTEPSRT-2{1'b0}}};

//////////////////////////////////////////////////////////////////////////////
// Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg signed [NBITPREMSRT-1:0]     PRemind         [NBSTEPSRT:0];
reg        [NBSTEPSRT-1:0]       PartialQuotient [NBSTEPSRT:0];
reg        [NBITQUOTIENTSRT-1:0] IntQuotient     [NBSTEPSRT:0];
reg        [NBITINPUTSRT-1:0]    IntDivisor      [NBSTEPSRT:0];
reg        [NBSTEPSRT:0]         DataEn;

integer                          Count;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// --------------------------------------------
// -- SRT algorithm
// --------------------------------------------
always @ (posedge PhyClk or negedge nPhyRst)
begin 
  if (!nPhyRst)
  begin
     for (Count=0; Count <= NBSTEPSRT; Count = Count+1)
     begin
        PRemind[Count]         <= $signed({{NBITPREMSRT}{1'b0}});
        IntQuotient[Count]     <= {{NBITQUOTIENTSRT}{1'b0}};
        IntDivisor[Count]      <= {{NBITINPUTSRT}{1'b0}};
        PartialQuotient[Count] <= {{NBSTEPSRT}{1'b0}};
        DataEn[Count]          <= 1'b0;
     end 
  end // if (!nPhyRst)
  else if (Clear)
  begin
     for (Count=0; Count <= NBSTEPSRT; Count = Count+1)
     begin
        DataEn[Count]          <= 1'b0;
     end 
  end // else if (Clear)
  else
  begin
     DataEn[0] <= DataEnIn;
     if (DataEnIn)
     begin
        // Initial step
        // Reminder set to dividend
        PRemind[0]         <= $signed({{NBITPREMSRT-NBITINPUTSRT{1'b0}}, Dividend});
        // PartialQuotient set to 2^Nq = 2^(nb_step_srt_g-1)
        PartialQuotient[0] <= {1'b1,{NBSTEPSRT-1{1'b0}}};
        // Quotient set to zero
        IntQuotient[0]     <= {{NBITQUOTIENTSRT}{1'b0}};
        // Divisor to follow the flip-flops.
        IntDivisor[0]      <= Divisor;
     end // if (DataEnIn)
     for (Count=1; Count <= NBSTEPSRT ; Count = Count+1)
     begin
        DataEn[Count]          <= DataEn[Count-1];
        PartialQuotient[Count] <= PartialQuotient[Count-1] >> 1;
        IntDivisor[Count]      <= IntDivisor[Count-1];

        // Compute next quotient digit:
        // Compare PRemind to 2^(Nq-1) = 2^(nb_step_srt_g-2)
        if (DataEn[Count-1]==1'b1)
        begin
          if ($signed(PRemind[Count-1]) >= $signed(MAX))
          begin
             PRemind[Count] <= $signed((PRemind[Count-1] -
                                        {{NBITPREMSRT-NBITINPUTSRT{1'b0}}, IntDivisor[Count-1]}) << 1);
             // Compute quotient.
             IntQuotient[Count] <= IntQuotient[Count-1] +
                                   {{NBITQUOTIENTSRT-NBSTEPSRT{1'b0}}, PartialQuotient[Count-1]};
          end //if (PRemind >= MAX)
          else if ($signed(PRemind[Count-1]) < $signed(MIN))
          begin
             PRemind[Count] <= $signed((PRemind[Count-1] +
                                       {{NBITPREMSRT-NBITINPUTSRT{1'b0}}, IntDivisor[Count-1]}) << 1);
             // Compute quotient.
             IntQuotient [Count]<= IntQuotient[Count-1] -
                                   {{NBITQUOTIENTSRT-NBSTEPSRT{1'b0}}, PartialQuotient[Count-1]};
          end // if (PRemind < MIN)
          else
          begin
             // Compute next partial remainder
             PRemind[Count]      <= PRemind[Count-1] << 1;
             IntQuotient [Count] <= IntQuotient [Count-1];
          end // else: !if(PRemind[Count-1] < MIN)
        end // if (DataEn[Count-1]==1)
     end // for (Count=1; Count <= NBSTEPSRT-1; Count = Count+1)
  end // else: !if(!nPhyRst)
end // always @ (posedge PhyClk or negedge nPhyRst)

assign Quotient  = IntQuotient[NBSTEPSRT];
assign DataEnOut = DataEn[NBSTEPSRT];

endmodule // DivPipe

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
