////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39422 $
// $Date: 2019-07-12 12:05:19 +0200 (Fri, 12 Jul 2019) $
// -------------------------------------------------------------------------
// Dependencies     :
// Description      : Divider
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON/DSP/verilog/rtl/SrtDivider.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

`default_nettype none

module SrtDivider #(parameter NBITINPUT               = 18, // = Na+1 = Nb+1 (min 3)
                    parameter NBITQUOTIENT            = 20, // = Nq+1 (min 2)
                    parameter NINTBITQUOTIENT         = 1,  // = NqINT. 0 to min(NBITQUOTIENT-1, NBITINPUT-1)
                    parameter NBITQUOTIENTSRT         = 22, // = Nqn. Warning: Nq < Nqn < 2Nq
                    parameter NBITPREMSRT             = 22, // = Nrn. Warning: Nq < Nqn < 2Nq
                    parameter SIZEBITSHIFTSRTQUOTIENT = 5   // size in bits of -NBITINPUT-NINTBITQUOTIENT+2
                    )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            // Clock and reset
            input   wire                      nPhyRst, // Active low reset
            input   wire                      PhyClk, // PHY Clock

            //Data & Control
            input   wire  [NBITINPUT-1:0]     Dividend,
            input   wire  [NBITINPUT-1:0]     Divisor,
            input   wire                      DataEnIn,
            input   wire                      Clear,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   reg   [NBITQUOTIENT-1:0] Quotient,
            output  wire                      DataEnOut
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
  localparam [NBITINPUT-1:0] INPUTMIN    = {1'b1, {NBITINPUT-1{1'b0}}};
  localparam [NBITINPUT-1:0] INPUTMINSAT = {1'b1, {NBITINPUT-2{1'b0}}, 1'b1};

    // Constants for unsaturated quotient constants...
  localparam [NBITQUOTIENT-2:0] ZEROSQUOT = 0;
  localparam [NBITQUOTIENT-2:0] ONESQUOT  = {NBITQUOTIENT-1{1'b1}};

    // Limitations on generics ensure ZEROSMSB_CT'high > 0
  localparam [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-NBITQUOTIENT:0] ZEROSMSB = 0;
  localparam [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-NBITQUOTIENT:0] ONESMSB  = {NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-NBITQUOTIENT+1{1'b1}};

    // Constants to compare the unsaturated quotient
  localparam [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] UNSATQUOTMAX = {ZEROSMSB, ONESQUOT};
  localparam [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] UNSATQUOTMIN = {ONESMSB, ZEROSQUOT};
    // Constant to saturate the quotient
  localparam [NBITQUOTIENT-1:0] QUOTMAX = {1'b0, ONESQUOT};
  localparam [NBITQUOTIENT-1:0] QUOTMIN = {1'b1, ZEROSQUOT};

  // pre-processing variables.
  localparam [SIZEBITSHIFTSRTQUOTIENT:0] NINTBITQUOTIENT_PARAM      = NINTBITQUOTIENT;
  localparam [SIZEBITSHIFTSRTQUOTIENT:0] SHIFT_STR_QUOTIENT_PARAM_0 = {{SIZEBITSHIFTSRTQUOTIENT+1}{1'b0}};
  localparam [SIZEBITSHIFTSRTQUOTIENT:0] SHIFT_STR_QUOTIENT_PARAM_1 = {{{SIZEBITSHIFTSRTQUOTIENT}{1'b0}},1'b1}; //'b1
  localparam [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] QUOTIENT_BUF_PARAM_1 =
                                                           {{{(NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1)}{1'b0}},1'b1}; //'b1

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
   //Data enabling variable
  reg DataEnff0;
  reg DataEnff1;
  reg DataEnSrtff0;
  reg DataEnSrtff1;

  // Post-processing variables.
  reg signed [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] QuotientBuf;

  reg [NBITINPUT-2 : 0] DividendAbsC;
  reg [NBITINPUT-2 : 0] DivisorAbsC;

  // pre-processing variables.
  reg                                    SignBit;
  reg signed [SIZEBITSHIFTSRTQUOTIENT:0] ShiftSrtQuotient;

  // Absolute and shifted values of clipped inputs, registered
  // These two vectors can be oversized, unused bits are removed by the
  // synthesis tool.
  reg [NBITINPUT-2:0] DividendAbsShift;
  reg [NBITINPUT-2:0] DivisorAbsShift;

  reg                             OneDetect;
  reg [SIZEBITSHIFTSRTQUOTIENT:0] ShiftDividendIndex;
  reg [SIZEBITSHIFTSRTQUOTIENT:0] ShiftDivisorIndex;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
  // Extend SRT result before shifting it back. It can be shifted of x bits,
  // x in [-NBITINPUT+1-NINTBITQUOTIENT+1, NBITINPUT-1-NINTBITQUOTIENT+1]
  // so the SRT result width should be extended with
  // NBITINPUT-NINTBITQUOTIENT bits. By limitation of the HDL this value is positive.
  wire signed [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] SrtQuotientExt;
  wire        [NBITINPUT-1 : 0]                               DividendSatC;
  wire        [NBITINPUT-1 : 0]                               DivisorSatC;
  wire        [NBITINPUT-1 : 0]                               DividendCc2;
  wire        [NBITINPUT-1 : 0]                               DivisorCc2;

  // SRT interface
  wire        [NBITQUOTIENT-1:0]    SrtDividend;
  wire        [NBITQUOTIENT-1:0]    SrtDivisor;
  wire signed [NBITQUOTIENTSRT-1:0] SrtQuotient;

  // Data enabling variable
  wire DataEnSrt;

  // Variable used in for loop
  integer Count;

  // Post-processing variables.
  reg signed [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] QuotientBufRound;
  reg signed [NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT-1:0] QuotientBufSign;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

  //--------------------------------------------
  //-- SRT divider
  //--------------------------------------------
   Divider # (.NBITINPUTSRT(NBITQUOTIENT),
              .NBSTEPSRT(NBITQUOTIENT),
              .NBITPREMSRT(NBITPREMSRT),
              .NBITQUOTIENTSRT(NBITQUOTIENTSRT))
   Divider_1 (.PhyClk(PhyClk),
              .nPhyRst(nPhyRst),
              .Clear(Clear),
              .DataEnIn(DataEnff1),
              .DataEnOut(DataEnSrt),
              .Dividend(SrtDividend),
              .Divisor(SrtDivisor),
              .Quotient(SrtQuotient)
              );

  //--------------------------------------------
  //-- Saturate the inputs so that a vector on n bits takes the values in
  //-- [-(2^(n-1)-1), 2^(n-1)-1]
  //-- i.e. if value 100.....00 is found, replace by 100.....01
  //--------------------------------------------
  //-- Saturate Dividend
  assign DividendSatC = (Dividend == INPUTMIN) ? INPUTMINSAT : Dividend;

  //-- Saturate Divisor
  assign DivisorSatC = (Divisor == INPUTMIN) ? INPUTMINSAT : Divisor;

  //--------------------------------------------
  //-- Take absolute value of the saturated data
  //--------------------------------------------
  //-- Absolute value of saturated Dividend
  assign DividendCc2 = ~(DividendSatC) + {{(NBITINPUT-1){1'b0}},1'b1};

  //-- Absolute value of saturated Divisor
  assign DivisorCc2 = ~(DivisorSatC) + {{(NBITINPUT-1){1'b0}},1'b1};

  always @ (posedge PhyClk or negedge nPhyRst)
  begin
    if(!nPhyRst)
    begin
      DivisorAbsC  <= {(NBITINPUT-1){1'b0}};
      DividendAbsC <= {(NBITINPUT-1){1'b0}};
    end
    else
    begin
      //-- Absolute value of saturated Divisor
      DivisorAbsC <= (DivisorSatC[NBITINPUT-1] == 1'b1) ? DivisorCc2[NBITINPUT-2:0] : DivisorSatC[NBITINPUT-2:0];
      //-- Absolute value of saturated Dividend
      DividendAbsC <= (DividendSatC[NBITINPUT-1] == 1'b1) ? DividendCc2[NBITINPUT-2:0] : DividendSatC[NBITINPUT-2:0];
    end
  end

  // SignBit set to 1 if dividend and divisor don't have the same sign.
  always @ (posedge PhyClk or negedge nPhyRst)
  begin
    // pre-processing variables.
    // sign of the operation (=1 if sign is different)
    if(!nPhyRst)
    begin
      SignBit <= 1'b0;
    end
    else if (DataEnIn)
    begin
       if (Dividend[NBITINPUT-1] != Divisor[NBITINPUT-1])
         SignBit <= 1'b1;
       else
         SignBit <= 1'b0;
    end
  end

  //--------------------------------------------
  //-- Preprocessing. This process bounds the divisor and the dividend
  //-- between 0.5 and 1.
  //--------------------------------------------

   // Prepare the shift value of the dividend and the divisor
   // for the post-processing.
   always @(*)
   begin
     // compute shift of dividend and Divisor.
     // compute the number of shift operations to bound the
     // dividend input to the SRT to 0.5 < Dividend < 1
     ShiftDividendIndex = SHIFT_STR_QUOTIENT_PARAM_0;
     OneDetect          = 1'b0;
     for (Count = NBITINPUT-2; Count >= 0; Count = Count - 1)
     begin
        if (OneDetect == 1'b0)
        begin
          if (DividendAbsC[Count] == 1'b1)
            OneDetect = 1'b1;
          else
            ShiftDividendIndex = ShiftDividendIndex + SHIFT_STR_QUOTIENT_PARAM_1;
        end //if (OneDetect==0)
     end //for (Count=0;...

     // Compute the number of shift operations to bound the
     // divisor input to the SRT to 0.5 < Divisor < 1
     ShiftDivisorIndex = SHIFT_STR_QUOTIENT_PARAM_0;
     OneDetect         = 1'b0;
     for (Count=NBITINPUT-2; Count>=0; Count = Count - 1)
     begin
        if (OneDetect == 1'b0)
        begin
          if (DivisorAbsC[Count])
            OneDetect = 1'b1;
          else
            ShiftDivisorIndex = ShiftDivisorIndex + SHIFT_STR_QUOTIENT_PARAM_1;
        end //if (OneDetect == 0)
     end //for ...
   end // always @(*)

   always @ (posedge PhyClk or negedge nPhyRst)
   begin
     if (!nPhyRst)
     begin
        ShiftSrtQuotient <= SHIFT_STR_QUOTIENT_PARAM_0;
        DataEnff0        <= 1'b0;
        DataEnff1        <= 1'b0;
        
        DividendAbsShift <= {(NBITINPUT-1){1'b0}};
        DivisorAbsShift  <= {(NBITINPUT-1){1'b0}};
     end
     else if (Clear)
     begin
        DataEnff0        <= 1'b0;
        DataEnff1        <= 1'b0;
     end
     else
     begin
        DataEnff0 <= DataEnIn;
        DataEnff1 <= DataEnff0;

        if (DataEnff0)
        begin
          // Bound dividend to 0.5 < dividend < 1
          DividendAbsShift <= DividendAbsC << ShiftDividendIndex;
          // Bound Divisor to 0.5 < Divisor < 1
          DivisorAbsShift  <= DivisorAbsC << ShiftDivisorIndex;

          // Quotient shift amount to get the correct weighted output.
          // Can be positive or negative
          ShiftSrtQuotient <= ShiftDivisorIndex  - 
                              ShiftDividendIndex -
                              $signed({{32-SIZEBITSHIFTSRTQUOTIENT{1'b0}},NINTBITQUOTIENT_PARAM}) +
                              $signed({{32-SIZEBITSHIFTSRTQUOTIENT{1'b0}},SHIFT_STR_QUOTIENT_PARAM_1});
        end // if (DataEnIn)
     end // else: !if(!nPhyRst)
   end // always @ (posedge PhyClk or negedge nPhyRst)

   // Keep NBITQUOTIENT bits of shifted values. In case not all bits of
   // dividend_shift and Divisor_shift registers are used, the not-used register
   // bits will be removed by the synthesis tool.

   // SRT dividend is on less bits than, or same number of bits as shifted
   // dividend and Divisor: keep only MSBs.
   generate
      if (NBITQUOTIENT <= NBITINPUT-1)
        begin : MSB_CUT
           assign SrtDividend = DividendAbsShift[(NBITINPUT-2):(NBITINPUT-NBITQUOTIENT-1)];
           assign SrtDivisor  = DivisorAbsShift[(NBITINPUT-2):(NBITINPUT-NBITQUOTIENT-1)];
        end

      else if (NBITQUOTIENT > NBITINPUT-1)
        begin : NO_MSB_CUT
           assign SrtDividend = {DividendAbsShift, {NBITQUOTIENT-NBITINPUT+1{1'b0}}};
           assign SrtDivisor  = {DivisorAbsShift, {NBITQUOTIENT-NBITINPUT+1{1'b0}}};
        end
   endgenerate

   //--------------------------------------------
   //-- Extend width of SRT output
   //--------------------------------------------
   // Signed extended.
   assign SrtQuotientExt = $signed({{NBITINPUT-NINTBITQUOTIENT{SrtQuotient[NBITQUOTIENTSRT-1]}},SrtQuotient});

   //--------------------------------------------
   //-- Postprocessing. Bound the quotient in the right
   //-- sign and range.
   //--------------------------------------------
   // Shift the quotient.
   always @ (posedge PhyClk or negedge nPhyRst)
   begin
     // Shift back of quotient after SRT algorithm
      if (!nPhyRst)
      begin
         QuotientBuf  <= $signed({{(NBITINPUT-NINTBITQUOTIENT+NBITQUOTIENTSRT)}{1'b0}});
         DataEnSrtff0 <= 1'b0;
         DataEnSrtff1 <= 1'b0;
      end
      else if (Clear)
      begin
         DataEnSrtff0 <= 1'b0;
         DataEnSrtff1 <= 1'b0;
      end
      else
      begin
        DataEnSrtff0 <= DataEnSrt;
        DataEnSrtff1 <= DataEnSrtff0;
        if (DataEnSrt)
        begin
          // Shift Quotient
          if ($signed(ShiftSrtQuotient) >= $signed(SHIFT_STR_QUOTIENT_PARAM_0))
            // Left shift of shift_srt_quotient (positive value).
            QuotientBuf <= SrtQuotientExt << ShiftSrtQuotient;
          else
            QuotientBuf <= SrtQuotientExt >>> -ShiftSrtQuotient;
        end // if (DataEnSrtOut)
      end // else: !if(!nPhyRst)
    end // always @ (posedge PhyClk or negedge nPhyRst)

   // Add the sign and round
   always @(*)
   begin
      // Round and correct sign bit (use different variables for easier debug)
      QuotientBufRound = QuotientBuf + QUOTIENT_BUF_PARAM_1;
      QuotientBufSign  = QuotientBufRound >>> 1;

      if (SignBit)
         QuotientBufSign = ~(QuotientBufSign) + QUOTIENT_BUF_PARAM_1;
   end // always @(*)

   // Saturate the quotient.
   always @ (posedge PhyClk or negedge nPhyRst)
   begin
     if (!nPhyRst)
       Quotient <= {NBITQUOTIENT{1'b0}};
     else if (DataEnSrtff0)
     begin
       // Saturate quotient
       if ($signed(QuotientBufSign) > $signed(UNSATQUOTMAX))
         Quotient <= QUOTMAX;
       else if ($signed(QuotientBufSign) < $signed(UNSATQUOTMIN))
         Quotient <= QUOTMIN;
       else
         Quotient <= QuotientBufSign[NBITQUOTIENT-1:0];
     end // else if (DataEnSrtff0)
   end // always @ (posedge PhyClk or negedge nPhyRst)

   assign DataEnOut = DataEnSrtff1;

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
