////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 16114 $
// $Date: 2014-09-23 17:14:09 +0200 (Tue, 23 Sep 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : AGC State Machine Core
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/AGCFSMCore.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module AGCFSMCore #(parameter AGCRAMWIDTH = 32
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,         // Active Low Reset
            input    wire                                  AGCClk,          // AGC Clock
            input    wire                                  AGCFSMSftRst,    // Soft Reset

            //Controls
            input    wire                                  EventValid,      // Event Valid Indication
            input    wire     [8:0]                        EventTargetAddr, // Address from Event
            input    wire                                  TransValid,      // Transition Valid
            input    wire     [AGCRAMWIDTH-1:0]            AGCMemRdData,    // Op Code from AGC RAM

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Address
            output   wire     [8:0]                       AGCMemAddr,      // Address for AGC RAM

            //Controls
            output   wire     [21:0]                      TransCond,       // Transition Condition
            output   wire     [7:0]                       AGCCommand,      // Commands to Modules
            output   wire     [23:0]                      AGCParam,        // Command Parameter 1
            output   reg                                  AGCCmdValid,     // AGC Command Valid 
            output   reg      [10:0]                      TimeOutValue,    // Time Out Count Value
            output   reg                                  TimeOutEn,       // Enable Time Out Counter
            output   wire     [14:0]                      DSPEn,           // Enable for DSP Blocks
            output   wire                                 AGCClear,        // Clear Status register
            output   wire                                 AGCMaskEvent,    // Mask event
            output   reg                                  AGCFSMExtEn,     // External enable
            
            //Debug
            output   wire     [8:0]                       DbgAgcFsm        // FSM address index
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam FSM_SIZE = 3;

localparam STATE_ENTRY     = 3'd0,
           READ_STATE_INFO = 3'd1,
           COMMAND         = 3'd2,
           TRANSITION_LOOP = 3'd3,
           SLEEP           = 3'd4;

localparam COMMAND_EN = 28,
           TIMEOUT_EN = 27,
           EXT_EN     = 26;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Wires Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [FSM_SIZE-1:0]      AGCFSMCS;
reg                [FSM_SIZE-1:0]      AGCFSMNS;
reg                [8:0]               NextAddr;
reg                [8:0]               TransTargetAddr;
reg                [8:0]               TransRetAddr;
reg                [2:0]               TransCnt;
reg                [2:0]               NumTrans;
reg                [7:0]               AGCCommandInt;
reg                [23:0]              AGCParamInt;
reg                [14:0]              DSPEnInt;
reg                [8:0]               AGCMemAddrInt;
reg                                    EventValidD;
reg                [21:0]              TransCondInt;
reg                                    CurrentEventValid;
reg                                    CurrentEventValidEnd;

//String definition to display FSM states in simulation environment
`ifdef SIMU_ON
reg   [50*8:0]         AGCFSMCurrStateStr;
reg   [50*8:0]         AGCFSMNextStateStr;
`endif //SIMU_ON

wire               [2:0]               NumTransInt;
wire                                   AGCClearInt;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//The AGC FSM Core implements the FSM transitions and translates the AGCMemRdData
//into Commands and Parameters. It enables the TimeOut Counter when required.
//This block also performs the address generation for the next AGCMemRdData to be
//read. The transition and event decoding is done outside the block.

//Sequential Block
always @ (posedge AGCClk or negedge nAGCRst)
   begin: FSM_Seq
      if (nAGCRst == 1'b0)
         AGCFSMCS <= STATE_ENTRY;
      else if (AGCFSMSftRst == 1'b1)
         AGCFSMCS <= STATE_ENTRY;
      else if ((EventValid == 1'b1) && (EventValidD == 1'b0))
         AGCFSMCS <= STATE_ENTRY;
      else 
         AGCFSMCS <= AGCFSMNS;
   end //FSM_Seq

//Pulse generation for event valid
always @ (posedge AGCClk or negedge nAGCRst)
   begin: EventValidD_Blk
      if (nAGCRst == 1'b0)
         EventValidD <= 1'b0;
      else
         EventValidD <= EventValid;
   end //EventValidD_Blk

//Combinatorial Block
always @ (*)
   begin: FSM_Combo
      case(AGCFSMCS)

         STATE_ENTRY: begin
            AGCFSMNS = READ_STATE_INFO;
         end //STATE_ENTRY

         READ_STATE_INFO: begin
            if (AGCMemRdData[COMMAND_EN] == 1'b1)
               AGCFSMNS = COMMAND;
            else if ((AGCMemRdData[31:29] == 3'h0))
               AGCFSMNS = SLEEP;
            else
               AGCFSMNS = TRANSITION_LOOP;
         end //READ_STATE_INFO

         COMMAND: begin
            AGCFSMNS = TRANSITION_LOOP;
         end //COMMAND

         TRANSITION_LOOP: begin
            if (TransValid == 1'b1)
               AGCFSMNS = STATE_ENTRY;
            else
               AGCFSMNS = TRANSITION_LOOP;
         end //TRANSITION_LOOP

         SLEEP: begin
            AGCFSMNS = SLEEP;
         end //SLEEP

         // pragma coverage block = off
         default: begin
            AGCFSMNS = STATE_ENTRY;
         end //default
         // pragma coverage block = on

      endcase //AGCFSMCS
   end //FSM_Combo

//Next Address Generation
always @ (*)
   begin: NextAddr_Blk
      if (AGCFSMCS == TRANSITION_LOOP) begin
         if (TransValid == 1'b1) begin
            NextAddr = TransTargetAddr;
         end
         else if ((TransCnt == NumTransInt-3'd1) || (NumTrans == 3'd1)) begin
            NextAddr = TransRetAddr;
         end
         else begin
            NextAddr = AGCMemAddrInt + 9'b1;
         end
      end
      else if (AGCFSMNS == SLEEP && AGCFSMCS != SLEEP)
         NextAddr = AGCMemAddrInt - 9'b1;
      else if ((AGCFSMCS == SLEEP) || 
              ((AGCFSMCS == READ_STATE_INFO) &&
              ((AGCMemRdData[31:29] == 3'h0) ||
               (AGCMemRdData[31:29] == 3'h1 && AGCMemRdData[28] == 1'b0))) ||
              ((AGCFSMCS == COMMAND) && (NumTrans == 3'd1))) begin
         // Sleep or no transition or 1 transition and no command then keep previous address
         NextAddr = AGCMemAddrInt;
      end
      else begin
         NextAddr = AGCMemAddrInt + 9'b1;
      end
   end //NextAddr_Blk

//Address Counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AddressCnt_Blk
      if (nAGCRst == 1'b0)
         AGCMemAddrInt <= 9'b0;
      else if (AGCFSMSftRst == 1'b1)
         AGCMemAddrInt <= 9'b0;
      else if ((EventValid == 1'b1) && (EventValidD == 1'b0))
         AGCMemAddrInt <= EventTargetAddr;
      else
         AGCMemAddrInt <= NextAddr;
   end //AddressCnt_Blk

assign AGCMemAddr = AGCMemAddrInt;

//Time Out Value and Enable
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TimeOut_Blk
      if (nAGCRst == 1'b0) begin
         TimeOutEn     <= 1'b0;
         TimeOutValue  <= 11'b0;
      end
      else if (AGCFSMSftRst == 1'b1) begin
         TimeOutEn     <= 1'b0;
         TimeOutValue  <= 11'b0;
      end
      else if (AGCFSMCS == READ_STATE_INFO && AGCMemRdData[TIMEOUT_EN] == 1'b1) begin
         TimeOutEn     <= 1'b1;
         TimeOutValue  <= AGCMemRdData[25:15];
      end
      else begin
         TimeOutEn     <= 1'b0;
         TimeOutValue  <= 11'b0;
      end
   end //TimeOut_Blk

//DSP Enable
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DSPEnInt_Blk
      if (nAGCRst == 1'b0)
         DSPEnInt <= 15'b0;
      else if (AGCFSMSftRst == 1'b1)
         DSPEnInt <= 15'b0;
      else if (AGCFSMCS == READ_STATE_INFO)
         DSPEnInt <= AGCMemRdData[14:0];
   end //DSPEnInt_Blk

assign DSPEn = DSPEnInt;

//AGC Clear
assign AGCClearInt = (AGCFSMCS == STATE_ENTRY) ? 1'b1: 1'b0;
assign AGCClear    = AGCClearInt;

//Event clear Handling
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Event_Blk
      if (nAGCRst == 1'b0)
         CurrentEventValid <= 1'b0;
      else if (AGCFSMSftRst == 1'b1)
         CurrentEventValid <= 1'b0;
      else if ((EventValid == 1'b1) || (EventValidD == 1'b1))
         CurrentEventValid <= 1'b1;
      else if (CurrentEventValidEnd == 1'b1)
         CurrentEventValid <= 1'b0;
   end //Event_Blk

always @ (posedge AGCClk or negedge nAGCRst)
   begin: EventEnd_Blk
      if (nAGCRst == 1'b0)
         CurrentEventValidEnd <= 1'b0;
      else if (AGCFSMSftRst == 1'b1)
         CurrentEventValidEnd <= 1'b0;
      else if (AGCFSMCS == READ_STATE_INFO)
         CurrentEventValidEnd <= 1'b1;
      else
         CurrentEventValidEnd <= 1'b0;
   end //EventEnd_Blk

assign AGCMaskEvent = CurrentEventValid;

//Transition Handling
//Target Address and Condition for Transition
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Trans_Blk
      if (nAGCRst == 1'b0) begin
         TransTargetAddr <= 9'b0;
         TransCondInt    <= 22'b0;
      end
      else if (AGCFSMCS == TRANSITION_LOOP) begin
         TransTargetAddr <= AGCMemRdData[8:0];
         TransCondInt    <= AGCMemRdData[31:10];
      end
      else begin
         TransTargetAddr <= 9'b0;
         TransCondInt    <= 22'b0;
      end
   end //Trans_Blk

assign TransCond = TransCondInt;

//Number of Transitions to be checked   
always @ (posedge AGCClk or negedge nAGCRst)
   begin: NumTrans_Blk
      if (nAGCRst == 1'b0)
         NumTrans <= 3'b0;
      else if (AGCFSMSftRst || AGCClearInt)
         NumTrans <= 3'b0;
      else if (AGCFSMCS == READ_STATE_INFO)
         NumTrans <= AGCMemRdData[31:29];
   end //NumTrans_Blk

assign NumTransInt = NumTrans - 3'b1;

//Transition Counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TransCnt_Blk
      if (nAGCRst == 1'b0)
         TransCnt <= 3'b0;
      else if (AGCFSMSftRst || AGCClearInt)
         TransCnt <= 3'b0;
      else if (AGCFSMCS == TRANSITION_LOOP) begin
         if (TransCnt == NumTransInt)
            TransCnt <= 3'b0;
         else
            TransCnt <= TransCnt + 3'b1;
      end
      else
         TransCnt <= 3'b0;
   end //TransCnt_Blk

//Loop back address for transition
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TransRetAddr_Blk
      if (nAGCRst == 1'b0)
         TransRetAddr <= 9'b0;
      else if (AGCFSMSftRst == 1'b1)
         TransRetAddr <= 9'b0;
      else if (AGCFSMNS == TRANSITION_LOOP && AGCFSMCS != TRANSITION_LOOP)
         TransRetAddr <= AGCMemAddrInt;
      else if (AGCFSMNS == SLEEP && AGCFSMCS != SLEEP)
         TransRetAddr <= AGCMemAddrInt-9'd1;
   end //TransRetAddr_Blk

//Command and Parameters
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Cmd_Blk
      if (nAGCRst == 1'b0) begin
         AGCCommandInt <= 8'b0;
         AGCCmdValid   <= 1'b0;
         AGCParamInt   <= 24'b0;
      end
      else if (AGCFSMSftRst == 1'b1 || AGCClearInt == 1'b1) begin
         AGCCommandInt <= 8'b0;
         AGCCmdValid   <= 1'b0;
         AGCParamInt   <= 24'b0;
      end
      else if (AGCFSMCS == COMMAND) begin
         AGCCommandInt <= AGCMemRdData[31:24];
         AGCParamInt   <= AGCMemRdData[23:0];
         AGCCmdValid   <= 1'b1;
      end
      else begin
         AGCCmdValid   <= 1'b0;
      end
   end //Cmd_Blk

assign AGCParam   = AGCParamInt;
assign AGCCommand = AGCCommandInt;

//Debug
assign DbgAgcFsm  = TransRetAddr;


// AGC External Enable Command
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AGCFSMExtEn_Blk
      if (nAGCRst == 1'b0) begin
         AGCFSMExtEn <= 1'b0;
      end
      else if (AGCFSMSftRst == 1'b1) begin
         AGCFSMExtEn <= 1'b0;
      end
      else if (AGCFSMCS == READ_STATE_INFO) begin
         AGCFSMExtEn <= AGCMemRdData[EXT_EN];
      end
   end //AGCFSMExtEn_Blk

//Additional Code to ease verification
//Display FSM State as a string in RTL simulation waveform
`ifdef SIMU_ON
 always @ (*)
   begin: AGCFSMCurrStateStr_Blk
      case (AGCFSMCS)
         STATE_ENTRY     : AGCFSMCurrStateStr = {"STATE_ENTRY"};
         READ_STATE_INFO : AGCFSMCurrStateStr = {"READ_STATE_INFO"};
         COMMAND         : AGCFSMCurrStateStr = {"COMMAND"};
         TRANSITION_LOOP : AGCFSMCurrStateStr = {"TRANSITION_LOOP"};
         SLEEP           : AGCFSMCurrStateStr = {"SLEEP"};
         default         : AGCFSMCurrStateStr = {"UNKNOWN"};
      endcase
 end

 always @ (*)
   begin: AGCFSMNextStateStr_Blk
      case (AGCFSMNS)
         STATE_ENTRY     : AGCFSMNextStateStr = {"STATE_ENTRY"};
         READ_STATE_INFO : AGCFSMNextStateStr = {"READ_STATE_INFO"};
         COMMAND         : AGCFSMNextStateStr = {"COMMAND"};
         TRANSITION_LOOP : AGCFSMNextStateStr = {"TRANSITION_LOOP"};
         SLEEP           : AGCFSMNextStateStr = {"SLEEP"};
         default         : AGCFSMNextStateStr = {"UNKNOWN"};
      endcase
 end
`endif

endmodule //AGCFSMCore

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
