////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: jandre $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 14736 $
// $Date: 2014-04-29 11:20:17 +0200 (Tue, 29 Apr 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : AGC FSM Top Level
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/AGCFSMTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module AGCFSMTop #(parameter NRX         = 4, // Number of Rx Chains (1 to 4)
                   parameter AGCRAMWIDTH = 32 // Datawidth of AGC RAM
                  )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            // Clock and Reset
            input    wire                                  AGCClk,       // AGC Clock
            input    wire                                  nAGCRst,      // Active Low Reset
            input    wire                                  AGCFSMSftRst, // Soft Reset

            // Controls
            // Saturation Detection
            input    wire                                  SatDetectRx0, // Rx Chain 0
            input    wire                                  SatDetectRx1, // Rx Chain 1
            input    wire                                  SatDetectRx2, // Rx Chain 2
            input    wire                                  SatDetectRx3, // Rx Chain 3

            // External Detection
            input    wire                                  ExtDetectRx0, // Rx Chain 0
            input    wire                                  ExtDetectRx1, // Rx Chain 1
            input    wire                                  ExtDetectRx2, // Rx Chain 2
            input    wire                                  ExtDetectRx3, // Rx Chain 3

            //Plateau Detection - Power Stable
            input    wire                                  PlatDetRx0, // Rx Chain 0
            input    wire                                  PlatDetRx1, // Rx Chain 1
            input    wire                                  PlatDetRx2, // Rx Chain 2
            input    wire                                  PlatDetRx3, // Rx Chain 3

            // From Correlators
            input    wire                                  OFDMACDiff,
            input    wire                                  OFDMCCDiff,
            input    wire                                  BarkerACDiff,
            input    wire                                  BarkerCCDiff,

            // From DSSS Continuous Correlator
            input    wire                                  DSSSDetect,

            // From RF Gain Setting Block
            input    wire                                  RFGainSetDoneP,
            input    wire                                  NoGainUpdateP,

            //InBand Power Threshold Crossing Detection
            input    wire                                  DetCrossUpRx0, //Rx Chain 0
            input    wire                                  DetCrossUpRx1, //Rx Chain 1
            input    wire                                  DetCrossUpRx2, //Rx Chain 2
            input    wire                                  DetCrossUpRx3, //Rx Chain 3
            input    wire                                  DetCrossDownRx0, //Rx Chain 0
            input    wire                                  DetCrossDownRx1, //Rx Chain 1
            input    wire                                  DetCrossDownRx2, //Rx Chain 2
            input    wire                                  DetCrossDownRx3, //Rx Chain 3

            // InBand Power Ramp Up/Down Detection
            input    wire                                  DetRampUpRx0, //Rx Chain 0
            input    wire                                  DetRampUpRx1, //Rx Chain 1
            input    wire                                  DetRampUpRx2, //Rx Chain 2
            input    wire                                  DetRampUpRx3, //Rx Chain 3
            input    wire                                  DetRampDownRx0, //Rx Chain 0
            input    wire                                  DetRampDownRx1, //Rx Chain 1
            input    wire                                  DetRampDownRx2, //Rx Chain 2
            input    wire                                  DetRampDownRx3, //Rx Chain 3

            // Power Disappearance Detection
            input    wire                                  ADCPowDisRx0, //Rx Chain 0
            input    wire                                  ADCPowDisRx1, //Rx Chain 1
            input    wire                                  ADCPowDisRx2, //Rx Chain 2
            input    wire                                  ADCPowDisRx3, //Rx Chain 3

            // From RF Gain Comparator Block
            input    wire                                  RFGainCp2MaxRx0,//Rx Chain 0
            input    wire                                  RFGainCp2MaxRx1,//Rx Chain 1
            input    wire                                  RFGainCp2MaxRx2,//Rx Chain 2
            input    wire                                  RFGainCp2MaxRx3,//Rx Chain 3
            input    wire                                  RFGainCp2MinRx0,//Rx Chain 0
            input    wire                                  RFGainCp2MinRx1,//Rx Chain 1
            input    wire                                  RFGainCp2MinRx2,//Rx Chain 2
            input    wire                                  RFGainCp2MinRx3,//Rx Chain 3
            input    wire                                  RFGainCp2Min4SatRx0,//Rx Chain 0
            input    wire                                  RFGainCp2Min4SatRx1,//Rx Chain 1
            input    wire                                  RFGainCp2Min4SatRx2,//Rx Chain 2
            input    wire                                  RFGainCp2Min4SatRx3,//Rx Chain 3

            // From AGC Detector Block
            input    wire                                  CmdCtrlFlag,

            // From Energy Comparator Block
            input    wire                                  InBdPowSupRx0,
            input    wire                                  InBdPowSupRx1,
            input    wire                                  InBdPowSupRx2,
            input    wire                                  InBdPowSupRx3,
            input    wire                                  InBdPowInfRx0,
            input    wire                                  InBdPowInfRx1,
            input    wire                                  InBdPowInfRx2,
            input    wire                                  InBdPowInfRx3,
            input    wire                                  IdPowRx0,
            input    wire                                  IdPowRx1,
            input    wire                                  IdPowRx2,
            input    wire                                  IdPowRx3,
            //
            input    wire                                  ADCPowdBVSupRx0,
            input    wire                                  ADCPowdBVSupRx1,
            input    wire                                  ADCPowdBVSupRx2,
            input    wire                                  ADCPowdBVSupRx3,
            input    wire                                  ADCPowdBVInfRx0,
            input    wire                                  ADCPowdBVInfRx1,
            input    wire                                  ADCPowdBVInfRx2,
            input    wire                                  ADCPowdBVInfRx3,
            input    wire                                  ADCPowdBmSupRx0,
            input    wire                                  ADCPowdBmSupRx1,
            input    wire                                  ADCPowdBmSupRx2,
            input    wire                                  ADCPowdBmSupRx3,
            input    wire                                  ADCPowdBmInfRx0,
            input    wire                                  ADCPowdBmInfRx1,
            input    wire                                  ADCPowdBmInfRx2,
            input    wire                                  ADCPowdBmInfRx3,

            // From modem control
            input    wire                                  HTSTFStartEstimP,
            input    wire                                  HTSTFGainUpdateP,
            input    wire                                  RxEndTimingP,
            input    wire                                  LSIGValid,
            input    wire                                  HTSIGValid,
            input    wire                                  SFDFound,
            input    wire                                  DemodRun,
            
            // From Registers
            input    wire                                  RifsEn,
            input    wire                                  RadarDet,
            input    wire                                  ModeA,
            input    wire                                  ModeB,
            input    wire      [3:0]                       GPSetting,
            input    wire                                  Force20,
            input    wire                                  Force40,

            // From Radar
            input    wire                                  FomSing,      // FOM singuliar status
            input    wire                                  FomHigh,      // FOM high status
            input    wire                                  FomMed,       // FOM low status
            input    wire                                  FreqSing,     // Frequency singuliar status
            input    wire                                  FreqDC,       // Frequency DC status
            input    wire                                  FreqLow,      // Frequency low status
            input    wire                                  FreqHigh,     // Frequency high status
            input    wire                                  LengthLow,    // Timer length low status
            input    wire                                  LengthHigh,   // Timer length high status
            input    wire                                  Meas1Done,    // Measure 1 done status
            input    wire                                  Meas2Done,    // Measure 2 done status
            input    wire                                  Meas3Done,    // Measure 3 done status

            // From AGC RAM
            input    wire      [AGCRAMWIDTH-1:0]           AGCMemRdData,

            // From AGC Configuration Registers
            input    wire      [3:0]                       ChCombMode, //Combining Rx Chains
            input    wire      [31:0]                      FSMEvent0Reg, //Event 0 Register
            input    wire      [31:0]                      FSMEvent1Reg, //Event 1 Register
            input    wire      [31:0]                      FSMEvent2Reg, //Event 2 Register
            input    wire      [31:0]                      FSMEvent3Reg, //Event 3 Register

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            // Address for AGC RAM
            output   wire      [8:0]                       AGCMemAddr, //Read Address

            // Controls
            output   wire      [7:0]                       AGCCommand, //Commands to Modules
            output   wire      [23:0]                      AGCParam, //Command Parameter 1
            output   wire                                  AGCCmdValid, //AGC Command Valid 
            output   wire      [14:0]                      DSPEn, //Enable for DSP Blocks
            output   wire                                  EventSat, //Saturation Event occurence to CCA
            output   wire                                  EventDis, //Disappearance Event occurence to CCA
            output   wire                                  AGCFSMExtEn, //AGC external enable

            // Debug port
            output   wire      [15:0]                      DbgAgc,     // Debug port
            output   wire      [8:0]                       DbgAgcFsm   // FSM address index
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Registers Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [21:0]              TransCond;
wire               [10:0]              TimeOutValue;
wire                                   TimeOutEn;
wire                                   TimeOut;
wire                                   AGCClear;
wire                                   AGCStatusClear;
wire                                   AGCMaskEvent;
wire                                   EventValid;
wire               [8:0]               EventTargetAddr;
wire                                   TransValid;
wire               [127:0]             AGCStatusReg;
wire                                   EventDisToCCA;

reg                [2:0]               SatCount;
reg                [2:0]               GenCount;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module instantiates the following AGC FSM related blocks:
// 1. AGCFSMCore
// 2. AGCStatusRegister
// 3. TransDec
// 4. EventDecTop
// 5. TimeOutCtr

// Instantiate AGCFSMCore
AGCFSMCore # (
              // Parameters
              .AGCRAMWIDTH(AGCRAMWIDTH)
             ) U_AGCFSMCore (
                             // Inputs
                             .nAGCRst(nAGCRst),
                             .AGCFSMSftRst(AGCFSMSftRst),
                             .AGCClk(AGCClk),
                             .EventValid(EventValid),
                             .EventTargetAddr(EventTargetAddr),
                             .TransValid(TransValid),
                             .AGCMemRdData(AGCMemRdData),

                             // Outputs
                             .TransCond(TransCond),
                             .AGCMemAddr(AGCMemAddr),
                             .AGCCommand(AGCCommand),
                             .AGCCmdValid(AGCCmdValid),
                             .AGCParam(AGCParam),
                             .TimeOutValue(TimeOutValue),
                             .TimeOutEn(TimeOutEn),
                             .DSPEn(DSPEn),
                             .AGCClear(AGCClear),
                             .AGCMaskEvent(AGCMaskEvent),
                             .AGCFSMExtEn(AGCFSMExtEn),
                             .DbgAgcFsm(DbgAgcFsm)
                            );

// AGC status register clear is or'ed with AGC FSM clear and AGC mask event
// in order to avoid multi-event generation before current event has been executed
assign AGCStatusClear = AGCClear | AGCMaskEvent;

// Instantiate AGCStatusRegister
AGCStatusRegister U_AGCSReg (
                             // Inputs
                             .AGCClk(AGCClk),
                             .nAGCRst(nAGCRst),
                             .AGCClear(AGCStatusClear),
                             .SatDetectRx0(SatDetectRx0),
                             .SatDetectRx1(SatDetectRx1),
                             .SatDetectRx2(SatDetectRx2),
                             .SatDetectRx3(SatDetectRx3),
                             .SatCount(SatCount),
                             .ExtDetectRx0(ExtDetectRx0),
                             .ExtDetectRx1(ExtDetectRx1),
                             .ExtDetectRx2(ExtDetectRx2),
                             .ExtDetectRx3(ExtDetectRx3),
                             .GenCount(GenCount),
                             .PlatDetRx0(PlatDetRx0),
                             .PlatDetRx1(PlatDetRx1),
                             .PlatDetRx2(PlatDetRx2),
                             .PlatDetRx3(PlatDetRx3),
                             .OFDMACDiff(OFDMACDiff),
                             .OFDMCCDiff(OFDMCCDiff),
                             .BarkerACDiff(BarkerACDiff),
                             .BarkerCCDiff(BarkerCCDiff),
                             .DSSSDetect(DSSSDetect),
                             .TimeOut(TimeOut),
                             .RFGainSetDoneP(RFGainSetDoneP),
                             .NoGainUpdateP(NoGainUpdateP),
                             .DetCrossUpRx0(DetCrossUpRx0),
                             .DetCrossUpRx1(DetCrossUpRx1),
                             .DetCrossUpRx2(DetCrossUpRx2),
                             .DetCrossUpRx3(DetCrossUpRx3),
                             .DetCrossDownRx0(DetCrossDownRx0),
                             .DetCrossDownRx1(DetCrossDownRx1),
                             .DetCrossDownRx2(DetCrossDownRx2),
                             .DetCrossDownRx3(DetCrossDownRx3),
                             .DetRampUpRx0(DetRampUpRx0),
                             .DetRampUpRx1(DetRampUpRx1),
                             .DetRampUpRx2(DetRampUpRx2),
                             .DetRampUpRx3(DetRampUpRx3),
                             .DetRampDownRx0(DetRampDownRx0),
                             .DetRampDownRx1(DetRampDownRx1),
                             .DetRampDownRx2(DetRampDownRx2),
                             .DetRampDownRx3(DetRampDownRx3),
                             .ADCPowDisRx0(ADCPowDisRx0),
                             .ADCPowDisRx1(ADCPowDisRx1),
                             .ADCPowDisRx2(ADCPowDisRx2),
                             .ADCPowDisRx3(ADCPowDisRx3),
                             .RFGainCp2MaxRx0(RFGainCp2MaxRx0),
                             .RFGainCp2MaxRx1(RFGainCp2MaxRx1),
                             .RFGainCp2MaxRx2(RFGainCp2MaxRx2),
                             .RFGainCp2MaxRx3(RFGainCp2MaxRx3),
                             .RFGainCp2MinRx0(RFGainCp2MinRx0),
                             .RFGainCp2MinRx1(RFGainCp2MinRx1),
                             .RFGainCp2MinRx2(RFGainCp2MinRx2),
                             .RFGainCp2MinRx3(RFGainCp2MinRx3),
                             .RFGainCp2Min4SatRx0(RFGainCp2Min4SatRx0),
                             .RFGainCp2Min4SatRx1(RFGainCp2Min4SatRx1),
                             .RFGainCp2Min4SatRx2(RFGainCp2Min4SatRx2),
                             .RFGainCp2Min4SatRx3(RFGainCp2Min4SatRx3),
                             .CmdCtrlFlag(CmdCtrlFlag),
                             .InBdPowSupRx0(InBdPowSupRx0),
                             .InBdPowSupRx1(InBdPowSupRx1),
                             .InBdPowSupRx2(InBdPowSupRx2),
                             .InBdPowSupRx3(InBdPowSupRx3),
                             .InBdPowInfRx0(InBdPowInfRx0),
                             .InBdPowInfRx1(InBdPowInfRx1),
                             .InBdPowInfRx2(InBdPowInfRx2),
                             .InBdPowInfRx3(InBdPowInfRx3),
                             .IdPowRx0(IdPowRx0),
                             .IdPowRx1(IdPowRx1),
                             .IdPowRx2(IdPowRx2),
                             .IdPowRx3(IdPowRx3),
                             .ADCPowdBVSupRx0(ADCPowdBVSupRx0),
                             .ADCPowdBVSupRx1(ADCPowdBVSupRx1),
                             .ADCPowdBVSupRx2(ADCPowdBVSupRx2),
                             .ADCPowdBVSupRx3(ADCPowdBVSupRx3),
                             .ADCPowdBVInfRx0(ADCPowdBVInfRx0),
                             .ADCPowdBVInfRx1(ADCPowdBVInfRx1),
                             .ADCPowdBVInfRx2(ADCPowdBVInfRx2),
                             .ADCPowdBVInfRx3(ADCPowdBVInfRx3),
                             .ADCPowdBmSupRx0(ADCPowdBmSupRx0),
                             .ADCPowdBmSupRx1(ADCPowdBmSupRx1),
                             .ADCPowdBmSupRx2(ADCPowdBmSupRx2),
                             .ADCPowdBmSupRx3(ADCPowdBmSupRx3),
                             .ADCPowdBmInfRx0(ADCPowdBmInfRx0),
                             .ADCPowdBmInfRx1(ADCPowdBmInfRx1),
                             .ADCPowdBmInfRx2(ADCPowdBmInfRx2),
                             .ADCPowdBmInfRx3(ADCPowdBmInfRx3),
                             .HTSTFStartEstimP(HTSTFStartEstimP),
                             .HTSTFGainUpdateP(HTSTFGainUpdateP),
                             .RxEndTimingP(RxEndTimingP),
                             .LSIGValid(LSIGValid),
                             .HTSIGValid(HTSIGValid),
                             .SFDFound(SFDFound),
                             .DemodRun(DemodRun),
                             .RifsEn(RifsEn),
                             .RadarDet(RadarDet),
                             .ModeA(ModeA),
                             .ModeB(ModeB),
                             .GPSetting(GPSetting),
                             .Force20(Force20),
                             .Force40(Force40),
                             .FomSing(FomSing),
                             .FomHigh(FomHigh),
                             .FomMed(FomMed),
                             .FreqSing(FreqSing),
                             .FreqDC(FreqDC),
                             .FreqLow(FreqLow),
                             .FreqHigh(FreqHigh),
                             .LengthLow(LengthLow),
                             .LengthHigh(LengthHigh),
                             .Meas1Done(Meas1Done),
                             .Meas2Done(Meas2Done),
                             .Meas3Done(Meas3Done),

                             // Output
                             .AGCStatusReg(AGCStatusReg)
                            );

// Instantiate TransDec
TransDec # (
            // Parameter
            .NRX(NRX)
            ) U_TransDec (
                          // Inputs
                          .TransCondReg(TransCond),
                          .AGCStatusReg(AGCStatusReg),
                          .ChCombMode(ChCombMode),

                          //Outputs
                          .TransValid(TransValid),
                          .DbgAgc(DbgAgc)
                         );

// Instantiate EventDecTop
EventDecTop # (
               // Parameter
               .NRX(NRX)
              ) U_EventDecTop (
                               // Inputs
                               .AGCClk(AGCClk),
                               .nAGCRst(nAGCRst),
                               .AGCStatusReg(AGCStatusReg),
                               .ChCombMode(ChCombMode),
                               .FSMEvent0Reg(FSMEvent0Reg),
                               .FSMEvent1Reg(FSMEvent1Reg),
                               .FSMEvent2Reg(FSMEvent2Reg),
                               .FSMEvent3Reg(FSMEvent3Reg),
                               .AGCMaskEvent(AGCMaskEvent),

                               // Outputs
                               .EventValid(EventValid),
                               .EventTargetAddr(EventTargetAddr),
                               .EventSat(EventSat),
                               .EventDis(EventDisToCCA)
                              );

// Instantiate TimeOutCtr
TimeOutCtr U_TimeOutCtr (
                         // Inputs
                         .AGCClk(AGCClk),
                         .nAGCRst(nAGCRst),
                         .TimeOutEn(TimeOutEn),
                         .TimeOutValue(TimeOutValue),
                         .TimeOutRst(AGCClear),

                         // Output
                         .TimeOut(TimeOut)
                         );

// Saturation event counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SatCount_Blk
      if (nAGCRst == 1'b0)
         SatCount <= 3'h0;
      else if ((AGCCommand == 8'd1 || AGCCommand == 8'd90) &&
               (AGCCmdValid == 1'b1))
         SatCount <= 3'h0;
      else if ((AGCCommand == 8'd91) && (AGCCmdValid == 1'b1) &&
               (SatCount < 3'h5))
         SatCount <= SatCount + 3'h1;
   end // SatCount_Blk

// Generic counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: GenCount_Blk
      if (nAGCRst == 1'b0)
         GenCount <= 3'h0;
      else if ((AGCCommand == 8'd1 || AGCCommand == 8'd95) &&
               (AGCCmdValid == 1'b1))
         GenCount <= 3'h0;
      else if ((AGCCommand == 8'd96) && (AGCCmdValid == 1'b1) &&
               (GenCount < 3'h5))
         GenCount <= GenCount + 3'h1;
   end // GenCount_Blk

// When AGC detects a false alarm, EventDis is sent to CCA
// else CCA generated on Threshold Crossing (ThreshDetSgnHys) stays high
assign EventDis = EventDisToCCA | ((AGCCommand == 8'd80) && AGCCmdValid);

endmodule // AGCFSMTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
