////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12473 $
// $Date: 2013-12-12 18:40:37 +0100 (Thu, 12 Dec 2013) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Correlation Comparator Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/CorrComp.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module CorrComp #(parameter DATAOUTOFDMAC  = 22, // OFDM Auto  Corr Width
                  parameter DATAOUTOFDMCC  = 20, // OFDM Cross Corr Width
                  parameter DATAOUTDSSSAC  = 22, // DSSS Auto  Corr Width
                  parameter DATAOUTDSSSCC  = 20, // DSSS Cross Corr Width
                  parameter DATAOUTOFDMREF = 19, // OFDM Reference  Width
                  parameter DATAOUTDSSSREF = 20  // DSSS Reference  Width
                  )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,   // Active Low Reset
            input    wire                                  AGCClk,    // AGC Clock

            //Controls
            input    wire                                  CorrDetEn,   // Block enable from FSM Core
            input    wire              [7:0]               AGCCommand,  // Command from FSM Core
            input    wire              [23:0]              AGCParam,    // Command Parameter
            input    wire                                  AGCCmdValid, // AGC Command Valid

            //Correlation and Reference Values
            input    wire             [DATAOUTOFDMAC-1:0]  OfdmACAccSum, // OFDM Auto Correlation Sum
            input    wire             [DATAOUTOFDMCC-1:0]  OfdmCCAccSum, // OFDM Cross Correlation Sum
            input    wire             [DATAOUTDSSSAC-1:0]  DsssACAccSum, // DSSS Auto Correlation Sum
            input    wire             [DATAOUTDSSSCC-1:0]  DsssCCAccSum, // DSSS Cross Correlation Sum
            input    wire             [DATAOUTDSSSREF-1:0] RefLevDsss,   // OFDM Reference Value
            input    wire             [DATAOUTOFDMREF-1:0] RefLevOfdm,   // DSSS Reference Value
            input    wire                                  RefLevDataEn,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Comparators
            output   wire                                  OFDMACDiff, // OFDM vs Noise (Auto)
            output   wire                                  OFDMCCDiff, // OFDM vs Noise (Cross)
            output   wire                                  DSSSACDiff, // DSSS vs Noise (Auto)
            output   wire                                  DSSSCCDiff  // DSSS vs Noise (Cross)
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [DATAOUTOFDMREF+1:0] OFDMCCMultInt;
wire               [DATAOUTOFDMREF+1:0] OFDMACMultInt;
wire               [DATAOUTDSSSREF+1:0] DSSSCCMultInt;
wire               [DATAOUTDSSSREF+1:0] DSSSACMultInt;
wire                                    OFDMACDiffInt;
wire                                    DSSSACDiffInt;
wire                                    OFDMCCDiffInt;
wire               [3:0]                OFDMCCDiffIntAcc;
wire               [2:0]                OFDMCCDiffIntAccSat;
wire               [2:0]                OFDMCCDiffIntAccMux;
wire                                    DSSSCCDiffInt;
wire               [4:0]                DSSSCCDiffIntAcc;
wire               [3:0]                DSSSCCDiffIntAccSat;
wire               [3:0]                DSSSCCDiffIntAccMux;
wire               [23:0]               AGCParam_aux;
wire                                    ComparePulse;

//Integers for loop generation
integer i,j,k,l;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                     CorrDetEnD;
reg                [3:0]                ComparePulseDelayLine;
reg                                     OFDMCrossCorrMode;
reg                                     DSSSCrossCorrMode;
reg                [2:0]                OFDMCrossCorrPeak;
reg                [3:0]                DSSSCrossCorrPeak;
reg                [6:0]                OFDMACThrReg;
reg                [6:0]                OFDMCCThrReg;
reg                [6:0]                DSSSACThrReg;
reg                [6:0]                DSSSCCThrReg;
reg                [DATAOUTOFDMREF+6:0] OFDMCCMult;
reg                [DATAOUTOFDMREF+6:0] OFDMACMult;
reg                [DATAOUTDSSSREF+6:0] DSSSCCMult;
reg                [DATAOUTDSSSREF+6:0] DSSSACMult;
reg                [2:0]                OFDMCCDelayLine[15:0];
reg                [3:0]                DSSSCCDelayLine[19:0];
reg                                     OFDMCCDiffPeak;
reg                                     DSSSCCDiffPeak;
reg                                     OFDMACDiffOut;
reg                                     OFDMCCDiffOut;
reg                                     DSSSACDiffOut;
reg                                     DSSSCCDiffOut;
reg                                     OFDMACDiffFix;
reg                                     OFDMCCDiffFix;
reg                                     DSSSACDiffFix;
reg                                     DSSSCCDiffFix;
reg               [DATAOUTOFDMAC-1:0]   OfdmACAccSumD;
reg               [DATAOUTOFDMCC-1:0]   OfdmCCAccSumD;
reg               [DATAOUTDSSSAC-1:0]   DsssACAccSumD;
reg               [DATAOUTDSSSCC-1:0]   DsssCCAccSumD;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module compares the autocorrelation and crosscorrelation values of OFDM
// and DSSS correlators. It is used for packet detection and discrimination.

// Set Cross Correlation Mode
// 0 = Single Maximum; 1 = Sum of Multiple Maxima
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CrossCorrMode_Blk
      if (nAGCRst == 1'b0)
      begin
         OFDMCrossCorrMode <= 1'b0;
         DSSSCrossCorrMode <= 1'b1;
      end
      else if ((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1))
      begin
         OFDMCrossCorrMode <= 1'b0;
         DSSSCrossCorrMode <= 1'b1;
      end
      else if ((AGCCommand == 8'd31) && (AGCCmdValid == 1'b1))
         OFDMCrossCorrMode <= AGCParam[0];
      else if ((AGCCommand == 8'd32) && (AGCCmdValid == 1'b1))
         DSSSCrossCorrMode <= AGCParam[0];
   end // CrossCorrMode_Blk

// Set Cross Correlation Peak
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CrossCorrPeak_Blk
      if (nAGCRst == 1'b0)
      begin
         OFDMCrossCorrPeak <= 3'd1;
         DSSSCrossCorrPeak <= 4'd7;
      end
      else if ((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1))
      begin
         OFDMCrossCorrPeak <= 3'd1;
         DSSSCrossCorrPeak <= 4'd7;
      end
      else if ((AGCCommand == 8'd31) && (AGCCmdValid == 1'b1))
         OFDMCrossCorrPeak <= AGCParam[6:4];
      else if ((AGCCommand == 8'd32) && (AGCCmdValid == 1'b1))
         DSSSCrossCorrPeak <= AGCParam[7:4];
   end // CrossCorrPeak_Blk

// OFDM and DSSS Threshold Registers (for comparison vs Noise)
always @ (posedge AGCClk or negedge nAGCRst)
   begin:ThrReg_Blk
      if (nAGCRst == 1'b0)
      begin
         OFDMACThrReg <= 7'd32;
         OFDMCCThrReg <= 7'd32;
         DSSSACThrReg <= 7'd32;
         DSSSCCThrReg <= 7'd32;
      end
      else if ((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1))
      begin
         // Reset Values
         OFDMACThrReg <= 7'd32;
         OFDMCCThrReg <= 7'd32;
         DSSSACThrReg <= 7'd32;
         DSSSCCThrReg <= 7'd32;
      end
      else if ((AGCCommand == 8'd31) && (AGCCmdValid == 1'b1))
      begin
         // Update OFDM Thresholds
         OFDMACThrReg <= AGCParam[22:16];
         OFDMCCThrReg <= AGCParam[14:8];
      end
      else if ((AGCCommand == 8'd32) && (AGCCmdValid == 1'b1))
      begin
         // Update DSSS Thresholds
         DSSSACThrReg <= AGCParam[22:16];
         DSSSCCThrReg <= AGCParam[14:8];
      end
   end // ThrReg_Blk

// Multiply with Threshold values
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Mult_Blk
      if (nAGCRst == 1'b0)
      begin
         OFDMCCMult <= {{(DATAOUTOFDMREF+7)}{1'b0}};
         OFDMACMult <= {{(DATAOUTOFDMREF+7)}{1'b0}};
         DSSSCCMult <= {{(DATAOUTDSSSREF+7)}{1'b0}};
         DSSSACMult <= {{(DATAOUTDSSSREF+7)}{1'b0}};
      end
      else if ((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1))
      begin
         // Reset Values
         OFDMCCMult <= {{(DATAOUTOFDMREF+7)}{1'b0}};
         OFDMACMult <= {{(DATAOUTOFDMREF+7)}{1'b0}};
         DSSSCCMult <= {{(DATAOUTDSSSREF+7)}{1'b0}};
         DSSSACMult <= {{(DATAOUTDSSSREF+7)}{1'b0}};
      end
      else if (RefLevDataEn == 1'b1)
      begin
         OFDMCCMult <= {{{(DATAOUTOFDMREF-7)}{1'b0}},OFDMCCThrReg} * RefLevOfdm;
         OFDMACMult <= {{{(DATAOUTOFDMREF-7)}{1'b0}},OFDMACThrReg} * RefLevOfdm;
         DSSSCCMult <= {{{(DATAOUTDSSSREF-7)}{1'b0}},DSSSCCThrReg} * RefLevDsss;
         DSSSACMult <= {{{(DATAOUTDSSSREF-7)}{1'b0}},DSSSACThrReg} * RefLevDsss;
      end
   end // Mult_Blk

// Divide above products by 32
assign OFDMCCMultInt = OFDMCCMult[DATAOUTOFDMREF+6:5];
assign OFDMACMultInt = OFDMACMult[DATAOUTOFDMREF+6:5];
assign DSSSCCMultInt = DSSSCCMult[DATAOUTDSSSREF+6:5];
assign DSSSACMultInt = DSSSACMult[DATAOUTDSSSREF+6:5];

// Delayed accumulators by 1 cc for comparison
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AccuSumD_Blk
      if (nAGCRst == 1'b0) begin
         OfdmACAccSumD <= {{DATAOUTOFDMAC}{1'b0}};
         OfdmCCAccSumD <= {{DATAOUTOFDMCC}{1'b0}};
         DsssACAccSumD <= {{DATAOUTDSSSAC}{1'b0}};
         DsssCCAccSumD <= {{DATAOUTDSSSCC}{1'b0}};
      end
      else begin
         OfdmACAccSumD <= OfdmACAccSum;
         OfdmCCAccSumD <= OfdmCCAccSum;
         DsssACAccSumD <= DsssACAccSum;
         DsssCCAccSumD <= DsssCCAccSum;
      end
   end // AccuSumD_Blk

// Compare values
assign OFDMCCDiffInt = (({{{(DATAOUTOFDMCC-DATAOUTOFDMREF)}{1'b0}}, OfdmCCAccSumD} > OFDMCCMultInt) && CorrDetEn) ? 1'b1 : 1'b0;
assign OFDMACDiffInt = ((OfdmACAccSumD > {{{((DATAOUTOFDMAC-1)-(DATAOUTOFDMREF+1))}{1'b0}}, OFDMACMultInt}) && CorrDetEn) ? 1'b1 : 1'b0;
assign DSSSCCDiffInt = (({{{((DATAOUTDSSSREF+1)-(DATAOUTDSSSCC-1) )}{1'b0}}, DsssCCAccSumD} > DSSSCCMultInt) && CorrDetEn) ? 1'b1 : 1'b0;
assign DSSSACDiffInt = ((DsssACAccSumD > DSSSACMultInt) && CorrDetEn) ? 1'b1 : 1'b0;

// Delayed CorrDetEn by 1 cc
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CorrDetEnD_Blk
      if (nAGCRst == 1'b0) begin
         CorrDetEnD            <= 1'b0;
         ComparePulseDelayLine <= 4'b0;
      end
      else begin
         CorrDetEnD <= CorrDetEn;
         if (!CorrDetEn)
           ComparePulseDelayLine <= 4'b0;
         else begin
           ComparePulseDelayLine[3:1] <= ComparePulseDelayLine[2:0];
           ComparePulseDelayLine[0]   <= ComparePulse | ComparePulseDelayLine[3];
         end
      end
   end // CorrDetEnD_Blk

assign ComparePulse = ((CorrDetEn & !CorrDetEnD) | ComparePulseDelayLine[3]);

// OFDM Cross Correlation accumulation for peak search
assign OFDMCCDiffIntAcc = {1'b0,OFDMCCDelayLine[15]} + {3'b0,OFDMCCDiffInt};

// Saturate OFDM accumulation
SatUnsigned # (
               .INPUT_WIDTH(4),
               .OUTPUT_WIDTH(3)
              ) U_SATOFDMCCACC (
                           .InputData(OFDMCCDiffIntAcc),
                           .SatData(OFDMCCDiffIntAccSat)
                          );

// OFDM Cross Correlation mux for peak search
assign OFDMCCDiffIntAccMux = (OFDMCrossCorrMode == 1'b1) ? OFDMCCDiffIntAccSat :
                             {2'b0,OFDMCCDiffInt};

// OFDM Cross Correlation Delay line for peak search
always @ (posedge AGCClk or negedge nAGCRst)
   begin: OFDMCCDelayLine_Blk
      if (nAGCRst == 1'b0)
      begin
         for (i = 0; i <= 15; i = i+1) begin
            OFDMCCDelayLine[i] <= 3'h0;
         end
      end
      else if (((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1)) || (CorrDetEn == 1'b0))
      begin
         for (i = 0; i <= 15; i = i+1) begin
            OFDMCCDelayLine[i] <= 3'h0;
         end
      end
      else if (ComparePulse == 1'b1)
      begin
         OFDMCCDelayLine[0] <= OFDMCCDiffIntAccMux;
         for (i = 1; i <= 15; i = i+1) begin
            OFDMCCDelayLine[i] <= OFDMCCDelayLine[i-1];
         end
      end
   end // OFDMCCDelayLine_Blk

// OFDM Cross Correlation peak detection
always@(*)
begin
  OFDMCCDiffPeak = 1'b0;
  for (j = 0; j <= 15; j = j+1) begin
     if(OFDMCCDelayLine[j] >= OFDMCrossCorrPeak)
        OFDMCCDiffPeak = CorrDetEn;
  end
end


// DSSS Cross Correlation accumulation for peak search
assign DSSSCCDiffIntAcc = {1'b0,DSSSCCDelayLine[19][3:0]} + {4'b0,DSSSCCDiffInt};

// Saturate DSSS accumulation
SatUnsigned # (
               .INPUT_WIDTH(5),
               .OUTPUT_WIDTH(4)
              ) U_SATDSSSCCACC (
                           .InputData(DSSSCCDiffIntAcc),
                           .SatData(DSSSCCDiffIntAccSat)
                          );

// DSSS Cross Correlation mux for peak search
assign DSSSCCDiffIntAccMux = (DSSSCrossCorrMode == 1'b1) ? DSSSCCDiffIntAccSat :
                             {3'b0,DSSSCCDiffInt};

// DSSS Cross Correlation Delay line for peak search
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DSSSCCDelayLine_Blk
      if (nAGCRst == 1'b0)
      begin
         for (k = 0; k <= 19; k = k+1) begin
            DSSSCCDelayLine[k][3:0] <= 4'h0;
         end
      end
      else if (((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1)) || (CorrDetEn == 1'b0))
      begin
         for (k = 0; k <= 19; k = k+1) begin
            DSSSCCDelayLine[k][3:0] <= 4'h0;
         end
      end
      else if (ComparePulse == 1'b1)
      begin
         DSSSCCDelayLine[0][3:0] <= DSSSCCDiffIntAccMux;
         for (k = 1; k <= 19; k = k+1) begin
            DSSSCCDelayLine[k][3:0] <= DSSSCCDelayLine[k-1][3:0];
         end
      end
   end // DSSSCCDelayLine_Blk

// DSSS Cross Correlation peak detection
always@(*)
begin
  DSSSCCDiffPeak = 1'b0;
  for (l = 0; l <= 19; l = l+1) begin
     if(DSSSCCDelayLine[l][3:0] >= DSSSCrossCorrPeak[3:0]) begin
        DSSSCCDiffPeak = CorrDetEn;
     end
  end
end


// Fix the outputs on command
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CompOut_Blk
      if (nAGCRst == 1'b0)
      begin
         OFDMACDiffOut <= 1'b0;
         OFDMCCDiffOut <= 1'b0;
         DSSSACDiffOut <= 1'b0;
         DSSSCCDiffOut <= 1'b0;
         OFDMACDiffFix <= 1'b0;
         OFDMCCDiffFix <= 1'b0;
         DSSSACDiffFix <= 1'b0;
         DSSSCCDiffFix <= 1'b0;
      end
      else if (((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1)) || (CorrDetEn == 1'b0))
      begin
         // Reset Values
         OFDMACDiffOut <= 1'b0;
         OFDMCCDiffOut <= 1'b0;
         DSSSACDiffOut <= 1'b0;
         DSSSCCDiffOut <= 1'b0;
         OFDMACDiffFix <= 1'b0;
         OFDMCCDiffFix <= 1'b0;
         DSSSACDiffFix <= 1'b0;
         DSSSCCDiffFix <= 1'b0;
      end
      else if ((AGCCommand == 8'd33) && (AGCCmdValid == 1'b1))
      begin
         // Register the comparator outputs and fix outputs
         OFDMCCDiffFix <= AGCParam[0];
         OFDMACDiffFix <= AGCParam[1];
         DSSSCCDiffFix <= AGCParam[8];
         DSSSACDiffFix <= AGCParam[9];
      end
      else
      begin
        // Register the comparator outputs and keep value on fix command (cmd=8'h33)
        if (OFDMCCDiffFix == 1'b0)
          OFDMCCDiffOut <= OFDMCCDiffPeak | OFDMCCDiffOut;
        if (OFDMACDiffFix == 1'b0)
          OFDMACDiffOut <= OFDMACDiffInt  | OFDMACDiffOut;
        if (DSSSCCDiffFix == 1'b0)
          DSSSCCDiffOut <= DSSSCCDiffPeak | DSSSCCDiffOut;
        if (DSSSACDiffFix == 1'b0)
          DSSSACDiffOut <= DSSSACDiffInt  | DSSSACDiffOut;
      end
   end // CompOut_Blk

//
assign AGCParam_aux = AGCParam; // To Avoid Lint Warning
//

// Output generation
assign OFDMACDiff = OFDMACDiffOut;
assign OFDMCCDiff = OFDMCCDiffOut;
assign DSSSACDiff = DSSSACDiffOut;
assign DSSSCCDiff = DSSSCCDiffOut;

endmodule // CorrComp

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
