////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 15050 $
// $Date: 2014-05-23 15:58:20 +0200 (Fri, 23 May 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : DSP Blocks Register Configuration Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/DSPDynConfig.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module DSPDynConfig #(parameter DBMPOW_WIDTH    = 13, // Data width of dBm power
                      parameter DBVPOW_WIDTH    = 12 // Data width of dBV power 
                     )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,   //Active Low Reset
            input    wire                                  AGCClk,    //AGC Clock

            //Controls from AGC FSM
            input    wire              [7:0]               AGCCommand,  //Command from FSM Core
            input    wire              [23:0]              AGCParam,    //Command Parameter
            input    wire                                  AGCCmdValid, //AGC Command Valid
            
            //Control from ModemIFReg
            input    wire                                  AGCOFDMLock, //OFDM lock
            input    wire                                  AGCDSSSLock, //DSSS lock

            //Control from register
            input    wire                                  InbdPowStartCaptureP, //Register command
            input    wire                                  StatusTrigConf,       //Confirmation for status register capture

            // In Band Power in dBm for 20 MHz packet
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx0, //Power in Rx Chain 0
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx1, //Power in Rx Chain 1
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx2, //Power in Rx Chain 2
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx3, //Power in Rx Chain 3

            // In Band Power in dBm for 20 MHz packet in Secondary Channel
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20SecPowdBm,

            // In Band Power in dBm for 40 MHz packet
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd40SecPowdBm,

            // ADC Power Estimate in dBVrms
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx0, //Power in Rx Chain 0
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx1, //Power in Rx Chain 1
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx2, //Power in Rx Chain 2
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx3, //Power in Rx Chain 3

            // ADC Power Max/Min index
            input    wire              [1:0]               ADCPowdBVMaxIndex, //Antenna max index
            input    wire              [1:0]               ADCPowdBVMinIndex, //Antenna min index

            // Inband Power Max/Min index
            input    wire              [1:0]               InbandPowMaxIndex, //Antenna max index
            input    wire              [1:0]               InbandPowMinIndex, //Antenna min index

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   wire      [2:0]                       InBd20SWLConfig,    //Delay Line Tap 
            output   wire      [2:0]                       InBd20SecSWLConfig, //Delay Line Tap 
            output   wire      [2:0]                       InBd40SecSWLConfig, //Delay Line Tap 
            output   wire      [1:0]                       ADCSWLSatConfig,    //Delay Line Tap 
            output   wire      [1:0]                       ADCSWLConfig,       //Delay Line Tap 
            output   wire      [1:0]                       OFDMACModeSel,      //Mode Select 
            output   wire                                  DSSSACModeSel,      //Mode Select
            output   wire      [1:0]                       DCMode,             //Select DC estimation mode
            output   wire      [1:0]                       AntSelDSSS,         //Antenna Mode Select for DSSS-CCK modem
            output   wire      [1:0]                       AntSelBWEst,        //Antenna Mode Select for BW estimator
            output   wire      [2:0]                       DSSSContCorrModeSel,//DSSS Continuous correlation Mode Select
            output   wire      [6:0]                       DSSSContCCThr,      //DSSS Continuous correlation Threshold
            output   reg                                   IQEstGainAccEn,     //IQ Imbalance gain accu enable
            output   reg                                   IQEstPhaseAccEn,    //IQ Imbalance phase accu enable

            //Interface with AGC status registers
            output  reg                                    StatusTrigReq,     //Request for status register capture
            output  reg   signed     [9:0]                 InBd20PowqdBmAnt0, //In Band Power 20P in qdBm for Rx Chain 0
            output  reg   signed     [9:0]                 InBd20PowqdBmAnt1, //In Band Power 20P in qdBm for Rx Chain 1
            output  reg   signed     [9:0]                 InBd20PowqdBmAnt2, //In Band Power 20P in qdBm for Rx Chain 2
            output  reg   signed     [9:0]                 InBd20PowqdBmAnt3, //In Band Power 20P in qdBm for Rx Chain 3
            output  reg                                    InBd20PowqdBmUpdate, //In Band Power 20P valid

            output  reg   signed     [9:0]                 InBd20SecPowqdBmAnt, //In Band Power 20S in qdBm

            output  reg   signed     [9:0]                 InBd40SecPowqdBmAnt, //In Band Power 40S in qdBm
            
            output  reg   signed     [8:0]                 ADCPowqdBVAnt0, //ADC Power Estimate in qdBVrms for Rx Chain 0
            output  reg   signed     [8:0]                 ADCPowqdBVAnt1, //ADC Power Estimate in qdBVrms for Rx Chain 1
            output  reg   signed     [8:0]                 ADCPowqdBVAnt2, //ADC Power Estimate in qdBVrms for Rx Chain 2
            output  reg   signed     [8:0]                 ADCPowqdBVAnt3, //ADC Power Estimate in qdBVrms for Rx Chain 3

            output  reg   signed     [9:0]                 InBd20PowNoiseqdBmAnt0, //In Band Power 20P in qdBm for Rx Chain 0 for noise floor
            output  reg   signed     [9:0]                 InBd20PowNoiseqdBmAnt1, //In Band Power 20P in qdBm for Rx Chain 1 for noise floor
            output  reg   signed     [9:0]                 InBd20PowNoiseqdBmAnt2, //In Band Power 20P in qdBm for Rx Chain 2 for noise floor
            output  reg   signed     [9:0]                 InBd20PowNoiseqdBmAnt3, //In Band Power 20P in qdBm for Rx Chain 3 for noise floor
            output  reg                                    InBd20PowNoiseqdBmUpdate, //In Band Power Noise 20P valid

            output  reg   signed     [9:0]                 InBd20SecPowNoiseqdBmAnt, //In Band Power 20S in qdBm for noise floor

            output  reg   signed     [9:0]                 InBd40SecPowNoiseqdBmAnt, //In Band Power 40S in qdBm for noise floor
            
            //Interface with Energy Comparator
            output  reg   signed     [DBMPOW_WIDTH-1:0]    IdThrqdBmRx0, //In Band Power Threshold in qdBm for Rx Chain 0
            output  reg   signed     [DBMPOW_WIDTH-1:0]    IdThrqdBmRx1, //In Band Power Threshold in qdBm for Rx Chain 1
            output  reg   signed     [DBMPOW_WIDTH-1:0]    IdThrqdBmRx2, //In Band Power Threshold in qdBm for Rx Chain 2
            output  reg   signed     [DBMPOW_WIDTH-1:0]    IdThrqdBmRx3  //In Band Power Threshold in qdBm for Rx Chain 3
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam    signed   [DBMPOW_WIDTH-1:0] SIG_DBMPOW_WIDTH_PARAM_0 = {{DBMPOW_WIDTH}{1'b0}};
localparam    signed   [8:0] SIG_9_PARAM_0  = 9'b0;
localparam    signed   [9:0] SIG_10_PARAM_0 = 10'b0;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Wires Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [2:0]               InBd20SWLConfigInt;
reg                [2:0]               InBd20SecSWLConfigInt;
reg                [2:0]               InBd40SecSWLConfigInt;
reg                [1:0]               ADCSWLSatConfigInt;
reg                [1:0]               ADCSWLConfigInt;
reg                [1:0]               OFDMACModeSelInt;
reg                                    DSSSACModeSelInt;
reg                [1:0]               DCModeInt;
reg                [1:0]               AntSelDSSSInt;
reg                [1:0]               AntSelBWEstInt;
reg                [2:0]               DSSSContCorrModeSelInt;
reg                [6:0]               DSSSContCCThrInt;
reg                [6:0]               AGCParamSamp;
reg                [3:0]               IQEstGainAccEnInt;
reg                [3:0]               IQEstPhaseAccEnInt;
// Power max/min decoding
wire                                   PowForceMode;
wire                                   PowMaxMode;
wire                                   PowMinMode;
wire                                   ADCPowMode;
wire                                   InbandPowMode;
// For lint purpose
wire               [23:0]              AGCParam_aux;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Some DSP blocks require dynamic parameters, which need to be changed over
//the course of the AGC sequence. These parameters are stored in the
//program memory, together with the state description. They are loaded into
//the registers in this module, using specific commands.

always @ (posedge AGCClk or negedge nAGCRst)
   begin: RegInt_Blk
      if (nAGCRst == 1'b0) begin
         InBd20SWLConfigInt     <= 3'd4;
         InBd20SecSWLConfigInt  <= 3'd4;
         InBd40SecSWLConfigInt  <= 3'd4;
         ADCSWLSatConfigInt     <= 2'd0;
         ADCSWLConfigInt        <= 2'd2;
         OFDMACModeSelInt       <= 2'd0;
         DSSSACModeSelInt       <= 1'b1;
         DCModeInt              <= 2'd1;
         AntSelDSSSInt          <= 2'd0;
         AntSelBWEstInt         <= 2'd0;
         DSSSContCorrModeSelInt <= 3'd1;
         DSSSContCCThrInt       <= 7'd32;
         IdThrqdBmRx0           <= SIG_DBMPOW_WIDTH_PARAM_0;
         IdThrqdBmRx1           <= SIG_DBMPOW_WIDTH_PARAM_0;
         IdThrqdBmRx2           <= SIG_DBMPOW_WIDTH_PARAM_0;
         IdThrqdBmRx3           <= SIG_DBMPOW_WIDTH_PARAM_0;
      end
      else if ((AGCCommand == 8'd1 || AGCCommand == 8'd60) && AGCCmdValid) begin
         //Reset parameters to default values
         InBd20SWLConfigInt     <= 3'd4;
         InBd20SecSWLConfigInt  <= 3'd4;
         InBd40SecSWLConfigInt  <= 3'd4;
         ADCSWLSatConfigInt     <= 2'd0;
         ADCSWLConfigInt        <= 2'd2;
         OFDMACModeSelInt       <= 2'd0;
         DSSSACModeSelInt       <= 1'b1;
         DCModeInt              <= 2'd1;
         AntSelDSSSInt          <= 2'd0;
         AntSelBWEstInt         <= 2'd0;
         DSSSContCorrModeSelInt <= 3'd1;
         DSSSContCCThrInt       <= 7'd32;
      end
      else if (AGCCommand == 8'd61 && AGCCmdValid) begin
         //Set InBand Power Estimation Window Size (for 20 MHz primary)
         InBd20SWLConfigInt   <= AGCParam[2:0];
         //Set InBand Power Estimation Window Size (for 20 MHz secondary)
         InBd20SecSWLConfigInt <= AGCParam[10:8];
         //Set InBand Power Estimation Window Size (for 40 MHz secondary)
         InBd40SecSWLConfigInt <= AGCParam[18:16];
      end
      else if (AGCCommand == 8'd62 && AGCCmdValid) begin
         //Set ADC Power Estimator Size
         ADCSWLSatConfigInt <= AGCParam[1:0];
         ADCSWLConfigInt    <= AGCParam[9:8];
      end
      else if (AGCCommand == 8'd63 && AGCCmdValid) begin
         //Set OFDM AC Mode and DSSS AC Mode Select Register
         OFDMACModeSelInt   <= AGCParam[1:0];
         DSSSACModeSelInt   <= AGCParam[8];
      end
      else if (AGCCommand == 8'd64 && AGCCmdValid) begin
         //Select DC estimation mode
         DCModeInt          <= AGCParam[1:0];
      end
      else if (AGCCommand == 8'd66 && AGCCmdValid) begin
         //Store InBand power for Energy Comparator
         IdThrqdBmRx0 <= InBd20PowdBmRx0;
         IdThrqdBmRx1 <= InBd20PowdBmRx1;
         IdThrqdBmRx2 <= InBd20PowdBmRx2;
         IdThrqdBmRx3 <= InBd20PowdBmRx3;
      end
      else if (AGCCommand == 8'd67 && AGCCmdValid) begin
         //Set antenna selection for DSSS-CCK
         if (AGCParam[0] == 1'b1) begin
           if (PowForceMode) // Force mode
             AntSelDSSSInt <= AGCParam[17:16];
           else if (InbandPowMode) begin // Inband pow mode
             if (PowMaxMode) // Max mode
               AntSelDSSSInt <= InbandPowMaxIndex;
             else if (PowMinMode) // Min mode
               AntSelDSSSInt <= InbandPowMinIndex;
           end
           else if (ADCPowMode) begin // ADC pow mode
             if (PowMaxMode) // Max mode
               AntSelDSSSInt <= ADCPowdBVMaxIndex;
             else if (PowMinMode) // Min mode
               AntSelDSSSInt <= ADCPowdBVMinIndex;
           end
         end
         //Set antenna selection for BW estimator
         if (AGCParam[1] == 1'b1) begin
           if (PowForceMode) // Force mode
             AntSelBWEstInt <= AGCParam[17:16];
           else if (InbandPowMode) begin // Inband pow mode
             if (PowMaxMode) // Max mode
               AntSelBWEstInt <= InbandPowMaxIndex;
             else if (PowMinMode) // Min mode
               AntSelBWEstInt <= InbandPowMinIndex;
           end
           else if (ADCPowMode) begin // ADC pow mode
             if (PowMaxMode) // Max mode
               AntSelBWEstInt <= ADCPowdBVMaxIndex;
             else if (PowMinMode) // Min mode
               AntSelBWEstInt <= ADCPowdBVMinIndex;
           end
         end
      end
      else if (AGCCommand == 8'd68 && AGCCmdValid == 1'b1) begin
         //Set DSSS Continuous CC parameters
         DSSSContCorrModeSelInt <= AGCParam[2:0];
         DSSSContCCThrInt       <= AGCParam[14:8];
      end
   end //RegInt_Blk

//Send command to IQ estimation
always @ (posedge AGCClk or negedge nAGCRst)
   begin: IQEstCtrlInt_Blk
      if (nAGCRst == 1'b0) begin
         IQEstGainAccEnInt  <= 4'b0;
         IQEstPhaseAccEnInt <= 4'b0;
         IQEstGainAccEn     <= 1'b0;
         IQEstPhaseAccEn    <= 1'b0;
      end
      else begin
         // Enlarge pulse to be captured by FE clock worst case @40MHz
         IQEstGainAccEn          <= |IQEstGainAccEnInt;
         IQEstPhaseAccEn         <= |IQEstPhaseAccEnInt;
         // Delay line
         IQEstGainAccEnInt[3:1]  <= IQEstGainAccEnInt[2:0];
         IQEstPhaseAccEnInt[3:1] <= IQEstPhaseAccEnInt[2:0];
         // Pulse generation
         IQEstGainAccEnInt[0]    <= 1'b0;
         IQEstPhaseAccEnInt[0]   <= 1'b0;
         if (AGCCommand == 8'd100 && AGCCmdValid == 1'b1) begin
            //Set DSSS Continuous CC parameters
            IQEstGainAccEnInt[0]  <= AGCParam[0];
            IQEstPhaseAccEnInt[0] <= AGCParam[8];
         end
      end
   end //IQEstCtrlInt_Blk

//Store InBand and ADC powers into status registers from AGC or register command
always @ (posedge AGCClk or negedge nAGCRst)
   begin: InbdRegInt_Blk
      if (nAGCRst == 1'b0) begin
         InBd20PowqdBmAnt0        <= SIG_10_PARAM_0;
         InBd20PowqdBmAnt1        <= SIG_10_PARAM_0;
         InBd20PowqdBmAnt2        <= SIG_10_PARAM_0;
         InBd20PowqdBmAnt3        <= SIG_10_PARAM_0;
         InBd20PowqdBmUpdate      <= 1'b0;
         InBd20SecPowqdBmAnt      <= SIG_10_PARAM_0;
         InBd40SecPowqdBmAnt      <= SIG_10_PARAM_0;
         InBd20PowNoiseqdBmAnt0   <= SIG_10_PARAM_0;
         InBd20PowNoiseqdBmAnt1   <= SIG_10_PARAM_0;
         InBd20PowNoiseqdBmAnt2   <= SIG_10_PARAM_0;
         InBd20PowNoiseqdBmAnt3   <= SIG_10_PARAM_0;
         InBd20PowNoiseqdBmUpdate <= 1'b0;
         InBd20SecPowNoiseqdBmAnt <= SIG_10_PARAM_0;
         InBd40SecPowNoiseqdBmAnt <= SIG_10_PARAM_0;
         ADCPowqdBVAnt0           <= SIG_9_PARAM_0;
         ADCPowqdBVAnt1           <= SIG_9_PARAM_0;
         ADCPowqdBVAnt2           <= SIG_9_PARAM_0;
         ADCPowqdBVAnt3           <= SIG_9_PARAM_0;
         StatusTrigReq            <= 1'b0;
         AGCParamSamp             <= 7'b0;
      end
      else begin
        // Send status register trigger
        if (AGCCommand == 8'd65 && AGCCmdValid)
          StatusTrigReq <= 1'b1;
        else if (StatusTrigConf)
          StatusTrigReq <= 1'b0;
        
        // AGCParam sampling
        if (AGCCommand == 8'd65 && AGCCmdValid)
          AGCParamSamp <= {AGCParam[10:8], AGCParam[3:0]};
        
        // For Inband power
        if ((StatusTrigConf && StatusTrigReq && AGCParamSamp[0]) || InbdPowStartCaptureP) begin
           InBd20PowqdBmAnt0   <= $signed(InBd20PowdBmRx0[9:0]);
           InBd20PowqdBmAnt1   <= $signed(InBd20PowdBmRx1[9:0]);
           InBd20PowqdBmAnt2   <= $signed(InBd20PowdBmRx2[9:0]);
           InBd20PowqdBmAnt3   <= $signed(InBd20PowdBmRx3[9:0]);
        end
        
        // Send InBd20PowqdBmUpdate and InBd20PowNoiseqdBmUpdate on command 65 and clear on AGC reset command or AGC unlock
        if (StatusTrigConf && StatusTrigReq) begin
           if (AGCParamSamp[0])
              InBd20PowqdBmUpdate      <= 1'b1;
           if (AGCParamSamp[4])
              InBd20PowNoiseqdBmUpdate <= 1'b1;
        end
        else if ((AGCCommand == 8'd1) && (AGCCmdValid == 1'b1)) begin
           InBd20PowqdBmUpdate      <= 1'b0;
           InBd20PowNoiseqdBmUpdate <= 1'b0;
        end
        else if (!AGCOFDMLock && !AGCDSSSLock) begin
           InBd20PowqdBmUpdate      <= 1'b0;
        end
        
        if ((StatusTrigConf && StatusTrigReq && AGCParamSamp[2]) || InbdPowStartCaptureP) begin
           InBd20SecPowqdBmAnt <= $signed(InBd20SecPowdBm[9:0]);
        end
        if ((StatusTrigConf && StatusTrigReq && AGCParamSamp[3]) || InbdPowStartCaptureP) begin
           InBd40SecPowqdBmAnt <= $signed(InBd40SecPowdBm[9:0]);
        end

        //Store ADC powers into status registers
        if (StatusTrigConf && StatusTrigReq && AGCParamSamp[1]) begin
           ADCPowqdBVAnt0 <= $signed(ADCPowdBVRx0[8:0]);
           ADCPowqdBVAnt1 <= $signed(ADCPowdBVRx1[8:0]);
           ADCPowqdBVAnt2 <= $signed(ADCPowdBVRx2[8:0]);
           ADCPowqdBVAnt3 <= $signed(ADCPowdBVRx3[8:0]);
        end
        
        // For noise floor
        if (StatusTrigConf && StatusTrigReq && AGCParamSamp[4]) begin
           InBd20PowNoiseqdBmAnt0 <= $signed(InBd20PowdBmRx0[9:0]);
           InBd20PowNoiseqdBmAnt1 <= $signed(InBd20PowdBmRx1[9:0]);
           InBd20PowNoiseqdBmAnt2 <= $signed(InBd20PowdBmRx2[9:0]);
           InBd20PowNoiseqdBmAnt3 <= $signed(InBd20PowdBmRx3[9:0]);
        end
        if (StatusTrigConf && StatusTrigReq && AGCParamSamp[5]) begin
           InBd20SecPowNoiseqdBmAnt <= $signed(InBd20SecPowdBm[9:0]);
        end
        if (StatusTrigConf && StatusTrigReq && AGCParamSamp[6]) begin
           InBd40SecPowNoiseqdBmAnt <= $signed(InBd40SecPowdBm[9:0]);
        end
      end
   end //InbdRegInt_Blk

// Power mode decoding
assign PowForceMode  = AGCParam[8];
assign PowMaxMode    = (AGCParam[10:9] == 2'b01) ? 1'b1 : 1'b0;
assign PowMinMode    = (AGCParam[10:9] == 2'b10) ? 1'b1 : 1'b0;
assign ADCPowMode    = (AGCParam[17:16] == 2'b10) ? 1'b1 : 1'b0;
assign InbandPowMode = (AGCParam[17:16] == 2'b01) ? 1'b1 : 1'b0;


//
assign AGCParam_aux = AGCParam; // To Avoid Lint Warning
//

//Outputs
assign InBd20SWLConfig     = InBd20SWLConfigInt;
assign InBd20SecSWLConfig  = InBd20SecSWLConfigInt;
assign InBd40SecSWLConfig  = InBd40SecSWLConfigInt;
assign ADCSWLSatConfig     = ADCSWLSatConfigInt;
assign ADCSWLConfig        = ADCSWLConfigInt;
assign OFDMACModeSel       = OFDMACModeSelInt;
assign DSSSACModeSel       = DSSSACModeSelInt;
assign DCMode              = DCModeInt;
assign AntSelDSSS          = AntSelDSSSInt;
assign AntSelBWEst         = AntSelBWEstInt;
assign DSSSContCorrModeSel = DSSSContCorrModeSelInt;
assign DSSSContCCThr       = DSSSContCCThrInt;

endmodule //DSPDynConfig

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
