////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Digital Gain Conversion Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/DigGainConv.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module DigGainConv (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  AGCClk,    //AGC Clock
            input    wire                                  nAGCRst,   //Active Low Reset

            //Digital Gain
            input    wire   signed     [6:0]               DigGaindBIn, //Gain in dB

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Digital Gain to Front End
            output   wire      [3:0]                       DigGain6dB, //6 dB Factor
            output   reg       [5:0]                       DigGainLin //Linear Factor
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam    signed   [6:0]  SIG_7_PARAM_0    = 7'b0;
localparam    signed   [8:0]  SIG_9_PARAM_0    = 9'b0;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [6:0]               DigGain6dBInt1;
wire               [4:0]               DigGain6dBInt2;
wire               [7:0]               DigGain6dBInt5;
wire    signed     [10:0]              DigGainLinInt2;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [4:0]               DigGain6dBInt3;
reg                [3:0]               DigGain6dBInt4;
reg     signed     [8:0]               DigGainLinInt1;
reg     signed     [6:0]               DigGaindBIn1D;
reg     signed     [6:0]               DigGaindBIn2D;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module converts the dB value of Digital Gain in to a combination of a
//6 dB factor and the remainder as a linear factor.

//Truncate the input gain by 1 bit and add 3 to it
assign DigGain6dBInt1 = (DigGaindBIn >>> 7'sd1) + 7'sd3;

//Saturate to 5 bits
  SatUnsigned #(
                .INPUT_WIDTH(7),
                .OUTPUT_WIDTH(5)
              )
              U_SatUnsigned (
                .InputData(DigGain6dBInt1),
                .SatData(DigGain6dBInt2)
              );

//Register saturation result
//If DigGain6dBInt1 is negative, force value to 0
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Reg1_Blk
      if (nAGCRst == 1'b0)
         DigGain6dBInt3 <= 5'd0;
      else if (DigGain6dBInt1[6] == 1'b1)
         DigGain6dBInt3 <= 5'd0;
      else
         DigGain6dBInt3 <= DigGain6dBInt2;

   end //Reg1_Blk

//Use this as the index of an LUT
always @ (posedge AGCClk or negedge nAGCRst)
   begin: LUT1_Blk
      if (nAGCRst == 1'b0) begin
         DigGain6dBInt4 <= 4'd1;
      end
      else begin
         case (DigGain6dBInt3)
            5'd00, 5'd01, 5'd02: DigGain6dBInt4 <= 4'd00;
            5'd03, 5'd04, 5'd05: DigGain6dBInt4 <= 4'd01;
            5'd06, 5'd07, 5'd08: DigGain6dBInt4 <= 4'd02;
            5'd09, 5'd10, 5'd11: DigGain6dBInt4 <= 4'd03;
            5'd12, 5'd13, 5'd14: DigGain6dBInt4 <= 4'd04;
            5'd15, 5'd16, 5'd17: DigGain6dBInt4 <= 4'd05;
            5'd18, 5'd19, 5'd20: DigGain6dBInt4 <= 4'd06;
            5'd21, 5'd22, 5'd23: DigGain6dBInt4 <= 4'd07;
            5'd24, 5'd25, 5'd26: DigGain6dBInt4 <= 4'd08;
            5'd27, 5'd28, 5'd29: DigGain6dBInt4 <= 4'd09;
            default            : DigGain6dBInt4 <= 4'd10;
         endcase //DigGain6dBInt3
      end

   end //LUT1_Blk

//Assign this value to the output
assign DigGain6dB = DigGain6dBInt4;

//Multiply this by 6
assign DigGain6dBInt5 = {2'b0,DigGain6dBInt4,2'b0} + {3'b0,DigGain6dBInt4,1'b0};

//Subtract this from a delayed version of the input
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Reg2_Blk
      if (nAGCRst == 1'b0)
         DigGainLinInt1 <= SIG_9_PARAM_0;
      else
         DigGainLinInt1 <= $signed({{2{DigGaindBIn2D[6]}},DigGaindBIn2D}) - $signed({1'b0, DigGain6dBInt5});

   end //Reg2_Blk

//Multiply this by 4 and add 24 to it  
assign DigGainLinInt2 = {DigGainLinInt1,2'b0} + 11'sd24;

//Use this as the index of a LUT
always @ (posedge AGCClk or negedge nAGCRst)
   begin: LUT2_Blk
      if (nAGCRst == 1'b0) begin
         DigGainLin <= 6'd32;
      end
      else begin
         case (DigGainLinInt2)
            11'd00:  DigGainLin <= 6'd32;
            11'd01:  DigGainLin <= 6'd33;
            11'd02:  DigGainLin <= 6'd33;
            11'd03:  DigGainLin <= 6'd34;
            11'd04:  DigGainLin <= 6'd35;
            11'd05:  DigGainLin <= 6'd37;
            11'd06:  DigGainLin <= 6'd38;
            11'd07:  DigGainLin <= 6'd39;
            11'd08:  DigGainLin <= 6'd40;
            11'd09:  DigGainLin <= 6'd41;
            11'd10:  DigGainLin <= 6'd42;
            11'd11:  DigGainLin <= 6'd44;
            11'd12:  DigGainLin <= 6'd45;
            11'd13:  DigGainLin <= 6'd46;
            11'd14:  DigGainLin <= 6'd47;
            11'd15:  DigGainLin <= 6'd49;
            11'd16:  DigGainLin <= 6'd50;
            11'd17:  DigGainLin <= 6'd52;
            11'd18:  DigGainLin <= 6'd53;
            11'd19:  DigGainLin <= 6'd55;
            11'd20:  DigGainLin <= 6'd57;
            11'd21:  DigGainLin <= 6'd58;
            11'd22:  DigGainLin <= 6'd60;
            11'd23:  DigGainLin <= 6'd62;
            default: DigGainLin <= 6'd32;
         endcase //DigGainLinInt2
      end

   end //LUT2_Blk

//Generate a 2 clock delayed version of DigGaindBIn
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Reg3_Blk
      if (nAGCRst == 1'b0) begin
         DigGaindBIn1D <= SIG_7_PARAM_0;
         DigGaindBIn2D <= SIG_7_PARAM_0;
      end
      else begin
         DigGaindBIn1D <= DigGaindBIn;
         DigGaindBIn2D <= DigGaindBIn1D;
      end

   end //Reg3_Blk

endmodule //DigGainConv

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
