////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 14729 $
// $Date: 2014-04-29 09:35:19 +0200 (Tue, 29 Apr 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Event Decoder Top Level
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/EventDecTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module EventDecTop #(parameter NRX = 4 //Number of Rx Chains (1 to 4)
                    )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input wire                             AGCClk,       // AGC Clock
            input wire                             nAGCRst,      // Active Low Reset

            //Status Register
            input wire         [127:0]             AGCStatusReg, //Status Register

            //Config Registers
            input wire         [3:0]               ChCombMode, //Combining Rx Chains
            input wire         [31:0]              FSMEvent0Reg, //Event 0 Register
            input wire         [31:0]              FSMEvent1Reg, //Event 1 Register
            input wire         [31:0]              FSMEvent2Reg, //Event 2 Register
            input wire         [31:0]              FSMEvent3Reg, //Event 3 Register

            //AGC FSM event mask
            input wire                             AGCMaskEvent, //Event mask from FSM

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //AGC FSM Control
            output wire                            EventValid, //Event occurence to FSM
            output reg       [8:0]                 EventTargetAddr, //Event Address
            //CCA Control
            output wire                            EventSat, // Saturation Event occurence to CCA
            output wire                            EventDis  // Disappearance Event occurence to CCA
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Registers Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [21:0]              Event0CondReg;
wire               [21:0]              Event1CondReg;
wire               [21:0]              Event2CondReg;
wire               [21:0]              Event3CondReg;
wire               [8:0]               Event0TargetAddr;
wire               [8:0]               Event1TargetAddr;
wire               [8:0]               Event2TargetAddr;
wire               [8:0]               Event3TargetAddr;
wire                                   Trans0Valid;
wire                                   Trans1Valid;
wire                                   Trans2Valid;
wire                                   Trans3Valid;
wire                                   Event0Valid;
wire                                   Event1Valid;
wire                                   Event2Valid;
wire                                   Event3Valid;

reg                                    EventOnGoing;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module instantiates 4 event decoders and a priority encoder. It
//indicates to the FSM Core when an even occurs. It also picks out the
//corresponding event target address and passes it to the FSM Core.

//Event is on-going
always @ (posedge AGCClk or negedge nAGCRst)
   begin: EventOnGoing_Blk
      if (nAGCRst == 1'b0)
         EventOnGoing <= 1'b0;
      else if (EventValid == 1'b1)
         EventOnGoing <= 1'b1;
      else if (AGCMaskEvent == 1'b0)
         EventOnGoing <= 1'b0;
   end // EventOnGoing_Blk

//Split FSMEventReg into Event Condition Register and Target Address
assign Event0CondReg = FSMEvent0Reg[31:10];
assign Event1CondReg = FSMEvent1Reg[31:10];
assign Event2CondReg = FSMEvent2Reg[31:10];
assign Event3CondReg = FSMEvent3Reg[31:10];

assign Event0TargetAddr = FSMEvent0Reg[8:0];
assign Event1TargetAddr = FSMEvent1Reg[8:0];
assign Event2TargetAddr = FSMEvent2Reg[8:0];
assign Event3TargetAddr = FSMEvent3Reg[8:0];

//Transition Decoder modules are used to detect occurence of events
//Instantiate 4 TransDec modules
TransDec # (
            //Parameters
            .NRX(NRX)
            ) U_EventDec0 (
                           //Inputs
                           .TransCondReg(Event0CondReg),
                           .AGCStatusReg(AGCStatusReg),
                           .ChCombMode(ChCombMode),
                           //Output
                           .TransValid(Trans0Valid),
                           .DbgAgc()
                           );

TransDec # (
            //Parameters
            .NRX(NRX)
            ) U_EventDec1 (
                           //Inputs
                           .TransCondReg(Event1CondReg),
                           .AGCStatusReg(AGCStatusReg),
                           .ChCombMode(ChCombMode),

                           //Output
                           .TransValid(Trans1Valid),
                           .DbgAgc()
                           );

TransDec # (
            //Parameters
            .NRX(NRX)
            ) U_EventDec2 (
                           //Inputs
                           .TransCondReg(Event2CondReg),
                           .AGCStatusReg(AGCStatusReg),
                           .ChCombMode(ChCombMode),

                           //Output
                           .TransValid(Trans2Valid),
                           .DbgAgc()
                           );

TransDec # (
            //Parameters
            .NRX(NRX)
            ) U_EventDec3 (
                           //Inputs
                           .TransCondReg(Event3CondReg),
                           .AGCStatusReg(AGCStatusReg),
                           .ChCombMode(ChCombMode),

                           //Output
                           .TransValid(Trans3Valid),
                           .DbgAgc()
                           );

//Generate the 4 EventValid
assign Event0Valid = Trans0Valid & ~EventOnGoing;
assign Event1Valid = Trans1Valid & ~EventOnGoing;
assign Event2Valid = Trans2Valid & ~EventOnGoing;
assign Event3Valid = Trans3Valid & ~EventOnGoing;

//Generate EventValid Output
assign EventValid = Event0Valid | Event1Valid | Event2Valid | Event3Valid;

//In order to select the EventTargetAddr, a priority encoder is implemented.
//Event0Valid is given highest priority, Event1Valid is given second priority
//and so on. This will avoid confusion when multiple events occur
//simultaneously.
always @ (*)
   begin: EventTargetAddr_Blk
      if (Event0Valid == 1'b1)
         EventTargetAddr = Event0TargetAddr;
      else if (Event1Valid == 1'b1)
         EventTargetAddr = Event1TargetAddr;
      else if (Event2Valid == 1'b1)
         EventTargetAddr = Event2TargetAddr;
      else if (Event3Valid == 1'b1)
         EventTargetAddr = Event3TargetAddr;
      else
         EventTargetAddr = 9'b0;

   end //EventTargetAddr_Blk

//Generate Event Output
assign EventSat = Event0Valid;
assign EventDis = Event2Valid;

endmodule //EventDecTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
