////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 28412 $
// $Date: 2016-10-04 15:49:09 +0200 (Tue, 04 Oct 2016) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Noise Variance and SNR Estimation Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/MDMCOMMON/RIUCORE/AGC/AGCFSM/verilog/rtl/NoiseVarEst.v $
// 
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

module NoiseVarEst #(parameter NRX          =  4, //Number of Rx Chains (1 to 4)
                     parameter DBMPOW_WIDTH = 13  //Data width of dBm power
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst, //Active LOW Reset
            input    wire                                  AGCClk,  //AGC Clock

            //Registers
            input    wire                                  SNRMeasSel,        //SNR Min or Max Selection

            //Rx gain
            input    wire   signed    [7:0]                RxGain0, //Gain for Rx Chain 0
            input    wire   signed    [7:0]                RxGain1, //Gain for Rx Chain 1
            input    wire   signed    [7:0]                RxGain2, //Gain for Rx Chain 2
            input    wire   signed    [7:0]                RxGain3, //Gain for Rx Chain 3

            //RF noise factor
            input    wire             [5:0]                RFGainNF0dB, //Noise Factor for Rx Chain 0
            input    wire             [5:0]                RFGainNF1dB, //Noise Factor for Rx Chain 1
            input    wire             [5:0]                RFGainNF2dB, //Noise Factor for Rx Chain 2
            input    wire             [5:0]                RFGainNF3dB, //Noise Factor for Rx Chain 3

            //Controls
            input    wire                                  SNREstStrtP, //Start Estimation

            //InBand Power Estimates in dBm
            input    wire   signed    [DBMPOW_WIDTH-1:0]   InBd20PowdBmRx0, //Rx Chain 0
            input    wire   signed    [DBMPOW_WIDTH-1:0]   InBd20PowdBmRx1, //Rx Chain 1
            input    wire   signed    [DBMPOW_WIDTH-1:0]   InBd20PowdBmRx2, //Rx Chain 2
            input    wire   signed    [DBMPOW_WIDTH-1:0]   InBd20PowdBmRx3, //Rx Chain 3

            //Digital Gain
            input    wire   signed    [6:0]                DigGainRx0, //For Rx Chain 0
            input    wire   signed    [6:0]                DigGainRx1, //For Rx Chain 1
            input    wire   signed    [6:0]                DigGainRx2, //For Rx Chain 2
            input    wire   signed    [6:0]                DigGainRx3, //For Rx Chain 3

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output  reg     signed    [7:0]                NoiseVardB,       // Noise variance in dB
            output  reg                                    NoiseVardBValid,  // Qualifies NoiseVardBValid
            output  reg               [7:0]                SNR,              // SNR
            output  reg                                    SNRValid          // Qualifies SNR
            );

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [7:0]               SNRSatRx0;
wire    signed     [7:0]               SNRSatRx1;
wire    signed     [7:0]               SNRSatRx2;
wire    signed     [7:0]               SNRSatRx3;
wire    signed     [DBMPOW_WIDTH-2:0]  SNRRndRx0;
wire    signed     [DBMPOW_WIDTH-2:0]  SNRRndRx1;
wire    signed     [DBMPOW_WIDTH-2:0]  SNRRndRx2;
wire    signed     [DBMPOW_WIDTH-2:0]  SNRRndRx3;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [DBMPOW_WIDTH:0]    SNRRegRx0;
reg     signed     [DBMPOW_WIDTH:0]    SNRRegRx1;
reg     signed     [DBMPOW_WIDTH:0]    SNRRegRx2;
reg     signed     [DBMPOW_WIDTH:0]    SNRRegRx3;
reg     signed     [7:0]               SNRSatRegRx0;
reg     signed     [7:0]               SNRSatRegRx1;
reg     signed     [7:0]               SNRSatRegRx2;
reg     signed     [7:0]               SNRSatRegRx3;
reg     signed     [7:0]               SNRMinInt0;
reg     signed     [7:0]               SNRMinInt1;
reg     signed     [7:0]               SNRMin;
reg     signed     [7:0]               SNRMaxInt0;
reg     signed     [7:0]               SNRMaxInt1;
reg     signed     [7:0]               SNRMax;
reg                [2:0]               DelayCnt;
reg     signed     [7:0]               NoiseVardBVRx0;
reg     signed     [7:0]               NoiseVardBVRx1;
reg     signed     [7:0]               NoiseVardBVRx2;
reg     signed     [7:0]               NoiseVardBVRx3;
reg     signed     [7:0]               NoiseVarMinInt0;
reg     signed     [7:0]               NoiseVarMinInt1;
reg     signed     [7:0]               NoiseVarMin;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module computes the Noise Variance in dB and in linear ,
// as well as the SNR of the current packet.

////////////////////////////////////
//SNR calculation
////////////////////////////////////

//Register the sum values
// 
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SNRReg_Blk
      if (nAGCRst == 1'b0) begin
         SNRRegRx0 <= $signed({{(DBMPOW_WIDTH+1)}{1'b0}});
         SNRRegRx1 <= $signed({{(DBMPOW_WIDTH+1)}{1'b0}});
         SNRRegRx2 <= $signed({{(DBMPOW_WIDTH+1)}{1'b0}});
         SNRRegRx3 <= $signed({{(DBMPOW_WIDTH+1)}{1'b0}});
      end
      else begin
         SNRRegRx0 <= InBd20PowdBmRx0 + 13'sd404 - $signed({5'b0,RFGainNF0dB,2'b0});
         SNRRegRx1 <= InBd20PowdBmRx1 + 13'sd404 - $signed({5'b0,RFGainNF1dB,2'b0});
         SNRRegRx2 <= InBd20PowdBmRx2 + 13'sd404 - $signed({5'b0,RFGainNF2dB,2'b0});
         SNRRegRx3 <= InBd20PowdBmRx3 + 13'sd404 - $signed({5'b0,RFGainNF3dB,2'b0});
      end
   end //SNRReg_Blk

//Round by 2 bits
Round #(
        .INPUT_WIDTH(DBMPOW_WIDTH+1),
        .OUTPUT_WIDTH(DBMPOW_WIDTH-1)
       )
        U_ROUNDINT0(
                    .InputData(SNRRegRx0),
                    .RoundData(SNRRndRx0)
                   );

generate
   if (NRX > 32'd1) begin : NRX1_rndgen
      Round #(
              .INPUT_WIDTH(DBMPOW_WIDTH+1),
              .OUTPUT_WIDTH(DBMPOW_WIDTH-1)
             )
              U_ROUNDINT1(
                          .InputData(SNRRegRx1),
                          .RoundData(SNRRndRx1)
                         );
   end
   else begin: NRX1_norndgen
      assign SNRRndRx1 = $signed({{(DBMPOW_WIDTH-1)}{1'b0}});
   end //NRX > 1

   if (NRX > 32'd2) begin : NRX2_rndgen
      Round #(
              .INPUT_WIDTH(DBMPOW_WIDTH+1),
              .OUTPUT_WIDTH(DBMPOW_WIDTH-1)
             )
              U_ROUNDINT2(
                          .InputData(SNRRegRx2),
                          .RoundData(SNRRndRx2)
                         );
   end
   else begin: NRX2_norndgen
      assign SNRRndRx2 = $signed({{(DBMPOW_WIDTH-1)}{1'b0}});
   end //NRX > 2

   if (NRX > 32'd3) begin : NRX3_rndgen
      Round #(
              .INPUT_WIDTH(DBMPOW_WIDTH+1),
              .OUTPUT_WIDTH(DBMPOW_WIDTH-1)
             )
              U_ROUNDINT3(
                          .InputData(SNRRegRx3),
                          .RoundData(SNRRndRx3)
                         );
   end
   else begin: NRX3_norndgen
      assign SNRRndRx3 = $signed({{(DBMPOW_WIDTH-1)}{1'b0}});
   end //NRX > 3
endgenerate

//Saturate the SNR Values to 8 bits
SatSymSigned #(
               .INPUT_WIDTH(DBMPOW_WIDTH-1),
               .OUTPUT_WIDTH(8)
              )
               U_SATINT0(
                         .InputData(SNRRndRx0),
                         .SatSymData(SNRSatRx0)
                        );

generate
   if (NRX > 32'd1) begin : NRX1_satgen
      SatSymSigned #(
                     .INPUT_WIDTH(DBMPOW_WIDTH-1),
                     .OUTPUT_WIDTH(8)
                    )
                     U_SATINT1(
                               .InputData(SNRRndRx1),
                               .SatSymData(SNRSatRx1)
                              );
   end
   else begin: NRX1_nosatgen
      assign SNRSatRx1 = 8'sd0;
   end //NRX > 1

   if (NRX > 32'd2) begin: NRX2_satgen
      SatSymSigned #(
                     .INPUT_WIDTH(DBMPOW_WIDTH-1),
                     .OUTPUT_WIDTH(8)
                    )
                     U_SATINT2(
                               .InputData(SNRRndRx2),
                               .SatSymData(SNRSatRx2)
                              );
   end
   else begin: NRX2_nosatgen
      assign SNRSatRx2 = 8'sd0;
   end //NRX > 2

   if (NRX > 32'd3) begin: NRX3_satgen
      SatSymSigned #(
                     .INPUT_WIDTH(DBMPOW_WIDTH-1),
                     .OUTPUT_WIDTH(8)
                    )
                     U_SATINT3(
                               .InputData(SNRRndRx3),
                               .SatSymData(SNRSatRx3)
                              );
   end
   else begin: NRX3_nosatgen
      assign SNRSatRx3 = 8'sd0;
   end //NRX > 3
endgenerate

//Register the saturated value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SNRSatReg_Blk
      if (nAGCRst == 1'b0) begin
         SNRSatRegRx0 <= 8'sd0;
         SNRSatRegRx1 <= 8'sd0;
         SNRSatRegRx2 <= 8'sd0;
         SNRSatRegRx3 <= 8'sd0;
      end
      else begin
         SNRSatRegRx0 <= SNRSatRx0;
         SNRSatRegRx1 <= SNRSatRx1;
         SNRSatRegRx2 <= SNRSatRx2;
         SNRSatRegRx3 <= SNRSatRx3;
      end
   end //SNRSatReg_Blk

//Find the minimum and the maximum of the SNR values
//SNR Min
always @ (*)
   begin: SNRMinInt_Blk
      if (NRX == 32'd4) begin
         SNRMinInt0 = (SNRSatRegRx0 < SNRSatRegRx1) ?
                       SNRSatRegRx0 : SNRSatRegRx1;

         SNRMinInt1 = (SNRSatRegRx2 < SNRSatRegRx3) ?
                       SNRSatRegRx2 : SNRSatRegRx3;

         SNRMin     = (SNRMinInt0 < SNRMinInt1) ?
                       SNRMinInt0 : SNRMinInt1;
     end
     else if (NRX >= 32'd3) begin
         SNRMinInt0 = (SNRSatRegRx0 < SNRSatRegRx1) ?
                       SNRSatRegRx0 : SNRSatRegRx1;

         SNRMinInt1 = SNRSatRegRx2;

         SNRMin     = (SNRMinInt0 < SNRMinInt1) ?
                       SNRMinInt0 : SNRMinInt1;
     end 
     else if (NRX >= 32'd2) begin
         SNRMinInt0 = (SNRSatRegRx0 < SNRSatRegRx1) ?
                       SNRSatRegRx0 : SNRSatRegRx1;

         SNRMinInt1 = 8'sd0;

         SNRMin     = SNRMinInt0;
     end
     else begin
         SNRMinInt0 = SNRSatRegRx0;

         SNRMinInt1 = 8'sd0;

         SNRMin     = SNRMinInt0;
     end
  end //SNRMinInt_Blk

//SNR Max
always @ (*)
   begin: SNRMaxInt_Blk
      if (NRX == 32'd4) begin
         SNRMaxInt0 = (SNRSatRegRx0 > SNRSatRegRx1) ?
                       SNRSatRegRx0 : SNRSatRegRx1;

         SNRMaxInt1 = (SNRSatRegRx2 > SNRSatRegRx3) ?
                       SNRSatRegRx2 : SNRSatRegRx3;

         SNRMax     = (SNRMaxInt0 > SNRMaxInt1) ?
                       SNRMaxInt0 : SNRMaxInt1;
     end
     else if (NRX > 32'd2) begin
         SNRMaxInt0 = (SNRSatRegRx0 > SNRSatRegRx1) ?
                       SNRSatRegRx0 : SNRSatRegRx1;

         SNRMaxInt1 = SNRSatRegRx2;

         SNRMax     = (SNRMaxInt0 > SNRMaxInt1) ?
                       SNRMaxInt0 : SNRMaxInt1;
     end
     else if (NRX > 32'd1) begin
         SNRMaxInt0 = (SNRSatRegRx0 > SNRSatRegRx1) ?
                       SNRSatRegRx0 : SNRSatRegRx1;

         SNRMaxInt1 = 8'sd0;

         SNRMax     = SNRMaxInt0;
     end
     else begin
         SNRMaxInt0 = SNRSatRegRx0;

         SNRMaxInt1 = 8'sd0;

         SNRMax     = SNRMaxInt0;
     end
  end //SNRMaxInt_Blk

//Either the min or the max value of SNR is given out depending on the value
//of SNRMeasSel (0 = Min; 1 = Max)
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SelSNR_Blk
      if (nAGCRst == 1'b0)
         SNR <= 8'd50;
      else if (DelayCnt == 3'd3) begin
         if (SNRMeasSel == 1'b0)
            SNR <= SNRMin & {8{~SNRMin[7]}}; // Limit SNR to 0
         else
            SNR <= SNRMax & {8{~SNRMax[7]}}; // Limit SNR to 0
      end
   end //SelSNR_Blk

//SNRValid
//SNRValid will indicate that the SNR value is stable. It will go high after
//3 clocks from estimation start (with SNREstStrtP).
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SNRValid_Blk
      if (nAGCRst == 1'b0)
         SNRValid <= 1'b0;
      else if (SNREstStrtP == 1'b1)
         SNRValid <= 1'b0;
      else if (DelayCnt == 3'd3)
         SNRValid <= 1'b1;
   end //SNRValid_Blk

//Delay Counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelayCnt_Blk
      if (nAGCRst == 1'b0)
         DelayCnt <= 3'd0;
      else if (SNREstStrtP == 1'b1)
         DelayCnt <= 3'd1;
      else if (DelayCnt == 3'd5)
         DelayCnt <= 3'd0;
      else if (DelayCnt > 3'd0)
         DelayCnt <= DelayCnt + 3'd1;
   end //DelayCnt_Blk


///////////////////////////////////////////////
//Noise Variance dBVrms (depending on NRX)
///////////////////////////////////////////////

//Add RF gain + Digital Gain value to NF + constant (in dB)
always @ (posedge AGCClk or negedge nAGCRst)
   begin:NoiseVardBV_Blk
      if (nAGCRst == 1'b0) begin
         NoiseVardBVRx0 <= 8'sd0;
         NoiseVardBVRx1 <= 8'sd0;
         NoiseVardBVRx2 <= 8'sd0;
         NoiseVardBVRx3 <= 8'sd0;
      end
      else begin
         NoiseVardBVRx0 <= $signed({2'b0,RFGainNF0dB}) - 8'sd36 + RxGain0 + $signed({DigGainRx0[6],DigGainRx0});
         NoiseVardBVRx1 <= $signed({2'b0,RFGainNF1dB}) - 8'sd36 + RxGain1 + $signed({DigGainRx1[6],DigGainRx1});
         NoiseVardBVRx2 <= $signed({2'b0,RFGainNF2dB}) - 8'sd36 + RxGain2 + $signed({DigGainRx2[6],DigGainRx2});
         NoiseVardBVRx3 <= $signed({2'b0,RFGainNF3dB}) - 8'sd36 + RxGain3 + $signed({DigGainRx3[6],DigGainRx3});
      end
   end //NoiseVardBV_Blk

//Find the minimum of the Noise Variance values. This will be utilized for the
//conversion process from dB to linear
always @ (*)
   begin: NoiseVarMinInt_Blk
      if (NRX == 32'd4) begin
         NoiseVarMinInt0 = (NoiseVardBVRx0 < NoiseVardBVRx1) ?
                            NoiseVardBVRx0 : NoiseVardBVRx1;

         NoiseVarMinInt1 = (NoiseVardBVRx2 < NoiseVardBVRx3) ?
                            NoiseVardBVRx2 : NoiseVardBVRx3;

         NoiseVarMin     = (NoiseVarMinInt0 < NoiseVarMinInt1) ?
                            NoiseVarMinInt0 : NoiseVarMinInt1;
     end
     else if (NRX > 32'd2) begin
         NoiseVarMinInt0 = (NoiseVardBVRx0 < NoiseVardBVRx1) ?
                            NoiseVardBVRx0 : NoiseVardBVRx1;

         NoiseVarMinInt1 = NoiseVardBVRx2;

         NoiseVarMin     = (NoiseVarMinInt0 < NoiseVarMinInt1) ?
                            NoiseVarMinInt0 : NoiseVarMinInt1;
     end
     else if (NRX > 32'd1) begin
         NoiseVarMinInt0 = (NoiseVardBVRx0 < NoiseVardBVRx1) ?
                            NoiseVardBVRx0 : NoiseVardBVRx1;

         NoiseVarMinInt1 = 8'sd0;

         NoiseVarMin     = NoiseVarMinInt0;
     end
     else begin
         NoiseVarMinInt0 = NoiseVardBVRx0;

         NoiseVarMinInt1 = 8'sd0;

         NoiseVarMin     = NoiseVarMinInt0;
     end
  end //NoiseVarMinInt_Blk

//Register the minimum Noise Variance Value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: NoiseVardBReg_Blk
      if (nAGCRst == 1'b0)
         NoiseVardB <= 8'sd0;
      else if (DelayCnt == 3'd1)
         NoiseVardB <= NoiseVarMin;
   end //NoiseVardBReg_Blk

//NoiseVarValid will qualify NoiseVardB
always @ (posedge AGCClk or negedge nAGCRst)
   begin: NoiseVardBValid_Blk
      if (nAGCRst == 1'b0)
         NoiseVardBValid <= 1'b0;
      else if (SNREstStrtP == 1'b1)
         NoiseVardBValid <= 1'b0;
      else if (DelayCnt == 3'd1)
         NoiseVardBValid <= 1'b1;
   end //NoiseVardBValid_Blk

endmodule //NoiseVarEst

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
