////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Ramp-Up/Ramp-Down/Plateau Detection Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/PlatRampDetect.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module PlatRampDetect #(parameter DBMPOW_WIDTH = 13 //Data width of dBm power
                       )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,   //Active Low Reset
            input    wire                                  AGCClk,    //AGC Clock

            //Config Registers
            input    wire              [7:0]               RegDetPlat, //Plateau Detect Value
            input    wire              [2:0]               DelRampUpTap, //Delay line tap out
            input    wire              [2:0]               DelRampDownTap, //Delay line tap out
            input    wire              [2:0]               DelPlatTap, //Delay line tap out

            //Power & Valid
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBdPowdBm, //InBand Power in dBm
            input    wire                                  InBdPowValid, //Qualifies InBdPowdBm

            //Controls
            input    wire                                  RampUpDetEn,  //Ramp-up enable
            input    wire                                  RampDownDetEn,//Ramp-down enable
            input    wire                                  PlatDetEn,    //Plateau enable
            input    wire              [7:0]               AGCCommand,   //Command from FSM Core
            input    wire                                  AGCCmdValid,  //AGC Command Valid

            input    wire              [7:0]               DetThrRampUp,  //Ramp Up Threshold Value
            input    wire              [7:0]               DetThrRampDown,//Ramp Down Threshold Value

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   wire                                  RampUpDet, //Ramp Up Detected
            output   wire                                  RampDownDet, //Ramp Down Detected
                     
                     
            output   reg                                   PlateauDet, //Plateau Detected
            output   wire signed     [DBMPOW_WIDTH-1:0]    InBdPowdBmDL0, //InBand Power delay line 0 in dBm
            output   wire                                  InBdPowDL0Valid, //Qualifies InBdPowdBmDL0
            output   wire signed     [DBMPOW_WIDTH-1:0]    InBdPowdBmDL1, //InBand Power delay line 1 in dBm
            output   wire                                  InBdPowDL1Valid //Qualifies InBdPowdBmDL1
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed  [DBMPOW_WIDTH-1:0] CONST_DBMPOW_WIDTH_ZERO = {{DBMPOW_WIDTH}{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [DBMPOW_WIDTH:0]    PowDiffRampUp;
wire    signed     [DBMPOW_WIDTH:0]    PowDiffRampDown;
wire    signed     [DBMPOW_WIDTH:0]    PowDiffPlat;
wire    signed     [DBMPOW_WIDTH:0]    PowDiffPlatc2;
wire               [DBMPOW_WIDTH:0]    PowDiffPlatAbs;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                    DelLineClr;
reg     signed     [DBMPOW_WIDTH-1:0]  DelayLine[23:0];
reg                [3:0]               SampleCount;
reg                [23:0]              InBdPowValidDel;
reg     signed     [DBMPOW_WIDTH-1:0]  RampUpSel;
reg     signed     [DBMPOW_WIDTH-1:0]  RampDownSel;
reg     signed     [DBMPOW_WIDTH-1:0]  PlatSel;
reg                                    InBdPowValidRampUp;
reg                                    InBdPowValidRampDown;
reg                                    InBdPowValidPlat;
reg                                    RampUpDetInt;
reg                                    RampDownDetInt;
reg                                    RampUpDetIntD;
reg                                    RampDownDetIntD;

integer i;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module implements a ramp-up/ramp-down detection circuit. This has to be
//instantiated per Rx chain, wherever needed. The current value is compared
//with a value at the delay line tap, and based on the difference and the
//reference values in the config registers, the detection signals are
//asserted. This block always runs on 5 MS/s, so the valid must also
//be at 5 MHz. The Valid signal should also be passed through the delay line
//so that the comparison is not done on junk samples at the output of the
//delay line.
//This module also implements a plateau detection circuit. It has been
//placed here in order to be able to share the same delay line.

//Delay Line
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Delay_Blk
      if (nAGCRst == 1'b0) begin
         for (i = 0; i<= 23; i=i+1)
            DelayLine[i] <= CONST_DBMPOW_WIDTH_ZERO;
      end
      else if (DelLineClr == 1'b1) begin
         for (i = 0; i<= 23; i=i+1)
            DelayLine[i] <= CONST_DBMPOW_WIDTH_ZERO;
      end
      else if (SampleCount == 4'h0) begin
         DelayLine[0] <= InBdPowdBm;
         for (i = 1; i<= 23; i=i+1)
            DelayLine[i] <= DelayLine[i-1];
      end
   end //Delay_Blk

//Sample Counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SampleCnt_Blk
      if (nAGCRst == 1'b0)
         SampleCount <= 4'b0;
      else if (DelLineClr == 1'b1)
         SampleCount <= 4'b0;
      else
         SampleCount <= SampleCount + 4'h1;
   end //SampleCnt_Blk

//Delay InBdPowValid for comparing the tapped value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ValidDel_Blk
      if (nAGCRst == 1'b0)
         InBdPowValidDel <= 24'b0;
      else if (DelLineClr == 1'b1)
         InBdPowValidDel <= 24'b0;
      else if (SampleCount == 4'h0)
         InBdPowValidDel <= {InBdPowValidDel[22:0], InBdPowValid};
   end //ValidDel_Blk

//Selecting the tap for RampUp Detection
always @ (*)
   begin: RampUpSel_Blk
      case (DelRampUpTap)
         3'b000 : RampUpSel = DelayLine[00]; //0.2 us
         3'b001 : RampUpSel = DelayLine[01]; //0.4 us
         3'b010 : RampUpSel = DelayLine[03]; //0.8 us
         3'b011 : RampUpSel = DelayLine[07]; //1.6 us
         3'b100 : RampUpSel = DelayLine[11]; //2.4 us
         3'b101 : RampUpSel = DelayLine[15]; //3.2 us
         3'b110 : RampUpSel = DelayLine[19]; //4.0 us
         default: RampUpSel = DelayLine[23]; //4.8 us
      endcase //DelRampUpTap

   end //RampUpSel_Blk

//Valid for RampUp Tap
always @ (*)
   begin: InBdPowValidRampUp_Blk
      case (DelRampUpTap)
         3'b000 : InBdPowValidRampUp = InBdPowValidDel[00]; //0.2 us
         3'b001 : InBdPowValidRampUp = InBdPowValidDel[01]; //0.4 us
         3'b010 : InBdPowValidRampUp = InBdPowValidDel[03]; //0.8 us
         3'b011 : InBdPowValidRampUp = InBdPowValidDel[07]; //1.6 us
         3'b100 : InBdPowValidRampUp = InBdPowValidDel[11]; //2.4 us
         3'b101 : InBdPowValidRampUp = InBdPowValidDel[15]; //3.2 us
         3'b110 : InBdPowValidRampUp = InBdPowValidDel[19]; //4.0 us
         default: InBdPowValidRampUp = InBdPowValidDel[23]; //4.8 us
      endcase //DelRampUpTap

   end //InBdPowValidRampUp_Blk

//Selecting the tap for RampDown Detection
always @ (*)
   begin: RampDownSel_Blk
      case (DelRampDownTap)
         3'b000 : RampDownSel = DelayLine[00]; //0.2 us
         3'b001 : RampDownSel = DelayLine[01]; //0.4 us
         3'b010 : RampDownSel = DelayLine[03]; //0.8 us
         3'b011 : RampDownSel = DelayLine[07]; //1.6 us
         3'b100 : RampDownSel = DelayLine[11]; //2.4 us
         3'b101 : RampDownSel = DelayLine[15]; //3.2 us
         3'b110 : RampDownSel = DelayLine[19]; //4.0 us
         default: RampDownSel = DelayLine[23]; //4.8 us
      endcase //DelRampDownTap

   end //RampDownSel_Blk

//Valid for RampDown Tap
always @ (*)
   begin: InBdPowValidRampDown_Blk
      case (DelRampDownTap)
         3'b000 : InBdPowValidRampDown = InBdPowValidDel[00]; //0.2 us
         3'b001 : InBdPowValidRampDown = InBdPowValidDel[01]; //0.4 us
         3'b010 : InBdPowValidRampDown = InBdPowValidDel[03]; //0.8 us
         3'b011 : InBdPowValidRampDown = InBdPowValidDel[07]; //1.6 us
         3'b100 : InBdPowValidRampDown = InBdPowValidDel[11]; //2.4 us
         3'b101 : InBdPowValidRampDown = InBdPowValidDel[15]; //3.2 us
         3'b110 : InBdPowValidRampDown = InBdPowValidDel[19]; //4.0 us
         default: InBdPowValidRampDown = InBdPowValidDel[23]; //4.8 us
      endcase //DelRampDownTap

   end //InBdPowValidRampDown_Blk

//Selecting the tap for Plateau Detection
always @ (*)
   begin: PlatSel_Blk
      case (DelPlatTap)
         3'b000 : PlatSel = DelayLine[00]; //0.2 us
         3'b001 : PlatSel = DelayLine[01]; //0.4 us
         3'b010 : PlatSel = DelayLine[03]; //0.8 us
         3'b011 : PlatSel = DelayLine[07]; //1.6 us
         3'b100 : PlatSel = DelayLine[11]; //2.4 us
         3'b101 : PlatSel = DelayLine[15]; //3.2 us
         3'b110 : PlatSel = DelayLine[19]; //4.0 us
         default: PlatSel = DelayLine[23]; //4.8 us
      endcase //DelPlatTap

   end //PlatSel_Blk

//Valid for Plateau Tap
always @ (*)
   begin: InBdPowValidPlat_Blk
      case (DelPlatTap)
         3'b000 : InBdPowValidPlat = InBdPowValidDel[00]; //0.2 us
         3'b001 : InBdPowValidPlat = InBdPowValidDel[01]; //0.4 us
         3'b010 : InBdPowValidPlat = InBdPowValidDel[03]; //0.8 us
         3'b011 : InBdPowValidPlat = InBdPowValidDel[07]; //1.6 us
         3'b100 : InBdPowValidPlat = InBdPowValidDel[11]; //2.4 us
         3'b101 : InBdPowValidPlat = InBdPowValidDel[15]; //3.2 us
         3'b110 : InBdPowValidPlat = InBdPowValidDel[19]; //4.0 us
         default: InBdPowValidPlat = InBdPowValidDel[23]; //4.8 us
      endcase //DelPlatTap

   end //InBdPowValidPlat_Blk

//Find the difference in Power
assign PowDiffRampUp      = $signed({DelayLine[0][DBMPOW_WIDTH-1],DelayLine[0]}) - 
                            $signed({RampUpSel[DBMPOW_WIDTH-1],RampUpSel});
assign PowDiffRampDown    = $signed({DelayLine[0][DBMPOW_WIDTH-1],DelayLine[0]}) - 
                            $signed({RampDownSel[DBMPOW_WIDTH-1],RampDownSel});
assign PowDiffPlat        = $signed({DelayLine[0][DBMPOW_WIDTH-1],DelayLine[0]}) - 
                            $signed({PlatSel[DBMPOW_WIDTH-1],PlatSel});
assign PowDiffPlatc2      = $signed(~PowDiffPlat[DBMPOW_WIDTH:0] + {{(DBMPOW_WIDTH){1'b0}},1'b1});
assign PowDiffPlatAbs     = (PowDiffPlat[DBMPOW_WIDTH] == 1'b1) ? PowDiffPlatc2 : PowDiffPlat;

//Ramp Up Detection
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RampUpDetInt_Blk
      if (nAGCRst == 1'b0)
         RampUpDetInt <= 1'b0;
      else if (RampUpDetEn == 1'b0)
         RampUpDetInt <= 1'b0;
      else if (InBdPowValidDel[0] == 1'b1 &&
               InBdPowValidRampUp == 1'b1 &&
               PowDiffRampUp > $signed({{DBMPOW_WIDTH-7{1'b0}},DetThrRampUp}))
         RampUpDetInt <= 1'b1;
      else
         RampUpDetInt <= 1'b0;

   end //RampUpDetInt_Blk

assign RampUpDet = RampUpDetInt & ~RampUpDetIntD & RampUpDetEn;

//Ramp Down Detection
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RampDownDetInt_Blk
      if (nAGCRst == 1'b0)
         RampDownDetInt <= 1'b0;
      else if (RampDownDetEn == 1'b0)
         RampDownDetInt <= 1'b0;
      else if (InBdPowValidDel[0] == 1'b1 &&
               InBdPowValidRampDown == 1'b1 &&
               PowDiffRampDown < $signed(-{{DBMPOW_WIDTH-7{1'b0}},DetThrRampDown}))
         RampDownDetInt <= 1'b1;
      else
         RampDownDetInt <= 1'b0;

   end //RampDownDetInt_Blk

assign RampDownDet = RampDownDetInt & ~RampDownDetIntD & RampDownDetEn;

//Plateau Detection
always @ (posedge AGCClk or negedge nAGCRst)
   begin: PlatDetInt_Blk
      if (nAGCRst == 1'b0)
         PlateauDet <= 1'b0;
      else if (PlatDetEn == 1'b0)
         PlateauDet <= 1'b0;
      else if (InBdPowValidDel[0] == 1'b1 &&
               InBdPowValidPlat == 1'b1 &&
               PowDiffPlatAbs < {{DBMPOW_WIDTH-7{1'b0}},RegDetPlat})
         PlateauDet <= 1'b1;
      else
         PlateauDet <= 1'b0;

   end //PlatDetInt_Blk

//DelLineClr
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelLineClr_Blk
      if (nAGCRst == 1'b0)
         DelLineClr <= 1'b0;
      else if ((AGCCommand == 8'd1 || AGCCommand == 8'd80) &&
               (AGCCmdValid == 1'b1))
         DelLineClr <= 1'b1;
      else
         DelLineClr <= 1'b0;

   end //DelLineClr_Blk

//Detection delayed
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DetIntD_Blk
      if (nAGCRst == 1'b0) begin
         RampUpDetIntD      <= 1'b0;
         RampDownDetIntD    <= 1'b0;
      end
      else begin
         RampUpDetIntD      <= RampUpDetInt;
         RampDownDetIntD    <= RampDownDetInt;
      end

   end //DetIntD_Blk


//InBand Power delay line element in dBm for cross detectors
assign InBdPowdBmDL0   = DelayLine[00];
assign InBdPowDL0Valid = InBdPowValidDel[00];
assign InBdPowdBmDL1   = DelayLine[01];
assign InBdPowDL1Valid = InBdPowValidDel[01];

endmodule //PlatRampDetect

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
