////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 15271 $
// $Date: 2014-06-11 10:42:43 +0200 (Wed, 11 Jun 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : RF Gain Setting Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/RFGainSet.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module RFGainSet #(parameter NRX          =  4, //Number of Rx Chains (1 to 4)
                   parameter DBVPOW_WIDTH = 12, //Data width of qdBV power
                   parameter DBMPOW_WIDTH = 13  //Data width of qdBm power
                  )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,          // Active Low Reset
            input    wire                                  AGCClk,           // AGC Clock

            //Controls from HTSIG decoder
            input    wire                                  BWEq20,           // Bandwidth is 20MHz
            input    wire                                  BWEq40,           // Bandwidth is 40MHz
            input    wire                                  BWEq80,           // Bandwidth is 80MHz
            input    wire                                  VHTSig,           // VHT Packet
            input    wire                                  NoACIMargin,      // Packet with no ACI margin
            input    wire              [6:0]               HTSIGMCS,         // MCS from HTSIG field

            //Controls from RC
            input    wire                                  RCProgRFDone,     // Programming Done

            //Registers
            input    wire              [6:0]               RegRFGainMax4Det,     // RF gain maximum for detection in dB
            input    wire signed       [6:0]               RegRFGainMin4Det,     // RF gain minimum for detection in dB
            input    wire              [6:0]               RegRFGainMax,         // RF gain maximum in dB
            input    wire signed       [6:0]               RegRFGainMin,         // RF gain minimum in dB
            input    wire              [3:0]               RegAntLossdB0,        // Antenna Loss in dB
            input    wire              [3:0]               RegAntLossdB1,        // Antenna Loss in dB
            input    wire              [3:0]               RegAntLossdB2,        // Antenna Loss in dB
            input    wire              [3:0]               RegAntLossdB3,        // Antenna Loss in dB
            input    wire signed       [3:0]               RegDigGain40vs20,     // Digital gain adjustment for 40MHz
            input    wire signed       [3:0]               RegDigGain80vs20,     // Digital gain adjustment for 80MHz
            input    wire              [5:0]               RegACI20MargMCS9,     // ACI margin in 20MHz for MCS9 (11ac)
            input    wire              [5:0]               RegACI20MargMCS8,     // ACI margin in 20MHz for MCS8 (11ac)
            input    wire              [5:0]               RegACI20MargMCS7,     // ACI margin in 20MHz for MCS7/15 (11n)
            input    wire              [5:0]               RegACI20MargMCS6,     // ACI margin in 20MHz for MCS6/14 (11n)
            input    wire              [5:0]               RegACI20MargMCS5,     // ACI margin in 20MHz for MCS5/13 (11n)
            input    wire              [5:0]               RegACI20MargMCS4,     // ACI margin in 20MHz for MCS4/12 (11n)
            input    wire              [5:0]               RegACI20MargMCS3,     // ACI margin in 20MHz for MCS3/11 (11n)
            input    wire              [5:0]               RegACI20MargMCS2,     // ACI margin in 20MHz for MCS2/10 (11n)
            input    wire              [5:0]               RegACI20MargMCS1,     // ACI margin in 20MHz for MCS1/9 (11n)
            input    wire              [5:0]               RegACI20MargMCS0,     // ACI margin in 20MHz for MCS0/8 (11n)
            input    wire              [5:0]               RegACI40MargMCS9,     // ACI margin in 40MHz for MCS9 (11ac)
            input    wire              [5:0]               RegACI40MargMCS8,     // ACI margin in 40MHz for MCS8 (11ac)
            input    wire              [5:0]               RegACI40MargMCS7,     // ACI margin in 40MHz for MCS7/15 (11n)
            input    wire              [5:0]               RegACI40MargMCS6,     // ACI margin in 40MHz for MCS6/14 (11n)
            input    wire              [5:0]               RegACI40MargMCS5,     // ACI margin in 40MHz for MCS5/13 (11n)
            input    wire              [5:0]               RegACI40MargMCS4,     // ACI margin in 40MHz for MCS4/12 (11n)
            input    wire              [5:0]               RegACI40MargMCS3,     // ACI margin in 40MHz for MCS3/11 (11n)
            input    wire              [5:0]               RegACI40MargMCS2,     // ACI margin in 40MHz for MCS2/10 (11n)
            input    wire              [5:0]               RegACI40MargMCS1,     // ACI margin in 40MHz for MCS1/9 (11n)
            input    wire              [5:0]               RegACI40MargMCS0,     // ACI margin in 40MHz for MCS0/8 (11n)
            input    wire              [5:0]               RegACI80MargMCS9,     // ACI margin in 80MHz for MCS9 (11ac)
            input    wire              [5:0]               RegACI80MargMCS8,     // ACI margin in 80MHz for MCS8 (11ac)
            input    wire              [5:0]               RegACI80MargMCS7,     // ACI margin in 80MHz for MCS7/15 (11n)
            input    wire              [5:0]               RegACI80MargMCS6,     // ACI margin in 80MHz for MCS6/14 (11n)
            input    wire              [5:0]               RegACI80MargMCS5,     // ACI margin in 80MHz for MCS5/13 (11n)
            input    wire              [5:0]               RegACI80MargMCS4,     // ACI margin in 80MHz for MCS4/12 (11n)
            input    wire              [5:0]               RegACI80MargMCS3,     // ACI margin in 80MHz for MCS3/11 (11n)
            input    wire              [5:0]               RegACI80MargMCS2,     // ACI margin in 80MHz for MCS2/10 (11n)
            input    wire              [5:0]               RegACI80MargMCS1,     // ACI margin in 80MHz for MCS1/9 (11n)
            input    wire              [5:0]               RegACI80MargMCS0,     // ACI margin in 80MHz for MCS0/8 (11n)
            input    wire                                  Force20,              //Force 20 Mode
            input    wire                                  Force40,              //Force 40 Mode

            //Controls from AGC FSM
            input    wire              [7:0]               AGCCommand,       // Command from FSM Core
            input    wire              [23:0]              AGCParam,         // Command Parameter 1
            input    wire                                  AGCCmdValid,      //AGC Command Valid 

            //Power Estimates in qdBV from ADC
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx0,     //Rx Chain 0
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx1,     //Rx Chain 1
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx2,     //Rx Chain 2
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBVRx3,     //Rx Chain 3

            //InBand Power Estimates in qdBm
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx0,  //Rx Chain 0
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx1,  //Rx Chain 1
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx2,  //Rx Chain 2
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBd20PowdBmRx3,  //Rx Chain 3

            // RF GainComputation Interface
            input    wire                                  AGCRFGainCompDone, 
            input    wire              [7:0]               RFGain0dB,  //Effective gain for Rx Chain 0
            input    wire              [7:0]               RFGain1dB,  //Effective gain for Rx Chain 1
            input    wire              [7:0]               RFGain2dB,  //Effective gain for Rx Chain 2
            input    wire              [7:0]               RFGain3dB,  //Effective gain for Rx Chain 3

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //To Radio Controller
            output wire                                    RCProgRF, //Program RF

            //To RF Gain and gain Comparator
            output wire signed         [7:0]               RxGain0, //Gain for Rx Chain 0
            output wire signed         [7:0]               RxGain1, //Gain for Rx Chain 1
            output wire signed         [7:0]               RxGain2, //Gain for Rx Chain 2
            output wire signed         [7:0]               RxGain3, //Gain for Rx Chain 3

            //To SNR Estimation
            output reg                                     SNREstStrtP,  //Start SNR Estimation
            output reg  signed         [DBMPOW_WIDTH-1:0]  InBandPowRx0, //Rx Chain 0 InBand Power captured for SNR
            output reg  signed         [DBMPOW_WIDTH-1:0]  InBandPowRx1, //Rx Chain 1 InBand Power captured for SNR
            output reg  signed         [DBMPOW_WIDTH-1:0]  InBandPowRx2, //Rx Chain 2 InBand Power captured for SNR
            output reg  signed         [DBMPOW_WIDTH-1:0]  InBandPowRx3, //Rx Chain 3 InBand Power captured for SNR

            //To AGC Core FSM
            output reg                                     RFGainSetDoneP, //Gain Setting complete
            output reg                                     NoGainUpdateP,  //No Gain update

            // RF Gain Computation Interface
            output reg                                     AGCRFGainCompReq,// Effective gain computation request
            output wire                [7:0]               AGCGainTgt0, // Gain Target for Rx Chain 0
            output wire                [7:0]               AGCGainTgt1, // Gain Target for Rx Chain 1
            output wire                [7:0]               AGCGainTgt2, // Gain Target for Rx Chain 2
            output wire                [7:0]               AGCGainTgt3, // Gain Target for Rx Chain 3

            //To DSP Blocks Register Configuration
            output reg                 [1:0]               ADCPowdBVMaxIndex, //Antenna max index
            output reg                 [1:0]               ADCPowdBVMinIndex, //Antenna min index
                                       
            //Digital Gain                  
            output wire signed         [6:0]               DigitalGaindB20Rx0, //For Rx Chain 0 20MHz BW
            output wire signed         [6:0]               DigitalGaindB20Rx1, //For Rx Chain 1 20MHz BW
            output wire signed         [6:0]               DigitalGaindB20Rx2, //For Rx Chain 2 20MHz BW
            output wire signed         [6:0]               DigitalGaindB20Rx3, //For Rx Chain 3 20MHz BW
            output wire signed         [6:0]               DigitalGaindB40Rx0, //For Rx Chain 0 40MHz BW
            output wire signed         [6:0]               DigitalGaindB40Rx1, //For Rx Chain 1 40MHz BW
            output wire signed         [6:0]               DigitalGaindB40Rx2, //For Rx Chain 2 40MHz BW
            output wire signed         [6:0]               DigitalGaindB40Rx3, //For Rx Chain 3 40MHz BW
            output wire signed         [6:0]               DigitalGaindB80Rx0, //For Rx Chain 0 80MHz BW
            output wire signed         [6:0]               DigitalGaindB80Rx1, //For Rx Chain 1 80MHz BW
            output wire signed         [6:0]               DigitalGaindB80Rx2, //For Rx Chain 2 80MHz BW
            output wire signed         [6:0]               DigitalGaindB80Rx3  //For Rx Chain 3 80MHz BW
            );                         


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam FSM_SIZE = 6;

localparam IDLE              = 6'd1,
           CAPTURE_ADC_LEVEL = 6'd2,
           COMPUTE_GAIN      = 6'd4,
           RF_GAIN_SPLIT_SET = 6'd8,
           PROG_RF           = 6'd16,
           COMPUTE_DIG_GAIN  = 6'd32;

localparam TGTGAINWIDTH    = (DBVPOW_WIDTH+1);
localparam DIGGAININTWIDTH = (DBMPOW_WIDTH+2);

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                                   CaptureADCPow;
wire                                   CaptureInbandPow;
wire                                   CaptureInbandPowACI;
wire    signed  [10:0]                 TgtGainRx0Int0;
wire    signed  [10:0]                 TgtGainRx1Int0;
wire    signed  [10:0]                 TgtGainRx2Int0;
wire    signed  [10:0]                 TgtGainRx3Int0;
wire    signed  [TGTGAINWIDTH-1:0]     TgtGainRx0Int2;
wire    signed  [TGTGAINWIDTH-1:0]     TgtGainRx1Int2;
wire    signed  [TGTGAINWIDTH-1:0]     TgtGainRx2Int2;
wire    signed  [TGTGAINWIDTH-1:0]     TgtGainRx3Int2;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainRx0Rnd;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainRx1Rnd;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainRx2Rnd;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainRx3Rnd;
wire    signed  [10:0]                 TgtGainACIInt0;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainACIRx0Int2;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainACIRx1Int2;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainACIRx2Int2;
wire    signed  [TGTGAINWIDTH-3:0]     TgtGainACIRx3Int2;
wire    signed  [10:0]                 BWTab;
wire    signed  [7:0]                  TgtGainRx0;
wire    signed  [7:0]                  TgtGainRx1;
wire    signed  [7:0]                  TgtGainRx2;
wire    signed  [7:0]                  TgtGainRx3;
wire    signed  [7:0]                  DelGainRx0;
wire    signed  [7:0]                  DelGainRx1;
wire    signed  [7:0]                  DelGainRx2;
wire    signed  [7:0]                  DelGainRx3;
wire    signed  [7:0]                  AbsGainRx0;
wire    signed  [7:0]                  AbsGainRx1;
wire    signed  [7:0]                  AbsGainRx2;
wire    signed  [7:0]                  AbsGainRx3;
wire    signed  [7:0]                  TgtGainRangeRx0;
wire    signed  [7:0]                  TgtGainRangeRx1;
wire    signed  [7:0]                  TgtGainRangeRx2;
wire    signed  [7:0]                  TgtGainRangeRx3;
wire    signed  [8:0]                  DelGainRx0Int;
wire    signed  [8:0]                  DelGainRx1Int;
wire    signed  [8:0]                  DelGainRx2Int;
wire    signed  [8:0]                  DelGainRx3Int;
wire    signed  [7:0]                  DelGainRangeRx0;
wire    signed  [7:0]                  DelGainRangeRx1;
wire    signed  [7:0]                  DelGainRangeRx2;
wire    signed  [7:0]                  DelGainRangeRx3;
wire    signed  [7:0]                  AbsGainRangeRx0;
wire    signed  [7:0]                  AbsGainRangeRx1;
wire    signed  [7:0]                  AbsGainRangeRx2;
wire    signed  [7:0]                  AbsGainRangeRx3;
wire    signed  [10:0]                 DigGainRx0Int0;
wire    signed  [10:0]                 DigGainRx1Int0;
wire    signed  [10:0]                 DigGainRx2Int0;
wire    signed  [10:0]                 DigGainRx3Int0;
wire    signed  [DIGGAININTWIDTH-3:0]  DigGainRx0Rnd;   
wire    signed  [DIGGAININTWIDTH-3:0]  DigGainRx1Rnd;   
wire    signed  [DIGGAININTWIDTH-3:0]  DigGainRx2Rnd;   
wire    signed  [DIGGAININTWIDTH-3:0]  DigGainRx3Rnd;   
wire                                   CondRx1Enable;
wire                                   CondRx2Enable;
wire                                   CondRx3Enable;
wire    signed  [7:0]                  RFGainMaxMux;
wire    signed  [7:0]                  RFGainMinMux;
wire                                   SameRFGain0dB;
wire                                   SameRFGain1dB;
wire                                   SameRFGain2dB;
wire                                   SameRFGain3dB;
wire                                   SameDigGainRx0;
wire                                   SameDigGainRx1;
wire                                   SameDigGainRx2;
wire                                   SameDigGainRx3;
wire                                   UpdateRFGain;
wire                                   UpdateDigGain;
wire    signed  [DBMPOW_WIDTH-3:0]     InBandPowACIRx0;
wire    signed  [DBMPOW_WIDTH-3:0]     InBandPowACIRx1;
wire    signed  [DBMPOW_WIDTH-3:0]     InBandPowACIRx2;
wire    signed  [DBMPOW_WIDTH-3:0]     InBandPowACIRx3;
wire            [1:0]                  ADCPowdBVMinIndex0;
wire            [1:0]                  ADCPowdBVMinIndex1;
wire            [1:0]                  ADCPowdBVMinIndexInt;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg             [FSM_SIZE-1:0]         RFGSFSMCS;
reg             [FSM_SIZE-1:0]         RFGSFSMNS;
reg                                    SetRFGain;
reg                                    SetDigGain;
reg                                    RunDigGain;
reg                                    RunAbsDigGain;
reg                                    IndepRFGain;
reg                                    IndepDigGain;
reg             [1:0]                  ADCPowdBVMaxIndex0;
reg             [1:0]                  ADCPowdBVMaxIndex1;
reg             [1:0]                  ADCPowdBVMaxIndexInt;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVMaxInt0;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVMaxInt1;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVMax;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVMaxReg;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVRegRx0;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVRegRx1;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVRegRx2;
reg     signed  [DBVPOW_WIDTH-1:0]     ADCPowdBVRegRx3;
reg     signed  [DBMPOW_WIDTH-1:0]     InBd20PowMaxInt0;
reg     signed  [DBMPOW_WIDTH-1:0]     InBd20PowMaxInt1;
reg     signed  [DBMPOW_WIDTH-1:0]     InBd20PowMax;
reg     signed  [10:0]                 TgtGainRx0Int1;
reg     signed  [10:0]                 TgtGainRx1Int1;
reg     signed  [10:0]                 TgtGainRx2Int1;
reg     signed  [10:0]                 TgtGainRx3Int1;
reg     signed  [TGTGAINWIDTH-3:0]     TgtGainACIInt1;
reg     signed  [TGTGAINWIDTH-3:0]     TgtGainRx0Int3;
reg     signed  [TGTGAINWIDTH-3:0]     TgtGainRx1Int3;
reg     signed  [TGTGAINWIDTH-3:0]     TgtGainRx2Int3;
reg     signed  [TGTGAINWIDTH-3:0]     TgtGainRx3Int3;
reg     signed  [10:0]                 DigGainRx0Int1;
reg     signed  [10:0]                 DigGainRx1Int1;
reg     signed  [10:0]                 DigGainRx2Int1;
reg     signed  [10:0]                 DigGainRx3Int1;
reg     signed  [7:0]                  TDAGainReg;
reg     signed  [7:0]                  CurGainRegRx0;
reg     signed  [7:0]                  CurGainRegRx1;
reg     signed  [7:0]                  CurGainRegRx2;
reg     signed  [7:0]                  CurGainRegRx3;
reg     signed  [7:0]                  RxGain0PreSplit;
reg     signed  [7:0]                  RxGain1PreSplit;
reg     signed  [7:0]                  RxGain2PreSplit;
reg     signed  [7:0]                  RxGain3PreSplit;
reg     signed  [7:0]                  RxGain0Split;
reg     signed  [7:0]                  RxGain1Split;
reg     signed  [7:0]                  RxGain2Split;
reg     signed  [7:0]                  RxGain3Split;
reg                                    LimRangeMode;
reg                                    RCProgRFInt;
reg                                    RCGainComp;
reg             [3:0]                  CmdValidDel;
reg             [5:0]                  GainCompDoneDel;
reg     signed  [DIGGAININTWIDTH-1:0]  DigGainRx0Int2;
reg     signed  [DIGGAININTWIDTH-1:0]  DigGainRx1Int2;
reg     signed  [DIGGAININTWIDTH-1:0]  DigGainRx2Int2;
reg     signed  [DIGGAININTWIDTH-1:0]  DigGainRx3Int2;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRx0Int3;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRx1Int3;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRx2Int3;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRx3Int3;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainMinInt0;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainMinInt1;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainMin;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRegRx0;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRegRx1;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRegRx2;
reg     signed  [DIGGAININTWIDTH-3:0]  DigGainRegRx3;
reg     signed  [6:0]                  CurDigGainRegRx0;
reg     signed  [6:0]                  CurDigGainRegRx1;
reg     signed  [6:0]                  CurDigGainRegRx2;
reg     signed  [6:0]                  CurDigGainRegRx3;
reg     signed  [7:0]                  TADigGainReg;
reg             [5:0]                  ACIMargin;

// The following 2 strings are useful during simulations
// to view States using strings instead of numbers
`ifdef SIMU_ON
  reg [25*8:0] CurrStateStr;
  reg [25*8:0] NextStateStr;
`endif


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module computes the gain values required, splits between gains
// and passes them on to the Radio Controller module.

// Rx path condition enable
generate
   if (NRX > 3) begin : U_CondRx4Enable_gen
      assign CondRx1Enable = 1'b1;
      assign CondRx2Enable = 1'b1;
      assign CondRx3Enable = 1'b1;
   end //NRX > 3
   else if (NRX == 3) begin : U_CondRx3Enable_gen
      assign CondRx1Enable = 1'b1;
      assign CondRx2Enable = 1'b1;
      assign CondRx3Enable = 1'b0;
   end //NRX = 3 
   else if (NRX == 2) begin : U_CondRx2Enable_gen
      assign CondRx1Enable = 1'b1;
      assign CondRx2Enable = 1'b0;
      assign CondRx3Enable = 1'b0;
   end //NRX = 2
   else begin : U_CondRx1Enable_gen
      assign CondRx1Enable = 1'b0;
      assign CondRx2Enable = 1'b0;
      assign CondRx3Enable = 1'b0;
   end //NRX = 1
endgenerate


/////////////////////////////////
// RF Gain Set FSM
/////////////////////////////////

//Sequential Block
always @ (posedge AGCClk or negedge nAGCRst)
   begin: FSM_Seq
      if (nAGCRst == 1'b0)
         RFGSFSMCS <= IDLE;
      else if (AGCCmdValid == 1'b1 && AGCCommand == 8'd1)
         RFGSFSMCS <= IDLE;
      else
         RFGSFSMCS <= RFGSFSMNS;
   end //FSM_Seq

//Combinatorial Block
always @ (*)
   begin: FSM_Combo
      case(RFGSFSMCS)

         IDLE: begin
            if (SetRFGain == 1'b1 || SetDigGain == 1'b1) begin
               if ((AGCCommand == 8'd14) || (AGCCommand == 8'd15))
                  RFGSFSMNS = CAPTURE_ADC_LEVEL;
               else
                  RFGSFSMNS = COMPUTE_GAIN;
            end
            else
               RFGSFSMNS = IDLE;
         end //IDLE

         CAPTURE_ADC_LEVEL: begin
            RFGSFSMNS = COMPUTE_GAIN;
         end //CAPTURE_ADC_LEVEL

         COMPUTE_GAIN: begin
            if (AGCRFGainCompDone == 1'b1)
               RFGSFSMNS = RF_GAIN_SPLIT_SET;
            else
               RFGSFSMNS = COMPUTE_GAIN;
         end //COMPUTE_GAIN

         RF_GAIN_SPLIT_SET: begin
            if (UpdateRFGain || (AGCCommand == 8'd12) || (AGCCommand == 8'd13))
              RFGSFSMNS = PROG_RF;
            else
              RFGSFSMNS = COMPUTE_DIG_GAIN;
         end //RF_GAIN_SPLIT_SET

         PROG_RF: begin
            if (RCProgRFDone)
               RFGSFSMNS = IDLE;
            else
               RFGSFSMNS = PROG_RF;
         end //PROG_RF

         COMPUTE_DIG_GAIN: begin
            if (GainCompDoneDel[5])
               RFGSFSMNS = IDLE;
            else
               RFGSFSMNS = COMPUTE_DIG_GAIN;
         end //COMPUTE_DIG_GAIN

         // pragma coverage block = off
         default: begin
            RFGSFSMNS = IDLE;
         end //default
         // pragma coverage block = on

      endcase //RFGSFSMCS

   end //FSM_Combo

/////////////////////////////////
// Gain Computation
/////////////////////////////////

//Finding Maximum of ADC Power
//Finding Maximum index of ADC Power
always @ (*)
begin: ADCMaxPowInt_Blk
   if ((ADCPowdBVRx0 > ADCPowdBVRx1) || !CondRx1Enable) begin
     ADCPowdBVMaxInt0   = ADCPowdBVRx0;
     ADCPowdBVMaxIndex0 = 2'd0;
   end
   else begin
     ADCPowdBVMaxInt0   = ADCPowdBVRx1;
     ADCPowdBVMaxIndex0 = 2'd1;
   end

   if ((ADCPowdBVRx2 > ADCPowdBVRx3) || !CondRx3Enable) begin
     ADCPowdBVMaxInt1   = ADCPowdBVRx2;
     ADCPowdBVMaxIndex1 = 2'd2;
   end
   else begin
     ADCPowdBVMaxInt1   = ADCPowdBVRx3;
     ADCPowdBVMaxIndex1 = 2'd3;
   end

   if ((ADCPowdBVMaxInt0 > ADCPowdBVMaxInt1) || !CondRx2Enable) begin
     ADCPowdBVMax         = ADCPowdBVMaxInt0;
     ADCPowdBVMaxIndexInt = ADCPowdBVMaxIndex0;
   end
   else begin
     ADCPowdBVMax         = ADCPowdBVMaxInt1;
     ADCPowdBVMaxIndexInt = ADCPowdBVMaxIndex1;
   end
end

//Finding Minimum index of ADC Power
generate
   if (NRX == 4) begin
      assign ADCPowdBVMinIndex0   = (ADCPowdBVRx0 > ADCPowdBVRx1) ? 2'd1 : 2'd0;
      assign ADCPowdBVMinIndex1   = (ADCPowdBVRx2 > ADCPowdBVRx3) ? 2'd3 : 2'd2;
      assign ADCPowdBVMinIndexInt = (ADCPowdBVMaxInt0 > ADCPowdBVMaxInt1)  ? ADCPowdBVMinIndex1 : ADCPowdBVMinIndex0;
   end // NRX == 4
   else if (NRX == 3) begin
      assign ADCPowdBVMinIndex0   = (ADCPowdBVRx0 > ADCPowdBVRx1) ? 2'd1 : 2'd0;
      assign ADCPowdBVMinIndexInt = (ADCPowdBVMaxInt0 > ADCPowdBVRx2)  ? 2'd2 : ADCPowdBVMinIndex0;
   end // NRX == 3
   else if (NRX == 2) begin
      assign ADCPowdBVMinIndexInt = (ADCPowdBVRx0 > ADCPowdBVRx1) ? 2'd1 : 2'd0;
   end // NRX == 2
   else begin
      assign ADCPowdBVMinIndexInt = 2'd0;
   end // NRX == 1
endgenerate

//Register Max Power
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCPowdBVMaxReg_Blk
      if (nAGCRst == 1'b0) begin
         ADCPowdBVMaxReg   <= $signed({{DBVPOW_WIDTH}{1'b0}});
         ADCPowdBVMaxIndex <= 2'b0;
         ADCPowdBVMinIndex <= 2'b0;
      end
      else begin
         ADCPowdBVMaxReg   <= ADCPowdBVMax;
         ADCPowdBVMaxIndex <= ADCPowdBVMaxIndexInt;
         ADCPowdBVMinIndex <= ADCPowdBVMinIndexInt;
      end
   end //ADCPowdBVMaxReg_Blk

//Capturing ADC dBV Power depending on Common/Independent Gain Setting
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCPowdBVReg_Blk
      if (nAGCRst == 1'b0) begin
         ADCPowdBVRegRx0 <= $signed({{DBVPOW_WIDTH}{1'b0}});
         ADCPowdBVRegRx1 <= $signed({{DBVPOW_WIDTH}{1'b0}});
         ADCPowdBVRegRx2 <= $signed({{DBVPOW_WIDTH}{1'b0}});
         ADCPowdBVRegRx3 <= $signed({{DBVPOW_WIDTH}{1'b0}});
      end
      else if (CaptureADCPow == 1'b1) begin
         if (IndepRFGain == 1'b0) begin
            ADCPowdBVRegRx0 <= ADCPowdBVMaxReg;
            ADCPowdBVRegRx1 <= ADCPowdBVMaxReg;
            ADCPowdBVRegRx2 <= ADCPowdBVMaxReg;
            ADCPowdBVRegRx3 <= ADCPowdBVMaxReg;
         end //Common Gain
         else begin //Independent Gain
            ADCPowdBVRegRx0 <= ADCPowdBVRx0;
            ADCPowdBVRegRx1 <= ADCPowdBVRx1;
            ADCPowdBVRegRx2 <= ADCPowdBVRx2;
            ADCPowdBVRegRx3 <= ADCPowdBVRx3;
         end //IndepRFGain == 1'b1
      end //CaptureADCPow == 1'b1
   end //ADCPowdBVReg_Blk

//Current Gain Registers
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CurGainReg_Blk
      if (nAGCRst == 1'b0) begin
         CurGainRegRx0 <= $signed({8{1'b0}});
         CurGainRegRx1 <= $signed({8{1'b0}});
         CurGainRegRx2 <= $signed({8{1'b0}});
         CurGainRegRx3 <= $signed({8{1'b0}});
      end
      else if (SetRFGain == 1'b1) begin // Update Gain
         CurGainRegRx0 <= RxGain0Split;
         CurGainRegRx1 <= RxGain1Split;
         CurGainRegRx2 <= RxGain2Split;
         CurGainRegRx3 <= RxGain3Split;
      end
   end //CurGainReg_Blk

/////////////////////////////////
// Extract AGC commands
/////////////////////////////////

//Load TDA Gain register (Target/Delta/Absolute)
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TDAGainReg_Blk
      if (nAGCRst == 1'b0)
         TDAGainReg <= $signed({8{1'b0}});
      else if (SetRFGain == 1'b1)
         TDAGainReg <= $signed(AGCParam[15:8]);
   end //TDAGainReg_Blk

//Decode AGC Commands for RF/Digital gain
always @ (*)
   begin: SetGain_Blk
      if ((AGCCmdValid == 1'b1) &&
          ((AGCCommand == 8'd12) ||
           (AGCCommand == 8'd13) ||
           (AGCCommand == 8'd14) ||
           (AGCCommand == 8'd15)))
      begin
         SetRFGain  = AGCParam[0];
         SetDigGain = AGCParam[1];
      end
      else
      begin
         SetRFGain  = 1'b0;
         SetDigGain = 1'b0;
      end
   end //SetGain_Blk

//Common/Independent RF Gain
always @ (posedge AGCClk or negedge nAGCRst)
   begin: IndepRFGain_Blk
      if (nAGCRst == 1'b0)
         IndepRFGain <= 1'b1;
      else if (AGCCmdValid == 1'b1 && AGCCommand == 8'd11)
         IndepRFGain <= AGCParam[0];
   end //IndepRFGain_Blk

//Common/Independent Digital Gain
always @ (posedge AGCClk or negedge nAGCRst)
   begin: IndepDigGain_Blk
      if (nAGCRst == 1'b0)
         IndepDigGain <= 1'b1;
      else if (AGCCmdValid == 1'b1 && AGCCommand == 8'd11)
         IndepDigGain <= AGCParam[1];
   end //IndepDigGain_Blk

//Start SNR estimation after digital gain complete
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SNREstStrtPReg_Blk
      if (nAGCRst == 1'b0)
         SNREstStrtP <= 1'b0;
      else if (GainCompDoneDel[5] == 1'b1 && ((AGCCommand == 8'd12) ||
           (AGCCommand == 8'd13) || (AGCCommand == 8'd14) ||
           (AGCCommand == 8'd15)))
         SNREstStrtP <= AGCParam[2];
      else // Generate pulse
         SNREstStrtP <= 1'b0;
   end //SNREstStrtPReg_Blk

//Set Limit Range mode
always @ (posedge AGCClk or negedge nAGCRst)
   begin: LimRangeModeReg_Blk
      if (nAGCRst == 1'b0)
         LimRangeMode <= 1'b0;
      else if (AGCCmdValid == 1'b1 && ((AGCCommand == 8'd12) ||
           (AGCCommand == 8'd13) || (AGCCommand == 8'd14) ||
           (AGCCommand == 8'd15)))
         LimRangeMode <= AGCParam[3];
   end //LimRangeModeReg_Blk

//Generate RCProgRF
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RCProgRFInt_Blk
      if (nAGCRst == 1'b0)
         RCProgRFInt <= 1'b0;
      else if (RCProgRFDone == 1'b1)
         RCProgRFInt <= 1'b0;
      else if (RFGSFSMNS == PROG_RF)
         RCProgRFInt <= 1'b1;
   end //RCProgRFInt_Blk

//RCProgRF should be a level
assign RCProgRF = RCProgRFInt;

//Capture ADC Power Values
assign CaptureADCPow = (RFGSFSMCS == CAPTURE_ADC_LEVEL) ? 1'b1 : 1'b0;

//Capture Inband Power Values
assign CaptureInbandPow = RunDigGain & ~CaptureInbandPowACI & CaptureADCPow;

//Capture Inband Power Values for ACI case
assign CaptureInbandPowACI = CaptureADCPow & (AGCCommand == 8'd15);

/////////////////////////////////
// Generate ACI Margin and BWTab
/////////////////////////////////

// BW Tab
assign BWTab = (BWEq80 == 1'b1) ? 11'sd6 : 
               (BWEq40 == 1'b1) ? 11'sd3 : 11'sd0;
// ACIMargin
always @ (posedge AGCClk or negedge nAGCRst)
begin: ACIMargin_Blk
   if (nAGCRst == 1'b0)
      ACIMargin <= 6'b0;
   else if (NoACIMargin)
   // In case of MuMIMO or NDP frame ACI margin must be by-passed
      ACIMargin <= 6'b0;
   else if (BWEq80 || Force20 || (Force40 && BWEq40)) begin
   // 20MHz BW or 40MHz frame in 40MHz BW or 80MHz frame in 80MHz BW 
     case (HTSIGMCS)
       7'd0      ,7'd16,7'd24,7'd32 : ACIMargin <= RegACI20MargMCS0;
       7'd1      ,7'd17,7'd25       : ACIMargin <= RegACI20MargMCS1;
       7'd2      ,7'd18,7'd26       : ACIMargin <= RegACI20MargMCS2;
       7'd3      ,7'd19,7'd27       : ACIMargin <= RegACI20MargMCS3;
       7'd4,7'd12,7'd20,7'd28       : ACIMargin <= RegACI20MargMCS4;
       7'd5,7'd13,7'd21,7'd29       : ACIMargin <= RegACI20MargMCS5;
       7'd6,7'd14,7'd22,7'd30       : ACIMargin <= RegACI20MargMCS6;
       7'd7,7'd15,7'd23,7'd31       : ACIMargin <= RegACI20MargMCS7;
       7'd8                         : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI20MargMCS8;
                                      else
                                        ACIMargin <= RegACI20MargMCS0;
                                      end
       7'd9                         : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI20MargMCS9;
                                      else
                                        ACIMargin <= RegACI20MargMCS1;
                                     end
       7'd10                        : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI20MargMCS9; // Set same as MCS9 for QAM-1024
                                      else
                                        ACIMargin <= RegACI20MargMCS2;
                                      end
       7'd11                        : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI20MargMCS9; // Set same as MCS9 for QAM-1024
                                      else
                                        ACIMargin <= RegACI20MargMCS3;
                                     end
       default                      : ACIMargin <= 6'b0; // For unequal modulation, no margin
     endcase
   end
   else if (BWEq40 || (Force40 && BWEq20)) begin
   // 20MHz frame in 40MHz BW or 40MHz frame in 80MHz BW 
     case (HTSIGMCS)
       7'd0      ,7'd16,7'd24,7'd32 : ACIMargin <= RegACI40MargMCS0;
       7'd1      ,7'd17,7'd25       : ACIMargin <= RegACI40MargMCS1;
       7'd2      ,7'd18,7'd26       : ACIMargin <= RegACI40MargMCS2;
       7'd3      ,7'd19,7'd27       : ACIMargin <= RegACI40MargMCS3;
       7'd4,7'd12,7'd20,7'd28       : ACIMargin <= RegACI40MargMCS4;
       7'd5,7'd13,7'd21,7'd29       : ACIMargin <= RegACI40MargMCS5;
       7'd6,7'd14,7'd22,7'd30       : ACIMargin <= RegACI40MargMCS6;
       7'd7,7'd15,7'd23,7'd31       : ACIMargin <= RegACI40MargMCS7;
       7'd8                         : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI40MargMCS8;
                                      else
                                        ACIMargin <= RegACI40MargMCS0;
                                      end
       7'd9                         : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI40MargMCS9;
                                      else
                                        ACIMargin <= RegACI40MargMCS1;
                                     end
       7'd10                        : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI40MargMCS9; // Set same as MCS9 for QAM-1024
                                      else
                                        ACIMargin <= RegACI40MargMCS2;
                                      end
       7'd11                        : begin
                                      if (VHTSig)
                                        ACIMargin <= RegACI40MargMCS9; // Set same as MCS9 for QAM-1024
                                      else
                                        ACIMargin <= RegACI40MargMCS3;
                                     end
       default                      : ACIMargin <= 6'b0; // For unequal modulation, no margin
     endcase
   end
   else begin
   // 20MHz frame in 80MHz BW
     case (HTSIGMCS)
       7'd0      ,7'd16,7'd24 : ACIMargin <= RegACI80MargMCS0;
       7'd1      ,7'd17,7'd25 : ACIMargin <= RegACI80MargMCS1;
       7'd2,7'd10,7'd18,7'd26 : ACIMargin <= RegACI80MargMCS2;
       7'd3,7'd11,7'd19,7'd27 : ACIMargin <= RegACI80MargMCS3;
       7'd4,7'd12,7'd20,7'd28 : ACIMargin <= RegACI80MargMCS4;
       7'd5,7'd13,7'd21,7'd29 : ACIMargin <= RegACI80MargMCS5;
       7'd6,7'd14,7'd22,7'd30 : ACIMargin <= RegACI80MargMCS6;
       7'd7,7'd15,7'd23,7'd31 : ACIMargin <= RegACI80MargMCS7;
       7'd8                   : begin
                                if (VHTSig)
                                  ACIMargin <= RegACI80MargMCS8;
                                else
                                  ACIMargin <= RegACI80MargMCS0;
                                end
       7'd9                   : begin
                                if (VHTSig)
                                  ACIMargin <= RegACI80MargMCS9;
                                else
                                  ACIMargin <= RegACI80MargMCS1;
                               end
       7'd10                  : begin
                                if (VHTSig)
                                  ACIMargin <= RegACI80MargMCS9; // Set same as MCS9 for QAM-1024
                                else
                                  ACIMargin <= RegACI80MargMCS2;
                                end
       7'd11                  : begin
                                if (VHTSig)
                                  ACIMargin <= RegACI80MargMCS9; // Set same as MCS9 for QAM-1024
                                else
                                  ACIMargin <= RegACI80MargMCS3;
                               end
       default                : ACIMargin <= 6'b0; // For unequal modulation, no margin
     endcase
   end
end

/////////////////////////////////
// Calculate the new gain values
/////////////////////////////////

//Target Mode with no ACI margin
assign TgtGainRx0Int0 = (CurGainRegRx0 + TDAGainReg) <<< 3'sd2;
assign TgtGainRx1Int0 = (CurGainRegRx1 + TDAGainReg) <<< 3'sd2;
assign TgtGainRx2Int0 = (CurGainRegRx2 + TDAGainReg) <<< 3'sd2;
assign TgtGainRx3Int0 = (CurGainRegRx3 + TDAGainReg) <<< 3'sd2;

always @ (posedge AGCClk or negedge nAGCRst)
   begin: TgtGain_Blk
      if (nAGCRst == 1'b0) begin
         TgtGainRx0Int1 <= $signed({11{1'b0}});
         TgtGainRx1Int1 <= $signed({11{1'b0}});
         TgtGainRx2Int1 <= $signed({11{1'b0}});
         TgtGainRx3Int1 <= $signed({11{1'b0}});
      end
      else begin
         TgtGainRx0Int1 <= TgtGainRx0Int0;
         TgtGainRx1Int1 <= TgtGainRx1Int0;
         TgtGainRx2Int1 <= TgtGainRx2Int0;
         TgtGainRx3Int1 <= TgtGainRx3Int0;
      end
   end //TgtGain_Blk

assign TgtGainRx0Int2 = $signed({TgtGainRx0Int1[10],TgtGainRx0Int1}) - ADCPowdBVRegRx0;
assign TgtGainRx1Int2 = $signed({TgtGainRx1Int1[10],TgtGainRx1Int1}) - ADCPowdBVRegRx1;
assign TgtGainRx2Int2 = $signed({TgtGainRx2Int1[10],TgtGainRx2Int1}) - ADCPowdBVRegRx2;
assign TgtGainRx3Int2 = $signed({TgtGainRx3Int1[10],TgtGainRx3Int1}) - ADCPowdBVRegRx3;

//Round TgtGainRxInt2 by 2 bits
generate 
   Round #(
           .INPUT_WIDTH(TGTGAINWIDTH),
           .OUTPUT_WIDTH(TGTGAINWIDTH-2)
          )
           U_ROUNDINT0(
                       .InputData(TgtGainRx0Int2),
                       .RoundData(TgtGainRx0Rnd)
                      );

   if (NRX > 1) begin
      Round #(
              .INPUT_WIDTH(TGTGAINWIDTH),
              .OUTPUT_WIDTH(TGTGAINWIDTH-2)
             )
              U_ROUNDINT1(
                          .InputData(TgtGainRx1Int2),
                          .RoundData(TgtGainRx1Rnd)
                         );
   end
   else begin
      assign TgtGainRx1Rnd = $signed({{(TGTGAINWIDTH-2)}{1'b0}});
   end

   if (NRX > 2) begin
      Round #(
              .INPUT_WIDTH(TGTGAINWIDTH),
              .OUTPUT_WIDTH(TGTGAINWIDTH-2)
             )
              U_ROUNDINT2(
                          .InputData(TgtGainRx2Int2),
                          .RoundData(TgtGainRx2Rnd)
                         );
   end
   else begin
      assign TgtGainRx2Rnd = $signed({{(TGTGAINWIDTH-2)}{1'b0}});
   end

   if (NRX > 3) begin
      Round #(
              .INPUT_WIDTH(TGTGAINWIDTH),
              .OUTPUT_WIDTH(TGTGAINWIDTH-2)
             )
              U_ROUNDINT3(
                          .InputData(TgtGainRx3Int2),
                          .RoundData(TgtGainRx3Rnd)
                         );
   end
   else begin
      assign TgtGainRx3Rnd = $signed({{(TGTGAINWIDTH-2)}{1'b0}});
   end

endgenerate

//Target Mode with ACI margin
assign TgtGainACIInt0 = $signed({{3{TDAGainReg[7]}},TDAGainReg}) + 11'sd13 -
                        $signed({5'b0, ACIMargin}) - BWTab;

always @ (posedge AGCClk or negedge nAGCRst)
   begin: TgtGainACI_Blk
      if (nAGCRst == 1'b0) begin
         TgtGainACIInt1 <= $signed({11{1'b0}});
      end
      else begin
         TgtGainACIInt1 <= TgtGainACIInt0;
      end
   end //TgtGainACI_Blk

//Finding maximum of Inband Power for ACI margin case
always @ (*)
begin: InbdMaxPowInt_Blk
   if ((InBandPowRx0 > InBandPowRx1) || !CondRx1Enable)
     InBd20PowMaxInt0 = InBandPowRx0;
   else
     InBd20PowMaxInt0 = InBandPowRx1;
   if ((InBandPowRx2 > InBandPowRx3) || !CondRx3Enable)
     InBd20PowMaxInt1 = InBandPowRx2;
   else
     InBd20PowMaxInt1 = InBandPowRx3;
   if ((InBd20PowMaxInt0 > InBd20PowMaxInt1) || !CondRx2Enable)
     InBd20PowMax     = InBd20PowMaxInt0;
   else
     InBd20PowMax     = InBd20PowMaxInt1;
end

//Mux on Inband Power for ACI margin case according to IndepRFGain
assign InBandPowACIRx0 = (IndepRFGain) ? InBandPowRx0[DBMPOW_WIDTH-1:2] : InBd20PowMax[DBMPOW_WIDTH-1:2];
assign InBandPowACIRx1 = (IndepRFGain) ? InBandPowRx1[DBMPOW_WIDTH-1:2] : InBd20PowMax[DBMPOW_WIDTH-1:2];
assign InBandPowACIRx2 = (IndepRFGain) ? InBandPowRx2[DBMPOW_WIDTH-1:2] : InBd20PowMax[DBMPOW_WIDTH-1:2];
assign InBandPowACIRx3 = (IndepRFGain) ? InBandPowRx3[DBMPOW_WIDTH-1:2] : InBd20PowMax[DBMPOW_WIDTH-1:2];

assign TgtGainACIRx0Int2 = TgtGainACIInt1 - InBandPowACIRx0;
assign TgtGainACIRx1Int2 = TgtGainACIInt1 - InBandPowACIRx1;
assign TgtGainACIRx2Int2 = TgtGainACIInt1 - InBandPowACIRx2;
assign TgtGainACIRx3Int2 = TgtGainACIInt1 - InBandPowACIRx3;

always @ (posedge AGCClk or negedge nAGCRst)
   begin: TgtGainInt_Blk
      if (nAGCRst == 1'b0) begin
         TgtGainRx0Int3 <= $signed({{(TGTGAINWIDTH-2)}{1'b0}});
         TgtGainRx1Int3 <= $signed({{(TGTGAINWIDTH-2)}{1'b0}});
         TgtGainRx2Int3 <= $signed({{(TGTGAINWIDTH-2)}{1'b0}});
         TgtGainRx3Int3 <= $signed({{(TGTGAINWIDTH-2)}{1'b0}});
      end
      else if (AGCCommand == 8'd14) begin // Target mode with no ACI margin
         TgtGainRx0Int3 <= TgtGainRx0Rnd;
         TgtGainRx1Int3 <= TgtGainRx1Rnd;
         TgtGainRx2Int3 <= TgtGainRx2Rnd;
         TgtGainRx3Int3 <= TgtGainRx3Rnd;
      end
      else if (AGCCommand == 8'd15) begin // Target mode with ACI margin
         if ($signed(TgtGainRx0Rnd) < $signed(TgtGainACIRx0Int2))
           TgtGainRx0Int3 <= TgtGainRx0Rnd;
         else
           TgtGainRx0Int3 <= TgtGainACIRx0Int2;
         if ($signed(TgtGainRx1Rnd) < $signed(TgtGainACIRx1Int2))
           TgtGainRx1Int3 <= TgtGainRx1Rnd;
         else
           TgtGainRx1Int3 <= TgtGainACIRx1Int2;
         if ($signed(TgtGainRx2Rnd) < $signed(TgtGainACIRx2Int2))
           TgtGainRx2Int3 <= TgtGainRx2Rnd;
         else
           TgtGainRx2Int3 <= TgtGainACIRx2Int2;
         if ($signed(TgtGainRx3Rnd) < $signed(TgtGainACIRx3Int2))
           TgtGainRx3Int3 <= TgtGainRx3Rnd;
         else
           TgtGainRx3Int3 <= TgtGainACIRx3Int2;
      end
   end //TgtGainInt_Blk

assign TgtGainRx0 = $signed(TgtGainRx0Int3[7:0]);
assign TgtGainRx1 = $signed(TgtGainRx1Int3[7:0]);
assign TgtGainRx2 = $signed(TgtGainRx2Int3[7:0]);
assign TgtGainRx3 = $signed(TgtGainRx3Int3[7:0]);

//Mux on RF gain max and min from LimRangeMode
assign RFGainMaxMux = (LimRangeMode) ? {1'b0,RegRFGainMax4Det} :
                                       {1'b0,RegRFGainMax};
assign RFGainMinMux = (LimRangeMode) ? {RegRFGainMin4Det[6],RegRFGainMin4Det} :
                                       {RegRFGainMin[6],    RegRFGainMin};

//Target Mode : Range between RFGainMinMux / RFGainMaxMux
assign TgtGainRangeRx0 = (TgtGainRx0 > RFGainMaxMux) ? RFGainMaxMux :
                         (TgtGainRx0 < RFGainMinMux) ? RFGainMinMux : TgtGainRx0;
assign TgtGainRangeRx1 = (TgtGainRx1 > RFGainMaxMux) ? RFGainMaxMux :
                         (TgtGainRx1 < RFGainMinMux) ? RFGainMinMux : TgtGainRx1;
assign TgtGainRangeRx2 = (TgtGainRx2 > RFGainMaxMux) ? RFGainMaxMux :
                         (TgtGainRx2 < RFGainMinMux) ? RFGainMinMux : TgtGainRx2;
assign TgtGainRangeRx3 = (TgtGainRx3 > RFGainMaxMux) ? RFGainMaxMux :
                         (TgtGainRx3 < RFGainMinMux) ? RFGainMinMux : TgtGainRx3;

//Delta Mode
assign DelGainRx0Int = CurGainRegRx0 - TDAGainReg;
assign DelGainRx1Int = CurGainRegRx1 - TDAGainReg;
assign DelGainRx2Int = CurGainRegRx2 - TDAGainReg;
assign DelGainRx3Int = CurGainRegRx3 - TDAGainReg;

assign DelGainRx0 = $signed(DelGainRx0Int[7:0]);
assign DelGainRx1 = $signed(DelGainRx1Int[7:0]);
assign DelGainRx2 = $signed(DelGainRx2Int[7:0]);
assign DelGainRx3 = $signed(DelGainRx3Int[7:0]);

//Delta Mode Range between RFGainMinMux / RFGainMaxMux
assign DelGainRangeRx0 = (DelGainRx0 > RFGainMaxMux) ? RFGainMaxMux :
                         (DelGainRx0 < RFGainMinMux) ? RFGainMinMux : DelGainRx0;
assign DelGainRangeRx1 = (DelGainRx1 > RFGainMaxMux) ? RFGainMaxMux :
                         (DelGainRx1 < RFGainMinMux) ? RFGainMinMux : DelGainRx1;
assign DelGainRangeRx2 = (DelGainRx2 > RFGainMaxMux) ? RFGainMaxMux :
                         (DelGainRx2 < RFGainMinMux) ? RFGainMinMux : DelGainRx2;
assign DelGainRangeRx3 = (DelGainRx3 > RFGainMaxMux) ? RFGainMaxMux :
                         (DelGainRx3 < RFGainMinMux) ? RFGainMinMux : DelGainRx3;

//Absolute Mode
assign AbsGainRx0 = TDAGainReg;
assign AbsGainRx1 = (CondRx1Enable) ? TDAGainReg : 8'b0;
assign AbsGainRx2 = (CondRx2Enable) ? TDAGainReg : 8'b0;
assign AbsGainRx3 = (CondRx3Enable) ? TDAGainReg : 8'b0;

//Absolute Mode Range between RFGainMinMux / RFGainMaxMux
assign AbsGainRangeRx0 = (AbsGainRx0 > RFGainMaxMux) ? RFGainMaxMux :
                         (AbsGainRx0 < RFGainMinMux) ? RFGainMinMux : AbsGainRx0;
assign AbsGainRangeRx1 = (AbsGainRx1 > RFGainMaxMux) ? RFGainMaxMux :
                         (AbsGainRx1 < RFGainMinMux) ? RFGainMinMux : AbsGainRx1;
assign AbsGainRangeRx2 = (AbsGainRx2 > RFGainMaxMux) ? RFGainMaxMux :
                         (AbsGainRx2 < RFGainMinMux) ? RFGainMinMux : AbsGainRx2;
assign AbsGainRangeRx3 = (AbsGainRx3 > RFGainMaxMux) ? RFGainMaxMux :
                         (AbsGainRx3 < RFGainMinMux) ? RFGainMinMux : AbsGainRx3;

//Select Target/Delta/Absolute depending on the AGC Command
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RxGainInt_Blk
      if (nAGCRst == 1'b0) begin
         RxGain0PreSplit <= $signed({8{1'b0}});
         RxGain1PreSplit <= $signed({8{1'b0}});
         RxGain2PreSplit <= $signed({8{1'b0}});
         RxGain3PreSplit <= $signed({8{1'b0}});
      end
      else if (RFGSFSMCS == COMPUTE_GAIN) begin
         if ((AGCCommand == 8'd14) || (AGCCommand == 8'd15)) begin //Target Mode
            RxGain0PreSplit <= TgtGainRangeRx0;
            RxGain1PreSplit <= TgtGainRangeRx1;
            RxGain2PreSplit <= TgtGainRangeRx2;
            RxGain3PreSplit <= TgtGainRangeRx3;
         end //(AGCCommand == 8'd14) || (AGCCommand == 8'd15)
         else if (AGCCommand == 8'd13) begin //Delta Mode
            RxGain0PreSplit <= DelGainRangeRx0;
            RxGain1PreSplit <= DelGainRangeRx1;
            RxGain2PreSplit <= DelGainRangeRx2;
            RxGain3PreSplit <= DelGainRangeRx3;
         end //AGCCommand == 8'd13
         else begin // AGCCommand == 8'd12 Absolute Mode
            RxGain0PreSplit <= AbsGainRangeRx0;
            RxGain1PreSplit <= AbsGainRangeRx1;
            RxGain2PreSplit <= AbsGainRangeRx2;
            RxGain3PreSplit <= AbsGainRangeRx3;
         end //else
      end //RFGSFSMCS == COMPUTE_GAIN
   end //RxGainInt_Blk

/////////////////////////////////
// RC Gain Computation done
/////////////////////////////////

//This needs to be generated once the new gain is ready for gain split
//The delay required for computing the gain is different for the
//Target/Delta/Absolute modes. Following is the delay w.r.t. CmdValid
//Target Mode   : 4 Clocks
//Delta Mode    : 2 Clocks
//Absolute Mode : 2 Clocks
always @ (*)
   begin:RCGainComp_Blk
      if (RFGSFSMCS == COMPUTE_GAIN) begin
         if (((AGCCommand == 8'd14) || (AGCCommand == 8'd15)) && CmdValidDel[2] == 1'b1) //Target Mode
            RCGainComp = 1'b1;
         else if (AGCCommand == 8'd13 && CmdValidDel[0] == 1'b1) //Delta Mode
            RCGainComp = 1'b1;
         else if (AGCCommand == 8'd12 && CmdValidDel[0] == 1'b1) //Abs Mode
            RCGainComp = 1'b1;
         else
            RCGainComp = 1'b0;
      end
      else
         RCGainComp = 1'b0;
   end //RCGainCompInt_Blk

//Generate AGCRFGainCompReq as level
always @ (posedge AGCClk or negedge nAGCRst)
   begin: AGCRFGainCompReq_Blk
      if (nAGCRst == 1'b0)
         AGCRFGainCompReq <= 1'b0;
      else if (AGCRFGainCompDone)
         AGCRFGainCompReq <= 1'b0;
      else if (RCGainComp)
         AGCRFGainCompReq <= 1'b1;
   end //AGCRFGainCompReq_Blk

//Delayed Versions of CmdValid
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CmdValidDel_Blk
      if (nAGCRst == 1'b0)
         CmdValidDel <= 4'b0;
      else
      begin
        CmdValidDel[3:1] <= CmdValidDel[2:0];
        if ((AGCCommand == 8'd12) || (AGCCommand == 8'd13) ||
            (AGCCommand == 8'd14) || (AGCCommand == 8'd15))
           CmdValidDel[0] <= AGCCmdValid;
      end
   end //CmdValidDel_Blk

//Delayed Versions of AGCRFGainCompDone
always @ (posedge AGCClk or negedge nAGCRst)
   begin: GainCompDoneDel_Blk
      if (nAGCRst == 1'b0)
         GainCompDoneDel <= 6'b0;
      else
      begin
        GainCompDoneDel[5:1] <= GainCompDoneDel[4:0];
        GainCompDoneDel[0]   <= AGCRFGainCompDone;
      end
   end //GainCompDoneDel_Blk


/////////////////////////////////
// Effective RF Gain computation
/////////////////////////////////
assign AGCGainTgt0 = RxGain0PreSplit;
assign AGCGainTgt1 = RxGain1PreSplit;
assign AGCGainTgt2 = RxGain2PreSplit;
assign AGCGainTgt3 = RxGain3PreSplit;

always @ (posedge AGCClk or negedge nAGCRst)
   begin: RxGainSplitReg_Blk
      if (nAGCRst == 1'b0)
      begin
         RxGain0Split <= 8'd0;
         RxGain1Split <= 8'd0;
         RxGain2Split <= 8'd0;
         RxGain3Split <= 8'd0;
       end
      else if (AGCRFGainCompDone)
      begin
         RxGain0Split <= RFGain0dB;
         RxGain1Split <= RFGain1dB;
         RxGain2Split <= RFGain2dB;
         RxGain3Split <= RFGain3dB;
      end
   end //RxGainSplitReg_Blk

assign RxGain0 = RxGain0Split;
assign RxGain1 = RxGain1Split;
assign RxGain2 = RxGain2Split;
assign RxGain3 = RxGain3Split;

/////////////////////////////////
// RF Gain status to AGC FSM
/////////////////////////////////
always @ (*)
   begin: RFGainSetDoneP_Blk
      if (RFGSFSMCS == PROG_RF)
         RFGainSetDoneP = RCProgRFDone;
      else if (RFGSFSMCS == COMPUTE_DIG_GAIN && GainCompDoneDel[5])
         RFGainSetDoneP = 1'b1;
      else
         RFGainSetDoneP = 1'b0;
   end //RFGainSetDoneP_Blk

always @ (*)
   begin: NoGainUpdateP_Blk
      if (RFGSFSMCS == COMPUTE_DIG_GAIN && GainCompDoneDel[5])
         NoGainUpdateP = ~ UpdateDigGain;
      else
         NoGainUpdateP = 1'b0;
   end //NoGainUpdateP_Blk

//Generates same gain signal if current gain is same as next gain
assign SameRFGain0dB = (RxGain0Split == CurGainRegRx0) ? 1'b1 : 1'b0;
assign SameRFGain1dB = (CondRx1Enable) ? ((RxGain1Split == CurGainRegRx1) ? 1'b1 : 1'b0) : 1'b1;
assign SameRFGain2dB = (CondRx2Enable) ? ((RxGain2Split == CurGainRegRx2) ? 1'b1 : 1'b0) : 1'b1;
assign SameRFGain3dB = (CondRx3Enable) ? ((RxGain3Split == CurGainRegRx3) ? 1'b1 : 1'b0) : 1'b1;

//Set update RF gain if RF gain is different than current RF gain
assign UpdateRFGain = ~ (SameRFGain0dB & SameRFGain1dB & SameRFGain2dB & SameRFGain3dB);

//Set update digital gain if digital gain is different than current digital gain
assign UpdateDigGain = ~ (SameDigGainRx0 & SameDigGainRx1 & SameDigGainRx2 & SameDigGainRx3);

/////////////////////////////////
//Digital Gain Computation
/////////////////////////////////

//Current Digital Gain Registers
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CurDigGainReg_Blk
      if (nAGCRst == 1'b0) begin
         CurDigGainRegRx0 <= $signed({7{1'b0}});
         CurDigGainRegRx1 <= $signed({7{1'b0}});
         CurDigGainRegRx2 <= $signed({7{1'b0}});
         CurDigGainRegRx3 <= $signed({7{1'b0}});
      end
      else if (SetDigGain == 1'b1) begin // Update Digital Gain
         CurDigGainRegRx0 <= $signed(DigGainRegRx0[6:0]);
         CurDigGainRegRx1 <= $signed(DigGainRegRx1[6:0]);
         CurDigGainRegRx2 <= $signed(DigGainRegRx2[6:0]);
         CurDigGainRegRx3 <= $signed(DigGainRegRx3[6:0]);
      end
   end //CurDigGainReg_Blk

//Load RA Digital Gain register (Target/Absolute)
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TADigGainReg_Blk
      if (nAGCRst == 1'b0)
         TADigGainReg <= $signed({8{1'b0}});
      else if (SetDigGain == 1'b1)
         TADigGainReg <= $signed(AGCParam[23:16]);
   end //TADigGainReg_Blk

assign DigGainRx0Int0 = (TADigGainReg - RxGain0Split + $signed({4'b0, RegAntLossdB0})) <<< 3'sd2;
assign DigGainRx1Int0 = (TADigGainReg - RxGain1Split + $signed({4'b0, RegAntLossdB1})) <<< 3'sd2;
assign DigGainRx2Int0 = (TADigGainReg - RxGain2Split + $signed({4'b0, RegAntLossdB2})) <<< 3'sd2;
assign DigGainRx3Int0 = (TADigGainReg - RxGain3Split + $signed({4'b0, RegAntLossdB3})) <<< 3'sd2;

//Capture Digital Gain request and absolute mode
always @ (posedge AGCClk or negedge nAGCRst)
   begin: CaptDigGain_Blk
      if (nAGCRst == 1'b0) begin
         RunDigGain    <= 1'b0;
         RunAbsDigGain <= 1'b0;
      end
      else if (SetDigGain == 1'b1) begin
         RunDigGain    <= 1'b1;
         RunAbsDigGain <= (AGCCommand == 8'd12);
      end
      else if (GainCompDoneDel[4] == 1'b1) begin
         RunDigGain <= 1'b0;
      end
   end //CaptDigGain_Blk

//Register Digital Gain 1st computation
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DigGainInt1_Blk
      if (nAGCRst == 1'b0) begin
         DigGainRx0Int1 <= $signed({11{1'b0}});
         DigGainRx1Int1 <= $signed({11{1'b0}});
         DigGainRx2Int1 <= $signed({11{1'b0}});
         DigGainRx3Int1 <= $signed({11{1'b0}});
      end
      else begin
         DigGainRx0Int1 <= DigGainRx0Int0;
         DigGainRx1Int1 <= DigGainRx1Int0;
         DigGainRx2Int1 <= DigGainRx2Int0;
         DigGainRx3Int1 <= DigGainRx3Int0;
      end
   end //DigGainInt1_Blk

//Capture InBand Power on CaptureADCPow
always @ (posedge AGCClk or negedge nAGCRst)
   begin: InBandPowCapt_Blk
      if (nAGCRst == 1'b0) begin
         InBandPowRx0 <= $signed({{DBMPOW_WIDTH}{1'b0}});
         InBandPowRx1 <= $signed({{DBMPOW_WIDTH}{1'b0}});
         InBandPowRx2 <= $signed({{DBMPOW_WIDTH}{1'b0}});
         InBandPowRx3 <= $signed({{DBMPOW_WIDTH}{1'b0}});
      end
      else if (CaptureInbandPowACI || CaptureInbandPow) begin
         InBandPowRx0 <= InBd20PowdBmRx0;
         InBandPowRx1 <= InBd20PowdBmRx1;
         InBandPowRx2 <= InBd20PowdBmRx2;
         InBandPowRx3 <= InBd20PowdBmRx3;
      end
   end //InBandPowCapt_Blk

always @ (posedge AGCClk or negedge nAGCRst)
   begin: DigGainInt2_Blk
      if (nAGCRst == 1'b0) begin
         DigGainRx0Int2 <= $signed({{DIGGAININTWIDTH}{1'b0}});
         DigGainRx1Int2 <= $signed({{DIGGAININTWIDTH}{1'b0}});
         DigGainRx2Int2 <= $signed({{DIGGAININTWIDTH}{1'b0}});
         DigGainRx3Int2 <= $signed({{DIGGAININTWIDTH}{1'b0}});
      end
      else begin
         DigGainRx0Int2 <= $signed({{4{DigGainRx0Int1[10]}},DigGainRx0Int1}) - 
                           $signed({{2{InBandPowRx0[DBMPOW_WIDTH-1]}},InBandPowRx0}) + 15'sd52;
         DigGainRx1Int2 <= $signed({{4{DigGainRx1Int1[10]}},DigGainRx1Int1}) - 
                           $signed({{2{InBandPowRx1[DBMPOW_WIDTH-1]}},InBandPowRx1}) + 15'sd52;
         DigGainRx2Int2 <= $signed({{4{DigGainRx2Int1[10]}},DigGainRx2Int1}) - 
                           $signed({{2{InBandPowRx2[DBMPOW_WIDTH-1]}},InBandPowRx2}) + 15'sd52;
         DigGainRx3Int2 <= $signed({{4{DigGainRx3Int1[10]}},DigGainRx3Int1}) -
                           $signed({{2{InBandPowRx3[DBMPOW_WIDTH-1]}},InBandPowRx3}) + 15'sd52;
      end
   end //DigGainInt2_Blk

//Round DigGainRxInt2 by 2 bits
generate 
   Round #(
           .INPUT_WIDTH(DIGGAININTWIDTH),
           .OUTPUT_WIDTH(DIGGAININTWIDTH-2)
          )
           U_ROUNDINT4(
                       .InputData(DigGainRx0Int2),
                       .RoundData(DigGainRx0Rnd)
                      );

   if (NRX > 1) begin
      Round #(
              .INPUT_WIDTH(DIGGAININTWIDTH),
              .OUTPUT_WIDTH(DIGGAININTWIDTH-2)
             )
              U_ROUNDINT5(
                          .InputData(DigGainRx1Int2),
                          .RoundData(DigGainRx1Rnd)
                         );
   end
   else begin
      assign DigGainRx1Rnd = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
   end

   if (NRX > 2) begin
      Round #(
              .INPUT_WIDTH(DIGGAININTWIDTH),
              .OUTPUT_WIDTH(DIGGAININTWIDTH-2)
             )
              U_ROUNDINT6(
                          .InputData(DigGainRx2Int2),
                          .RoundData(DigGainRx2Rnd)
                         );
   end
   else begin
      assign DigGainRx2Rnd = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
   end

   if (NRX > 3) begin
      Round #(
              .INPUT_WIDTH(DIGGAININTWIDTH),
              .OUTPUT_WIDTH(DIGGAININTWIDTH-2)
             )
              U_ROUNDINT7(
                          .InputData(DigGainRx3Int2),
                          .RoundData(DigGainRx3Rnd)
                         );
   end
   else begin
      assign DigGainRx3Rnd = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
   end

endgenerate

//Register the rounded value
always @ (posedge AGCClk or negedge nAGCRst)
   begin:DigGainInt3_Blk
      if (nAGCRst == 1'b0) begin
         DigGainRx0Int3 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRx1Int3 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRx2Int3 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRx3Int3 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
      end
      else begin
         DigGainRx0Int3 <= DigGainRx0Rnd;
         DigGainRx1Int3 <= DigGainRx1Rnd;
         DigGainRx2Int3 <= DigGainRx2Rnd;
         DigGainRx3Int3 <= DigGainRx3Rnd;
      end
   end //DigGainInt3_Blk

//The final digital gain has to be selected based on IndepDigGain.
//If IndepDigGain is 0, then the minimum of the 4 gains has to be chosen.
//Find the minimum of all the digital gains
generate
   if (NRX == 4) begin
      always @ (*)
         begin: DigGainMinInt4_Blk
            DigGainMinInt0 = (DigGainRx0Int3 < DigGainRx1Int3) ?
                              DigGainRx0Int3 : DigGainRx1Int3;

            DigGainMinInt1 = (DigGainRx2Int3 < DigGainRx3Int3) ?
                              DigGainRx2Int3 : DigGainRx3Int3;

            DigGainMin     = (DigGainMinInt0 < DigGainMinInt1) ?
                              DigGainMinInt0 : DigGainMinInt1;
         end //DigGainMinInt_Blk
   end // NRX == 4
   else if (NRX == 3) begin
      always @ (*)
         begin: DigGainMinInt3_Blk
            DigGainMinInt0 = (DigGainRx0Int3 < DigGainRx1Int3) ?
                              DigGainRx0Int3 : DigGainRx1Int3;
            DigGainMinInt1 = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});

            DigGainMin     = (DigGainMinInt0 < DigGainRx2Int3) ?
                              DigGainMinInt0 : DigGainRx2Int3;
         end //DigGainMinInt_Blk
   end // NRX == 3
   else if (NRX == 2) begin
      always @ (*)
         begin: DigGainMinInt2_Blk
            DigGainMinInt0 = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
            DigGainMinInt1 = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
            DigGainMin     = (DigGainRx0Int3 < DigGainRx1Int3) ?
                              DigGainRx0Int3 : DigGainRx1Int3;
         end //DigGainMinInt_Blk
   end // NRX == 2
   else begin
      always @ (*)
         begin: DigGainMinInt1_Blk
            DigGainMinInt0 = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
            DigGainMinInt1 = $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
            DigGainMin     = DigGainRx0Int3;
         end //DigGainMinInt_Blk
   end // NRX == 1
endgenerate

//Selection based on IndepDigGain
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DigGainReg_Blk
      if (nAGCRst == 1'b0) begin
         DigGainRegRx0 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRegRx1 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRegRx2 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRegRx3 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
      end
      else if (AGCCmdValid == 1'b1 && AGCCommand == 8'd1)
      begin
         DigGainRegRx0 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRegRx1 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRegRx2 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
         DigGainRegRx3 <= $signed({{(DIGGAININTWIDTH-2)}{1'b0}});
      end
      else if ((GainCompDoneDel[4] == 1'b1) && (RunDigGain == 1'b1)) begin
         if (RunAbsDigGain == 1'b1) begin             // Absolute Mode
            DigGainRegRx0 <= $signed({{5{TADigGainReg[7]}},TADigGainReg[7:0]});
            DigGainRegRx1 <= $signed({{5{TADigGainReg[7]}},TADigGainReg[7:0]});
            DigGainRegRx2 <= $signed({{5{TADigGainReg[7]}},TADigGainReg[7:0]});
            DigGainRegRx3 <= $signed({{5{TADigGainReg[7]}},TADigGainReg[7:0]});
         end
         else if (IndepDigGain == 1'b0) begin         // Target Mode with Common Gain
            DigGainRegRx0 <= DigGainMin;
            DigGainRegRx1 <= DigGainMin;
            DigGainRegRx2 <= DigGainMin;
            DigGainRegRx3 <= DigGainMin;
         end 
         else begin                                   // Target Mode with Independent Gain
            DigGainRegRx0 <= DigGainRx0Int3;
            DigGainRegRx1 <= DigGainRx1Int3;
            DigGainRegRx2 <= DigGainRx2Int3;
            DigGainRegRx3 <= DigGainRx3Int3;
         end //IndepDigGain == 1'b1
      end //
   end //DigGainReg_Blk

//Generates same digital gain signal if current digital gain is same as next digital gain
assign SameDigGainRx0 = (DigGainRegRx0[6:0] == CurDigGainRegRx0) ? 1'b1 : 1'b0;
assign SameDigGainRx1 = (CondRx1Enable) ? ((DigGainRegRx1[6:0] == CurDigGainRegRx1) ? 1'b1 : 1'b0) : 1'b1;
assign SameDigGainRx2 = (CondRx2Enable) ? ((DigGainRegRx2[6:0] == CurDigGainRegRx2) ? 1'b1 : 1'b0) : 1'b1;
assign SameDigGainRx3 = (CondRx3Enable) ? ((DigGainRegRx3[6:0] == CurDigGainRegRx3) ? 1'b1 : 1'b0) : 1'b1;

//Digital gain final outputs
// 20MHz BW
assign DigitalGaindB20Rx0 = $signed(DigGainRegRx0[6:0]);
assign DigitalGaindB20Rx1 = $signed(DigGainRegRx1[6:0]);
assign DigitalGaindB20Rx2 = $signed(DigGainRegRx2[6:0]);
assign DigitalGaindB20Rx3 = $signed(DigGainRegRx3[6:0]);
// 40MHz BW
assign DigitalGaindB40Rx0 = $signed(DigGainRegRx0[6:0]) - 7'sd3 + $signed({{3{RegDigGain40vs20[3]}},RegDigGain40vs20});
assign DigitalGaindB40Rx1 = $signed(DigGainRegRx1[6:0]) - 7'sd3 + $signed({{3{RegDigGain40vs20[3]}},RegDigGain40vs20});
assign DigitalGaindB40Rx2 = $signed(DigGainRegRx2[6:0]) - 7'sd3 + $signed({{3{RegDigGain40vs20[3]}},RegDigGain40vs20});
assign DigitalGaindB40Rx3 = $signed(DigGainRegRx3[6:0]) - 7'sd3 + $signed({{3{RegDigGain40vs20[3]}},RegDigGain40vs20});
// 80MHz BW
assign DigitalGaindB80Rx0 = $signed(DigGainRegRx0[6:0]) - 7'sd6 + $signed({{3{RegDigGain80vs20[3]}},RegDigGain80vs20});
assign DigitalGaindB80Rx1 = $signed(DigGainRegRx1[6:0]) - 7'sd6 + $signed({{3{RegDigGain80vs20[3]}},RegDigGain80vs20});
assign DigitalGaindB80Rx2 = $signed(DigGainRegRx2[6:0]) - 7'sd6 + $signed({{3{RegDigGain80vs20[3]}},RegDigGain80vs20});
assign DigitalGaindB80Rx3 = $signed(DigGainRegRx3[6:0]) - 7'sd6 + $signed({{3{RegDigGain80vs20[3]}},RegDigGain80vs20});


// assigning Strings to States so that it would be easy
// to debug in simulation
`ifdef SIMU_ON
always@(*)
begin
  case (RFGSFSMCS)
    IDLE                : CurrStateStr = {"IDLE"};
    CAPTURE_ADC_LEVEL   : CurrStateStr = {"CAPTURE_ADC_LEVEL"};
    COMPUTE_GAIN        : CurrStateStr = {"COMPUTE_GAIN"};
    RF_GAIN_SPLIT_SET   : CurrStateStr = {"RF_GAIN_SPLIT_SET"};
    PROG_RF             : CurrStateStr = {"PROG_RF"};
    COMPUTE_DIG_GAIN    : CurrStateStr = {"COMPUTE_DIG_GAIN"};
    default             : CurrStateStr = {"UNKNOWN"};
  endcase
end
always@(*)
begin
  case (RFGSFSMNS)
    IDLE                : NextStateStr = {"IDLE"};
    CAPTURE_ADC_LEVEL   : NextStateStr = {"CAPTURE_ADC_LEVEL"};
    COMPUTE_GAIN        : NextStateStr = {"COMPUTE_GAIN"};
    RF_GAIN_SPLIT_SET   : NextStateStr = {"RF_GAIN_SPLIT_SET"};
    PROG_RF             : NextStateStr = {"PROG_RF"};
    COMPUTE_DIG_GAIN    : NextStateStr = {"COMPUTE_DIG_GAIN"};
    default             : NextStateStr = {"UNKNOWN"};
  endcase
end
`endif

endmodule //RFGainSet

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
