////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Ramp-Up/Ramp-Down Detection Block for CCA
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/RampDetect.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module RampDetect #(parameter DBMPOW_WIDTH = 11 //Data width of dBm power
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,   // Active Low Reset
            input    wire                                  AGCClk,    // AGC Clock

            //Config Registers
            input    wire              [5:0]               RegDetRampUp,   // Ramp Up Detect Value
            input    wire              [5:0]               RegDetRampDown, // Ramp Down Detect Value
            input    wire              [2:0]               DelRampUpTap,   // Delay line tap out
            input    wire              [2:0]               DelRampDownTap, // Delay line tap out

            //Power & Valid
            input    wire   signed     [DBMPOW_WIDTH-1:0]  InBdPowdBm,   // InBand Power in dBm
            input    wire                                  InBdPowValid, // Qualifies InBdPowdBm

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   wire                                  RampUpDet,  // Ramp Up Detected
            output   wire                                  RampDownDet // Ramp Down Detected
            );



//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [DBMPOW_WIDTH:0]  PowDiffRampUp;
wire    signed     [DBMPOW_WIDTH:0]  PowDiffRampDown;

//Integers
integer i;


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [DBMPOW_WIDTH-1:0]  DelayLine[20:0];
reg                [3:0]               SampleCount;
reg                [20:0]              InBdPowValidDel;
reg     signed     [DBMPOW_WIDTH-1:0]  RampUpSel;
reg     signed     [DBMPOW_WIDTH-1:0]  RampDownSel;
reg                                    InBdPowValidRampUp;
reg                                    InBdPowValidRampDown;
reg                                    RampUpDetInt;
reg                                    RampDownDetInt;
reg                                    RampUpDetIntD;
reg                                    RampDownDetIntD;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module implements a ramp-up/ramp-down detection circuit. This has to be
//instantiated per Rx chain, wherever needed. The current value is compared
//with a value at the delay line tap, and based on the difference and the
//reference values in the config registers, the detection signals are
//asserted. This block always runs on 5 MS/s, so the valid must also
//be at 5 MHz. The Valid signal should also be passed through the delay line
//so that the comparison is not done on junk samples at the output of the
//delay line.

//Delay Line
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Delay_Blk
      if (nAGCRst == 1'b0) begin
         for (i = 0; i<= 20; i=i+1)
            DelayLine[i] <= $signed({{(DBMPOW_WIDTH)}{1'b0}});
      end
      else if (SampleCount == 4'h0 && InBdPowValid) begin
         DelayLine[0]    <= InBdPowdBm;
         for (i = 1; i<= 20; i=i+1)
            DelayLine[i] <= DelayLine[i-1];
      end
   end //Delay_Blk

//Sample Counter
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SampleCnt_Blk
      if (nAGCRst == 1'b0)
         SampleCount <= 4'b0;
      else
         SampleCount <= SampleCount + 4'h1;
   end //SampleCnt_Blk

//Delay InBdPowValid for comparing the tapped value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ValidDel_Blk
      if (nAGCRst == 1'b0)
         InBdPowValidDel <= 21'b0;
      else if (SampleCount == 4'h0)
         InBdPowValidDel <= {InBdPowValidDel[19:0], InBdPowValid};

   end //ValidDel_Blk

//Selecting the tap for RampUp Detection
always @ (*)
   begin: RampUpSel_Blk
      case (DelRampUpTap)
         3'b000 : RampUpSel = DelayLine[00]; //0.2 us
         3'b001 : RampUpSel = DelayLine[03]; //0.8 us
         3'b010 : RampUpSel = DelayLine[07]; //1.6 us
         3'b011 : RampUpSel = DelayLine[11]; //2.4 us
         3'b100 : RampUpSel = DelayLine[15]; //3.2 us
         default: RampUpSel = DelayLine[20]; //4.2 us
      endcase //DelRampUpTap

   end //RampUpSel_Blk

//Valid for RampUp Tap
always @ (*)
   begin: InBdPowValidRampUp_Blk
      case (DelRampUpTap)
         3'b000 : InBdPowValidRampUp = InBdPowValidDel[00]; //0.2 us
         3'b001 : InBdPowValidRampUp = InBdPowValidDel[03]; //0.8 us
         3'b010 : InBdPowValidRampUp = InBdPowValidDel[07]; //1.6 us
         3'b011 : InBdPowValidRampUp = InBdPowValidDel[11]; //2.4 us
         3'b100 : InBdPowValidRampUp = InBdPowValidDel[15]; //3.2 us
         default: InBdPowValidRampUp = InBdPowValidDel[20]; //4.2 us
      endcase //DelRampUpTap

   end //InBdPowValidRampUp_Blk

//Selecting the tap for RampDown Detection
always @ (*)
   begin: RampDownSel_Blk
      case (DelRampDownTap)
         3'b000 : RampDownSel = DelayLine[00]; //0.2 us
         3'b001 : RampDownSel = DelayLine[03]; //0.8 us
         3'b010 : RampDownSel = DelayLine[07]; //1.6 us
         3'b011 : RampDownSel = DelayLine[11]; //2.4 us
         3'b100 : RampDownSel = DelayLine[15]; //3.2 us
         default: RampDownSel = DelayLine[20]; //4.2 us
      endcase //DelRampDownTap

   end //RampDownSel_Blk

//Valid for RampDown Tap
always @ (*)
   begin: InBdPowValidRampDown_Blk
      case (DelRampDownTap)
         3'b000 : InBdPowValidRampDown = InBdPowValidDel[00]; //0.2 us
         3'b001 : InBdPowValidRampDown = InBdPowValidDel[03]; //0.8 us
         3'b010 : InBdPowValidRampDown = InBdPowValidDel[07]; //1.6 us
         3'b011 : InBdPowValidRampDown = InBdPowValidDel[11]; //2.4 us
         3'b100 : InBdPowValidRampDown = InBdPowValidDel[15]; //3.2 us
         default: InBdPowValidRampDown = InBdPowValidDel[20]; //4.2 us
      endcase //DelRampDownTap
   end //InBdPowValidRampUp_Blk

//Find the difference in Power
assign PowDiffRampUp   = $signed({DelayLine[0][DBMPOW_WIDTH-1],DelayLine[0]}) - 
                         $signed({RampUpSel[DBMPOW_WIDTH-1],RampUpSel});
assign PowDiffRampDown = $signed({DelayLine[0][DBMPOW_WIDTH-1],DelayLine[0]}) - 
                         $signed({RampDownSel[DBMPOW_WIDTH-1],RampDownSel});

//Ramp Up Detection
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RampUpDet_Blk
      if (nAGCRst == 1'b0)
         RampUpDetInt <= 1'b0;
      else if (InBdPowValidDel[0] == 1'b1 &&
               InBdPowValidRampUp == 1'b1 &&
               PowDiffRampUp > $signed({{DBMPOW_WIDTH-5{1'b0}},RegDetRampUp}))
         RampUpDetInt <= 1'b1;
      else
         RampUpDetInt <= 1'b0;
   end //RampUpDet_Blk

assign RampUpDet = RampUpDetInt & ~RampUpDetIntD;

//Ramp Down Detection
always @ (posedge AGCClk or negedge nAGCRst)
   begin: RampDownDet_Blk
      if (nAGCRst == 1'b0)
         RampDownDetInt <= 1'b0;
      else if (InBdPowValidDel[0] == 1'b1 &&
               InBdPowValidRampDown == 1'b1 &&
               PowDiffRampDown < $signed(-{{DBMPOW_WIDTH-5{1'b0}},RegDetRampDown}))
         RampDownDetInt <= 1'b1;
      else
         RampDownDetInt <= 1'b0;
   end //RampDownDet_Blk

assign RampDownDet = RampDownDetInt & ~RampDownDetIntD;

//Detection delayed
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DetIntD_Blk
      if (nAGCRst == 1'b0) begin
         RampUpDetIntD   <= 1'b0;
         RampDownDetIntD <= 1'b0;
      end
      else begin
         RampUpDetIntD   <= RampUpDetInt;
         RampDownDetIntD <= RampDownDetInt;
      end
   end //DetIntD_Blk

endmodule //RampDetect

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
