////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: asboui $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 3355 $
// $Date: 2012-07-24 13:16:23 +0200 (Tue, 24 Jul 2012) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Saturation Detection Block
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dbchef@svn.frso.rivierawaves.com:/svn/rw_wlan_nx/trunk/IPs/HW/Modem/Src/common/AGC/AGCFSM/verilog/SatDetect.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module SatDetect #(parameter DBVPOW_WIDTH = 12  //Data width of ADC Sat power
                  )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,   //Active Low Reset
            input    wire                                  AGCClk,    //AGC Clock

            //Controls
            input    wire                                  SatDetEn, //Block enable
            input    wire              [7:0]               AGCCommand,  //Command from FSM Core
            input    wire                                  AGCCmdValid, //AGC Command Valid

            //Config Registers
            input    wire   signed     [9:0]               SatLowThr, //Low Threshold
            input    wire   signed     [9:0]               SatDynHighThr, //Dynamic Low Threshold
            input    wire   signed     [9:0]               SatDynLowThr, //Dynamic High Threshold
            input    wire              [6:0]               SatDelayCount, //Delay Counter

            //Power
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx0, //From Rx Chain 0
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx1, //From Rx Chain 1
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx2, //From Rx Chain 2
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCSatPowdBVRx3, //From Rx Chain 3
            input    wire                                  ADCSatPowValid, //Qualifies Power

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   wire                                  SatDetectRx0, //For Rx Chain 0
            output   wire                                  SatDetectRx1, //For Rx Chain 1
            output   wire                                  SatDetectRx2, //For Rx Chain 2
            output   wire                                  SatDetectRx3  //For Rx Chain 3
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                                   DelayCnt0En;
wire                                   DelayCnt1En;
wire                                   DelayCnt2En;
wire                                   DelayCnt3En;
wire                                   PowGtDynHighRx0;
wire                                   PowGtDynHighRx1;
wire                                   PowGtDynHighRx2;
wire                                   PowGtDynHighRx3;
wire                                   PowGtDynLowRx0;
wire                                   PowGtDynLowRx1;
wire                                   PowGtDynLowRx2;
wire                                   PowGtDynLowRx3;
wire                                   FSMBlkRst;


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [6:0]               DelayCnt0;
reg                [6:0]               DelayCnt1;
reg                [6:0]               DelayCnt2;
reg                [6:0]               DelayCnt3;
reg                                    SatDetectIntRx0;
reg                                    SatDetectIntRx1;
reg                                    SatDetectIntRx2;
reg                                    SatDetectIntRx3;
reg                [1:0]               Div0Count50NS; //Counter for 50ns delay.
reg                [1:0]               Div1Count50NS; //Counter for 50ns delay.
reg                [1:0]               Div2Count50NS; //Counter for 50ns delay.
reg                [1:0]               Div3Count50NS; //Counter for 50ns delay.


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module detects saturation events on all the Rx Chains and reports the
//same to the AGC FSM. It also keeps a count of the number of saturation
//events that occured during a scenario.

//For Rx Chain 0
//Enable for Delay Counter
assign DelayCnt0En = ((ADCSatPowValid == 1'b1) &&
                      ($signed(ADCSatPowdBVRx0[DBVPOW_WIDTH-1:2]) > SatLowThr)) ?
                      1'b1 : 1'b0;

//Div Counter for 50 ns delay
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Div0Count50NS_Blk
      if (nAGCRst == 1'b0)
         Div0Count50NS <= 2'h0;
      else if (!DelayCnt0En)
         Div0Count50NS <= 2'h0;
      else
         Div0Count50NS <= Div0Count50NS + 2'h1;
   end //Div0Count50NS_Blk

//Delay Counter 0
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelayCnt0_Blk
      if (nAGCRst == 1'b0)
         DelayCnt0 <= 7'b0;
      else if (!DelayCnt0En)
         DelayCnt0 <= SatDelayCount;
      else if ((DelayCnt0 != 7'b0) && (Div0Count50NS == 2'h0) && SatDetEn)
         DelayCnt0 <= DelayCnt0 - 7'b1;
   end //DelayCnt0_Blk

//Compare input power with dynamic thresholds
assign PowGtDynHighRx0 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx0[DBVPOW_WIDTH-1:2]) > SatDynHighThr)) ?
                          1'b1 : 1'b0;

assign PowGtDynLowRx0 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx0[DBVPOW_WIDTH-1:2]) > SatDynLowThr)) ?
                          1'b1 : 1'b0;

//Mux the threshold crossings depending on the value of DelayCnt0
//and use it to detect saturation
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SatDetectIntRx0_Blk
      if (nAGCRst == 1'b0)
         SatDetectIntRx0 <= 1'b0;
      else if (FSMBlkRst || !SatDetEn)
         SatDetectIntRx0 <= 1'b0;
      else if ((DelayCnt0 == 7'b0) && PowGtDynHighRx0)
         SatDetectIntRx0 <= 1'b1; 
      else if ((DelayCnt0 != 7'b0) && PowGtDynLowRx0)
         SatDetectIntRx0 <= 1'b1;
      else
         SatDetectIntRx0 <= 1'b0;
   end //SatDetectIntRx0_Blk

//For Rx Chain 1
//Enable for Delay Counter
assign DelayCnt1En = ((ADCSatPowValid == 1'b1) &&
                      ($signed(ADCSatPowdBVRx1[DBVPOW_WIDTH-1:2]) > SatLowThr)) ?
                      1'b1 : 1'b0;

//Div Counter for 50 ns delay
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Div1Count50NS_Blk
      if (nAGCRst == 1'b0)
         Div1Count50NS <= 2'h0;
      else if (!DelayCnt1En)
         Div1Count50NS <= 2'h0;
      else
         Div1Count50NS <= Div1Count50NS + 2'h1;
   end //Div1Count50NS_Blk

//Delay Counter 1
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelayCnt1_Blk
      if (nAGCRst == 1'b0)
         DelayCnt1 <= 7'b0;
      else if (!DelayCnt1En)
         DelayCnt1 <= SatDelayCount;
      else if ((DelayCnt1 != 7'b0) && (Div1Count50NS == 2'h0) && SatDetEn)
         DelayCnt1 <= DelayCnt1 - 7'b1;
   end //DelayCnt1_Blk

//Compare input power with dynamic thresholds
assign PowGtDynHighRx1 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx1[DBVPOW_WIDTH-1:2]) > SatDynHighThr)) ?
                          1'b1 : 1'b0;

assign PowGtDynLowRx1 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx1[DBVPOW_WIDTH-1:2]) > SatDynLowThr)) ?
                          1'b1 : 1'b0;

//Mux the threshold crossings depending on the value of DelayCnt1
//and use it to detect saturation
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SatDetectIntRx1_Blk
      if (nAGCRst == 1'b0)
         SatDetectIntRx1 <= 1'b0;
      else if (FSMBlkRst || !SatDetEn)
         SatDetectIntRx1 <= 1'b0;
      else if ((DelayCnt1 == 7'b0) && PowGtDynHighRx1)
         SatDetectIntRx1 <= 1'b1; 
      else if ((DelayCnt1 != 7'b0) && PowGtDynLowRx1)
         SatDetectIntRx1 <= 1'b1;
      else
         SatDetectIntRx1 <= 1'b0;
   end //SatDetectIntRx1_Blk

//For Rx Chain 2
//Enable for Delay Counter
assign DelayCnt2En = ((ADCSatPowValid == 1'b1) &&
                      ($signed(ADCSatPowdBVRx2[DBVPOW_WIDTH-1:2]) > SatLowThr)) ?
                      1'b1 : 1'b0;

//Div Counter for 50 ns delay
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Div2Count50NS_Blk
      if (nAGCRst == 1'b0)
         Div2Count50NS <= 2'h0;
      else if (!DelayCnt2En)
         Div2Count50NS <= 2'h0;
      else
         Div2Count50NS <= Div2Count50NS + 2'h1;
   end //Div2Count50NS_Blk

//Delay Counter 2
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelayCnt2_Blk
      if (nAGCRst == 1'b0)
         DelayCnt2 <= 7'b0;
      else if (!DelayCnt2En)
         DelayCnt2 <= SatDelayCount;
      else if ((DelayCnt2 != 7'b0) && (Div2Count50NS == 2'h0) && SatDetEn)
         DelayCnt2 <= DelayCnt2 - 7'b1;
   end //DelayCnt2_Blk

//Compare input power with dynamic thresholds
assign PowGtDynHighRx2 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx2[DBVPOW_WIDTH-1:2]) > SatDynHighThr)) ?
                          1'b1 : 1'b0;

assign PowGtDynLowRx2 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx2[DBVPOW_WIDTH-1:2]) > SatDynLowThr)) ?
                          1'b1 : 1'b0;

//Mux the threshold crossings depending on the value of DelayCnt2
//and use it to detect saturation
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SatDetectIntRx2_Blk
      if (nAGCRst == 1'b0)
         SatDetectIntRx2 <= 1'b0;
      else if (FSMBlkRst || !SatDetEn)
         SatDetectIntRx2 <= 1'b0;
      else if ((DelayCnt2 == 7'b0) && PowGtDynHighRx2)
         SatDetectIntRx2 <= 1'b1;
      else if ((DelayCnt2 != 7'b0) && PowGtDynLowRx2)
         SatDetectIntRx2 <= 1'b1;
      else
         SatDetectIntRx2 <= 1'b0;
   end //SatDetectIntRx2_Blk

//For Rx Chain 3
//Enable for Delay Counter
assign DelayCnt3En = ((ADCSatPowValid == 1'b1) &&
                      ($signed(ADCSatPowdBVRx3[DBVPOW_WIDTH-1:2]) > SatLowThr)) ?
                      1'b1 : 1'b0;

//Div Counter for 50 ns delay
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Div3Count50NS_Blk
      if (nAGCRst == 1'b0)
         Div3Count50NS <= 2'h0;
      else if (!DelayCnt3En)
         Div3Count50NS <= 2'h0;
      else
         Div3Count50NS <= Div3Count50NS + 2'h1;
   end //Div3Count50NS_Blk

//Delay Counter 3
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelayCnt3_Blk
      if (nAGCRst == 1'b0)
         DelayCnt3 <= 7'b0;
      else if (!DelayCnt3En)
         DelayCnt3 <= SatDelayCount;
      else if ((DelayCnt3 != 7'b0) && (Div3Count50NS == 2'h0) && SatDetEn)
         DelayCnt3 <= DelayCnt3 - 7'b1;
   end //DelayCnt3_Blk

//Compare input power with dynamic thresholds
assign PowGtDynHighRx3 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx3[DBVPOW_WIDTH-1:2]) > SatDynHighThr)) ?
                          1'b1 : 1'b0;

assign PowGtDynLowRx3 = ((ADCSatPowValid == 1'b1) &&
                          ($signed(ADCSatPowdBVRx3[DBVPOW_WIDTH-1:2]) > SatDynLowThr)) ?
                          1'b1 : 1'b0;

//Mux the threshold crossings depending on the value of DelayCnt3
//and use it to detect saturation
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SatDetectIntRx3_Blk
      if (nAGCRst == 1'b0)
         SatDetectIntRx3 <= 1'b0;
      else if (FSMBlkRst || !SatDetEn)
         SatDetectIntRx3 <= 1'b0;
      else if ((DelayCnt3 == 7'b0) && PowGtDynHighRx3)
         SatDetectIntRx3 <= 1'b1;
      else if ((DelayCnt3 != 7'b0) && PowGtDynLowRx3)
         SatDetectIntRx3 <= 1'b1;
      else
         SatDetectIntRx3 <= 1'b0;
   end //SatDetectIntRx3_Blk

//Mask final output as saturation event is not a pulse
assign SatDetectRx0 = SatDetectIntRx0 & SatDetEn;
assign SatDetectRx1 = SatDetectIntRx1 & SatDetEn;
assign SatDetectRx2 = SatDetectIntRx2 & SatDetEn;
assign SatDetectRx3 = SatDetectIntRx3 & SatDetEn;

//Decode AGC Commands
assign FSMBlkRst = (AGCCmdValid == 1'b1 && AGCCommand == 8'd1) ? 1'b1 : 1'b0;

endmodule //SatDetect

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
