////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Signed Threshold Crossing Disappearance with FSM Enable
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/ThreshDisSgnFSM.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module ThreshDisSgnFSM #(parameter DBVPOW_WIDTH = 12 //Data width of dBm power
                        )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,   //Active Low Reset
            input    wire                                  AGCClk,    //AGC Clock

            //Config Registers
            input    wire   signed     [DBVPOW_WIDTH-1:0]  RegADCPowDisThrdBV, //Pow Disappearance
            input    wire              [3:0]               RegADCPowDisNdl,    //Delay line tap out

            //Power & Valid
            input    wire   signed     [DBVPOW_WIDTH-1:0]  ADCPowdBV,      //ADC Power in dBV
            input    wire                                  ADCPowValid,    //Qualifies ADCPowdBV

            //Controls
            input    wire                                  ADCPowDisDetEn, //Block enable
            input    wire              [7:0]               AGCCommand,     //Command from FSM Core
            input    wire                                  AGCCmdValid,    //AGC Command Valid 

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output  wire                                   ADCPowDis //Disappearance Detected
            );

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                    DelLineClr;
reg                [9:0]               DelayLine;
reg                [3:0]               SampleCount;
reg                                    ADCPowDisSel;
reg                                    ADCPowDisInt;
reg                                    ADCPowDisIntD;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module implements a disappearance detection circuit. This has to be
//instantiated per Rx chain, wherever needed. The current value is compared
//with a value at the delay line tap, and based on the
//reference values in the config registers, the disappearance signal is
//asserted. This block always runs on 5 MSamples/s, so the valid must
//be at 5 MHz. The Valid signal should also be passed through the delay line
//so that the comparison is not done on junk samples at the output of the
//delay line.

//Delay Line
always @ (posedge AGCClk or negedge nAGCRst)
   begin: Delay_Blk
      if (nAGCRst == 1'b0) begin
         DelayLine <= 10'b0;
      end
      else if ((DelLineClr == 1'b1) || (ADCPowDisDetEn == 1'b0)) begin
         DelayLine <= 10'b0;
      end
      else if ((ADCPowValid == 1'b1) && (SampleCount == 4'h0)) begin
         // Shiht register
         DelayLine[9:1] <= DelayLine[8:0];
         
         if ($signed(ADCPowdBV) <= $signed(RegADCPowDisThrdBV))
           DelayLine[0] <= 1'b1;
         else
           DelayLine[0] <= 1'b0;
      end
   end //Delay_Blk

//Sample Counter for 5MS/s
always @ (posedge AGCClk or negedge nAGCRst)
   begin: SampleCnt_Blk
      if (nAGCRst == 1'b0)
         SampleCount <= 4'h0;
      else if ((DelLineClr == 1'b1) || (ADCPowDisDetEn == 1'b0))
         SampleCount <= 4'h0;
      else if (ADCPowValid == 1'b1)
         SampleCount <= SampleCount + 4'h1;
   end //SampleCnt_Blk

//Selecting the bit vector for Disappearance
always @ (*)
   begin: ADCPowDisSel_Blk
      case (RegADCPowDisNdl)
        4'b0000: ADCPowDisSel = (DelayLine[0]   == 1'b1) ? 1'b1 : 1'b0;
        4'b0001: ADCPowDisSel = (DelayLine[1:0] == 2'b11) ? 1'b1 : 1'b0;
        4'b0010: ADCPowDisSel = (DelayLine[2:0] == 3'b111) ? 1'b1 : 1'b0;
        4'b0011: ADCPowDisSel = (DelayLine[3:0] == 4'b1111) ? 1'b1 : 1'b0;
        4'b0100: ADCPowDisSel = (DelayLine[4:0] == 5'b11111) ? 1'b1 : 1'b0;
        4'b0101: ADCPowDisSel = (DelayLine[5:0] == 6'b111111) ? 1'b1 : 1'b0;
        4'b0110: ADCPowDisSel = (DelayLine[6:0] == 7'b1111111) ? 1'b1 : 1'b0;
        4'b0111: ADCPowDisSel = (DelayLine[7:0] == 8'b11111111) ? 1'b1 : 1'b0;
        4'b1000: ADCPowDisSel = (DelayLine[8:0] == 9'b111111111) ? 1'b1 : 1'b0;
        default: ADCPowDisSel = (DelayLine[9:0] == 10'b1111111111) ? 1'b1 : 1'b0;
      endcase //RegADCPowDisNdl
   end //ADCPowDisSel_Blk

//Disappearance Detection
always @ (posedge AGCClk or negedge nAGCRst)
   begin: ADCPowDisInt_Blk
      if (nAGCRst == 1'b0)
         ADCPowDisInt <= 1'b0;
      else if (ADCPowDisSel == 1'b1)
         ADCPowDisInt <= 1'b1;
      else
         ADCPowDisInt <= 1'b0;
   end //ADCPowDis_Blk

assign ADCPowDis = ADCPowDisInt & ~ADCPowDisIntD & ADCPowDisDetEn;

//DelLineClr
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DelLineClr_Blk
      if (nAGCRst == 1'b0)
         DelLineClr <= 1'b0;
      else if ((AGCCommand == 8'd1 || AGCCommand == 8'd80) &&
               (AGCCmdValid == 1'b1))
         DelLineClr <= 1'b1;
      else
         DelLineClr <= 1'b0;
   end //DelLineClr_Blk

//Disappearance detection delayed
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DisDetIntD_Blk
      if (nAGCRst == 1'b0)
         ADCPowDisIntD <= 1'b0;
      else
         ADCPowDisIntD <= ADCPowDisInt;
   end //DisDetIntD_Blk

endmodule //ThreshDisSgnFSM

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
