////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Time Out Counter for AGC FSM 
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/AGCFSM/verilog/rtl/TimeOutCtr.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module TimeOutCtr (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nAGCRst,     //Active Low Reset
            input    wire                                  AGCClk,   //FSM Clock

            // Controls
            input    wire                                  TimeOutEn, //Qualifies TimeOutValue
            input    wire              [10:0]              TimeOutValue, //Count Value for Timer
            input    wire                                  TimeOutRst, //TimeOut Counter Reset

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Controls
            output    reg                                  TimeOut //Time Out Event Indication
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam FSM_SIZE = 4;

localparam IDLE  = 4'd1,
           LOAD  = 4'd2,
           COUNT = 4'd4,
           TMOUT = 4'd8;


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [FSM_SIZE-1:0]      TCFSMCS;
reg                [FSM_SIZE-1:0]      TCFSMNS;
reg                [10:0]              TimerReg;
reg                [10:0]              TimeOutValInt;
reg                [2:0]               DivCount100NS;

// The following 2 strings are useful during simulations
// to view States using strings instead of numbers
`ifdef SIMU_ON
  reg [25*8:0] CurrStateStr;
  reg [25*8:0] NextStateStr;
`endif

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This module implements a simple time out counter. It gets loaded with the
//terminal count value when TimeOutEn is asserted by the AGC FSM. After that,
//it decrements the TimeOutValue on every clock until it reaches zero. Upon
//reaching zero, TimeOut is asserted which is used as a transition condition
//in the decoder. TimeOutRst brings the counter back to its intiail state.

//TimeOutCounter FSM
//Sequential Block
always @ (posedge AGCClk or negedge nAGCRst)
   begin: FSM_Seq
      if (nAGCRst == 1'b0)
         TCFSMCS <= IDLE;
      else if (TimeOutRst == 1'b1)
         TCFSMCS <= IDLE;
      else
         TCFSMCS <= TCFSMNS;
   end //FSM_Seq

//Combinatorial Block
always @ (*)
   begin: FSM_Combo
      case(TCFSMCS)

         IDLE: begin
            if (TimeOutEn == 1'b1)
               TCFSMNS = LOAD;
            else
               TCFSMNS = IDLE;
         end //IDLE

         LOAD: begin
            TCFSMNS = COUNT;
         end //LOAD

         COUNT: begin
            if (TimerReg == 11'b0)
               TCFSMNS = TMOUT;
            else
               TCFSMNS = COUNT;
         end //COUNT

         TMOUT: begin
            TCFSMNS = IDLE;
         end //TMOUT

         // pragma coverage block = off
         default: begin
            TCFSMNS = IDLE;
         end //default
         // pragma coverage block = on

      endcase //TCFSMCS
   end //FSM_Combo

//TimerReg
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TimerReg_Blk
      if (nAGCRst == 1'b0)
         TimerReg <= 11'b0;
      else if (TCFSMCS == LOAD)
         TimerReg <= TimeOutValInt;
      else if ((TimerReg != 11'b0) && (DivCount100NS == 3'h0))
         TimerReg <= TimerReg - 11'b1;
   end //TimerReg_Blk

//TimeOut
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TimeOut_Blk
      if (nAGCRst == 1'b0)
         TimeOut <= 1'b0;
      else if (TCFSMCS == TMOUT)
         TimeOut <= 1'b1;
      else
         TimeOut <= 1'b0;
   end //TimeOut_Blk

//TimeOutValInt
//Counter Terminal Value
always @ (posedge AGCClk or negedge nAGCRst)
   begin: TimeOutVal_Blk
      if (nAGCRst == 1'b0)
         TimeOutValInt <= 11'b0;
      else if (TimeOutRst == 1'b1)
         TimeOutValInt <= 11'b0;
      else if (TimeOutEn == 1'b1 && TimeOutValue > 11'd1)
         TimeOutValInt <= TimeOutValue - 11'd1;
   end //TimeOutVal_Blk

//Div Counter for 100 ns delay
always @ (posedge AGCClk or negedge nAGCRst)
   begin: DivCount100NS_Blk
      if (nAGCRst == 1'b0)
         DivCount100NS <= 3'h0;
      else if (TimeOutEn == 1'b1)
         DivCount100NS <= 3'h0;
      else
         DivCount100NS <= DivCount100NS + 3'h1;
   end //DivCount100NS_Blk

// assigning Strings to States so that it would be easy
// to debug in simulation
`ifdef SIMU_ON
always@(*)
begin
  case (TCFSMCS)
    IDLE                : CurrStateStr = {"IDLE"};
    LOAD                : CurrStateStr = {"LOAD"};
    COUNT               : CurrStateStr = {"COUNT"};
    TMOUT               : CurrStateStr = {"TMOUT"};
    default             : CurrStateStr = {"UNKNOWN"};
  endcase
end
always@(*)
begin
  case (TCFSMNS)
    IDLE                : NextStateStr = {"IDLE"};
    LOAD                : NextStateStr = {"LOAD"};
    COUNT               : NextStateStr = {"COUNT"};
    TMOUT               : NextStateStr = {"TMOUT"};
    default             : NextStateStr = {"UNKNOWN"};
  endcase
end
`endif

endmodule //TimeOutCtr

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
