//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 12940 $
// $Date: 2014-01-16 14:13:28 +0100 (Thu, 16 Jan 2014) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Ofdm Modem state based CCA
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/CCAFSM/verilog/rtl/MdmStateCca.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module MdmStateCca(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  AGCClk,
            input    wire                                  nAGCRst,

            // Data and Control
            input    wire                                  FsmRst,
            input    wire                                  AGCOFDMLock,
            input    wire                                  AGCDSSSLock,
            input    wire                                  SFDFound,
            input    wire                                  LSIGValid,
            input    wire                                  HTSIGValid,
            input    wire                                  RxEndTimingP,
            input    wire                                  BWDetected,
            input    wire                                  BWSup20,
            input    wire                                  BWEq40,
            input    wire                                  BWEq80,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   wire [10:0]                           RxStateOut,
            //
            output   reg  [3:0]                            DbgCcaFsm
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam IDLE  = 12'b000000000001;
localparam OFDM0 = 12'b000000000010;
localparam OFDM1 = 12'b000000000100;
localparam OFDM2 = 12'b000000001000;
localparam OFDM3 = 12'b000000010000;
localparam OFDM4 = 12'b000000100000;
localparam OFDM5 = 12'b000001000000;
localparam OFDM6 = 12'b000010000000;
localparam OFDM7 = 12'b000100000000;
localparam DSSS0 = 12'b001000000000;
localparam DSSS1 = 12'b010000000000;
localparam DSSS2 = 12'b100000000000;


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg      [11:0]  CurrState;
reg      [11:0]  NextState;
reg      [10:0]  NextRxStateOut;
reg      [10:0]  RxStateOutInt;
reg              AGCOFDMLockD;
reg              AGCDSSSLockD;
reg              RxEndTimingPD;
reg              LSIGValidD;
reg              HTSIGValidD;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire             AGCOFDMLockRise;
wire             AGCDSSSLockRise;
wire             RxEndTimingPRise;
wire             LSIGValidRise;
wire             HTSIGValidRise;

// The following 2 strings are useful during simulations
// to view States using strings instead of numbers
`ifdef SIMU_ON
  reg [25*8:0] CCACurrStateStr;
  reg [25*8:0] CCANextStateStr;
`endif


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Output assignment
  assign RxStateOut = RxStateOutInt;

// Rise detection
  always @ (posedge AGCClk or negedge nAGCRst) begin
    if (nAGCRst == 1'b0) begin
      AGCOFDMLockD  <= 1'b0;
      AGCDSSSLockD  <= 1'b0;
      RxEndTimingPD <= 1'b0;
      LSIGValidD    <= 1'b0;
      HTSIGValidD   <= 1'b0;
    end
    else begin
      AGCOFDMLockD  <= AGCOFDMLock;
      AGCDSSSLockD  <= AGCDSSSLock;
      RxEndTimingPD <= RxEndTimingP;
      LSIGValidD    <= LSIGValid;
      HTSIGValidD   <= HTSIGValid;
    end
  end

assign AGCOFDMLockRise  = AGCOFDMLock  & ~AGCOFDMLockD;
assign AGCDSSSLockRise  = AGCDSSSLock  & ~AGCDSSSLockD;
assign RxEndTimingPRise = RxEndTimingP & ~RxEndTimingPD;
assign LSIGValidRise    = LSIGValid    & ~LSIGValidD;
assign HTSIGValidRise   = HTSIGValid   & ~HTSIGValidD;

//State Machine Sequential Part
  always @ (posedge AGCClk or negedge nAGCRst) begin :SeqBlk
    if (nAGCRst == 1'b0) begin
      CurrState     <= IDLE;
      RxStateOutInt <= 11'd0;
    end
    else if(FsmRst) begin
      CurrState     <= IDLE;
      RxStateOutInt <= 11'd0;
    end
    else begin
      CurrState     <= NextState;
      RxStateOutInt <= NextRxStateOut;
    end
  end //SeqBlk

//Next State assignment
  always @ (*) begin :NextStateBlk
    case (CurrState) //synopsys full_case parallel_case
      IDLE: begin
        if (AGCOFDMLockRise)
          NextState = OFDM0;
        else if (AGCDSSSLockRise)
          NextState = DSSS0;
        else
          NextState = IDLE;
      end

      OFDM0: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else if (BWDetected) begin
          if (BWSup20)
            NextState = OFDM2;
          else
            NextState = OFDM1;
        end
        else
          NextState = OFDM0;
      end

      OFDM1: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else if (LSIGValidRise)
          NextState = OFDM3;
        else if (HTSIGValidRise) begin
          if (BWEq80)
            NextState = OFDM7;
          else if (BWEq40)
            NextState = OFDM6;
          else
            NextState = OFDM5;
        end
        else
          NextState = OFDM1;
      end

      OFDM2: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else if (LSIGValidRise)
          NextState = OFDM4;
        else if (HTSIGValidRise) begin
          if (BWEq80)
            NextState = OFDM7;
          else if (BWEq40)
            NextState = OFDM6;
          else
            NextState = OFDM5;
        end
        else
          NextState = OFDM2;
      end

      OFDM3: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else if (HTSIGValidRise) begin
          if (BWEq80)
            NextState = OFDM7;
          else if (BWEq40)
            NextState = OFDM6;
          else
            NextState = OFDM5;
        end
        else
          NextState = OFDM3;
      end

      OFDM4: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else if (HTSIGValidRise) begin
          if (BWEq80)
            NextState = OFDM7;
          else if (BWEq40)
            NextState = OFDM6;
          else
            NextState = OFDM5;
        end
        else
          NextState = OFDM4;
      end

      OFDM5: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else
          NextState = OFDM5;
      end

      OFDM6: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else
          NextState = OFDM6;
      end

      OFDM7: begin
        if (RxEndTimingPRise || !AGCOFDMLock)
          NextState = IDLE;
        else
          NextState = OFDM7;
      end

      DSSS0: begin
        if (RxEndTimingPRise || !AGCDSSSLock)
          NextState = IDLE;
        else if (SFDFound)
          NextState = DSSS1;
        else
          NextState = DSSS0;
      end

      DSSS1: begin
        if (RxEndTimingPRise || !AGCDSSSLock)
          NextState = IDLE;
        else if (LSIGValidRise)
          NextState = DSSS2;
        else
          NextState = DSSS1;
      end

      DSSS2: begin
        if (RxEndTimingPRise || !AGCDSSSLock)
          NextState = IDLE;
        else
          NextState = DSSS2;
      end

      // pragma coverage block = off
      default: begin
        NextState = IDLE;
      end
      // pragma coverage block = on

    endcase
  end //NextStateBlk

//Output assignment
  always @ (*) begin :OutputBlk
    case (CurrState)
      IDLE: begin
          NextRxStateOut = 11'd0;
          DbgCcaFsm      = 4'd0;
      end

      OFDM0: begin
          NextRxStateOut = 11'd1;
          DbgCcaFsm      = 4'd1;
      end

      OFDM1: begin
          NextRxStateOut = 11'd2;
          DbgCcaFsm      = 4'd2;
      end

      OFDM2: begin
          NextRxStateOut = 11'd4;
          DbgCcaFsm      = 4'd3;
      end

      OFDM3: begin
          NextRxStateOut = 11'd8;
          DbgCcaFsm      = 4'd4;
      end

      OFDM4: begin
          NextRxStateOut = 11'd16;
          DbgCcaFsm      = 4'd5;
      end

      OFDM5: begin
          NextRxStateOut = 11'd32;
          DbgCcaFsm      = 4'd6;
      end

      OFDM6: begin
          NextRxStateOut = 11'd64;
          DbgCcaFsm      = 4'd7;
      end

      OFDM7: begin
          NextRxStateOut = 11'd128;
          DbgCcaFsm      = 4'd8;
      end

      DSSS0: begin
          NextRxStateOut = 11'd256;
          DbgCcaFsm      = 4'd9;
      end

      DSSS1: begin
          NextRxStateOut = 11'd512;
          DbgCcaFsm      = 4'd10;
      end

      DSSS2: begin
          NextRxStateOut = 11'd1024;
          DbgCcaFsm      = 4'd11;
      end

      // pragma coverage block = off
      default: begin
          NextRxStateOut = 11'd0;
          DbgCcaFsm      = 4'd0;
      end
      // pragma coverage block = on

    endcase
  end //OutputBlk

// assigning Strings to States so that it would be easy
// to debug in simulation
`ifdef SIMU_ON
always@(*)
begin
  case (CurrState)
    IDLE      : CCACurrStateStr = {"IDLE"};

    OFDM0     : CCACurrStateStr = {"OFDM0"};
    OFDM1     : CCACurrStateStr = {"OFDM1"};
    OFDM2     : CCACurrStateStr = {"OFDM2"};
    OFDM3     : CCACurrStateStr = {"OFDM3"};
    OFDM4     : CCACurrStateStr = {"OFDM4"};
    OFDM5     : CCACurrStateStr = {"OFDM5"};
    OFDM6     : CCACurrStateStr = {"OFDM6"};
    OFDM7     : CCACurrStateStr = {"OFDM7"};

    DSSS0     : CCACurrStateStr = {"DSSS0"};
    DSSS1     : CCACurrStateStr = {"DSSS1"};
    DSSS2     : CCACurrStateStr = {"DSSS2"};
    default   : CCACurrStateStr = {"UNKNOWN"};
  endcase
end
always@(*)
begin
  case (NextState)
    IDLE      : CCANextStateStr = {"IDLE"};

    OFDM0     : CCANextStateStr = {"OFDM0"};
    OFDM1     : CCANextStateStr = {"OFDM1"};
    OFDM2     : CCANextStateStr = {"OFDM2"};
    OFDM3     : CCANextStateStr = {"OFDM3"};
    OFDM4     : CCANextStateStr = {"OFDM4"};
    OFDM5     : CCANextStateStr = {"OFDM5"};
    OFDM6     : CCANextStateStr = {"OFDM6"};
    OFDM7     : CCANextStateStr = {"OFDM7"};

    DSSS0     : CCANextStateStr = {"DSSS0"};
    DSSS1     : CCANextStateStr = {"DSSS1"};
    DSSS2     : CCANextStateStr = {"DSSS2"};
    default   : CCANextStateStr = {"UNKNOWN"};
  endcase
end
`endif

endmodule // MdmStateCca

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
