//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 10209 $
// $Date: 2013-09-23 10:03:49 +0200 (Mon, 23 Sep 2013) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Reference level calculation module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/Correlator/verilog/rtl/CorrRefLev.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module CorrRefLev #(parameter DATAINWIDTH = 13, // Input data from TXRX FE and delay lines
                    parameter OFDMOUTWIDTH = 17, // Accumulated RefLev output
                    parameter DSSSOUTWIDTH = 18  // Accumulated RefLev output
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  AGCClk,
            //Reset resynchronized to AGCClk
            input    wire                                  nAGCRst,

            //Control Signals
            //Enable for Correlation Reference cal block, from AGC FSM
            input    wire                                  RefLevBlockEn,
            //Subtract zeros instead of samples from delay block for 0.8
            input    wire                                  DsssSubZero,
            input    wire                                  OfdmSubZero,
            //Enable for input data
            input    wire                                  DataInEn,
            //Data Input
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DI,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DQ,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Output Accumulated Reference level
            output   wire          [DSSSOUTWIDTH-1:0]  RefLevDsss,
            output   wire          [OFDMOUTWIDTH-1:0]  RefLevOfdm,
            //Enable for output data
            output   wire                              RefLevDataEn
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam             [DATAINWIDTH-1:0] SIG_DATAINWIDTH_PARAM_0    = {DATAINWIDTH{1'b0}};
localparam             [DATAINWIDTH+3:0] SIG_DATAINWIDTH_P4_PARAM_0 = {DATAINWIDTH+4{1'b0}};
localparam             [DATAINWIDTH+4:0] SIG_DATAINWIDTH_P5_PARAM_0 = {DATAINWIDTH+5{1'b0}};


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [DATAINWIDTH-1:0]   ModApproxOut;
wire               [DATAINWIDTH-1:0]   ModApproxOut16D;
wire               [DATAINWIDTH-1:0]   ModApproxOut20D;
wire               [DATAINWIDTH-1:0]   MuxOut16D;
wire               [DATAINWIDTH-1:0]   MuxOut20D;
wire               [DATAINWIDTH+3:0]   AccSumOfdm;
wire               [DATAINWIDTH+4:0]   AccSumDsss;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [DATAINWIDTH+3:0]   AccSumOfdmReg;
reg                [DATAINWIDTH+4:0]   AccSumDsssReg;
reg                [2:0]               DelDataInEn;
reg                                    DelDsssSubZero;
reg                                    DelOfdmSubZero;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//delayed version of input control signals
  always @ (posedge AGCClk or negedge nAGCRst) begin:EnDelayBlk
    if (nAGCRst == 1'b0) begin
      DelDataInEn    <= 3'd0;
      DelDsssSubZero <= 1'd0;
      DelOfdmSubZero <= 1'd0;
    end
    else begin
      if (RefLevBlockEn == 1'b0) begin
        DelDataInEn    <= 3'd0;
        DelDsssSubZero <= 1'd0;
        DelOfdmSubZero <= 1'd0;
      end
      else begin
        DelDataInEn[0]   <= DataInEn;
        DelDataInEn[2:1] <= DelDataInEn[1:0];
        if (DataInEn == 1'b1) begin
          DelDsssSubZero <= DsssSubZero;
          DelOfdmSubZero <= OfdmSubZero;
        end
      end
    end
  end //EnDelayBlk

//Mod Approx Data path
  ModApprox # (
              .SUM_WIDTH(DATAINWIDTH),
              .CORRVAL_WIDTH(DATAINWIDTH)
              )
              U_ModApprox (
                //Inputs
                .PhyClk(AGCClk),
                .nPhyRst(nAGCRst),
                .ISum(DataInI),
                .QSum(DataInQ),
                .ComputeOn(RefLevBlockEn),

                //Output
                .CorrVal(ModApproxOut)
              );

//Mod Approx 16 delayed data path
  ModApprox # (
              .SUM_WIDTH(DATAINWIDTH),
              .CORRVAL_WIDTH(DATAINWIDTH)
              )
              U_ModApprox16D (
                //Inputs
                .PhyClk(AGCClk),
                .nPhyRst(nAGCRst),
                .ISum(DataIn16DI),
                .QSum(DataIn16DQ),
                .ComputeOn(RefLevBlockEn),

                //Output
                .CorrVal(ModApproxOut16D)
              );

//Mod Approx 20 delayed data path
  ModApprox # (
              .SUM_WIDTH(DATAINWIDTH),
              .CORRVAL_WIDTH(DATAINWIDTH)
              )
              U_ModApprox20D (
                //Inputs
                .PhyClk(AGCClk),
                .nPhyRst(nAGCRst),
                .ISum(DataIn20DI),
                .QSum(DataIn20DQ),
                .ComputeOn(RefLevBlockEn),

                //Output
                .CorrVal(ModApproxOut20D)
              );

//Muxing the Output of ModApprox with zero when DsssSubZero is set to 1
//This is to avoid invalid inputs from the delay line initially
  assign MuxOut16D = (DelOfdmSubZero | OfdmSubZero) ? SIG_DATAINWIDTH_PARAM_0 : ModApproxOut16D;
  assign MuxOut20D = (DelDsssSubZero | DsssSubZero) ? SIG_DATAINWIDTH_PARAM_0 : ModApproxOut20D;

//Accumulation of data to calculate the reference level
  assign AccSumOfdm = {4'b0,ModApproxOut} - {4'b0,MuxOut16D} + AccSumOfdmReg;
  assign AccSumDsss = {5'b0,ModApproxOut} - {5'b0,MuxOut20D} + AccSumDsssReg;

//Delay. Registering the data path
  always @ (posedge AGCClk or negedge nAGCRst) begin: RegBlk
    if (nAGCRst == 1'b0) begin
      AccSumOfdmReg <= SIG_DATAINWIDTH_P4_PARAM_0;
      AccSumDsssReg <= SIG_DATAINWIDTH_P5_PARAM_0;
    end
    else begin
      if (RefLevBlockEn == 1'b0) begin
        AccSumOfdmReg <= SIG_DATAINWIDTH_P4_PARAM_0;
        AccSumDsssReg <= SIG_DATAINWIDTH_P5_PARAM_0;
      end
      else if (DelDataInEn[1] == 1'b1)begin
        AccSumOfdmReg <= AccSumOfdm;
        AccSumDsssReg <= AccSumDsss;
      end
    end
  end // RegBlk

//Output Assignment
  assign RefLevDsss = AccSumDsssReg;
  assign RefLevOfdm = AccSumOfdmReg;

  assign RefLevDataEn = DelDataInEn[2];

endmodule // CorrRefLev

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////

