//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 10209 $
// $Date: 2013-09-23 10:03:49 +0200 (Mon, 23 Sep 2013) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Reference level calculation top module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/Projects/WLAN_NX_SDM_DS_CEL/HW/Modem/RIU/AGC/Correlator/verilog/rtl/CorrRefLevTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module CorrRefLevTop #(parameter NRX            = 4,   // Number of RX chains
                       parameter DATAINWIDTH    = 13,  // Input data from TXRX FE and delay lines
                       parameter OFDMOUTWIDTH   = 19,  // Ofdm Corr Reference output
                       parameter DSSSOUTWIDTH   = 20  // Dsss Corr Reference output
                      )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////

            //Clock and Reset
            input    wire                                  AGCClk,
            //Reset resynchronized to AGCClk
            input    wire                                  nAGCRst,

            //Control Signals
            //Enable Correlator ref calculation block, from AGC FSM
            input    wire                                  RefLevBlockEn,
            //Add zeros instead of samples from delay block for first 1.0 us
            input    wire                                  DsssSubZero,
            //Add zeros instead of samples from delay block for first 0.8 us
            input    wire                                  OfdmSubZero,
            //Enable for input data
            input    wire                                  DataInEn,

            //Data Input RxChain1
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DI1,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DQ1,

            //Data Input RxChain2
            input    wire   signed     [DATAINWIDTH-1:0]   DataInI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataInQ2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn16DQ2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DI2,
            input    wire   signed     [DATAINWIDTH-1:0]   DataIn20DQ2,

            //Data Input RxChain3
            input   wire   signed      [DATAINWIDTH-1:0]   DataInI3,
            input   wire   signed      [DATAINWIDTH-1:0]   DataInQ3,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn16DI3,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn16DQ3,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn20DI3,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn20DQ3,

            //Data Input RxChain4
            input   wire   signed      [DATAINWIDTH-1:0]   DataInI4,
            input   wire   signed      [DATAINWIDTH-1:0]   DataInQ4,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn16DI4,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn16DQ4,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn20DI4,
            input   wire   signed      [DATAINWIDTH-1:0]   DataIn20DQ4,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Output Accumulated sum for AutoCorrelation
            output  wire               [DSSSOUTWIDTH-1:0]  RefLevDsss,
            output  wire               [OFDMOUTWIDTH-1:0]  RefLevOfdm,
            //Enable for output data
            output  wire                                   RefLevDataEn
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam             [DSSSOUTWIDTH-1:0] DSSSOUTWIDTH_PARAM_0 = {DSSSOUTWIDTH{1'b0}};
localparam             [OFDMOUTWIDTH-1:0] OFDMOUTWIDTH_PARAM_0 = {OFDMOUTWIDTH{1'b0}};


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [DSSSOUTWIDTH-1:0]  RegSumAccDsss;
reg                [OFDMOUTWIDTH-1:0]  RegSumAccOfdm;
reg                [DSSSOUTWIDTH-1:0]  RegSumAccDsssD1;
reg                [OFDMOUTWIDTH-1:0]  RegSumAccOfdmD1;
reg                [3:0]               DelDataOutEn;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [DSSSOUTWIDTH-1:0]  SumAccDsss;
wire               [OFDMOUTWIDTH-1:0]  SumAccOfdm;

wire               [3:0]               DataOutEn;
wire               [DATAINWIDTH+4:0]   RefLevDsssIn [3:0];
wire               [DATAINWIDTH+3:0]   RefLevOfdmIn [3:0];

wire    signed     [DATAINWIDTH-1:0]   DataInI [3:0];
wire    signed     [DATAINWIDTH-1:0]   DataInQ [3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn16DI [3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn16DQ [3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn20DI [3:0];
wire    signed     [DATAINWIDTH-1:0]   DataIn20DQ [3:0];

//Genvar
genvar                                 j,i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Assign Inputs to Array variable
  assign DataInI[0]    = DataInI1;
  assign DataInQ[0]    = DataInQ1;
  assign DataIn16DI[0] = DataIn16DI1;
  assign DataIn16DQ[0] = DataIn16DQ1;
  assign DataIn20DI[0] = DataIn20DI1;
  assign DataIn20DQ[0] = DataIn20DQ1;

  assign DataInI[1]    = DataInI2;
  assign DataInQ[1]    = DataInQ2;
  assign DataIn16DI[1] = DataIn16DI2;
  assign DataIn16DQ[1] = DataIn16DQ2;
  assign DataIn20DI[1] = DataIn20DI2;
  assign DataIn20DQ[1] = DataIn20DQ2;

  assign DataInI[2]    = DataInI3;
  assign DataInQ[2]    = DataInQ3;
  assign DataIn16DI[2] = DataIn16DI3;
  assign DataIn16DQ[2] = DataIn16DQ3;
  assign DataIn20DI[2] = DataIn20DI3;
  assign DataIn20DQ[2] = DataIn20DQ3;

  assign DataInI[3]    = DataInI4;
  assign DataInQ[3]    = DataInQ4;
  assign DataIn16DI[3] = DataIn16DI4;
  assign DataIn16DQ[3] = DataIn16DQ4;
  assign DataIn20DI[3] = DataIn20DI4;
  assign DataIn20DQ[3] = DataIn20DQ4;

//Data is registered with delayed version of DataInEn
  always @ (posedge AGCClk or negedge nAGCRst) begin:EnDelayBlk
    if (nAGCRst == 1'b0) begin
      DelDataOutEn <= 4'd0;
    end
    else begin
      if (RefLevBlockEn == 1'b0) begin
        DelDataOutEn <= 4'd0;
      end
      else begin
        DelDataOutEn[0] <= DataOutEn[0];
        DelDataOutEn[3:1] <= DelDataOutEn[2:0];
      end
    end
  end //EnDelayBlk

generate
  for (j=0; j<NRX; j=j+1) begin: GenCorrRefLev
    CorrRefLev U_CorrRefLev(
                 //Inputs
                 .AGCClk(AGCClk),
                 .nAGCRst(nAGCRst),
                 .RefLevBlockEn(RefLevBlockEn),
                 .DsssSubZero(DsssSubZero),
                 .OfdmSubZero(OfdmSubZero),
                 .DataInEn(DataInEn),
                 .DataInI(DataInI[j]),
                 .DataInQ(DataInQ[j]),
                 .DataIn16DI(DataIn16DI[j]),
                 .DataIn16DQ(DataIn16DQ[j]),
                 .DataIn20DI(DataIn20DI[j]),
                 .DataIn20DQ(DataIn20DQ[j]),

                 //Outputs
                 .RefLevDataEn(DataOutEn[j]),
                 .RefLevDsss(RefLevDsssIn[j]),
                 .RefLevOfdm(RefLevOfdmIn[j])
                 );
  end // GenCorrRefLev
endgenerate

//Adding the outputs from all the RX chains
generate
  for(i=NRX;i<4;i=i+1) begin: GenAdder

     assign RefLevDsssIn[i] = {{(DATAINWIDTH+5)}{1'b0}};
     assign RefLevOfdmIn[i] = {{(DATAINWIDTH+4)}{1'b0}};
     assign DataOutEn[i]    = 1'b0;

  end //GenAdder
endgenerate

  assign SumAccDsss = {{2{RefLevDsssIn[0][DATAINWIDTH+4]}},RefLevDsssIn[0]} +
                      {{2{RefLevDsssIn[1][DATAINWIDTH+4]}},RefLevDsssIn[1]} +
                      {{2{RefLevDsssIn[2][DATAINWIDTH+4]}},RefLevDsssIn[2]} +
                      {{2{RefLevDsssIn[3][DATAINWIDTH+4]}},RefLevDsssIn[3]};

  assign SumAccOfdm = {{2{RefLevOfdmIn[0][DATAINWIDTH+3]}},RefLevOfdmIn[0]} +
                      {{2{RefLevOfdmIn[1][DATAINWIDTH+3]}},RefLevOfdmIn[1]} +
                      {{2{RefLevOfdmIn[2][DATAINWIDTH+3]}},RefLevOfdmIn[2]} +
                      {{2{RefLevOfdmIn[3][DATAINWIDTH+3]}},RefLevOfdmIn[3]};

//Registering Sum Data
//Data Out is delays by 3 PhyClocks, so that all correlators and RefLev have
//Same delay
  always @ (posedge AGCClk or negedge nAGCRst) begin: RegSumData
    if (nAGCRst == 1'b0) begin
      RegSumAccDsss <= DSSSOUTWIDTH_PARAM_0;
      RegSumAccOfdm <= OFDMOUTWIDTH_PARAM_0;
    end
    else begin
      if (RefLevBlockEn == 1'b0) begin
        RegSumAccDsss <= DSSSOUTWIDTH_PARAM_0;
        RegSumAccOfdm <= OFDMOUTWIDTH_PARAM_0;
      end
      else if (DelDataOutEn[1] == 1'b1) begin
        RegSumAccDsss <= SumAccDsss;
        RegSumAccOfdm <= SumAccOfdm;
      end
    end
  end // RegSumData

//Registering output which must be always positive
  always @ (posedge AGCClk or negedge nAGCRst) begin: RegSumData2
    if (nAGCRst == 1'b0) begin
      RegSumAccDsssD1 <= DSSSOUTWIDTH_PARAM_0;
      RegSumAccOfdmD1 <= OFDMOUTWIDTH_PARAM_0;
    end
    else begin
      if (RefLevBlockEn == 1'b0) begin
        RegSumAccDsssD1 <= DSSSOUTWIDTH_PARAM_0;
        RegSumAccOfdmD1 <= OFDMOUTWIDTH_PARAM_0;
      end
      else begin
        RegSumAccDsssD1 <= RegSumAccDsss & {DSSSOUTWIDTH{~RegSumAccDsss[DSSSOUTWIDTH-1]}};
        RegSumAccOfdmD1 <= RegSumAccOfdm & {OFDMOUTWIDTH{~RegSumAccOfdm[OFDMOUTWIDTH-1]}};
      end
    end
  end // RegSumData2

//Output Assignment
  assign RefLevDsss = RegSumAccDsssD1;
  assign RefLevOfdm = RegSumAccOfdmD1;
  assign RefLevDataEn = DelDataOutEn[3];

endmodule // CorrRefLevTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
